#include "sdk-headers.h"

#ifdef SCITERJS
#include "sdk.js/include/sciter-x-graphics.h"
#else
#include "sdk/include/sciter-x-graphics.h"
#endif

#include "tool/tool.h"
#include "gool/gool.h"
#include "html/html.h"

#ifndef __GRAPHICS_API_CPP__
#define __GRAPHICS_API_CPP__

inline gool::argb torgba(SC_COLOR c) {
  return gool::argb(gool::r(c), gool::g(c), gool::b(c), gool::a(c));
}

GRAPHIN_RESULT SCAPI imageCreate(HIMG *poutImg, UINT width, UINT height,
                                 SBOOL withAlpha) {
  if (!poutImg) return GRAPHIN_BAD_PARAM;
  if (!width || !height) return GRAPHIN_BAD_PARAM;
  gool::bitmap *pbm = new gool::bitmap(gool::size(width, height), !!withAlpha);
  if (pbm) {
    pbm->add_ref();
    *poutImg = pbm;
    return GRAPHIN_OK;
  }
  return GRAPHIN_PANIC;
}

GRAPHIN_RESULT SCAPI imageCreateFromPixmap(HIMG *poutImg, UINT pixmapWidth,
                                           UINT pixmapHeight, SBOOL withAlpha,
                                           const BYTE *pixmapPixels) {
  if (!poutImg) return GRAPHIN_BAD_PARAM;
  if (!pixmapWidth || !pixmapHeight) return GRAPHIN_BAD_PARAM;
  if (!pixmapPixels) return GRAPHIN_BAD_PARAM;

  gool::bitmap *pbm =
      new gool::bitmap(gool::size(pixmapWidth, pixmapHeight), !!withAlpha);
  if (pbm) {
    pbm->add_ref();
    tool::slice<gool::argb> pixels = tool::slice<gool::argb>(
        (const gool::argb *)pixmapPixels, pixmapWidth * pixmapHeight);
    pbm->set_bits(pixels);
    *poutImg = pbm;
    return GRAPHIN_OK;
  }
  return GRAPHIN_PANIC;
}

GRAPHIN_RESULT SCAPI imageAddRef(HIMG himg) {
  if (!himg) return GRAPHIN_BAD_PARAM;
  himg->add_ref();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI imageRelease(HIMG himg) {
  if (!himg) return GRAPHIN_BAD_PARAM;
  himg->release();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI imageGetInfo(HIMG himg, UINT *width, UINT *height,
                                  SBOOL *usesAlpha) {
  if (!himg) return GRAPHIN_BAD_PARAM;
  if (!width || !height) return GRAPHIN_BAD_PARAM;
  if (!usesAlpha) return GRAPHIN_BAD_PARAM;

  gool::size sz = himg->dimension();
  *width        = sz.x;
  *height       = sz.y;
  *usesAlpha    = himg->is_transparent();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI imageClear(HIMG himg, SC_COLOR byColor) {
  if (!himg) return GRAPHIN_BAD_PARAM;
  gool::argb c = torgba(byColor);
  himg->clear(c);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI imageLoad(const BYTE *bytes, UINT num_bytes,
                               HIMG *pout_img) {
  // return GRAPHIN_NOTSUPPORTED;
  if (!pout_img) return GRAPHIN_BAD_PARAM;
  if (!bytes || !num_bytes) return GRAPHIN_BAD_PARAM;

  tool::handle<gool::image> pbm = // note: will add_ref the pointer
      gool::image::create(tool::bytes(bytes, num_bytes), tool::string());
  if (pbm) {
    //pbm->add_ref();
    *pout_img = pbm.detach(); // will keep pointer add_refed
    return GRAPHIN_OK;
  }
  return GRAPHIN_PANIC;
}

GRAPHIN_RESULT SCAPI imageSave // save png/jpeg/etc. image to stream of bytes
    (HIMG himg, image_write_function *pfn,
     void *prm, /* function and its param passed "as is" */
     UINT  packaging /* SCITER_IMAGE_ENCODING */,
     UINT  quality /* png: 0, jpeg:, 10 - 100 */) 
{
  if (!himg || !pfn) return GRAPHIN_BAD_PARAM;
  //SCITER_IMAGE_ENCODING enc;
  //SCITER_IMAGE_ENCODING_RAW, // [a,b,g,r,a,b,g,r,...] vector
  //  SCITER_IMAGE_ENCODING_PNG,
  //  SCITER_IMAGE_ENCODING_JPG,
  //  SCITER_IMAGE_ENCODING_WEBP,
  gool::image::PACKAGING pack = gool::image::UNKNOWN;
  switch (SCITER_IMAGE_ENCODING(packaging))
  {
    case SCITER_IMAGE_ENCODING_RAW:
    {
      handle<gool::bitmap> bmp = himg->get_bitmap(nullptr, himg->dimension());
      if (bmp) {
        auto bytes = bmp->pixels_as_bytes();
        pfn(prm, bytes.start, (UINT)bytes.length);
        return GRAPHIN_OK;
      }
      else {
        auto bits = himg->get_data();
        pfn(prm, bits.start, (UINT)bits.length);
        return GRAPHIN_OK;
      }
    }
    case SCITER_IMAGE_ENCODING_PNG: pack = gool::image::PACKAGING::PNG; break;
    case SCITER_IMAGE_ENCODING_JPG: pack = gool::image::PACKAGING::JPG; break;
    case SCITER_IMAGE_ENCODING_WEBP: pack = gool::image::PACKAGING::WEBP; break;
    default:
      return GRAPHIN_BAD_PARAM;
  }
  array<byte> data;
  himg->save(data, pack, quality);
  pfn(prm, data.cbegin(), (UINT)data.length());

  return GRAPHIN_OK;
}

SC_COLOR SCAPI RGBA(UINT red, UINT green, UINT blue, UINT alpha /*= 255*/) {
  return gool::rgba(red, green, blue, alpha);
}

GRAPHIN_RESULT SCAPI gCreate(HIMG img, HGFX *pout_gfx) {
  return GRAPHIN_NOTSUPPORTED;
}

GRAPHIN_RESULT SCAPI gAddRef(HGFX gfx) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->add_ref();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gRelease(HGFX gfx) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->release();
  return GRAPHIN_OK;
}

// Draws line from x1,y1 to x2,y2 using current lineColor and lineGradient.
GRAPHIN_RESULT SCAPI gLine(HGFX gfx, SC_POS x1, SC_POS y1, SC_POS x2,
                           SC_POS y2) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->draw_line(gool::pointf(x1, y1), gool::pointf(x2, y2));
  return GRAPHIN_OK;
}

// Draws rectangle using current lineColor/lineGradient and
// fillColor/fillGradient with (optional) rounded corners.
GRAPHIN_RESULT SCAPI gRectangle(HGFX gfx, SC_POS x1, SC_POS y1, SC_POS x2,
                                SC_POS y2) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  // gool::rectf(x1,y1,x2,y2);
  gfx->draw_rectangle(gool::pointf(x1, y1), gool::pointf(x2 - x1, y2 - y1));
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI
               gRoundedRectangle(HGFX gfx, SC_POS x1, SC_POS y1, SC_POS x2, SC_POS y2,
                                 const SC_DIM *radii8 /*SC_DIM[8] - four rx/ry pairs */) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->draw_rectangle(
      gool::pointf(x1, y1), gool::pointf(x2 - x1, y2 - y1),
      gool::sizef(radii8[0], radii8[1]), gool::sizef(radii8[2], radii8[3]),
      gool::sizef(radii8[4], radii8[5]), gool::sizef(radii8[6], radii8[7]));
  return GRAPHIN_OK;
}

// Draws circle or ellipse using current lineColor/lineGradient and
// fillColor/fillGradient.
GRAPHIN_RESULT SCAPI gEllipse(HGFX gfx, SC_POS x, SC_POS y, SC_DIM rx,
                              SC_DIM ry) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->draw_ellipse(gool::pointf(x, y), gool::sizef(rx, ry));
  return GRAPHIN_OK;
}

// Draws closed arc using current lineColor/lineGradient and
// fillColor/fillGradient.
GRAPHIN_RESULT SCAPI gArc(HGFX gfx, SC_POS x, SC_POS y, SC_POS rx, SC_POS ry,
                          SC_ANGLE start, SC_ANGLE sweep) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->draw_arc(gool::pointf(x, y), gool::sizef(rx, ry), start, sweep);
  return GRAPHIN_OK;
}

// Draws star.
GRAPHIN_RESULT SCAPI gStar(HGFX gfx, SC_POS x, SC_POS y, SC_POS r1, SC_POS r2,
                           SC_ANGLE start, UINT rays) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  gfx->draw_star(gool::pointf(x, y), gool::sizef(r1), gool::sizef(r2), start,
                 rays);
  return GRAPHIN_OK;
}

// Closed polygon.
GRAPHIN_RESULT SCAPI gPolygon(HGFX gfx, const SC_POS *xy, UINT num_points) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  tool::handle<gool::path> pg = gool::app()->create_path();
  pg->move_to(gool::pointf(xy[0], xy[1]));
  int pn = 2;
  for (UINT n = 1; n < num_points; ++n, pn += 2)
    pg->line_to(gool::pointf(xy[pn], xy[pn + 1]));
  pg->close();
  gfx->draw_path(pg);
  return GRAPHIN_OK;
}

// Polyline.
GRAPHIN_RESULT SCAPI gPolyline(HGFX gfx, const SC_POS *xy, UINT num_points) {
  if (!gfx) return GRAPHIN_BAD_PARAM;
  tool::handle<gool::path> pg = gool::app()->create_path();
  pg->move_to(gool::pointf(xy[0], xy[1]));
  int pn = 2;
  for (UINT n = 1; n < num_points; ++n, pn += 2)
    pg->line_to(gool::pointf(xy[pn], xy[pn + 1]));
  gfx->draw_path(pg, true, false);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathCreate(HPATH *ppath) {
  if (!ppath) return GRAPHIN_BAD_PARAM;
  gool::path *pg = gool::app()->create_path();
  if (pg) {
    pg->add_ref();
    *ppath = pg;
    return GRAPHIN_OK;
  }
  return GRAPHIN_PANIC;
}

GRAPHIN_RESULT SCAPI pathAddRef(HPATH path) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->add_ref();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathRelease(HPATH path) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->release();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathMoveTo(HPATH path, SC_POS x, SC_POS y, SBOOL relative) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->move_to(gool::pointf(x, y), relative != FALSE);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathLineTo(HPATH path, SC_POS x, SC_POS y, SBOOL relative) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->line_to(gool::pointf(x, y), relative != FALSE);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathArcTo(HPATH path, SC_POS x, SC_POS y, SC_ANGLE angle,
                               SC_DIM rx, SC_DIM ry, SBOOL is_large_arc,
                               SBOOL clockwise, SBOOL relative) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->arc_to(gool::pointf(x, y), gool::sizef(rx, ry), angle,
               is_large_arc != FALSE, clockwise != FALSE, relative != FALSE);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathQuadraticCurveTo(HPATH path, SC_POS xc, SC_POS yc,
                                          SC_POS x, SC_POS y, SBOOL relative) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->quadratic_to(gool::pointf(x, y), gool::sizef(xc, yc),
                     relative != FALSE);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathBezierCurveTo(HPATH path, SC_POS xc1, SC_POS yc1,
                                       SC_POS xc2, SC_POS yc2, SC_POS x,
                                       SC_POS y, SBOOL relative) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->cubic_to(gool::pointf(x, y), gool::sizef(xc1, yc1),
                 gool::sizef(xc2, yc2), relative != FALSE);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI pathClosePath(HPATH path) {
  if (!path) return GRAPHIN_BAD_PARAM;
  path->close();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gDrawPath(HGFX hgfx, HPATH path, DRAW_PATH_MODE dpm) {
  if (!hgfx || !path) return GRAPHIN_BAD_PARAM;
  hgfx->draw_path(path, (dpm & DRAW_STROKE_ONLY) != 0,
                  (dpm & DRAW_FILL_ONLY) != 0);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gRotate(HGFX hgfx, SC_ANGLE radians, const SC_POS *cx /*= 0*/, const SC_POS *cy /*= 0*/) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  if (cx && cy)
    hgfx->rotate(radians, gool::pointf(*cx, *cy));
  else
    hgfx->rotate(radians);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gTranslate(HGFX hgfx, SC_POS cx, SC_POS cy) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->translate(gool::pointf(cx, cy));
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gScale(HGFX hgfx, SC_DIM x, SC_DIM y) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->scale(gool::sizef(x, y));
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gSkew(HGFX hgfx, SC_DIM x, SC_DIM y) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->skew(gool::pointf(x, y));
  return GRAPHIN_OK;
}

// all above in one shot
GRAPHIN_RESULT SCAPI gTransform(HGFX hgfx, SC_POS m11, SC_POS m12, SC_POS m21,
                                SC_POS m22, SC_POS dx, SC_POS dy) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::affine_mtx_f mtx(m11, m12, m21, m22, dx, dy);
  hgfx->transform(mtx);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gStateSave(HGFX hgfx) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->push_state();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gStateRestore(HGFX hgfx) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->pop_state();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gFlush(HGFX hgfx) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->flush();
  return GRAPHIN_OK;
}


GRAPHIN_RESULT SCAPI gLineWidth(HGFX hgfx, SC_DIM width) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->set_stroke_width(width);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gLineJoin(HGFX hgfx, SCITER_LINE_JOIN_TYPE type) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->line_join(gool::LINE_JOIN(type));
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gLineCap(HGFX hgfx, SCITER_LINE_CAP_TYPE type) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->cap_style(gool::CAP_STYLE(type));
  return GRAPHIN_OK;
}

// SC_COLOR for solid lines/strokes
GRAPHIN_RESULT SCAPI gLineColor(HGFX hgfx, SC_COLOR c) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->set_stroke(torgba(c));
  return GRAPHIN_OK;
}

// SC_COLOR for solid fills
GRAPHIN_RESULT SCAPI gFillColor(HGFX hgfx, SC_COLOR c) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->set_fill(torgba(c));
  return GRAPHIN_OK;
}

// setup parameters of linear gradient of lines.
GRAPHIN_RESULT SCAPI gLineGradientLinear(HGFX hgfx, SC_POS x1, SC_POS y1,
                                         SC_POS x2, SC_POS y2,
                                         const SC_COLOR_STOP *stops, UINT nstops) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::linear_brush lb(gool::pointf(x1, y1), gool::pointf(x2, y2));
  for (uint n = 0; n < nstops; ++n)
    lb.add_stop(stops[n].offset, torgba(stops[n].color));
  hgfx->set_stroke(lb);
  return GRAPHIN_OK;
}

// setup parameters of linear gradient of fills.
GRAPHIN_RESULT SCAPI gFillGradientLinear(HGFX hgfx, SC_POS x1, SC_POS y1,
                                         SC_POS x2, SC_POS y2,
                                         const SC_COLOR_STOP *stops, UINT nstops) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::linear_brush lb(gool::pointf(x1, y1), gool::pointf(x2, y2));
  for (uint n = 0; n < nstops; ++n)
    lb.add_stop(stops[n].offset, torgba(stops[n].color));
  hgfx->set_fill(lb);
  return GRAPHIN_OK;
}

// setup parameters of line gradient radial fills.
GRAPHIN_RESULT SCAPI gLineGradientRadial(HGFX hgfx, SC_POS x, SC_POS y,
                                         SC_DIM rx, SC_DIM ry,
                                         const SC_COLOR_STOP *stops, UINT nstops) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::radial_brush rb(gool::pointf(x, y), gool::sizef(rx, ry));
  for (uint n = 0; n < nstops; ++n)
    rb.add_stop(stops[n].offset, torgba(stops[n].color));
  hgfx->set_stroke(rb);
  return GRAPHIN_OK;
}

// setup parameters of gradient radial fills.
GRAPHIN_RESULT SCAPI gFillGradientRadial(HGFX hgfx, SC_POS x, SC_POS y,
                                         SC_DIM rx, SC_DIM ry,
                                         const SC_COLOR_STOP *stops, UINT nstops) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::radial_brush rb(gool::pointf(x, y), gool::sizef(rx, ry));
  for (uint n = 0; n < nstops; ++n)
    rb.add_stop(stops[n].offset, torgba(stops[n].color));
  hgfx->set_fill(rb);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gFillMode(HGFX hgfx,
                               SBOOL even_odd /* false - fill_non_zero */) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->set_fill_even_odd(even_odd != FALSE);
  return GRAPHIN_OK;
}

// create text layout using element's styles
GRAPHIN_RESULT SCAPI textCreateForElement(HTEXT *ptext, LPCWSTR text,
                                          UINT textLength, HELEMENT he, LPCWSTR className) {
  if (!he) return GRAPHIN_BAD_PARAM;
  if (!ptext) return GRAPHIN_BAD_PARAM;
  html::element *b = (html::element *)he;
  if (b) {
    html::view *pv = b->pview();
    if (pv) {
      handle<gool::text_layout> tl = pv->create_text_layout(tool::wchars(text, textLength));
      if (tl) {
        *ptext = tl;
        tl->set_host(b,chars_of(className));
        tl->add_ref();
        return GRAPHIN_OK;
      } else {
        return GRAPHIN_PANIC;
      }
    }
  }
  return GRAPHIN_BAD_PARAM;
}

// create text layout using explicit format declaration
GRAPHIN_RESULT SCAPI textCreateForElementAndStyle(HTEXT *ptext, LPCWSTR text, UINT textLength, HELEMENT he, LPCWSTR style, UINT styleLength) {
  if (!style || !styleLength) return GRAPHIN_BAD_PARAM;
  if (!ptext) return GRAPHIN_BAD_PARAM;
  html::element *b = (html::element *)he;
  if (b) {
    html::view *pv = b->pview();
    if (pv) {
      handle<gool::text_layout> tl = pv->create_text_layout(tool::wchars(text, textLength));
      if (tl) {
        *ptext = tl;
        tl->set_style(wchars(style, styleLength));
        tl->set_host(b);
        tl->add_ref();
        return GRAPHIN_OK;
      }
      else {
        return GRAPHIN_PANIC;
      }
    }
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI textAddRef(HTEXT text) {
  if (!text) return GRAPHIN_BAD_PARAM;
  text->add_ref();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI textRelease(HTEXT text) {
  if (!text) return GRAPHIN_BAD_PARAM;
  text->release();
  return GRAPHIN_OK;
}


GRAPHIN_RESULT SCAPI textGetMetrics(HTEXT text, SC_DIM *minWidth,
                                    SC_DIM *maxWidth, SC_DIM *height,
                                    SC_DIM *ascent, SC_DIM *descent,
                                    UINT *nLines) {
  if (!text) return GRAPHIN_BAD_PARAM;

  if (minWidth) *minWidth = text->width_min();
  if (maxWidth) *maxWidth = text->width_max();
  if (height) *height = text->height();
  if (ascent) *ascent = text->ascent();
  if (descent) *descent = text->height() - text->ascent();
  if (nLines) *nLines = text->get_lines_count();

  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI textSetBox(HTEXT text, SC_DIM width, SC_DIM height) {
  if (!text) return GRAPHIN_BAD_PARAM;

  text->set_width(width);
  text->set_height(height);

  return GRAPHIN_OK;
}

// draw text with position (1..9 on MUMPAD) at px,py
// Ex: gDrawText( 100,100,5) will draw text box with its center at 100,100 px
GRAPHIN_RESULT SCAPI gDrawText(HGFX hgfx, HTEXT text, SC_POS px, SC_POS py,
                               UINT position) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  if (!text) return GRAPHIN_BAD_PARAM;

  gool::pointf dst(px, py);
  gool::rectf  rc(dst, text->get_box());
  if (position) rc.pointOf(position, dst);
  hgfx->draw(text, rc.s);

  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gDrawPath(HGFX hgfx, HIMG himg, SC_POS x, SC_POS y,
                                const SC_DIM *w /*= 0*/, const SC_DIM *h /*= 0*/,
                                const UINT *ix /*= 0*/, const UINT *iy /*= 0*/,
                                const UINT *iw /*= 0*/, const UINT *ih /*= 0*/,
                                const float *opacity /*= 0*/) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  if (!himg) return GRAPHIN_BAD_PARAM;

  gool::rectf dst(gool::pointf(x, y), himg->dimension());
  gool::rect  src(gool::point(0, 0), himg->dimension());

  byte op = 255;
  if (opacity) op = byte(*opacity * 255.0f);

  if (w && h) dst = gool::rectf(gool::pointf(x, y), gool::sizef(*w, *h));

  if (ix && iy && iw && ih)
    src = gool::rect(gool::point(*ix, *iy), gool::size(*iw, *ih));

  hgfx->draw_image(himg, dst, src, op);
  return GRAPHIN_OK;
}

// SECTION: coordinate space

GRAPHIN_RESULT SCAPI gWorldToScreen(HGFX hgfx, SC_POS *inout_x,
                                    SC_POS *inout_y) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  if (!inout_x) return GRAPHIN_BAD_PARAM;

  if (inout_y) {
    gool::pointf pt(*inout_x, *inout_y);
    hgfx->world_to_screen(pt);
    *inout_x = pt.x;
    *inout_y = pt.y;
  }
  else {
    hgfx->world_to_screen(*inout_x);
  }
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gScreenToWorld(HGFX hgfx, SC_POS *inout_x,
                                    SC_POS *inout_y) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  if (!inout_x) return GRAPHIN_BAD_PARAM;

  if (inout_y) {
    gool::pointf pt(*inout_x, *inout_y);
    hgfx->screen_to_world(pt);
    *inout_x = pt.x;
    *inout_y = pt.y;
  }
  else {
    hgfx->screen_to_world(*inout_x);
  }

  return GRAPHIN_OK;
}

// SECTION: clipping

GRAPHIN_RESULT SCAPI gPushClipBox(HGFX hgfx, SC_POS x1, SC_POS y1, SC_POS x2,
                                  SC_POS y2, float opacity) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  gool::rectf rc(x1, y1, x2, y2);
  byte        op = byte(opacity * 255.f);
  if (op == 0 || rc.empty()) return GRAPHIN_BAD_PARAM;
  hgfx->push_layer(rc, op);
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI gPushClipPath(HGFX hgfx, HPATH hpath,
                                   float opacity /*=1.f*/) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  byte op = byte(opacity * 255.f);
  if (op == 0 || !hpath) return GRAPHIN_BAD_PARAM;
  hgfx->push_layer(hpath, op);
  return GRAPHIN_OK;
}

// pop clip layer previously set by gPushClipBox or gPushClipPath
GRAPHIN_RESULT SCAPI gPopClip(HGFX hgfx) {
  if (!hgfx) return GRAPHIN_BAD_PARAM;
  hgfx->pop_layer();
  return GRAPHIN_OK;
}

GRAPHIN_RESULT SCAPI imagePaint(HIMG himg, image_paint_function *pPainter,
                                void *prm) // paint on image using graphics
{
  // gool::app()->create_bitmap_bits_graphics(
  if (!himg) return GRAPHIN_BAD_PARAM;
  if (!himg->is_bitmap()) return GRAPHIN_NOTSUPPORTED;

  gool::argb initc = gool::argb::undefined();

  handle<gool::bitmap>   bmp = static_cast<gool::bitmap *>(himg);
  handle<gool::graphics> bgr =
      gool::app()->create_bitmap_bits_graphics(bmp, initc);
  if (bgr) {
    gool::size dim = bmp->dim();
    pPainter(prm, bgr.ptr(), dim.x, dim.y);
    while (bgr->n_layers() > 0)
      bgr->pop_layer();
    return GRAPHIN_OK;
  }
  return GRAPHIN_FAILURE;
}

tool::value &value_of(VALUE *pval) { return *((tool::value *)pval); }

GRAPHIN_RESULT SCAPI vWrapGfx(HGFX hgfx, VALUE *toValue) {
  if (hgfx) {
    value_of(toValue).set_resource(hgfx);
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vWrapPath(HIMG himg, VALUE *toValue) {
  if (himg) {
    value_of(toValue).set_resource(himg);
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vWrapPath(HPATH hpath, VALUE *toValue) {
  if (hpath) {
    value_of(toValue).set_resource(hpath);
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vWrapText(HTEXT htext, VALUE *toValue) {
  if (htext) {
    value_of(toValue).set_resource(htext);
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

const tool::value &value_of(const VALUE *pval) {
  return *((const tool::value *)pval);
}

GRAPHIN_RESULT SCAPI vUnWrapGfx(const VALUE *fromValue, HGFX *phgfx) {
  if (phgfx) {
    *phgfx = value_of(fromValue).get_resource<gool::graphics>();
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vUnWrapPath(const VALUE *fromValue, HIMG *phimg) {
  if (phimg) {
    *phimg = value_of(fromValue).get_resource<gool::image>();
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vUnWrapPath(const VALUE *fromValue, HPATH *phpath) {
  if (phpath) {
    *phpath = value_of(fromValue).get_resource<gool::path>();
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

GRAPHIN_RESULT SCAPI vUnWrapText(const VALUE *fromValue, HTEXT *phtext) {
  if (phtext) {
    *phtext = value_of(fromValue).get_resource<gool::text_layout>();
    return GRAPHIN_OK;
  }
  return GRAPHIN_BAD_PARAM;
}

static SciterGraphicsAPI gfxapi = {
    imageCreate,
    imageCreateFromPixmap,
    imageAddRef,
    imageRelease,
    imageGetInfo,
    imageClear,
    imageLoad,
    imageSave,

    RGBA,

    gCreate,
    gAddRef,
    gRelease,
    gLine,
    gRectangle,
    gRoundedRectangle,
    gEllipse,
    gArc,
    gStar,
    gPolygon,
    gPolyline,

    pathCreate,
    pathAddRef,
    pathRelease,
    pathMoveTo,
    pathLineTo,
    pathArcTo,
    pathQuadraticCurveTo,
    pathBezierCurveTo,
    pathClosePath,

    gDrawPath,
    gRotate,
    gTranslate,
    gScale,
    gSkew,
    gTransform,

    gStateSave,
    gStateRestore,

    gLineWidth,
    gLineJoin,
    gLineCap,
    gLineColor,
    gFillColor,
    gLineGradientLinear,
    gFillGradientLinear,
    gLineGradientRadial,
    gFillGradientRadial,
    gFillMode,

    textCreateForElement,
    textCreateForElementAndStyle,

    textAddRef,
    textRelease,
    textGetMetrics,
    textSetBox,

    gDrawText,
    gDrawPath,

    gWorldToScreen,
    gScreenToWorld,

    gPushClipBox,
    gPushClipPath,
    gPopClip,

    imagePaint,

    vWrapGfx,
    vWrapPath,
    vWrapPath,
    vWrapText,

    vUnWrapGfx,
    vUnWrapPath,
    vUnWrapPath,
    vUnWrapText,

    gFlush,
};

SciterGraphicsAPI *SCAPI SciterGraphicsAPI_api() { return &gfxapi; }

#endif
