
#ifndef __XGL_WINDOW_BASE_H_INCLUDED__
#define __XGL_WINDOW_BASE_H_INCLUDED__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html/html.h"
#include "xgl-graphics.h"

#include "xgl/skia/include/core/SkSurfaceProps.h"
#include "xgl/skia/include/gpu/GrContext.h"
#include "xgl/skia/include/gpu/gl/GrGLInterface.h"

#if defined(WINDOWS)
  #define WIN32_LEAN_AND_MEAN
  #include <gl/GL.h>
  #include "xgl/skia/src/utils/win/SkWGL.h"
  #if !defined(WINDOWLESS)
    #include "win/win-layered.h"
  #endif 
#elif defined(OSX)
  #include <OpenGL/gl.h>
  #include <OpenGL/glu.h>
  #include <OpenGL/glext.h>
  #include <GLUT/glut.h>
  #include <OpenGL/gl.h>
  #include <OpenGL/CGLTypes.h>
#elif defined(ANDROID)
  #include <EGL/egl.h>
  #include <GLES/gl.h>
#elif defined(LINUX)
  #if !defined(WINDOWLESS)
    #include <gtk/gtk.h>
  #endif
  #include <GL/gl.h>
  #if !defined(EGL_API_FB)
    #include <GL/glx.h>
  #endif
#endif


struct GrGLInterface;

namespace xgl {

  using namespace tool;
  using namespace gool;
  using namespace html;

  struct AttachmentInfo {
      AttachmentInfo()
        : fSampleCount(0)
        , fStencilBits(0)
        , fColorBits(0) {}

        int fSampleCount;
        int fStencilBits;
        int fColorBits;
  };


  class BackendDevice : public resource
  {
public:
    BackendDevice(html::iwindow* pw)
      : fWindow(pw)
      , fSurfaceProps(SkSurfaceProps::kLegacyFontHost_InitType)
    {
      fBitmap.allocN32Pixels(0, 0);
      fIsTransparent = pw->get_transparency();
    }

    ~BackendDevice();

    virtual bool isRaster() const { return false; }
    virtual bool isGpu() const { return false; }
    virtual bool isReady() const { return false; }

    virtual bool attach() = 0;
    virtual bool detach() = 0;

    virtual void enable_vsync(bool enable) {}

    virtual bool    render(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) = 0;
    virtual bool    render_layered(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) { assert(false); return false; }

    // don't do preambula / postambula
    virtual bool    render_layer(element* pel,function<bool(xgl::graphics* gfx, element* pel)> painter, gool::rect paint_rc, void* hdc = nullptr) { assert(false); return false; }

    virtual void    start_drawing() {}
    virtual bool    update_request() { return false; }

    virtual int     width() const { return int(fWindow->client_dim().x * backing_scale()); }
    virtual int     height() const { return int(fWindow->client_dim().y * backing_scale()); }
    virtual void    invalidate(int left, int top, int width, int height) { refresh(fWindow, rect::make_xywh(left, top, width, height)); }
    virtual HWINDOW getHWnd() const { return fWindow->get_hwnd(); }

    SkSurfaceProps  getSurfaceProps() const { return fSurfaceProps; }
    void            setSurfaceProps(const SkSurfaceProps& props) { fSurfaceProps = props; }

    virtual void    setColorType(SkColorType ct) { resize(width(), height(), ct); }
    virtual void    on_dpi_change();
    virtual void    resize(int width, int height, SkColorType = kUnknown_SkColorType) {
        //SkImageInfo ninfo = fBitmap.info().makeWH(width,height);
        //if( ninfo != fBitmap.info() )
        //{
          //fBitmap.allocPixels(ninfo);
          invalidate(0,0,width,height);
        //}
    }

    virtual float   backing_scale() const;

    virtual void    refresh(iwindow* self, const rect& area);

    SkImageInfo     info() const { return fBitmap.info(); }

protected:
    SkBitmap        fBitmap;
    html::iwindow*  fWindow;
    SkSurfaceProps  fSurfaceProps;
    SkColorType     fColorType;
    bool            fIsTransparent;
  };

  class DeviceOpenGL : public BackendDevice
  {
    typedef BackendDevice super;

  public:
    DeviceOpenGL(html::iwindow* pw) : BackendDevice(pw)
#if SK_SUPPORT_GPU
  #if defined(WINDOWLESS)
    // nothing
  #elif defined(WINDOWS)
    , fHGLRC(nullptr)
  #elif defined(OSX)
    , fGLContext(nullptr)
  #elif defined(LINUX)
    , fGLContext(nullptr)
  #endif
    , fRenderTarget(nullptr)
    , fInterface(nullptr)
    , fContext(nullptr)
#endif
    {
    }

    ~DeviceOpenGL();

    virtual bool isGpu() const override { return true; }
    virtual bool isReady() const override { return fContext != nullptr; }


    virtual bool attach() override;
    virtual bool detach() override;
            bool present();
    virtual bool render(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) override;
    virtual bool render_layered(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) override;
    virtual bool render_layer(element* ell, function<bool(xgl::graphics* gfx,element* pel)> painter, gool::rect paint_rc, void* hdc = nullptr) override;

    virtual void start_drawing() override;
    virtual bool update_request() override;

    virtual void enable_vsync(bool enable) override;

    SkSurface*  createSurface();

    virtual void resize(int width, int height, SkColorType = kUnknown_SkColorType) override;
    virtual void refresh(iwindow* self, const rect& area) override;

  protected:
    bool            init();
    void            setUpRenderTarget();

    AttachmentInfo      fAttachmentInfo;
    SkSurface*      fSurface;
    GrRenderTarget*          fRenderTarget;
    const GrGLInterface*     fInterface;
    GrContext*               fContext;
  #if defined(WINDOWLESS)
    // none
  #elif defined(WINDOWS)
    void*           fHGLRC;
  #elif defined(OSX)
    void*           fGLContext;
  #elif defined(LINUX)
    GdkGLContext*   fGLContext;
    GdkWindow *     fXWindow;
  #endif

  };

  class DeviceRaster : public BackendDevice
  {
    typedef BackendDevice super;
  public:

    DeviceRaster(html::iwindow* pw)
      : BackendDevice(pw) {
          
      }

    ~DeviceRaster();

    virtual bool isRaster() const override { return true; }
    virtual bool isReady() const override { return true; }

    virtual bool attach() override { return true; }
    virtual bool detach() override { fBitmap.reset(); return true; }
            bool present(rect paint_rc, void* hdc);

    virtual bool render(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) override;
    virtual bool render_layered(function<bool(xgl::graphics* gfx)> painter, gool::rect paint_rc, void* hdc = nullptr) override;

    virtual void resize(int width, int height, SkColorType = kUnknown_SkColorType) override;
  };


}

#endif // __XGL_WINDOW_BASE_H_INCLUDED__
