
#include "xgl.h"
//#include "xgl-view.h"

#include "SkRRect.h"

#include "xgl/skia/include/core/SkBitmap.h"
#include "xgl/skia/include/core/SkRegion.h"
#include "xgl/skia/include/effects/SkDashPathEffect.h"

namespace xgl {

  bool path::is_inside(pointf pt)
  {
     rect rc = bounds();
     SkRegion reg, clip;
     clip.setRect(cvt(rc));
     reg.setPath(_path,clip);
     return reg.contains(int(pt.x),int(pt.y));
  }

  void path::start(pointf pt, bool filled)
  {
    bool t = even_odd();
    _path.reset();
    set_even_odd(t);
    _path.moveTo(cvt(pt));
  }
  void path::move_to(pointf pt, bool rel)
  {
    rel ? _path.rMoveTo(pt.x,pt.y) : _path.moveTo(cvt(pt));
  }
  void path::line_to(pointf pt, bool rel)
  {
    rel ? _path.rLineTo(pt.x, pt.y) : _path.lineTo(cvt(pt));
  }
  void path::hline_to(float x, bool rel)
  {
    SkPoint last;
    _path.getLastPt(&last);
    line_to(pointf(x,rel?0:lastp().y),rel);
  }

  void path::vline_to(float y, bool rel)
  {
    SkPoint last;
    _path.getLastPt(&last);
    line_to(pointf(rel?0:lastp().x,y),rel);
  }

  void arcTo(SkPath& path, SkScalar rx, SkScalar ry, SkScalar angle, bool arcLarge,
    bool arcSweepCCW, SkScalar x, SkScalar y) {
    SkPoint srcPts[2];
    path.getLastPt(&srcPts[0]);
    // If rx = 0 or ry = 0 then this arc is treated as a straight line segment (a "lineto")
    // joining the endpoints.
    // http://www.w3.org/TR/SVG/implnote.html#ArcOutOfRangeParameters
    if (!rx || !ry) {
      return;
    }
    // If the current point and target point for the arc are identical, it should be treated as a
    // zero length path. This ensures continuity in animations.
    srcPts[1].set(x, y);
    if (srcPts[0] == srcPts[1]) {
      return;
    }
    rx = SkScalarAbs(rx);
    ry = SkScalarAbs(ry);
    SkVector midPointDistance = srcPts[0] - srcPts[1];
    midPointDistance.scale(0.5f);
    SkMatrix pointTransform;
    pointTransform.setRotate(-angle);
    SkPoint transformedMidPoint;
    pointTransform.mapPoints(&transformedMidPoint, &midPointDistance, 1);
    SkScalar squareRx = rx * rx;
    SkScalar squareRy = ry * ry;
    SkScalar squareX = transformedMidPoint.fX * transformedMidPoint.fX;
    SkScalar squareY = transformedMidPoint.fY * transformedMidPoint.fY;
    // Check if the radii are big enough to draw the arc, scale radii if not.
    // http://www.w3.org/TR/SVG/implnote.html#ArcCorrectionOutOfRangeRadii
    SkScalar radiiScale = squareX / squareRx + squareY / squareRy;
    if (radiiScale > 1) {
      radiiScale = SkScalarSqrt(radiiScale);
      rx *= radiiScale;
      ry *= radiiScale;
    }
    pointTransform.setScale(1 / rx, 1 / ry);
    pointTransform.preRotate(-angle);
    SkPoint unitPts[2];
    pointTransform.mapPoints(unitPts, srcPts, (int)SK_ARRAY_COUNT(unitPts));
    SkVector delta = unitPts[1] - unitPts[0];
    SkScalar d = delta.fX * delta.fX + delta.fY * delta.fY;
    SkScalar scaleFactorSquared = SkTMax(1 / d - 0.25f, 0.f);
    SkScalar scaleFactor = SkScalarSqrt(scaleFactorSquared);
    if (arcSweepCCW != arcLarge) {  // flipped from the original implementation
      scaleFactor = -scaleFactor;
    }
    delta.scale(scaleFactor);
    SkPoint centerPoint = unitPts[0] + unitPts[1];
    centerPoint.scale(0.5f);
    centerPoint.offset(-delta.fY, delta.fX);
    unitPts[0] -= centerPoint;
    unitPts[1] -= centerPoint;
    SkScalar theta1 = SkScalarATan2(unitPts[0].fY, unitPts[0].fX);
    SkScalar theta2 = SkScalarATan2(unitPts[1].fY, unitPts[1].fX);
    SkScalar thetaArc = theta2 - theta1;
    if (thetaArc < 0 && !arcSweepCCW) {  // arcSweep flipped from the original implementation
      thetaArc += SK_ScalarPI * 2;
    }
    else if (thetaArc > 0 && arcSweepCCW) {  // arcSweep flipped from the original implementation
      thetaArc -= SK_ScalarPI * 2;
    }
    pointTransform.setRotate(angle);
    pointTransform.preScale(rx, ry);
    int segments = SkScalarCeilToInt(SkScalarAbs(thetaArc / (SK_ScalarPI / 2)));
    SkScalar thetaWidth = thetaArc / segments;
    SkScalar t = SkScalarTan(0.5f * thetaWidth);
    if (!SkScalarIsFinite(t)) {
      return;
    }
    SkScalar startTheta = theta1;
    SkScalar w = SkScalarSqrt(SK_ScalarHalf + SkScalarCos(thetaWidth) * SK_ScalarHalf);
    for (int i = 0; i < segments; ++i) {
      SkScalar endTheta = startTheta + thetaWidth;
      SkScalar cosEndTheta, sinEndTheta = SkScalarSinCos(endTheta, &cosEndTheta);
      unitPts[1].set(cosEndTheta, sinEndTheta);
      unitPts[1] += centerPoint;
      unitPts[0] = unitPts[1];
      unitPts[0].offset(t * sinEndTheta, -t * cosEndTheta);
      SkPoint mapped[2];
      pointTransform.mapPoints(mapped, unitPts, (int)SK_ARRAY_COUNT(unitPts));
      path.conicTo(mapped[0], mapped[1], w);
      startTheta = endTheta;
    }
  }

  void path::arc_to(pointf to, sizef r, float rotation_angle, bool large_arc, bool clockwise, bool rel_to)
  {
    if (rel_to) {
      SkPoint currentPoint;
      _path.getLastPt(&currentPoint);
      arcTo(_path, r.x, r.y, rotation_angle, large_arc, !clockwise, currentPoint.x() + to.x, currentPoint.y() + to.y);
    }
    else 
      arcTo(_path,r.x,r.y,rotation_angle,large_arc,!clockwise,to.x,to.y);
  } 

  void path::add_arc(pointf c, sizef r, float angle_start, float angle_sweep)
  {
    SkRect oval = SkRect::MakeLTRB(c.x - r.x, c.y - r.y, c.x + r.x, c.y + r.y);
    _path.arcTo(oval, SkRadiansToDegrees(angle_start), SkRadiansToDegrees(angle_sweep),false);
    //_path.addArc(cvt(box), SkRadiansToDegrees(angle_start), SkRadiansToDegrees(angle_sweep));
  }
  void path::quadratic_to(pointf pt, pointf cp, bool rel)
  {
    if (rel)
      _path.rQuadTo(cp.x, cp.y, pt.x, pt.y);
    else
      _path.quadTo(cp.x, cp.y, pt.x, pt.y);
  }
  void path::cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel) {
    if (rel)
      _path.rCubicTo(cp1.x, cp1.y, cp2.x, cp2.y, pt.x, pt.y);
    else
      _path.cubicTo(cp1.x, cp1.y, cp2.x, cp2.y, pt.x, pt.y);
  }

  void path::close()
  {
    _path.close();
  }

  gool::path* path::combine(gool::path::COMBINE_MODE mode, const gool::path* other)
  {
    SkRegion rgn_clip(SkIRect::MakeLargest());
    SkRegion rgn; rgn.setPath(_path,rgn_clip);
    SkRegion rgn_other; rgn_other.setPath(static_cast<const xgl::path*>(other)->_path, rgn_clip);
       
    SkRegion::Op op;
     
    switch (mode) {
      default:
      case COMBINE_MODE_UNION:      op = SkRegion::kUnion_Op; break;
      case COMBINE_MODE_INTERSECT:  op = SkRegion::kIntersect_Op; break;
      case COMBINE_MODE_XOR:        op = SkRegion::kXOR_Op; break;
      case COMBINE_MODE_EXCLUDE:    op = SkRegion::kDifference_Op; break;
    }

    rgn.op(rgn_other, op);
    path* out = new path();
    rgn.getBoundaryPath(&out->_path);
    return out;
  }
    
  gool::application* graphics::app() const {
    return xgl::app_factory();
  }

  SkBitmap& sk_bitmap(gool::bitmap* pbm) {
    if (pbm->sk_bitmap.isNull() || pbm->_used_generation != pbm->_generation) {
      gool::size sz = pbm->dim();
        
      
      pbm->sk_bitmap.reset();
      //SkImageInfo info = SkImageInfo::Make(sz.x, sz.y, kBGRA_8888_SkColorType, kPremul_SkAlphaType);
      SkImageInfo info = SkImageInfo::MakeN32Premul(sz.x, sz.y);
      pbm->sk_bitmap.setInfo(info);
      pbm->sk_bitmap.allocPixels();
        
      if (pbm->sk_bitmap.lockPixelsAreWritable()) {

        if (pbm->sk_bitmap.colorType() == kRGBA_8888_SkColorType) {
          BGR32_space_converter cvt(pbm->dim());
          tool::slice<gool::argb> pixels = pbm->get_bits();
          cvt.convert_from_rgb32(pixels,(byte*)pbm->sk_bitmap.getPixels());
        }
        else {
          tool::tslice<gool::argb> target((gool::argb*) pbm->sk_bitmap.getPixels(), sz.x * sz.y);
          target.copy( pbm->get_bits() );
        }
        pbm->sk_bitmap.unlockPixels();
        pbm->_generation = pbm->_used_generation = pbm->sk_bitmap.getGenerationID();
      }
    }
    return pbm->sk_bitmap;
  }

  void graphics::fill( argb c, const rect& dst) {

    SkPaint paint;
    paint.setColor(cvt(c));
    canvas()->drawIRect(cvt(dst), paint);
  }

  void graphics::fill( argb c, const rect& dst, const size& radii)
  {
    SkRRect rr;
    rr.setRectXY(cvtf(dst), SkScalar(radii.x), SkScalar(radii.y));
    SkPaint paint;
    paint.setColor(cvt(c));
    canvas()->drawRRect(rr, paint);
  }

  void brush::fill(graphics* gfx)
  {
    //cairo_set_source(gfx->target(), pat );
    //cairo_fill(gfx->target());

  }

  void brush::stroke(graphics* gfx)
  {
    //cairo_set_source(gfx->target(), pat );
    //cairo_stroke(gfx->target());
  }

  void            graphics::image_rendering_mode(IMAGE_RENDERING irm)
  {
    switch (irm) {
      case image_rendering_default: this->_registers.state.setFilterQuality(kLow_SkFilterQuality); break;
      case image_rendering_speed:
      case image_rendering_undefined:  
            this->_registers.state.setFilterQuality(kNone_SkFilterQuality); break;
      case image_rendering_quality: this->_registers.state.setFilterQuality(kMedium_SkFilterQuality); break;
    }
  }
  IMAGE_RENDERING graphics::image_rendering_mode() const
  {
    SkFilterQuality fq = this->_registers.state.getFilterQuality();
    switch (fq) {
      default:
      case kLow_SkFilterQuality:  return image_rendering_default;
      case kNone_SkFilterQuality: return image_rendering_speed;
      case kMedium_SkFilterQuality: return image_rendering_quality;
    }
    return image_rendering_default;
  }

  void graphics::fill( const gool::brush* lb, const rect& dst)
  {

    SkPaint paint;
    //paint.setAntiAlias(true);
    paint.setStyle(SkPaint::kFill_Style);

    switch (lb->type())
    {
    case gool::brush::SOLID:
    {
      solid_brush br(*static_cast<const gool::solid_brush*>(lb));
      paint.setShader(br.shader());
      break;
    }
    case gool::brush::LINEAR:
    {
      linear_gradient_brush br(*static_cast<const gool::linear_brush*>(lb));
      paint.setShader(br.shader());
      break;
    }
    case gool::brush::RADIAL:
    {
      radial_gradient_brush br(*static_cast<const gool::radial_brush*>(lb));
      paint.setShader(br.shader());
      break;
    }
    default:
      return;
    }

    canvas()->drawIRect(cvt(dst), paint);
  }

  void graphics::fill( image* img, const gool::path* dst)
  {
    handle<bitmap> bmp = img->get_bitmap(this,img->dim());
    if (!bmp) return;

    size src_img_size = img->dim();
    //size src_size = src.size();

    rectf dst_rect = dst->bounds();

    //SkBitmap bitmap_surface;
    //sk_bitmap(bmp).extractSubset(&bitmap_surface, cvt(src));

    //SkMatrix sm = SkMatrix::(float(offset.x), float(offset.y));
    //if (!celldim.empty() && src_size != celldim)
    //  sm.postScale(float(celldim.x) / src_size.x, float(celldim.y) / src_size.y);

    ref<SkShader> shader = SkShader::CreateBitmapShader(sk_bitmap(bmp) , SkShader::kRepeat_TileMode, SkShader::kRepeat_TileMode);

    SkPaint paint = _registers.state;
    paint.setShader(shader);

    SkAutoCanvasRestore acr(canvas(), true);

    xgl::path* p = (xgl::path*)dst;

    // investiage why it works strangely, I suspect 
    canvas()->save();
    canvas()->clipPath(p->sk_path()/*,SkRegion::kDifference_Op*/);
    canvas()->translate(float(dst_rect.s.x), float(dst_rect.s.y));
    canvas()->drawPaint(paint);
    canvas()->restore();
  }

  void graphics::do_push_clip(const rect& r)
  {
    canvas()->save();
    canvas()->clipRect(cvtf(r));
  }
  void graphics::do_pop_clip()
  {
    canvas()->restore();
  }

  void graphics::draw( const gool::path* dst, argb c, float width ) {
     xgl::path* p = (xgl::path*)dst;

     SkPaint paint;
     paint.setColor(cvt(c));
     paint.setStyle(SkPaint::kStroke_Style);
     paint.setAntiAlias(true);
     paint.setStrokeWidth(width);
     canvas()->drawPath(p->sk_path(), paint);
  }
  void graphics::fill( argb c, const gool::path* dst)
  {
     xgl::path* p = (xgl::path*)dst;
     SkPaint paint;
     paint.setColor(cvt(c));
     paint.setAntiAlias(true);
     paint.setStyle(SkPaint::Style::kFill_Style);
     canvas()->drawPath(p->sk_path(), paint);

  }

  void graphics::draw_line_v(const rect& rc, argb c, int stroke, int step, int_v voff)
  {
    if(c.alfa == 0)
        return;
    int s1 = stroke;

    SkPaint paint;
    paint.setColor(cvt(c));
    paint.setStyle(SkPaint::kStroke_Style);
    paint.setAntiAlias(true);

    if (rc.height() == 1)
      paint.setStrokeCap(SkPaint::Cap::kSquare_Cap);
    else
      paint.setStrokeCap(SkPaint::Cap::kRound_Cap);

    float w = float(rc.width());
    float w2 = w / 2.0f;

    pointf p1 = { rc.left() + w2,float(rc.top() + w2) };
    pointf p2 = { rc.left() + w2,float(rc.bottom() - w2) };

    paint.setStrokeWidth(w);

    float off = rc.height() / step / 2.0f + voff;

    float dashes[] = { max(0.0f,stroke - w), step - stroke + w };
    ref<SkPathEffect> eff = SkDashPathEffect::Create(dashes, 2, off);
    paint.setPathEffect(eff);

    canvas()->drawLine(p1.x, p1.y, p2.x, p2.y, paint);
  }
  void graphics::draw_line_h(const rect& rc, argb c, int stroke, int step, int_v voff)
  {
    if(c.alfa == 0)
        return;
    int s1 = stroke;

    SkPaint paint;
    paint.setColor(cvt(c));
    paint.setStyle(SkPaint::kStroke_Style);
    paint.setAntiAlias(true);

    if (rc.height() == 1)
      paint.setStrokeCap(SkPaint::Cap::kSquare_Cap);
    else
      paint.setStrokeCap(SkPaint::Cap::kRound_Cap);

    float w = float(rc.height());
    float w2 = w / 2.0f;

    paint.setStrokeWidth(w);

    float off = rc.width() / step / 2.0f + voff.val(0);

    pointf p1 = { float(rc.left()) + w2,rc.top() + w2 };
    pointf p2 = { float(rc.right()) - w2,rc.top() + w2 };

    float dashes[] = { max(0.0f,stroke - w), step - stroke + w };
    ref<SkPathEffect> eff = SkDashPathEffect::Create(dashes, 2, off);
    paint.setPathEffect(eff);

    canvas()->drawLine(p1.x, p1.y, p2.x, p2.y, paint);
  }

  void graphics::draw_line(pointf p1, pointf p2 , argb c, int width)
  {
    SkPaint paint;
    paint.setColor(cvt(c));
    canvas()->drawLine(p1.x, p1.y, p2.x, p2.y, paint);
  }

  point graphics::offset(point noff)
  {
    point off = _offset;
    _offset = noff;
    _clip_rc -= noff;

    canvas()->translate(SkScalar(noff.x), SkScalar(noff.y));
    return off;
  }

  bool graphics::set_antialiasing(bool onoff) {
    _registers.state.setAntiAlias(onoff);
    return true;
  }
   
    uint graphics::push_state()
  {
     canvas()->save();
     uint lv = uint(_registers_stack.length());
     _registers_stack.push(_registers);
     return lv;
  }

  bool graphics::pop_state()
  {
     if( !_registers_stack.size() )
       return false;
     _registers = _registers_stack.pop();
     canvas()->restore();
     return true;
  }

  void graphics::restore_state(uint level)
  {
     while( _registers_stack.length() && _registers_stack.length() > level )
       pop_state();
  }

  /*const text_format& graphics::get_text_format()
  {
    static text_format ss; return ss;
  }
  void  graphics::set_text_format(const text_format& tf)
  {
  }*/

  SkPaint get_filtered_paint(function<bool(gool::filter_graph_builder*)> filters);

  void graphics::push_layer(const gool::rect& clip, byte opacity, function<bool(gool::filter_graph_builder*)> filters)
  {
    auto rc = cvtf(clip);
    if (opacity < 255) // semitransparent layer
      canvas()->saveLayerAlpha(&rc, opacity);
    else if (filters) {
      SkPaint paint = get_filtered_paint(filters);
      canvas()->saveLayer(&rc, &paint);
    } else
      canvas()->saveLayer(&rc,nullptr);
    canvas()->clipRect(cvtf(clip), SkRegion::kIntersect_Op, true);

    _layers_stack.push(layer_registers());
  }
  void graphics::push_layer(const gool::path* clip, byte opacity)
  {
    xgl::path* p = (xgl::path*)clip;

    SkRect rc = p->sk_path().getBounds();

    if (opacity < 255) // semitransparent layer
      canvas()->saveLayerAlpha(&rc, opacity);
    else
      canvas()->saveLayer(&rc, nullptr);

    canvas()->clipPath(p->sk_path(),SkRegion::kIntersect_Op, true);
    _layers_stack.push(layer_registers());
  }

  void graphics::push_layer(const gool::bitmap* clip, bool draw1a, byte opacity)
  {
    //xgl::path* p = (xgl::path*)clip;

    SkRect rc = cvtf(gool::rect(clip->dim()));

    if (opacity < 255) // semitransparent layer
      canvas()->saveLayerAlpha(&rc, opacity);
    else
      canvas()->saveLayer(&rc, nullptr);

    //canvas()->clipPath(p->sk_path(), SkRegion::kIntersect_Op, true);
    layer_registers lr; lr.mask = clip; lr.draw1a = draw1a;
    _layers_stack.push(lr);
  }

  void graphics::pop_layer()
  {
    assert(_layers_stack.size());
    layer_registers lr = _layers_stack.pop();
    if (lr.mask) {
      SkPaint paint = _registers.state;
      paint.setXfermodeMode(lr.draw1a ? SkXfermode::kDstIn_Mode : SkXfermode::kDstOut_Mode);
      canvas()->drawBitmap(sk_bitmap(lr.mask), 0, 0, &paint/*, SkCanvas::kFast_SrcRectConstraint*/);
    }
    canvas()->restore();
  }

  uint graphics::n_layers() {
    return uint(_layers_stack.size());
  }

  CAP_STYLE graphics::cap_style() {
    
    switch (this->_registers.state.getStrokeCap()) {
       case SkPaint::kButt_Cap:   return CAP_BUTT;
       case SkPaint::kRound_Cap:  return CAP_ROUND;
       case SkPaint::kSquare_Cap: return CAP_SQUARE;
     }
     return CAP_BUTT;
  }
  void graphics::cap_style(CAP_STYLE ncs) {
      switch(ncs) {
        default:
        case CAP_BUTT:  this->_registers.state.setStrokeCap(SkPaint::kButt_Cap); break;
        case CAP_ROUND: this->_registers.state.setStrokeCap(SkPaint::kRound_Cap); break;
        case CAP_SQUARE: this->_registers.state.setStrokeCap(SkPaint::kSquare_Cap); break;
      }
  }
  LINE_JOIN graphics::line_join() {
      switch (this->_registers.state.getStrokeJoin()) 
      {
        case SkPaint::kMiter_Join: return JOIN_MITER;
        case SkPaint::kRound_Join: return JOIN_ROUND;
        case SkPaint::kBevel_Join: return JOIN_BEVEL;
      }
      return JOIN_MITER;
  }
  void graphics::line_join(LINE_JOIN ncs, float mitter_limit) {
      switch(ncs) {
        default:
        case JOIN_MITER: this->_registers.state.setStrokeJoin(SkPaint::kMiter_Join); this->_registers.state.setStrokeMiter(mitter_limit);  break;
        case JOIN_ROUND: this->_registers.state.setStrokeJoin(SkPaint::kRound_Join); break;
        case JOIN_BEVEL: this->_registers.state.setStrokeJoin(SkPaint::kBevel_Join); break;
      }
  }
  DASH_STYLE graphics::dash_style() {
    //int dash_count = cairo_get_dash_count(_target);
    //if(dash_count == 0 )
      return DASH_STYLE_SOLID;
    //if( dash_count != 2 )
    //  return DASH_STYLE_SOLID;
    //double dashes[2];
    //double offset;
    //cairo_get_dash(_target,dashes,&offset);
    //if( dashes[0] == 1.0 && dashes[1] == 1.0)
    //  return DASH_STYLE_DOT;
    //return DASH_STYLE_DASH;
  }
  void graphics::dash_style( DASH_STYLE st ) {
    switch(st) {
      case DASH_STYLE_SOLID: this->_registers.state.setPathEffect(nullptr); break;
      case DASH_STYLE_DOT: {
        float w = this->_registers.state.getStrokeWidth();
        float dashes[] = {1.0f * w,1.0f * w};
        ref<SkPathEffect> eff = SkDashPathEffect::Create(dashes, 2, 0);
        this->_registers.state.setPathEffect(eff);
        //cairo_set_dash(_target,dashes,2,0);
        break;
      }
      case DASH_STYLE_DASH: {
        float w = this->_registers.state.getStrokeWidth();
        float dashes[] = {2.0f * w,2.0f * w};
        ref<SkPathEffect> eff = SkDashPathEffect::Create(dashes, 2, 0);
        this->_registers.state.setPathEffect(eff);
        break;
      }
    }
  }

  void graphics::custom_dash_style(slice<float> pattern, float offset)
  {
    float w = this->_registers.state.getStrokeWidth();
    if (w > 0.0f) {
      array<float> dashes = pattern;//{ 2.0f * w,2.0f * w };
      //for (auto& d : dashes) { d *= w; }
      ref<SkPathEffect> eff = SkDashPathEffect::Create(dashes.cbegin(), dashes.size(), offset);
      this->_registers.state.setPathEffect(eff);
    }

  }

  void graphics::set_stroke( argb c)
  {
     if( c == argb::no_color() )
       _registers.stroke_brush = nullptr;
     else
       _registers.stroke_brush = new solid_brush(c);
  }
  void graphics::set_stroke( const gool::linear_brush& lb )
  {
    _registers.stroke_brush = new linear_gradient_brush(lb);
  }
  void graphics::set_stroke( const radial_brush& rb )
  {
    _registers.stroke_brush = new radial_gradient_brush(rb);
  }
  void graphics::set_stroke_width( float w )
  {
    this->_registers.state.setStrokeWidth(w);
  }

  void graphics::set_fill( argb c )
  {
     if( c == argb::no_color() )
       _registers.fill_brush = nullptr;
     else
       _registers.fill_brush = new solid_brush(c);
  }
  void graphics::set_fill( const linear_brush& lb )
  {
    _registers.fill_brush = new linear_gradient_brush(lb);
  }

  void graphics::set_fill( const radial_brush& rb )
  {
    _registers.fill_brush = new radial_gradient_brush(rb);
  }
  void graphics::set_fill( const image* ib )
  {
    _registers.fill_brush = new image_brush(ib);
  }
  void graphics::set_fill_even_odd( bool even_odd )
  {
    _registers.even_odd = even_odd;
  }

  void gradient_brush::get_stops(const gool::gradient_brush& bin, array<SkColor>& colors, array<SkScalar>& positions)
  {
     slice<gool::color_stop>   color_stops = bin.stops();
     colors.size(color_stops.size());
     positions.size(color_stops.size());
     for (uint i = 0; i < color_stops.length; ++i) {
       auto c = color_stops[i].color;
       colors[i] = cvt(c);
       positions[i] = color_stops[i].position;
     }
  }

  void graphics::render_figure(bool fill, bool stroke, function<void(SkCanvas* canvas, const SkPaint& paint)> worker)
  {
    if (fill && _registers.fill_drawable()) {
      SkPaint paint = _registers.state;
      paint.setStyle(SkPaint::kFill_Style);
      paint.setShader(_registers.fill_brush->_shader);
      worker(canvas(), paint);
    }
    if (stroke && _registers.stroke_drawable()) {
      SkPaint paint = _registers.state;
      paint.setStyle(SkPaint::kStroke_Style);
      paint.setShader(_registers.stroke_brush->_shader);
      worker(canvas(), paint);
    }
  }


  void graphics::draw_ellipse(pointf center, sizef radius, bool stroke, bool fill)
  {
    SkRect rc = { center.x - radius.x, center.y - radius.y, center.x + radius.x, center.y + radius.y };
    render_figure(fill, stroke, [rc](SkCanvas* canvas, const SkPaint& paint){
      canvas->drawOval(rc, paint);
    });
  }
  
  /*void graphics::draw_arc(pointf center, sizef radius, float angle_start, float angle_sweep)
  {
    SkPath path;
    SkRect rect = { center.x - radius.x, center.y - radius.y, center.x + radius.x, center.y + radius.y };
        
    if (_registers.fill_drawable()) {
      path.moveTo(cvt(center));
      path.addArc(rect, SkRadiansToDegrees(angle_start), SkRadiansToDegrees(angle_sweep));
      path.lineTo(cvt(center));
      path.close();
    } else 
      path.addArc(rect, SkRadiansToDegrees(angle_start), SkRadiansToDegrees(angle_sweep));

    render_figure(true, true, [&](SkCanvas* canvas, const SkPaint& paint) 
    { 
      canvas->drawPath(path, paint);
    });
  } */

  void graphics::draw_rectangle(pointf org, sizef dim, bool stroke, bool fill)
  {
    rectf rc(org, dim);
    render_figure(fill, stroke, [rc](SkCanvas* canvas, const SkPaint& paint){
      canvas->drawRect(cvt(rc), paint);
    });
  }

  void graphics::draw_path( const gool::path* dst, bool stroke, bool fill)
  {
    render_figure(fill, stroke, [dst](SkCanvas* canvas, const SkPaint& paint){
      canvas->drawPath(static_cast<const xgl::path*>(dst)->sk_path(), paint);
    });
  }
  void graphics::draw_path( const gool::polygonf& p, bool stroke, bool fill)
  {
     path pa; pa.set(p);
     this->draw_path(&pa,stroke,fill);
  }

  void graphics::draw_line(pointf start, pointf end) {
    render_figure(false, true, [start,end](SkCanvas* canvas, const SkPaint& paint){
      canvas->drawLine(start.x, start.y, end.x, end.y, paint);
    });
  }

  void graphics::translate(pointf pt)
  {
    this->canvas()->translate(pt.x, pt.y);
  }
  void graphics::rotate(float radians, pointf center)
  {
    //this->canvas()->rotate(>translate(pt.x, pt.y);
    if(center.x != 0 || center.y != 0) {
      this->canvas()->translate(center.x, center.y);
      this->canvas()->rotate(SkRadiansToDegrees(radians));
      this->canvas()->translate(-center.x, -center.y);
    } else {
      this->canvas()->rotate(radians * 57.2957795f);
    }
  }

  void graphics::scale(sizef sz, pointf center)
  {
    if (center.x != 0 || center.y != 0) {
      this->canvas()->translate(center.x, center.y);
      this->canvas()->scale(sz.x,sz.y);
      this->canvas()->translate(-center.x, -center.y);
    }
    else {
      this->canvas()->scale(sz.x, sz.y);
    }
  }
  void graphics::skew(sizef sz, pointf center)
  {

    if (center.x != 0 || center.y != 0) {
      this->canvas()->translate(center.x, center.y);
      this->canvas()->skew(sz.x, sz.y);
      this->canvas()->translate(-center.x, -center.y);
    }
    else {
      this->canvas()->skew(sz.x, sz.y);
    }
  }
  void graphics::transform(const affine_mtx_f& m)
  {
    SkMatrix mtx;
    SkScalar affine[6] = { m.sx, m.shy, m.shx, m.sy, m.tx, m.ty };
    mtx.setAffine(affine);
    this->canvas()->concat(mtx);
  }

  void graphics::reset_transform() {
    this->canvas()->setMatrix(SkMatrix::I());
  }

  affine_mtx_f graphics::transform() const
  {
    SkMatrix mtx = this->canvas()->getTotalMatrix();
    SkScalar affine[6];
    mtx.asAffine(affine);
    affine_mtx_f m;
    m.sx  = affine[0];
    m.shy = affine[1];
    m.shx = affine[2];
    m.sy = affine[3]; 
    m.tx = affine[4];
    m.ty = affine[5];
    return m;
  }

  pointf graphics::world_to_screen(pointf pt) const
  {
    auto r = this->canvas()->getTotalMatrix().mapXY(pt.x, pt.y);
    return cvt(r);
  }

  pointf graphics::screen_to_world(pointf pt) const
  {
    SkMatrix imtx;
    this->canvas()->getTotalMatrix().invert(&imtx);
    auto r = imtx.mapXY(pt.x, pt.y);
    return cvt(r);
  }

  void graphics::do_draw( image* img, const rectf&  dst, const rect& src, byte opacity)
  {
    handle<gool::bitmap> bmp = img->get_bitmap(this,src.size());
	  if (!bmp) return;

    size src_img_size = img->dim();
    size src_size = src.size();
    
    SkPaint paint = _registers.state;

    paint.setAlpha(opacity);

    SkIRect isrc = cvt(src);
    
    this->canvas()->drawBitmapRect(sk_bitmap(bmp), &isrc, cvt(dst), &paint/*, SkCanvas::kFast_SrcRectConstraint*/);
  }
  
  void graphics::do_expand(image* img, const rect& dst, const SECTION_DEFS& sds, rect area)
  {
    auto siq = _registers.state.getFilterQuality();
    _registers.state.setFilterQuality(kNone_SkFilterQuality); // to avoid border effect
    super::do_expand(img, dst, sds, area);
    _registers.state.setFilterQuality(siq);
  }

  void graphics::fill( image* img, const rect& dst, const rect& src, point offset, size celldim)
  {
    handle<gool::bitmap> bmp = img->get_bitmap(this, src.size());
	  if (!bmp) return;

    size src_img_size = img->dim();
    size src_size = src.size();
    
    SkBitmap bitmap_surface;
    sk_bitmap(bmp).extractSubset(&bitmap_surface, cvt(src));
    
    SkMatrix sm = SkMatrix::MakeTrans(float(offset.x),float(offset.y));
    if (!celldim.empty() && src_size != celldim)
      sm.postScale(float(celldim.x) / src_size.x, float(celldim.y) / src_size.y);

    ref<SkShader> shader = SkShader::CreateBitmapShader(bitmap_surface, SkShader::kRepeat_TileMode, SkShader::kRepeat_TileMode, &sm);

    SkPaint paint = _registers.state;
    paint.setShader(shader);

    SkAutoCanvasRestore acr(canvas(), true);
   
    canvas()->clipRect(cvtf(dst));
    canvas()->translate(float(dst.s.x), float(dst.s.y));
    canvas()->drawPaint(paint);

  }


}
