// packfolder.cpp : Defines the entry point for the console application.
//

#include "tool/tool.h"

#ifdef WIN32

  //#include "targetver.h"

  #include <stdio.h>
  #include <tchar.h>

#ifndef MONOLITHIC

  #pragma comment( lib, "tool" )
  #pragma comment( lib, "uv" )
  #pragma comment( lib, "ws2_32" )
  #pragma comment(lib,  "iphlpapi.lib")
  #pragma comment(lib,  "psapi.lib")
  #pragma comment(lib,  "userenv.lib")

#endif // !MONOLITIC

#endif 

#include "tool/tool.h"
#include "sdk-headers.h"


static void usage(void)
{
    fprintf(stderr,"usage: packfolder.exe folder outfile [options]\n\n"
            "[-i \"foo/*;...\"]          include files or folders, if defined only matching items are included\n"
            "[-x \"*.ex1;*.ex2;...\"]    exclude files or folders, if defined matching items are excluded from archive\n\n"
            "[-v \"varname\"]            name of blob variable\n"
            "[-csharp]                 generates C# class with a byte[] field literal\n"
            "[-dlang]                  generates D ubyte[] literal\n"
            "[-go]                     generates Go []byte literal\n"
            "[-binary]                 generates binary file\n"
    );
    exit(1);
}

struct packed {
  tool::array<byte> data;
  void printf( const char* format, ... ) 
  {
    char buffer [ 2049 ];
    va_list args;
    va_start ( args, format );
#ifdef WINDOWS
    int n = _vsnprintf_s( buffer, 2048, format, args );
#else
    int n = vsnprintf( buffer, 2048, format, args );
#endif
    va_end ( args );
    buffer [ 2048 ] = 0;
    data.push( (const byte*) buffer, n );
  }
};

#ifdef WINDOWS
typedef tool::wchars tchars;
#define TCHARS WCHARS
#define TCVT(WS) WS

int _tmain(int argc, _TCHAR* argv[])
#else
typedef tool::chars tchars;
#define TCHARS CHARS
#define TCVT(WS) tool::ustring(WS)()

int main(int argc, const char * argv[])
#endif
{
  if( argc < 3 )
    usage();

  tool::ustring fname = argv[2];

  tool::string varname = "resources";
  tool::ustring exclusions;
  tool::ustring inclusions;

  enum class OutputMode {
    Binary,
    Cpp,
    Csharp,
    Dlang,
    Go,
  };

  unsigned exclusive_options = 0;
  OutputMode output_mode = OutputMode::Cpp;

  for(int n = 3; n < argc; ++n) {
    tchars mod = tool::chars_of(argv[n]);

    if( mod == TCHARS("-v") ) {
      if( n+1 >= argc )
        usage();
      varname = argv[n+1];
      ++n;
    }
    else if( mod == TCHARS("-x") ) {
      if( n+1 >= argc )
        usage();
        exclusions = tool::ustring(argv[n+1]);
      ++n;
    }
    else if (mod == TCHARS("-i")) {
      if (n + 1 >= argc)
        usage();
      inclusions = tool::ustring(argv[n + 1]);
      ++n;
    }
    else if( mod == TCHARS("-cpp") ) {
      output_mode = OutputMode::Cpp;
      exclusive_options++;
    }
    else if( mod == TCHARS("-dlang") ) {
      output_mode = OutputMode::Dlang;
      exclusive_options++;
    }
    else if( mod == TCHARS("-csharp") ) {
      output_mode = OutputMode::Csharp;
      exclusive_options++;
    }
    else if (mod == TCHARS("-go")) {
      output_mode = OutputMode::Go;
      exclusive_options++;
    }
    else if (mod == TCHARS("-binary")) {
      output_mode = OutputMode::Binary;
      exclusive_options++;
    }
  }

  if(exclusive_options > 1)
  {
	  fprintf(stderr, "Error: either one of: -go, -dlang, -csharp or -binary\n");
	  exit(1);
  }

  packed pack;

  switch (output_mode)
  {
  case OutputMode::Cpp:
    pack.printf("const unsigned char %s[] = {\n\t", varname.c_str());
    break;
  case OutputMode::Dlang:
    pack.printf("ubyte[] %s = [\n\t", varname.c_str());
    break;
  case OutputMode::Go:
    pack.printf("package main\n\nvar %s []byte = []byte {\n\t", varname.c_str());
    break;
  case OutputMode::Csharp:
    pack.printf(
      "#if !SCITER_APP_RESOURCE_SKIP\n"\
      "namespace SciterAppResource\n"\
      "{\n"\
      "#if SCITER_APP_RESOURCE_PUBLIC\n"
      "\tpublic\n"
      "#endif\n"
      "\tstatic class ArchiveResource\n"\
      "\t{\n"\
      "\t\tpublic static readonly byte[] %s = new byte[] {\n\t", varname.c_str());
    break;
  }

  int line_counter = 0;

  auto out_text = [&](tool::bytes data) -> uint {
    for(size_t n = 0; n < data.length; ++n) {
      pack.printf("0x%02x,",data[n]);
      if(++line_counter > 40 ) {
        line_counter = 0;
        pack.printf("\n\t");
      }
    }

    return uint(data.length);
  };

  auto out_bin = [&](tool::bytes data) -> uint {
    pack.data.push(data);
    return uint(data.length);
  };

  if( output_mode == OutputMode::Binary )
    tool::sar::pack_folder(TCVT(tool::chars_of(argv[1])), inclusions(), exclusions(), out_bin);
  else
    tool::sar::pack_folder(TCVT(tool::chars_of(argv[1])), inclusions(), exclusions(), out_text);

  switch (output_mode)
  {
  case OutputMode::Dlang:
    pack.printf("\n];\n");
    break;
  case OutputMode::Csharp:
    pack.printf("\n\t\t};\n\t}\n}\n#endif\n");
    break;
  case OutputMode::Cpp:
    pack.printf("\n};\n");
    break;
  case OutputMode::Go:
    pack.printf("\n}\n");
    break;
  }

  {
    tool::mm_file mf;
    if( mf.open(fname) && mf.bytes() == pack.data()) {
      fprintf(stdout,"resources not changed\n");
      return 0;
    }
  }

#ifdef WINDOWS
  FILE* fout = nullptr; 
  _wfopen_s(&fout,argv[2], W("wb"));
#else
  FILE* fout = fopen(argv[2], "wb");
#endif
  if( !fout ) {
#ifdef WINDOWS
    fwprintf(stderr,L"cannot open %s for writing\n",argv[2]);
#else
    fprintf(stderr,"cannot open %s for writing\n",argv[2]);
#endif
    exit(2);
  }
  fwrite(pack.data.cbegin(),1,pack.data.length(),fout);
  fclose(fout);

  return 0;
}

namespace tool {
  namespace async {

    void dispatch::start_timer() {
      assert(false);
    }
    void dispatch::stop_timer() {
      assert(false);
    }
  }
}
