//
//  osx-video.m
//  sciter-osx
//
//  Created by Andrew Fedoniouk on 2014-10-21.
//  Copyright (c) 2014 andrew fedoniouk. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>


#include "osx-sciter-popup.h"
#include "osx-sciter-view.h"

#include "sdk-headers.h"
#ifdef SCITERJS
  #include "sdk.js/include/sciter-x-video-api.h"
#else
  #include "sdk/include/sciter-x-video-api.h"
#endif
#include "osx-cvt.h"

#include "sdk-headers.h"


@class CameraFrameReceiver;

static tool::value get_camera_devices_list();
static CameraFrameReceiver* start_camera_streaming( sciter::video_destination* to, const tool::value& device_id );
static void stop_camera_streaming( CameraFrameReceiver* &bridge );

namespace html { namespace behavior {
    
    struct camera_ctl_factory: ctl_factory
    {
        camera_ctl_factory(): ctl_factory("camera") {}
        virtual ctl* create(element* el) override;
    };
    
    static camera_ctl_factory* _camera_ctl_factory = 0;
 
    void init_camera()
    {
        ctl_factory::add( _camera_ctl_factory = new camera_ctl_factory() );
    }
    
    struct camera_ctl: ctl
    {
        typedef ctl super;
        
        sciter::om::hasset<sciter::video_destination> rendering_site;
        CameraFrameReceiver* bridge;
        
        camera_ctl(): ctl(HANDLE_BEHAVIOR_EVENT), bridge(nullptr) {}
        
        virtual CTL_TYPE get_type() { return CTL_UNKNOWN; }
        virtual const string& behavior_name() const override;
        
        virtual bool is_ext() { return true; }
        
        virtual bool attach(view& v, element* self) override { return super::attach(v,self);  }
        virtual void detach(view& v, element* self) override { stop_camera_streaming(bridge); super::detach(v,self);  }
        
        virtual bool on (view& v, element* self, event_behavior& evt ) override;
        virtual bool on_x_method_call(view& v, element *self, const char* name, const value* argv, size_t argc, value& retval) override;
        
    };
    
    ctl* camera_ctl_factory::create(element* el)
    {
        return new camera_ctl();
    }
    
    const string& camera_ctl::behavior_name() const {
        return _camera_ctl_factory->name;
    }
    
 
    bool camera_ctl::on_x_method_call(view& v, element *self, const char* name, const value* argv, size_t argc, value& retval)
    {
        chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if( argc == ARGC && fname == CHARS(#NAME))
        
        METHOD(0, devices)
        {
            retval = get_camera_devices_list();
            return true;
        }

        METHOD(1, streamFrom)
        {
            stop_camera_streaming(bridge);
            bridge = start_camera_streaming( rendering_site, argv[0] );
            retval = tool::value( bridge != nullptr );
            return true;
        }

        METHOD(0, stopStreaming)
        {
            stop_camera_streaming(bridge);
            return true;
        }
     
        
#undef METHOD
        return super::on_x_method_call(v, self, name, argv, argc,retval); 
    }
    
    bool camera_ctl::on (view& v, element* self, event_behavior& evt ) {
        if( evt.cmd != VIDEO_BIND_RQ )
            return false;
        
        if( evt.reason == 0 )
            return true; // first phase, accept it
        
        rendering_site = (sciter::video_destination*) evt.reason;
        
        return true;
    }
    
    
}}


tool::value get_camera_devices_list() {

    tool::array<tool::value> vout;
    NSArray *devices = [AVCaptureDevice devicesWithMediaType: AVMediaTypeVideo ];
    for (AVCaptureDevice *device in devices) {
        NSString *name = [device localizedName];
        vout.push( tool::value(cvt(name)) );
    }
    return tool::value::make_array(vout());
}


@interface CameraFrameReceiver :  NSObject <AVCaptureVideoDataOutputSampleBufferDelegate>
{
    //osx::iwindow* iwindow;
    AVCaptureSession*          session;
    AVCaptureDevice*           camera;
    AVCaptureDeviceInput*      cameraInput; // This is the data input for the camera that allows us to capture frames
    AVCaptureVideoDataOutput*  cameraOutput;
    sciter::video_destination* destination;
    bool                       firstFrameReceived;
    uint                       frameTick;
}
@end

@implementation CameraFrameReceiver

- ( id ) init
{
    // 1. Initialize the parent class(es) up the hierarchy and create self:
    self = [ super init ];
    
    // 2. Initialize members:
    session           = NULL;
    camera            = NULL;
    cameraInput       = NULL;
    cameraOutput      = NULL;
    destination       = NULL;
    firstFrameReceived = false;
    frameTick = 0;
    
    return self;
}


- ( void ) captureOutput: ( AVCaptureOutput * ) captureOutput
   didOutputSampleBuffer: ( CMSampleBufferRef ) sampleBuffer
          fromConnection: ( AVCaptureConnection * ) connection
{
    // 1. Check if this is the output we are expecting:
    if ( captureOutput == cameraOutput )
    {
        uint currentTick = tool::get_ticks();
        if( frameTick && (currentTick - frameTick) < 40 ) {
            return; // 24 FPS only
        }
        
        frameTick = currentTick;
        
        // 2. If it's a video frame, copy it from the sample buffer:
        [ self copyVideoFrame: sampleBuffer ];
    }
}

- ( void ) copyVideoFrame: ( CMSampleBufferRef ) sampleBuffer
{
    // 1. Get a pointer to the pixel buffer:
    CVPixelBufferRef pixelBuffer = ( CVPixelBufferRef ) CMSampleBufferGetImageBuffer( sampleBuffer );

    // 2. Obtain access to the pixel buffer by locking its base address:
    CVOptionFlags lockFlags = 0; // If you are curious, look up the definition of CVOptionFlags
    CVReturn status = CVPixelBufferLockBaseAddress( pixelBuffer, lockFlags );
    assert( kCVReturnSuccess == status );
    
    // 3. Copy bytes from the pixel buffer
    // 3.1. First, work out how many bytes we need to copy:
    size_t bytesPerRow = CVPixelBufferGetBytesPerRow( pixelBuffer );
    size_t height = CVPixelBufferGetHeight( pixelBuffer );
    UINT   numBytes = UINT(bytesPerRow * height);
    
    // 3.2. Then work out where in memory we'll need to start copying:
    BYTE * bytes = (BYTE *)CVPixelBufferGetBaseAddress( pixelBuffer );
    
    @synchronized( self ) {
    
      if(!firstFrameReceived) {
        int frameWidth = (int)CVPixelBufferGetWidth( pixelBuffer );
        int frameHeight = (int) height;
        firstFrameReceived = true;
        destination->start_streaming(frameWidth,frameHeight, sciter::COLOR_SPACE_RGB32);
        //NSLog(@"started streaming frames %d,%d", frameWidth, frameHeight);
      }
      destination->render_frame(bytes,numBytes);
 
    }
    
    // 6. Let go of the access to the pixel buffer by unlocking the base address:
    CVOptionFlags unlockFlags = 0; // If you are curious, look up the definition of CVOptionFlags
    CVPixelBufferUnlockBaseAddress( pixelBuffer, unlockFlags );
    
}

- ( void ) stop
{
    if ( NULL == session )
    {
        // The camera was never started, don't bother stpping it
        return;
    }
    
    // Make sure we don't pull the rug out of the camera thread's feet.
    // Get hold of a mutex with @synchronized and then stop
    // and tidy up the capture session.
    @synchronized( self )
    {
        if ( [ session isRunning ] )
        {
            [ session stopRunning ];
            
            assert( NULL != cameraOutput );
            [ session removeOutput: cameraOutput ];
            
            assert( NULL != cameraInput );
            [ session removeInput: cameraInput ];
            
            [session release];      session = NULL;
            [camera release];       camera = NULL;
            [cameraInput release];  cameraInput = NULL;
            [cameraOutput release]; cameraOutput = NULL;
            firstFrameReceived = false;
        }
    }
}

- ( void ) setupVideoOutput
{
    // 1. Create the video data output
    cameraOutput = [ [ AVCaptureVideoDataOutput alloc ] init ];
    
    // 2. Create a queue for capturing video frames
    dispatch_queue_t captureQueue = dispatch_queue_create( "captureQueue", DISPATCH_QUEUE_SERIAL );
    
    // 3. Use the AVCaptureVideoDataOutputSampleBufferDelegate capabilities of CameraDelegate:
    [ cameraOutput setSampleBufferDelegate: self queue: captureQueue ];
    
    dispatch_release( captureQueue );
    
    // 4. Set up the video output
    // 4.1. Do we care about missing frames?
    cameraOutput.alwaysDiscardsLateVideoFrames = YES;
    
    // 4.2. We want the frames in some RGB format, which is what ActionScript can deal with
    //NSNumber * framePixelFormat = [ NSNumber numberWithInt: kCVPixelFormatType_32BGRA ];
    
    
    id keys[] = { ( id ) kCVPixelBufferPixelFormatTypeKey, (id)kCVPixelBufferWidthKey, (id)kCVPixelBufferHeightKey };
    id vals[] = { [ NSNumber numberWithInt: kCVPixelFormatType_32BGRA ],
                  [ NSNumber numberWithInt: 640 ],
                  [ NSNumber numberWithInt: 480 ] };
        
   
    
    //cameraOutput.videoSettings = [ NSDictionary dictionaryWithObject: framePixelFormat
    //                                                           forKey: ( id ) kCVPixelBufferPixelFormatTypeKey ];
    
    cameraOutput.videoSettings = [ NSDictionary dictionaryWithObjects: vals
                                                              forKeys: keys
                                                                count:3 ];
  
    
    // 5. Add the video data output to the capture session
    [ session addOutput: cameraOutput ];
}


- ( BOOL ) attachCamera : ( AVCaptureDevice* ) device
            destination : (sciter::video_destination*) dest
{
    // 0. Assume we've found the camera and set up the session first:
    assert( NULL == camera );
    assert( NULL == session );
    
    if(destination)
        destination->asset_release();
    
    destination = dest;
    destination->asset_add_ref();
    
    firstFrameReceived = false;
    
    camera = device; [camera retain];
    
    //2. Make sure we have a capture session
    session = [ [ AVCaptureSession alloc ] init ];
   
    
    // 1. Initialize the camera input
    cameraInput = NULL;
    
    // 2. Request a camera input from the camera
    NSError * error = NULL;
    cameraInput = [ AVCaptureDeviceInput deviceInputWithDevice: camera error: &error ];
    
    // Check if we've got any errors
    if ( NULL != error )
    {
        return false;
    }
    
    [cameraInput retain];

    
    [session beginConfiguration];
    
    // 3. Choose a preset for the session.
    NSString * cameraResolutionPreset = AVCaptureSessionPresetLow;
    
    // 4. Check if the preset is supported on the device by asking the capture session:
    if ( ![ session canSetSessionPreset: cameraResolutionPreset ] )
    {
        // Optional TODO: Send an error event to ActionScript
        return false;
    }
    
    // 4.1. The preset is OK, now set up the capture session to use it
    session.sessionPreset = cameraResolutionPreset;
    
    
    // 3. We've got the input from the camera, now attach it to the capture session:
    if ( [ session canAddInput: cameraInput ] )
    {
        [ session addInput: cameraInput ];
    }
    else
    {
        // TODO: send an error event to ActionScript
        return false;
    }

    
    [ self setupVideoOutput ];
    
    [session commitConfiguration];
    
    
    // 4. Done, the attaching was successful, return true to signal that
    return true;
}

- ( void ) start
{
    [session startRunning];
}

@end



CameraFrameReceiver* start_camera_streaming( sciter::video_destination* to, const tool::value& device_id ) {
    NSArray *devices = [AVCaptureDevice devicesWithMediaType: AVMediaTypeVideo ];
    AVCaptureDevice *device = nullptr;
    
    if( device_id.is_string() ) {
      tool::ustring device_name = device_id.get( WSTR("") );
      for (device in devices) {
          NSString *name = [device localizedName];
          if(cvt(name) == device_name)
             goto FOUND;
      }
    } else if(device_id.is_int()) {
      uint idx = device_id.get(0);
      if( idx < [devices count] ) {
        device = [devices objectAtIndex: idx ];
        goto FOUND;
      }
    }
        
    return nullptr;
FOUND:
    
    CameraFrameReceiver* bridge = [[CameraFrameReceiver alloc] init];
    if( !bridge )
        return nullptr;
    
    if(![bridge attachCamera:device destination:to])
    {
        [bridge release];
        return nullptr;
    }
    
    [ bridge start ];
    
    return bridge;
    
}

void stop_camera_streaming( CameraFrameReceiver* &bridge ) {
    if( bridge ) {
      [bridge stop];
      [bridge release ];
      bridge = nullptr;
    }
}

