/*
*/

#include "osx-sciter.h"
#include "osx-sciter-graphics.h"
#include "osx-ref.h"
#include <ApplicationServices/ApplicationServices.h>
#include <Carbon/Carbon.h>

#import <IOKit/graphics/IOGraphicsLib.h>


//typedef void (*AppFocusChangeCallback)();
//extern AppFocusChangeCallback appFocusChangeCallback;
//typedef bool (*CheckEventBlockedByModalComps) (NSEvent*);
//extern CheckEventBlockedByModalComps isEventBlockedByModalComps;

@interface NSEvent (SciterDeviceDelta)
 - (CGFloat) scrollingDeltaX;
 - (CGFloat) scrollingDeltaY;
 - (BOOL) hasPreciseScrollingDeltas;
 - (BOOL) isDirectionInvertedFromDevice;
@end


NSScreen* primaryScreen() {
   return [[NSScreen screens] firstObject];
}

gool::size  osx_screen_dim()
{
    NSScreen* pps = primaryScreen();
    return pps ? fromns<int>( [pps frame].size ) : gool::size(1000,1000);
}

gool::rect flipped_screen_rect (gool::rect r)
{
    gool::point pt = r.start();
    pt.y = osx_screen_dim().y - (pt.y + r.height()) - 1;
    r.pointOf(7,pt);
    return r;
}

gool::rect osx_screen_rect (gool::rect r)
{
    gool::point pi = r.start();
    pi.y = osx_screen_dim().y - pi.y - 1;
    r.pointOf( 1, pi );
    return r;
}

gool::point osx_screen_point (gool::point pt)
{
    pt.y = osx_screen_dim().y - pt.y;
    return pt;
}

bool osx_iwindow_has_close_button(html::iwindow* iw)
{
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    return ([pnw styleMask] & NSClosableWindowMask) != 0;
}

bool osx_iwindow_is_visible(html::iwindow* iw) {
    
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    return [pnw isVisible];
}

gool::rect osx_iwindow_screen_pos(html::iwindow* iw) {
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    NSRect r = [pnw frame];
    return flipped_screen_rect(fromns<int>( r ));
}

gool::rect  osx_window_decoration(html::iwindow* iw)
{
    NSView*    pnv = (NSView*) iw->get_hwnd();
    NSWindow*  pnw = [pnv window];
    NSUInteger mas = [pnw styleMask];

    NSRect ra; ra.origin.x = 0;
               ra.origin.y = 0;
               ra.size.width = 0;
               ra.size.height = 0;

    NSRect rb = [NSWindow frameRectForContentRect: ra styleMask:mas];

    gool::rect rc = flipped_screen_rect(fromns<int>( rb ));
    gool::rect r0 = flipped_screen_rect(fromns<int>( ra ));

    return gool::rect( r0.s.x - rc.s.x,
                 r0.s.y - rc.s.y,
                 rc.e.x - r0.e.x,
                 rc.e.y - r0.e.y );
}

void osx_iwindow_replace_child(html::iwindow* iw, gool::rect box, bool visible) {
    
    NSView* pnv = (NSView*) iw->get_hwnd();
    
    NSView* psupernv = [pnv superview];
    if( psupernv ) {
       NSRect superrc = [psupernv frame];
       NSRect dst = tons(box);
       if( !psupernv.isFlipped )
         dst.origin.y = superrc.size.height - dst.origin.y - dst.size.height;
       [pnv setFrame: dst ];
    }

    [pnv setHidden: visible ? NO : YES ];
}


void osx_iwindow_screen_pos(html::iwindow* iw, gool::point pt) {
    
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    [pnw setFrameTopLeftPoint: tons( osx_screen_point(pt) )];
}

void osx_iwindow_screen_pos(html::iwindow* iw, gool::rect rc) {
   
    rc = osx_screen_rect(rc);
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    
#ifdef MAC_OS_X_VERSION_10_9    
    //if([pnw respondsToSelector:@selector(occlusionState)]
    //    && ([pnw occlusionState] &  NSWindowOcclusionStateVisible) == 0)
    //  return;
#endif
    
    //gool::rect pr = cvt<int>([pnw frame]);
    BOOL visible = [pnw isVisible];
    
    [pnw setFrame: tons(rc) display: visible ];
    
    if( iw->type == html::FRAME_WINDOW || iw->type == html::DIALOG_WINDOW)
        return;
    
    if( !iw->root() )
        return;
    
    if( !iw->root()->pview() )
        return;
    
    bool is_detached = (iw->windowing_mode == html::ELEMENT_WINDOW_DETACHED || iw->windowing_mode == html::ELEMENT_WINDOW_DETACHED_TOPMOST );
    
    if( [pnw parentWindow] && is_detached ) {
        [[pnw parentWindow] removeChildWindow:pnw ];
        //[pnw invalidateShadow];
        // oskryp: restore popup level for topmost
        if (iw->windowing_mode == html::ELEMENT_WINDOW_DETACHED_TOPMOST)
          [pnw setLevel:NSPopUpMenuWindowLevel];
        
    }
    else if( ![pnw parentWindow] && !is_detached )
    {
        html::view* pv = iw->root()->pview();
        NSView* pnv = (NSView*) pv->get_hwnd();
        if( iw->anchor() )
            pnv = (NSView*) iw->anchor()->get_window(*pv)->get_hwnd();
        if(iw->windowing_mode != html::ELEMENT_WINDOW_UNKNOWN && pnv)
          [[pnv window] addChildWindow:pnw ordered:NSWindowAbove];
        //[pnw invalidateShadow];
    }
    //else if( pr.size() != rc.size() )
   
}

bool osx_iwindow_set_title(html::iwindow* iw, const wchar* title)
{
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    @autoreleasepool {
        NSString* caption = cvt(title);
        [pnw setTitle: caption ];
        return true;
    }
}
tool::ustring osx_iwindow_get_title(const html::iwindow* iw)
{
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    return cvt( [pnw title]);
}

bool osx_set_application_icon(html::iwindow* iw, gool::image* pimg)
{
    //NSView* pnv = (NSView*) iw->get_hwnd();
    tool::handle<gool::bitmap> pbm = pimg->get_bitmap(nullptr, gool::size(256));
    if( pbm )
    {
        //???? pbm->top_to_bottom_inplace();
        CGImageRef cgim = osx_image( pbm, nullptr );
        [NSApp setApplicationIconImage: [[NSImage alloc] initWithCGImage:cgim size:NSZeroSize]]; 
        return true;
    }
    return false;
}


void osx_iwindow_close(html::iwindow* iw, bool force) {
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    if( force )
      [pnw close];
    else
      [pnw performClose: pnw];
}

void osx_iwindow_update(html::iwindow* iw) {
    NSView* pnv = (NSView*) iw->get_hwnd();
    [pnv displayIfNeeded];
}

void osx_iwindow_refresh(html::iwindow* iw, const gool::rect& rc)
{
    //printf("osx_iwindow_refresh %d %d %d %d\n", rc.left(), rc.top(), rc.width(), rc.height() );
    NSView* pnv = (NSView*) iw->get_hwnd();
    if(iw->is_painting) {
      [pnv performSelector:@selector(refresh) withObject:pnv afterDelay:0.016];
    }
    else if(rc.empty())
      [pnv setNeedsDisplay:TRUE];
    else
      [pnv setNeedsDisplayInRect: tons(rc)];
}

gool::size  osx_iwindow_client_dim(html::iwindow* iw)
{
  NSView* pnv = (NSView*) iw->get_hwnd();
  NSSize sz = pnv.bounds.size;
  if( iw->uses_physical_size() )
  {
      NSView* nsv = (NSView*)iw->get_hwnd();
      NSWindow* nsw = [nsv window];
      float bsf = [nsw backingScaleFactor];
      sz.width *= bsf;
      sz.height *= bsf;
  }
  return fromns<int>( sz );
}
gool::size  osx_iwindow_window_dim(html::iwindow* iw)
{
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSWindow* pnw = [pnv window];
    NSRect r = [pnw frame];
    return fromns<int>( r.size );
}
gool::point osx_iwindow_client_screen_pos(html::iwindow* iw)
{
    NSView* pnv = (NSView*) iw->get_hwnd();
    NSRect r = [[pnv window] convertRectToScreen:[pnv bounds]];
    return flipped_screen_rect(fromns<int>(r)).start();
}

gool::point osx_iwindow_cursor_pos(html::iwindow* iw)
{
    gool::point pt = fromns<int>([NSEvent mouseLocation]);
    pt.y = osx_screen_dim().y - pt.y;
    return pt;
}
//void        osx_iwindow_refresh(html::iwindow* iw, const gool::rect& rc);
//void        osx_iwindow_update(html::iwindow* iw);

gool::rect  osx_screen_workarea(gool::rect rc)
{
    //NSPoint mouseLoc = [NSEvent mouseLocation];
    NSPoint pt = tons(rc.pointOf(5));
    NSEnumerator *screenEnum = [[NSScreen screens] objectEnumerator];
    NSScreen* screen;
    while ((screen = [screenEnum nextObject])) {
        if( NSPointInRect(pt, [screen visibleFrame]) )
            return flipped_screen_rect(fromns<int>([screen visibleFrame]));
    }
    return gool::rect(osx_screen_dim());
}

gool::rect  osx_screen_workarea( HWINDOW hwnd, gool::rect rc )
{
    NSView* nsv = (NSView*)hwnd;
    NSWindow* nsw = [nsv window];
    NSScreen* nsc = (nsw && [nsw screen]) ? [nsw screen] : primaryScreen();
    
    NSRect frc = [nsc visibleFrame];
    if( !frc.size.width || !frc.size.height )
        return osx_screen_workarea(gool::rect());
    return flipped_screen_rect(fromns<int>(frc));
}

gool::size  osx_pixels_per_inch(html::iwindow* iw) {
    gool::size sz = gool::size(96,96);
    if( iw && iw->uses_physical_size() )
    {

       NSView* nsv = (NSView*)iw->get_hwnd();
       NSWindow* nsw = [nsv window];
       float bsf = [nsw backingScaleFactor];
       sz *= bsf;
    }
    return sz;
}

gool::size  osx_physical_pixels_per_inch(html::iwindow* iw) {
    gool::size sz = gool::size(96,96);
    if( iw )
    {
        NSView* nsv = (NSView*)iw->get_hwnd();
        NSWindow* nsw = [nsv window];
        float bsf = [nsw backingScaleFactor];
        sz *= bsf;
    }
    return sz;
}


gool::point get_mouse_pos(NSEvent* e, NSView* view)
{
    NSWindow* thisWindow = [view window];
    if( thisWindow == [e window]) {
      NSPoint p = [view convertPoint: [e locationInWindow] fromView: nil];
      return gool::point(p.x,p.y);
    } else {
      NSWindow* otherWindow = [e window];
        
      NSPoint p = [[otherWindow contentView] convertPoint: [e locationInWindow] fromView: nil];
      gool::point pt = gool::point(p.x,p.y);
        
      gool::rect this_rc = fromns<int>([thisWindow convertRectToScreen:[view bounds]]);
      gool::rect other_rc = fromns<int>([otherWindow convertRectToScreen:[[otherWindow contentView] bounds]]);
      this_rc = flipped_screen_rect(this_rc);
      other_rc = flipped_screen_rect(other_rc);
        
      pt += other_rc.start() - this_rc.start();
       
      return pt;
    }
    
}

uint get_alts() {
    
    uint alts = 0;
    CGEventSourceStateID eventSource = kCGEventSourceStateCombinedSessionState;
    if( CGEventSourceKeyState(eventSource, kVK_Option))  alts |= html::ALT_ALT;
    if( CGEventSourceKeyState(eventSource, kVK_Control))  alts |= html::ALT_CONTROL;
    if( CGEventSourceKeyState(eventSource, kVK_Shift))    alts |= html::ALT_SHIFT;
    return alts;
}

uint get_alts(NSEvent* ev) {
    
    unsigned long flags = [ev modifierFlags];
    uint alts = 0;
    
    if( flags & NSAlternateKeyMask) alts |= html::ALT_ALT;
    if( flags & NSControlKeyMask )  alts |= html::ALT_CONTROL;
    if( flags & NSShiftKeyMask )    alts |= html::ALT_SHIFT;
    if( flags & NSCommandKeyMask )  alts |= html::ALT_COMMAND;
    
    return alts;
}

uint get_mouse_buttons() {
    NSUInteger nb = [NSEvent pressedMouseButtons];
    uint buttons = 0;
    if( nb & (1 << 0) ) buttons |= html::MAIN_BUTTON;
    if( nb & (1 << 1) ) buttons |= html::PROP_BUTTON;
    if( nb & (1 << 2) ) buttons |= html::MIDDLE_BUTTON;
    return buttons;
}

void each_extension(const wchar* filter, std::function<void(wchars)> cb)
{
    wchars f = chars_of(filter);
    while( f.length ) {
        f.chop(WCHARS("|"));
        wchars extlist = f.chop(WCHARS("|"));
        while(extlist.length) {
          wchars maskext = extlist.chop(WCHARS(";"));
          wchars ext = maskext.r_tail('.');
          if( ext != WCHARS("*") )
              cb(ext);
        }
    }
}


//bool osx_ask_file_name(bool to_save, const ustring& caption, ustring& filename, const wchar* def_ext, const wchar* filter)
array<ustring> osx_ask_file_name(osx::view::AFN_MODE mode, const ustring& caption, const ustring& filename, const wchar* def_ext, const wchar* filter)
{
    
    auto initExtensions = [filter]( NSSavePanel* panel ) {
        NSMutableArray* list = [NSMutableArray arrayWithCapacity:4];
        int cnt = 0;
        each_extension(filter, [&list,&cnt](wchars ext) {
            [list addObject: cvt(ext) ];
            ++cnt;
        });
        if(cnt)
            [panel setAllowedFileTypes:list];
        //else
        //    [list release]; - as we are not used init/alloc
    };
    
    array<ustring> ret;
    
//@autoreleasepool {
 
    if( mode == osx::view::AFN_SAVE ) {
       NSSavePanel * panel = [NSSavePanel savePanel];
      [panel setFloatingPanel:YES];
      [panel setTitle: cvt(caption)];
      //if(caption.length())
      //   [panel setMessage: cvt(caption)];

      [panel setDirectoryURL:[NSURL fileURLWithPath:cvt(filename)]];
        
      [panel setAllowsOtherFileTypes:YES];
      [panel setCanCreateDirectories:YES];
        
      if( !filename().ends_with(WCHARS("/"))) {
        tool::ustring fn = filename().r_tail(WCHARS("/"));
        if( fn.is_defined() )
          [panel setNameFieldStringValue:cvt(fn)];
      }
        
      if(filter)
          initExtensions(panel);
        
      NSInteger result = [panel runModal];
      if(result == NSOKButton)
      {
        ret.push(cvt([[panel URL] path]));
        return ret;
      }
        
    } else {
        
        NSOpenPanel * panel = [NSOpenPanel openPanel];
        [panel setAllowsMultipleSelection: (mode == osx::view::AFN_OPEN_MULTIPLE ? YES : NO) ];
        [panel setCanChooseDirectories:NO];
        [panel setCanChooseFiles:YES];
        [panel setFloatingPanel:YES];
        //if(caption.length())
        //  [panel setTitle: cvt(caption)];
        if(caption.length())
          [panel setMessage: cvt(caption)];
        [panel setDirectoryURL:[NSURL fileURLWithPath:cvt(filename)]];
        if(filter)
           initExtensions(panel);
        
        NSInteger result = [panel runModal];
        if(result == NSOKButton)
        {
            for(NSURL* url in [panel URLs])
            {
                ret.push( cvt( [ url path ] ));
            }
            return ret;
        }
    }
//}
    return ret;
}

bool osx_ask_folder_name(const ustring& caption, ustring& foldername)
{
    //@autoreleasepool {
    
    NSOpenPanel * panel = [NSOpenPanel openPanel];
    [panel setAllowsMultipleSelection:NO];
    [panel setCanChooseDirectories:YES];
    [panel setCanChooseFiles:NO];
    [panel setFloatingPanel:YES];
    if(foldername.length())
      [panel setDirectoryURL:[NSURL fileURLWithPath:cvt(foldername)]];
    
    NSInteger result = [panel runModal];
    if(result == NSOKButton)
    {
        foldername.clear();
        for(NSURL* url in [panel URLs])
        {
            if( foldername.length() )
                foldername += W(";");
            foldername += cvt( [ url path ] );
        }
        return true;
    }
    //}
    return false;
}

bool osx_do_events(osx::view* iw, html::DO_EVENT_MANNER m, bool& result )
{
    handle<osx::view> iwindow = iw;
    
    result = true;

    if ( m == html::DO_EVENT_ONLY_IO )
      return iw->html::view::exec_idle();


    if( m != html::DO_EVENT_UNTIL_MOUSE_UP )
        do {
            @autoreleasepool
            {
                NSView* pnv = (NSView*) iwindow->get_hwnd();
                if( !pnv )
                    return false;
                NSWindow* pnw = [pnv window];
                if( !pnw )
                    return false;
                //if(![pnw isVisible])
                //    return false;

                
                /* following is hanging up, why?
                 
                 SInt32 r = CFRunLoopRunInMode (kCFRunLoopDefaultMode, m == html::DO_EVENT_MANNER::DO_EVENT_NOWAIT ? 0: 0.02, true);
                
                switch(m) {
                    case html::DO_EVENT_MANNER::DO_EVENT_NOWAIT:
                        return true;
                    case html::DO_EVENT_MANNER::DO_EVENT_WAIT:
                        if( r != kCFRunLoopRunTimedOut )
                            return true;
                        break;
                    case html::DO_EVENT_MANNER::DO_EVENT_ALL:
                        if( r != kCFRunLoopRunTimedOut )
                            return true;
                    default:
                        return false;
                }*/
                
                NSDate* nd = [NSDate dateWithTimeIntervalSinceNow: 0.020];

                    
                NSEvent* e = [NSApp nextEventMatchingMask: NSAnyEventMask
                                                untilDate: nd
                                                   inMode: NSDefaultRunLoopMode
                                                  dequeue: YES];
                    
                if (e != nil)
                   [NSApp sendEvent: e];
                
                pnv = (NSView*) iwindow->get_hwnd();
                if( !pnv )
                    return false; // window was closed while handling the event

                html::view::each([](html::view* pv){ pv->on_idle(); });
                
                //iw->on_idle();
                
                switch(m) {
                    case html::DO_EVENT_MANNER::DO_EVENT_NOWAIT:
                        return true;
                    case html::DO_EVENT_MANNER::DO_EVENT_WAIT:
                        if( e != nil )
                          return true;
                        break;
                    case html::DO_EVENT_MANNER::DO_EVENT_ALL:
                        if( e == nil )
                          return true;
                        continue;
                    default:
                        return false;
                }
                
            }
        } while( true );
    
    else {

        //bool loop = true;
        
        //NSPoint newDragLocation;
        
        [[NSCursor closedHandCursor] push];
       
        /*for (;;) {
            @autoreleasepool {
                
                NSView*   pnv = (NSView*) iwindow->get_hwnd();
                if( !pnv ) return false;
                NSWindow* pnw = [pnv window];
                if( !pnw ) return false;
                // get the next event that is a mouse-up or mouse-dragged event
                
                NSEvent *localEvent = [ pnw nextEventMatchingMask:NSLeftMouseUpMask | NSLeftMouseDraggedMask];
                if(!localEvent) return false;
               
               
                switch ([localEvent type])
                {
                    case NSLeftMouseDragged:
                        
                        // convert the new drag location into the view coords
                        //newDragLocation = [pnv convertPoint:[localEvent locationInWindow] fromView:pnv];
                        //newDragLocation = newDragLocation;
                        [pnv mouseDragged:localEvent];
                        break;
                        
                    case NSLeftMouseUp:
                        // mouse up has been detected,
                        // we can exit the loop
                        //loop = false;
                        
                        // finished dragging, restore the cursor
                        [NSCursor pop];
                        
                        // the rectangle has moved, we need to reset our cursor
                        // rectangle
                        [[pnv window] invalidateCursorRectsForView:pnv];
                        
                        [pnv mouseUp:localEvent];
                        
                        return true;
                    default:
                        // Ignore any other kind of event. 
                        break;
                }

                
            }
        }*/
        
        NSView*   pnv = (NSView*) iwindow->get_hwnd();
        if( !pnv ) return false;
        NSWindow* pnw = [pnv window];
        if( !pnw ) return false;
        // get the next event that is a mouse-up or mouse-dragged event
        
        [pnw makeKeyAndOrderFront:pnv];
        
        //NSTrackingArea* nsta = [NSTrackingArea alloc];
        //[nsta
        //  initWithRect: [pnv frame]
        //  options: NSTrackingMouseMoved | NSTrackingActiveAlways | NSTrackingEnabledDuringMouseDrag
        //  owner: pnv
        //  userInfo:nil];
        
        //[pnv addTrackingArea:nsta];
        
        bool r = false;
        
        do {
            @autoreleasepool {
                
                NSEvent *localEvent =
                [pnw
                    nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask | NSKeyDownMask)
                    untilDate:[NSDate distantFuture]
                    inMode:NSDefaultRunLoopMode
                    dequeue:YES];

                if(!localEvent)
                    continue;
                
                pnv = (NSView*) iwindow->get_hwnd();
                if( !pnv )
                    return false; // window was closed while handling the event
                
                switch ([localEvent type])
                {
                    case NSLeftMouseDragged:
                        
                        [pnv mouseDragged:localEvent];
                        [NSApp updateWindows];
                        break;
                        
                    case NSLeftMouseUp:
                        // mouse up has been detected,
                        // we can exit the loop
                        // finished dragging, restore the cursor
                        [NSCursor pop];
                        // the rectangle has moved, we need to reset our cursor
                        // rectangle
                        [[pnv window] invalidateCursorRectsForView:pnv];
                        [pnv mouseUp:localEvent];
                        r = true;
                        break;
                    case NSKeyDown:
                        if( [localEvent keyCode] == KB_ESCAPE ) {
                            result = false;
                            return true;
                        }
                        // fall through
                    default:
                        [pnw sendEvent:localEvent];
                        [NSApp updateWindows];
                        break;
                }
                
                
            }
   
        } while (!r);
        
        
        //if(nsta) {
        //  [pnv removeTrackingArea: nsta];
        //  [nsta release];
        //}
        
        return r;
        
    }
    return false;
    
}

extern uint module_version(bool major);

void setup_window(NSWindow *nw, NSView* nv, osx::view* pv);

namespace osx
{
    
    void view::stop()
    {
        super::stop();
        //setup_mouse_idle(false, point());
        stop_request_idle();
        stop_timers();
        trayicon_remove();
        
        NSView* pnv = (NSView*) get_hwnd();
        set_hwnd(nullptr);
        [pnv release];
    }
    
    bool view::render(void* hdc, gool::rect dirty_rc) {
        CGContextRef cg = (CGContextRef) hdc;
        handle<osx::graphics> context = new osx::graphics(cg, this->is_layered() ); // (float) [view frame].size.height, displayScale
        render( context, dirty_rc );
        //context->fill(argb(255,0,0), dirty_rc);
        return true;
    }

    uint view::dblclick_timeout() const {
      //uint r = uint([NSEvent doubleClickInterval] * 1000);
      //return r;
      return 400;
    }
    
    bool view::set_resizeable(bool on) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        auto mask = [pnw styleMask];
        if( on )
            mask |= NSResizableWindowMask;
        else
            mask &= ~NSResizableWindowMask;
        
        [pnw setStyleMask: mask];
        
//        ::setup_window(pnw, pnv, this);
        
        return true;
        
    }
    bool view::get_resizeable() const
    {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        auto mask = [pnw styleMask];
        return (mask & NSResizableWindowMask) != 0;
    }
    
    bool        view::set_frame_type(FRAME_TYPE on)
    {
        //release_backend();
        
        super::set_frame_type(on);

        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        unsigned int window_style = 0;
        bool layered = false;
        switch( this->type )
        {
            case UNDEFINED_WINDOW_TYPE:
            case TOOLTIP_WINDOW:
            case PRINT_VIEW:
            case CHILD_WINDOW: return false;
                
            case POPUP_WINDOW:
                window_style = NSBorderlessWindowMask; break;
            case TOOL_WINDOW:
                window_style = NSTitledWindowMask | NSUnifiedTitleAndToolbarWindowMask | NSClosableWindowMask; break;
            case FRAME_WINDOW:
                window_style = NSTitledWindowMask | NSMiniaturizableWindowMask | NSClosableWindowMask | NSResizableWindowMask; break;
            case DIALOG_WINDOW:
                window_style = NSTitledWindowMask | NSClosableWindowMask; break;
        }
        
        switch (on)
        {
            default:
            case STANDARD:
                break;
            case LAYERED:
                layered = true;
                window_style = NSBorderlessWindowMask; break;
            case SOLID:
            case SOLID_WITH_SHADOW:
                window_style = NSBorderlessWindowMask; break;
            case STANDARD_EXTENDED:
                {
                  window_style |= NSFullSizeContentViewWindowMask;
                  [pnw setTitleVisibility:NSWindowTitleHidden];
                  [pnw setTitlebarAppearsTransparent:YES];
                }
                break;
                
        }
        
        NSRect placement = [pnw frame];
        [pnw setStyleMask:window_style];
        [pnw setFrame:placement display:NO];
        
        [pnw setHasShadow: ( on == html::STANDARD || on == html::STANDARD_EXTENDED || on == html::SOLID_WITH_SHADOW ? YES : NO )];
        
        if( layered ) {
            [pnw setOpaque:NO];
            [pnw setBackgroundColor:[NSColor clearColor]];
        } else {
            [pnw setOpaque: YES];
            //breaks OpenGL: [pnv setWantsLayer: YES]; // enable H/W acceleration, NOTE this disables click through
        }
        
        _frame_type = on;
        
        ::setup_window(pnw, pnv, this);
       
        return true;
        
    }
    FRAME_TYPE  view::get_frame_type() const
    {
        if( is_layered() )
            return LAYERED;
        return _frame_type;
    }
    
    bool view::set_minimizable(bool on)
    {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        auto mask = [pnw styleMask];
        if( on ) {
            mask |= NSMiniaturizableWindowMask;
            [[pnw standardWindowButton:NSWindowMiniaturizeButton] setHidden:NO];
        }
        else {
            mask &= ~NSMiniaturizableWindowMask;
            [[pnw standardWindowButton:NSWindowMiniaturizeButton] setHidden:YES];
        }
        
        [pnw setStyleMask: mask];
        [pnw invalidateShadow];
        
        return true;
        
    }
    
    bool view::get_minimizable()
    {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        auto mask = [pnw styleMask];
        return (mask & NSMiniaturizableWindowMask) != 0;
        
    }
    
    bool view::set_maximizable(bool on)
    {
      NSView* pnv = (NSView*) get_hwnd();
      if( !pnv )
        return false;
      NSWindow* pnw = [pnv window];
      if( !pnw )
        return false;
        
      [[pnw standardWindowButton:NSWindowZoomButton] setHidden:!on];
        
      //NSButton *button = [pnw standardWindowButton:NSWindowZoomButton];
      //[button setEnabled:on];
      return true;
    }
    
    bool view::get_maximizable() {
      //return get_resizeable();
      NSView* pnv = (NSView*) get_hwnd();
      if( !pnv )
          return false;
      NSWindow* pnw = [pnv window];
      if( !pnw )
          return false;

      NSButton *button = [pnw standardWindowButton:NSWindowZoomButton];
      return [button isEnabled];      
    }
    
    void view::aspect_ratio(tool::float_v v) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return;
        
        if( v.is_defined() ) {
            NSSize r; r.width = 1000 * v.val(0.0f);
            r.height = 1000;
            [pnw setContentAspectRatio:r];
        } else {
            [pnw setResizeIncrements:NSMakeSize(1,1)];
        }
    }

    tool::float_v view::aspect_ratio() const {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return tool::float_v();
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return tool::float_v();

        NSSize r = [pnw contentAspectRatio];
        
        if( r.width && r.height ) {
            return float(r.width) / float(r.height);
        } else {
            return tool::float_v();
        }
    }
    
    bool view::get_min_size(gool::size& sz) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        NSSize nsz = [pnw contentMinSize];
        if( nsz.width && nsz.height )
        {
            sz.x = nsz.width;
            sz.y = nsz.height;
            return true;
        }
        return false;
    }
    
    bool view::get_max_size(gool::size& sz) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        NSSize nsz = [pnw contentMaxSize];
        if( nsz.width != CGFLOAT_MAX && nsz.height != CGFLOAT_MAX )
        {
            sz.x = nsz.width;
            sz.y = nsz.height;
            return true;
        }
        return false;
    }
    
    bool view::set_min_size(gool::size sz) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        NSSize nsz;
        nsz.width = sz.x;
        nsz.height = sz.y;
        
        [pnw setContentMinSize: nsz];
        return true;
    }
    
    bool view::set_max_size(gool::size sz) {
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;

        NSSize nsz;
        nsz.width = sz.x ? CGFLOAT_TYPE(sz.x) : CGFLOAT_MAX;
        nsz.height = sz.y ? CGFLOAT_TYPE(sz.y) : CGFLOAT_MAX;
        
        [pnw setContentMaxSize: nsz];
        return true;
    }
    
    bool view::show_modal() {
        
        handle<view> _this = this;
        handle<html::view> _this_parent = this->parent();
        
        if(_this_parent)
        {
           auto st = _this_parent->get_window_state();
           if( st == WINDOW_HIDDEN || st == WINDOW_MINIMIZED )
             _this_parent->set_window_state(WINDOW_SHOWN);
        }
        
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
            return false;
        NSWindow* pnw = [pnv window];
        if( !pnw )
            return false;
        
        BOOL r = [pnw canBecomeKeyWindow];
        if( !r )
            return false;
        
        [pnw retain];
        
        [pnw makeKeyAndOrderFront: pnv];
                
        NSApplication* app = [NSApplication sharedApplication];
        
        NSModalSession session = [app beginModalSessionForWindow:pnw];
        bool result = true;
        
        for( ;; ) {
         if ([app runModalSession:session] != NSModalResponseContinue)
             break;
         if(!osx_do_events(this, html::DO_EVENT_WAIT, result))
             break;
        }
        
        [app endModalSession: session];
        
        if(_this_parent) {
            _this_parent->set_enabled(true);
        }
        
        
        //remnants of other experiments:
        //  1. while(osx_do_events(this, html::DO_EVENT_WAIT)) {;} */
        //  2. [app runModalForWindow: pnw ];
        
        [pnw release];
        
        return this->dialog_retval != NOTHING_VALUE;
    }
    
    
    void view::init_media_vars() {
        
        super::init_media_vars();
        //media_variables _media_vars;
        
        gool::size min_w_size(0,0);
        gool::size max_w_size = osx_screen_dim();
        gool::size w_size = max_w_size;
        gool::size device_size = max_w_size;
        
        uint bits_per_pixel = 0;
        uint num_colors = 0;
        
        gool::size dpi = pixels_per_inch();
        gool::size dpcm;
        
        bool high_contrast = false;
        bool has_pen = false;
        bool has_mouse = false;
        bool has_mouse_wheel = false;
        bool has_horizontal_mouse_wheel = false; //GetSystemMetrics(SM_MOUSEHORIZONTALWHEELPRESENT) != 0;
        bool screen_reader = false;
        bool slow_machine = false;

#if defined(PLATFORM_DESKTOP)
        has_pen = false;
        has_mouse = true;
        has_mouse_wheel = true;
        slow_machine = false;
#elif defined(PLATFORM_HANDHELD)
        //has_pen = GetSystemMetrics(SM_PENWINDOWS) != 0;
        has_mouse = GetSystemMetrics(SM_MOUSEPRESENT) != 0;
        //has_mouse_wheel = GetSystemMetrics(SM_MOUSEWHEELPRESENT) != 0;
        //slow_machine = GetSystemMetrics(SM_SLOWMACHINE) != 0;
#endif
        
        device_size = max_w_size;
        
        //screen_reader = ...;
        
        NSScreen *screen = primaryScreen();
        
        bits_per_pixel =  (uint) NSBitsPerPixelFromDepth([screen depth]);
        uint64 nc = uint64(1) << bits_per_pixel;
        num_colors = nc > 0xFFFFFFFF?0xFFFFFFFF:int(nc);

        NSDictionary *description = [screen deviceDescription];
        CGSize displaySizeMM = CGDisplayScreenSize( [[description objectForKey:@"NSScreenNumber"] unsignedIntValue]);

        size physical_screen_mm( int(displaySizeMM.width),int(displaySizeMM.height) );
        
        if(physical_screen_mm.x && physical_screen_mm.y)
        {
            dpcm.x = (device_size.x * 10) / physical_screen_mm.x;
            dpcm.y = (device_size.y * 10) / physical_screen_mm.y;
        }
        else
        {
            dpcm.x = (dpi.x * 254) / 100;
            dpcm.y = (dpi.y * 254) / 100;
        }

        if( get_hwnd() )
        {
            w_size = this->client_dim();
        }
        _media_vars["width"] = tool::value::make_length(w_size.x,tool::value::px);
        _media_vars["height"] = tool::value::make_length(w_size.y,tool::value::px);
        _media_vars["min-width"] = tool::value::make_length(min_w_size.x,tool::value::px);
        _media_vars["min-height"] = tool::value::make_length(min_w_size.y,tool::value::px);
        _media_vars["max-width"] = tool::value::make_length(max_w_size.x,tool::value::px);
        _media_vars["max-height"] = tool::value::make_length(max_w_size.y,tool::value::px);
        _media_vars["aspect-ratio"] = tool::value(double(w_size.x)/double(w_size.y));
        
        _media_vars["monitors"] = tool::value( int([[NSScreen screens] count ]));
        _media_vars["device-width"] = tool::value::make_length(device_size.x,tool::value::px);
        _media_vars["device-height"] = tool::value::make_length(device_size.y,tool::value::px);
        _media_vars["device-aspect-ratio"] = tool::value(double(device_size.x)/double(device_size.y));
        
        _media_vars["orientation-portrait"] = device_size.y > device_size.x;
        _media_vars["orientation-landscape"] = device_size.y < device_size.x;
        _media_vars["color"] = int(bits_per_pixel);
        _media_vars["color-index"] = int(num_colors);
        //monochrome ???
        
        if(dpi.x == dpi.y) // dots per inch
        {
            _media_vars["resolution"] = dpi.x;
            _media_vars["min-resolution"] = dpi.x;
            _media_vars["max-resolution"] = dpi.x;
        }
        else
        {
            // A ‘resolution’ (without a "min-" or "max-" prefix) query never matches a device with non-square pixels.
            _media_vars["min-resolution"] = min(dpi.x,dpi.y);
            _media_vars["max-resolution"] = max(dpi.x,dpi.y);
        }
        
        if(dpcm.x == dpcm.y) // dots per sentimeter
        {
            _media_vars["resolution-dpcm"] = dpcm.x;
            _media_vars["min-resolution-dpcm"] = dpcm.x;
            _media_vars["max-resolution-dpcm"] = dpcm.x;
        }
        else
        {
            // A ‘resolution’ (without a "min-" or "max-" prefix) query never matches a device with non-square pixels.
            _media_vars["min-resolution-dpcm"] = min(dpcm.x,dpcm.y);
            _media_vars["max-resolution-dpcm"] = max(dpcm.x,dpcm.y);
        }
        
        _media_vars["high-contrast"] = high_contrast;
        _media_vars["contrast-screen"] = high_contrast; // Symantec's var.
        
        _media_vars["has-pen"] = has_pen;
        _media_vars["has-mouse"] = has_mouse;
        _media_vars["has-mouse-wheel"] = has_mouse_wheel;
        _media_vars["has-horizontal-mouse-wheel"] = has_horizontal_mouse_wheel;
        _media_vars["screen-reader"] = screen_reader;
        _media_vars["slow-machine"] = slow_machine;
        
        _media_vars["engine"] = ustring(W("sciter"));
        _media_vars["engine-version-minor"] = (int)module_version(false);
        _media_vars["engine-version-major"] = (int)module_version(true);
        
        static uint ver[4] = { SCITER_VERSION };
        
        _media_vars["sciter"] = value(ver[0]);
        
        _media_vars["os"] = ustring(tool::environment::get_os_version_name());
        _media_vars["platform"] = ustring("OSX");
        
        /*_media_vars["old-themes"] = environment::get_os_version() < environment::WIN_VISTA;
        _media_vars["new-themes"] = environment::get_os_version() >= environment::WIN_VISTA;*/
        
        //BOOL isDWM = _IsCompositionActive ? _IsCompositionActive()():FALSE;
        _media_vars["composition-supported"] = tool::value(true);
        _media_vars["on-glass"] = tool::value(get_transparency()); 
        
        _media_vars["graphics-layer"] = tool::value(graphics_caps()); 

        NSString *osxMode = [[NSUserDefaults standardUserDefaults] stringForKey:@"AppleInterfaceStyle"];
        _media_vars["ui-ambience"] = ( osxMode && [osxMode containsString:@"Dark"] ) ? W("dark") : W("light"); //If osxMode is nil then it isn't in dark mode, but if osxMode is @"Dark" then it is in dark mode.        
        _media_vars["ui-blurbehind"] = true;
        
    }
    
#if 0
    void view::post_request_render_layered()
    {
        //static NSArray<NSRunLoopMode> *modes = @[ NSRunLoopCommonModes, NSModalPanelRunLoopMode, NSEventTrackingRunLoopMode ];
        //NSView* pnv = (NSView*) get_hwnd();
        //if( !pnv )
        //  return;
        //[pnv performSelector:@selector(render) withObject:pnv afterDelay:0 inModes: modes ];
        
        //static NSArray<NSString*> *modes = @[ NSRunLoopCommonModes,
        //                                      NSModalPanelRunLoopMode,
        //                                      NSEventTrackingRunLoopMode ];
        static NSArray *modes = nullptr;
        if(!modes) {
           modes = [NSArray arrayWithObjects:NSRunLoopCommonModes,NSModalPanelRunLoopMode,NSEventTrackingRunLoopMode, nil];
           [modes retain];
        }
        NSView* pnv = (NSView*) get_hwnd();
        if( !pnv )
          return;
        [pnv performSelectorOnMainThread:@selector(render) withObject:pnv waitUntilDone:FALSE modes: modes ];
    }
#endif
    
    void  view::enable_ime( bool on) {
        _ime_enabled = on;
        NSView* pnv = (NSView*) get_hwnd();
        [pnv unmarkText];
    }
    
}

namespace osx 
{
  bool view::set_focus(helement b, FOCUS_CAUSE cause, bool postfactum)
  {
      if( !postfactum && b && is_enabled() ){
          NSView*  nsv = (NSView*)get_hwnd();
          if(nsv) {
              NSWindow* nsw = [nsv window];
              if( nsw && ![nsw isKeyWindow] && [nsw isVisible])
                [nsw makeKeyAndOrderFront:nsv];
          }
      }
      return super::set_focus(b,cause,postfactum);
  }
    
  bool        view::set_blurbehind(BLUR_BEHIND bb)
  {
    #if 0 && __MAC_OS_X_VERSION_MAX_ALLOWED >= __MAC_10_10
          return false;
    #else
      if( get_blurbehind() == bb )
          return false;
      //return false;
      if( bb ) {
          
         NSView* nv = (NSView*)get_hwnd();
          
         NSVisualEffectView *blurView = [[NSVisualEffectView alloc] initWithFrame:nv.frame];
         set_transparency(true);
      
         [blurView setBlendingMode: NSVisualEffectBlendingMode::NSVisualEffectBlendingModeBehindWindow];
         switch( bb )
         {
             case html::BLUR_BEHIND_CUSTOM:
                 
                 [blurView setMaterial:NSVisualEffectMaterial::NSVisualEffectMaterialSidebar]; break;
                 
             case html::BLUR_BEHIND_ULTRA_DARK:
               //#if __MAC_OS_X_VERSION_MAX _ALLOWED > __MAC_10_10
               [blurView setMaterial:NSVisualEffectMaterial::NSVisualEffectMaterialUltraDark]; break;
               //#else
               //  [blurView setMaterial:NSVisualEffectMaterial::NSVisualEffectMaterialDark]; break;
               //#endif
               case html::BLUR_BEHIND_DARK: 
                 [blurView setMaterial: NSVisualEffectMaterial::NSVisualEffectMaterialDark]; break;
             default:
             case html::BLUR_BEHIND_LIGHT:
               [blurView setMaterial:
               //#if __MAC_OS_X_VERSION_MAX_ALLOWED > __MAC_10_10
                 NSVisualEffectMaterial::NSVisualEffectMaterialMediumLight]; break;
               //#else
               //  NSVisualEffectMaterial::NSVisualEffectMaterialLight]; break;
               //#endif
             case html::BLUR_BEHIND_ULTRA_LIGHT:
               [blurView setMaterial: NSVisualEffectMaterial::NSVisualEffectMaterialLight]; break;
         }
         //[blurView setState: NSVisualEffectState:: NSVisualEffectStateActive /*NSVisualEffectStateFollowsWindowActiveState*/];
         [blurView setState: NSVisualEffectState:: NSVisualEffectStateFollowsWindowActiveState];
         [blurView setAutoresizesSubviews:YES];

         NSWindow* nw = [nv window];
          
         [nv retain];
         [nw setContentView: blurView];
         [blurView addSubview:nv];
         [nv setAutoresizingMask: kCALayerWidthSizable | kCALayerHeightSizable ];
         
         //[nw setTitlebarAppearsTransparent:TRUE];
         //[nw setStyleMask: [nw styleMask] | NSFullSizeContentViewWindowMask];
          
         [nv release];
         
      } else {
         set_transparency(false);
         NSView* nv = (NSView*)get_hwnd();
         NSWindow* nw = [nv window];
         assert ( [[nv window] contentView] != nv );
         [nv retain];
         [nv removeFromSuperview];
         [nw setContentView:nv];
         [nv release];
         //[nw setTitlebarAppearsTransparent:FALSE];
      }
      super::set_blurbehind(bb);
      
      set_transparency(!!bb);
      
      return true;
    #endif
   }
 
   bool view::request_attention(WRA_MODE wm)
   {
       if(request_id.is_defined()) {
           [NSApp cancelUserAttentionRequest: (NSInteger)request_id.val(0) ];
           request_id.clear();
       }
       
       switch(wm) {
           case WRA_ATTENTION_CRITICAL: request_id = [NSApp requestUserAttention: NSCriticalRequest ]; break;
           case WRA_ATTENTION: request_id = [NSApp requestUserAttention: NSInformationalRequest ]; break;
           default: break;
       }
       return true;
   }

   bool view::trayicon_setup(const view::tray_icon_params& params){
       if(!trayicon_item) {
         NSStatusBar* sb = [NSStatusBar systemStatusBar];
         NSStatusItem* si = [sb statusItemWithLength:NSSquareStatusItemLength];
         trayicon_item = si;
         if(!trayicon_item)
           return false;
         [(NSStatusItem*)trayicon_item retain];
         NSButton* button = [(NSStatusItem*)trayicon_item button];
         [button setAction:@selector(trayiconClick:)];
         NSView* pnv = (NSView *) get_hwnd();
         [button setTarget: pnv];
       }
       if(params.img) {
         rect rc;
         trayicon_place(rc);
         handle<gool::bitmap> hbm = params.img->get_bitmap(nullptr, rc.size());
         if(!hbm)
             return false;
         CGImageRef imref = osx_image( hbm );
         @autoreleasepool {
            NSImage* pimg = [[NSImage alloc] initWithCGImage:imref size: NSZeroSize];
            [[(NSStatusItem*)trayicon_item button] setImage: pimg];
         }
         if(trayicon_image)
           CFRelease((CGImageRef)trayicon_image);
         trayicon_image = imref;
       }
       if(params.tooltip.is_defined()) {
          @autoreleasepool {
            [[(NSStatusItem*)trayicon_item button] setToolTip:cvt(params.tooltip)];
          }
       }
       return true;
   }
   bool view::trayicon_remove()
   {
     if(!trayicon_item)
       return false;
     NSStatusBar* sb = [NSStatusBar systemStatusBar];
     [sb removeStatusItem:(NSStatusItem*)trayicon_item];
     [(NSStatusItem*)trayicon_item release];
     trayicon_item = nullptr;
     //if(trayicon_image)
     //  CFRelease((CGImageRef)trayicon_image);
     return true;
   }
   bool view::trayicon_place(rect& rc) {
     if(!trayicon_item)
           return false;
     NSRect nrc = [[(NSStatusItem*)trayicon_item button] frame];
     nrc.origin.x = 0;
     nrc.origin.y = 0;
     nrc = [[(NSStatusItem*)trayicon_item button] convertRect:nrc toView:nil];
     NSRect rectOnScreen = [[[(NSStatusItem*)trayicon_item button] window] convertRectToScreen:nrc];
     rc = flipped_screen_rect(fromns<int>( rectOnScreen ));
     return true;
   }
  
   bool view::is_at_position(point screen_pos)
   {
      NSPoint pt = tons( osx_screen_point(screen_pos) );
      NSView* nv = (NSView*)get_hwnd();
      NSWindow* nw = [nv window];
      auto wid = [NSWindow windowNumberAtPoint:pt belowWindowWithWindowNumber:0 ];
      if( [nw windowNumber] == wid )
        return true;
      for(auto pcw: windows) {
        NSView* ncv = (NSView*)pcw->get_hwnd();
        NSWindow* ncw = [ncv window];
        if( [ncw windowNumber] == wid )
          return true;
      }
      return false;
   }


 
}


static void KeyArrayCallback(const void* key, const void* value, void* context) { CFArrayAppendValue((CFMutableArrayRef)context, key);  }

static bool osx_screen_name_id( NSScreen * pns, tool::ustring& name, int_v& sid )
{
    
    @autoreleasepool {
        NSDictionary* screenDictionary = [pns deviceDescription];
        NSNumber* screenID = [screenDictionary objectForKey:@"NSScreenNumber"];
        CGDirectDisplayID aID = [screenID unsignedIntValue];
        sid = aID;
        
        CFStringRef localName = nullptr;
        io_connect_t displayPort = CGDisplayIOServicePort(aID);
        CF::ref<CFDictionaryRef> dict = IODisplayCreateInfoDictionary(displayPort, 0);
        CFDictionaryRef names = (CFDictionaryRef) CFDictionaryGetValue(dict, CFSTR(kDisplayProductName));
        if(names)
        {
            CF::ref<CFArrayRef> langKeys = CFArrayCreateMutable(kCFAllocatorDefault, 0, &kCFTypeArrayCallBacks );
            CFDictionaryApplyFunction(names, KeyArrayCallback, (void*)langKeys.data());
            CF::ref<CFArrayRef> orderLangKeys = CFBundleCopyPreferredLocalizationsFromArray(langKeys);
            if(orderLangKeys && CFArrayGetCount(orderLangKeys))
            {
                CFStringRef langKey = (CFStringRef)CFArrayGetValueAtIndex(orderLangKeys, 0);
                localName = (CFStringRef)CFDictionaryGetValue(names, langKey);
            }
            name = cvt(localName);
            return true;
        }
        return false;

    }
    
    
}


namespace html {
    
    int number_of_screens() {
        return (int) [[NSScreen screens] count];
    }
    
    int screen_of( html::iwindow* hw ) {
        NSView* pnv = (NSView *) hw->get_hwnd();
        if( !pnv )
            return -1;
        NSWindow *pnw = [pnv window];
        if( !pnw )
            return -1;
        NSScreen * pns = [pnw screen];
        if( !pns )
            return -1;
        NSUInteger idx = [[NSScreen screens] indexOfObject: pns];
        return idx == NSNotFound ? -1 : int(idx);
    }
    
    bool get_screen_info( int n, screen_info& si ) {
    
        if (n < 0 || n >= [[NSScreen screens] count])
           return false;
    
        NSScreen * pns = [[NSScreen screens] objectAtIndex: n];
        if(!pns)
            return false;
        si.monitor = flipped_screen_rect(fromns<int>( [pns frame] ));
        si.workarea = flipped_screen_rect(fromns<int>( [pns visibleFrame] ));
        si.is_primary = n == 0; //[NSScreen mainScreen] == pns;
           // The screen at index 0 in the returned array corresponds to the primary screen of the user’s system.
           // This is the screen that contains the menu bar and whose origin is at the point (0, 0).
        osx_screen_name_id(pns, si.device_name,si.sid);
        return true;
    }
    
    gool::bitmap* get_screen_shot(int n) {
        
        screen_info si;
        if(!get_screen_info( n, si ))
            return nullptr;
        return gool::bitmap::create(CGDisplayCreateImage( si.sid.val(0) ), false);
        
    }
    
    
    bool iwindow::set_topmost(bool on)
    {
      NSView* pnv = (NSView *) get_hwnd();
      if( !pnv )
        return false;
      NSWindow *pnw = [pnv window];
      if( !pnw )
        return false;
      
      if( on ) {
        [pnw setLevel:NSFloatingWindowLevel];
        [pnw orderFront:pnw];
      }
      else
          switch( this->type )
          {
              case html::POPUP_WINDOW:
              case html::TOOL_WINDOW:   [pnw setLevel: NSFloatingWindowLevel]; break;
              case html::DIALOG_WINDOW: [pnw setLevel: NSModalPanelWindowLevel]; break;
              default:
                  [pnw setLevel: NSNormalWindowLevel]; break;
          }
      return true;
    }
    bool iwindow::get_topmost() {
      NSView* pnv = (NSView *) get_hwnd();
      if( !pnv )
        return false;
      NSWindow *pnw = [pnv window];
      if( !pnw )
        return false;
      return [pnw level] == NSFloatingWindowLevel;
    }
    
    EVENT_SOURCE get_current_event_source() {
        auto event = [NSApp currentEvent];
        if(!event)
            return ES_UNAVAILABLE;
        auto type = [event type];
        if( type == NSEventTypeKeyUp || type == NSEventTypeKeyDown )
            return ES_KEYBOARD;
        
        if( type >= NSEventTypeLeftMouseDown && type <= NSEventTypeMouseMoved )
            switch([event subtype])
        {
            case NSEventSubtypeTouch: return ES_TOUCHPAD; // ?ES_TOUCH
            case NSEventSubtypeMouseEvent: return ES_MOUSE;
            default: break;
        }
        return ES_UNAVAILABLE;
    }
}

bool osx_set_clipboard_data(html::clipboard::data *cbdata, NSPasteboard *pasteboard );

namespace osx {


    bool view::activate(bool and_bring_to_front) {
        NSView* pnv = (NSView *) get_hwnd();
        if( !pnv )
            return false;
        NSWindow *pnw = [pnv window];
        if( !pnw )
            return false;
        if(and_bring_to_front) {
            [[NSApplication sharedApplication] activateIgnoringOtherApps:TRUE];
            [pnw makeKeyAndOrderFront:pnw];
        }
        else
            [pnw makeKeyWindow];
        return true;
    }

    bool view::supports_native_ui_window_move() const {
        NSView* pnv = (NSView *) get_hwnd();
        if( !pnv )
            return false;
        NSWindow *pnw = [pnv window];
        if( !pnw )
            return false;
        return [pnw respondsToSelector:@selector(performWindowDragWithEvent:)];
        //return false;
    }
    
    bool view::perform_window_move() {
        if( !currentEvent ) return false;
        NSView* pnv = (NSView *) get_hwnd();
        if( !pnv )
            return false;
        [pnv windowWillMove: nil];
        [[pnv window] performWindowDragWithEvent:(NSEvent*)currentEvent];
        return true;
    }
    
    bool view::do_drag(clipboard::data *pd, uint &ddm /*in out, DD_MODE flags*/, element *src,gool::bitmap *drag_image, point offset)
    {
        NSView* pnv = (NSView *) get_hwnd();
        if( !pnv )
            return false;
        NSWindow *pnw = [pnv window];
        if( !pnw )
            return false;
        NSEvent* pevt = [[NSApplication sharedApplication] currentEvent];
        NSPoint pos = [pevt locationInWindow];
        NSSize  off = {CGFloat(0), CGFloat(0)};
        
        dd_completion = 0;
        dd_operation = ddm;

        pos.x -= offset.x;
        pos.y -= offset.y;
        
        CF::ref<CGImageRef> imref = osx_image( drag_image );

        @autoreleasepool {
       
          NSImage* pimg = [[NSImage alloc] initWithCGImage:imref size: NSZeroSize];
          
          NSPasteboard* pasteboard = [NSPasteboard pasteboardWithName:NSDragPboard];
          if(osx_set_clipboard_data(pd, pasteboard)) {
            [pnw dragImage:pimg at:pos offset:off event:pevt pasteboard:pasteboard source:pnv slideBack:YES];
            ddm = (uint) dd_completion;
            return true;
          }
        }
        return false;
    }

    
    
}


