#include "osx-sciter.h"

#include "html/html-layout-text-analysis.h"
#include "html/html-layout-text-flow.h"
#include "html/html-view.h"

#ifndef CORETEXT_LAYOUT

namespace osx {
    
    using namespace html;
    
    class text_analysis
    : public html::tflow::text_analysis
    {
        
    public:
        // the only public method that wraps text analysis functionality:
        static bool exec(view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes);
        
    private:
        
        text_analysis() {}
        
        bool generate_results(
                              view& v, element* el, slice<hnode> nodes,
                              OUT tool::array<wchar>& text,
                              OUT tool::array<tflow::text_run>& runs,
                              OUT tool::array<tflow::LINE_BREAKPOINT>& breakpoints
                              );
        
        void shape_glyph_runs( view& v, element* elem, tflow::text_flow& tf);
        void shape_glyph_run( view& v, tflow::text_flow& tf, element* elem, uint run_index, IN OUT uint& glyph_start);
        
        void analyze_line_breakpoints();
        void analyze_visuals();
        
    };
    
    
    bool text_analysis::exec(
                             view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes)
    {
        text_analysis inst;
        
        if(inst.generate_results(v,el, nodes,tf._text,tf._runs,tf._breakpoints))
        {
            inst.shape_glyph_runs( v,el, tf);
            return true;
        }
        return false;
    }
    
    bool text_analysis::generate_results(
                                         view& v, element* elem, slice<hnode> nodes,
                                         OUT tool::array<wchar>& text,
                                         OUT tool::array<tflow::text_run>& runs,
                                         OUT tool::array<tflow::LINE_BREAKPOINT>& breakpoints )
    {
        hstyle es = elem->get_style(v);
        ustring lang = elem->get_lang();
        
        _reading_direction = es->direction == direction_rtl?
        tflow::READING_DIRECTION_RIGHT_TO_LEFT:
        tflow::READING_DIRECTION_LEFT_TO_RIGHT;
        
        uint  next_run_index = 0;
        text.clear();
        runs.clear();
        breakpoints.clear();
       
        auto push_run = [&](node* pnode, uint start, uint length, uint dom_start, char_mark marks)
        {
            linked_text_run& run = _runs.push();
            run.node = pnode;
            run.next_run_index   = ++next_run_index;
            run.text_start       = start;
            run.text_length      = length;
            run.bidi_level(pnode->get_style(v)->direction == direction_rtl? 1:0);
            run.text_node_start  = dom_start;
            run.marks            = marks;
        };
        
        if( elem->c_style->collapse_ws() )
        {
            auto receiver = [&](wchars span, char_marks marks, node* pnode)
            {
                uint start = static_cast<uint>(text.length());
                if( span.length == 1 && span[0] == 0)
                {
                    text.push(span); // inline block
                    push_run(pnode, start, static_cast<::uint32>(text.length()) - start,0, char_mark());
                }
                else
                    tflow::visible_chars_collapse_ws(pnode,span,marks,text,push_run,lang);
            };
            tflow::flatten_nodes(v,elem, nodes, receiver);
        }
        else
        {
            auto receiver = [&](wchars span, char_marks marks, node* pnode)
            {
                uint start = static_cast<uint>(text.length());
                if( span.length == 1 && span[0] == 0)
                {
                    text.push(span); // inline block
                    push_run(pnode, start, static_cast<uint>(text.length()) - start,0, char_mark());
                }
                else
                    tflow::visible_chars_break_nl(pnode,span,marks,text,push_run,lang);
                
                //push_run(pnode, start, static_cast<uint32>(text.length()) - start,0);
            };
            tflow::flatten_nodes(v,elem, nodes, receiver);
        }
        
        if(text.length() == 0
           && nodes.length == 1
           && nodes[0]->is_text()
           && nodes[0].ptr_of<html::text>()->chars.length() == 0)
        {
            // special case for empty paragraphs with forced flow:text
            // note: needed for content-editable
            text.push(0x200B /* zero width space*/);
            push_run(nodes[0],0,1,0,char_mark());
        }
        
        _text = text();
        if(_text.length == 0)
            return true;
        
        // Analyzes the text using each of the analyzers and returns
        // their results as a series of runs.
        
        try
        {
            // Initially start out with one result that covers the entire range.
            // This result will be subdivided by the analysis processes.
            
            // Allocate enough room to have one breakpoint per code unit.
            _breakpoints.length(_text.length);
            
            // Call each of the analyzers in sequence, recording their results.
            
            analyze_line_breakpoints();
            analyze_visuals();
            
            //if (FAILED(hr = textAnalyzer->AnalyzeLineBreakpoints(   this, 0, UINT32(_text.length), this))) return false;
            //if (FAILED(hr = textAnalyzer->AnalyzeBidi(              this, 0, UINT32(_text.length), this))) return false;
            //if (FAILED(hr = textAnalyzer->AnalyzeScript(            this, 0, UINT32(_text.length), this))) return false;
            
            
            for (index_t i = _runs.last_index(); i >= 0; --i) // new runs (if any) will have numbers total_runs
            {
                //uint rl = (uint)_runs.length();
                linked_text_run& r = _runs[i];
                html::style *rs = r.get_style(v);
                if(r.is_inline_block_pos(_text))
                {
                    r.used_font = v.get_font(rs);
                    r.no_visual(false);
                    //r.script.shapes = DWRITE_SCRIPT_SHAPES_NO_VISUAL;
                    continue;
                }
                
                if(rs != es && !rs->can_wrap() && r.text_length)
                {
                    uint nm = r.text_start + r.text_length - 1;
                    for(uint n = r.text_start + 1; n < nm; ++n)
                    {
                        tflow::LINE_BREAKPOINT &bp = _breakpoints[n];
                        if(bp.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
                            bp.break_condition_after = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
                        if(bp.break_condition_before != tflow::BREAK_CONDITION_MUST_BREAK)
                            bp.break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
                    }
                    tflow::LINE_BREAKPOINT &bp1 = _breakpoints[r.text_start];
                    tflow::LINE_BREAKPOINT &bp2 = _breakpoints[nm];
                    if(bp1.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
                        bp1.break_condition_after = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
                    if(bp2.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
                        bp2.break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
                }
                
                wchars run_chars = r.chars(_text.start);
                
                //if(!setup_used_run_font(v, rs, r, run_chars, lang))
                //  view::debug_printf(OT_DOM, OS_ERROR,"Font fallback failure for font %S fallback %S.\n",
                //    r.decl_font->name.c_str(),
                //    r.used_font->name.c_str());
                
                //r.decl_font = v.get_font(rs);
                handle<osx::font> decl_font = static_cast<osx::font*>(v.get_font(rs));
                
                // check each character in the run for supported glyphs, split the run and set substitution font.
                
                while( run_chars.length )
                {
                    const wchar* start = run_chars.start;
                    const wchar* end = start;
                    
                    uint ucp = utf16::getc(run_chars);
                    
                    if(decl_font->has_glyph_for(ucp))
                        continue;
                    end = run_chars.start;
                    
                    handle<gool::font> fb_font;
                    wchars font_list = decl_font->name;
                    if(!app()->get_used_font(fb_font,font_list,decl_font,lang,writing_script(ucp),ucp))
                        continue;
                    
                    while(run_chars.length)
                    {
                        end = run_chars.start;
                        if(!fb_font->has_glyph_for(utf16::getc(run_chars)))
                            break;
                        end = run_chars.start;
                    }
                    uint text_position = uint(start - _text.start);
                    uint text_length   = uint(end - start);
                    set_current_run(text_position);
                    split_current_run(text_position);
                    while (text_length > 0)
                    {
                        linked_text_run& run  = fetch_next_run(&text_length);
                        run.used_font = fb_font;
                    }
                }
            }
            
            // Exchange our results with the caller's.
            breakpoints.swap(_breakpoints);
            
            size_t total_runs = _runs.length();
            
            // Resequence the resulting runs in order before returning to caller.
            runs.length(total_runs);
            
            uint nextRunIndex = 0;
            for (int i = 0; i < int(total_runs); ++i)
            {
                runs[i]         = _runs[nextRunIndex];
                //wchars tt = runs[i].chars(_text.start);
                nextRunIndex    = _runs[nextRunIndex].next_run_index;
            }
            
        }
        catch (...)
        {
            return false;
        }
        return true;
    }
    
    // VERY basic
    
    void text_analysis::analyze_line_breakpoints() {
        wchars text = _text;
        tflow::LINE_BREAKPOINT* bp = _breakpoints.begin();
        
        tflow::LINE_BREAKPOINT _pbp = {0};
        tflow::LINE_BREAKPOINT* pbp = &_pbp;
        
        bool p_is_punct = false;
        
        for(wchar pc = 0; !!text; ++text )
        {
            wchar c = *text;
            bool is_punct = ucispunct(c);
            bp->is_whitespace = ucisspace(*text);
            bp->is_soft_hyphen = *text == 0x00AD;
            bp->break_condition_after = bp->break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
            if( pbp->is_whitespace && !bp->is_whitespace ) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            } else if( pc == '\r' && c != '\n' ) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            } else if( pc == '\n' ) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            else if( p_is_punct && !is_punct ) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            pbp = bp++;
            pc = c;
            p_is_punct = is_punct;
        }
    }
    
    void text_analysis::analyze_visuals() {
        wchars text = _text;
        
        for (index_t i = _runs.last_index(); i >= 0; --i) // new runs (if any) will have numbers total_runs
        {
            linked_text_run& r = _runs[i];
            if(r.is_inline_block_pos(_text))
            {
                r.no_visual(false);
                continue;
            }
            
            wchars run_chars = r.chars(_text.start);
            
            while( run_chars.length )
            {
                const wchar* start = run_chars.start;
                const wchar* end = start;
                
                uint ucp = utf16::getc(run_chars);
                if( ucp >= ' ' )
                    continue;
                
                end = run_chars.start;
                
                while(run_chars.length)
                {
                    end = run_chars.start;
                    if(utf16::getc(run_chars) >= ' ')
                        break;
                    end = run_chars.start;
                }
                uint text_position = uint(start - _text.start);
                uint text_length   = uint(end - start);
                set_current_run(text_position);
                split_current_run(text_position);
                while (text_length > 0)
                {
                    linked_text_run& run  = fetch_next_run(&text_length);
                    run.no_visual(true);
                }
            }
        }
    }
    
    
    uint estimate_glyph_count(uint text_length)
    {
        return 3 * text_length / 2 + 16;
    }

    struct glyphs_shaper
    {
      view& v;
      element* elem;
      tflow::text_flow& tf;

      style* cs; 

      CF::ref<CTFontRef>       ctFont;
      CF::ref<CFDictionaryRef> ctAttributesDict;
      //CF::ref<CTFontRef>        

      glyphs_shaper(view& v_, element* elem_, tflow::text_flow& tf_): v(v_),elem(elem_),tf(tf_), cs(0) {}

      //void text_analysis::shape_glyph_run(uint run_index, IN OUT uint& glyph_start) {
      //}



    };


    
    void text_analysis::shape_glyph_runs( view& v, element* elem, tflow::text_flow& tf)
    {
        // Shapes all the glyph runs in the layout.
        
        uint text_length = static_cast<uint>(tf._text.length());
        if(!text_length)
            return;
        
        // Estimate the maximum number of glyph indices needed to hold a string.
        uint estimated_glyph_count = estimate_glyph_count(text_length);
        
        tf._glyph_indices.size(estimated_glyph_count);
        tf._glyph_offsets.size(estimated_glyph_count);
        tf._glyph_advances.size(estimated_glyph_count);
        tf._glyph_clusters.size(text_length);
        
        uint glyph_start = 0;
        
        //uint total = _runs.length();
        
        // Shape each run separately. This is needed whenever script, locale,
        // or reading direction changes.
        for (uint runIndex = 0; runIndex < tf._runs.length(); ++runIndex)
            shape_glyph_run(v,tf,elem, runIndex, glyph_start);
        
        tf._glyph_indices.size(glyph_start);
        tf._glyph_offsets.size(glyph_start);
        tf._glyph_advances.size(glyph_start);
        tf._glyph_justified_advances = tf._glyph_advances;
        
    }
    
#if 1
    
    void text_analysis::shape_glyph_run(
                                        view& v, tflow::text_flow& tf, element* elem,
                                        uint run_index,
                                        IN OUT uint& glyph_start
                                        )
    {
        // Shapes a single run of text into glyphs.
        // Alternately, you could iteratively interleave shaping and line
        // breaking to reduce the number glyphs held onto at once. It's simpler
        // for this demostration to just do shaping and line breaking as two
        // separate processes, but realize that this does have the consequence that
        // certain advanced fonts containing line specific features (like Gabriola)
        // will shape as if the line is not broken.
        
        
        tflow::text_run& run  = tf._runs[run_index];
        
        uint text_start        = run.text_start;
        uint text_length       = run.text_length;
        uint maxglyph_count    = static_cast<uint>(tf._glyph_indices.size() - glyph_start);
        uint actual_glyph_count = 0;
        
        element* el = run.get_element();
        
        if( run.is_inline_block_pos(tf._text()))
        {
            el->check_layout(v);
            //assert(el->is_inline_block_element(v));
            run.glyph_start          = glyph_start;
            run.glyph_count          = 1;
            if (text_length > maxglyph_count)
            {
                uint totalGlyphsArrayCount = glyph_start + 1;
                tf._glyph_indices.size(totalGlyphsArrayCount);
            }
            tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_advances.length()));
            tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_offsets.length()));
            tf._glyph_advances[glyph_start] = static_cast<float>(el->min_inline_margin_box_width(v));
            tf._glyph_indices[glyph_start] = 0;
            glyph_start    += 1;
            return;
        }
        
        
        ustring  locale_name = el->get_lang();
        //html::style* style = el->get_style(v);
        osx::font* pf = static_cast<osx::font*>(run.get_used_font(v));
        
        //float  font_em_size = run.decl_font->size;
        
        //pf = used_font_face(pf,run);
        //d2d::asset<IDWriteFontFace> font_face = dw_font_face(pf, run);
        
        run.glyph_start          = glyph_start;
        run.glyph_count          = 0;
        
        if (text_length == 0)
            return; // Nothing to do..
        
        ////////////////////
        // Allocate space for shaping to fill with glyphs and other information,
        // with about as many glyphs as there are text characters. We'll actually
        // need more glyphs than codepoints if they are decomposed into separate
        // glyphs, or fewer glyphs than codepoints if multiple are substituted
        // into a single glyph. In any case, the shaping process will need some
        // room to apply those rules to even make that determintation.
        
        if (text_length > maxglyph_count)
        {
            maxglyph_count = estimate_glyph_count(text_length);
            uint totalGlyphsArrayCount = glyph_start + maxglyph_count;
            tf._glyph_indices.size(totalGlyphsArrayCount);
        }
        
        //juce::String text = juce::String(&tf._text[text_start],text_length);
        //Array <int> glyphs;
        //Array <float> xOffsets;
        
        //pf->juce_font()->getGlyphPositions(text,glyphs,xOffsets);
        
        //
        
        actual_glyph_count = osx_get_glyph_indices( pf, tf._text(text_start, text_start + text_length), &tf._glyph_indices[glyph_start], maxglyph_count );
        
        assert(actual_glyph_count);
        
        if( run.bidi_level() & 1 )
            std::reverse( &tf._glyph_indices[glyph_start], &tf._glyph_indices[glyph_start+actual_glyph_count] );
        
        
        tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_advances.length()));
        tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_offsets.length()));
        
        osx_get_glyph_advances( pf, tf._glyph_indices(glyph_start, glyph_start + actual_glyph_count), &tf._glyph_advances[glyph_start] );
        
        /*tflow::GLYPH_OFFSET zgo = {0};
         
         float xoff = xOffsets[0];
         
         for( uint i = 0; i < actual_glyph_count; ++i ) {
         float xoff_next = xOffsets.getUnchecked(i+1);
         tf._glyph_advances[i + glyph_start] = xoff_next - xoff; xoff = xoff_next;
         tf._glyph_offsets[i + glyph_start] = zgo;
         tf._glyph_indices[i + glyph_start] = glyphs[i];
         }*/
        
        for( uint i = 0; i < text_length; ++i )
            tf._glyph_clusters[text_start + i] = i;
        
        ////////////////////
        // Certain fonts, like Batang, contain glyphs for hidden control
        // and formatting characters. So we'll want to explicitly force their
        // advance to zero.
        if (run.no_visual())
            tf._glyph_advances.set(glyph_start, glyph_start + actual_glyph_count, 0.0f);
        
        ////////////////////
        // Set the final glyph count of this run and advance the starting glyph.
        run.glyph_count = actual_glyph_count;
        glyph_start    += actual_glyph_count;
    }
#else
    void text_analysis::shape_glyph_run(
                                        view& v, tflow::text_flow& tf, element* elem,
                                        uint run_index,
                                        IN OUT uint& glyph_start
                                        )
    {
        // Shapes a single run of text into glyphs.
        // Alternately, you could iteratively interleave shaping and line
        // breaking to reduce the number glyphs held onto at once. It's simpler
        // for this demostration to just do shaping and line breaking as two
        // separate processes, but realize that this does have the consequence that
        // certain advanced fonts containing line specific features (like Gabriola)
        // will shape as if the line is not broken.
        
        
        tflow::text_run& run  = tf._runs[run_index];
        
        uint text_start        = run.text_start;
        uint text_length       = run.text_length;
        uint maxglyph_count    = static_cast<uint>(tf._glyph_indices.size() - glyph_start);
        //uint actual_glyph_count = 0;
        
        element* el = run.get_element();
        
        if( run.is_inline_block_pos(tf._text()))
        {
            el->check_layout(v);
            //assert(el->is_inline_block_element(v));
            run.glyph_start          = glyph_start;
            run.glyph_count          = 1;
            if (text_length > maxglyph_count)
            {
                uint totalGlyphsArrayCount = glyph_start + 1;
                tf._glyph_indices.size(totalGlyphsArrayCount);
            }
            tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_advances.length()));
            tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_offsets.length()));
            tf._glyph_advances[glyph_start] = static_cast<float>(el->min_inline_margin_box_width(v));
            tf._glyph_indices[glyph_start] = 0;
            glyph_start    += 1;
            return;
        }
        
        
        //ustring  locale_name = el->get_lang();
        //html::style* style = el->get_style(v);
        osx::font* pf = static_cast<osx::font*>(run.get_used_font(v));

        
        CTFontRef                base_ctFont = pf->ct_font();
        CF::ref<CFDictionaryRef> base_attributes;
        
        CTWritingDirection direction = (run.bidi_level() & 1) ? kCTWritingDirectionRightToLeft: kCTWritingDirectionLeftToRight;
        
        CTParagraphStyleSetting settings[] = { { kCTParagraphStyleSpecifierBaseWritingDirection, sizeof(direction), &direction } };
        
        CF::ref<CTParagraphStyleRef> para_attributes = CTParagraphStyleCreate(settings, items_in(settings));
        
        
        const void *keys[] = { kCTFontAttributeName, kCTParagraphStyleAttributeName, nullptr };
        const void *vals[] = { base_ctFont, para_attributes.data(), nullptr };
        
        base_attributes = ::CFDictionaryCreate(NULL,
                                               keys,
                                               vals,
                                               2, // count of attributes
                                               &kCFCopyStringDictionaryKeyCallBacks,
                                               &kCFTypeDictionaryValueCallBacks);
        
        
        CF::ref<CFStringRef> string = ::CFStringCreateWithCharactersNoCopy(kCFAllocatorDefault, (const UniChar*)(_text.start + text_start), text_length, kCFAllocatorNull);
        
        // Now we can create an attributed string
        CF::ref<CFAttributedStringRef> attrStringObj = ::CFAttributedStringCreate(kCFAllocatorDefault, string, base_attributes);

        // Create the CoreText line from our string, then we're done with it
        CF::ref<CTLineRef> line = ::CTLineCreateWithAttributedString(attrStringObj);
        // and finally retrieve the glyph data and store into the gfxTextRun
        CFArrayRef glyphRuns = ::CTLineGetGlyphRuns(line);
        uint32_t numRuns = (uint32_t)::CFArrayGetCount(glyphRuns);
                                                                           
        //assert(numRuns == 1);
                                                                           
        CTRunRef ctRun = (CTRunRef)::CFArrayGetValueAtIndex(glyphRuns, 0);
        CFRange  ctRange = CTRunGetStringRange(ctRun);
        uint     runGlyphCount = (uint)CTRunGetGlyphCount(ctRun);
                                                                           
        run.glyph_start          = glyph_start;
        run.glyph_count          = runGlyphCount;
                                                                           
        if (text_length == 0)
            return; // Nothing to do..
        
        tf._glyph_indices.size( glyph_start + runGlyphCount );
        tf._glyph_advances.size ( glyph_start + runGlyphCount );
        tf._glyph_offsets.size( glyph_start + runGlyphCount );
        
        //actual_glyph_count = osx_get_glyph_indices( pf, tf._text(text_start, text_start + text_length), &tf._glyph_indices[glyph_start], maxglyph_count );
                                                                           
        CGSize advances[runGlyphCount];
        CGGlyph glyphs[runGlyphCount];
        CTRunGetAdvances(ctRun, CFRangeMake(0, 0), advances);
        CTRunGetGlyphs(ctRun, CFRangeMake(0, 0), glyphs);
                                                                           
        for(uint i = 0; i < runGlyphCount; ++i)
        {
           tf._glyph_indices[glyph_start + i] = glyphs[i];
           tf._glyph_advances[glyph_start + i] = advances[i].width;
           tf._glyph_offsets[glyph_start + i] = tflow::GLYPH_OFFSET();
        }
       
        //if( run.bidi_level() & 1 )
        //    std::reverse( tf._glyph_indices.begin() + glyph_start, tf._glyph_indices.begin() + glyph_start + runGlyphCount );
        
        
        //osx_get_glyph_advances( pf, tf._glyph_indices(glyph_start, glyph_start + actual_glyph_count), &tf._glyph_advances[glyph_start] );
        
        /*tflow::GLYPH_OFFSET zgo = {0};
         
         float xoff = xOffsets[0];
         
         for( uint i = 0; i < actual_glyph_count; ++i ) {
         float xoff_next = xOffsets.getUnchecked(i+1);
         tf._glyph_advances[i + glyph_start] = xoff_next - xoff; xoff = xoff_next;
         tf._glyph_offsets[i + glyph_start] = zgo;
         tf._glyph_indices[i + glyph_start] = glyphs[i];
         }*/
        
        for( uint i = 0; i < text_length; ++i )
            tf._glyph_clusters[text_start + i] = i;
        
        ////////////////////
        // Certain fonts, like Batang, contain glyphs for hidden control
        // and formatting characters. So we'll want to explicitly force their
        // advance to zero.
        if (run.no_visual())
            tf._glyph_advances.set(glyph_start, glyph_start + runGlyphCount, 0.0f);
        
        ////////////////////
        // Set the final glyph count of this run and advance the starting glyph.
        glyph_start    += run.glyph_count;
    }
    
#endif
    
    
    void view::setup_text_flow( html::element* elem, html::tflow::text_flow& tf, tool::slice< tool::handle<html::node> > nodes ) 
    {
        text_analysis::exec(*this,elem, tf, nodes);
    }
    
}

#endif

