
#include "osx-sciter.h"
#include "osx-sciter-application.h"

#include "html/html-layout-text-analysis.h"
#include "html/html-layout-text-flow.h"

#include <algorithm>

#if defined(CORETEXT_LAYOUT) 


namespace osx {
    
    using namespace html;
    
    class text_analysis
    : public html::tflow::text_analysis
    {
        
    public:
        // the only public method that wraps text analysis functionality:
        static bool exec(view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes);
        
    private:
        
        text_analysis() {}
        
        bool generate_results(
                              view& v,
                              element* el,
                              tflow::text_flow& tf,
                              slice<hnode> nodes );
        
        //void shape_glyph_runs( view& v, element* elem, tflow::text_flow& tf);
        //void shape_glyph_run( view& v, tflow::text_flow& tf, element* elem, uint run_index, IN OUT uint& glyph_start);
        
        void analyze_line_breakpoints();
        //void analyze_visuals();
        
        
        // cached font descriptor, created the first time it's needed
        static CTFontDescriptorRef get_default_features_descriptor() {
            static CF::ref<CTFontDescriptorRef>    default_features_descriptor = create_default_features_descriptor();
            return default_features_descriptor;
        }
        static CTFontDescriptorRef create_default_features_descriptor();
        
    };
    
    
    // Construct the font attribute descriptor that we'll apply by default when creating a CTFontRef.
    // This will turn off line-edge swashes by default, because we don't know the actual line breaks
    // when doing glyph shaping.
    CTFontDescriptorRef text_analysis::create_default_features_descriptor()
    {
        CTFontDescriptorRef res = nullptr;
        
        SInt16 val;
        
        val = kSmartSwashType;
        CFNumberRef swashesType = ::CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt16Type, &val);
        
        val = kLineInitialSwashesOffSelector;
        CFNumberRef lineInitialsOffSelector = ::CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt16Type, &val);
        
        CFTypeRef keys[]   = { kCTFontFeatureTypeIdentifierKey, kCTFontFeatureSelectorIdentifierKey };
        CFTypeRef values[] = { swashesType, lineInitialsOffSelector };
        
        CFDictionaryRef featureSettings[2];
        
        featureSettings[0] =  ::CFDictionaryCreate(kCFAllocatorDefault,
                                                   (const void **) keys,
                                                   (const void **) values,
                                                   items_in(keys),
                                                   &kCFTypeDictionaryKeyCallBacks,
                                                   &kCFTypeDictionaryValueCallBacks);
        
        ::CFRelease(lineInitialsOffSelector);
        
        val = kLineFinalSwashesOffSelector;
        CFNumberRef lineFinalsOffSelector = ::CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt16Type, &val);
        values[1] = lineFinalsOffSelector;
        featureSettings[1] = ::CFDictionaryCreate(kCFAllocatorDefault,
                                                  (const void **) keys,
                                                  (const void **) values,
                                                  items_in(keys),
                                                  &kCFTypeDictionaryKeyCallBacks,
                                                  &kCFTypeDictionaryValueCallBacks);
        ::CFRelease(lineFinalsOffSelector);
        ::CFRelease(swashesType);
        
        CFArrayRef featuresArray = ::CFArrayCreate(kCFAllocatorDefault,
                                                   (const void **) featureSettings,
                                                   items_in(featureSettings),
                                                   &kCFTypeArrayCallBacks);
        ::CFRelease(featureSettings[0]);
        ::CFRelease(featureSettings[1]);
        
        const CFTypeRef attrKeys[]   = { kCTFontFeatureSettingsAttribute };
        const CFTypeRef attrValues[] = { featuresArray };
        CFDictionaryRef attributesDict = ::CFDictionaryCreate(kCFAllocatorDefault,
                                                              (const void **) attrKeys,
                                                              (const void **) attrValues,
                                                              items_in(attrKeys),
                                                              &kCFTypeDictionaryKeyCallBacks,
                                                              &kCFTypeDictionaryValueCallBacks);
        ::CFRelease(featuresArray);
        
        res = ::CTFontDescriptorCreateWithAttributes(attributesDict);
        ::CFRelease(attributesDict);
        return res;
    }
    
    
    
    
    
    bool text_analysis::exec( view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes)
    {
        text_analysis inst;
        
        if(inst.generate_results(v,el,tf, nodes))
        {
            //inst.shape_glyph_runs( v,el, tf);
            return true;
        }
        return false;
    }
    
    bool text_analysis::generate_results(view& v, element* elem, tflow::text_flow& tf, slice<hnode> nodes)
    {
        //tool::array<wchar>& text = tf._text;
        //tool::array<tflow::text_run>& runs = tf._runs;
        //tool::array<tflow::LINE_BREAKPOINT>& breakpoints = tf._breakpoints;
        
        
        hstyle es = elem->get_style(v);
        ustring lang = elem->get_lang();
        
        _reading_direction = es->direction == direction_rtl?
        tflow::READING_DIRECTION_RIGHT_TO_LEFT:
        tflow::READING_DIRECTION_LEFT_TO_RIGHT;
        
        uint  next_run_index = 0;
        tf._text.clear();
        tf._runs.clear();
        tf._breakpoints.clear();
        
      
        auto push_run = [&](node* pnode, uint start, uint length, uint dom_start, char_mark marks)
        {
            linked_text_run& run = _runs.push();
            run.node = pnode;
            run.next_run_index   = ++next_run_index;
            run.text_start       = start;
            run.text_length      = length;
            run.bidi_level(pnode->get_style(v)->direction == direction_rtl? 1:0);
            run.text_node_start  = dom_start;
            run.marks            = marks;
        };
        
        if( elem->c_style->collapse_ws() )
        {
            auto receiver = [&](wchars span, char_marks marks,node* pnode)
            {
                uint start = static_cast<uint>(tf._text.length());
                if( span.length == 1 && span[0] == 0)
                {
                    tf._text.push(span); // inline block
                    push_run(pnode, start, static_cast<::uint32>(tf._text.length()) - start,0,char_mark());
                }
                else
                    tflow::visible_chars_collapse_ws(pnode,span,marks,tf._text,push_run,lang);
            };
            tflow::flatten_nodes(v,elem, nodes, receiver);
        }
        else
        {
            auto receiver = [&](wchars span, char_marks marks, node* pnode)
            {
                uint start = static_cast<uint>(tf._text.length());
                if( span.length == 1 && span[0] == 0)
                {
                    tf._text.push(span); // inline block
                    push_run(pnode, start, static_cast<uint>(tf._text.length()) - start,0,char_mark());
                }
                else
                    tflow::visible_chars_break_nl(pnode,span,marks,tf._text,push_run,lang);
                
                //push_run(pnode, start, static_cast<uint32>(text.length()) - start,0);
            };
            tflow::flatten_nodes(v,elem, nodes, receiver);
        }
        
        if(tf._text.length() == 0
           && nodes.length == 1
           && nodes[0]->is_text()
           && nodes[0].ptr_of<html::text>()->chars.length() == 0)
        {
            // special case for empty paragraphs with forced flow:text
            // note: needed for content-editable
            tf._text.push(0x200B /* zero width space*/);
            push_run(nodes[0],0,1,0, char_mark());
        }
        
        _text = tf._text();
        if(_text.length == 0)
            return true;
        
        tf._glyph_clusters.length(_text.length,0);
        
        tool::array<uint16>       glyph_indices;
        tool::array<float>        glyph_advances;
        tool::array<tflow::GLYPH_OFFSET> glyph_offsets;
        
        /// +++++++++++++++
        
        osx::font* pf = static_cast<osx::font*>(v.get_font(es));
        
        bool                     isRightToLeft = _reading_direction == tflow::READING_DIRECTION_RIGHT_TO_LEFT;
        
        CTFontRef                base_ctFont = pf->ct_font();
        CF::ref<CFDictionaryRef> base_attributes;
        
        CTWritingDirection direction = isRightToLeft ? kCTWritingDirectionRightToLeft: kCTWritingDirectionLeftToRight;
        
        CTParagraphStyleSetting settings[] = {
            { kCTParagraphStyleSpecifierBaseWritingDirection, sizeof(direction), &direction }
        };
        
        CF::ref<CTParagraphStyleRef> para_attributes = CTParagraphStyleCreate(settings, items_in(settings));
       
        
        const void *keys[] = { kCTFontAttributeName, kCTParagraphStyleAttributeName, nullptr };
        const void *vals[] = { base_ctFont, para_attributes.data(), nullptr };
        
        base_attributes = ::CFDictionaryCreate(NULL,
                                               keys,
                                               vals,
                                               2, // count of attributes
                                               &kCFCopyStringDictionaryKeyCallBacks,
                                               &kCFTypeDictionaryValueCallBacks);
        
        
        // Create a CFAttributedString with text and style info, so we can use CoreText to lay it out.
        
        
        // If there's a possibility of any bidi, we wrap the text with direction overrides
        // to ensure neutrals or characters that were bidi-overridden in HTML behave properly.
        //const UniChar beginLTR[]    = { 0x202d, 0x20 };
        //const UniChar beginRTL[]    = { 0x202e, 0x20 };
        //const UniChar endBidiWrap[] = { 0x20, 0x2e, 0x202c };
        
        CF::ref<CFStringRef> string = ::CFStringCreateWithCharactersNoCopy(kCFAllocatorDefault,
                                          reinterpret_cast<const UniChar*>(_text.start), _text.length, kCFAllocatorNull);
        
        
        // Now we can create an attributed string
        CF::ref<CFAttributedStringRef> attrString = ::CFAttributedStringCreate(kCFAllocatorDefault, string, base_attributes);
        CF::ref<CFMutableAttributedStringRef> attrStringObj = //::CFMutableAttributedStringCreate(kCFAllocatorDefault, stringObj, attrObj);
                                                              CFAttributedStringCreateMutableCopy(NULL, 0, attrString);
        
        CFAttributedStringBeginEditing(attrStringObj);
        
        CF::ref<CFStringRef> attrN = CFSTR("runNo");
        
        for(int n = 0; n < _runs.size(); ++n )
        {
           auto& r = _runs[n];
           CF::ref<CFNumberRef> valN = CFNumberCreate(NULL, kCFNumberIntType, &n);
           CFAttributedStringSetAttribute(attrStringObj, CFRangeMake(r.text_start, r.text_length), attrN, valN);
            
           osx::font* rf = static_cast<osx::font*>(r.get_used_font(v));
           //if( rf != pf )
           CFAttributedStringSetAttribute(attrStringObj, CFRangeMake(r.text_start, r.text_length), kCTFontAttributeName, rf->ct_font());
           //r.used_font = rf;
        }

        CFAttributedStringEndEditing(attrStringObj);
        
        
        // Create the CoreText line from our string, then we're done with it
        CF::ref<CTLineRef> line = ::CTLineCreateWithAttributedString(attrStringObj);
        
        // and finally retrieve the glyph data and store into the gfxTextRun
        CFArrayRef glyphRuns = ::CTLineGetGlyphRuns(line);
        uint32_t numRuns = (uint32_t)::CFArrayGetCount(glyphRuns);
        
        
        uint glyph_position = 0;
        
        //uint nRunsBefore = tf._runs.length();
        osx::application* papp = static_cast<osx::application*>(v.app.ptr());
        
        assert( numRuns >= _runs.size() );
        
        double line_width = 0;
        
        double line_advances = 0;
        
        // Iterate through the glyph runs.
        for (uint32_t runIndex = 0; runIndex < numRuns; runIndex++) {
            CTRunRef ctRun = (CTRunRef)::CFArrayGetValueAtIndex(glyphRuns, runIndex);
            CFRange  ctRange = CTRunGetStringRange(ctRun);
            
            uint runGlyphCount = (uint)CTRunGetGlyphCount(ctRun);
            
            CGSize advances[runGlyphCount];
            //CGPoint positions[runGlyphCount];
            CGGlyph glyphs[runGlyphCount];
            
            // Grab the glyphs, positions, and font
            CTRunGetAdvances(ctRun, CFRangeMake(0, 0), advances);
            //CTRunGetPositions(ctRun, CFRangeMake(0, 0), positions);
            CTRunGetGlyphs(ctRun, CFRangeMake(0, 0), glyphs);
            CFDictionaryRef attributes = CTRunGetAttributes(ctRun);
            CTFontRef runFont = (CTFontRef) CFDictionaryGetValue(attributes, kCTFontAttributeName);
            
            byte rtl = (CTRunGetStatus(ctRun) & kCTRunStatusRightToLeft) ? 1 : 0;
            
            uint text_position = uint(ctRange.location);
            uint text_length = uint(ctRange.length);
            
            uint glyph_run_position = glyph_position;
            set_current_run(text_position);
            split_current_run(text_position);
            while(text_length > 0)
            {
                linked_text_run& run = fetch_next_run(&text_length);
                run.glyph_start = glyph_position;
                run.glyph_count = runGlyphCount; // ????????????????????????????????
                //run.script_analysis.fRTL = !!CTRunGetStatus(ctRun) & kCTRunStatusRightToLeft;
                //run.script_analysis.fLayoutRTL = run.script_analysis.fRTL;
                run.bidi_level(rtl);
                
                osx::font* rf = static_cast<osx::font*>(run.get_used_font(v));
                
                if( runFont != rf->ct_font() ) {
                  run.used_font = papp->register_font(runFont);
                  rf = static_cast<osx::font*>(run.used_font.ptr());
                }

                assert(text_length == 0);
                assert(runGlyphCount);
                
                glyph_indices.length( glyph_position + runGlyphCount );
                glyph_advances.length( glyph_position + runGlyphCount );
                glyph_offsets.length( glyph_position + runGlyphCount );
                
                for(uint i = 0; i < runGlyphCount; ++i)
                {
                    glyph_indices[glyph_position + i] = glyphs[i];
                    glyph_advances[glyph_position + i] = advances[i].width;
                    glyph_offsets[glyph_position + i] = tflow::GLYPH_OFFSET();
                }
                
                
                //if( rtl ) {
//                    for(int i = 0; i < runGlyphCount; ++i)
//                        printf("%d ",clusters[i]);
//                    printf("\n");
                //    //for(int i = runGlyphCount - 1; i >= 0; --i)
                //    //    tf._glyph_clusters[text_position + clusters[i]] = clusters[i];
                //
                //}
                
                //CFIndex clusters[runGlyphCount];
                //CTRunGetStringIndices(ctRun, CFRangeMake(0, 0), clusters );

                //if( !rtl )
                //    for(int i = runGlyphCount - 1; i >= 0 ; --i)
                //        //    tf._glyph_clusters[i] = i;
                //        tf._glyph_clusters[clusters[i]] = i;
                //else
                //    for(int i = runGlyphCount - 1; i >= 0 ; --i)
                //        //    tf._glyph_clusters[i] = i;
                //        tf._glyph_clusters[clusters[i]] = runGlyphCount - 1 - i;
                
                
                
            }
            

            if(rtl) {
                std::reverse(&glyph_indices[glyph_position],glyph_indices.begin() + glyph_position + runGlyphCount);
                std::reverse(&glyph_advances[glyph_position],glyph_advances.begin() + glyph_position + runGlyphCount);
                std::reverse(&glyph_offsets[glyph_position],glyph_offsets.begin() + glyph_position + runGlyphCount);
            }
            glyph_position += runGlyphCount;
        }
        
        size_t total_runs = _runs.length();
        tf._runs.length(total_runs);
        next_run_index = 0;
        
        assert (tf._glyph_clusters.length() == _text.length);
        
        glyph_position = 0;
        
        tf._glyph_indices.reserve(glyph_indices.length());
        tf._glyph_advances.reserve(glyph_advances.length());
        tf._glyph_offsets.reserve(glyph_offsets.length());
  
        for( uint i = 0; i < total_runs; ++i )
        {
            tf._runs[i] = _runs[next_run_index];
            next_run_index = _runs[next_run_index].next_run_index;
            uint tl = tf._runs[i].text_length;
            uint ts = tf._runs[i].text_start;
            
            auto& tr = tf._runs[i];
            
           
            for(uint n = 0; n < tl; ++n)
               tf._glyph_clusters[ts + n] = n;

            //printf("%d/%d ", tr.glyph_start, tr.glyph_count);
            tf._glyph_indices.push(glyph_indices(tr.glyph_start,tr.glyph_start + tr.glyph_count));
            tf._glyph_advances.push(glyph_advances(tr.glyph_start,tr.glyph_start + tr.glyph_count));
            tf._glyph_offsets.push(glyph_offsets(tr.glyph_start,tr.glyph_start + tr.glyph_count));
            
            tr.glyph_start = glyph_position;
            glyph_position += tr.glyph_count;
            
        }
        //printf("\n");
        
        
        //if( isRightToLeft ) {
        //  for(int i = 0; i < _text.size(); ++i)
        //     printf("%d ",tf._glyph_clusters[i]);
        //  printf("\n");
        //    //for(int i = runGlyphCount - 1; i >= 0; --i)
        //    //    tf._glyph_clusters[text_position + clusters[i]] = clusters[i];
        //}
        
        _breakpoints.length(_text.length);
        analyze_line_breakpoints();

        tf._glyph_justified_advances = tf._glyph_advances();
        tf._breakpoints.swap(_breakpoints);
        
        
        return true;
    }

  
    
    void text_analysis::analyze_line_breakpoints() {
        wchars text = _text;
        tflow::LINE_BREAKPOINT* bp = _breakpoints.begin();
        
        tflow::LINE_BREAKPOINT _pbp = { 0 };
        tflow::LINE_BREAKPOINT* pbp = &_pbp;
        
        bool p_is_punct = false;
        
        for (wchar pc = 0; !!text; ++text)
        {
            wchar c = *text;
            bool is_punct = ucispunct(c) != 0;
            bp->is_whitespace = ucisspace(*text);
            bp->is_soft_hyphen = *text == 0x00AD;
            bp->break_condition_after = bp->break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
            if (pbp->is_whitespace && !bp->is_whitespace) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            else if (pc == '\r' && c != '\n') {
                pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            else if (pc == '\n') {
                pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            else if (p_is_punct && !is_punct) {
                pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
                bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
            }
            pbp = bp++;
            pc = c;
            p_is_punct = is_punct;
        }
    }
    
    
    
    uint estimate_glyph_count(uint text_length)
    {
        return 3 * text_length / 2 + 16;
    }

#if 0
    void text_analysis::shape_glyph_runs( view& v, element* elem, tflow::text_flow& tf)
    {
        // Shapes all the glyph runs in the layout.
        
        uint text_length = static_cast<uint>(tf._text.length());
        if(!text_length)
            return;
        
        // Estimate the maximum number of glyph indices needed to hold a string.
        uint estimated_glyph_count = estimate_glyph_count(text_length);
        
        tf._glyph_indices.size(estimated_glyph_count);
        tf._glyph_offsets.size(estimated_glyph_count);
        tf._glyph_advances.size(estimated_glyph_count);
        tf._glyph_clusters.size(text_length);
        
        uint glyph_start = 0;
        
        //uint total = _runs.length();
        
        // Shape each run separately. This is needed whenever script, locale,
        // or reading direction changes.
        for (uint runIndex = 0; runIndex < tf._runs.length(); ++runIndex)
            shape_glyph_run(v,tf,elem, runIndex, glyph_start);
        
        tf._glyph_indices.size(glyph_start);
        tf._glyph_offsets.size(glyph_start);
        tf._glyph_advances.size(glyph_start);
        tf._glyph_justified_advances = tf._glyph_advances;
        
    }
    
    void text_analysis::shape_glyph_run(
                                        view& v, tflow::text_flow& tf, element* elem,
                                        uint run_index,
                                        IN OUT uint& glyph_start
                                        )
    {
        // Shapes a single run of text into glyphs.
        // Alternately, you could iteratively interleave shaping and line
        // breaking to reduce the number glyphs held onto at once. It's simpler
        // for this demostration to just do shaping and line breaking as two
        // separate processes, but realize that this does have the consequence that
        // certain advanced fonts containing line specific features (like Gabriola)
        // will shape as if the line is not broken.
        
        
        tflow::text_run& run  = tf._runs[run_index];
        
        uint text_start        = run.text_start;
        uint text_length       = run.text_length;
        uint maxglyph_count    = static_cast<uint>(tf._glyph_indices.size() - glyph_start);
        uint actual_glyph_count = 0;
        
        element* el = run.get_element();
        
        if( run.is_inline_block_pos(tf._text()))
        {
            el->check_layout(v);
            //assert(el->is_inline_block_element(v));
            run.glyph_start          = glyph_start;
            run.glyph_count          = 1;
            if (text_length > maxglyph_count)
            {
                uint totalGlyphsArrayCount = glyph_start + 1;
                tf._glyph_indices.size(totalGlyphsArrayCount);
            }
            tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_advances.length()));
            tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_offsets.length()));
            tf._glyph_advances[glyph_start] = static_cast<float>(el->min_inline_margin_box_width(v));
            tf._glyph_indices[glyph_start] = 0;
            glyph_start    += 1;
            return;
        }
        
        
        ustring  locale_name = el->get_lang();
        //html::style* style = el->get_style(v);
        osx::font* pf = static_cast<osx::font*>(run.get_used_font());
        
        //float  font_em_size = run.decl_font->size;
        
        //pf = used_font_face(pf,run);
        //d2d::asset<IDWriteFontFace> font_face = dw_font_face(pf, run);
        
        run.glyph_start          = glyph_start;
        run.glyph_count          = 0;
        
        if (text_length == 0)
            return; // Nothing to do..
        
        ////////////////////
        // Allocate space for shaping to fill with glyphs and other information,
        // with about as many glyphs as there are text characters. We'll actually
        // need more glyphs than codepoints if they are decomposed into separate
        // glyphs, or fewer glyphs than codepoints if multiple are substituted
        // into a single glyph. In any case, the shaping process will need some
        // room to apply those rules to even make that determintation.
        
        if (text_length > maxglyph_count)
        {
            maxglyph_count = estimate_glyph_count(text_length);
            uint totalGlyphsArrayCount = glyph_start + maxglyph_count;
            tf._glyph_indices.size(totalGlyphsArrayCount);
        }
        
        //juce::String text = juce::String(&tf._text[text_start],text_length);
        //Array <int> glyphs;
        //Array <float> xOffsets;
        
        //pf->juce_font()->getGlyphPositions(text,glyphs,xOffsets);
        
        //
        
        actual_glyph_count = osx_get_glyph_indices( pf, tf._text(text_start, text_start + text_length), &tf._glyph_indices[glyph_start], maxglyph_count );
        
        tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_advances.length()));
        tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_offsets.length()));
        
        osx_get_glyph_advances( pf, tf._glyph_indices(glyph_start, glyph_start + actual_glyph_count), &tf._glyph_advances[glyph_start] );
        
        /*tflow::GLYPH_OFFSET zgo = {0};
         
         float xoff = xOffsets[0];
         
         for( uint i = 0; i < actual_glyph_count; ++i ) {
         float xoff_next = xOffsets.getUnchecked(i+1);
         tf._glyph_advances[i + glyph_start] = xoff_next - xoff; xoff = xoff_next;
         tf._glyph_offsets[i + glyph_start] = zgo;
         tf._glyph_indices[i + glyph_start] = glyphs[i];
         }*/
        
        for( uint i = 0; i < text_length; ++i )
            tf._glyph_clusters[text_start + i] = i;
        
        ////////////////////
        // Certain fonts, like Batang, contain glyphs for hidden control
        // and formatting characters. So we'll want to explicitly force their
        // advance to zero.
        if (run.no_visual())
            tf._glyph_advances.set(glyph_start, glyph_start + actual_glyph_count, 0.0f);
        
        ////////////////////
        // Set the final glyph count of this run and advance the starting glyph.
        run.glyph_count = actual_glyph_count;
        glyph_start    += actual_glyph_count;
    }
    
#endif
    
    void view::setup_text_flow( html::element* elem, html::tflow::text_flow& tf, tool::slice< tool::handle<html::node> > nodes ) 
    {
        text_analysis::exec(*this, elem, tf, nodes);
    }
    
}

#endif
