#include "xom.h"
#include "xcontext.h"
#include "xview.h"
#include "xevent.h"
#include "xclipboard.h"

namespace qjs {

  html::ctl* create_canvas_for(xview* pv, html::element* b);

  JSClassID Window_class_id = 0;

  void xview::set_init_script(const char* script_text) {
#pragma TODO("SCITERJS")
  }

  void xview::stop()  {
    //dialog_retval.clear(); - this must survive stop() - will be read after
    super::stop();
    parameters.clear();
    subscriptions.clear(); // detach event handlers
    if(is_main())
      dialog_retval.clear();
    obj.clear();
    qjs::hruntime::current().release();
  }

  void xview::run(html::hdocument pd, chars text, chars url, bool is_module, int line_no) {

    context& c = context::inst(pd);

    uint flags = is_module ? JS_EVAL_TYPE_MODULE : JS_EVAL_TYPE_GLOBAL;
                 //JS_EVAL_TYPE_MODULE;
                 //JS_EVAL_TYPE_GLOBAL;
    try {
      c.eval(text, url, flags, line_no);
      //JS_RunGC(JS_GetRuntime(c));
    }
    catch(qjs::exception) {
      c.report_exception();
    }
  }

  void xview::on_load_start(html::document *pdoc) {

    //if (pdoc == doc())
    //  obj.clear(); // WRONG (but need check reload): loading document in existing window, clear remnants of previous doc/context (if any)

    context& c = context::create(this, pdoc);
    if (pdoc->debug_mode() && pdoc->allow_own_scripts && pdoc->pdocrq)
      {
        handle<request> hrq = new pump::request("sciter:debug-peer.js", html::DATA_SCRIPT);
        hrq->dst = pdoc;
        hrq->dst_view = this;
        if (load_data(hrq, true) && hrq->data.length()) {
          run(pdoc, chars_of(hrq->data()), CHARS("sciter:debug-peer.js"), true);
          pdoc->pdocrq->dst = pdoc;
          notify_resource_arrival(pdoc->pdocrq);
        }
        JS_SetDebuggerMode(c, TRUE);
      }
    super::on_load_start(pdoc);
  }

  void xview::on_load_end(html::document *d, bool existing_doc_replaced) {

    handle<request> pdocrq;
    swap(pdocrq, d->pdocrq);

    super::on_load_end(d, existing_doc_replaced);

    html::hdocument pdoc = d;
    handle<xview> holder(this);

    context& c = context::inst(pdoc);

    try
    {
      if (d->allow_own_scripts) {

        array<html::helement> script_elements;
        find_all(*this, script_elements, pdoc, WCHARS("script:not([type]),script[type='text/javascript'],script[type='module']"));
        for (int n = 0; n < script_elements.size(); ++n) {
          c.load_script_element(script_elements[n]);
          if (_finished) return;
        }
      }

    // resolve prototypes
    if (d->allow_script_behaviors)
      d->check_prototypes_tree(*this);

    d->operational = true;

      if (d->allow_own_scripts && d->ns) {
        hvalue ready = c.get_prop<hvalue>("ready", pdoc->obj);
        if (JS_IsFunction(c,ready)) {
          bool r; c.call(r, ready, JS_DupValue(c,pdoc->obj));
        }
      }
    }
      catch (exception) {
        c.report_exception();
      }

    //if (existing_doc_replaced)
      //CsCollectGarbage(vm);
  }

  value xview::get_retval() {
    if (doc()) {
      context& c = context::inst(doc());
      return c.get<value>(dialog_retval);
    }
    return value();
  }

  bool xview::on_unload(html::document *pd) {
    bool r = super::on_unload(pd);

    context& c = context::inst(pd);

    for (int i = subscriptions.last_index(); i >= 0; --i) {
      subscription* ps = subscriptions[i];
      if(c.is_function_of_this_realm(ps->fcn))
        subscriptions.remove(i);
    }

    tool::async::dispatch::heartbit_current(); // to close pending connections

    // TO REMOVE - too fragile
    /*if (r && pd->is_reloading) {
      if (c.pdoc() == pd && c.pview() == this) {
        hvalue pv = c.get_prop<hvalue>("data", c.global());
        size_t len = 0;
        uint8_t *buf = JS_WriteObject(c, &len, pv, 0);
        reload_data = bytes(buf,len);
        js_free(c, buf);
      }
    }*/

    JS_RunGC(JS_GetRuntime(c));

    return r;
  }

  html::ctl *xview::create_behavior(html::element *b, const string &name) {
    if (name == CHARS("graphics"))
      return create_canvas_for(this, b);
    return super::create_behavior(b, name);
  }

  bool rx_do_component_update(xcontext& c, html::helement el);
  bool rx_do_component_error(xcontext& c, html::helement el, tool::value err, helement context);

  bool xview::handle_element_event(html::helement b, html::event &evt)
  {
#ifdef DEBUG
    //if (evt.event_group() == html::HANDLE_COMMAND && evt.cmd == (event_command::EXEC | EVENT_SINKING) && b->is_document())
    //  b = b;
    //if (evt.event_group() == html::HANDLE_BEHAVIOR_EVENT && evt.event_type() == html::BUTTON_CLICK && evt.target->tag == tag::T_BUTTON)
    //  b = b;
#endif

    if (!b)
      return handle_view_event(evt);

    evt.processing_target = b;

    html::hdocument doc = b->doc();
    if (!doc) return false;

    if (doc->ns) {
      context& c = context::inst(doc);
      assert(c.pdoc() == doc);

      if (evt.event_group() == html::HANDLE_BEHAVIOR_EVENT && evt.cmd == html::COMPONENT_UPDATE)
        return rx_do_component_update(c, evt.target);

      if (evt.event_group() == html::HANDLE_BEHAVIOR_EVENT && evt.cmd == html::COMPONENT_ERROR)
        return rx_do_component_error(c, b, evt.data, evt.target);

      return b->handle_event(c, evt);
    }
    else {
      //auto eg = evt.event_group();
      //auto ec = evt.cmd_no_flags();
      return false;
    }

  }

  bool xview::may_have_on_size_handler(html::element *b) {
    return b->flags.has_on_size_script || super::may_have_on_size_handler(b);
  }

  bool xview::handle_view_event(html::event &evt) {

    bool handled = false;

#ifdef DEBUG
    if (evt.cmd == html::DOCUMENT_CLOSE_REQUEST) {
      ustring en = evt.event_full_name();
      evt.cmd = evt.cmd;
    }
#endif

    //if (evt.target) ???????
    //  return false;

    if (subscriptions.length() == 0)
      goto CHECK_CLASS_HANDLERS;

    for (int n = subscriptions.last_index(); n >= 0; --n)
    {
      if (n >= subscriptions.size()) {
        n = subscriptions.size();
        continue;
      }
      handle<html::subscription> ps = subscriptions[n];
      bool to_remove = ps->once;
      html::helement principal;
      if (ps->fcn && evt.match(*this, *ps, principal))
      {
        bool was_handled = evt.is_handled();
        context& c = context::inst(doc());
        try {
          hvalue rv;
          if (JS_IsFunction(c, ps->fcn)) {
            if( ps->receiver )
              c.call(rv, ps->fcn, ps->receiver.ptr(), &evt);
            else
              c.call(rv, ps->fcn, this, &evt);
          }
          else {
            hvalue handleEvent = c.get_prop<hvalue>(c.known_atoms().handleEvent, ps->fcn);
            assert(handleEvent);
            c.call(rv, handleEvent, ps->fcn, &evt);
          }
          if (rv == JS_TRUE)
            evt.stop_propagation();
          if (evt.is_handled() && !was_handled)
            handled = true;
        }
        catch (exception) {
          c.report_exception();
          to_remove = true;
          break;
        }
      }
      if (to_remove)
        subscriptions.remove(n);
    }

  CHECK_CLASS_HANDLERS:
    if (!handled && obj && doc())
    {
      context& c = context::inst(doc());
      hvalue map = JS_GetPrototype(c, obj);
      hvalue element_proto = JS_GetClassProto(c, Element_class_id);
      hvalue document_proto = JS_GetClassProto(c, Document_class_id);

      while (map && JS_IsObject(map))
      {
        if (map == element_proto || map == document_proto)
          break;
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(c, &tab, &len, map, JS_GPN_STRING_MASK);
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars pname(c, tab[n].atom);
          //dbg_printf("atom %s\n", pname);
          if (pname.length > 2 && pname[2] == ' ' && pname[0] == 'o' && pname[1] == 'n') {
            chars name = pname(3);
            chars selector;
            int pos = name.index_of(CHARS(" at "));
            if (pos > 0) {
              selector = name(pos + 4);
              name.length = uint(pos);
            }
            html::subscription sub(name);
            sub.event_selector = ustring(selector);

            html::helement principal;
            if (evt.match(*this, sub, principal)) {
              hvalue proto = JS_GetPrototype(c, obj);
              hvalue handler = JS_GetProperty(c, proto, tab[n].atom);
              assert(JS_IsFunction(c, handler));
              try {
                hvalue rv;
                c.call(rv, handler, obj, &evt);
                if (rv == JS_FALSE)
                  evt.stop_propagation();

                if (handled = evt.is_handled())
                  break;
              }
              catch (exception) {
                c.report_exception();
              }
            }
          }

        }
        js_free_prop_enum(c, tab, len);
        map = JS_GetPrototype(c, map);
      }
    }

    return handled;
  }


  bool xview::handle_element_timer(html::element *b, html::timer_def &td)
  {
    if ((td.kind == html::SCRIPT_TIMER || td.kind == html::SCRIPT_INTERVAL) && td.pcb) {
      assert(b->is_document()); // sciter.js uses only document timers
      html::hdocument pd = b->doc();
      if (pd && pd->pview() == this) {
        context& c = context::inst(pd);
        try {
          auto_state<html::timer_def*> _(processing_timer, &td);
          bool rv; 
          if (td.pcb.ptr_of<timer_callback>()->fun)
            c.call(rv, td.pcb.ptr_of<timer_callback>()->fun, b);
          else
            return false;
        } catch (exception) {
          c.report_exception();
          return false;
        }
      }
      if (td.kind == html::SCRIPT_INTERVAL)
        return true; // keep going
      else
        return false;
    }
    else if ((td.kind == html::SCRIPT_ELEMENT_TIMER) && td.pcb) {
      bool keep_ticking = false;
      html::hdocument pd = b->doc();
      if (pd && pd->pview() == this) {
        context& c = context::inst(pd);
        try {
          hvalue rv;
          if(c.is_function(td.pcb.ptr_of<timer_callback>()->fun))
            c.call(rv, td.pcb.ptr_of<timer_callback>()->fun, b);
          keep_ticking = rv == JS_TRUE;
        }
        catch (exception) {
          c.report_exception();
          return false;
        }
      }
      return keep_ticking;
    }

    return false;
  }

  bool xview::call_element_method(element *b, const char *name, slice<tool::value> args, tool::value &retval)
  {
    /*document* pd = b->doc();
    if (!pd || !b->obj)
      return false;
    context& c = context::inst(pd);
    hvalue method = c.get_prop<hvalue>(name, b->obj);
    if (!c.is_function(method))
      return false;
    hvalue rv;
    c.call(rv, method, b->obj, );*/
    return false;
  }

  bool xview::get_element_value(helement b, value &v, bool ignore_text_value) {
    document* pd = b->doc();
    if (!pd || !b->obj)
      return get_element_native_value(b, v, ignore_text_value);
    context& c = context::inst(pd);
    v = c.get_prop<tool::value>(c.known_atoms().value, b->obj);
    return true;
    //if (ignore_text_value)
    //return get_element_native_value(b, v, ignore_text_value);
  }
  bool xview::set_element_value(helement b, const value &v, bool ignore_text_value) {
    document* pd = b->doc();
    if (!pd || !b->obj)
      return set_element_native_value(b, v, ignore_text_value);
    context& c = context::inst(pd);
    c.set_prop<tool::value>(c.known_atoms().value, b->obj,v);
    return true;
    //if (ignore_text_value)
    //return set_element_native_value(b, v, ignore_text_value);
  }




  bool xview::exec_idle() {
    bool r = super::exec_idle();
    if (!has_pending_script_jobs() || !doc() || !doc()->allow_own_scripts )
      return r;
    context& c = context::inst(doc());
    // execute the pending jobs
    JSContext *ctx1 = nullptr;
    try {
      int err = JS_ExecutePendingJob(JS_GetRuntime(c), &ctx1);
      if ((err < 0) && ctx1)
        c.report_exception(JS_GetException(ctx1));
    }
    catch (qjs::exception) {
      c.report_exception();
    }

    return r || has_pending_script_jobs();
  }

  bool xview::has_pending_script_jobs() const
  {
    hruntime& hr = hruntime::current();
    //assert((JSRuntime *)hr);
    if((JSRuntime *)hr)
      return !!JS_IsJobPending(hr);
    return false;
  }


  bool xview::wants_idle() const {
    return super::wants_idle() || has_pending_script_jobs();
  }

  void xview::on_style_resolved(handle<html::element> b, const html::style *prev_style) {

    super::on_style_resolved(b, prev_style);

    if (this->dismissing) return;

    if (b->c_style == html::element::null_style)
      return;

    if (!b->is_connected())
      return;

    html::hstyle cs = b->c_style;

    tool::string name = cs->prototype;
    tool::string prev_name = prev_style->prototype;

    //auto_state<xview *> _1(vm->_current_view, this);

#ifdef _DEBUG
    //if (b->tag == html::tag::T_BODY)
    //  name = name;
    //if (name == CHARS("MainController"))
    //  name = name;
    //if(b->is_id_test())
    //  name = name;
#endif // _DEBUG


    html::helement el = b;

    if (el->flags.script_need_attached_call)
    {
#ifdef DEBUG
      //el->dbg_report("forced attached");
#endif
      el->flags.script_need_attached_call = 0;
      html::document* pd = b->doc();
      if (pd) {
        context& c = context::inst(pd);
        on_element_prototype_change(c, b);
        on_mount_element(c, b);
      }
    }

    if (name || prev_name) {
      process_prototype(el, name, prev_name);
    }

    if (cs->handler_list.defined()) {
      process_handlers(el, cs->handler_list);
    }
  }

  void xview::process_handlers(handle<html::element> b, const html::handler_list_v &handlers)
  {
    html::hdocument pd = b->doc();
    if (!pd) return;
    if (!pd->ns) return;
    if (!pd->allow_script_behaviors) return;
    if (!pd->state.ready()) return; // do it later

    auto already_processed = [b](const html::handler_list_v::item *fd) -> bool {
      bool found = false;
      html::each_resource<html::handler_list_v::item>(
        b->meta(), [fd, &found](const html::handler_list_v::item *it) -> bool {
        if (fd->funcname == it->funcname) {
          found = true;
          return true;
        }
        return false;
      });
      return found;
    };

    auto one = [&](const html::handler_list_v::item *hd)
    {
      context& c = context::inst(pd);

      try {
        hvalue fcn = c.eval_silent(hd->funcname);
        if (!fcn.is_defined() || c.is_exception(fcn)) {
          tool::string script_url = hd->url;
          if (script_url.is_empty())
            return; // not this time

          if (auto pm = c.load_script_module(script_url)) {
            fcn = JS_GetModuleExportItemStr(c, pm, hd->funcname);
            if (!JS_IsFunction(c, fcn))
            {
              JS_ThrowTypeError(c, tool::string::format("'%s' is not a function in module '%s'", hd->funcname.c_str(), script_url.c_str()));
              throw qjs::exception();
            }
          }
          else
            throw qjs::exception();
        }
        bool dummy;
        b->meta.push(hd); // mark it as processed
        c.call(dummy, fcn, b, hd->params);
      }
      catch (qjs::exception) {
        c.report_exception();
      }
    }; // one

    int nelements = handlers.list->size();
    for (int n = 0; n < nelements; ++n) {
      const html::handler_list_v::item *pit = handlers.list->operator[](n);
      if (!already_processed(pit)) one(pit);
    }
  }

  void xview::on_element_prototype_change(xcontext& c, handle<html::element> b) {
    b->flags.script_draws_background = c.get_prop<bool>(c.known_atoms().paintBackground,b->obj);
    b->flags.script_draws_foreground = c.get_prop<bool>(c.known_atoms().paintForeground, b->obj);
    b->flags.script_draws_content = c.get_prop<bool>(c.known_atoms().paintContent, b->obj);
    b->flags.script_draws_outline = c.get_prop<bool>(c.known_atoms().paintOutline, b->obj);
    b->flags.has_on_size_script = c.get_prop<bool>(c.known_atoms().onsizechange, b->obj);
    b->flags.has_on_pre_size_script = c.get_prop<bool>(c.known_atoms().onsizechanging, b->obj);
    b->flags.has_on_visibility_changed_script = c.get_prop<bool>(c.known_atoms().onvisibilitychange, b->obj);
    b->flags.script_has_is_point_inside = c.get_prop<bool>(c.known_atoms().isPointInside, b->obj);
  }

  void xview::on_element_client_size_changed(html::element *b) {

    super::on_element_client_size_changed(b);
    if (!b->flags.has_on_size_script)
      return;
    document* pd = b->doc();
    if (!pd)
      return;
    context& c = context::inst(pd);
    hvalue cb = c.get_prop<hvalue>(c.known_atoms().onsizechange, b->obj);
    if (c.is_function(cb)) {
      try {
        bool dummy;
        c.call(dummy, cb, b->obj);
      }
      catch (exception) {
        c.report_exception();
        b->flags.has_on_size_script = 0;
      }
    } else
      b->flags.has_on_size_script = 0;
  }

  void xview::on_element_client_size_changing(html::element *b, bool horizontal,int val) {
    super::on_element_client_size_changing(b,horizontal,val);
    if (!b->flags.has_on_pre_size_script)
      return;
    document* pd = b->doc();
    if (!pd)
      return;
    context& c = context::inst(pd);
    hvalue cb = c.get_prop<hvalue>(c.known_atoms().onsizechanging, b->obj);
    if (c.is_function(cb)) {
      try {
        bool dummy;
        b->flags.has_on_pre_size_script = 0; // to prevent recursive calls
        c.call(dummy, cb, b->obj, horizontal, ppx_to_dip(gool::size(val,val)).x);
        b->flags.has_on_pre_size_script = 1;
      }
      catch (exception) {
        c.report_exception();
      }
    }
  }


  void xview::on_element_visibility_changed(html::element *b, bool on_off) {
    if (!b->flags.has_on_visibility_changed_script)
      return;
    document* pd = b->doc();
    if (!pd)
      return;
    context& c = context::inst(pd);
    hvalue cb = c.get_prop<hvalue>(c.known_atoms().onvisibilitychange, b->obj);
    if (c.is_function(cb)) {
      try {
        bool dummy;
        c.call(dummy, cb, b->obj, on_off);
      }
      catch (exception) {
        c.report_exception();
        b->flags.has_on_visibility_changed_script = 0;
      }
    }
    else
      b->flags.has_on_visibility_changed_script = 0;
  }


  void xview::process_prototype(handle<html::element> b, const tool::string &name, const tool::string &prev_name)
  {
    html::hdocument pd = b->doc();
    if (!pd)
      return;
    if (!pd->ns)
      return;
    if (!pd->allow_script_behaviors)
      return;

    if ((name != prev_name) && (prev_name.length() != 0))
    {
      if (b->obj) {
        context& c = context::inst(pd);

        //if (b->flags.script_reactors_prototype) {
        //  CsWarning(vm, string::format("attempt to remove prototype '%s' set by Reactor", prev_name.c_str()));
        //  return;
        //}

        on_element_prototype_change(c, b);

        c.pxview()->on_unmount_element(c, b);

      }
    }

    if (name.length())
    {
      if (b->flags.strayed)
        return;
      if (!pd->state.ready())
        return; // do it later, behaviors MUST be assigned after <script>s

      context& c = context::inst(pd);

      if(b->obj) {
        if (hvalue constructor = c.get_prop<hvalue>(c.known_atoms().constructor, b->obj)) {
          if (tool::string classname = c.get_prop<tool::string>(c.known_atoms().name, constructor)) {
            if (name == classname)
              return; // this element already has this class
          }
        }
      }

      try {
        hvalue cls = c.eval_silent(name);
        if (!cls.is_defined() || c.is_exception(cls))
        {
          tool::string script_url = b->c_style->prototype_url;
          if (script_url.is_empty())
            return; // not this time

          if (auto pm = c.load_script_module(script_url)) {
            cls = JS_GetModuleExportItemStr(c, pm, name);
            if (!JS_IsFunction(c, cls))
            {
              JS_ThrowTypeError(c, tool::string::format("'%s' is not a class/function in module '%s'", name.c_str(), script_url.c_str()));
              throw qjs::exception();
            }
          }
          else
            throw qjs::exception();
        }

        hvalue newinst = JS_CallConstructor(c, cls, 0, 0);
        if (JS_IsException(newinst))
          throw qjs::exception();

        /*if (b->obj) { // this trick breaks private fields :(
          hvalue obj = c.val(b.ptr());
          //JS_SwapObjects(c, obj, newinst)
        }
        else*/
        {
          b->obj = JS_DupValue(c,newinst);
          b->add_ref();
          JS_SetOpaque(b->obj, (html::node*)b);
        }
        on_element_prototype_change(c, b);
        on_mount_element(c, b);

      }
      catch (qjs::exception) {
        c.report_exception();
      }
    }

  }

  void xview::on_element_removing(html::element *b) {

    html::hdocument pd = b->doc();
    if (!pd)
      return;
    if (!pd->ns)
      return;
    if (!pd->allow_script_behaviors)
      return;

    if (b->obj)
    {
      context& c = context::inst(pd);
      on_unmount_element(c, b);
    }
  }

  void xview::notify_resource_arrival(handle<pump::request> prq) {
    if (prq->dst) {
      html::hdocument pd = prq->dst->doc();
      if (pd && pd->ns) {
        context& c = context::inst(pd);
        if (c.request_complete_handler) try {
          bool dummy;
          c.call(dummy, c.request_complete_handler, JS_UNDEFINED, prq.ptr_of<html::response>());
        }
        catch (exception) {
          c.report_exception();
        }
      }
    }
  }

  bool window_move(xcontext&, xview* pv, int x, int y, int_v w, int_v h, bool client) {

    gool::point pos(x, y);
    bool        and_size = w.is_defined() && h.is_defined();
    gool::size  sz(w.val(0), h.val(0));

    html::VIEW_TYPE vt = pv->view_type();
    switch (vt) {
      case html::VIEW_WINDOW:
      case html::VIEW_DIALOG:
      {
        if (!and_size) {
          if (client)
            sz = pv->client_dim();
          else
            sz = pv->window_dim();
        }
        gool::rect rc(pos, sz);
        //rect wd = pv->get_frame_type() == html::STANDARD ? pv->window_decoration() : rect(0,0,0,0);
        //if (client_cc) rc >>= wd;
        pv->move_window(rc, client);
        return true;
      }
    }
    return false;
  }

  bool window_move_to(xcontext&, xview* pv, int monitor, float x, float y, float_v w, float_v h, bool client) {

    gool::pointf pos(x, y);
    bool         and_size = w.is_defined() && h.is_defined();
    gool::sizef  sz(w.val(0), h.val(0));

    bool r = true;

    screen_info si;

    if (!get_screen_info(monitor, si)) {
      get_screen_info(0, si);
      r = false;
    }

    size isz;
    size ipos;

    html::VIEW_TYPE vt = pv->view_type();
    switch (vt) {
    case html::VIEW_WINDOW:
    case html::VIEW_DIALOG:
      {
        if (!and_size) {
          if (client)
            isz = pv->client_dim();
          else
            isz = pv->window_dim();
          sz = pv->ppx_to_dip(isz);
        }
        ipos.x = int(pos.x * si.dpi.x / 96);
        ipos.y = int(pos.y * si.dpi.y / 96);
        isz.x = int(sz.x * si.dpi.x / 96);
        isz.y = int(sz.y * si.dpi.y / 96);

        ipos += si.monitor.s;

        gool::rect rc(ipos, isz);
        rc.inscribe(si.workarea);
      
        pv->move_window(rc, client);
      } break;
    }
    return r;
  }



  bool window_set_state(xcontext&, xview* pv, int state) {
    return pv->set_window_state((gool::WINDOW_STATE)state);
  }
  int  window_get_state(xcontext&, xview* pv) {
    return pv->get_window_state();
  }

  ustring window_get_caption(xcontext&, xview* pv) {
    return pv->get_window_title();
  }
  void window_set_caption(xcontext&, xview* pv, ustring cap) {
    pv->set_window_title(cap);
  }

  helement window_events_root(xcontext&, xview* pv) {
    return pv->event_capture_element;
  }
  void window_set_events_root(xcontext&, xview* pv, helement el) {
    pv->set_event_capture(el);
  }

  helement window_focus(xcontext&, xview* pv) {
    return pv->get_focus_element();
  }
  void window_set_focus(xcontext&, xview* pv, helement el) {
    pv->set_focus(el, html::BY_CODE);
  }

  helement window_focusable(xcontext&, xview* pv, string op, helement rel) {
    html::view::FOCUS_CMD_TYPE fcmd;

    if (op == CHARS("next")) fcmd = html::view::FOCUS_NEXT;
    else if (op == CHARS("prior")) fcmd = html::view::FOCUS_PREV;
    else if (op == CHARS("first")) fcmd = html::view::FOCUS_FIRST;
    else if (op == CHARS("last")) fcmd = html::view::FOCUS_LAST;
    else throw qjs::om::type_error("unknown command");

    bool end_reached = false; // ???
    rel = pv->get_next_focus_element(fcmd, rel, end_reached);
    return rel;
  }

  bool window_update(xcontext& c, xview* pv) {
    pv->update();
    return true;
  }

  hvalue window_perform_drag(xcontext& c, xview* pv, hvalue items, string mode, hvalue what, int x, int y) {

    handle<html::clipboard::data> pdata = object2cbdata(c, pv, items);
    uint dd_mode = html::dd_copy;
    if (mode == CHARS("any"))
      dd_mode = html::dd_copy_or_move;
    else if (mode == CHARS("copy"))
      dd_mode = html::dd_copy;
    else if (mode == CHARS("move"))
      dd_mode = html::dd_move;

    if (c.isa<html::element*>(what)) {
      handle<html::element> pel = c.get<html::element*>(what);
      if (!pv->drag_element(pel, pdata, dd_mode))
        return JS_NULL;
    }
    else if (c.isa<gool::image*>(what)) {
      handle<gool::image> pimg = c.get<gool::image*>(what);
      hbitmap pbmp;
      if (pimg && pimg->is_bitmap())
        pbmp = pimg.ptr_of<gool::bitmap>();

      if (!pbmp || !pv->drag_data(pbmp, point(x,y), pdata, dd_mode))
        return JS_NULL;
    }
    else
      return JS_NULL;

    switch (dd_mode) {
      case html::dd_copy: return c.val(CHARS("copy"));
      case html::dd_move: return c.val(CHARS("move"));
    }
    return JS_NULL;
  }


  array<int> window_get_min_size(xcontext&, xview* pv) {
    size sz;
    if (pv->get_min_size(sz))
      return array<int> {sz.x, sz.y};
    return array<int>();
  }
  bool window_set_min_size(xcontext&, xview* pv, array<int> sz) {
    if (sz.length() == 2)
      pv->set_min_size(size(sz[0], sz[1]));
    else
      pv->set_min_size();
    return true;
  }

  array<int> window_get_max_size(xcontext&, xview* pv) {
    size sz;
    if (pv->get_max_size(sz))
      return array<int> {sz.x, sz.y};
    return array<int>();
  }
  bool window_set_max_size(xcontext&, xview* pv, array<int> sz) {
    if (sz.length() == 2)
      pv->set_max_size(size(sz[0], sz[1]));
    else
      pv->set_max_size();
    return true;
  }

  string window_blurbehind(xcontext&, xview* pv) {
    switch (pv->get_blurbehind()) {
      case html::BLUR_BEHIND::BLUR_BEHIND_NONE: return CHARS("none");
      case html::BLUR_BEHIND::BLUR_BEHIND_CUSTOM: return CHARS("custom");
      case html::BLUR_BEHIND::BLUR_BEHIND_ULTRA_DARK: return CHARS("ultra-dark");
      case html::BLUR_BEHIND::BLUR_BEHIND_DARK: return CHARS("dark");
      case html::BLUR_BEHIND::BLUR_BEHIND_LIGHT: return CHARS("light");
      case html::BLUR_BEHIND::BLUR_BEHIND_ULTRA_LIGHT: return CHARS("ultra-light");
    }
    return string();
  }
  void window_set_blurbehind(xcontext&, xview* pv, string mode) {
    if(mode == CHARS("none")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_NONE);
    else if(mode == CHARS("custom")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_CUSTOM);
    else if (mode == CHARS("ultra-dark")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_ULTRA_DARK);
    else if (mode == CHARS("dark")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_DARK);
    else if (mode == CHARS("ultra-light")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_ULTRA_LIGHT);
    else if (mode == CHARS("light")) pv->set_blurbehind(html::BLUR_BEHIND::BLUR_BEHIND_LIGHT);
    else throw qjs::om::type_error("invalid blurbehind value");
  }

  bool window_resizable(xcontext&, xview* pv) { return pv->get_resizeable(); }
  void window_set_resizable(xcontext&, xview* pv, bool on) { pv->set_resizeable(on); }

  bool window_maximizable(xcontext&, xview* pv) { return pv->get_maximizable(); }
  void window_set_maximizable(xcontext&, xview* pv, bool on) { pv->set_maximizable(on); }

  bool window_minimizable(xcontext&, xview* pv) { return pv->get_minimizable(); }
  void window_set_minimizable(xcontext&, xview* pv, bool on) { pv->set_minimizable(on); }

  bool window_topmost(xcontext&, xview* pv) { return pv->get_topmost(); }
  void window_set_topmost(xcontext&, xview* pv, bool on) { pv->set_topmost(on); }

  bool window_enabled(xcontext&, xview* pv) { return pv->is_enabled(); }
  void window_set_enabled(xcontext&, xview* pv, bool on) { pv->set_enabled(on); }


  hvalue window_aspect_ratio(xcontext& c, xview* pv) {
    float_v ratio = pv->aspect_ratio();
    if (ratio.is_defined())
      return c.val(ratio.val(0.0f));
    return JS_NULL;
  }
  void window_set_aspect_ratio(xcontext& c, xview* pv, hvalue ratio) {
    float t;
    if(c.is_float(ratio,t))
      pv->aspect_ratio(t);
    else
      pv->aspect_ratio(float_v());
  }

  xview* window_subscribe(xcontext& c, xview* pv, ustring name, qjs::hvalue fcn, html::subscription::flags fs)
  {
    if (!c.is_event_handler(fcn)) {
      string sn = conv<string>::unwrap(c, fcn);
      throw qjs::om::type_error(string::format("%s is not an event handler", sn.c_str()));
    }
    pv->subscribe(fcn, name, ustring(), fs);
    return pv;
  }

  xview* window_on(xcontext& c, xview* pv, ustring name, qjs::hvalue fcn)
  {
    if (!c.is_function(fcn)) {
      string sn = conv<string>::unwrap(c, fcn);
      throw qjs::om::type_error(string::format("%s is not a function", sn.c_str()));
    }
    pv->subscribe(fcn, name, c.pdoc());
    return pv;
  }

  xview* window_off(xcontext& c, xview* pv, qjs::hvalue what)
  {
    if (c.is_event_handler(what))
      pv->unsubscribe(what);
    else
      pv->unsubscribe(conv<ustring>::unwrap(c, what));
    return pv;
  }

  bool window_activate(xcontext& c, xview* pv, bool bringToFront) {
    return pv->activate(bringToFront);
  }

  bool window_is_active(xcontext& c, xview* pv) {
    return pv->is_active();
  }

  //void init_event(xcontext& c, hvalue def, html::event_behavior& evt);

  bool window_dispatch_event(xcontext& c, xview* pe, html::event_proxy* ppevt, bool post)
  {
    assert(ppevt->evt().event_group() == html::HANDLE_BEHAVIOR_EVENT);
    html::event_behavior& evt = static_cast<html::event_behavior&>(ppevt->evt());

    if (post) {
      pe->post_behavior_event(evt, true);
      return true;
    }
    else {
      if (pe->dispatch_event_level > 32)
        throw qjs::om::range_error("cyclic event");
      auto_state<uint> _(pe->dispatch_event_level, pe->dispatch_event_level+1);
      bool r = pe->send_behavior_event(evt);
      return (r || evt.is_handled()) ? false : true; /* note inverse logic */
    }
  }

  string window_graphics_backend(xcontext&, xview* pv) {
    return pv->graphics_backend();
  }

  hvalue window_parameters(xcontext&, xview* pv) {
    return pv->parameters;
  }

  hvalue window_retval(xcontext&, xview* pv) {
    return pv->dialog_retval;
  }


  bool window_close(xcontext& c, hvalue hv, hvalue retval)
  {
    xview* pv = conv<xview*>::try_unwrap(c, hv);
    if (!pv) return false;
    
    html::VIEW_TYPE vt = pv->view_type();
    switch (vt) {
    case html::VIEW_WINDOW:
    case html::VIEW_DIALOG:
      // THROW
      pv->dialog_retval = retval;
      pv->ask_close_window();
      break;
    }
    return true;
  }

  bool window_is_alive(xcontext& c, hvalue hv)
  {
    xview* pv = conv<xview*>::try_unwrap(c, hv);
    if (!pv) return false;
    return true;
  }

  hvalue window_select_file(xcontext& c, xview* pv, hvalue options) {

#pragma TODO("FEATURE_SYSINFO")

    //if ((c->features & FEATURE_SYSINFO) == 0) return false;

    ustring  filter = c.get_prop<ustring>("filter", options, W("All Files (*.*)|*.*"));
    ustring  ext = c.get_prop<ustring>("extension", options, W("html"));
    ustring  initialPath = c.get_prop<ustring>("path", options, W(""));
    string   purpose = c.get_prop<string>("mode", options, W("open"));
    ustring  caption = c.get_prop<ustring>("caption", options, W("Select file:"));

    if (initialPath.length())
      initialPath = url::file_url_to_path(initialPath);

    if (filter.length() == 0) return JS_NULL;
    //if (sfilter[sfilter.size() - 1] != 0) sfilter += '|'; ???

    ustring filename = url::file_url_to_path(initialPath);

    xview::AFN_MODE mode = xview::AFN_OPEN;
    if (purpose == CHARS("open-multiple"))
      mode = xview::AFN_OPEN_MULTIPLE;
    else if (purpose == CHARS("save"))
      mode = xview::AFN_SAVE;

    array<ustring> filenames = pv->ask_file_name(mode, caption, filename, ext, filter);
    array<string> fileurls(filenames.size());
    if (fileurls.size() == 1) {
      return c.val(url::path_to_file_url(filenames[0]));
    }
    if (fileurls.size()) {
      for (int n = 0; n < fileurls.size(); ++n)
        fileurls[n] = url::path_to_file_url(filenames[n]);
      return c.val(fileurls);
    }
    return JS_NULL;
  }

  string window_select_folder(xcontext& c, xview* pv, hvalue options) {
#pragma TODO("FEATURE_SYSINFO")
    //if ((c->features & FEATURE_SYSINFO) == 0) return UNDEFINED_VALUE;

    ustring  caption = c.get_prop<ustring>("caption", options, W("Select folder:"));
    ustring  initialPath = c.get_prop<ustring>("path", options, W(""));

    ustring name;
    if (initialPath.length())
      name = url::file_url_to_path(initialPath);

    if (pv->ask_folder_name(caption, name))
      return url::path_to_file_url(name);

    return string();
  }

  hvalue window_media_var(xcontext& c, xview* pv, string mvname, hvalue value_to_set)
  {
    html::media_variables &media_vars = pv->media_vars();
    if (value_to_set.is_defined()) {
      media_vars[mvname] = c.get<tool::value>(value_to_set);
      pv->update_media_vars(true);
      return hvalue();
    }
    else {
      return c.val(media_vars[mvname]);
    }
  }

  hvalue window_media_vars(xcontext& c, xview* pv, hvalue vars)
  {
    html::media_variables &media_vars = pv->media_vars();

    if (vars.is_defined()) {
      tool::value vvars = c.get<tool::value>(vars).isolate();
      if (pv)
        pv->set_media_vars(vvars, true, true);
      else
        xview::set_default_media_vars(vvars, false);
      return hvalue();
    }
    else {
      tool::value map = tool::value::make_map();
      const html::media_variables &media_vars = pv ? pv->media_vars() : html::default_media_vars_provider.media_vars();
      for (int n = 0; n < media_vars.size(); ++n) {
        tool::value k = tool::value(media_vars.key(n)); // UT_SYMBOL
        map.set_prop(k, media_vars.value(n));
      }
      return c.val(map);
    }
  }

  hvalue window_trayicon(xcontext& c, xview* pv, hvalue what) {
    html::view::tray_icon_params tip;

    if (c.is_object(what)) {

      tip.img = c.get_prop<himage>("image", what,nullptr);
      tip.tooltip = c.get_prop<ustring>("text", what);
      if (tip.img.is_defined() || tip.tooltip.is_defined()) {
        pv->trayicon_setup(tip);
        return JS_TRUE;
      }
    }
    else if (c.is_string(what) && c.get<string>(what) == CHARS("remove")) {
      pv->trayicon_remove();
      return JS_TRUE;
    }
    else if (c.is_string(what) && c.get<string>(what) == CHARS("place")) {
      rect rc;
      if (pv->trayicon_place(rc)) {
        array<int> src = { rc.left(), rc.top(), rc.width(), rc.height() };
        return c.val(src);
      }
    }
    return JS_NULL;
  }

  bool window_request_attention(xcontext& c, xview* pv, string how) {
    if (how().starts_with(CHARS("info")))
      return pv->request_attention(view::WRA_ATTENTION);
    else if (how == "alert")
      return pv->request_attention(view::WRA_ATTENTION_CRITICAL);
    else 
      return pv->request_attention(view::WRA_STOP);
  }
  
  string window_frame_type(xcontext& c, xview* pv)
  {
    switch (pv->get_frame_type()) {
    default:
    case html::STANDARD: return CHARS("standard");
    case html::LAYERED:  return CHARS("transparent");
    case html::SOLID:    return CHARS("solid");
    case html::SOLID_WITH_SHADOW: return CHARS("solid-with-shadow");
    case html::STANDARD_EXTENDED: return CHARS("extended");
    }
  }

  void window_set_frame_type(xcontext& c, xview* pv, string val)
  {
    if (val == CHARS("standard")) pv->set_frame_type(html::STANDARD);
    else if (val == CHARS("transparent")) pv->set_frame_type(html::LAYERED);
    else if (val == CHARS("solid")) pv->set_frame_type(html::SOLID);
    else if (val == CHARS("solid-with-shadow")) pv->set_frame_type(html::SOLID_WITH_SHADOW);
    else if (val == CHARS("extended")) pv->set_frame_type(html::STANDARD_EXTENDED);
  }

  hvalue window_icon(xcontext& c, xview* pv)
  {
    if (pv->get_icon()) {
      handle<gool::image> pimg = pv->get_icon();
      return c.val(pimg);
    }
    return JS_NULL;
  }

  void window_set_icon(xcontext& c, xview* pv, himage img)
  {
    pv->set_icon(img);
  }

  // Window.func - static props and functions
  //xview* js_window_this(html::context& c);
  //hvalue js_window_screenbox(html::context& c, JSValue screenNo, JSValue area, JSValue part);

  xview* Window_this(xcontext& c) {
    return c.pxview();
  }

  hvalue Window_share(xcontext& c) {
    hruntime& hrt = hruntime::current();
    if (hrt.shared.is_undefined())
      hrt.shared = JS_NewObject(c);
    return hrt.shared;
  }

  int Window_screens_total(xcontext& c) {
    return html::number_of_screens();
  }

  handle<document> window_document(xcontext& c, xview* pv)
  {
    return pv->doc();
  }

  handle<xview> window_parent(xcontext& c, xview* pv)
  {
    return static_cast<xview*>(pv->parent());
  }

  hvalue window_box(xcontext& c, xview* pv, string part, string box, string relto, tristate_v as_ppx)
  {
    if (!pv) return hvalue();

    rect rc(pv->client_dim());

    if (box == CHARS("client")) {
      if (relto == CHARS("screen") || relto == CHARS("desktop"))
        rc += pv->client_screen_pos();
      else if (relto == CHARS("monitor")) {
        rc += pv->client_screen_pos();
        screen_info si;
        get_screen_info(screen_of(pv), si);
        rc -= si.monitor.s;
        if (!as_ppx) {
          as_ppx = true;
          rc.s.x = rc.s.x * 96 / si.dpi.x;
          rc.s.y = rc.s.y * 96 / si.dpi.y;
          rc.e.x = rc.e.x * 96 / si.dpi.x;
          rc.e.y = rc.e.y * 96 / si.dpi.y;
        }
      }
    }
    else if (box == CHARS("border")) {
      rc = rect(pv->window_dim());
      if (relto == CHARS("screen") || relto == CHARS("desktop"))
        rc += pv->screen_pos();
      else if (relto == CHARS("monitor")) {
        rc += pv->screen_pos();
        screen_info si;
        get_screen_info(screen_of(pv), si);
        rc -= si.monitor.s;
        if (!as_ppx) {
          as_ppx = true;
          rc.s.x = rc.s.x * 96 / si.dpi.x;
          rc.s.y = rc.s.y * 96 / si.dpi.y;
          rc.e.x = rc.e.x * 96 / si.dpi.x;
          rc.e.y = rc.e.y * 96 / si.dpi.y;
        }
      }
      else {
        rc += pv->screen_pos();
        rc -= pv->client_screen_pos();
      }
    }
    else if( box == CHARS("cursor")) {
      rc = rect(pv->cursor_pos(),size(0,0));
      if (relto == CHARS("screen") || relto == CHARS("desktop"))
        rc += pv->client_screen_pos();
      else if (relto == CHARS("monitor")) {
        rc += pv->screen_pos();
        screen_info si;
        get_screen_info(screen_of(pv), si);
        rc -= si.monitor.s;
        if (!as_ppx) {
          as_ppx = true;
          rc.s.x = rc.s.x * 96 / si.dpi.x;
          rc.s.y = rc.s.y * 96 / si.dpi.y;
          rc.e.x = rc.e.x * 96 / si.dpi.x;
          rc.e.y = rc.e.y * 96 / si.dpi.y;
        }
      }
    }
    return c.box_part(part(), rc, as_ppx.val(0) ? xcontext::AS_PPX : xcontext::AS_PX );
  }

  tool::value window_behavior_call(xcontext& c, xview* pv, tool::string name, tool::array<tool::value> args) {
    html::scripting_params params;
    params.method_name = name.c_str();
    params.argv = args.cbegin();
    params.argc = uint(args.length());
    if (!pv->call_behavior_method(&params))
      throw qjs::om::type_error(string::format("unknown name '%s'", name.c_str()));
    return params.retval;
  }

  bool window_load(xcontext& c, xview* pv, string url)
  {
    return pv->load_url(url);
  }

  int window_screen_no(xcontext& c, xview* pv) {
    return screen_of(pv);
  }
  hvalue window_screenbox(xcontext& c, xview* pv, string what, string part, tristate_v as_ppx)
  {
    int   monitor_no = screen_of(pv);

    html::screen_info si;

    if (!html::get_screen_info(monitor_no, si)) return JS_UNDEFINED;

    rect rc;

    if (what == CHARS("frame"))
      rc = si.monitor;
    else if (what == CHARS("workarea"))
      rc = si.workarea;
    else if (what == CHARS("device"))
      return c.val(si.device_name);
    else if (what == CHARS("isPrimary"))
      return c.val(si.is_primary);
#ifdef SCREENSHOT_SUPPORT
    else if (what == CHARS("snapshot")) {
      handle<gool::bitmap> bmp = html::get_screen_shot(monitor_no);
      if (!bmp) return JS_NULL;
      return c.val(bmp.ptr_of<gool::image>());
    }
#endif
    else
      throw qjs::om::type_error("unknown option");

    return c.box_part(part(), rc, as_ppx.val(1) ? xcontext::AS_PPX : xcontext::AS_PX);
  }

  uint Window_ticks(xcontext& c) {
    return tool::get_ticks();
  }

  hvalue Window_screenbox(xcontext& c, int monitor_no, string what, string part, tristate_v as_ppx)
  {
    html::screen_info si;

    if (!html::get_screen_info(monitor_no, si)) return JS_UNDEFINED;

    rect rc;

    if (what == CHARS("frame"))
      rc = si.monitor;
    else if (what == CHARS("workarea"))
      rc = si.workarea;
    else if (what == CHARS("device"))
      return c.val(si.device_name);
    else if (what == CHARS("isPrimary"))
      return c.val(si.is_primary);
    else if (what == CHARS("devicePixelRatio"))
      return c.val( si.dpi.x / 96.0);
#ifdef SCREENSHOT_SUPPORT
    else if (what == CHARS("snapshot")) {
      handle<bitmap> bmp = html::get_screen_shot(monitor_no);
      if (!bmp) return JS_NULL;
      return c.val(bmp.ptr_of<gool::image>());
    }
#endif
    else
      throw qjs::om::type_error("unknown option");

    if (!as_ppx.val(1)) {
      rc.s.x = rc.s.x * 96 / si.dpi.x;
      rc.s.y = rc.s.y * 96 / si.dpi.y;
      rc.e.x = rc.e.x * 96 / si.dpi.x;
      rc.e.y = rc.e.y * 96 / si.dpi.y;
    }

    return c.box_part(part(), rc, xcontext::AS_PPX);

  }

  bool window_post_event_all(xcontext& c, hvalue what, tristate_v only_if_not_there) {

    if (c.is_function(what)) {
      html::posted_event be;
      be.target = c.pdoc();
      be.cbf = what;
      be.is_bubbling = false;
      c.pxview()->post_event(be, only_if_not_there.val(1));
      return true;
    }
    else if(auto ppevt = c.get<event_proxy*>(what)){
      assert(ppevt->evt().event_group() == html::HANDLE_BEHAVIOR_EVENT);
      html::event_behavior& evt = static_cast<html::event_behavior&>(ppevt->evt());
      xview::each([&evt](handle<html::view> hv) {
        hv->post_behavior_event(evt, true);
      });
      return true;
    }
    return false;
  }

  bool window_send_event_all(xcontext& c, hvalue what, tristate_v only_if_not_there) {

    if (auto ppevt = c.get<event_proxy*>(what)) {
      assert(ppevt->evt().event_group() == html::HANDLE_BEHAVIOR_EVENT);
      html::event_behavior& evt = static_cast<html::event_behavior&>(ppevt->evt());
      xview::each_if([&evt](handle<html::view> hv) -> bool {
        return hv->send_behavior_event(evt);
      });
      return false;
    }
    return false;
  }


  bool window_do_event(xcontext& c, handle<xview> pv, string mode) {

    html::DO_EVENT_MANNER m = html::DO_EVENT_WAIT;
    if (mode == CHARS("wait"))
      m = html::DO_EVENT_WAIT;
    else if (mode == CHARS("noWait"))
      m = html::DO_EVENT_NOWAIT;
    else if (mode == CHARS("untilQuit"))
      m = html::DO_EVENT_ALL;
    else if (mode == CHARS("untilMouseUp"))
      m = html::DO_EVENT_UNTIL_MOUSE_UP;
    else if (mode == CHARS("I/O"))
      m = html::DO_EVENT_ONLY_IO;

    if (!pv)
      throw om::type_error("window closed");
    bool result = true;
    pv->do_event(m, result);
    return result;
  }

  hvalue window_modal(xcontext& c, xview* pv, hvalue options)
  {

    handle<html::view> xd;
    handle<html::pump::request> prq;

    if (pv->dismissing)
      throw om::type_error("parent window is closing");

    window_params params(html::DIALOG_WINDOW);

    params.parent = pv;
    params.debug_mode = pv->debug_mode();

    if(c.get_prop<bool>("url", options) || c.get_prop<bool>("html", options))
    {
      params.url = c.get_prop<string>("url", options);
      params.url = html::combine_url(c.pdoc()->uri().src, url::escape(params.url));

      prq = new html::pump::request(params.url, html::DATA_HTML);

      string html = c.get_prop<ustring>("html", options,ustring());

      if (html.is_defined()) {
        prq->data = html.chars_as_bytes();
      }
      else if (params.url.length())
      {
        if (!pv->load_data(prq, true))
          throw om::type_error(string::format("file not found: %s", params.url.c_str()));
      }

      params.data = prq->data();
      params.alignment = 5;
      params.caption = c.get_prop<ustring>("caption", options, params.caption);
      params.alignment = c.get_prop<int>("alignment", options, params.alignment);
      params.screen_no = c.get_prop<int_v>("screen", options);
      params.dim.x = c.get_prop<int_v>("width", options);
      params.dim.y = c.get_prop<int_v>("height", options);
      params.pos.x = c.get_prop<int_v>("x", options);
      params.pos.y = c.get_prop<int_v>("y", options);
      params.client_coordinates = c.get_prop<bool>("client", options);
      params.direct_window = c.get_prop<bool>("direct", options, false);
      params.window_parameters = c.get_prop<hvalue>("parameters", options);
    }
    else
    {
      prq = new html::pump::request("sciter:msgbox.htm", html::DATA_HTML);
      if (!pv->load_data(prq, true))
        return JS_UNDEFINED;

      params.window_parameters = options;

      params.alignment = -5;
      //params.screen_no = html::screen_of(pv);
      params.data = prq->data();
      params.url = CHARS("sciter:msgbox.htm");
      params.direct_window = false;

    }

    xd = app()->create_dialog(params);
    xd->notify_data_arrived(nullptr, prq);

    bool r = xd->show_modal();
    hvalue retval = xd.ptr_of<xview>()->dialog_retval;
    xd.ptr_of<xview>()->dialog_retval.clear();
    return r ? retval : hvalue();
    //return hvalue();
  }

  helement screen_element_at(xcontext& c,int x, int y) {
    point screen_pt(x, y);
    return c.pview()->element_under_cursor(screen_pt);
  }

  JSOM_PASSPORT_BEGIN(Window_def, qjs::xview)
    JSOM_FUNC_DEF("move", window_move),
    JSOM_FUNC_DEF("moveTo", window_move_to),
    JSOM_FUNC_DEF("close", window_close),
    JSOM_FUNC_DEF("selectFile", window_select_file),
    JSOM_FUNC_DEF("selectFolder", window_select_folder),
    JSOM_FUNC_DEF("mediaVar", window_media_var),
    JSOM_FUNC_DEF("mediaVars", window_media_vars),

    JSOM_RO_PROP_DEF("graphicsBackend", window_graphics_backend),
    JSOM_RO_PROP_DEF("parameters", window_parameters),
    JSOM_FUNC_DEF("addEventListener", window_subscribe),
    JSOM_FUNC_DEF("dispatchEvent", window_dispatch_event),
    JSOM_FUNC_DEF("on", window_on),
    JSOM_FUNC_DEF("off", window_off),
    JSOM_FUNC_DEF("activate", window_activate),
    JSOM_FUNC_DEF("box", window_box),
    JSOM_FUNC_DEF("xcall", window_behavior_call),
    JSOM_FUNC_DEF("load", window_load),

    // attributes
    JSOM_RO_PROP_DEF("isAlive", window_is_alive),
    JSOM_PROP_DEF("state", window_get_state, window_set_state),
    JSOM_PROP_DEF("blurBehind", window_blurbehind, window_set_blurbehind),
    JSOM_PROP_DEF("frameType", window_frame_type, window_set_frame_type),
    JSOM_PROP_DEF("icon", window_icon, window_set_icon),
    JSOM_PROP_DEF("caption", window_get_caption, window_set_caption),
    JSOM_PROP_DEF("minSize", window_get_min_size, window_set_min_size),
    JSOM_PROP_DEF("maxSize", window_get_max_size, window_set_max_size),
    JSOM_PROP_DEF("isResizable", window_resizable, window_set_resizable),
    JSOM_PROP_DEF("isMaximizable", window_maximizable, window_set_maximizable),
    JSOM_PROP_DEF("isMinimizable", window_minimizable, window_set_minimizable),
    JSOM_PROP_DEF("isTopmost", window_topmost, window_set_topmost),
    JSOM_PROP_DEF("isEnabled", window_enabled, window_set_enabled),
    JSOM_RO_PROP_DEF("isActive", window_is_active ),
    JSOM_PROP_DEF("aspectRatio", window_aspect_ratio, window_set_aspect_ratio),

    JSOM_FUNC_DEF("trayIcon", window_trayicon),
    JSOM_FUNC_DEF("requestAttention", window_request_attention),

    JSOM_RO_PROP_DEF("document", window_document),

    JSOM_FUNC_DEF("screenBox", window_screenbox),
    JSOM_RO_PROP_DEF("screen", window_screen_no), JSOM_RO_PROP_DEF("monitor", window_screen_no),
    JSOM_FUNC_DEF("modal", window_modal),
    JSOM_FUNC_DEF("doEvent", window_do_event),
    JSOM_PROP_DEF("eventsRoot", window_events_root, window_set_events_root),

    JSOM_RO_PROP_DEF("parent", window_parent),

    JSOM_PROP_DEF("focus", window_focus, window_set_focus),
    JSOM_FUNC_DEF("focusable", window_focusable),

    JSOM_FUNC_DEF("performDrag", window_perform_drag),
    JSOM_FUNC_DEF("update", window_update),

  JSOM_PASSPORT_END

  JSOM_PASSPORT_BEGIN(Window_static_def, qjs::xcontext)
    JSOM_GLOBAL_RO_PROP_DEF("this", Window_this),
    JSOM_GLOBAL_RO_PROP_DEF("share", Window_share),
    JSOM_GLOBAL_RO_PROP_DEF("screens", Window_screens_total), JSOM_GLOBAL_RO_PROP_DEF("monitors", Window_screens_total),
    JSOM_GLOBAL_FUNC_DEF("screenBox", Window_screenbox),
    JSOM_GLOBAL_FUNC_DEF("ticks", Window_ticks),
    JSOM_GLOBAL_FUNC_DEF("post", window_post_event_all),
    JSOM_GLOBAL_FUNC_DEF("send", window_send_event_all),
    JSOM_GLOBAL_FUNC_DEF("elementAt", screen_element_at),

  JSOM_PASSPORT_END

  JSCFunctionListEntry Window_constants[] = {
    int_cdef("WINDOW_STATE_NA", gool::WINDOW_STATE_NA),
    int_cdef("WINDOW_SHOWN", gool::WINDOW_SHOWN),
    int_cdef("WINDOW_MINIMIZED", gool::WINDOW_MINIMIZED),
    int_cdef("WINDOW_MAXIMIZED", gool::WINDOW_MAXIMIZED),
    int_cdef("WINDOW_HIDDEN", gool::WINDOW_HIDDEN),
    int_cdef("WINDOW_FULL_SCREEN", gool::WINDOW_FULL_SCREEN),

    int_cdef("FRAME_WINDOW", gool::FRAME_WINDOW),
    int_cdef("DIALOG_WINDOW", gool::DIALOG_WINDOW),
    int_cdef("POPUP_WINDOW", gool::POPUP_WINDOW),
    int_cdef("TOOL_WINDOW", gool::TOOL_WINDOW),

  };

  xview* create_frame(xcontext& c, JSValue cfg, JSValue obj, xview* pv, html::WINDOW_TYPE wt = html::FRAME_WINDOW)
  {
    window_params params(wt);

    if (wt)
      params.set(wt);

    int_v wtype = c.get_prop<int_v>("type", cfg);
    if (wtype.is_defined())
      params.set((html::WINDOW_TYPE)wtype.val(0));

    if (!pv) {
      pv = c.get_prop<xview *>("parent", cfg, nullptr);
      if (!pv) {
        params.is_detached = true;
        pv = c.pxview();
      }
    }

    params.window_obj = JS_DupValue(c,obj);
    params.debug_mode = c.pdoc()->debug_mode();
    params.parent = pv;

    params.url = c.get_prop<string>("url", cfg);

    handle<html::pump::request> prq = new html::pump::request(params.url, html::DATA_HTML);

    string html = c.get_prop<string>("html", cfg, "");

    if (html.is_defined()) {
      //u8::from_utf16(html, prq->data, true);
      prq->data = html.chars_as_bytes();
    }
    else if (params.url.length()) {
      params.url = html::combine_url(pv->doc()->uri(), url::escape(params.url));
    }
    else
      throw qjs::om::type_error("either url or html are required");

    //params.data = prq->data();
    params.caption = c.get_prop<ustring>("caption", cfg,"(no caption)");
    params.alignment = c.get_prop<int>("alignment", cfg, 5);
    params.screen_no = c.get_prop<int_v>("screen", cfg);
    params.dim.x = c.get_prop<int_v>("width", cfg/*, (int)pv->pixels_per_dip(800)*/);
    params.dim.y = c.get_prop<int_v>("height", cfg/*, (int)pv->pixels_per_dip(600)*/);
    params.pos.x = c.get_prop<int_v>("x", cfg);
    params.pos.y = c.get_prop<int_v>("y", cfg);
    params.client_coordinates = c.get_prop<bool>("client", cfg);
    params.direct_window = c.get_prop<tristate_v>("direct", cfg);
    params.window_type = (html::WINDOW_TYPE)c.get_prop<int>("type", cfg, html::FRAME_WINDOW);

    int_v wstate = c.get_prop<int_v>("state", cfg);
    if (wstate.is_defined())
      params.window_state = (html::WINDOW_STATE)wstate.val(0);
    else
      params.window_state = html::WINDOW_STATE::WINDOW_SHOWN;

    params.window_parameters = c.get_prop<hvalue>("parameters", cfg);

    handle<html::view> frm = app()->create_frame(params);
    if (frm) {
      qjs::xview* pxv = frm.ptr_of<qjs::xview>();
      if (prq->data.length())
        frm->notify_data_arrived(nullptr, prq);
      if (params.window_state &&
        (frm->get_window_state() != params.window_state))
        frm->set_window_state(params.window_state);
      return pxv;
    }

    return nullptr;
  }

  int window_get_own_property(JSContext *ctx, JSPropertyDescriptor *desc, JSValueConst obj, JSAtom prop)
  {
    xview* pthis = conv<xview*>::try_unwrap(ctx, obj);
    if (!pthis)
      return false;
    hvalue val;

    atom_chars pname(ctx, prop);
    for (html::ctl *pbh = pthis->ground_behavior; pbh; pbh = pbh->next) {
      if (auto apsp = pbh->asset_get_passport()) {
        if (apsp->name == prop) {
          if (desc) {
            desc->flags = JS_PROP_C_W_E;
            desc->getter = JS_UNDEFINED;
            desc->setter = JS_UNDEFINED;
            desc->value = JS_UNDEFINED;
            som_asset_t* pass = pbh->as_asset();
            desc->value = conv<som_asset_t*>::wrap(ctx, pass);
          }
          return TRUE;
        }
      }
    }
    return FALSE;
  }

  JSClassExoticMethods Window_exotic_methods =
  {
    &window_get_own_property, // int(*get_own_property)(JSContext *ctx, JSPropertyDescriptor *desc, JSValueConst obj, JSAtom prop);
    NULL, // int(*get_own_property_names)(JSContext *ctx, JSPropertyEnum **ptab, uint32_t *plen, JSValueConst obj);
    /* return < 0 if exception, or TRUE/FALSE */
    NULL, // &element_delete_property,
    NULL, // int(*define_own_property)(JSContext *ctx, JSValueConst this_obj, JSAtom prop, JSValueConst val, JSValueConst getter, JSValueConst setter, int flags);
    NULL, //&element_has_property,
    NULL, //&element_get_property,
    NULL, //&element_set_property
  };

  void xview::init_class(JSContext* ctx)
  {
    xcontext c(ctx);

    JS_NewClassID(&Window_class_id);

    static JSClassDef Window_class = {
      "Window",
      [](JSRuntime *rt, JSValue val) { // finalizer
        xview* pc = (xview *)JS_GetOpaque(val, Window_class_id);
        if (pc) {
          pc->release();
        }
      },
      [](JSRuntime *rt, JSValueConst val, JS_MarkFunc *mark_func) {
        handle<xview> pc = (xview *)JS_GetOpaque(val, Window_class_id);
        if (pc && !pc->get_hwnd()) {
          JS_SetOpaque(val, nullptr);
          pc->release();
        }
        //JS_MarkValue(rt, pc->dialog_retval, mark_func);
      },
      NULL,
      &Window_exotic_methods
    };

    JS_NewClass(JS_GetRuntime(c), Window_class_id, &Window_class);
    JSValue window_proto = JS_NewObject(c);

    auto list = Window_def();
    JS_SetPropertyFunctionList(c, window_proto, list.start, list.length);

    static auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      JSValue obj;
      hvalue  proto;
      xcontext c(ctx);
      if( argc == 0 || !JS_IsObject(argv[0]))
        return JS_ThrowTypeError(ctx,"invalid window creation parameters");
      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetProperty(ctx, new_target, c.known_atoms().prototype);
      if (JS_IsException(proto))
        return JS_EXCEPTION;
      obj = JS_NewObjectProtoClass(ctx, proto, Window_class_id);
      if (JS_IsException(obj))
        return JS_EXCEPTION;
      xview* pxv = create_frame(c, argv[0], obj, nullptr);
      if (pxv)
        return pxv->obj;
      return JS_EXCEPTION;
    };

    hvalue window_class = JS_NewCFunction2(c, ctor, "Window", 2, JS_CFUNC_constructor, 0);
    JS_DefinePropertyValueStr(c, c.global(), "Window", JS_DupValue(c, window_class), JS_PROP_CONFIGURABLE);

    JS_SetPropertyFunctionList(c, window_class, Window_constants, items_in(Window_constants));
    auto Window_statics = Window_static_def();
    JS_SetPropertyFunctionList(c, window_class, Window_statics.start, Window_statics.length);

    JS_SetConstructor(c, window_class, window_proto);
    JS_SetClassProto(c, Window_class_id, window_proto);

    if (c.pdoc() == c.pview()->doc()) {
      // to force expando object creation for root document
      hvalue force_obj = c.val(c.pxview());
    }

  }
}

bool sciter_call_scripting_function(html::element *b, const char *name,  const tool::value *argv, uint argc, tool::value &retval, bool this_method) {
  if (!b)
    return false;
  if (this_method)
    return qjs::call(b, tool::string(name), int(argc), argv, &retval);
  else {
    html::document* pd = b->doc();
    if (!pd) return false;
    return qjs::call(pd, tool::string(name), int(argc), argv, &retval);
  }
}

// OBSOLETE, not used
bool sciter_get_expando(html::element *b, tool::value &pval, bool force_creation)
{
  return false;
}
