#pragma once

#ifndef __xdomjs_xtokenizer_h__
#define __xdomjs_xtokenizer_h__

#include "xcontext.h"
#include "xconv.h"
#include "xrange.h"
#include "html.h"

namespace qjs
{
  #define EOC wchar(-1)

  typedef tool::markup::scanner<wchar> markup_scanner;

  struct xtok_stream : public resource, public tool::markup::instream<wchar> {
    typedef tool::markup::instream<wchar> super;
    xtok_stream(ustring url) : super(url), chunk_src(nullptr), got_eof(false) {}

    virtual wchar get_char() {
      if (!chunk) {
        buffer.clear();
        if (got_eof || !get_chunk(buffer, chunk_src)) {
          got_eof = true;
          return 0;
        }
        chunk = buffer();
        ++line_no;
        return '\n';
      }

      if (until.length() && chunk.starts_with(until())) 
        return wchar(-1);

      wchar t = chunk++;
      if (t == '\n') ++line_no;
      return t;
    }
    uint get_pos() const { return uint(chunk.start - buffer.cbegin()); }

    virtual bool get_chunk(array<wchar> &ch, void *&ch_src) = 0;

    array<wchar> buffer;
    wchars       chunk;
    void *       chunk_src;
    ustring      until;
    bool         got_eof;
  };

  struct element_xtok_stream : public xtok_stream {

    typedef xtok_stream super;

    html::each_node gen;

    element_xtok_stream(html::element *pel) : super(string()), gen(pel) {}

    virtual bool get_chunk(array<wchar> &ch, void *&ch_src) {
      html::hnode pn;
      while (gen(pn)) {
        if (pn->is_text()) {
          html::text *pt = pn.ptr_of<html::text>();
          ch = pt->chars();
          ch_src = pn;

          if (pt->marks.length()) {
            html::element *pel = pn->nearest_text_box();
            if (pel) {
              pt->marks.clear();
              pel->drop_layout();
            }
          }
          return true;
        }
      }
      return false;
    }
  };


  struct xtokenizer : public resource {
    handle<xtok_stream> in;
    handle<xtokenizer>  next;
    hrange              range;
    ustring             saved_tail;

    xtokenizer(xtok_stream *ps, xtokenizer *n = nullptr) : in(ps), next(n) {
      range = new qjs::range();
    }

    virtual hvalue get_token(xcontext& c) = 0;
    virtual ustring get_value(xcontext& c) = 0;
    virtual ustring get_attr(xcontext& c) { return ustring(); }
    virtual ustring get_tag(xcontext& c) { return ustring(); }

    virtual bool is_markup() const { return false; }
    virtual bool is_source() const { return false; }
  };

  struct xtokenizer_markup : public xtokenizer {
    typedef xtokenizer super;
    markup_scanner scan;
    xtokenizer_markup(xtok_stream *ps, xtokenizer *n = nullptr) : super(ps, n), scan(*ps) 
    {
      scan.token_start_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->range->start = html::bookmark(pn, ps->get_pos() + off, false);
      };
      scan.token_end_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->range->end = html::bookmark(pn, ps->get_pos() + off, false);
      };
    }

    virtual bool is_markup() const { return true; }

    hvalue get_token(xcontext& c) override {
      switch (scan.get_token(false)) {
        case markup_scanner::TT_ERROR: return JS_AtomToString(c, c.known_atoms().error);
        case markup_scanner::TT_EOF: return JS_NULL; 
        case markup_scanner::TT_TAG_START: return JS_AtomToString(c, c.known_atoms().tag_start);
        case markup_scanner::TT_TAG_END:  return JS_AtomToString(c, c.known_atoms().tag_end);
        case markup_scanner::TT_TAG_HEAD_END: return JS_AtomToString(c, c.known_atoms().tag_head_end);
        case markup_scanner::TT_EMPTY_TAG_END: return JS_AtomToString(c, c.known_atoms().tag_empty_end);
        case markup_scanner::TT_ATTR: return JS_AtomToString(c, c.known_atoms().tag_attr);
        case markup_scanner::TT_TEXT: return JS_AtomToString(c, c.known_atoms().text);
        case markup_scanner::TT_COMMENT: return JS_AtomToString(c, c.known_atoms().comment);
        case markup_scanner::TT_CDATA: return JS_AtomToString(c, c.known_atoms().cdata);
        case markup_scanner::TT_PI: return JS_AtomToString(c, c.known_atoms().pi);
        case markup_scanner::TT_DOCTYPE: return JS_AtomToString(c, c.known_atoms().doc_type);
        case markup_scanner::TT_ENTITIY: return JS_AtomToString(c, c.known_atoms().entity);
        default: return JS_UNDEFINED;
      }
    }

    virtual ustring get_value(xcontext& c) override { return scan.get_value(); }
    virtual ustring get_attr(xcontext& c) override { return scan.get_attr_name(); }
    virtual ustring get_tag(xcontext& c) override { return scan.get_tag_name(); }
  };

  struct xtokenizer_source : public xtokenizer {
    typedef xtokenizer super;
    source_scanner     scan;

    xtokenizer_source(xtok_stream *ps, xtokenizer *n, string mime_type)
      : super(ps, n), scan(ps, mime_type)
    {

      scan.token_start_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->range->start = html::bookmark(pn, ps->get_pos() + off, false);
      };
      scan.token_end_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->range->end = html::bookmark(pn, ps->get_pos() + off, false);
      };
    }

    hvalue get_token(xcontext& c) override {
      switch (scan.get_token()) {
        case source_scanner::T_EOF: return JS_NULL;
        case source_scanner::T_NUMBER: return JS_AtomToString(c, c.known_atoms().number);
        case source_scanner::T_NUMBER_UNIT: return JS_AtomToString(c, c.known_atoms().number_unit);
        case source_scanner::T_COLOR: return JS_AtomToString(c, c.known_atoms().color);
        case source_scanner::T_STRING: return JS_AtomToString(c, c.known_atoms().string);
        case source_scanner::T_NAME: return JS_AtomToString(c, c.known_atoms().name);
        case source_scanner::T_COMMENT: return JS_AtomToString(c, c.known_atoms().comment);
        case source_scanner::T_OPERATOR: return JS_AtomToString(c, c.known_atoms().$operator);
        case source_scanner::T_OPAREN: return JS_AtomToString(c, c.known_atoms().opening_paren);
        case source_scanner::T_CPAREN:  return JS_AtomToString(c, c.known_atoms().closing_paren);
        case source_scanner::T_END_OF_ISLAND: return JS_AtomToString(c, c.known_atoms().end_of_island);
        case source_scanner::T_ERROR: return JS_AtomToString(c, c.known_atoms().error);
        default: return JS_AtomToString(c, c.known_atoms().unknown);
      }
    }

    ustring get_value(xcontext& c) override {
      return scan.get_value();
    }
  };

  extern JSClassID Tokenizer_class_id;

  void init_Tokenizer_class(xcontext& c);

  typedef handle<xtokenizer> hxtokenizer;

  template <> struct conv<hxtokenizer>
  {
    static hxtokenizer unwrap(JSContext * ctx, JSValueConst v)
    {
      hxtokenizer pr = (xtokenizer*)JS_GetOpaque(v, Tokenizer_class_id);
      return pr;
    }

    static JSValue wrap(JSContext * ctx, hxtokenizer pr) noexcept
    {
      if (!pr)
        return JS_NULL;
      JSValue tokenizer_obj = JS_NewObjectClass(ctx, Tokenizer_class_id);
      pr->add_ref();
      JS_SetOpaque(tokenizer_obj, pr.ptr());
      return tokenizer_obj;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Tokenizer_class_id) != NULL; }
  };

}

#endif
