#include "xtokenizer.h"
#include "xdom.h"
#include "xom.h"
#include "xrange.h"

namespace qjs
{
  using namespace html;
  using namespace tool;

  JSClassID Tokenizer_class_id;

  bool is_markup(chars mime) {
    return mime == CHARS("text/html") || mime == CHARS("text/svg") || mime == CHARS("text/xml");
  }

  hvalue tokenizer_push(xcontext&c, hvalue obj, string type, string until) 
  {
    handle<xtokenizer> xtkn = c.get<handle<xtokenizer>>(obj);
    if (!xtkn) return JS_FALSE;

    handle<xtokenizer> nxtkn;

    if (is_markup(type)) {
      if (xtkn->is_markup()) throw qjs::om::type_error("already is a markup tokenizer");
      nxtkn = new xtokenizer_markup(xtkn->in, xtkn);
    }
    else {
      if (xtkn->is_source()) throw qjs::om::type_error("already is a code tokenizer");
      nxtkn = new xtokenizer_source(xtkn->in, xtkn, type);
    }

    nxtkn->saved_tail = nxtkn->in->until;
    nxtkn->in->until = until;

    xtkn->release();
    nxtkn->add_ref();
    JS_SetOpaque(obj, nxtkn.detach());
    return obj;
  }
  hvalue tokenizer_pop(xcontext&c, hvalue obj)
  {
    handle<xtokenizer> xtkn = c.get<handle<xtokenizer>>(obj);
    if (!xtkn) return JS_FALSE;

    handle<xtokenizer> nxtkn = xtkn->next;
    if (!nxtkn) return obj;

    nxtkn->in->until = nxtkn->saved_tail;

    xtkn->next = nullptr;
    xtkn->release();
    nxtkn->add_ref();
    JS_SetOpaque(obj, nxtkn.detach());
    return obj;
  }

  array<int> tokenizer_element_type(xcontext&c, handle<xtokenizer> xtkn, string tag) {
    html::tag::symbol_t t = html::tag::symbol(tag, false);
    if (!t) 
      return { 0,0,0 };
    else 
      return {
        html::tag::type(t),
        html::tag::content_model(t),
        html::tag::parsing_model(t)
      };
  }

  hrange tokenizer_token_range(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return nullptr;
    return xtkn->range;
  }

  hvalue tokenizer_token_type(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return hvalue();
    return xtkn->get_token(c);
  }

  ustring tokenizer_token_value(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return ustring();
    return xtkn->get_value(c);
  }

  ustring tokenizer_tag(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return ustring();
    return xtkn->get_tag(c);
  }

  ustring tokenizer_attr_name(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return ustring();
    return xtkn->get_attr(c);
  }

  helement tokenizer_element(xcontext&c, handle<xtokenizer> xtkn) {
    if (!xtkn) return nullptr;
    html::node *pn = static_cast<html::node *>(xtkn->in->chunk_src);
    if (pn) return pn->get_element();
    return nullptr;
  }
  
  JSOM_PASSPORT_BEGIN(Tokenizer_def, xtokenizer)
    JSOM_CONST_STR("[Symbol.toStringTag]", "Tokenizer", JS_PROP_CONFIGURABLE),
    //JSOM_PROP_DEF("link", state_link, state_set_link),
    JSOM_RO_PROP_DEF("element", tokenizer_element),
    JSOM_FUNC_DEF("push", tokenizer_push),
    JSOM_FUNC_DEF("pop", tokenizer_pop),
    JSOM_FUNC_DEF("nextToken", tokenizer_token_type),

    JSOM_FUNC_DEF("elementType", tokenizer_element_type),
    JSOM_RO_PROP_DEF("tokenRange", tokenizer_token_range),
    JSOM_RO_PROP_DEF("tokenValue", tokenizer_token_value),
    JSOM_RO_PROP_DEF("markupTag", tokenizer_tag),
    JSOM_RO_PROP_DEF("markupAttributeName", tokenizer_attr_name),
    JSOM_RO_PROP_DEF("markupAttributeValue", tokenizer_token_value),
  JSOM_PASSPORT_END

  void init_Tokenizer_class(context& c)
  {
    JS_NewClassID(&Tokenizer_class_id);

    static JSClassDef Tokenizer_class = {
      "Tokenizer",
      [](JSRuntime *rt, JSValue val)
      {
        xtokenizer* pe = (xtokenizer*)JS_GetOpaque(val,Tokenizer_class_id);  //object_of<html::element>(val);
        if (pe) pe->release();
      }
    };

    JS_NewClass(JS_GetRuntime(c), Tokenizer_class_id, &Tokenizer_class);
    JSValue tokenizer_proto = JS_NewObject(c);

    auto list = Tokenizer_def();
    JS_SetPropertyFunctionList(c, tokenizer_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      html::helement el;
      string   type;
      xcontext c(ctx);
      handle<xtokenizer> htz;
      handle<xtok_stream> ps;

      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto))
        goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, Tokenizer_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj))
        goto fail;

      el = argc > 0 ? c.get<helement>(argv[0]): nullptr;
      if (!el) {
        JS_ThrowTypeError(ctx, "not a DOM element");
        goto fail;
      }
      
      ps = new element_xtok_stream(el);

      type = argc > 1 ? c.get<string>(argv[1]) : string();
      if (type.like("text/htm*") || type == CHARS("text/svg") || type.like("text/xml*")) {
        htz = new xtokenizer_markup(ps);
      } else if (type.like("text/*") || type.like("*/json") || type.like("*script") || type.like("application/*")) {
        htz = new xtokenizer_source(ps, nullptr, type == CHARS("text/css"));
      } else {
        JS_ThrowTypeError(ctx, "not supported mime type");
        goto fail;
      }
      
      JS_SetOpaque(obj, htz.detach());
      return obj;
    fail:
      //js_free(ctx, s);
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue tokenizer_class = JS_NewCFunction2(c, ctor, "Tokenizer", 2, JS_CFUNC_constructor, 0);
    JS_DefinePropertyValueStr(c, c.global(), "Tokenizer", JS_DupValue(c, tokenizer_class), JS_PROP_CONFIGURABLE);

    JS_SetConstructor(c, tokenizer_class, tokenizer_proto);
    JS_SetClassProto(c, Tokenizer_class_id, tokenizer_proto);
  }

}