#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "sciter-revision.h"
#include "tool/tl_lzf.h"

namespace qjs {

  using namespace html;

  static helement sciter_select(xcontext& c, ustring selector) {
    return html::find_first(*c.pview(), c.pdoc(), selector);
  }

  static array<helement> sciter_select_all(xcontext& c, ustring selector) {
    array<helement> r;
    find_all(*c.pview(), r, c.pdoc(), selector);
    return r;
  }

  document* sciter_on(xcontext& c, ustring name, qjs::hvalue vselector, qjs::hvalue fcn)
  {
    ustring selector;
    if (fcn.is_nothing()) {
      fcn = vselector;
    }
    else {
      selector = conv<ustring>::unwrap(c, vselector);
    }
    if (!c.is_event_handler(fcn)) {
      string sn = conv<string>::unwrap(c, fcn);
      throw qjs::om::type_error(string::format("%s is not an event handler", sn.c_str()));
    }
    c.pdoc()->subscribe(fcn, name, selector);
    return c.pdoc();
  }
  document* sciter_off(xcontext& c, qjs::hvalue what)
  {
    if (c.is_event_handler(what))
      c.pdoc()->unsubscribe(what);
    else
      c.pdoc()->unsubscribe(conv<ustring>::unwrap(c, what));
    return c.pdoc();
  }

  int device_pixels(xcontext& c, JSValue of_what, ustring axis)
  {
    if (conv<int>::isa(c, of_what)) {
      int dips = conv<int>::unwrap(c, of_what);
      if (axis == CHARS("height"))
        return (int)c.pview()->pixels_per_dip(gool::size(dips)).y;
      else
        return (int)c.pview()->pixels_per_dip(gool::size(dips)).x;
    }
    else if (conv<float>::isa(c, of_what)) {
      float dips = conv<float>::unwrap(c, of_what);
      if (axis == CHARS("height"))
        return (int)c.pview()->pixels_per_dip(gool::sizef(dips)).y;
      else
        return (int)c.pview()->pixels_per_dip(gool::sizef(dips)).x;
    }

    else {
      tool::value v = tool::value::parse(conv<ustring>::unwrap(c, of_what));
      if (!v.is_length())
        throw qjs::om::type_error("not a length");
      if (axis == CHARS("height"))
        return html::pixels(*c.pview(), c.pdoc(), html::size_v(v)).width();
      else
        return html::pixels(*c.pview(), c.pdoc(), html::size_v(v)).height();
    }
  }

  string sciter_printf(xcontext& c, string fmt, slice<JSValue> args) {
    return c.format(fmt(), args);
  }

  string sciter_uuid(xcontext& c, bool sequential) 
  {
    string uid = tool::unique_id();
    if (sequential) {
      tool::datetime_t t = tool::date_time::now().time();
      // this ensures that two consequtive calls return distinct values.
      static tool::mutex      guard;
      static tool::datetime_t last_t = 0;
      tool::critical_section  _(guard);
      if (t <= last_t) t = last_t + 1;
      last_t = t;
      return tool::string::format("%08X%08X.%s", hidword(t),lodword(t), uid.c_str());
    }
    else
      return uid;
  }

  array<byte> sciter_encode(xcontext& c, ustring data, string encoding) {
    if (!encoding) encoding = CHARS("utf-8");
    array<byte> encoded;
    if (!tool::encode_bytes(data(), encoded, encoding))
      throw qjs::om::type_error("unknown encoding");
    return encoded;
  }

  ustring sciter_decode(xcontext& c, array<byte> data, string encoding) {
    if (!encoding) encoding = CHARS("utf-8");
    ustring decoded;
    if (!tool::decode_bytes(data(), decoded, encoding))
      throw qjs::om::type_error("unknown encoding");
    return decoded;
  }

  static tool::array<byte> compress(xcontext& c, tool::array<byte> what, string alg)
  {
    tool::array<byte> compressed;
    if (alg == CHARS("gz"))
      tool::gzip(true, what, compressed, false);
    else if (alg == CHARS("gzip"))
      tool::gzip(true, what, compressed, true);
    else
      tool::lzf::compress(what, compressed);
    return compressed;
  }

  static tool::array<byte> decompress(xcontext& c, tool::array<byte> what, string alg)
  {
    tool::array<byte> decompressed;
    if (alg == CHARS("gz"))
      tool::gzip(false, what, decompressed, false);
    else if (alg == CHARS("gzip"))
      tool::gzip(false, what, decompressed, true);
    else
      tool::lzf::decompress(what, decompressed);
    return decompressed;
  }

  static string md5(xcontext& c, tool::array<byte> what)
  {
    return tool::md5(what).to_string();
  }

  static uint crc32(xcontext& c, tool::array<byte> what)
  {
    return tool::crc32(what);
  }

  static string toBase64(xcontext& c, tool::array<byte> what)
  {
    tool::array<char> b64;
    tool::base64_encode(what, b64);
    return b64();
  }

  static tool::array<byte> fromBase64(xcontext& c, string what)
  {
    array<byte> data;
    tool::base64_decode(what, data);
    return data;
  }

  static hvalue compileFile(xcontext& c, string filename, tristate_v module) {
    int eval_flags;
    hvalue obj;

    handle<html::request> rq = new html::request(filename, RESOURCE_DATA_TYPE::DATA_SCRIPT);
    if(!c.pview()->load_data(rq, true))
      return JS_ThrowTypeError(c, "could not load '%s'", filename.c_str());

    eval_flags = JS_EVAL_FLAG_COMPILE_ONLY;
    if (module.is_undefined())
      module = filename.like("*.mjs") || JS_DetectModule((const char *)rq->data.cbegin(), rq->data.length());
    if (module)
      eval_flags |= JS_EVAL_TYPE_MODULE;
    else
      eval_flags |= JS_EVAL_TYPE_GLOBAL;
    obj = JS_Eval(c, (const char *)rq->data.cbegin(), rq->data.length(), filename, eval_flags);
    if (JS_IsException(obj))
      return obj;

    uint8_t *out_buf; size_t out_buf_len;
    int flags = JS_WRITE_OBJ_BYTECODE;
    out_buf = JS_WriteObject(c, &out_buf_len, obj, flags);
    if (!out_buf)
      //return JS_GetException(c);
      return JS_ThrowTypeError(c, "could not serialize '%s'", filename.c_str());

    obj = c.val(bytes(out_buf, out_buf_len));
    js_free(c, out_buf);
    return obj;
  }
  
  static hvalue import(xcontext& c, string filename) {
    return JS_ImportModuleSync(c, filename);
  }
    
  hvalue load_native_library(xcontext& c, ustring dll_name);

  JSOM_PASSPORT_BEGIN(Sciter_def, qjs::xcontext)
    JSOM_GLOBAL_FUNC_DEF("$", sciter_select),
    JSOM_GLOBAL_FUNC_DEF("$$", sciter_select_all),
    JSOM_GLOBAL_FUNC_DEF("on", sciter_on),
    JSOM_GLOBAL_FUNC_DEF("off", sciter_off),
    JSOM_GLOBAL_FUNC_DEF("devicePixels", device_pixels),
    //JSOM_GLOBAL_FUNC_DEF("printf", sciter_printf),
    JSOM_GLOBAL_FUNC_DEF("uuid", sciter_uuid),
    JSOM_GLOBAL_FUNC_DEF("loadLibrary", load_native_library),

    JSOM_GLOBAL_FUNC_DEF("encode", sciter_encode),
    JSOM_GLOBAL_FUNC_DEF("decode", sciter_decode),

    JSOM_GLOBAL_FUNC_DEF("compress", compress),
    JSOM_GLOBAL_FUNC_DEF("decompress", decompress),
    JSOM_GLOBAL_FUNC_DEF("md5", md5),
    JSOM_GLOBAL_FUNC_DEF("crc32", crc32),
    JSOM_GLOBAL_FUNC_DEF("toBase64", toBase64),
    JSOM_GLOBAL_FUNC_DEF("fromBase64", fromBase64),

    JSOM_GLOBAL_FUNC_DEF("compileFile", compileFile),
    JSOM_GLOBAL_FUNC_DEF("import", import),

    JSOM_CONST_STR("VERSION", SCITER_VERSION_FULL_STR, JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
    JSOM_CONST_STR("REVISION", SCITER_REVISION_STR, JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
    JSOM_CONST_STR("QUICKJS_VERSION", QUICKJS_VERSION, JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE)
  JSOM_PASSPORT_END

  static auto member_defs = Sciter_def();

  static int sciter_module_init(JSContext *ctx, JSModuleDef *m)
  {
    return JS_SetModuleExportList(ctx, m, member_defs.start, member_defs.length);
  }

  void init_sciter_module(context& c)
  {
    JSModuleDef *m = JS_NewCModule(c, "@sciter", sciter_module_init);
    if (!m) return;
    JS_AddModuleExportList(c, m, member_defs.start, member_defs.length);
  }

}
