#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "xcontext.h"
#include "xgraphics.h"

namespace html
{
  using namespace qjs;
}

namespace qjs {

  using namespace html;

  JSClassID Graphics_class_id = 0;

  template <>
  inline xgraphics* object_of<xgraphics>(JSValue val) {
    return (xgraphics*)JS_GetOpaque(val, Graphics_class_id);
  }

#if 1
  xgraphics* xgraphics::draw(xcontext& c, xgraphics* pg, hvalue what, hvalue how)
  {
    auto gfx = pg->get_gfx(c);
    if (!gfx) return pg;

    void* ptr = nullptr;
    JSClassID cid = JS_GetClassID(what, &ptr);
    if (!ptr)
      return pg;

    bool   fill = true;
    bool   stroke = true;
    pointf pos;
    if (c.is_object(how)) {
      fill = c.get_prop<bool>(c.known_atoms().fill, how, false);
      stroke = c.get_prop<bool>(c.known_atoms().stroke, how, false);
      if (!fill && !stroke) fill = true;
      pos.x = c.get_prop<float>(c.known_atoms().x, how, 0.0f);
      pos.y = c.get_prop<float>(c.known_atoms().y, how, 0.0f);
      if (fill) {
        string fill_rule = c.get_prop<string>(c.known_atoms().fill, how, string("nonzero"));
        gfx->set_fill_even_odd(fill_rule == CHARS("evenodd"));
      }
    }

    if (cid == Graphics_Path_class_id) {
      xpath* xp = static_cast<xpath*>(ptr);
      gfx->translate(pos);
      gfx->draw_path(xp->cpath,stroke,fill);
      gfx->translate(-pos);
    }
    else if (cid == Graphics_Text_class_id) 
    {
      xtext* xp = static_cast<xtext*>(ptr);

      sizef ppx = c.pview()->ppx_to_dip(sizef(1, 1));

      gool::argb clr = c.get_prop<gool::argb>(c.known_atoms().fill, how, gool::argb(0,0,0,0));
      int alignment = c.get_prop<int>(c.known_atoms().alignment, how, 0);

      //pointf dst;
      //dst.x = c.get_prop<float>(c.known_atoms().x, how, 0.0f);
      //dst.y = c.get_prop<float>(c.known_atoms().y, how, 0.0f);
      
      rectf rc(pos, xp->tl->get_box() * ppx);
      if (alignment) rc.pointOf(alignment, pos);

      gool::state _(gfx);

      gfx->translate(rc.s);
      gfx->scale(ppx);

      if (clr.alfa)
        gfx->draw(xp->tl, pointf(0,0), clr);
      else
        gfx->draw(xp->tl, pointf(0, 0));

    }
    else if (cid == Image_class_id) {
      image* xp = static_cast<image*>(ptr);
      sizef dim = xp->dim();
      dim.x = c.get_prop<float>(c.known_atoms().width, how, dim.x);
      dim.y = c.get_prop<float>(c.known_atoms().height, how, dim.y);
      //dim = c.pview()->ppx_to_dip(dim);

      point src_pos;
      size  src_dim = xp->dim();

      src_pos.x = c.get_prop<int>(c.known_atoms().srcX, how, 0);
      src_pos.y = c.get_prop<int>(c.known_atoms().srcY, how, 0);

      src_dim.x = c.get_prop<int>(c.known_atoms().srcWidth, how, src_dim.x);
      src_dim.y = c.get_prop<int>(c.known_atoms().srcHeight, how, src_dim.y);

      byte opacity = byte(255.0f * c.get_prop<float>(c.known_atoms().opacity, how, 1.0f));
     
      gfx->draw_image(xp,gool::rectf(pos,dim), gool::rect(src_pos,src_dim),opacity);
    }
    else if (cid == Element_class_id) {
      element* el = static_cast<element*>(static_cast<node*>(ptr));
      //sizef dim = el->dim();
      //dim.x = c.get_prop<float>(c.known_atoms().width, how, dim.x);
      //dim.y = c.get_prop<float>(c.known_atoms().height, how, dim.y);
      gool::state _(gfx);
      gfx->scale(c.pview()->ppx_to_dip(sizef(1, 1)));
      el->draw(*c.pview(), gfx, pos);
    }

    return pg;
  }
#endif

  JSOM_PASSPORT_BEGIN(Graphics_def, xgraphics)
    JSOM_CONST_STR("[Symbol.toStringTag]", "Graphics2D", JS_PROP_CONFIGURABLE),

    JSOM_FUNC_DEF("clearRect", xgraphics::clear_rect),

    JSOM_FUNC_DEF("beginPath", xgraphics::begin_path),
    JSOM_FUNC_DEF("moveTo", xgraphics::move_to),
    JSOM_FUNC_DEF("lineTo", xgraphics::line_to),
    JSOM_FUNC_DEF("quadraticCurveTo", xgraphics::quadratic_curve_to),
    JSOM_FUNC_DEF("bezierCurveTo", xgraphics::bezier_curve_to),
    JSOM_FUNC_DEF("arc", xgraphics::arc),
    JSOM_FUNC_DEF("arcTo", xgraphics::arc_to),
    JSOM_FUNC_DEF("ellipse", xgraphics::ellipse),
    JSOM_FUNC_DEF("rect", xgraphics::rect),
    JSOM_FUNC_DEF("closePath", xgraphics::close_path),

    JSOM_FUNC_DEF("stroke", xgraphics::stroke),
    JSOM_FUNC_DEF("fill", xgraphics::fill),

    JSOM_FUNC_DEF("fillRect", xgraphics::fill_rect),
    JSOM_FUNC_DEF("fillText", xgraphics::fill_text),
    JSOM_FUNC_DEF("strokeRect", xgraphics::stroke_rect),

    JSOM_PROP_DEF("strokeStyle", xgraphics::get_stroke_style, xgraphics::set_stroke_style),
    JSOM_PROP_DEF("lineWidth", xgraphics::get_stroke_width, xgraphics::set_stroke_width),
    JSOM_PROP_DEF("strokeWidth", xgraphics::get_stroke_width, xgraphics::set_stroke_width),
    JSOM_PROP_DEF("fillStyle", xgraphics::get_fill_style, xgraphics::set_fill_style),
    JSOM_FUNC_DEF("setLineDash", xgraphics::set_line_dash),
    JSOM_FUNC_DEF("setStrokeDash", xgraphics::set_line_dash),

    JSOM_PROP_DEF("lineCap", xgraphics::get_line_cap, xgraphics::set_line_cap),
    JSOM_PROP_DEF("lineJoin", xgraphics::get_line_join, xgraphics::set_line_join),

    JSOM_FUNC_DEF("save", xgraphics::save),
    JSOM_FUNC_DEF("restore", xgraphics::restore),

    JSOM_FUNC_DEF("scale", xgraphics::scale),
    JSOM_FUNC_DEF("translate", xgraphics::translate),
    JSOM_FUNC_DEF("rotate", xgraphics::rotate),
    JSOM_FUNC_DEF("transform", xgraphics::transform),
    JSOM_FUNC_DEF("setTransform", xgraphics::reset_transform),

    JSOM_PROP_DEF("font", xgraphics::get_font, xgraphics::set_font),

    JSOM_FUNC_DEF("createLinearGradient", xgraphics::create_linear_gradient),
    JSOM_FUNC_DEF("createRadialGradient", xgraphics::create_radial_gradient),


    JSOM_FUNC_DEF("pushLayer", xgraphics::push_layer),
    JSOM_FUNC_DEF("popLayer", xgraphics::pop_layer),

    JSOM_FUNC_DEF("draw", xgraphics::draw),

  JSOM_PASSPORT_END

  void init_Image_class(context& c, hvalue graphics);
  void init_Color_class(context& c, hvalue graphics);
  void init_Text_class(context& c, hvalue graphics);
  void init_Path_class(context& c, hvalue graphics);
  void init_Brush_class(context& c, hvalue graphics);

  void init_Graphics_class(context& c)
  {
    JS_NewClassID(&Graphics_class_id);

    static JSClassDef Graphics_class = {
      "Graphics2D",
      [](JSRuntime *rt, JSValue val)
      {
        xgraphics* pg = object_of<xgraphics>(val);
        if (pg) {
          pg->commit();
          pg->release();
        }
      }
    };

    JS_NewClass(JS_GetRuntime(c), Graphics_class_id, &Graphics_class);
    JSValue graphics_proto = JS_NewObject(c);

    auto list = Graphics_def();
    JS_SetPropertyFunctionList(c, graphics_proto, list.start, list.length);

    //hvalue node_proto = JS_GetClassProto(c, Node_class_id);
    //JS_SetPrototype(c, graphics_proto, node_proto);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto))
        goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, Graphics_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj))
        goto fail;
      JS_SetOpaque(obj, 0);
      return obj;
    fail:
      //js_free(ctx, s);
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue graphics_class = JS_NewCFunction2(c, ctor, "Graphics", 2, JS_CFUNC_constructor, 0);

    JS_DefinePropertyValueStr(c, c.global(), "Graphics", JS_DupValue(c, graphics_class), JS_PROP_CONFIGURABLE);

    //auto static_list = Graphics_static_def();
    //JS_SetPropertyFunctionList(c, graphics_class, static_list.start, static_list.length);

    JS_SetConstructor(c, graphics_class, graphics_proto);
    JS_SetClassProto(c, Graphics_class_id, graphics_proto);

    init_Image_class(c, graphics_class);
    init_Color_class(c, graphics_class);
    init_Path_class(c, graphics_class);
    init_Text_class(c, graphics_class);
    init_Brush_class(c, graphics_class);

  }

  float  canvas_width(xcontext& c, element* pel);
  float  canvas_height(xcontext& c, element* pel);
  bool   canvas_set_width(xcontext& c, element* pel, float v);
  bool   canvas_set_height(xcontext& c, element* pel, float v);

  hvalue canvas_context(xcontext& c, element* pel, string type);
  bool   canvas_update(xcontext& c, element* pel, hvalue func);

  JSOM_PASSPORT_BEGIN(Element_Canvas_def, html::element)
    JSOM_PROP_DEF("width", canvas_width, canvas_set_width),
    JSOM_PROP_DEF("height", canvas_height, canvas_set_height),
    JSOM_FUNC_DEF("getContext", canvas_context),
    JSOM_FUNC_DEF("canvasUpdate", canvas_update),
  JSOM_PASSPORT_END

  struct canvas : public html::ctl
  {
    DEFINE_TYPE_ID(canvas)

    weak_handle<html::element> host;
    handle<xgraphics>          xgfx;
    size   dim;

    hvalue  obj;

    virtual html::CTL_TYPE get_type() { return html::CTL_IMAGE; }

    canvas(xview* pv, html::element* self) : ctl(html::HANDLE_DRAW | html::HANDLE_SIZE) { host = self; }

    virtual const string &behavior_name() const override {
      static string name("graphics");
      return name;
    }

    virtual bool get_expando(html::context& hc, script_expando& seo) 
    { 
      xcontext& c = static_cast<xcontext&>(hc);
      if (!obj) {
        obj = JS_NewObject(c);
        auto list = Element_Canvas_def();
        JS_SetPropertyFunctionList(c, obj, list.start, list.length);
      }
      seo = JS_DupValue(c, obj);
      return true;
    }

    xgraphics* get_graphics() {
      if (!xgfx && host) {
        auto w = host->atts.get_width_size();
        auto h = host->atts.get_height_size();
        if (w.is_defined() && h.is_defined()) {
          html::view* pv = host->pview();
          if (pv) {
            dim.x = html::pixels(*pv, host, w).width();
            dim.y = html::pixels(*pv, host, h).height();
          }
        }
        else {
          if (!host->dim().empty())
            dim = host->dim();
          else
            dim = size(300, 150);
        }
        xgfx = new xgraphics(host, dim);
      }
      return xgfx;
    }

    virtual bool attach(html::view &v, html::element *self) override {
      return ctl::attach(v, self);
    }
    virtual void detach(html::view &v, html::element *self) override {
      obj.clear();
      if(xgfx)
        xgfx->detach();
      return ctl::detach(v, self);
    }

    virtual bool draw_content(html::view &v, html::element *self, gool::graphics *sf, gool::point pos) override
    {
      if (xgfx) {
        handle<bitmap> bmp = xgfx->get_bitmap_for(sf);
        sf->draw(bmp, pos);
        return true;
      } else 
        return false;
    }

    //virtual image *get_fore_image(html::view &v, html::element *self) override {
    //  return img;
    //}

#if 0
    SOM_PASSPORT_BEGIN(canvas)
/*      SOM_FUNCS(
        SOM_FUNC_EX(load, api_load),
        )
      SOM_PROPS(
        SOM_RO_VIRTUAL_PROP(graphics, get_graphics),
      )*/
    SOM_PASSPORT_END
#endif
  };

  float canvas_width(xcontext& c, element* pel) {
    int w;
    if (!pel->is_layout_valid())
      w = pel->declared_width(*c.pview(), 0);
    else
      w = pel->dim().x;
    return c.pview()->ppx_to_dip(sizef(float(w), float(w))).x;
  }
  float canvas_height(xcontext& c, element* pel) {
    int h;
    if (!pel->is_layout_valid())
      h = pel->declared_height(*c.pview(), 0);
    else
      h = pel->dim().y;
    return c.pview()->ppx_to_dip(sizef(float(h), float(h))).y;
  }

  bool   canvas_set_width(xcontext& c, element* pel, float v)
  {
    pel->set_style_attribute(html::cssa_width, value::make_length(v, value::px));
    return true;
  }
  bool   canvas_set_height(xcontext& c, element* pel, float v) {
    pel->set_style_attribute(html::cssa_height, value::make_length(v, value::px));
    return true;
  }
  
  hvalue canvas_context(xcontext& c, element* pel, string type)
  {
    if (type == CHARS("2d")) {
      canvas* pc = pel->get_behavior<canvas>();
      xgraphics* pg = pc->get_graphics();
      return conv<xgraphics*>::wrap(c, pg);
    }
    return JS_NULL;
  }

  bool canvas_update(xcontext& c, element* pel, hvalue func) {
    if (!c.is_function(func))
      throw qjs::om::type_error("function expected");
    canvas* pc = pel->get_behavior<canvas>();
    xgraphics* pg = pc->get_graphics();
    ON_SCOPE_EXIT(pg->commit());
    try {
      bool dummy;
      c.call(dummy, func, pel, pg);
      pel->refresh(*c.pview());
      return true;
    } catch(qjs::exception) {
      c.report_exception();
    }
    return false;
  }
 

  html::ctl* create_canvas_for(xview* pv, html::element* b) {
    return new canvas(pv, b);
  }

  
  bool immediate_draw(xcontext& c, element* pel, gool::graphics* pg, gool::point pos, JSAtom name) {
    handle<xgraphics> xgfx = new xgraphics(c.pview(), pel, pg, false);

    hvalue func = c.get_prop<hvalue>(name, pel->obj);
    if (!c.is_function(func)) return false;

    gool::aa_mode _0(pg);
    gool::state   _1(pg);
    tool::auto_state<html::graphics *> _3(c.pview()->drawing_surface, pg);

    pg->translate(pos);
    
    xgfx->setup_gfx(c);

    //ON_SCOPE_EXIT(xgfx->obj.clear());

    bool handled = false;

    try {
      c.call(handled, func, pel->obj, xgfx.ptr());
    }
    catch (qjs::exception) {
      c.report_exception();
      return false;
    }
    return handled;
  }

  bool xview::on_element_draw_background(gool::graphics *pg, html::element *el, gool::point pos)
  {
    html::document* pd = el->doc();
    if (!pd) return false;
    if (!el->obj) return false;
    xcontext c(pd->ns);
    return immediate_draw(c, el, pg, pos, c.known_atoms().paintBackground);
  }
  bool xview::on_element_draw_foreground(gool::graphics *pg, html::element *el, gool::point pos)
  {
    html::document* pd = el->doc();
    if (!pd) return false;
    if (!el->obj) return false;
    xcontext c(pd->ns);
    return immediate_draw(c, el, pg, pos, c.known_atoms().paintForeground);
  }
  bool xview::on_element_draw_content(gool::graphics *pg, html::element *el, gool::point pos)
  {
    html::document* pd = el->doc();
    if (!pd) return false;
    if (!el->obj) return false;
    xcontext c(pd->ns);
    return immediate_draw(c, el, pg, pos, c.known_atoms().paintContent);
  }
  bool xview::on_element_draw_outline(gool::graphics *pg, html::element *el, gool::point pos)
  {
    html::document* pd = el->doc();
    if (!pd) return false;
    if (!el->obj) return false;
    xcontext c(pd->ns);
    return immediate_draw(c, el, pg, pos, c.known_atoms().paintOutline);
  }

}