#include "xsciter.h"
#include "xdom.h"
#include "xgraphics.h"

namespace tis {

  xview *xvm::current_view() {
    if (_current_view) return _current_view;
    html::element *pd = xvm::current_doc();
    if (!pd) return nullptr;
    return static_cast<xview *>(pd->pview());
  }

  html::document *xvm::current_doc() {
    value        gns      = currentScope()->globals;
    static value self_sym = CsSymbolOf("self");
    value        self_obj = NULL_VALUE;

    if (!CsNamespaceP(gns)) return nullptr;

    if (!CsGetProperty(this, gns, self_sym, &self_obj)) return nullptr;

    html::element *pel = element_ptr_no_throw(this, self_obj);
    if (pel && pel->is_document()) return pel->cast<html::document>();
    return nullptr;
  }

  struct scripting_debug_output : public html::view_debug_output,
    gc_callback,
    functor
  {
    DEFINE_TYPE_ID(scripting_debug_output)

      xview *pv;
      value  lcb;

    struct item {
      uint        subs;
      uint        sev;
      tool::ustring text; // text to print into stdout, etc.
      tool::value   data; // data to output to inspector's console
      tool::value   mode;
      tool::value   location;
      bool        as_stringizer = false;
    };

    array<item> items;

    scripting_debug_output(xview *pxv, value cb)
      : pv(pxv), gc_callback(pxv->vm), lcb(cb) {}
    virtual ~scripting_debug_output() {}
    virtual void print(uint subs, uint sev, wchars msg) {
      item it;
      it.subs = subs;
      it.sev = sev;
      it.text = msg;
      items.push(it);
      pv->post(this);
    }

    virtual void log(uint subs, uint sev, slice<tis::value> list, tis::value mode, tis::value location, bool as_stringizer)
    {
      item it;
      array<tool::value> tlist(list.length);

      string_stream ss;

      if (mode != UNDEFINED_VALUE) {
        CsDisplay(c, mode, &ss);
        ss.put_str(": ");
      }
      
      for (uint n = 0; n < list.length; ++n) {
        if(n > 0) ss.put_str(" ");
        CsPrintData(c, list[n], &ss, false);
        tlist[n] = value_to_value_def(pv->vm, list[n]);
      }

      ss.put_str("\n");

      it.subs = subs;
      it.sev = sev;
      it.text = ss.to_ustring();
      it.data = tool::value::make_array(tlist());
      it.mode = value_to_value_def(pv->vm, mode);
      it.location = value_to_value_def(pv->vm, location);
      it.as_stringizer = as_stringizer;
      items.push(it);
      pv->post(this);
    }

    virtual bool is_debug_peer() const { return true; }

    virtual bool operator()() {
      //::debug_print(subs, sev, msg);
      auto_scope as(c, CsMethodNamespace(lcb));
      for (int n = 0; n < items.size(); ++n) {
        const item &it = items[n];
        TRY{
          value vmsg = 0, vmode = 0, vloc = 0;
          PROTECT(vmsg, vmode, vloc);
          if(it.data.is_defined())
            vmsg = value_to_value(c, it.data);
          else
            vmsg = string_to_value(c, it.text);
          vmode = value_to_value(c, it.mode);
          vloc = value_to_value(c, it.location);
          value res = CsCallFunction(&as, lcb, 6, 
            CsMakeInteger(it.subs),
            CsMakeInteger(it.sev), 
            vmsg, 
            vmode, 
            vloc, 
            (it.as_stringizer ? TRUE_VALUE : FALSE_VALUE));
          if (res != TRUE_VALUE) pv->do_debug_print(it.subs, it.sev, it.text);
        }
        CATCH_ERROR_NE {
          assert(false);
          pv->do_debug_print(it.subs, it.sev, it.text);
        }
      }
      items.clear();
      return true;
    }

    virtual void on_GC(VM *c) { lcb = CsCopyValue(c, lcb); }
  };

  bool xvm::has_log_values() {
    xview* pv = current_view();
    return pv && pv->debug_output;
  }

  bool xvm::log_values(slice<value> list, value mode, value location, bool as_stringizer)
  {
    xview* pv = current_view();
    if (!pv) 
      return false;

    if (pv->debug_output && pv->debug_output->is_of_type<scripting_debug_output>()) {
      scripting_debug_output* sdo = pv->debug_output.ptr_of<scripting_debug_output>();
      sdo->log(3, 0, list, mode, location, as_stringizer);
      return true;
    }
    else {
      //string_stream s;
      int out = 0;
      if (mode != UNDEFINED_VALUE) {
        standardOutput->put_str(value_to_string(mode));
        standardOutput->put_str(": ");
        ++out;
      }
      if (as_stringizer)
        for (size_t n = 0; n < list.length; ++n) {
          value v = list[n];
          if (out++)
            standardOutput->put(' ');
          if ((n & 1) == 0) {
            standardOutput->put_str(value_to_string(v));
            continue;
          }
          CsPrintData(this, v, standardOutput,false);
        }
      else
        for (size_t n = 0; n < list.length; ++n) {
          value v = list[n];
          if (out++)
            standardOutput->put(' ');
          CsPrintData(this, v, standardOutput, false);
        }
      standardOutput->put('\n');
    }
    return true;
  }

  tool::ustring xvm::lang() {
    if (html::document *pd = current_doc())
      return pd->get_lang();
    else if(xview* pv = current_view())
      return pv->get_lang();
    return tool::ustring();
  }

  int xvm::pixels_per_inch() {
    xview *pv = current_view();
    if (pv) return pv->pixels_per_inch().y;
    return gool::resolution_provider::desktop().pixels_per_inch().y;
  }

  bool xvm::resolveNamedColor(value tuple, uint* pout) {
    tool::value tv = value_to_value(this,tuple);
    if (!tv.is_color())
      return false;
    if (html::element::drawing_element) {
      gool::color_v cv = tv;
      *pout = cv.to_color();
      return true;
    }
    return false;
  }

  tool::value xvm::x_value_to_value(tis::value v) {
    tool::value rv;
    if (CsIsType(v, this->pathDispatch)) {
      auto ptr = path_ptr(v);
      if (ptr) rv.set_resource(ptr, tool::value::UTR_PATH);
    } else if (CsIsType(v, this->imageDispatch)) {
      auto ptr = image_ptr(this, v);
      if (ptr) rv.set_resource(ptr, tool::value::UTR_IMAGE);
    } else if (CsIsType(v, this->graphicsDispatch)) {
      auto ptr = graphics_ptr(v);
      if (ptr && ptr->gfx) rv.set_resource(ptr->gfx.ptr(), tool::value::UTR_GRAPHICS);
    } else if (CsIsType(v, this->textDispatch)) {
      auto ptr = text_ptr(v);
      if (ptr && ptr->tl) rv.set_resource(ptr->tl.ptr(), tool::value::UTR_TEXT);
    } else if (CsIsType(v, this->elementDispatch)) {
      auto ptr = element_ptr(this,v);
      if (ptr) rv.set_resource(ptr, tool::value::UTR_DOM_ELEMENT);
    } else if (CsIsType(v, this->nodeDispatch)) {
      auto ptr = node_ptr(this, v);
      if (ptr) rv.set_resource(ptr, tool::value::UTR_DOM_NODE);
    }

    return rv;
  }
  tis::value xvm::x_value_to_value(tool::value v) {
    if (!v.is_resource()) return NULL_VALUE;
    {
      // gool::graphics *pgfx = v.get_resource<gool::graphics>();
      // handle<gfx_ctl> graphics_ctl = new gfx_ctl();
      // graphics_ctl->gfx = pgfx;
      // return CsMakeCPtrObject(c,c->graphicsDispatch,&graphics_ctl);
    }
    if (gool::image *pimg = v.get_resource<gool::image>()) {
      return image_object(this, pimg);
    }
    else if (gool::path *ppath = v.get_resource<gool::path>()) {
      return path_object(this, ppath);
    }
    else if (html::element* pel = v.get_resource<html::element>())
    {
      return element_object(this, pel);
    }
    
#pragma TODO("do others from the above please")
    /*gool::text_layout *ptl = v.get_resource<gool::text_layout>();
    if( ptl ) {
      text_ctl* xpath = new text_ctl();
      return path_object(this,xpath);
    }*/
    return NULL_VALUE;
  }

  void destroy_view(xvm *c, value obj) { CsSetCObjectValue(obj, 0); }

  struct auto_pevt {
    handle<xview> _self;
    html::event * _pevt;
    auto_pevt(xview *self, html::event *evt) : _self(self), _pevt(self->pevt) {
      _self->pevt = evt;
    }
    ~auto_pevt() { _self->pevt = _pevt; }
  };

  value view_sym() {
    static value vs = CsSymbolOf("view");
    return vs;
  }

  xview *xview_ptr(xvm *c, value obj) {
    if (CsGetDispatch(obj) != c->viewDispatch) {
      // assert(false);
      return 0;
    }
    xview *v = static_cast<xview *>(CsCObjectValue(obj));
    if (!v || v->vm != c) return 0;
    // assert (uint_ptr(v) > 0x100 );
    return v;
  }

  xview *xview_ptr(xvm *c) {
    return static_cast<xview *>(html::view::get_current());
  }

  /*value get_ns(html::element* b)
  {
    html::document* pd = b->doc();
    return pd? pd->ns : NULL_VALUE;
  }*/

  value get_ns(html::element *b) {
    html::document *pd = b->doc();
    if (pd) {
      if (pd->ns) return pd->ns;
      if (pd->parent) return get_ns(pd->parent);
    }
    return NULL_VALUE;
  }

  value CSF_all(xvm *c, value obj)
  {
    auto list = xview::get_all();
    value vlist = CsMakeVector(c, list.size());
    for (int n = 0; n < list.size(); ++n)
      CsSetVectorElementI(vlist, n, list[n].ptr_of<xview>()->view_obj);
    return vlist;
  }

  static value CSF_root(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return NULL_VALUE;
    return element_object(c, xv->doc());
    // return xv->root_obj;
  }

  static value CSF_owner(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return NULL_VALUE;

    if (xv->parent()) {
      xview *xpv = (xview *)xv->parent();
      return xpv->view_obj;
    }

    return NULL_VALUE;
  }

  static value CSF_load(xvm *c) {
    value obj;
    // wchar *path = 0;
    // int_t  pathlen = 0;
    bool now = false;

    value p1 = 0;
    CsParseArguments(c, "V=*V|B", &obj, c->viewDispatch, &p1, &now);

    xview *pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    if (CsStringP(p1)) {
      tool::ustring uin(CsStringAddress(p1), CsStringSize(p1));
      tool::string  in  = url::escape(uin);
      string        out = html::combine_url(pv->doc()->uri().src, in);
      return pv->load_url(out, now) ? TRUE_VALUE : FALSE_VALUE;
    } else if (CsFileP(c, p1) && CsFileStream(p1)->is_string_stream()) {
      string_stream *ss = (string_stream *)CsFileStream(p1);
      // if( CsStringP(p2) )
      //{
      //  tool::string path(CsStringAddress(p2) , CsStringSize(p2));
      //  pv->file_load((const char*)ss->buf.head(),ss->buf.size(),path)?
      //  TRUE_VALUE: FALSE_VALUE;
      //}
      // else
      tool::string          path = html::combine_url(pv->doc()->uri().src,
                                            tool::string(ss->stream_name()));
      handle<html::request> hrq  = new html::request(path, html::DATA_HTML);
      hrq->rq_type               = html::RQ_PUSH;
      hrq->data                  = ss->buf();
      bool r                     = pv->load(hrq);
      pv->on_data_loaded(hrq);
      return r ? TRUE_VALUE : FALSE_VALUE;
    } else {
      CsThrowKnownError(c, CsErrUnexpectedTypeError, p1,
                        "source is neither url:string nor in-memory stream");
    }
    return UNDEFINED_VALUE;
  }

  static value CSF_close(xvm *c) {
    value obj;
    value retval = UNDEFINED_VALUE;
    CsParseArguments(c, "V=*|V", &obj, c->viewDispatch, &retval);

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    html::VIEW_TYPE vt = pv->view_type();
    switch (vt) {
    case html::VIEW_WINDOW:
    case html::VIEW_DIALOG:
      // THROW
      pv->dialog_retval = retval;
      pv->ask_close_window();
      break;
    }
    return UNDEFINED_VALUE;
  }

#if defined(XSCITER)

  static value CSF_callback(xvm *c) {
    value obj;
    value p1 = UNDEFINED_VALUE, p2 = UNDEFINED_VALUE;
    int_t ch;
    CsParseArguments(c, "V=*i|V|V", &obj, c->viewDispatch, &ch, &p1, &p2);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    tool::value jp1, jp2, jr;
    jp1 = value_to_value(c, p1);
    jp2 = value_to_value(c, p2);

    // if(
    // ::SendMessage((HWND)pv->doc()->hctl,WM_XSCITER_CALLBACK,ch,LPARAM(pars))
    // == 0xAF )
    if (pv->host_callback(ch, jp1, jp2, jr)) { return value_to_value(c, jr); }
    return UNDEFINED_VALUE;
  }

#endif

  static value CSF_selectFile(xvm *c) {
    if ((c->features & FEATURE_SYSINFO) == 0) return UNDEFINED_VALUE;
    value obj;
    //#open/#save, filter, ext
    const wchar *filter =
        //W("HTML Files (*.htm,*.html)|*.HTM;*.HTML|All Files (*.*)|*.*");
        W("All Files (*.*)|*.*");
    const wchar *ext = W("html");
    wchars       initialPath;
    value        purpose;
    wchars       caption;

    CsParseArguments(c, "V=*V|S|S|S#|S#", &obj, c->viewDispatch, &purpose,
                     &filter, &ext, &initialPath.start, &initialPath.length,
                     &caption.start, &caption.length);

    xview *pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    ustring sfilter(filter);
    ustring scaption(caption);

    if (sfilter.length() == 0) return UNDEFINED_VALUE;
    if (sfilter[sfilter.size() - 1] != 0) sfilter += '|';

    ustring sfilename = url::file_url_to_path(initialPath);
    ustring sext = ext;

    xview::AFN_MODE mode = xview::AFN_OPEN;
    if (purpose == CsSymbolOf(CHARS("open-multiple")))
      mode = xview::AFN_OPEN_MULTIPLE;
    else if (purpose == CsSymbolOf(CHARS("save")))
      mode = xview::AFN_SAVE;

    array<ustring> filenames =
        pv->ask_file_name(mode, scaption, sfilename, sext, sfilter);

    if (filenames.size() == 1) {
       return CsMakeCString(c, url::path_to_file_url(filenames[0]));
    }
    if (filenames.size() > 1) {
      value arr = CsMakeVector(c, filenames.size());
      PROTECT(arr);
      for (int n = 0; n < filenames.size(); ++n) {
        value fn = CsMakeCString(c, url::path_to_file_url(filenames[n]));
        CsSetVectorElement(c, arr, n, fn);
      }
      return arr;
    }

    return UNDEFINED_VALUE;
  }

  static value CSF_selectFolder(xvm *c) {
    if ((c->features & FEATURE_SYSINFO) == 0) return UNDEFINED_VALUE;

    value  obj;
    wchars caption = WCHARS("Select folder...");
    wchars path;
    CsParseArguments(c, "V=*|S#|S#", &obj, c->viewDispatch, &caption.start,
                     &caption.length, &path.start, &path.length);

    xview *pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    ustring name = url::file_url_to_path(path);

    if (pv->ask_folder_name(caption, name)) { return CsMakeCString(c, url::path_to_file_url(name)); }
    //#endif
    return UNDEFINED_VALUE;
  }

  static value CSF_selectPrinter(xvm *c) {
#if 0
    value obj;
    const wchar* caption = W("Select printer...");
    CsParseArguments(c,"V=*|S", &obj,c->viewDispatch,&caption);

    xview* pv = xview_ptr(c,obj);
    if(!pv)
      return UNDEFINED_VALUE;

    PRINTDLGW pd; memzero(pd);
    pd.lStructSize = sizeof(pd);
    pd.hwndOwner   = pv->get_hwnd();
    pd.nFromPage = pd.nMinPage = 1;
    pd.nToPage = pd.nMaxPage = 1;//max(pd.nFromPage,total_pages.val(100));
    pd.hInstance = HINST_THISCOMPONENT;

    //if(on_default)
    //  pd.Flags = PD_RETURNDEFAULT | PD_RETURNDC;
    //else
    //{
    pd.Flags = PD_RETURNDC | PD_USEDEVMODECOPIESANDCOLLATE; //  | PD_NOSELECTION | PD_USEDEVMODECOPIESANDCOLLATE
    //}
    if( !::PrintDlgW(&pd) )
    {
      DWORD dwError = CommDlgExtendedError();
      dwError = dwError;
      return FALSE_VALUE;
    } else {
/*      surface sf(pd.hDC);
      size dim(::GetDeviceCaps(sf, PHYSICALWIDTH),
               ::GetDeviceCaps(sf, PHYSICALHEIGHT));
      rect rc(
       -::GetDeviceCaps(sf,PHYSICALOFFSETX),
       -::GetDeviceCaps(sf,PHYSICALOFFSETY),
        ::GetDeviceCaps(sf, PHYSICALWIDTH) - ::GetDeviceCaps(sf,PHYSICALOFFSETX),
        ::GetDeviceCaps(sf, PHYSICALHEIGHT) - ::GetDeviceCaps(sf,PHYSICALOFFSETY));
      print_content(v,self,sf,rc,dim,pd.nFromPage,pd.nToPage); */
    }



    DEVMODE* pdm = (DEVMODE*) ::GlobalLock(pd.hDevMode);
    pdm = pdm;
    ::GlobalUnlock(pd.hDevMode);

    DOCINFO di;
    di.cbSize      = sizeof(DOCINFO);
    di.lpszDocName = L"Sciter doc";
    di.lpszOutput  = nullptr;
    StartDoc  (pd.hDC, &di);
    StartPage (pd.hDC);

    rect rc(
       -::GetDeviceCaps(pd.hDC,PHYSICALOFFSETX),
       -::GetDeviceCaps(pd.hDC,PHYSICALOFFSETY),
        ::GetDeviceCaps(pd.hDC, PHYSICALWIDTH) - ::GetDeviceCaps(pd.hDC,PHYSICALOFFSETX),
        ::GetDeviceCaps(pd.hDC, PHYSICALHEIGHT) - ::GetDeviceCaps(pd.hDC,PHYSICALOFFSETY));
    //rect rc(0,0,1000,1000);


    //HBITMAP hbmp = CreateCompatibleBitmap(pd.hDC,rc.width(), rc.height());

    //HDC dc = ::CreateCompatibleDC(nullptr);

    //HBITMAP holdbmp = (HBITMAP)::SelectObject(dc,hbmp);

    //handle<dib32> bitmap = new dib32(size(1000,1000));

    //d2d::dc_graphics gfx(pd.hDC,rc);
    {

      ::SetMapMode(pd.hDC,MM_TEXT);

      {
        handle<xprintview> xpv = xprintview::create(pv);
        handle<gdi::graphics> gfx = new gdi::graphics(pd.hDC);

        size ppi;
        ppi.x = GetDeviceCaps(pd.hDC, LOGPIXELSX);
        ppi.y = GetDeviceCaps(pd.hDC, LOGPIXELSY);
        xpv->pixels_per_inch(ppi);

        //size sz = xpv->pixels_per_dip(size(1,1));

        /*int nLogPx = ::GetDeviceCaps(pd.hDC, LOGPIXELSX);
         // Get GDI+ resolution
        int nGdiPlusLogPx = (int) gfx->target()->GetDpiX();
         // set to pixels
         gfx->target()->SetPageUnit(Gdiplus::UnitPixel);
         // Adjust to match
         gfx->target()->SetPageScale(((float) nGdiPlusLogPx / (float) nLogPx)); */


        //gfx->scale(sizef(1/6.0f,1/6.0f));
        xpv->set_graphics(gfx);
        xpv->load_url("file://d:/print-test.htm");

        size wa = xpv->pixels_per_dip( size(800,1000) );

        xpv->on_size( rc.size() );
        gfx->set_clip_rc( rc );
        xpv->paint();
        xpv->stop();
      }

      //gfx->scale(gool::sizef(3,3));
      //mswin::window* pw = (mswin::window*)pv;
      //pw->render_on(gfx,rc);

      //gfx->render_target()->BeginDraw();
      //gfx->fill(argb(255,0,0), rect(100,100,200,200));
      //HRESULT result = gfx->render_target()->EndDraw();
      //result = result;
    }

    //::BitBlt(pd.hDC,0,0,rc.width(),rc.height()/2,dc,0,0,SRCCOPY);
    //::BitBlt(pd.hDC,0,rc.height()/2,rc.width(),rc.height(),dc,0,rc.height()/2,SRCCOPY);

    //::SelectObject(dc,holdbmp);
    //::DeleteDC(dc);
    //::DeleteObject(hbmp);

    EndPage   (pd.hDC);
    EndDoc    (pd.hDC);

#if defined(PLATFORM_WINCE)
    return UNDEFINED_VALUE;
#endif

#endif
    return UNDEFINED_VALUE;
  }

  static value CSF_box(xvm *c) {
    value obj;

    symbol_t side;
    symbol_t box   = uint(-1);
    symbol_t relTo = uint(-1);

    CsParseArguments(c, "V=*L|L|L", &obj, c->viewDispatch, &side, &box, &relTo);

    side  = get_sym_id(side, id_left);
    box   = get_sym_id(box, id_client);
    relTo = get_sym_id(relTo, id_self);

    xview *pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    rect rc(pv->client_dim());
    switch (box) {
    case id_client:
      if (relTo == id_screen) 
        rc += pv->client_screen_pos();
      break;
    case id_border:
      rc = rect(pv->window_dim());
      if (relTo == id_screen)
        rc += pv->screen_pos();
      else {
        rc += pv->screen_pos();
        rc -= pv->client_screen_pos();
      }
      break;
    }

    switch (side) {
    case id_left: return CsMakeInteger(rc.left());
    case id_right: return CsMakeInteger(rc.right());
    case id_top: return CsMakeInteger(rc.top());
    case id_bottom: return CsMakeInteger(rc.bottom());
    case id_width: return CsMakeInteger(rc.width());
    case id_height: return CsMakeInteger(rc.height());
    case id_rect:
      CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()),
                 CsMakeInteger(rc.right()), CsMakeInteger(rc.bottom()));
    case id_rectw:
      CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()),
                 CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    case id_position:
      //CsSetRVal(c, 1, CsMakeInteger(rc.left()));
      //return CsMakeInteger(rc.top());
      CS_RETURN2(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()));

    case id_dimension:
      //CsSetRVal(c, 1, CsMakeInteger(rc.width()));
      //return CsMakeInteger(rc.height());
      CS_RETURN2(c, CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    }

    return UNDEFINED_VALUE;
  }

  static value CSF_move(xvm *c) {
    value       obj;
    gool::point pos;
    gool::size  sz;
    bool        client_cc = false;
    bool        and_size  = false;
    switch (CsArgCnt(c)) {
    case 4:
    case 5:
      CsParseArguments(c, "V=*ii|B", &obj, c->viewDispatch, &pos.x, &pos.y,
                       &client_cc);
      break;
    case 6:
    case 7:
      CsParseArguments(c, "V=*iiii|B", &obj, c->viewDispatch, &pos.x, &pos.y,
                       &sz.x, &sz.y, &client_cc);
      and_size = true;
      break;
    default:
      CsParseArguments(c, "V=*ii", &obj, c->viewDispatch, &pos.x, &pos.y);
      break;
    }

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return UNDEFINED_VALUE;

    html::VIEW_TYPE vt = pv->view_type();
    switch (vt) {
    case html::VIEW_WINDOW:
    case html::VIEW_DIALOG: {
      if (!and_size) {
        if (client_cc)
          sz = pv->dimension();
        else
          sz = pv->window_dim();
      }
      rect rc(pos, sz);
      //rect wd = pv->get_frame_type() == html::STANDARD ? pv->window_decoration() : rect(0,0,0,0);
      //if (client_cc) rc >>= wd;
      pv->move_window(rc, client_cc);
      return TRUE_VALUE;
    }
    default:
      CsThrowKnownError(c, CsErrGenericError,
                        "view.move() cannot be applied to this window type");
      break;
    }
    return UNDEFINED_VALUE;
  }

  static value CSF_subscribe(xvm *c) {
    value  obj;
    value  fun;
    wchars namens;
    CsParseArguments(c, "V=*S#m", &obj, c->viewDispatch, &namens.start, &namens.length, &fun);

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return obj;

    return CsEventObjectAdd(c, obj, fun, namens);
  }

  static value CSF_unsubscribe(xvm *c) {
    value obj;
    value what;
    CsParseArguments(c, "V=*V", &obj, c->viewDispatch, &what);

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return obj;

    return CsEventObjectRemoveV(c, obj, what);
  }

  static value CSF_state(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return NULL_VALUE;
    html::WINDOW_STATE ws = xv->get_window_state();
    return int_value(ws);
  }

  static void CSF_set_state(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (CsIntegerP(val)) {
      xv->set_window_state((html::WINDOW_STATE)to_int(val));
      xv->check_mouse();
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, val, "integer");
  }

  /*static value CSF_frameless(xvm *c,value obj)
  {
    xview* xv = xview_ptr(c, obj);
    return xv->get_frameless_ctl()? FALSE_VALUE: TRUE_VALUE;
  }

  static void CSF_set_frameless(xvm *c,value obj, value val)
  {
    xview* xv = xview_ptr(c, obj);
    if( CsTrueP(val) )
    {
      html::element * d = xv->doc();
      if(d && d->parent == 0)
      {
        handle<html::window_controls> wctls = new html::window_controls;
        wctls->button_close = html::find_first(*xv,d,WCHARS("#window-close"));
        wctls->button_icon = html::find_first(*xv,d,WCHARS("#window-icon"));
        wctls->button_min = html::find_first(*xv,d,WCHARS("#window-minimize"));
        wctls->button_max = html::find_first(*xv,d,WCHARS("#window-maximize"));
        wctls->caption = html::find_first(*xv,d,WCHARS("#window-caption"));
        wctls->corner = html::find_first(*xv,d,WCHARS("#window-corner"));
        xv->set_frameless_ctl(wctls);
      }
    }
    else
    {
      xv->set_frameless_ctl(0);
    }

  }*/

  static value CSF_caption(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      ustring us = xv->get_window_title();
      return CsMakeString(c, us);
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_caption(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (xv && CsStringP(val)) { xv->set_window_title(CsStringAddress(val)); }
  }

  static value CSF_windowFrame(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      switch (xv->get_frame_type()) {
      case html::STANDARD: return CsSymbolOf(WCHARS("standard"));
      case html::LAYERED:  return CsSymbolOf(WCHARS("transparent"));
      case html::SOLID:    return CsSymbolOf(WCHARS("solid"));
      case html::SOLID_WITH_SHADOW: return CsSymbolOf(WCHARS("solid-with-shadow"));
      case html::STANDARD_EXTENDED: return CsSymbolOf(WCHARS("extended"));
      }
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_windowFrame(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
           if (val == CsSymbolOf(WCHARS("standard"))) xv->set_frame_type(html::STANDARD);
      else if (val == CsSymbolOf(WCHARS("transparent"))) xv->set_frame_type(html::LAYERED);
      else if (val == CsSymbolOf(WCHARS("solid"))) xv->set_frame_type(html::SOLID);
      else if (val == CsSymbolOf(WCHARS("solid-with-shadow"))) xv->set_frame_type(html::SOLID_WITH_SHADOW);
      else if (val == CsSymbolOf(WCHARS("extended"))) xv->set_frame_type(html::STANDARD_EXTENDED);
    }
  }
  
  static value CSF_eventsRoot(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      html::helement b = xv->get_event_capture();
      if (b) return element_object(c, b);
    }
    return NULL_VALUE;
  }

  static void CSF_set_eventsRoot(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    html::helement b;
    if (val != UNDEFINED_VALUE && val != NULL_VALUE) b = element_ptr(c, val);
    xv->set_event_capture(b);
  }

  static value CSF_parameters(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv && xv->param.is_set())
      return xv->param.val;
    return NULL_VALUE;
  }

  static void CSF_set_parameters(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (xv)
      xv->param.pin(c, val);
  }

  static value CSF_event(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv && xv->event_obj.is_set())
      return xv->event_obj.val;
    return NULL_VALUE;
  }


  static value CSF_focus(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      html::element *b = xv->get_focus_element();
      if (b) return element_object(c, b);
    }
    return NULL_VALUE;
  }

  static void CSF_set_focus(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    html::helement b;
    if (val != UNDEFINED_VALUE && val != NULL_VALUE) b = element_ptr(c, val);
    xv->set_focus(b, html::BY_CODE);
  }

  static value CSF_focusable(xvm *c) {
    value  obj;
    value  op;
    value  element = 0;
    CsParseArguments(c, "V=*V=|V=", &obj, c->viewDispatch, &op, &CsSymbolDispatch, &element, c->elementDispatch);

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return obj;

    html::view::FOCUS_CMD_TYPE fcmd;
    html::element *pcurr = nullptr;

    if (op == CsSymbolOf("next")) fcmd = html::view::FOCUS_NEXT;
    else if (op == CsSymbolOf("prior")) fcmd = html::view::FOCUS_PREV;
    else if (op == CsSymbolOf("first")) fcmd = html::view::FOCUS_FIRST;
    else if (op == CsSymbolOf("last")) fcmd = html::view::FOCUS_LAST;
    else CsThrowKnownError(c, CsErrValueError, op);

    if (element)
      pcurr = element_ptr(c, element);
    bool end_reached;
    pcurr = pv->get_next_focus_element(fcmd, pcurr, end_reached);
    if (pcurr)
      return element_object(c, pcurr);
    return NULL_VALUE;
  }

  static value CSF_activate(xvm *c) {
    value  obj;
    value  op = 0;
    CsParseArguments(c, "V=*|V=", &obj, c->viewDispatch, &op, &CsSymbolDispatch);

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) return obj;

    pv->activate(op == CsSymbolOf("toFront"));

    return TRUE_VALUE;
  }

  static value CSF_aspectRatio(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      float_v ratio = xv->aspect_ratio();
      if (ratio.is_defined())
        return CsMakeFloat(ratio.val(0.0f));
      else
        return NULL_VALUE;
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_aspectRatio(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    float_v ratio;
    if (CsFloatP(val))
      ratio = (float)CsFloatValue(val);
    else if (CsIntegerP(val))
      ratio = (float)CsIntegerValue(val);
    xv->aspect_ratio(ratio);
  }

  static value CSF_isTopmost(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeBoolean(xv->get_topmost()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_isTopmost(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == TRUE_VALUE)
      xv->set_topmost(true);
    else if (val == FALSE_VALUE)
      xv->set_topmost(false);
    else
      CsUnexpectedTypeError(c, val, "boolean");
  }

  static value CSF_isEnabled(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeBoolean(xv->is_enabled()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_isEnabled(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == TRUE_VALUE)
      xv->set_enabled(true);
    else if (val == FALSE_VALUE)
      xv->set_enabled(false);
    else
      CsUnexpectedTypeError(c, val, "boolean");
  }


  static value CSF_minSize(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      gool::size sz;
      if (xv->get_min_size(sz)) CS_RETURN2(c, sz.x, sz.y);
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_minSize(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if(CsValueListP(c->val) && CsIntegerP(CsValueListElement(c->val, 0)) && CsIntegerP(CsValueListElement(c->val, 1)))
      xv->set_min_size(gool::size(CsIntegerValue(CsValueListElement(c->val,0)), CsIntegerValue(CsValueListElement(c->val, 1))));
    else if (CsIntegerP(val))
      xv->set_min_size(gool::size(CsIntegerValue(val), CsIntegerValue(val)));
    else
      CsUnexpectedTypeError(c, val, "integer");
  }

  static value CSF_maxSize(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      gool::size sz;
      if (xv->get_max_size(sz)) CS_RETURN2(c, sz.x, sz.y);
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_maxSize(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (CsValueListP(c->val))
      xv->set_max_size(gool::size(CsIntegerValue(CsValueListElement(c->val, 0)), CsIntegerValue(CsValueListElement(c->val, 1))));
    else if (CsIntegerP(val))
      xv->set_max_size(gool::size(CsIntegerValue(val), CsIntegerValue(val)));
    else
      CsUnexpectedTypeError(c, val, "integer");
  }

  static value CSF_animationsEnabled(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return xv->animations_disabled ? FALSE_VALUE : TRUE_VALUE; }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_animationsEnabled(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    xv->animations_disabled = CsToBoolean(c, val) == FALSE_VALUE;
  }

  static value CSF_icon(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv && xv->get_icon()) {
      handle<gool::image> pimg  = xv->get_icon();
      return image_object(c, pimg);
    }
    return NULL_VALUE;
  }

  static void CSF_set_icon(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == NULL_VALUE || val == UNDEFINED_VALUE)
      xv->set_icon(nullptr);
    else if (CsStringP(val))
      xv->set_icon_url(tool::string(CsStringChars(val)));
    else {
      gool::image *pimg = image_ptr(c, val);
      if (pimg)
        xv->set_icon(pimg);
      else
        CsUnexpectedTypeError(c, val, "image");
    }
  }

  static value CSF_isResizeable(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeBoolean(xv->get_resizeable()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_isResizeable(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == TRUE_VALUE)
      xv->set_resizeable(true);
    else if (val == FALSE_VALUE)
      xv->set_resizeable(false);
    else
      CsUnexpectedTypeError(c, val, "boolean");
  }

  static value CSF_resizePolicy(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeInteger(xv->get_resize_policy()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_resizePolicy(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (CsIntegerP(val))
      xv->set_resize_policy(CsIntegerValue(val));
    else
      CsUnexpectedTypeError(c, val, "integer");
  }



  static value CSF_isMaximizable(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeBoolean(xv->get_maximizable()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_isMaximizable(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == TRUE_VALUE)
      xv->set_maximizable(true);
    else if (val == FALSE_VALUE)
      xv->set_maximizable(false);
    else
      CsUnexpectedTypeError(c, val, "boolean");
  }

  static value CSF_isMinimizable(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) { return CsMakeBoolean(xv->get_minimizable()); }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_isMinimizable(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    if (val == TRUE_VALUE)
      xv->set_minimizable(true);
    else if (val == FALSE_VALUE)
      xv->set_minimizable(false);
    else
      CsUnexpectedTypeError(c, val, "boolean");
  }

  static value CSF_blurBehind(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (xv) {
      html::BLUR_BEHIND bb = xv->get_blurbehind();
      if (bb == html::BLUR_BEHIND_NONE) return UNDEFINED_VALUE;
      enum_v  vbb = xv->get_blurbehind();
      ustring us  = enum_to_string(vbb, html::blur_edef);
      return CsMakeSymbol(c, us());
    }
    return NULL_VALUE;
  }

  static void CSF_set_blurBehind(xvm *c, value obj, value val) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return;
    enum_v bb;

    if (!CsSymbolP(val)) CsUnexpectedTypeError(c, val, "symbol");

    ustring us = CsSymbolName(val);
    html::parse_enum(bb, us(), html::blur_edef);

    xv->set_blurbehind((html::BLUR_BEHIND)bb.val(0));
  }

  static value CSF_isActive(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return UNDEFINED_VALUE;
    return (xv->doc() && xv->doc()->state.owns_focus()) ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_backend(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return UNDEFINED_VALUE;
    int gfx = xv->graphics_caps();
    return CsMakeInteger(gfx);
  }

  static value CSF_backendName(xvm *c, value obj) {
    xview *xv = xview_ptr(c, obj);
    if (!xv) return UNDEFINED_VALUE;

    ustring us = xv->graphics_backend();
    return CsSymbolOf(us());
  }

  static value CSF_doEvent(xvm *c) {
    value obj;

    symbol_t t = limits<symbol_t>::max_value();

    CsParseArguments(c, "V=*|L", &obj, c->viewDispatch, &t);
    t = get_sym_id(t, id_wait);

    html::DO_EVENT_MANNER m = html::DO_EVENT_WAIT;
    if (t == id_wait)
      m = html::DO_EVENT_WAIT;
    else if (t == id_nowait)
      m = html::DO_EVENT_NOWAIT;
    else if (t == id_all)
      m = html::DO_EVENT_ALL;
    else if (t == id_untilMouseUp)
      m = html::DO_EVENT_UNTIL_MOUSE_UP;
    else if (t == id_onlyIO)
      m = html::DO_EVENT_ONLY_IO;

    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) {
      CsThrowKnownError(c, CsErrGenericError, "view is already closed");
    }
    bool result = true;
    if (!pv->do_event(m, result)) { throw script_exception(0); }
    return result ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_update(xvm *c) {
    value obj;
    bool  and_draw = false;
    CsParseArguments(c, "V=*|B", &obj, c->viewDispatch,&and_draw);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv)
      CsThrowKnownError(c, CsErrGenericError, "view is already closed");
    else {
      pv->refresh();
      if(and_draw)
        pv->update();
      else
        pv->commit_update();
    }

    return UNDEFINED_VALUE;
  }

  static value CSF_display(xvm *c) {
    value obj;
    uint  milliseconds = 200;
    value op           = UNDEFINED_VALUE;
    value mode         = UNDEFINED_VALUE;
    value direction    = UNDEFINED_VALUE;
    CsParseArguments(c, "V=*|V|V|V|T", &obj, c->viewDispatch, &op, &mode,
                     &direction, &milliseconds);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv)
      CsThrowKnownError(c, CsErrGenericError, "view is already closed");
    else {

      static value show = CsSymbolOf(WCHARS("show"));
      static value hide = CsSymbolOf(WCHARS("hide"));

      static value mode_blend   = CsSymbolOf(WCHARS("blend"));
      static value mode_roll    = CsSymbolOf(WCHARS("roll"));
      static value mode_slide   = CsSymbolOf(WCHARS("slide"));
      static value mode_instant = CsSymbolOf(WCHARS("instant"));

      if (mode == UNDEFINED_VALUE || mode == mode_instant) {
        pv->set_window_state(op == show ? gool::WINDOW_SHOWN
                                        : gool::WINDOW_HIDDEN);
        return TRUE_VALUE;
      }
      uint animation_mode = op == show ? 0 : html::iwindow::AWF_HIDE;
      if (mode == mode_roll)
        animation_mode |= html::iwindow::AWF_ROLL;
      else if (mode == mode_slide)
        animation_mode |= html::iwindow::AWF_SLIDE;
      else if (mode == mode_blend)
        animation_mode |= html::iwindow::AWF_BLEND;

      static value dir_from_right  = CsSymbolOf(WCHARS("from-right"));
      static value dir_from_top    = CsSymbolOf(WCHARS("from-top"));
      static value dir_from_bottom = CsSymbolOf(WCHARS("from-bottom"));
      static value dir_from_left   = CsSymbolOf(WCHARS("from-left"));
      if (direction == dir_from_left)
        animation_mode |= html::iwindow::AWF_HOR_POSITIVE;
      else if (direction == dir_from_right)
        animation_mode |= html::iwindow::AWF_HOR_NEGATIVE;
      else if (direction == dir_from_top)
        animation_mode |= html::iwindow::AWF_VER_POSITIVE;
      else if (direction == dir_from_bottom)
        animation_mode |= html::iwindow::AWF_VER_NEGATIVE;

      pv->show(animation_mode, milliseconds);
      return TRUE_VALUE;
    }

    return UNDEFINED_VALUE;
  }

  handle<html::clipboard::data> object2cbdata(xvm *c, xview *pv, value vitems) {
    handle<html::clipboard::data> pdata = new html::clipboard::data();

    each_property gen(c, vitems);

    for (value key, val; gen(key, val);) {
      wchars name = value_to_wchars(key);
      if (name == WCHARS("text"))
        pdata->add(new html::clipboard::text_item(value_to_string(val)));
      else if (name == WCHARS("html")) {
        if (CsInstanceOf(c, val, c->elementDispatch->obj)) {
          html::element *pel = element_ptr(c, val);
          if (pel) {
            array<char> html;
            html::clipboard::html_cf(*pv, pel->this_pos(false),
                                     pel->this_pos(true), html);
            pdata->add(html::clipboard::html_item::from_cf_html(html()));
          }
        } else {
          ustring         whtml_fragment = value_to_string(val);
          string          html_fragment  = u8::cvt(whtml_fragment);
          html::document *pd             = c->current_doc();
          if (!pd) pd = pv->doc();
          array<char> html;
          html::clipboard::html_cf(html_fragment(), pd->uri().src(), html);
          pdata->add(html::clipboard::html_item::from_cf_html(html()));
        }
      } else if (name == WCHARS("link")) {
        if (CsObjectP(val)) {
          tool::ustring caption;
          tool::ustring url;
          CsGetProperty(c, val, "caption", caption);
          CsGetProperty(c, val, "url", url);
          pdata->add(new html::clipboard::link_item(caption, url));
        } else if (CsStringP(val)) {
          tool::ustring caption;
          tool::ustring url = value_to_string(val);
          pdata->add(new html::clipboard::link_item(caption, url));
        }
      }
      /*else if( name == WCHARS("image") ) {
      handle<tis::gool::image> pimg = image_ptr(c,CsTupleElement(item,0));
      if( pimg && pimg->img )
      pdata->add( new html::clipboard::image_item(pimg->img) );
      }*/
      else if (name == WCHARS("file")) {
        array<ustring> files;
        if (CsVectorP(val)) {
          tool::slice<value> vals = CsVectorElements(c, val);
          for (uint i = 0; i < vals.length; ++i)
            files.push(value_to_string(vals[i]));
        } else if (CsStringP(val)) {
          files.push(value_to_string(val));
        }
        if (files.size()) pdata->add(new html::clipboard::file_item(files));
      } else if (name == WCHARS("json")) {
        string_stream s;
        s.set_encoder(stream::null_encoder());
        if (CsPrintJsonData(c, val, &s, false)) {
          array<byte> data;
          s.get_content(data);
          pdata->add(new html::clipboard::json_item(data()));
        }
      }
    }
    return pdata;
  }

  static value CSF_performDrag(xvm *c) {
    value obj = 0, vel = 0, vitems = 0, vmode = CsSymbolOf(WCHARS("any"));
    value vimg = 0; 
    point off;

    if (CsParseArguments(c, "|V=*V=V=|V=", &obj, c->viewDispatch, &vel, c->elementDispatch, &vitems, &CsObjectDispatch, &vmode, &CsSymbolDispatch)) ;
    else CsParseArguments(c, "V=*V=iiV=|V=", &obj, c->viewDispatch, &vimg, c->imageDispatch,&off.x,&off.y,&vitems, &CsObjectDispatch, &vmode, &CsSymbolDispatch);

    handle<xview>         pv  = xview_ptr(c, obj);

    handle<html::clipboard::data> pdata = object2cbdata(c, pv, vitems);

    uint dd_mode = html::dd_copy;
    if (vmode == CsSymbolOf(WCHARS("any")))
      dd_mode = html::dd_copy_or_move;
    else if (vmode == CsSymbolOf(WCHARS("copy")))
      dd_mode = html::dd_copy;
    else if (vmode == CsSymbolOf(WCHARS("move")))
      dd_mode = html::dd_move;

    if (vel) {
      handle<html::element> pel = element_ptr(c, vel);
      if(!pv->drag_element(pel, pdata, dd_mode))
        return NULL_VALUE;
    }
    else if(vimg) {
      handle<gool::image> pimg = image_ptr(c, vimg);
      hbitmap pbmp;
      if (pimg && pimg->is_bitmap())
        pbmp = pimg.ptr_of<gool::bitmap>();
        
      if (!pbmp || !pv->drag_data(pbmp, off, pdata, dd_mode))
        return NULL_VALUE;
    } else
      return NULL_VALUE;
    
    switch (dd_mode) {
      case html::dd_copy: return CsSymbolOf(WCHARS("copy"));
      case html::dd_move: return CsSymbolOf(WCHARS("move"));
    }
    return NULL_VALUE;
  }

  static value CSF_clipboardPut(xvm *c) {
    value obj, data = 0;
    CsParseArguments(c, "V=**V", &obj, c->viewDispatch, &data);
    handle<xview> pv = xview_ptr(c, obj);

    handle<html::clipboard::data> cbdata;

    if (CsStringP(data)) {
      html::clipboard::empty();
      html::clipboard::set_text(CsStringChars(data));
      return TRUE_VALUE;
    } else if (CsObjectP(data)) {
      html::clipboard::empty();
      cbdata = object2cbdata(c, pv, data);
      if (html::clipboard::set(cbdata))
        return TRUE_VALUE;
    } else {
      gool::image *xim = image_ptr(c, data);
      if (xim) {
        html::clipboard::empty();
        html::clipboard::set(xim);
        return TRUE_VALUE;
      }
    }
    CsUnexpectedTypeError(c, data, "string, object or Image");
    return UNDEFINED_VALUE;
  }

  static const value sym_text = CsSymbolOf("text");
  static const value sym_html = CsSymbolOf("html");
  static const value sym_picture = CsSymbolOf("picture");
  static const value sym_json = CsSymbolOf("json");
  static const value sym_url = CsSymbolOf("url");
  static const value sym_file = CsSymbolOf("file");

  value clipboard_format_sym(html::clipboard::clipboard_format cf) {
    switch (cf) {
      case html::clipboard::cf_text: return sym_text;
      case html::clipboard::cf_html: return sym_html;
      case html::clipboard::cf_picture: return sym_picture;
      case html::clipboard::cf_json: return sym_json;
      case html::clipboard::cf_hyperlink: return sym_url;
      case html::clipboard::cf_file: return sym_file;
      default: return UNDEFINED_VALUE;
    }
  }

  html::clipboard::clipboard_format clipboard_format_by_sym(value cf) {
    if(cf == sym_text) return html::clipboard::cf_text;
    if (cf == sym_html) return html::clipboard::cf_html;
    if (cf == sym_picture) return html::clipboard::cf_picture;
    if (cf == sym_json) return html::clipboard::cf_json;
    if (cf == sym_url) return html::clipboard::cf_hyperlink;
    if (cf == sym_file) return html::clipboard::cf_file;
    return html::clipboard::cf_undefined;
  }

  static value CSF_clipboardGet(xvm *c) {
    value p1;
    CsParseArguments(c, "***V", &p1);

    if (CsMethodP(p1)) {
      // cbcb _cbcb(c,p1);
      auto cb = [c, p1](html::clipboard::clipboard_format cf) -> bool {
        value r = tis::CsCallFunction(CsCurrentScope(c), p1, 1,
                                      clipboard_format_sym(cf));
        return r != FALSE_VALUE;
      };
      html::clipboard::available_formats(cb);
      return UNDEFINED_VALUE;
    } else if (CsSymbolP(p1)) {

      handle<html::clipboard::data> data = html::clipboard::get(clipboard_format_by_sym(p1));

      if (p1 == clipboard_format_sym(html::clipboard::cf_text)) {
        auto pitem = data->get<html::clipboard::text_item>();
        if (pitem)
          return CsMakeString(c, pitem->val);

      } else if (p1 == clipboard_format_sym(html::clipboard::cf_html)) {

        auto pitem = data->get<html::clipboard::html_item>();
        if (pitem) {
          value url = 0;
          value html = 0;
          PROTECT(url, html);
          url = CsMakeString(c, pitem->url);
          html = CsMakeString(c, pitem->val);
          CS_RETURN2(c, url, html);
        }

      } else if (p1 == clipboard_format_sym(html::clipboard::cf_picture)) {

        auto pitem = data->get<html::clipboard::image_item>();
        if (pitem && pitem->image) {
          return image_object(c, pitem->image);
        }
      } else if (p1 == clipboard_format_sym(html::clipboard::cf_hyperlink)) {

        auto pitem = data->get<html::clipboard::link_item>();
        if (pitem) {
          value url = 0;
          value caption = 0;
          PROTECT(url, caption);
          url = CsMakeString(c, pitem->url);
          caption = CsMakeString(c, pitem->caption);
          CS_RETURN2(c, url, caption);
        }

      } else if (p1 == clipboard_format_sym(html::clipboard::cf_file)) {
        auto pitem = data->get<html::clipboard::file_item>();
        if (pitem) {
          array<tool::value> filenames;
          for (auto fn : pitem->filenames) filenames.push(tool::value(fn));
          return value_to_value(c, tool::value::make_array(filenames()));
        }
      } else if (p1 == clipboard_format_sym(html::clipboard::cf_json)) {
        auto pitem = data->get<html::clipboard::json_item>();
        if (pitem) {
          ustring     jsdata = u8::cvt(pitem->val());
          auto        chars = jsdata();
          tool::value v = xjson::parse(chars, false);
          return value_to_value(c, v);
        }
      }

      return NULL_VALUE;
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, p1, "function or symbol");
    return UNDEFINED_VALUE;
  }

#pragma warning(push)
#pragma warning(disable : 4715) // warning C4715: 'tis::CSF_clipboard' : not all
                                // control paths return a value

  // I cannot convince MS VC++ compiler to do not warn about 'not all control
  // paths return a value' Either 64-bit or 32-bit compiler will reject return
  // value here, :(
  static value CSF_clipboard(xvm *c) {
    value p1;
    CsParseArguments(c, "**V", &p1);
    if (CsMethodP(p1)) {
      // cbcb _cbcb(c,p1);
      // html::clipboard::available_formats(_cbcb);
      PROTECT(p1);
      auto cb = [c, &p1](html::clipboard::clipboard_format cf) -> bool {
        value r = tis::CsCallFunction(CsCurrentScope(c), p1, 1,
                                      clipboard_format_sym(cf));
        return r != FALSE_VALUE;
      };
      html::clipboard::available_formats(cb);
      return CsMakeInteger((int)html::clipboard::get_sequence_num());
    } else if (p1 == CsSymbolOf("get"))
      return CSF_clipboardGet(c);
    else if (p1 == CsSymbolOf("put") || p1 == CsSymbolOf("set"))
      return CSF_clipboardPut(c);
    else
      CsUnexpectedTypeError(c, p1, "function, #get or #put");
    return UNDEFINED_VALUE;
  }

#pragma warning(pop)

  static value CSF_mediaVars(xvm *c) {
    value obj;
    value vars  = 0;
    bool  reset = false;
    CsParseArguments(c, "V*|V=|B", &obj, &vars,
                     &CsObjectDispatch, &reset);

    handle<xview> pv = xview_ptr(c, obj);
    //if (!pv) 
    //  CsThrowKnownError(c, CsErrGenericError, "view is already closed");

    if (vars) {
      tool::value vvars = value_to_value(c, vars);
      if (pv)
        pv->set_media_vars(vvars, reset, true);
      else
        xview::set_default_media_vars(vvars, reset);
    } else {
      tool::value map = tool::value::make_map();
      const html::media_variables &media_vars = pv ? pv->media_vars() : html::default_media_vars_provider.media_vars();
      for (int n = 0; n < media_vars.size(); ++n) {
        tool::value k = tool::value(media_vars.key(n)); // UT_SYMBOL
        map.set_prop(k, media_vars.value(n));
      }
      return value_to_value(c, map);
    }
    return UNDEFINED_VALUE;
  }

  static value CSF_mediaVar(xvm *c) {
    value  obj;
    value  var = 0;
    wchars name;
    CsParseArguments(c, "V=*S#|V", &obj, c->viewDispatch, &name.start,
                     &name.length, &var);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) CsThrowKnownError(c, CsErrGenericError, "view is already closed");

    html::media_variables &media_vars = pv->media_vars();

    if (var) {
      media_vars[name] = value_to_value(c, var);
      pv->update_media_vars(true);
      return UNDEFINED_VALUE;
    } else {
      return value_to_value(c, media_vars[name]);
    }
  }

  static value CSF_cursorLocation(xvm *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, c->viewDispatch);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) CsThrowKnownError(c, CsErrGenericError, "view is already closed");

    point pt = pv->cursor_pos();
    CS_RETURN2(c, CsMakeInteger(pt.x), CsMakeInteger(pt.y));
  }

  static value CSF_screenBox(xvm *c) {
    value obj  = 0;
    int   mn   = 0;
    value what = 0;
    value how  = 0;
    if (CsIntegerP(CsGetArg(c, 3)))
      CsParseArguments(c, "**iV=|V=", &mn, &what, &CsSymbolDispatch, &how,
                       &CsSymbolDispatch);
    else {
      CsParseArguments(c, "V=*V=|V=", &obj, c->viewDispatch, &what,
                       &CsSymbolDispatch, &how, &CsSymbolDispatch);
      handle<xview> pv = xview_ptr(c, obj);
      if (!pv)
        CsThrowKnownError(c, CsErrGenericError, "view is already closed");
      mn = html::screen_of(pv);
    }

    html::screen_info si;

    if (!html::get_screen_info(mn, si)) return UNDEFINED_VALUE;

    rect rc;

    static value s_frame     = CsSymbolOf(WCHARS("frame"));
    static value s_workarea  = CsSymbolOf(WCHARS("workarea"));
    static value s_device    = CsSymbolOf(WCHARS("device"));
    static value s_isPrimary = CsSymbolOf(WCHARS("isPrimary"));
    static value s_snapshot  = CsSymbolOf(WCHARS("snapshot"));

    if (what == s_frame)
      rc = si.monitor;
    else if (what == s_workarea)
      rc = si.workarea;
    else if (what == s_device)
      return CsMakeString(c, si.device_name);
    else if (what == s_isPrimary)
      return CsMakeBoolean(si.is_primary);
#ifdef SCREENSHOT_SUPPORT
    else if (what == s_snapshot) {
      handle<bitmap> bmp = html::get_screen_shot(mn);
      if (!bmp) return NULL_VALUE;
      return image_object(c, bmp);
    }
#endif
    else
      CsUnexpectedTypeError(c, what, "unknown option");

    //static value s_rect      = CsSymbolOf(WCHARS("rect"));
    static value s_rectw     = CsSymbolOf(WCHARS("rectw"));
    static value s_position  = CsSymbolOf(WCHARS("position"));
    static value s_dimension = CsSymbolOf(WCHARS("dimension"));
    static value s_left      = CsSymbolOf(WCHARS("left"));
    static value s_right     = CsSymbolOf(WCHARS("right"));
    static value s_top       = CsSymbolOf(WCHARS("top"));
    static value s_bottom    = CsSymbolOf(WCHARS("bottom"));
    static value s_width     = CsSymbolOf(WCHARS("width"));
    static value s_height    = CsSymbolOf(WCHARS("height"));

    if (how == s_rectw)
      CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()), CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    if (how == s_position)
      CS_RETURN2(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()));
    if (how == s_dimension) 
      CS_RETURN2(c, CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    if (how == s_left) return CsMakeInteger(rc.left());
    if (how == s_right) return CsMakeInteger(rc.right());
    if (how == s_top) return CsMakeInteger(rc.top());
    if (how == s_bottom) return CsMakeInteger(rc.bottom());
    if (how == s_width) return CsMakeInteger(rc.width());
    if (how == s_height) return CsMakeInteger(rc.height());

    CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()), CsMakeInteger(rc.right()), CsMakeInteger(rc.bottom()));
  }

  static value CSF_screens(xvm *c, value obj) {
    return int_value(html::number_of_screens());
  }

  static value CSF_screen(xvm *c, value obj) {
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) CsThrowKnownError(c, CsErrGenericError, "view is already closed");
    return int_value(html::screen_of(pv));
  }

  static value CSF_trayicon(xvm *c) {
    value obj = 0;
    value what = 0;
    CsParseArguments(c, "V=*V", &obj, c->viewDispatch, &what);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv)
      CsThrowKnownError(c, CsErrGenericError, "view is already closed");

    html::view::tray_icon_params tip;

    if (CsObjectP(what)) {
      value vimg = 0;
      CsGetProperty(c, what, CsSymbolOf(WCHARS("image")), &vimg);
      if (vimg) {
        gool::image *xim = image_ptr(c, vimg);
        if(xim)
          tip.img = xim;
      }
      CsGetProperty(c, what, "text", tip.tooltip);
      if (tip.img.is_defined() || tip.tooltip.is_defined()) {
        pv->trayicon_setup(tip);
        return TRUE_VALUE;
      }
    }
    else if (what == CsSymbolOf(WCHARS("remove"))) {
      pv->trayicon_remove();
      return TRUE_VALUE;
    }
    else if (what == CsSymbolOf(WCHARS("place"))) {
      rect rc;
      if(pv->trayicon_place(rc))
        CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()), CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    }
    return FALSE_VALUE;
  }

  static value CSF_logHandler(xvm *c) {
    value obj, cb = UNDEFINED_VALUE;
    CsParseArguments(c, "V=*|V", &obj, c->viewDispatch, &cb);

    handle<xview> pxv = xview_ptr(c, obj);
    if (!pxv) return FALSE_VALUE;

    if (CsMethodP(cb))
      pxv->set_debug_output(new scripting_debug_output(pxv, cb));
    else
      pxv->set_debug_output(nullptr);

    return TRUE_VALUE;
  }

  /*  function<void(html::url_data*)> create_url_data_tracker(xvm* c, value cb)
    {
      pvalue pcb(c,cb);
      return [pcb] (html::url_data* udata) {
        VM* c = pcb.pvm;
        auto_scope as(c,CsMethodNamespace(pcb));
        TRY
        {
          value obj = CsMakeObject(c, c->objectObject);
          PROTECT(obj);
          CsSetProperty(c,obj,"url",udata->url());
          CsSetProperty(c,obj,"contentUrl",udata->content_url());
          CsSetProperty(c,obj,"type", int_t(udata->data_type));
          CsSetProperty(c,obj,"mimeType", udata->data_content_type());
          CsSetProperty(c,obj,"encoding", udata->data_content_encoding());
          CsSetProperty(c,obj,"data",  CsMakeByteVector(c, udata->data() ));
          CsCallFunction(&as,pcb,1, obj );
        }
        CATCH_ERROR_NE
        {
          assert(false);
          //CsDisplay(pvm,pvm->val,pvm->standardError);
        }

      };
    }*/

  /*static value CSF_resourceTracker(xvm *c)
  {
    value obj, cb = UNDEFINED_VALUE;
    CsParseArguments(c,"V=*|V=", &obj,c->viewDispatch, &cb, &CsMethodDispatch);

    handle<xview> pxv = xview_ptr(c,obj);
    if( !pxv )
      return FALSE_VALUE;

    if( CsMethodP(cb) ) {
      pxv->debug_mode(true);
      pxv->url_data_tracker = create_url_data_tracker(c,cb);
    }
    else {
      pxv->debug_mode(false);
      pxv->url_data_tracker = nullptr;
    }

    return TRUE_VALUE;
  }*/

  static value CSF_getNodeByUID(xvm *c) {
    value obj;
    int_t uid;
    CsParseArguments(c, "V=*i", &obj, c->viewDispatch, &uid);

    handle<xview> pxv = xview_ptr(c, obj);
    if (!pxv || !pxv->doc()) return NULL_VALUE;

    handle<html::node> pn = pxv->doc()->get_node_by_uid(uid);

    if (!pn) return NULL_VALUE;

    if (pn->is_element())
      return element_object(c, pn.ptr_of<html::element>());
    else
      return node_object(c, pn.ptr_of<html::element>());
  }

  static value CSF_highlightedNode(xvm *c, value obj) {
    xview *pxv = xview_ptr(c, obj);
    if (pxv) {
      html::element *pel = pxv->get_highlighted();
      return pel ? element_object(c, pel) : NULL_VALUE;
    }
    return UNDEFINED_VALUE;
  }

  static void CSF_set_highlightedNode(xvm *c, value obj, value nobj) {
    xview *pxv = xview_ptr(c, obj);
    if (!pxv) return;

    if (nobj == NULL_VALUE) {
      pxv->set_highlighted(nullptr);
      return;
    }

    handle<html::node> pn = node_ptr(c, nobj);
    if (pn->is_element())
      pxv->set_highlighted(pn.ptr_of<html::element>());
    else
      pxv->set_highlighted(pn->get_owner());
  }

  static value CSF_getElementDetails(xvm *c) {
    value vobj;
    value eobj = NULL_VALUE;
    CsParseArguments(c, "V=*V=", &vobj, c->viewDispatch, &eobj,
                     c->elementDispatch);

    handle<xview> pxv = xview_ptr(c, vobj);
    if (!pxv) return UNDEFINED_VALUE;

    handle<html::element> b = element_ptr(c, eobj);

    if (!b) return UNDEFINED_VALUE;

    tool::value map = tool::value::make_map();

    {
      tool::value retval;
      b->applied_style_rules_report(*pxv, retval);
      map.set_prop("appliedStyleRules", retval);
    }

    {
      tool::value retval;
      b->used_style_props_report(*pxv, retval);
      map.set_prop("usedStyleProperties", retval);
    }

    {
      tool::value a = tool::value::make_map();
      for (int n = 0; n < b->atts.size(); ++n)
        a.set_prop(b->atts.name(n), tool::value(b->atts.value(n)));
      map.set_prop("attributes", a);
    }

    map.set_prop("states", tool::value((uint)b->state.data));

    return value_to_value(c, map);
  }

#if 0
  static value CSF_mutationPush(xvm *c) {
    value vobj;
    value eobj = NULL_VALUE;
    CsParseArguments(c, "V=*V=", &vobj, c->viewDispatch, &eobj, c->elementDispatch);

    handle<xview> pxv = xview_ptr(c, vobj);
    if (!pxv) return UNDEFINED_VALUE;

    handle<html::element> b = element_ptr(c, eobj);
    if (!b) return UNDEFINED_VALUE;

    bool first = pxv->mutator_push(b);
    return first ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_mutationPop(xvm *c) {
    value vobj;
    CsParseArguments(c, "V=*", &vobj, c->viewDispatch);

    handle<xview> pxv = xview_ptr(c, vobj);
    if (!pxv) return UNDEFINED_VALUE;

    bool last = pxv->mutator_pop();
    return last ? TRUE_VALUE : FALSE_VALUE;
  }
#endif

  static value CSF_post(xvm *c) {
    value m;
    bool  only_if_not_there = false;

    CsParseArguments(c, "**m|B", &m, &only_if_not_there);

    if (c->views.length() == 0) return FALSE_VALUE;

    xview *pv = c->views[0];
    if (!pv) return FALSE_VALUE;

    html::posted_event be;
    be.target = nullptr;
    be.cbf    = m;

    pv->post_event(be, only_if_not_there);
    return UNDEFINED_VALUE;
  }

  value CSF_http_request(xvm *c);

  extern value CSF_dialog(xvm *c); // string|stream, pos
  extern value CSF_window(xvm *c); // string|stream, pos
  extern value CSF_msgbox(xvm *c);
  
  static value CSF_isActive(xvm *c, value obj);

  value CSF_audio(xvm *c);

  /* file methods */
  static c_method methods[] = {
      // C_METHOD_ENTRY_X( "this", CSF_ctor     ),
      C_METHOD_ENTRY_X("close", CSF_close), 
      C_METHOD_ENTRY_X("load", CSF_load),
      C_METHOD_ENTRY_X("selectFile", CSF_selectFile),
      C_METHOD_ENTRY_X("selectFolder", CSF_selectFolder),
      C_METHOD_ENTRY_X("selectPrinter", CSF_selectPrinter),
      C_METHOD_ENTRY_X("dialog", CSF_dialog),
      C_METHOD_ENTRY_X("window", CSF_window),
#ifdef AUDIO_SUPPORT
      C_METHOD_ENTRY_X("audio", CSF_audio),
#endif
      C_METHOD_ENTRY_X("msgbox", CSF_msgbox), 
      C_METHOD_ENTRY_X("box", CSF_box),
      C_METHOD_ENTRY_X("move", CSF_move),
      C_METHOD_ENTRY_X("display", CSF_display),
      C_METHOD_ENTRY_X("doEvent", CSF_doEvent),
      C_METHOD_ENTRY_X("update", CSF_update),
      C_METHOD_ENTRY_X("clipboard", CSF_clipboard),
      C_METHOD_ENTRY_X("mediaVars", CSF_mediaVars),
      C_METHOD_ENTRY_X("mediaVar", CSF_mediaVar),
      C_METHOD_ENTRY_X("cursorLocation", CSF_cursorLocation),
      C_METHOD_ENTRY_X("screenBox", CSF_screenBox),
      C_METHOD_ENTRY_X("trayIcon", CSF_trayicon),
      C_METHOD_ENTRY_X("subscribe", CSF_subscribe),
      C_METHOD_ENTRY_X("on", CSF_subscribe),
      C_METHOD_ENTRY_X("unsubscribe", CSF_unsubscribe),
      C_METHOD_ENTRY_X("off", CSF_unsubscribe),
      C_METHOD_ENTRY_X("performDrag", CSF_performDrag),
      C_METHOD_ENTRY_X("focusable", CSF_focusable),

      C_METHOD_ENTRY_X("activate", CSF_activate),

      C_METHOD_ENTRY_X("request", CSF_http_request),
      C_METHOD_ENTRY_X("post", CSF_post),

      C_METHOD_ENTRY_X("windowMove", CSF_move),

      // debug peer support methods
      C_METHOD_ENTRY_X("logHandler", CSF_logHandler),
      C_METHOD_ENTRY_X("getNodeByUID", CSF_getNodeByUID),
      // C_METHOD_ENTRY_X( "highlightNode",      CSF_highlightNode  ),
      C_METHOD_ENTRY_X("getElementDetails", CSF_getElementDetails),
#if 0
      C_METHOD_ENTRY_X("mutationPush", CSF_mutationPush),
      C_METHOD_ENTRY_X("mutationPop", CSF_mutationPop),
#endif

#if defined(XSCITER)
      C_METHOD_ENTRY_X("hostCallback", CSF_callback),
#endif

      C_METHOD_ENTRY(0, 0)};

  /* String properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY_X("root", CSF_root, 0),
      VP_METHOD_ENTRY_X("all", CSF_all, 0),
      VP_METHOD_ENTRY_X("parent", CSF_owner, 0),
      VP_METHOD_ENTRY_X("state", CSF_state, CSF_set_state),
      VP_METHOD_ENTRY_X("parameters",  CSF_parameters, CSF_set_parameters),
      VP_METHOD_ENTRY_X("event",  CSF_event, 0),
      VP_METHOD_ENTRY_X("eventsRoot", CSF_eventsRoot, CSF_set_eventsRoot),
      VP_METHOD_ENTRY_X("focus", CSF_focus, CSF_set_focus),
      VP_METHOD_ENTRY_X("backend", CSF_backend, 0),
      VP_METHOD_ENTRY_X("backendName", CSF_backendName, 0),

      VP_METHOD_ENTRY_X("screens", CSF_screens, 0),
      VP_METHOD_ENTRY_X("screen", CSF_screen, 0),

      VP_METHOD_ENTRY_X("caption", CSF_caption, CSF_set_caption),
      VP_METHOD_ENTRY_X("aspectRatio", CSF_aspectRatio, CSF_set_aspectRatio),
      VP_METHOD_ENTRY_X("isTopmost", CSF_isTopmost, CSF_set_isTopmost),
      VP_METHOD_ENTRY_X("isResizeable", CSF_isResizeable, CSF_set_isResizeable),
      VP_METHOD_ENTRY_X("isResizable", CSF_isResizeable, CSF_set_isResizeable),
      VP_METHOD_ENTRY_X("isMaximizable", CSF_isMaximizable, CSF_set_isMaximizable),
      VP_METHOD_ENTRY_X("isMinimizable", CSF_isMinimizable, CSF_set_isMinimizable),
      VP_METHOD_ENTRY_X("blurBehind", CSF_blurBehind, CSF_set_blurBehind),
      VP_METHOD_ENTRY_X("minSize", CSF_minSize, CSF_set_minSize),
      VP_METHOD_ENTRY_X("maxSize", CSF_maxSize, CSF_set_maxSize),

      VP_METHOD_ENTRY_X("animationsEnabled", CSF_animationsEnabled, CSF_set_animationsEnabled),

      VP_METHOD_ENTRY_X("windowState", CSF_state, CSF_set_state),
      VP_METHOD_ENTRY_X("windowCaption", CSF_caption, CSF_set_caption),
      VP_METHOD_ENTRY_X("windowFrame", CSF_windowFrame, CSF_set_windowFrame),
      VP_METHOD_ENTRY_X("windowAspectRatio", CSF_aspectRatio, CSF_set_aspectRatio),
      VP_METHOD_ENTRY_X("windowTopmost", CSF_isTopmost, CSF_set_isTopmost),
      VP_METHOD_ENTRY_X("windowEnabled", CSF_isEnabled, CSF_set_isEnabled),
      VP_METHOD_ENTRY_X("windowResizable", CSF_isResizeable, CSF_set_isResizeable),
      VP_METHOD_ENTRY_X("windowMaximizable", CSF_isMaximizable, CSF_set_isMaximizable),
      VP_METHOD_ENTRY_X("windowMinimizable", CSF_isMinimizable, CSF_set_isMinimizable),
      VP_METHOD_ENTRY_X("windowBlurbehind", CSF_blurBehind, CSF_set_blurBehind),
      VP_METHOD_ENTRY_X("windowMinSize", CSF_minSize, CSF_set_minSize),
      VP_METHOD_ENTRY_X("windowMaxSize", CSF_maxSize, CSF_set_maxSize),
      VP_METHOD_ENTRY_X("windowIcon", CSF_icon, CSF_set_icon),
      VP_METHOD_ENTRY_X("windowIsActive", CSF_isActive, 0),

      VP_METHOD_ENTRY_X("windowResizePolicy", CSF_resizePolicy, CSF_set_resizePolicy),

      VP_METHOD_ENTRY_X("highlightedNode", CSF_highlightedNode, CSF_set_highlightedNode),



#ifdef PLATFORM_WINCE
      VP_METHOD_ENTRY_X("sip", CSF_SIP, CSF_set_SIP),
      VP_METHOD_ENTRY_X("fullscreen", CSF_Fullscreen, CSF_set_Fullscreen),
      VP_METHOD_ENTRY_X("sipUp", CSF_SIPState, CSF_set_SIPState),
      VP_METHOD_ENTRY_X("menu", CSF_menu, 0),
#endif

      VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {
      // CONSTANT_ENTRY_X("WINDOW_SHOWN"       , int_value(WINDOW_SHOWN )),
      CONSTANT_ENTRY_X("WINDOW_SHOWN", int_value(html::WINDOW_SHOWN)),
      CONSTANT_ENTRY_X("WINDOW_MINIMIZED", int_value(html::WINDOW_MINIMIZED)),
      CONSTANT_ENTRY_X("WINDOW_MAXIMIZED", int_value(html::WINDOW_MAXIMIZED)),
      CONSTANT_ENTRY_X("WINDOW_HIDDEN", int_value(html::WINDOW_HIDDEN)),
      CONSTANT_ENTRY_X("WINDOW_FULL_SCREEN", int_value(html::WINDOW_FULL_SCREEN)),
      CONSTANT_ENTRY_X("ANIMATION_TIMER_SPAN", int_value(html::ANIMATION_TIMER_SPAN)),

      CONSTANT_ENTRY_X("FRAME_WINDOW", int_value(html::FRAME_WINDOW)),
      CONSTANT_ENTRY_X("TOOL_WINDOW", int_value(html::TOOL_WINDOW)),
      CONSTANT_ENTRY_X("POPUP_WINDOW", int_value(html::POPUP_WINDOW)),
      CONSTANT_ENTRY_X(0, 0)};

  // this method delegates view.something(...) calls to event_handler attached
  // to the view.
  static bool HandleViewCall(xvm *c, value obj, value tag, int argc, value *pretval) {

    //value obj;
    //CsParseArguments(c, "V=*", &obj, c->viewDispatch);
    handle<xview> pv = xview_ptr(c, obj);
    if (!pv) CsThrowKnownError(c, CsErrGenericError, "view is already closed");

    //OBSOLETE: if (pv->call_behavior_method(0, c, tag, *pretval)) { return true; }

    argc -= 2;

    array<tool::value> argv;
    argv.size(argc);

    for (int i = 0; i < argc; ++i)
      argv[i] = value_to_value(c, CsGetArg(c, i + 3));

    string      sym = CsSymbolName(tag);
    tool::value retval;

    if (pv->call_behavior_method(0, sym, argv.head(), argv.size(), retval)) {
      *pretval = value_to_value(c, retval, true);
      return true;
    }
    return false;
  }

  static void xview_scan(VM *c, value obj) {
    handle<xview> pv = xview_ptr((xvm *)c, obj);
    if (pv) pv->on_gc();
    CsCObjectScan(c, obj);
  }

  static value xview_copy(VM *c, value obj) { return CsDefaultCopy(c, obj); }

  static bool GetViewProperty(xvm *c, value &obj, value tag, value *pValue) {
    xview *pv = xview_ptr(c, obj);
    if (!pv) return false;
#if 0
    if (pv->ground_behavior) {
      tristate_v r = CsGetNativeBehaviorProperty(c, pv->ground_behavior, tag, pValue);
      if (r.is_defined()) return !!r.val();
    }
#else
    if (pv->ground_behavior) {
      handle<html::ctl> pb = pv->ground_behavior;
      while (pb) {
        if (som_passport_t* psp = pb->asset_get_passport()) {
          if (psp->name == tag) {
            *pValue = CsMakeAssetObject(c, pb->as_asset());
            return true;
          }
        }
        pb = pb->next;
      }
    }
#endif
    return CsGetCObjectProperty(c, obj, tag, pValue);
  }

  static bool SetViewProperty(xvm *c, value obj, value tag, value val) {
#if 0
    xview *pv = xview_ptr(c, obj);
    if (!pv) return false;
    if (pv->ground_behavior) {
      tristate_v r = CsSetNativeBehaviorProperty(c, pv->ground_behavior, tag, val);
      if (r.is_defined()) return !!r.val();
    }
#endif
    return CsSetCObjectProperty(c, obj, tag, val);
  }

  void xvm::init_view_class() {
    // create the 'View' type
    dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(this), "View", methods, properties, constants);
    if (!pd)
      CsInsufficientMemory(this);
    else {
      pd->baseType = &CsCObjectDispatch;
      pd->destroy = (destructor_t)destroy_view;
      pd->scan = xview_scan;
      pd->copy = xview_copy;
      pd->handleCall = (call_method_t)HandleViewCall;
      pd->binOp = CsDefaultObjectBinOp;
      pd->getProperty = (get_property_t)GetViewProperty;
      pd->setProperty = (set_property_t)SetViewProperty;
    }
    viewDispatch = pd;
  }

  void xvm::GC_started() {
    elementListObject = CsCopyValue(this, elementListObject);
    nodeListObject    = CsCopyValue(this, nodeListObject);

    for (index_t i = views.last_index(); i >= 0; --i)
      views[i]->on_gc();
  }

  /*tool::async::tasks *xvm::get_tasks() {
    xview *pxv = current_view();
    return pxv ? pxv : nullptr;
  }*/

  /*  struct gc_functor : public html::element_functor
    {
        xvm* vm;
        virtual bool do_it(html::element *b)
        {
          if( b->obj )
            b->obj = CsCopyValue(vm,b->obj);
          FOREACH(n, b->subscriptions )
          {
            html::subscription* ps = b->subscriptions[n];
            ps->fcn = CsCopyValue(vm,ps->fcn);
          }
          return false;
        }
    };*/

  void xview::on_gc() {
    // if( is_unloading ) - unfortunately GC can be called while handling
    // document close.
    //  return;

    if (last_gc_generation == vm->gc_generation) return;

    last_gc_generation = vm->gc_generation;

    html::document *pd = doc();

    function<bool(html::node *)> scan = [this, &scan](html::node *n) -> bool {
      if (n->obj) {
        // if(CsIsNewObjectP(vm,el->obj))
        //  return false;
        n->obj = CsCopyValue(vm, n->obj);
      }
      if (!n->is_element()) return false;

      html::element *el = static_cast<html::element *>(n);

      if (el->is_document() && static_cast<html::document *>(el)->ns)
        static_cast<html::document *>(el)->ns =
            CsCopyValue(vm, static_cast<html::document *>(el)->ns);

      /*FOREACH(i, el->subscriptions )
      {
        html::subscription* ps = el->subscriptions[i];
        ps->fcn = CsCopyValue(vm,ps->fcn);
      }*/
      html::ctl *bhv = el->behavior;
      while (bhv) {
        bhv->scan_owned_elements(scan);
        bhv = bhv->next;
      }
      return false;
    };

    html::tree_scanner ts(pd);

    // copy_element(vm,this,pd);

    ts.add_root(mouse_over_element);
    ts.add_root(mouse_down_element);
    ts.add_root(mouse_capture_element);
    ts.add_root(focus_element);
    ts.add_root(event_capture_element);
    ts.add_root(bfc);
    // ts.add_root(dragging_source_element);
    ts.add_root(gesture_element);
    if (highlighted_ctl)
      ts.add_root(highlighted_ctl->el);

    FOREACH(n, windows){
      handle<iwindow> pw = windows[n];
      ts.add_root(pw->root());
      ts.add_root(pw->anchor());
      ts.add_root(pw->pfocus());
    }

    {
      to_update.foreach_element([&](html::element *el) -> bool {
        ts.add_root(el);
        return false;
      });
    }

    {
      critical_section _p(posted_guard);
      FOREACH(n, posted_functors) {
        functor *pf = posted_functors[n];
        if (pf->is_of_type<html::element>()) {
          tool::resource *pr = pf;
          html::element * pb = static_cast<html::element *>(pr);
          ts.add_root(pb);
        }
      }
      FOREACH(n, _posted_functors) {
        functor *pf = _posted_functors[n];
        if (pf->is_of_type<html::element>()) {
          tool::resource *pr = pf;
          html::element * pb = static_cast<html::element *>(pr);
          ts.add_root(pb);
        }
      }
    }

    ts.each_node(scan);

    FOREACH(n, element_timers) {
      html::timer_def &td = element_timers[n];
      if (td.kind == html::SCRIPT_TIMER && td.id) td.id = CsCopyValue(vm, td.id);
    }

    {
      critical_section _p(posted_guard);
      FOREACH(k, posted_events) {
        handle<html::posted_event> pe = posted_events[k];
        if (pe->cbf) pe->cbf = CsCopyValue(vm, pe->cbf);
      }
      FOREACH(k, _posted_events) {
        handle<html::posted_event> pe = _posted_events[k];
        if (pe->cbf) pe->cbf = CsCopyValue(vm, pe->cbf);
      }
    }

}

tool::ustring xvm::resolve_url(const tool::ustring &base,
                               const tool::ustring &relative) {
  return html::combine_url(string(base), string(relative));
}

stream *xvm::open_stream(const tool::ustring &url, bool as_text) {
  value gns = currentScope()->globals;

  static value self_sym = CsSymbolOf("self");

  value self_obj = NULL_VALUE;

  if (!CsGetProperty(this, gns, self_sym, &self_obj))
    return super::open_stream(url, as_text);

  html::element *self = element_ptr_no_throw(this, self_obj);
  if (!self) return super::open_stream(url, as_text);

  html::view *pv = self->pview();
  if (!pv) return super::open_stream(url, as_text);

  handle<html::pump::request> prq =
      new html::pump::request(url::escape(url), html::DATA_SCRIPT);
  prq->dst = self->doc();
  if (pv->load_data(prq, true)) {
    if (prq->success_flag) return new binary_i_stream(prq->data, url);
    return nullptr;
  }

  return super::open_stream(url, as_text);
}

bool xview::handle_on_idle() {
  auto_state<xview *> _(vm->_current_view, this);
  bool r = super::handle_on_idle();
  purge_deleted_elements();
  if(vm)
    vm->deliver_notifications();
  return r;
}

void xview::run(stream &in, html::hdocument pd, int line_no) {
  TRY {
    auto_state<xview *> _(vm->_current_view, this);
    tis::auto_scope as(vm, pd->ns);
    CsLoadStream(&as, &in, 0, line_no);
  }
  CATCH_ERROR_NE {
    // string_stream s;
    // CsDisplay(vm,vm->val,&s);
    // vm->standardError->put_str(s.to_ustring());
    CsHandleUnhandledError(vm);
  }
}

value xview::eval(stream &in, html::hdocument pd, int line_no) {
  TRY{
    auto_state<xview *> _(vm->_current_view, this);
    tis::auto_scope as(vm, pd->ns);
    return CsEvalStream(&as, pd->ns, &in, line_no);
  }
  CATCH_ERROR_NE{
    // string_stream s;
    // CsDisplay(vm,vm->val,&s);
    // vm->standardError->put_str(s.to_ustring());
    CsHandleUnhandledError(vm);
  }
  return UNDEFINED_VALUE;
}


#ifdef _DEBUG
void printPins(VM *c, const char *msg);
#endif

bool xview::ask_unload(html::document *pd, view::UNLOAD_REASON reason) {
  if (!super::ask_unload(pd, reason)) return false;
  if (!pd) return true;
  if (pd && view_obj.is_set()) {
    value r  = TRUE_VALUE;
    value re = UNDEFINED_VALUE;
    switch (reason) {
    case UNLOAD_BY_CHROME:
      re = CsSymbolOf(WCHARS("by-chrome"));
      break; // user clicked close button on window
    case UNLOAD_BY_CODE:
      re = CsSymbolOf(WCHARS("by-code"));
      break; // view.close() issued
    case UNLOAD_BY_LOAD:
      re = CsSymbolOf(WCHARS("by-load"));
      break; // document unload-old/load-new
    }
    if (send_notification(pd, element_object(vm, pd), get_sym_by_id(id_closing), re, r)) {
      if (r == FALSE_VALUE) return false;
    }
  }
  return true;
}

bool xview::on_unload(html::document *pdoc) {
  handle<html::document> pd = pdoc;
  bool is_root_doc = pd == doc();
  super::on_unload(pd);
  if (!pd) return true;
  if (pd && view_obj.is_set()) {

    if (is_root_doc)
      fire_event(WCHARS("closing"));

    value r  = TRUE_VALUE;
    // pd->send_custom_event(*this,WCHARS("close"));
    send_notification(pdoc, element_object(vm, pd), get_sym_by_id(id_closed), r);

    // tis::printPins(vm, "xview::on_unload 1");
    pd->ns = UNDEFINED_VALUE;

    if (is_root_doc) {
      fire_event(WCHARS("close"));
      CsCollectGarbage(vm);
    } 
  }

#ifdef _DEBUG
  //   dbg_printf( "!!!! total elements = %d\n", html::element::_total);
  dbg_printf("!!!! total styles = %d\n", html::style::_total);
#endif

  return true;
}

void load_script_element(xview *pv, html::hdocument pd, html::helement b) {
  string src = b->atts[html::attr::a_src];
  ustring media = b->atts[html::attr::a_media];
  if (media.is_defined()) {
    if (!pv->match_media_type(media))
      return;
  }
  if (!src.is_empty()) {
    src = html::combine_url(b->doc()->uri().src, src);
    handle<html::pump::request> prq =
        new html::pump::request(src, html::DATA_SCRIPT);
    prq->dst = pd.ptr();
    if (pv->load_data(prq, true)) {
      // include_handler ih(pv);
      binary_i_stream in(prq->data, tool::url::unescape(src), false);
      in.set_encoder(stream::utf8_encoder());
      pv->run(in, pd);
    } else {
      pv->vm->standardError->printf(W("Warning: cannot load %S\n"), src.c_str());
    }
  }

  ustring text = b->get_text(*pv);

  if (text.length()) {
    ustring id  = b->attr_id();
    ustring src = //WRONG, it must still be an url: tool::url::unescape(b->doc()->uri().src);
                  b->doc()->uri().src;
    /* WTF???? if (id.length()) {
      src += "#";
      src += string(id);
    }*/
    string_i_stream in(text, src);
    // include_handler ih(pv);
    pv->run(in, pd, b->line_no());
  }
}

bool xview::load_script(html::document *d, const string &url,
                        const string &mime_type, wchars text) {
  auto_scope      as(vm, d->ns);
  ustring         src = tool::url::unescape(url);
  string_i_stream in(text, src);
  // include_handler ih(this);
  run(in, d);
  return true;
}
bool xview::include_script(html::document *d, const string &url,
                           const string &mime_type) {
  auto_scope as(vm, d->ns);
  ustring    src = url;//WRONG tool::url::unescape(url);
  return CsInclude(CsCurrentScope(vm), string_to_value(vm, src), true) !=
         UNDEFINED_VALUE;
}

//void CSF_make_element(xvm *c, value tuple, tool::handle<html::element> &b);

bool xview::mount_component(html::helement b) {
  html::hdocument d = b->doc();
  if (!d) return false;
  string src = b->atts[html::attr::a_src];
  if (!src.is_empty()) {
    src = html::combine_url(d->uri().src, src);
    if (!include_script(d, src)) {
      view::debug_printf(html::OT_DOM, html::OS_ERROR, "cannot load component file <%s> at (%s(%d))\n", src.c_str(), d->uri().src.c_str(), b->line_no().val(0));
      return false;
    }
  }
  ustring type = b->attr_type();
  bool is_content = false;
  if (type.is_undefined()) {
    is_content = true;
    type = WCHARS("content");
  }
  ustring text = b->get_text(*this);
  ustring script;
  ustring attributes;
  
  for (auto at : b->atts.items)
  {
    if (at.att_sym == html::attr::a_type)
      continue;

    if (attributes.length())
      attributes.append(WCHARS(" "));
    ustring aname = html::attr::symbol_name(at.att_sym);
    ustring aval = at.att_value;
    attributes.append(aname());
    if(aval.length())
    {
      attributes.append(WCHARS("="));
      if( aval().starts_with('{') && aval().ends_with('}') )
        attributes.append(aval());
      else {
        attributes.append(WCHARS("\""));
        attributes.append(xml_escape_seq(aval())());
        attributes.append(WCHARS("\""));
      }
    }
  }
      
  if (text.length())
    script = ustring::format(W("<%s %s>%s</%s>"), type.c_str(), attributes.c_str(), text.c_str(), type.c_str());
  else
    script = ustring::format(W("<%s %s />"), type.c_str(), attributes.c_str());

  TRY {

  string_i_stream in(script(), d->uri().src);
  eval(in, d, b->line_no());
  if (!CsTupleP(vm->val)) // we've got vnode
    return false;

  auto_scope as(vm, d->ns);

  html::helement p = b->parent;
  ElementCreatorCtx ctx(this, vm->val,p.ptr());
  html::helement mb = ctx.make();
  if (!mb) return false;
  if (is_content) {
    int pos = int(b->node_index);
    b->remove(true);
    p->insert_nodes(pos,mb->nodes(), this);
  }
  else
    p->replace_child(b, mb, this);

  } CATCH_ERROR_NE{
    CsHandleUnhandledError(vm);
  }
      
  return true;
}

static c_method global_methods[] = {C_METHOD_ENTRY_X("$", CSF_$_global),
                                    C_METHOD_ENTRY_X("$$", CSF_$$_global),
                                    C_METHOD_ENTRY_X(0, 0)};

static string _init_script;
    
void xview::set_init_script(const char* script_text) {
  _init_script = script_text;
}

void xview::on_load_start(html::document *d) {
    
  if (!vm) // unloading
    return;
  if (d->parent)
    // d->ns = CsMakeObject(vm,d->parent->doc()->ns); // namespace of in-frame
    // document is derived from parent document namespace,
    //                                               // is this right thing to
    //                                               do? It can be usable in
    //                                               case when host document
    //                                               supplies behaviors for the
    //                                               child.
    d->ns = CsNewNamespaceInstance(vm, d->parent->doc()->ns, NULL_VALUE);
  else
    d->ns = CsNewNamespaceInstance(vm, vm->globalScope.globals, NULL_VALUE);

  TRY {
    auto_scope as(vm, d->ns);

    value doc_obj = element_object(vm, d);
    CsAddConstant(vm, d->ns, CsSymbolOf("self"), doc_obj);
    CsAddConstant(vm, d->ns, CsSymbolOf("view"), view_obj);
    CsEnterMethods(vm, d->ns, global_methods);

#if 1
    if (!d->parent && debug_mode() /*&& d->uri().src() != CHARS("sciter:msgbox.htm")*/) {
      value dp = 0;
      if (!CsGetProperty(vm, view_obj, "connectToInspector", dp, &CsMethodDispatch)) {
        handle<request> hrq = new pump::request("sciter:debug-peer.tis", html::DATA_SCRIPT);
        hrq->dst      = d;
        hrq->dst_view = this;
        if (load_data(hrq, true)) {
          if (hrq->data.length()) {
            auto_scope      as(vm, d->ns);
            binary_i_stream in(hrq->data(), W("sciter:debug-peer.tis"), false);
            in.set_encoder(stream::utf8_encoder());
            run(in, d);
          }
        } else
          debug_printf(html::OT_DOM, html::OS_WARNING,"sciter:debug-peer.tis\n");
      }
    }
#endif
    if (!d->parent && _init_script.length()) {
          auto_scope      as(vm, d->ns);
          binary_i_stream in(_init_script.chars_as_bytes(), W("sciter:init-script.tis"), false);
          in.set_encoder(stream::utf8_encoder());
          run(in, d);
    }
      
  }
  CATCH_ERROR(e) {
    if (e.number != 0) // close() called
      tis::CsHandleUnhandledError(vm);
  }
  super::on_load_start(d);
}

extern value CSF_select(xvm *c);
extern value CSF_selectParent(xvm *c);

/*static c_method global_methods[] = {
  C_METHOD_ENTRY_X( "$",                  CSF_select  ),
  C_METHOD_ENTRY_X( 0,               0          )
};*/

void xview::on_load_end(html::document *d, bool existing_doc_replaced) {
    
  if (!vm) // unloading
    return;

  super::on_load_end(d, existing_doc_replaced);

  html::hdocument pdoc = d;
  handle<xview> holder(this);
  // if(!html::find_first(d,"script[type='text/tiscript']"))
  //  return;

  TRY {
    // CsEnterMethods(vm, d->ns, global_methods);
    {
      // styler_functor sf(this);
      // d->do_for_each(sf); // create styles

      // load and execute scripts

      if (d->allow_own_scripts) {
        array<html::helement> script_elements;
        find_all(*this, script_elements, pdoc,
                 WCHARS("script[type='text/tiscript'],script[type='application/tiscript'],script[src$='.tis']"));
        for (int n = 0; n < script_elements.size(); ++n) {
          load_script_element(this, pdoc, script_elements[n]);
          if (_finished) return;
        }
      }

      // resolve prototypes

      if (d->allow_script_behaviors)
        d->check_prototypes_tree(*this);
      
      if (d->allow_own_scripts) {
        value r = TRUE_VALUE;
        pdoc->operational = true;
        send_notification(pdoc, element_object(vm, d), get_sym_by_id(id_ready), r);
      }
    }
  }
  CATCH_ERROR(e) {
    if (e.number != 0) // close() called
      tis::CsHandleUnhandledError(vm);
  }

  if (existing_doc_replaced) 
    CsCollectGarbage(vm);
}

static void invoke_attached(xvm *c, html::helement el)
{
  value obj = element_object(c, el);

  value cls = CsObjectClass(obj);
  PROTECT(obj, obj, cls);

  value m = 0;

  if (!CsGetProperty1(c, cls, get_sym_by_id(id_attached), &m))
    return; // no such method...

  html::document* pd = el->doc();
  if (!pd)
    return;

  if (CsMethodP(m)) {
    auto_scope as(c, pd->ns);
    TRY{
      CsCallMethod(c, obj, m, cls, 0);
    }
      CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(c);
    }
  }
}

void check_scripting_methods(xview* pv, html::helement b) {
  value obj = element_object_nc(pv->vm, b);
  if (obj) {
    auto has_method = [&](uint id) -> bool {
      value method = NULL_VALUE;
      if (CsGetProperty(pv->vm, obj, get_sym_by_id(id), &method))
        return CsMethodP(method);
      else
        return false;
    };
    b->flags.has_on_size_script = has_method(id_onSize);
    b->flags.has_on_visibility_changed_script = has_method(id_onVisibilityChanged);
    b->flags.script_draws_background = has_method(id_paintBackground);
    b->flags.script_draws_foreground = has_method(id_paintForeground);
    b->flags.script_draws_content = has_method(id_paintContent);
    b->flags.script_draws_outline = has_method(id_paintOutline);
    b->flags.script_has_is_point_inside = has_method(id_isPointInside);
  }
}


void xview::on_style_resolved(handle<html::element> b,
                              const html::style *   prev_style) {

#ifdef _DEBUG
  //b->dbg_report("xview::on_style_resolved");
#endif

  super::on_style_resolved(b, prev_style);

  if (this->dismissing) return;

  if (vm->collecting_garbage) return;

  if (b->c_style == html::element::null_style)
    return;

  if (!b->is_connected())
    return;

  html::hstyle cs = b->c_style;

  tool::string name      = cs->prototype;
  tool::string prev_name = prev_style->prototype;

  auto_state<xview *> _1(vm->_current_view, this);

#ifdef _DEBUG
  //if (name == CHARS("MainController"))
  //  name = name;
  if(b->attr_id() == WCHARS("meeting"))
    name = name;
#endif // _DEBUG

  html::helement el = b;

  if (el->flags.script_need_attached_call)
  {
#ifdef DEBUG
    el->dbg_report("forced attached");
#endif
    el->flags.script_need_attached_call = 0;
    invoke_attached(vm, el);
    check_scripting_methods(this, el);
  }
  
  if (name || prev_name) {
    process_prototype(el, name, prev_name);
  }

  if (cs->handler_list.defined()) { 
    process_handlers(el, cs->handler_list); 
  }
}

void xview::process_prototype(handle<html::element> b, const tool::string &name,
                              const tool::string &prev_name) {

  tool::string    script_url = b->c_style->prototype_url;
  html::hdocument pd         = b->doc();
  if (!pd)
    return;

  if (!pd->allow_script_behaviors)
    return;

  if ((name != prev_name) && (prev_name.length() != 0)) {
    {
      value obj = element_object_nc(vm, b);
      if (!obj) return;
      if (CsObjectClass(obj) == vm->elementDispatch->obj || CsObjectClass(obj) == UNDEFINED_VALUE) // if it is not assigned already
        return;

      if (b->flags.script_reactors_prototype) {
        CsWarning(vm, string::format("attempt to remove prototype '%s' set by Reactor", prev_name.c_str()));
        return;
      }
    }

    TRY{
      invoke_event_function(pd, element_object(vm, b), get_sym_by_id(id_detached));
      CsSetObjectClass(element_object(vm, b), vm->elementDispatch->obj);
    }
    CATCH_ERROR_NE{ tis::CsHandleUnhandledError(vm); }
      // WRONG: CsSetObjectClass(obj, UNDEFINED_VALUE);
      // as it can be assigned manually.
  }


  if (name.length()) {
    if (b->flags.strayed) return;
    if (!pd->state.ready()) 
      return; // do it later, behaviors MUST be assigned after <script>s

    if (script_url.length()) {
      if (!include_script(pd, script_url,"text/tiscript")) // value r = CsInclude(&as, script_url, true);
        return;
    }
    TRY {
      value bhvo = CsGetGlobalValueByPath(vm, pd->ns, name);

      if (bhvo != UNDEFINED_VALUE) {
        value obj = 0;
        PROTECT_P(this->vm, bhvo, obj);
        obj = element_object(vm, b);
        if (CsObjectClass(obj) != bhvo) // if it is not assigned already
        {
          CsSetObjectClass(obj, bhvo);
          CsMergeThisVarsFromClass(vm, obj, bhvo);
          invoke_event_function(pd, obj, get_sym_by_id(id_attached));
          goto CHECK_METHODS;
        } else
          return;
      } 
      else {
        CsWarning(vm, string::format("prototype, class '%s' not found", name.c_str()).c_str());
      }
    }
    CATCH_ERROR_NE { tis::CsHandleUnhandledError(vm); }
  }

  if (name != prev_name) 
    CHECK_METHODS: check_scripting_methods(this, b);
 }

// this calls script function(s) defined by CSS::handler proprty.
// each function if found is called precisely once.
void xview::process_handlers(handle<html::element>       b,
                             const html::handler_list_v &handlers) {
  html::hdocument pd = b->doc();
  if (!pd) return;
  if (!pd->ns) return;
  if (!pd->allow_script_behaviors) return;
  if (!pd->state.ready()) return; // do it later

  //auto_state<html::element*> 

  auto already_processed = [b](const html::handler_list_v::item *fd) -> bool {
    bool found = false;
    html::each_resource<html::handler_list_v::item>(
        b->meta(), [fd, &found](const html::handler_list_v::item *it) -> bool {
          if (fd->funcname == it->funcname) {
            found = true;
            return true;
          }
          return false;
        });
    return found;
  };

  auto one = [&](const html::handler_list_v::item *hd) {

    if (hd->url.length()) {
      if (!include_script(pd, hd->url, "text/tiscript")) 
        return;
    }
    TRY {
      value callable = CsGetGlobalValueByPath(vm, pd->ns, hd->funcname);

      if (CsMethodP(callable) || CsCMethodP(callable)) {
        //
        b->meta.push(hd); // mark it as processed
        value obj = 0;
        PROTECT_P(this->vm, obj, callable);
        obj = element_object(this->vm, b);
        if (hd->params.size()) {
          value params = dictionary_to_value(vm, hd->params);
          invoke_event_function(pd, obj, callable, params);
        } else
          invoke_event_function(pd, obj, callable);
      }
      else
        vm->standardOutput->printf(W("warning:aspect '%S' not found\n"), hd->funcname.c_str());
    }
    CATCH_ERROR_NE { tis::CsHandleUnhandledError(vm); }
  }; // one

  int nelements = handlers.list->size();
  for (int n = 0; n < nelements; ++n) {
    const html::handler_list_v::item *pit = handlers.list->operator[](n);
    if (!already_processed(pit)) one(pit);
  }
}

void xview::on_element_removing(html::element *pb) 
{
  ON_SCOPE_EXIT(super::on_element_removing(pb));

  if (vm->collecting_garbage) return;
  // if (this->dismissing)
  //  return;

  value obj = element_object_nc(vm, pb);
  if (!obj) return;

  html::hdocument pdoc = pb->doc();
  if (!pdoc) pdoc = this->doc();
  if (!pdoc) return;
  if (!pdoc->allow_script_behaviors) return;

  value ns = pdoc->ns;
  if (!ns) return;

  TRY{
    CsEventObjectFire(vm,obj,get_sym_by_id(id_detached));
  }
  CATCH_ERROR_NE{
    tis::CsHandleUnhandledError(vm);
  }
}

#if 1

void xview::enqueue_for_deletion(html::element *pb) {
  if (deletion_queue.size() == 0) request_idle();
  deletion_queue.push(pb);
}

void xview::purge_deleted_elements() {
  int nm = deletion_queue.size();
  for (int n = 0; n < nm; ++n) {
    html::helement pb  = deletion_queue[n];
    value          obj = element_object_nc(vm, pb);
    if (!obj) 
      continue;
    if (!pb->flags.strayed)
      continue;
    CsPurgeObject(obj);
    CsSetCObjectValue(obj, 0);
    pb->obj = 0;
    if (pb->is_document())
      pb.ptr_of<html::document>()->ns = 0;
    pb->release();
  }
  if (nm) {
    deletion_queue.clear();
    if (vm && !dismissing)
      CsCollectGarbage(vm);
  }
}

void xview::on_element_removed(html::element *_pb) {
  handle<html::element> pb = _pb;

  ON_SCOPE_EXIT(super::on_element_removed(pb));

  if (vm->collecting_garbage) return;
  // if (this->dismissing)
  //  return;

  value obj = element_object_nc(vm, pb);
  if (!obj) return;

  enqueue_for_deletion(_pb);

  if (CsObjectClass(obj) == UNDEFINED_VALUE ||
      CsObjectClass(obj) == vm->elementDispatch->obj) // if it is not assigned already
    return;

  html::hdocument pdoc = pb->doc();
  if (!pdoc) pdoc = this->doc();
  if (!pdoc) return;

  value ns = pdoc->ns;
  if (!ns) return;

  if (!pdoc->allow_script_behaviors) return;
  
  TRY {
    invoke_event_function(pdoc, obj, get_sym_by_id(id_detached));
    obj = element_object_nc(vm, pb);
    if (!obj) return;
    CsSetObjectClass(obj, vm->elementDispatch->obj);
  }
  CATCH_ERROR_NE { 
    tis::CsHandleUnhandledError(vm); 
  }
}

#else

void xview::on_element_removed(html::element *_pb) {
  handle<html::element> pb = _pb;
  super::on_element_removed(pb);
  if (vm->collecting_garbage) return;
  // if (this->dismissing)
  //  return;

  value obj = element_object_nc(vm, pb);
  if (!obj) return;
  if (CsObjectClass(obj) == UNDEFINED_VALUE) // if it is not assigned already
    goto FORGET;

  html::document *pdoc = pb->doc();
  if (!pdoc) pdoc = this->doc();
  if (!pdoc) goto FORGET;

  value ns = pdoc->ns;
  if (!ns) goto FORGET;

  TRY {
    invoke_event_function(ns, obj, get_sym_by_id(id_detached));
    obj = element_object_nc(vm, pb);
    if (!obj) return;
    // that's too weak: CsSetObjectClass(obj, vm->elementDispatch->obj);
  }
  CATCH_ERROR_NE { tis::CsHandleUnhandledError(vm); }
FORGET:
  CsPurgeObject(obj);
  CsSetCObjectValue(obj, 0);
  pb->obj = 0;
  pb->release();
}
#endif

string xview::get_sciter_home() {
  if (home_url.length() == 0) home_url = WCHARS("file://") + tool::get_home_dir();
  return home_url;
}

bool is_script(html::pump::request *rq) {
  // if( rq->data_content_type.like("text/javascript*")) return true;
  // if( rq->data_content_type.like("application/javascript*")) return true;
  if (rq->data_content_type.like("text/tiscript*")) return true;
  if (rq->data_content_type.like("application/tiscript*")) return true;
  return false;
}
bool is_json_data(html::pump::request *rq, bool force_json) {
  if (rq->data_content_type.like("application/json*")) return true;
  if (rq->data_content_type.like("text/json*")) return true;
  if (force_json && rq->data_content_type.like("text/*")) return true;
  return false;
}
bool is_text(html::pump::request *rq) {
  if (rq->data_content_type.length() == 0) {
    if (rq->data.get_index(0) >= 0) return false;
    return true;
  }
  if (rq->data_content_type.like("text/*")) return true;
  if (chars(rq->data_content_type).head(';').like("*+xml")) return true;
  return false;
}

enum DATA_PROTOCOL {
  PROTOCOL_BASIC,
  PROTOCOL_MULTIPART,
  PROTOCOL_JSON,
};

enum REQUEST_REPRESENTATION_FORMAT {
  OF_AUTO,
  OF_STREAM,
  OF_JSON,
  OF_BYTES,
  OF_STRING,
};

value parse_response(xvm *vm, html::request *rq, uint of) {
  
  value val = UNDEFINED_VALUE;

  PROTECT_VM(val);

  if (of == OF_AUTO) {
    if (is_script(rq))
      of = OF_JSON;
    else if (is_json_data(rq, false))
      of = OF_JSON;
    else if (is_text(rq))
      of = OF_STREAM;
    else
      of = OF_BYTES;
  }

  switch (of) {
  case OF_JSON: {
#if 0
    string_stream s(rq->data());
    TRY {
      value      dummy = CsMakeObject(vm, UNDEFINED_VALUE);
      auto_scope as(vm, dummy);
      val = CsEvalDataStream(&as, &s);
    }
    CATCH_ERROR(e) {
      if (e.number == 0) // close() called
        return true;
      else
        CsHandleUnhandledError(vm);
      val = vm->val;
    }
    s.close();
#else
    ustring     jsdata = u8::cvt(rq->data());
    auto        chars = jsdata();
    tool::value v = xjson::parse(chars, false);
    val = value_to_value(vm, v);
#endif
    return val;
  }

  case OF_STREAM: {
    tool::ustring     str;
    string_stream_sd *ps = nullptr;
    if (tool::decode_bytes(rq->data(), str, rq->data_content_encoding))
      ps = new string_stream_sd(str.c_str(), str.length());
    else
      ps = new string_stream_sd(rq->data());
    ps->name = rq->real_url();
    return CsMakeFile(vm, ps);
  }
  case OF_STRING: {
    tool::ustring str;
    if (!tool::decode_bytes(rq->data(), str, rq->data_content_encoding))
      str = u8::cvt(rq->data());
    return string_to_value(vm, str);
  }
  default:
  case OF_BYTES: {
    val = CsMakeByteVector(vm, rq->data.head(), (int_t)rq->data.size());
    CsSetByteVectorType(val, CsMakeCString(vm, rq->data_content_type));
    CsSetByteVectorName(val, CsMakeCString(vm, rq->content_url));
    return val;
  }
  }
}

bool on_raw_data_loaded(xview *xv, html::pump::request *rq) { return false; }

// bool xview::on_data_loaded(html::pump::request* rq)
//{
//  ++rq->ready;
//  if( rq->data_type != html::DATA_RAW_DATA )
//    return super::on_data_loaded(rq);

//  // handling async data request
//  handle<xrequest> xrq = (xrequest*)rq;
//  if( !xrq->callback_obj.is_set() || !xrq->dst)
//  {
//    //assert(false);
//    //return false; // "sync" request
//    return false;
//  }

//  if(!xrq->dst->pview())
//  {
//    assert(false);
//    return false;
//  }

//  pvalue val(vm); val = parse_response(vm, xrq );
//  pvalue obj(vm); obj = element_object(vm, xrq->dst);

//  TRY
//  {
//    auto_scope as(vm,CsMethodNamespace(xrq->callback_obj.val));
//    CsSendMessage(vm, obj, xrq->callback_obj, 2, val.val,
//    CsMakeInteger(xrq->status));
//  }
//  CATCH_ERROR(e)
//  {
//    if(e.number == 0) // close() called
//    {
//      return true;
//    }
//    else
//      tis::CsHandleUnhandledError( vm );
//  }
//  return super::on_data_loaded(rq);
//}

bool xview::invoke_event_function(html::hdocument pd, html::event &evt, html::element *b,
                                  value iobj, value selector) {

#ifdef DEBUG
#endif // DEBUG


  assert(vm && pd);

  if (!vm || vm->collecting_garbage || !pd || !pd->ns) return false;

  auto_pevt holder(this, &evt);

  evt_group = selector;
  bool res  = false;

  value obj  = iobj;
  value tobj = iobj;
  value m = 0;

  PROTECT_VM(obj, tobj, m);

  auto_state<xview *> _(vm->_current_view, this);
  auto_scope as(vm, pd->ns);

  if (selector == UNDEFINED_VALUE || !CsGetProperty1(vm, tobj, selector, &m)) {
    // dispatch* pd = CsGetDispatch(tobj);
    // return false;
    goto TRY_SUBSCRIPTIONS; // no such method...
  }

  if (CsMethodP(m) && event_obj.val) {
    TRY {
      // dispatch* disp = CsGetDispatch(eventObject);
      value r = CsCallMethod(vm, obj, m, tobj, 1, event_obj.val);
      if (!res) res = r == TRUE_VALUE;
    }
    CATCH_ERROR(e) {
      if (e.number == 0) // close() called
      {
        return true;
      } else {
        tis::CsHandleUnhandledError(vm);
        return false;
      }
    }
  }

  // second step, subscribed events invocation:
TRY_SUBSCRIPTIONS : {

#ifdef _DEBUG
  if (evt.event_group() == HANDLE_MOUSE && 
    evt.cmd == (html::MOUSE_DOWN | html::EVENT_SINKING) && 
    b->is_id_test() )
    b = b;
#endif

  auto exec_event_cb = [&](html::element *_this, html::element *_that,
                           value cb) -> bool {
    if (!event_obj.val) return false;
    TRY {
      value thisObj = 0, thatObj = 0;
      PROTECT_VM(thisObj, thatObj,cb);
      thisObj = element_object(this->vm, _this);
      thatObj = element_object(this->vm, _that);
      value r = CsCallMethod(this->vm, thisObj, cb, thisObj, 2, event_obj.val, thatObj);
      return r == TRUE_VALUE;
    }
    CATCH_ERROR(e) {
      if (e.number == 0) // close() called
        return true;
      else {
        tis::CsHandleUnhandledError(vm);
        return false;
      }
    }
    return false;
  };

  /*auto oneach = [&](html::subscription* subs, html::element* principal) ->
  bool { if( !subs->fcn ) return false; return
  exec_event_cb(principal,b,subs->fcn);
  };*/

  auto handle_matching_events = [&](value obj, bool handlingClass) {
    each_event_handler gen(this->vm, obj);

    for (value fun, v_name, v_ns, v_selector;
         gen(fun, v_name, v_ns, v_selector);) {
      if (CsIntegerP(v_ns)) {
        uint egroup = (uint)CsIntegerValue(v_ns);
        if (CsIntegerP(v_name)) {
          uint ecmd = (uint)CsIntegerValue(v_name);
          if (evt.event_group() == egroup && evt.cmd == ecmd &&
              exec_event_cb(b, evt.target, fun))
            res = true;
        } else {
          if (evt.event_group() == egroup && exec_event_cb(b, evt.target, fun))
            res = true;
        }
      } else {
        html::subscription sub(value_to_wchars(v_name));
        sub.event_ns = value_to_wchars(v_ns);
        sub.event_selector = value_to_wchars(v_selector);
        html::helement principal;
        if (evt.match(*this, sub, principal,b)) {
          if (handlingClass) {
            html::element *ethis = b;
            html::element *ethat = evt.target;
            if (sub.event_selector.is_defined()) {
              ethat = principal;
              ethis = b;
            }
            if (ethis && /*ethat &&*/ exec_event_cb(ethis, ethat, fun))
              res = true;
          } else {
            html::element *ethis = b;
            html::element *ethat = evt.target;
            if (sub.event_selector.is_defined()) {
              ethis = principal;
              ethat = principal;
            }
            if (ethis && /*ethat &&*/ exec_event_cb(ethis, ethat, fun))
              res = true;
          }
        }
      }
    }
  };

  /*if(b->each_handler(*this,evt,oneach))
    res = true;
  else*/
  {
    // element itself and class subscriptions
    for (tobj = obj; tobj && CsObjectOrMethodP(tobj);
         tobj = CsObjectClass(tobj)) {
      handle_matching_events(tobj, tobj != obj);
    }
    if (b->is_document()) {
      // for the document: look for event functions defined in its namespace as
      // free event-functions:
      if (pd && CsObjectOrMethodP(pd->ns))
        handle_matching_events(pd->ns, false);
    }
  }
}
  return res;
}

bool xview::send_notification(html::hdocument pd, value obj, value selector, value &result) {
  if (!vm || vm->collecting_garbage) return false;
  value tobj = obj;
  value m    = 0;
  PROTECT_VM(obj, tobj, m, selector);

  auto_state<xview *> _1(vm->_current_view, this);
  handle<VM>          _2(vm);

  auto_scope as(vm, pd->ns);

  while (!CsGetProperty1(vm, tobj, selector, &m)) {
    // dispatch* pd = CsGetDispatch(tobj);
    if (!CsObjectOrMethodP(tobj) || (tobj = CsObjectClass(tobj)) == 0) {
      return false; // no such method...
    }
    // CsDumpObject(vm, obj);
  }

  if (CsMethodP(m)) {
    auto_scope as(vm, pd->ns);
    TRY {
      value r = CsCallMethod(vm, obj, m, tobj, 0);
      result  = r;
      return true;
    }
    CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(vm);
      return false;
    }
  }
  return false;
}

bool xview::send_notification(html::hdocument pd, value obj, value selector, value p,
                              value &result) {
  if (!vm || vm->collecting_garbage) return false;

  value tobj = obj;
  PROTECT_VM(obj, tobj, p, selector);

  auto_state<xview *> _(vm->_current_view, this);

  value m = 0;
  while (!CsGetProperty1(vm, tobj, selector, &m)) {
    if (!CsDerivedFromObjectP(tobj))
      return false; // no such method...
    tobj = CsObjectClass(tobj);
    if (!tobj) 
      return false; // no such method...
    if (!CsDerivedFromObjectP(tobj))
      return false; // no such method...
    // CsDumpObject(vm, obj);
  }

  if (CsMethodP(m)) {
    auto_scope as(vm, pd->ns);
    TRY {
      value r = CsCallMethod(vm, obj, m, tobj, 1, p);
      result  = r;
      return true;
    }
    CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(vm);
      return false;
    }
  }
  return false;
}

bool xview::send_notification(html::hdocument pd, value obj, value selector, value p1,
                              value p2, value &result) {
  if (!vm || vm->collecting_garbage) return false;

  value tobj = obj;
  PROTECT_VM(obj, tobj, p1, p2, selector);

  value m = 0;
  while (!CsGetProperty1(vm, tobj, selector, &m)) {
    // dispatch* pd = CsGetDispatch(tobj);
    if (!CsObjectOrMethodP(tobj) || (tobj = CsObjectClass(tobj)) == 0) {
      return false; // no such method...
    }
    // CsDumpObject(vm, obj);
  }

  if (CsMethodP(m)) {
    auto_scope as(vm, pd->ns);
    TRY {
      result = CsCallMethod(vm, obj, m, tobj, 2, p1, p2);
      return true;
    }
    CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(vm);
      return false;
    }
  }
  return false;
}


bool xview::invoke_event_function(html::hdocument pd, value obj, value selector,
                                  value arg) {
  if (!vm || vm->collecting_garbage) return false;

  evt_group = selector;

  bool res = false;

  value next = obj;
  PROTECT_VM(obj, next, arg);

  auto_state<xview *> _(vm->_current_view, this);

  value m = 0;

  if (CsMethodP(selector))
    m = selector;
  else if (!CsGetProperty1(vm, next, selector, &m))
    return false; // no such method...

  if (CsMethodP(m)) {
    auto_scope as(vm, pd->ns);
    TRY {
      value r = arg ? CsCallMethod(vm, obj, m, next, 1, arg)
                    : CsCallMethod(vm, obj, m, next, 0);
      res = r == TRUE_VALUE;
    }
    CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(vm);
    }
  }
  return res;
}

#if 0
  bool xview::invoke_function(const string& name, value* argv, int argc, value& ret)
  {
    /*evt_group = selector;

    bool res = false;

    value tobj = obj;
    value m = 0;
    while(!CsGetProperty1(vm,tobj,selector,&m))
    {
        dispatch* pd = CsGetDispatch(tobj);
        if (!CsObjectOrMethodP(tobj) || (tobj = CsObjectClass(tobj)) == nullptr)
        {
            pevt = 0;
            return false; // no such method...
        }
    }

    if(CsMethodP(m))
    {
      auto_scope as(vm,self);
      TRY
      {
        value r = CsSendMessage(vm,obj,m,0);
        res = r == TRUE_VALUE;
      }
      CATCH_ERROR(e)
      {
        e;
        CsDisplay(vm,vm->val,vm->standardError);
      }
    }
    pevt = 0;
    return res;
    */
    return false;
  }
#endif

bool xview::call_behavior_method(html::method_params *p) {
  return super::call_behavior_method(p);
}
bool xview::call_behavior_method(html::element *b, html::method_params *p) {
  return super::call_behavior_method(b, p);
}

//static value value_sym = 0;

bool xview::get_element_value(html::helement b, tool::value &v, bool ignore_text_value) {
  b->get_style(*this);
  value               obj = element_object_nc(vm, b);
  auto_state<xview *> _(vm->_current_view, this);
  if (obj /*&& (b->c_style->prototype.is_defined() || b->c_style->handler_list.defined())*/ && !vm->collecting_garbage) {
    //if (!value_sym) value_sym = CsSymbolOf("value");
    value      val = 0;
    auto_scope as(vm, get_ns(b));
    TRY {
      // dispatch* disp = CsGetDispatch(eventObject);
      if (CsGetProperty1(vm, obj, VALUE_SYM, &val)) {
        v = value_to_value(vm, val);
        return true;
      }
    }
    CATCH_ERROR_NE { tis::CsHandleUnhandledError(vm); }
  }
  return super::get_element_value(b, v, ignore_text_value);
}
bool xview::set_element_value(html::helement b, const tool::value &v,bool ignore_text_value) {
#ifdef _DEBUG
  b->dbg_report("set_element_value");
#endif
  b->get_style(*this);
  value obj = element_object_nc(vm, b);
  if (obj /*&& b->c_style->prototype.defined()*/ && !vm->collecting_garbage) {
    PROTECT_VM(obj);
    value val = 0;
    auto_scope as(vm, get_ns(b));
    TRY {
      val = value_to_value(vm, v);
      if (CsSetProperty(vm, obj, VALUE_SYM, val)) {
        return true;
      }
    }
    CATCH_ERROR_NE { tis::CsHandleUnhandledError(vm); }
  }
  return super::set_element_value(b, v, ignore_text_value);
}

bool xview::get_element_property(html::helement b, wchars name, ustring &v) {
  value obj = element_object_nc(vm, b);
  if (obj) {
    value val;
    if (CsGetProperty(vm, obj, CsSymbolOf(name), &val)) {
      v = value_to_string(val);
      return true;
    }
  }
  return super::get_element_property(b, name, v);
}

bool xview::on_element_event(html::element *b, html::event_mouse &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onMouse)))
      return true;
    if (evt.cmd == html::MOUSE_CLICK && (obj = element_object_nc(vm, b))) {
        value r = FALSE_VALUE;
        send_notification(pd, obj, get_sym_by_id(id_onClick), r);
        if (r == TRUE_VALUE) return true;
    }
  }
  return super::on_element_event(b, evt);
}

bool xview::on_element_event(html::element *b, html::event_key &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onKey)))
      return true;
  }
  return super::on_element_event(b, evt);
}

bool xview::on_element_event(html::element *b, html::event_exchange &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onExchange)))
      return true;
  }
  return super::on_element_event(b, evt);
}

bool xview::on_element_event(html::element *b, html::event_gesture &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onGesture)))
      return true;
  }
  return super::on_element_event(b, evt);
}

bool xview::on_element_event(html::element *b, html::event_focus &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onFocus)))
      return true;
  }
  return super::on_element_event(b, evt);
}

bool xview::on_element_event(html::element *b, html::event_scroll &evt) {
  value obj = element_object_nc(vm, b);
  html::hdocument pd = b->doc();
  if (obj && pd) {
    if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onScroll)))
      return true;
  }
  return super::on_element_event(b, evt);
}

bool xview::on(html::view &v, html::element *self, html::event_behavior &evt) {
  if (evt.cmd == html::DOCUMENT_CREATED && evt.target == doc())
    this->fire_event(WCHARS("newdocument"));
  return super::on(v, self, evt);
}

bool xview::on_element_event(html::element *b, html::event_behavior &evt) {
  // if( evt.cmd == html::DOCUMENT_CREATED ||
  //    evt.cmd == (html::DOCUMENT_CREATED | html::EVENT_SINKING))
  //  return super::on_element_event(b,evt);

#ifdef _DEBUG
  if (evt.cmd == html::CUSTOM)
    b = b;
  else if (evt.cmd == (html::BUTTON_CLICK | html::EVENT_SINKING))
    b = b;
  else if (evt.cmd == (html::BUTTON_CLICK))
    b = b;
#endif // _DEBUG


  value obj = element_object_nc(vm, b);
  if (!obj) return super::on_element_event(b, evt);

  html::hdocument pd = b->doc();

  if (!pd) 
    return super::on_element_event(b, evt);

  PROTECT_VM(obj);

  if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onControlEvent)))
    return true;

  value r = FALSE_VALUE;
  if (b == evt.target) switch (evt.cmd) {
    case html::HYPERLINK_CLICK:
    case html::MENU_ITEM_CLICK:
    case html::BUTTON_CLICK:
      if (send_notification(pd, obj, get_sym_by_id(id_onClick), r))
        return r == TRUE_VALUE;
      break;
    case html::EDIT_VALUE_CHANGED:
    //case html::SELECT_SELECTION_CHANGED:
    case html::SELECT_VALUE_CHANGED:
    case html::BUTTON_STATE_CHANGED:
    case html::FORM_VALUE_CHANGED:
      if (send_notification(pd, obj, get_sym_by_id(id_onValueChanged), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::POPUP_REQUEST:
      if (send_notification(pd, obj, get_sym_by_id(id_onPopupCreated), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::POPUP_READY:
      if (send_notification(pd, obj, get_sym_by_id(id_onPopupShown), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::POPUP_DISMISSED:
      if (send_notification(pd, obj, get_sym_by_id(id_onPopupClosed), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::FORM_SUBMIT:
      if (send_notification(pd, obj, CsSymbolOf("onSubmit"), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::FORM_RESET:
      if (send_notification(pd, obj, CsSymbolOf("onReset"), element_object(vm, evt.source), r))
        return r == TRUE_VALUE;
      break;
    case html::ANIMATION:
#ifdef _DEBUG
      if (evt.reason == 0)
        evt.reason = evt.reason;
#endif
      if (send_notification(pd, obj, evt.reason ? get_sym_by_id(id_onAnimationStart) : get_sym_by_id(id_onAnimationEnd), r))
        return r == TRUE_VALUE;
      break;
    }
  return super::on_element_event(b, evt);
}
bool xview::on_element_event(html::element *b, html::event_command &evt) {
  // if( b == evt.target ) {
  value obj = element_object_nc(vm, b);
  if (!obj) 
    return super::on_element_event(b, evt);

  html::hdocument pd = b->doc();
  if (!pd) 
    return super::on_element_event(b, evt);

  // PROTECT_VM(obj,ns);
  //#ifdef _DEBUG
  //      if (b->tag == html::tag::T_HTMLAREA &&
  //      evt.command.like(W("document*")) && evt.cmd ==
  //      html::event_command::CHECK)
  //        b = b;
  //#endif

  if (invoke_event_function(pd, evt, b, obj, get_sym_by_id(id_onCommand)))
    return true;
  //}
  return super::on_element_event(b, evt);
}
bool xview::on_element_timer(html::element *b, html::timer_def &td)
{
  if (td.kind == html::SCRIPT_TIMER) {
    auto_state<xview *> _(vm->_current_view, this);
    value               obj = element_object_nc(vm, b);
    html::hdocument     pd = b->doc();

    if (pd && td.id && obj && tis::CsMethodP(td.id)) {
      PROTECT_VM(td.id);
      bool       res = false;
      auto_scope as(vm, pd->ns);
      TRY {
        value r = CsSendMessage(vm, obj, td.id, 0);
        res     = r == TRUE_VALUE;
      }
      CATCH_ERROR(e) {
        if (e.number == 0) // close() called
          ;                // throw e;
        else
          tis::CsHandleUnhandledError(vm);
      }
      return res;
    }
  }
  return super::on_element_timer(b, td);
}

bool xview::on_element_draw_background(html::graphics *pg, html::element *el,
                                       point pos) {
  value           sym_draw = get_sym_by_id(id_paintBackground);
  html::hdocument pd = el->doc();
  if (!pd)
    return false;

  gool::aa_mode _0(pg);
  gool::state   _1(pg);
  tool::auto_state<html::graphics *> _2(drawing_surface, pg);
  pg->translate(pos);
  auto_graphics gfx(vm, pg, el);

  value obj = element_object_nc(vm, el);
  if (!obj) {
#ifdef _DEBUG
    el->dbg_report("painting wrong");
#endif

    //bool cn = el->is_connected();
    //this->on_style_resolved(el, el->p_style);
    assert(false);
    el->flags.script_draws_background = 0;
    return false;
  }
  value result = 0;
  return send_notification(pd, obj, sym_draw, gfx.graphics_obj, result) && result == TRUE_VALUE;
}
bool xview::on_element_draw_foreground(html::graphics *pg, html::element *el, point pos) {
  html::hdocument pd = el->doc();
  if (!pd)
    return false;

  value         sym_draw = get_sym_by_id(id_paintForeground);
  gool::aa_mode _0(pg);
  gool::state   _1(pg);
  pg->translate(pos);
  //tool::auto_state<html::element*>   _2(html::element::drawing_element, el);
  tool::auto_state<html::graphics *> _3(drawing_surface, pg);
  auto_graphics                      gfx(vm, pg, el);
  // value gfx = get_graphics(pg,el);
  value obj = element_object_nc(vm, el);
  if (!obj) {
    assert(false);
    el->flags.script_draws_background = 0;
    return false;
  }
  value result = 0;
  return send_notification(pd, obj, sym_draw, gfx.graphics_obj, result) && result == TRUE_VALUE;
}

bool xview::on_element_draw_outline(html::graphics *pg, html::element *el,
                                    point pos) {
  html::hdocument pd = el->doc();
  if (!pd)
    return false;

  value         sym_draw = get_sym_by_id(id_paintOutline);
  gool::aa_mode _0(pg);
  gool::state   _1(pg);
  pg->translate(pos);
  //tool::auto_state<html::element*>   _2(html::element::drawing_element, el);
  tool::auto_state<html::graphics *> _3(drawing_surface, pg);
  auto_graphics                      gfx(vm, pg, el);
  // value gfx = get_graphics(pg,el);
  value obj = element_object_nc(vm, el);
  if (!obj) {
    assert(false);
    el->flags.script_draws_outline = 0;
    return false;
  }
  value result = 0;
  return send_notification(pd, obj, sym_draw, gfx.graphics_obj, result) && result == TRUE_VALUE;
}

bool xview::on_element_draw_content(html::graphics *pg, html::element *el,
                                    point pos) {
  html::hdocument pd = el->doc();
  if (!pd)
    return false;

  value         sym_draw = get_sym_by_id(id_paintContent);
  gool::aa_mode _0(pg);
  gool::state   _1(pg);
  pg->translate(pos);
  //tool::auto_state<html::element*>   _2(html::element::drawing_element, el);
  tool::auto_state<html::graphics *> _3(drawing_surface, pg);
  // value gfx = get_graphics(pg,el);
  auto_graphics gfx(vm, pg, el);
  value         obj = element_object_nc(vm, el);
  if (!obj) {
    assert(false);
    el->flags.script_draws_background = 0;
    return false;
  }
  value result = 0;
  return send_notification(pd, obj, sym_draw, gfx.graphics_obj, result) && result == TRUE_VALUE;
}

bool xview::hit_test_element(html::element *el, point zpt) {

  if (!el->flags.script_has_is_point_inside) return true;

  html::hdocument pd = el->doc();
  if (!pd)
    return false;

  value sym_isPointInside = get_sym_by_id(id_isPointInside);
  value obj               = element_object_nc(vm, el);
  if (!obj) {
    assert(false);
    el->flags.script_has_is_point_inside = 0;
    return false;
  }
  value result = 0;
  return send_notification(pd, obj, sym_isPointInside, CsMakeInteger(zpt.x),CsMakeInteger(zpt.y), result) && result == TRUE_VALUE;
}

void xview::dispatch_posted_event(handle<html::posted_event> pe) {
  if (pe->cbf && CsMethodP(pe->cbf) && vm) {
    auto_state<xview *> _(vm->_current_view, this);
    value      ns = pe->target ? get_ns(pe->target) : vm->globalScope.globals;
    auto_scope as(vm, ns);
    TRY {
      value elobj = pe->target ? element_object(vm, pe->target) : ns;
      CsSendMessage(vm, elobj, pe->cbf, 0);
    }
    CATCH_ERROR(e) {
      if (e.number == 0) // close() called
        return;
      else
        tis::CsHandleUnhandledError(vm);
    }
  } else
    super::dispatch_posted_event(pe);
}

bool xview::send_behavior_event(html::event_behavior &evt) {
  return super::send_behavior_event(evt);
}

void xview::on_element_behavior_changed(html::element *b,
                                        const string & old_names,
                                        const string & new_names) {
  super::on_element_behavior_changed(b, old_names, new_names);
  value obj = element_object_nc(vm, const_cast<html::element *>(b));
  if (obj == 0) return;
  CsSetDispatch(obj, vm->elementDispatch);
}

bool xview::may_have_on_size_handler(html::element *b) {
  return super::may_have_on_size_handler(b) || b->flags.has_on_size_script;
}

void xview::on_element_client_size_changed(html::element *b) {
  super::on_element_client_size_changed(b);

  if (b->flags.has_on_size_script) {
    html::hdocument pd = b->doc();
    if (!pd)
      return;
    value obj = element_object_nc(vm, b);
    if (obj) {
      invoke_event_function(pd, obj, get_sym_by_id(id_onSize));
    }
  }
}

void xview::on_element_visibility_changed(html::element *b, bool on_off) {
  super::on_element_visibility_changed(b, on_off);

  if (b->flags.has_on_visibility_changed_script) {
    html::hdocument pd = b->doc();
    if (!pd)
      return;

    value obj = element_object_nc(vm, b);

    if (obj) {
      invoke_event_function(pd, obj,
                            get_sym_by_id(id_onVisibilityChanged),
                            on_off ? TRUE_VALUE : FALSE_VALUE);
    }
  }
}

void xview::on_size(gool::size sz) {
  super::on_size(sz);
  // value view_o;
  // if( doc() && doc()->ns && tis::CsGetProperty(vm,
  // doc()->ns,view_sym(),&view_o))
  if (view_obj.is_set() && doc()) {
    // CsDumpObject(vm,view_obj);
    invoke_event_function(doc(), view_obj, get_sym_by_id(id_onSize));
    this->fire_event(WCHARS("size"));
  }
}
void xview::on_move() {
  super::on_move();
  if (view_obj.is_set() && doc()) {
    invoke_event_function(doc(), view_obj, get_sym_by_id(id_onMove));
    this->fire_event(WCHARS("move"));
  }
}

bool xview::on_move_request(gool::rect &rc) {
  if (view_obj.is_set()) {
    tis::value vrect = 0;
    PROTECT_VM(vrect);
    auto rect_creator = [&rc, this, &vrect]() -> tis::value {
      vrect = CsMakeObject(this->vm, UNDEFINED_VALUE);
      CsSetProperty(vm, vrect, "x", CsMakeInteger(rc.left()));
      CsSetProperty(vm, vrect, "y", CsMakeInteger(rc.top()));
      CsSetProperty(vm, vrect, "width", CsMakeInteger(rc.width()));
      CsSetProperty(vm, vrect, "height", CsMakeInteger(rc.height()));
      return vrect;
    };

    if (this->fire_event(WCHARS("moving"), rect_creator) && vrect) {
      gool::point pt = rc.start();
      gool::size  sz = rc.size();
      CsGetProperty(vm, vrect, "x", pt.x);
      CsGetProperty(vm, vrect, "y", pt.y);
      CsGetProperty(vm, vrect, "width", sz.x);
      CsGetProperty(vm, vrect, "height", sz.y);
      rc = rect(pt, sz);
      return true;
    }
  }
  return false;
}

bool xview::on_size_request(int side_id, gool::rect &rc) {
  bool r = super::on_size_request(side_id, rc);
  if (view_obj.is_set()) {
    tis::value vrect = 0;
    PROTECT_VM(vrect);
    auto rect_creator = [&rc, this, &vrect, side_id]() -> tis::value {
      vrect = CsMakeObject(this->vm, UNDEFINED_VALUE);
      CsSetProperty(vm, vrect, "x", CsMakeInteger(rc.left()));
      CsSetProperty(vm, vrect, "y", CsMakeInteger(rc.top()));
      CsSetProperty(vm, vrect, "width", CsMakeInteger(rc.width()));
      CsSetProperty(vm, vrect, "height", CsMakeInteger(rc.height()));
      CsSetProperty(vm, vrect, "side", CsMakeInteger(side_id));
      return vrect;
    };

    if (this->fire_event(WCHARS("sizing"), rect_creator) && vrect) {
      gool::point pt = rc.start();
      gool::size  sz = rc.size();
      CsGetProperty(vm, vrect, "x", pt.x);
      CsGetProperty(vm, vrect, "y", pt.y);
      CsGetProperty(vm, vrect, "width", sz.x);
      CsGetProperty(vm, vrect, "height", sz.y);
      rc = rect(pt, sz);
      return r;
    }
  }
  return r;
}

void xview::on_start_ui_replacement(tristate_v sizing) {
  super::on_start_ui_replacement(sizing);
  if (view_obj.is_set()) this->fire_event(WCHARS("replacement-start"));
}

void xview::on_end_ui_replacement(tristate_v sizing) {
  super::on_end_ui_replacement(sizing);
  if (view_obj.is_set()) this->fire_event(WCHARS("replacement-end"));
}

void xview::on_dpi_changed(size dpi, rect proposed_window_rect) {
  super::on_dpi_changed(dpi, proposed_window_rect);
  if (doc() && view_obj.is_set()) {
    invoke_event_function(doc(), view_obj,
                          get_sym_by_id(id_onResolutionChanged));
    this->fire_event(WCHARS("resolutionchange"));
  }
}

bool xview::on_activate(html::ACTIVATE_MODE am) {
  super::on_activate(am);
  if (view_obj.is_set() && doc()) {
    value p = FALSE_VALUE;
    switch (am) {
      case html::ACTIVATED: p = TRUE_VALUE; break;
      case html::MOUSE_ACTIVATED: p = CsSymbolOf("by-mouse"); break;
      default: break;
    }
    invoke_event_function(doc(), view_obj, CsSymbolOf("activated"), p);
    this->fire_event(WCHARS("activate"), [&p]() { return p; });
  }
  return true;
}

void xview::on_state_changed() {
  super::on_state_changed();
  handle<xview> self = this;
  async([self]() -> bool {
    if (self->view_obj.is_set() && self->doc()) {
      element_object(self->vm, self->doc());
      self->invoke_event_function(self->doc(), self->view_obj, get_sym_by_id(id_onStateChanged));
      self->fire_event(WCHARS("statechange"));
    }
    return true;
  });
}

/*void xview::setup_window(html::document* d)
{
  if( d->parent )
    return;
  html::element* head = html::find_first(*this,d,WCHARS("head"), false);
  if(head)
  {
    html::element* title = html::find_first(*this,head,WCHARS("title"), false);
    if(title)
    {
      ustring t = title->get_text(*this);
      set_window_title(t);
    }
  }
}*/

// tool::cabinet xview::cache; // cache of all zip contents

bool xview::on_element_data_arrived(html::element *b, pump::request *prq) {
  bytes  dt  = prq->data();
  string url = prq->url;
  // on_file_load(dt, url);
  if (dt.start != prq->data.head()) {
    prq->data.clear();
    prq->data.push(dt);
    prq->url = url;
  }
  if (super::on_element_data_arrived(b, prq)) {
    // was: "THIS PRODUCES SOME STRANGE EFFECT while changing locales"
    // Found it. Here goes esoteric russian knowledge:
    //   in debug mode CsCollectGarbage spills GC statistic in console
    //   window. And any output in console during window creation kills
    //   WM_INPUTLANGCHANGEREQUEST for that window. Forever. Surprise, surprise!
    //
    //   But don't ask me how deep in that shit I've found this solution.
    // wrong here: CsCollectGarbage(vm);
    return true;
  }
  return false;
}

value CsRequestObject(xvm *c, html::request *prq);

void xview::on_data_request_notify(html::element *b, pump::request *rq) {
  // if (rq->consumed)
  //  return;
  value obj = b ? element_object_nc(vm, b) : view_obj.val;
  if (!obj) return;

  value next = obj;
  value m    = 0;
  PROTECT_VM(obj, next, m);

  static value selector = CsSymbolOf(W("onRequest"));

  if (!CsGetProperty1(vm, next, selector, &m)) return; // no such method...
  if (!CsMethodP(m)) return;

  html::hdocument pd = b ? b->doc() : doc();
  if (!pd) return;

  auto_state<xview *> _(vm->_current_view, this);

  auto_scope as(vm, pd->ns);
  TRY {
    value robj = CsRequestObject(vm, rq);
    CsCallMethod(vm, obj, m, next, 1, robj);
  }
  CATCH_ERROR(e) {
    if (e.number != 0) // close() called
      tis::CsHandleUnhandledError(vm);
  }
  return ;
}

void xview::on_data_arrived_notify(html::element *dst, pump::request *prq) {
  html::helement        b  = dst;
  handle<html::request> rq = prq;

  auto deliver_notification = [b, rq, this]() -> bool {

    value obj = b ? element_object_nc(vm, b) : view_obj.val;
    if (!obj) return true;

    value next = obj;
    value m    = 0;
    PROTECT_VM(obj, next, m);

    static value selector = CsSymbolOf(W("onRequestResponse"));

    if (!CsGetProperty1(vm, next, selector, &m))
      return true; // no such method...
    if (!CsMethodP(m)) return true;

    html::hdocument pd = b ? b->doc() : doc();
    if (!pd) return true;

    auto_state<xview *> _(vm->_current_view, this);

    auto_scope as(vm, CsMethodNamespace(m));
    TRY {
      value robj   = CsRequestObject(vm, rq);
      value r      = CsCallMethod(vm, obj, m, next, 1, robj);
      rq->consumed = r == TRUE_VALUE;
    }
    CATCH_ERROR(e) {
      if (e.number != 0) // close() called
        tis::CsHandleUnhandledError(vm);
    }
    return true; // complete
  };
  //if (b) // that should not be async - script should be able to consume it
  //  async(deliver_notification);
  //else
  deliver_notification();

}

bool xview::fire_event(wchars name, function<tis::value()> param) {
  if (view_obj)
    return CsEventObjectFire(vm, view_obj, CsSymbolOf(name), param) ==
           TRUE_VALUE;
  return false;
}

bool xview::trayicon_notify(point screen_pt, uint buttons, html::mouse_events evt) {
  auto genparam = [&]() -> tis::value {
    value p = CsMakeObject(vm, UNDEFINED_VALUE);
    PROTECT_VM(p);
    CsSetProperty(vm, p, "x", screen_pt.x);
    CsSetProperty(vm, p, "y", screen_pt.y);
    CsSetProperty(vm, p, "buttons", int(buttons));
    return p;
  };

  ustring event_name;
  switch (evt) {
    //case html::MOUSE_DOWN: event_name = WCHARS("trayicon-mousedown"); break;
    case html::MOUSE_CLICK: event_name = WCHARS("trayicon-click"); break;
    //case html::MOUSE_MOVE: event_name = WCHARS("trayicon-mousemove"); break;
    default: return false;
  }
  return fire_event(event_name(), genparam);
}

  enum OUTBOUND_PROTOCOL {
    PROTOCOL_GET = 0,
    PROTOCOL_POST,
    PROTOCOL_POST_DATA,
    PROTOCOL_PUT_DATA,
    PROTOCOL_GET_JSON,
    PROTOCOL_POST_JSON,
    PROTOCOL_PUT_JSON,
    PROTOCOL_DELETE_JSON,
    PROTOCOL_DELETE,
    PROTOCOL_size,
  };

  static value getProtocolSymbol(OUTBOUND_PROTOCOL protocol) {
    static value syms[PROTOCOL_size] = {0};
    if (!syms[0]) {
      syms[0] = CsSymbolOf("get");
      syms[1] = CsSymbolOf("post");
      syms[2] = CsSymbolOf("post-data");
      syms[3] = CsSymbolOf("put-data");
      syms[4] = CsSymbolOf("json");
      syms[5] = CsSymbolOf("post-json");
      syms[6] = CsSymbolOf("put-json");
      syms[7] = CsSymbolOf("delete-json");
      syms[8] = CsSymbolOf("delete");
    }
    return syms[protocol];
  }

// extern value parse_response(xvm* vm, html::request* rq );

  void CSF_handle_headers(xvm *c, html::request *xrq, value headers) {
    if (headers && CsObjectP(headers)) {
      each_property gen(c, headers);
      // ustring h;
      for (value key, val; gen(key, val);) {
        string name = u8::cvt(value_to_string(key));
        string sval = u8::cvt(value_to_string(CsToString(c, val)));
        // h += ustring::format(W("%s:%s\r\n"),name.c_str(),sval.c_str());
        xrq->rq_headers[name] = sval;
      }
      // xrq->headers = h.utf8();
    }
  }

  static void CSF_handle_params(xvm *c, html::request *xrq, value params,
                                DATA_PROTOCOL protocol) {
    if (!params) return;
    if (!CsObjectP(params)) return;

    switch (protocol) {
    case PROTOCOL_JSON: {
      string_stream data;
      CsPrintJsonData(c, params, &data, false);
      tool::bytes bs = data.buf();
      bs.prune(3); // remove utf-8 BOM
      xrq->data.push(bs);
      xrq->data_content_type = "application/json;charset=utf-8";
    } break;

    case PROTOCOL_MULTIPART: {
      html::pump::multipart_composer mpc(xrq);
      each_property                  gen(c, params);
      for (value key, val; gen(key, val);) {
        string name = u8::cvt(value_to_string(key));
        if (CsByteVectorP(val)) {
          string filename = name;
          string mimetype;
          value  tn = CsByteVectorName(val);
          if (CsStringP(tn)) filename = url::escape(value_to_string(tn));
          value tt = CsByteVectorType(val);
          if (CsStringP(tt)) mimetype = u8::cvt(value_to_string(tt));
          mpc.add(name, CsByteVectorBytes(val), filename, mimetype);
        } else {
          ustring sval = value_to_string(CsToString(c, val));
          mpc.add(name, sval);
        }
      }
    } break;

    default: {
      string_stream name;
      string_stream data;

      each_property gen(c, params);
      for (value key, val; gen(key, val);) {
        name.clear();
        data.clear();

        CsDisplay(c, key, &name);
        CsDisplay(c, val, &data);

        html::pump::param param;
        param.name  = name.to_ustring();
        param.value = data.to_ustring();
        xrq->rq_params.push(param);
      }
    } break;
    }
  }

  value CSF_request(xvm *c) {

    if ((c->features & FEATURE_SOCKET_IO) == 0) {
      CsThrowKnownError(c, CsErrNotAllowed, "SOCKET IO");
    }

    value  obj;
    wchars url;

    value  callback = 0;
    value  params   = 0;
    value  protocol = 0;
    value  headers  = 0;

    CsParseArguments(c, "V=*VV=S#|V=|V=", &obj, c->elementDispatch, &callback,
                     &protocol, &CsSymbolDispatch, &url.start, &url.length, &params,
                     &CsObjectDispatch, &headers, &CsObjectDispatch);

    value data = UNDEFINED_VALUE;
  
    PROTECT(data,params,protocol,headers);

    html::helement self = element_ptr(c, obj);
    if (!self) return FALSE_VALUE;
    handle<html::view> pv = self->pview();
    if (!pv) return FALSE_VALUE;

    // bool protocol_json = false;
    // bool protocol_data = false; // multipart/form-data

    DATA_PROTOCOL data_protocol = PROTOCOL_BASIC;

    string uri = html::combine_url(self->doc()->uri().src, url::escape(url));

    pvalue cb;

    if (callback && CsMethodP(callback))
      cb.pin(c, callback);

    handle<html::request> xrq = new html::request(uri, html::DATA_RAW_DATA);

    auto on_data_ready_async = [self, cb, data_protocol](html::request *xrq) -> bool {
      value obj = element_object((xvm*)cb.pvm, self);
      value val = 0;
      PROTECT_P(cb.pvm,obj, val);
      val = parse_response((xvm*)cb.pvm, xrq, data_protocol == PROTOCOL_JSON ? OF_JSON : OF_AUTO);

      TRY {
        auto_scope as(cb.pvm, get_ns(self));
        CsSendMessage(cb.pvm, obj, cb.val, 2, val, CsMakeInteger(xrq->status));
      }
      CATCH_ERROR(e) {
        if (e.number == 0) // close() called
          return true;
        else
          tis::CsHandleUnhandledError(cb.pvm);
      }
      return true;
    };
    auto on_data_ready_sync = [c, &data, data_protocol](html::request *xrq) -> bool {
      data = parse_response(c, xrq, data_protocol == PROTOCOL_JSON ? OF_JSON : OF_AUTO);
      return true;
    };

    if (cb.is_set())
      xrq->add(on_data_ready_async);
    else
      xrq->add(on_data_ready_sync);

    CSF_handle_headers(c, xrq, headers);

    xrq->rq_type = html::RQ_GET;
    if (protocol == getProtocolSymbol(PROTOCOL_POST))
      xrq->rq_type = html::RQ_POST;
    else if (protocol == getProtocolSymbol(PROTOCOL_POST_DATA)) {
      xrq->rq_type  = html::RQ_POST;
      data_protocol = PROTOCOL_MULTIPART;
    } else if (protocol == getProtocolSymbol(PROTOCOL_PUT_DATA)) {
      xrq->rq_type  = html::RQ_PUT;
      data_protocol = PROTOCOL_MULTIPART;
    } else if (protocol == getProtocolSymbol(PROTOCOL_POST_JSON) ||
               protocol == getProtocolSymbol(PROTOCOL_GET_JSON)) {
      xrq->rq_type  = html::RQ_POST;
      data_protocol = PROTOCOL_JSON;
    } else if (protocol == getProtocolSymbol(PROTOCOL_PUT_JSON)) {
      xrq->rq_type  = html::RQ_PUT;
      data_protocol = PROTOCOL_JSON;
    } else if (protocol == getProtocolSymbol(PROTOCOL_DELETE_JSON)) {
      xrq->rq_type  = html::RQ_DELETE;
      data_protocol = PROTOCOL_JSON;
    } else if (protocol == getProtocolSymbol(PROTOCOL_DELETE)) {
      xrq->rq_type = html::RQ_DELETE;
    }

    xrq->data_type = html::DATA_RAW_DATA;
    xrq->dst       = self;

    CSF_handle_params(c, xrq, params, data_protocol);

    // if( xrq->callback_obj ) // async request
    // return pv->request_data(xrq)? TRUE_VALUE: FALSE_VALUE ;
    if (cb.is_set()) {
      return pv->request_data(xrq) ? TRUE_VALUE : FALSE_VALUE;
    }

    // sync requests
    unsigned timeout = (callback && CsIntegerP(callback)) ? CsIntegerValue(callback) : 10000;
    timeout += get_ticks();

    bool is_timeout = false;
    // value v = UNDEFINED_VALUE;

    if (!pv->request_data(xrq)) {
      for (;;) {
        if (xrq->ready_flag) {
          pv->dispatch_request(xrq);
          break;
        }

        if (get_ticks() > timeout) {
          is_timeout = true;
          goto TIMEOUT;
        }

        if (!self->pview()) goto NOTINTHEDOM;

        if (!pv->do_event(html::DO_EVENT_NOWAIT)) goto DOEVENT_FAILURE;
        // if(!html::do_ui_event())
        //  goto TIMEOUT;
      }
    }

    // v = parse_response(c,xrq);
    // self->state_off(*pv,html::S_BUSY);
    if (CsErrorP(data)) {
      // c->val = v;
      // c->vals = 1;
      // THROW_ERROR( CsIntegerValue(CsErrorNo(v)) );
      CS_RETURN2(c, CsMakeInteger(xrq->status), data);
    }
    CS_RETURN2(c, CsMakeInteger(xrq->status), data);

  TIMEOUT:
    self->state_off(*pv, html::S_BUSY);
  NOTINTHEDOM:
    if (is_timeout)
      CsThrowKnownError(c, CsErrGenericError,
                        "time is over to complete the request");
    else
      CsThrowKnownError(c, CsErrGenericError, "element is not in the DOM");
    // return UNDEFINED_VALUE;
  DOEVENT_FAILURE:
    CsThrowKnownError(c, CsErrGenericError, "window is not visible");
    return UNDEFINED_VALUE;
  }

  enum HTTP_COMMAND { HTTP_GET, HTTP_POST, HTTP_PUT, HTTP_DELETE };

  // struct progress_cb: public tool::functor {
  //}

  struct closure_block : public resource {
    pvalue obj;
    pvalue success;
    pvalue error;
    pvalue complete;
  };

  value CSF_http_request(xvm *c) {

    if ((c->features & FEATURE_SOCKET_IO) == 0) {
      CsThrowKnownError(c, CsErrNotAllowed, "SOCKET IO");
    }

    value vobj;

    pvalue obj(c);

    CsParseArguments(c, "V*V=", &vobj, &obj.val, &CsObjectDispatch);

    handle<xview>          pview;
    handle<html::document> self;

    pview = xview_ptr(c, vobj);
    if (pview) {
      self = pview->doc();
    } else {
      self = c->current_doc();
      if (!self) return UNDEFINED_VALUE;
      pview = static_cast<xview *>(self->pview());
      if (!pview) return UNDEFINED_VALUE;
    }

    ustring url;
    ustring target_path;

    pvalue success(c);
    pvalue error(c);
    pvalue complete(c);
    pvalue progress(c);
    value  httpc    = 0;
    value  params   = 0;
    value  protocol = 0;
    value  headers  = 0;
    value  output   = 0;
    // value     data = UNDEFINED_VALUE;

    if (!CsGetProperty(c, obj, "url", url))
      CsThrowKnownError(c, CsErrGenericError, "no url provided");

    string uri = html::combine_url(self->doc()->uri().src, url::escape(url));
    // if( !uri.like("http://*") && !uri.like("https://*") )
    //  CsThrowKnownError(c, CsErrGenericError, "url is neither http nor https");

    HTTP_COMMAND htc = HTTP_GET;

    ustring proxy_host;
    int_v   proxy_port;

    ustring username;
    ustring password;

    bool no_cache = false;
    bool try_sync = false;

    CsGetProperty(c, obj, "success", success.val, &CsMethodDispatch);
    CsGetProperty(c, obj, "error", error.val, &CsMethodDispatch);
    CsGetProperty(c, obj, "complete", complete.val, &CsMethodDispatch);
    CsGetProperty(c, obj, "progress", progress.val, &CsMethodDispatch);
    CsGetProperty(c, obj, "protocol", protocol, &CsSymbolDispatch);
    CsGetProperty(c, obj, "type", httpc, &CsSymbolDispatch);
    CsGetProperty(c, obj, "params", params, &CsObjectDispatch);
    CsGetProperty(c, obj, "headers", headers, &CsObjectDispatch);
    CsGetProperty(c, obj, "toFile", target_path);
    CsGetProperty(c, obj, "output", output, &CsSymbolDispatch);
    CsGetProperty(c, obj, "username", username);
    CsGetProperty(c, obj, "password", password);
    CsGetProperty(c, obj, "noCache", no_cache);
    CsGetProperty(c, obj, "trySync", try_sync);

    if ((c->features & FEATURE_FILE_IO) == 0 && target_path.length()) {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
    }
  
    REQUEST_REPRESENTATION_FORMAT output_format = OF_AUTO;

    if (output) {
      if (output == CsSymbolOf(WCHARS("string")))
        output_format = OF_STRING;
      else if (output == CsSymbolOf(WCHARS("bytes")))
        output_format = OF_BYTES;
      else if (output == CsSymbolOf(WCHARS("json")))
        output_format = OF_JSON;
      else if (output == CsSymbolOf(WCHARS("stream")))
        output_format = OF_STREAM;
    }
    // CsGetProperty(c,obj,"data",data,&CsObjectP);

    // if( target_path.length() )
    //    target_path = target_path;

    CsGetProperty(c, obj, "proxyHost", proxy_host);
    CsGetProperty(c, obj, "proxyPort", proxy_port._v);

    if (!httpc)
      htc = HTTP_GET;
    else if (httpc == CsSymbolOf(W("post")))
      htc = HTTP_POST;
    else if (httpc == CsSymbolOf(W("put")))
      htc = HTTP_PUT;
    else if (httpc == CsSymbolOf(W("delete")))
      htc = HTTP_DELETE;

    DATA_PROTOCOL data_protocol = PROTOCOL_BASIC;

    handle<html::request> xrq = new html::request(uri, html::DATA_RAW_DATA);

    xrq->data_represantaion_hint = output_format;

    xrq->proxy_host = proxy_host;
    xrq->proxy_port = proxy_port;

    xrq->username = u8::cvt(username);
    xrq->password = u8::cvt(password);

    xrq->no_cache = no_cache;

    CSF_handle_headers(c, xrq, headers);

    tool::ustring protocol_s;
    if (protocol) protocol_s = CsSymbolName(protocol);

    if (protocol_s == WCHARS("json"))
      data_protocol = PROTOCOL_JSON;
    else if (protocol_s == WCHARS("multipart"))
      data_protocol = PROTOCOL_MULTIPART;

    xrq->rq_type = html::RQ_GET;
    switch (htc) {
    default: xrq->rq_type = html::RQ_GET; break;
    case HTTP_POST: xrq->rq_type = html::RQ_POST; break;
    case HTTP_PUT: xrq->rq_type = html::RQ_PUT; break;
    case HTTP_DELETE: xrq->rq_type = html::RQ_DELETE; break;
    }

    xrq->data_type = html::DATA_RAW_DATA;
    xrq->dst = self.ptr();

    CSF_handle_params(c, xrq, params, data_protocol);

    auto on_data_ready = [=](html::request *prq) -> bool {
      pvalue val(c);

      if (target_path.length()) {
        ustring file_name = url::file_url_to_path(target_path);
        //if (target_path.like(W("file://*"))) file_name = target_path()(7);

        FILE *f = nullptr;
  #if defined(WINDOWS)
        if (0 != _wfopen_s(&f, file_name, L"w+b")) f = nullptr;
  #else
        f = fopen(string(file_name), "w+b");
  #endif
        if (f) {
          size_t r = fwrite(xrq->data.cbegin(), 1, xrq->data.length(), f);
          fclose(f);
          if (r == xrq->data.length())
            val = string_to_value(c, md5(xrq->data()).to_string());
          else
            val = FALSE_VALUE;
        } else
          val = FALSE_VALUE;
      }
      else
      {
        if (data_protocol == PROTOCOL_JSON)
          val = parse_response(c, xrq, OF_JSON);
        else
          val = parse_response(c, xrq, output_format);
      }

      TRY {
        if (xrq->success_flag && success.is_set()) {
          auto_scope as(c, self->ns);
          CsSendMessage(c, obj, success, 2, val.val, CsMakeInteger(xrq->status));
        }
        if (!xrq->success_flag) {
          ustring errm = ustring::format(W("%d"), xrq->status);
          value   errv = 0, errdata = 0;
          PROTECT(errv, errdata);
          errv    = string_to_value(c, errm);
          errdata = CsMakeObject(c, c->objectObject);
          CsSetProperty(c, errdata, "status", CsMakeInteger(xrq->status));
          CsSetProperty(c, errdata, "response", val.val);
          val = CsError(c, CsErrHTTPError, errv, errdata);
        }
        if (!xrq->success_flag && error.is_set()) {
          auto_scope as(c, self->ns);
          CsSendMessage(c, obj, error, 2, val.val, CsMakeInteger(xrq->status));
        }
        if (complete.is_set()) {
          auto_scope as(c, self->ns);
          CsSendMessage(c, obj, complete, 2, val.val, CsMakeInteger(xrq->status));
        }
      }
      CATCH_ERROR(e) {
        if (e.number == 0) // close() called
          return true;
        else
          CsHandleUnhandledError(c);
      }
      return true;
    };

    xrq->add(on_data_ready);

    if (progress.is_set()) {
      auto on_data_progress = [=](html::request *xrq, uint data_size,
                                  uint total_size) -> bool {
        TRY {
          auto_scope as(c, self->ns);
          CsSendMessage(c, obj, progress, 2, CsMakeInteger(data_size),
                        total_size ? CsMakeInteger(total_size) : UNDEFINED_VALUE);
        }
        CATCH_ERROR(e) {
          if (e.number == 0) // close() called
            return true;
          else
            tis::CsHandleUnhandledError(c);
        }
        return true;
      };
      xrq->add_progress(on_data_progress);
    }

    if( try_sync )
      pview->request_data(xrq);
    else 
      pview->async([pview, xrq]() -> bool {
        pview->request_data(xrq);
        return true;
      });
    return CsRequestObject(c, xrq);
  }

}

bool sciter_get_view_expando(tis::xview *pv, tool::value &pval) {
  // tis::dispatch *pd = tis::CsGetDispatch(pv->view_obj);
  // assert( pd == pv->vm->viewDispatch ); pd = pd;
  if (!pv->view_obj) return false;
  pval = tis::value_to_value(pv->vm, pv->view_obj);
  return true;
}
