#include "xsciter.h"
#include "xview.h"

namespace tis {

  html::bookmark value2bookmark(xvm *c, value obm);
  value          bookmark2value(xvm *c, html::bookmark bm);

  html::selection_ctx *selection_ptr(xvm *pvm, value o) {
    if (CsGetDispatch(o) == pvm->selectionDispatch)
      return static_cast<html::selection_ctx *>(CsCObjectValue(o));
    return 0;
  }

  value selection_object(xvm *pvm, html::element *pel) {
    if (!pel) return NULL_VALUE;
    html::selection_ctx *psel = pel->get_selection_ctx();
    if (psel) {
      psel->add_ref();
      return CsMakeCPtrObject(pvm, pvm->selectionDispatch, psel);
    }
    return NULL_VALUE;
  }

  static value CSF_isCollapsed(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    return self->anchor == self->caret ? TRUE_VALUE : FALSE_VALUE;
  }
  static value CSF_anchor(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->anchor.valid()) return NULL_VALUE;
    return bookmark2value(c, self->anchor);
  }
  static value CSF_caret(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) 
      return NULL_VALUE;
    return bookmark2value(c, self->caret);
  }
  static value CSF_start(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->anchor.valid()) return NULL_VALUE;
    return bookmark2value(c, self->anchor < self->caret ? self->anchor
                                                        : self->caret);
  }
  static value CSF_end(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) return NULL_VALUE;
    return bookmark2value(c, self->anchor > self->caret ? self->anchor
                                                        : self->caret);
  }

  static value CSF_text(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) return NULL_VALUE;
    auto        sel = self->normalized();
    html::view *pv  = self->caret.node->pview();
    if (!pv) return UNDEFINED_VALUE;
    html::ostream_w os;
    html::emit_range_text(*pv, os, sel.first, sel.second);
    return CsMakeString(c, os.data());
  }

  static value CSF_html(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) return NULL_VALUE;
    html::view *pv = self->caret.node->pview();
    if (!pv) return NULL_VALUE;
    auto            sel = self->normalized();
    html::ostream_w os;
    html::emit_range_html(*pv, os, sel.first, sel.second, nullptr);
    return CsMakeString(c, os.data());
  }

  static value CSF_type(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) return NULL_VALUE;
    html::view *pv = self->caret.node->pview();
    if (!pv) return NULL_VALUE;
    switch (self->get_selection_type(*pv)) {
      case html::ONLY_CARET: return CsSymbolOf("caret");
      case html::TEXT_RANGE: return CsSymbolOf("text");
      case html::SINGLE_BLOCK: return CsSymbolOf("blocks");
      case html::CELL_RANGE : return CsSymbolOf("cells");
    }
    return NULL_VALUE;
  }

  static value CSF_collapse(xvm *c) {
    value obj;
    value how;
    CsParseArguments(c, "V=*|V=", &obj, c->selectionDispatch, &how,
                     &CsSymbolDispatch);
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid()) return UNDEFINED_VALUE;
    html::view *pv = self->caret.node->pview();
    if (!pv) return UNDEFINED_VALUE;

    static value $toCaret  = CsSymbolOf("toCaret");
    static value $toAnchor = CsSymbolOf("toAnchor");
    static value $toStart  = CsSymbolOf("toStart");
    static value $toEnd    = CsSymbolOf("toEnd");

    if (how == $toCaret)
      self->set_caret_to(*pv, self->caret, false);
    else if (how == $toAnchor)
      self->set_caret_to(*pv, self->anchor, false);
    else if (how == $toStart)
      self->set_caret_to(
          *pv, self->anchor < self->caret ? self->anchor : self->caret, false);
    else if (how == $toEnd)
      self->set_caret_to(
          *pv, self->anchor >= self->caret ? self->anchor : self->caret, false);
    else
      return FALSE_VALUE;
    return bookmark2value(c, self->caret);
  }

  static value CSF_select(xvm *c) {

    value obj;

    if (CsGetDispatch(CsGetArgSafe(c, 3)) == c->elementDispatch) {
      value el, what = UNDEFINED_VALUE;
      CsParseArguments(c, "V=*V=|V", &obj, c->selectionDispatch, &el, c->elementDispatch, &what);
      html::selection_ctx *self = selection_ptr(c, obj);
      if (!self) return UNDEFINED_VALUE;

      html::element *pel = element_ptr(c, el);
      if(!pel) return UNDEFINED_VALUE;
      html::view *   pv  = pel->pview();
      if (!pv)
        CsThrowKnownError(c, CsErrUnexpectedTypeError, el,
                          "element is not in the DOM");

      if (what == CsSymbolOf(WCHARS("content")))
        self->select(*pv, pel->end_caret_pos(*pv), pel->start_caret_pos(*pv));
      else
        self->select(*pv, pel->end_pos(),pel->start_pos());
    } 
    /*else if (CsGetDispatch(CsGetArgSafe(c, 3)) == c->nodeDispatch) 
    {
        value nod, what = UNDEFINED_VALUE;
        CsParseArguments(c, "V=*V=|V", &obj, c->selectionDispatch, &nod, c->nodeDispatch, &what);
        html::selection_ctx *self = selection_ptr(c, obj);
        if (!self) return UNDEFINED_VALUE;

        html::node *pn = node_ptr(c, nod);
        html::view *   pv = pn->pview();
        if (!pv)
          CsThrowKnownError(c, CsErrUnexpectedTypeError, nod, "node is not in the DOM");

        if (what == CsSymbolOf(WCHARS("content")))
          self->select(*pv, pel->end_caret_pos(*pv), pel->start_caret_pos(*pv));
        else
          self->select(*pv, pel->end_pos(), pel->start_pos());
    }*/
    else {

      value vanchor = 0, vcaret = 0;
      CsParseArguments(c, "V=*V=|V=", &obj, c->selectionDispatch, &vcaret,
                       &CsTupleDispatch, &vanchor, &CsTupleDispatch);
      html::selection_ctx *self = selection_ptr(c, obj);
      if (!self) return UNDEFINED_VALUE;

      html::bookmark caret = value2bookmark(c, vcaret);
      if (!caret.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vcaret,
                          "invalid caret position");
      html::bookmark anchor = vanchor ? value2bookmark(c, vanchor) : caret;
      if (!anchor.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vanchor,
                          "invalid anchor position");

      html::view *pv = caret.node->pview();
      if (!pv)
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vcaret,
                          "caret position is not in the DOM");

      self->select(*pv, caret, anchor);
    }
    return TRUE_VALUE;
  }
    
  /*static value CSF_select(xvm *c) {

    value obj;

    value cb = 0;
    CsParseArguments(c, "V=*V=", &obj, c->selectionDispatch, &cb, &CsMethodP);
    html::selection_ctx* self = selection_ptr(c, obj);
    if (!self || !self->caret.valid()) return UNDEFINED_VALUE;

    html::view* pv = self->caret.node->pview();

    PROTECT(cb);

    html::element* celement = nullptr;
    auto match = [&](html::node* n, bool& skip) -> bool {
      html::helement t = n->get_element();
      return false;
    };
    self->selection_each(*pv, match);
    return celement ? TRUE_VALUE : FALSE_VALUE;
  }*/

  static value CSF_advance(xvm *c) {
    value obj;

    static value $caret  = CsSymbolOf("caret");
    static value $anchor = CsSymbolOf("anchor");
    static value $start  = CsSymbolOf("start");
    static value $end    = CsSymbolOf("end");
    static value $both   = CsSymbolOf("both");

    value what = $both, how;

    static value $nextChar  = CsSymbolOf("next");
    static value $prevChar  = CsSymbolOf("prior");
    static value $wordStart = CsSymbolOf("wordStart");
    static value $wordEnd   = CsSymbolOf("wordEnd");
    static value $lineStart = CsSymbolOf("lineStart");
    static value $lineEnd   = CsSymbolOf("lineEnd");
    static value $first     = CsSymbolOf("first");
    static value $last      = CsSymbolOf("last");

    CsParseArguments(c, "V=*V=|V=", &obj, c->selectionDispatch, &how,
                     &CsSymbolDispatch, &what, &CsSymbolDispatch);

    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    // if( !self->caret.valid() ) return UNDEFINED_VALUE;
    html::view *pv = self->selection_root()->pview();
    if (!pv) return UNDEFINED_VALUE;

    html::bookmark target;

    if (what == $both)
      target = self->caret;
    else if (what == $caret)
      target = self->caret;
    else if (what == $anchor)
      target = self->anchor;
    else if (what == $start) {
      if (self->caret <= self->anchor) {
        target = self->caret;
        what   = $caret;
      } else {
        target = self->anchor;
        what   = $anchor;
      }
    } else if (what == $end) {
      if (self->caret >= self->anchor) {
        target = self->caret;
        what   = $caret;
      } else {
        target = self->anchor;
        what   = $anchor;
      }
    } else if (how == $first || how == $last)
      ;
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, what, "target identifier");

    bool r;

    if (how == $nextChar)
      r = target.valid() && html::advance(*pv, self->selection_root(), target,
                                          html::ADVANCE_RIGHT);
    else if (how == $prevChar)
      r = target.valid() && html::advance(*pv, self->selection_root(), target,
                                          html::ADVANCE_LEFT);
    else if (how == $wordStart)
      r = target.valid() && html::advance(*pv, self->selection_root(), target,
                                          html::ADVANCE_WORD_START);
    else if (how == $wordEnd)
      r = html::advance(*pv, self->selection_root(), target,
                        html::ADVANCE_WORD_END);
    else if (how == $lineStart)
      r = target.valid() && html::advance(*pv, self->selection_root(), target,
                                          html::ADVANCE_HOME);
    else if (how == $lineEnd)
      r = target.valid() &&
          html::advance(*pv, self->selection_root(), target, html::ADVANCE_END);
    else if (how == $first)
      r = html::advance(*pv, self->selection_root(), target,
                        html::ADVANCE_FIRST);
    else if (how == $last)
      r = html::advance(*pv, self->selection_root(), target,
                        html::ADVANCE_LAST);
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, how,
                        "wrong step identifier");

    if (!target.valid()) return FALSE_VALUE;

    if (what == $both)
      self->select(*pv, target, target);
    else if (what == $caret)
      self->select(*pv, target, self->anchor);
    else if (what == $anchor)
      self->select(*pv, self->caret, target);

    return TRUE_VALUE;
  }

  static value CSF_applyMark(xvm *c) {
    // return UNDEFINED_VALUE;

    value _this = CsGetArgSafe(c, 1);
    value marks = UNDEFINED_VALUE;

    html::bookmark start, end;

    if (CsIsType(_this, c->selectionDispatch)) {
      CsParseArguments(c, "V=*V", &_this, c->selectionDispatch, &marks);
      html::selection_ctx *self = selection_ptr(c, _this);
      if (!self) return UNDEFINED_VALUE;
      auto startend = self->normalized();
      start         = startend.first;
      end           = startend.second;
    }

    else {

      value vanchor = 0, vcaret = 0;
      CsParseArguments(c, "**V=V=V", &vcaret, &CsTupleDispatch, &vanchor,
                       &CsTupleDispatch, &marks);

      html::bookmark caret = value2bookmark(c, vcaret);
      if (!caret.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vcaret,
                          "invalid caret position");
      html::bookmark anchor = value2bookmark(c, vanchor);
      if (!anchor.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vanchor,
                          "invalid anchor position");
      if (caret < anchor) {
        start = caret;
        end   = anchor;
      } else {
        start = anchor;
        end   = caret;
      }
      start.linearize();
      end.linearize();
    }

    //string marks_wsl;
    array<string> list;

#ifdef DEBUG
    if (CsStringChars(marks) == WCHARS("keyword"))
      list = list;
#endif

    if (CsStringP(marks))
      list.push(string(CsStringChars(marks)));
    else if (CsVectorP(marks)) {
      int n = CsVectorSize(c, marks);
      for (int i = 0; i < n; ++i) {
        string nm = value_to_string(CsVectorElement(c, marks, i));
        list.push(nm);
      }
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, marks,
                        " is not a string or array of strings");

    html::view *pv = start.node->pview();
    if (!pv) return CsMakeInteger(0);

    uint changes = html::apply_marks(*pv, start, end, list());

    return CsMakeInteger(changes);
  }
  static value CSF_clearMark(xvm *c) {
    value _this = CsGetArgSafe(c, 1);
    value marks = UNDEFINED_VALUE;

    html::bookmark start, end;

    if (CsIsType(_this, c->selectionDispatch)) {
      CsParseArguments(c, "V=*V", &_this, c->selectionDispatch, &marks);
      html::selection_ctx *self = selection_ptr(c, _this);
      if (!self) return UNDEFINED_VALUE;
      auto startend = self->normalized();
      start = startend.first;
      end = startend.second;
    }
    else if (CsIsType(CsGetArgSafe(c, 3), c->elementDispatch)) {
      html::element *pel = element_ptr(c, CsGetArgSafe(c, 3));
      html::view *   pv = pel->pview();
      if (!pv) return CsMakeInteger(0);
      uint changes =
        html::clear_marks(*pv, pel->start_pos(), pel->end_pos(), chars());
      return CsMakeInteger(changes);
    }
    else if (CsIsType(CsGetArgSafe(c, 3), c->nodeDispatch)) {
      html::node *pn = node_ptr(c, CsGetArgSafe(c, 3));
      html::view *pv = pn->pview();
      if (!pv) return CsMakeInteger(0);
      uint changes =
        html::clear_marks(*pv, pn->start_pos(), pn->end_pos(), chars());
      return CsMakeInteger(changes);
    }
    else {

      value vanchor = 0, vcaret = 0;
      CsParseArguments(c, "**V=V=V", &vcaret, &CsTupleDispatch, &vanchor,
        &CsTupleDispatch, &marks);

      html::bookmark caret = value2bookmark(c, vcaret);
      if (!caret.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vcaret,
          "invalid caret position");
      html::bookmark anchor = value2bookmark(c, vanchor);
      if (!anchor.valid())
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vanchor,
          "invalid anchor position");
      if (caret < anchor) {
        start = caret;
        end = anchor;
      }
      else {
        start = anchor;
        end = caret;
      }
      start.linearize();
      end.linearize();
    }

    string marks_wsl;

    if (CsStringP(marks))
      marks_wsl = string(CsStringChars(marks));
    else if (CsVectorP(marks)) {
      int n = CsVectorSize(c, marks);
      for (int i = 0; i < n; ++i) {
        if (i) marks_wsl += CHARS(" ");
        string nm = value_to_string(CsVectorElement(c, marks, i));
        marks_wsl += nm();
      }
    }
    else if (marks != UNDEFINED_VALUE)
      CsThrowKnownError(c, CsErrUnexpectedTypeError, marks,
        " is not a string or array of strings");

    html::view *pv = start.node->pview();
    if (!pv) return CsMakeInteger(0);

    uint changes = html::clear_marks(*pv, start, end, marks_wsl);

    return CsMakeInteger(changes);
  }
  static value CSF_BFC(xvm *c, value obj) {
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    if (!self->caret.valid()) return NULL_VALUE;
    if (!self->anchor.valid()) return NULL_VALUE;

    html::view *pv = self->caret.node->pview();
    if (!pv) return NULL_VALUE;

    html::element * cp = html::node::find_common_parent(self->caret.node, self->anchor.node);
    if (!cp) return NULL_VALUE;

    if (cp->is_floats_container(*pv))
      return element_object(c, cp);

    html::element* bfc = cp->floats_parent(*pv);
    return bfc ? element_object(c, bfc) : NULL_VALUE;
  }

  static value CSF_box(xvm *c) {

    value    obj = 0;
    symbol_t side;

    CsParseArguments(c, "V=*L", &obj, c->selectionDispatch, &side);
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid() || !self->anchor.valid()) return UNDEFINED_VALUE;

    side = get_sym_id(side, id_left);

    html::view* pv = self->caret.node->pview();
    if (!pv) return UNDEFINED_VALUE;

    rect rc;
    
    switch (self->get_selection_type(*pv))
    {
      case html::ONLY_CARET:
      case html::TEXT_RANGE:
      {
        rect crc = self->caret.get_caret_rect(*pv);
        rect arc = self->anchor.get_caret_rect(*pv);
        rc = crc | arc;
      } break;
      case html::SINGLE_BLOCK:
      case html::CELL_RANGE: {
        FOREACH(i, self->selected)
          rc |= self->selected[i]->border_box(*pv, html::element::TO_VIEW);
      }  break;
    }

    switch (side) {
    case id_left: return CsMakeInteger(rc.left());
    case id_right: return CsMakeInteger(rc.right());
    case id_top: return CsMakeInteger(rc.top());
    case id_bottom: return CsMakeInteger(rc.bottom());
    case id_width: return CsMakeInteger(rc.width());
    case id_height: return CsMakeInteger(rc.height());
    case id_rect:
      CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()),
        CsMakeInteger(rc.right()), CsMakeInteger(rc.bottom()));
    case id_rectw:
      CS_RETURN4(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()),
        CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    case id_position:
      //CsSetRVal(c, 1, CsMakeInteger(rc.left()));
      //return CsMakeInteger(rc.top());
      CS_RETURN2(c, CsMakeInteger(rc.left()), CsMakeInteger(rc.top()));

    case id_dimension:
      //CsSetRVal(c, 1, CsMakeInteger(rc.width()));
      //return CsMakeInteger(rc.height());
      CS_RETURN2(c, CsMakeInteger(rc.width()), CsMakeInteger(rc.height()));
    }
    //CsThrowKnownError(c, CsErrUnexpectedTypeError, CsGetArg(c,3), "invalid parameter value");
    CsThrowKnownError(c, CsErrValueError, CsGetArg(c, 3));
    return UNDEFINED_VALUE;
  }

  static value CSF_contains(xvm *c) {

    value    obj = 0;
    wchars   selector;

    CsParseArguments(c, "V=*S#", &obj, c->selectionDispatch, &selector.start,&selector.length);
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid() || !self->anchor.valid()) return UNDEFINED_VALUE;
    html::view* pv = self->caret.node->pview();
    if (!pv) return UNDEFINED_VALUE;

    html::element* pel = self->selection_contains(*pv, selector);
    return pel ? element_object(c, pel) : NULL_VALUE;
  }

  static value CSF_elements(xvm *c) {

    value    obj = 0;
    wchars   selector;

    CsParseArguments(c, "V=*S#", &obj, c->selectionDispatch, &selector.start, &selector.length);
    html::selection_ctx *self = selection_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    if (!self->caret.valid() || !self->anchor.valid()) return UNDEFINED_VALUE;
    html::view* pv = self->caret.node->pview();
    if (!pv) return UNDEFINED_VALUE;

    array<html::helement> list;
    self->selection_contains(*pv, selector,list);
    value arr = CsMakeVector(c, list.size());
    if (list.size()) {
      PROTECT(arr);
      for (int n = 0; n < list.size(); ++n) {
        value vel = element_object(c, list[n]);
        CsSetVectorElement(c, arr, n, vel);
      }
    }
    return arr;
  }


  /* Selection methods */
  static c_method methods[] = {C_METHOD_ENTRY_X("collapse", CSF_collapse),
                               C_METHOD_ENTRY_X("select", CSF_select),
                               C_METHOD_ENTRY_X("advance", CSF_advance),

                               C_METHOD_ENTRY_X("applyMark", CSF_applyMark),
                               C_METHOD_ENTRY_X("clearMark", CSF_clearMark),

                               C_METHOD_ENTRY_X("box", CSF_box),
                               C_METHOD_ENTRY_X("contains", CSF_contains),
                               C_METHOD_ENTRY_X("elements", CSF_elements),

                               C_METHOD_ENTRY_X(0, 0)};

  /* Richtext properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY_X("isCollapsed", CSF_isCollapsed, 0),
      VP_METHOD_ENTRY_X("anchor", CSF_anchor, 0),
      VP_METHOD_ENTRY_X("caret", CSF_caret, 0),
      VP_METHOD_ENTRY_X("start", CSF_start, 0),
      VP_METHOD_ENTRY_X("end", CSF_end, 0),
      VP_METHOD_ENTRY_X("text", CSF_text, 0),
      VP_METHOD_ENTRY_X("html", CSF_html, 0),
      VP_METHOD_ENTRY_X("type", CSF_type, 0),
      VP_METHOD_ENTRY_X("blockFormattingContext", CSF_BFC, 0),

      VP_METHOD_ENTRY_X(0, 0, 0)};

  static constant constants[] = {CONSTANT_ENTRY_X(0, 0)};

  void destroy_selection_ctx(xvm *c, value obj) {
    html::selection_ctx *sp = selection_ptr(c, obj);
    if (sp) {
      CsSetCObjectValue(obj, 0);
      sp->release();
    }
  }

  void xvm::init_selection_class() {
    selectionDispatch = CsEnterCPtrObjectType(CsGlobalScope(this), "Selection", methods, properties);
    if (!selectionDispatch) CsInsufficientMemory(this);
    CsEnterConstants(this, selectionDispatch->obj, constants);

    selectionDispatch->destroy = (destructor_t)destroy_selection_ctx;
  }

} // namespace tis
