#include "xsciter.h"
#include "xdom.h"
#include "xview.h"
#include "tool/tl_audio.h"

#ifdef  AUDIO_SUPPORT

namespace tis {


  struct audio : public tool::audio {

    typedef tool::audio super;

    weak_handle<xview> pview;

    audio(xview* pv):pview(pv) {}

    value obj = 0;
    
    ustring url;

    virtual void on_heartbit() override 
    {
      if (!pview) return;
      if (pview->dismissing) { stop(); return; }
      super::on_heartbit();
      handle<audio> holder = this;
      pview->async([holder]() -> bool {
        CsEventObjectFire(holder->pview->vm, holder->obj, CsSymbolOf(CHARS("heartbit")));
        return true;
      });
    }

    virtual void on_stop() override {
      if (!pview) return;
      super::on_stop();
      handle<audio> holder = this;
      pview->async([holder]() -> bool {
        CsEventObjectFire(holder->pview->vm, holder->obj, CsSymbolOf(CHARS("stop")));
        holder->stop();
        return true;
      });
    }

    bool stop() override  {
      auto r = super::stop();
      CsSetCObjectValue(obj, nullptr);
      obj = UNDEFINED_VALUE;
      release();
      pview = nullptr;
      return r;
    }

  };

  inline audio *audio_ptr(xvm *c, value obj) {
    assert(CsGetDispatch(obj) == c->audioDispatch);
    if (!CsIsType(obj, c->audioDispatch)) return nullptr;
    return static_cast<audio *>(CsCObjectValue(obj));
  }

  // view.audio(url) : Audio
  value CSF_audio(xvm *c) {

    wchars url;
    value vobj = 0;

    CsParseArguments(c, "V=*S#",&vobj,c->viewDispatch, &url.start, &url.length);

    handle<audio> paudio = new audio(xview_ptr(c,vobj));

    value obj = CsMakeCPtrObject(c, c->audioDispatch, paudio.ptr());
    paudio->obj = obj;
    paudio->url = url;
    paudio->add_ref();

    return obj;
  }

  static void AudioRelease(VM *c, value obj) {
    audio *prq = audio_ptr(static_cast<xvm *>(c), obj);
    if (prq) {
      prq->obj = 0;
      CsSetCObjectValue(obj, 0);
      prq->release();
    }
  }

  static void AudioScan(VM *c, value obj) {
    audio *p = audio_ptr(static_cast<xvm *>(c), obj);
    if (p) p->obj = CsCopyValue(c, p->obj);
    CsCObjectScan(c, obj);
  }

  static int_t AudioHash(value obj) {
    // its address is a perfect hash
    return (int)(int64)CsCObjectValue(obj);
  }

  static bool AudioPrint(VM *c, value val, stream *s, bool toLocale) {
    audio *p = audio_ptr(static_cast<xvm *>(c), val);
    if (p) {
      s->put_str("Audio(");
      s->put_str(p->url);
      s->put_str(")");
    }
    else
      s->put_str("Audio()");
    return true;
  }

  value CSF_audio_play(xvm *c) {
    value  obj;
    CsParseArguments(c, "V=*", &obj, c->audioDispatch);

    audio *p = audio_ptr(c, obj);
    if(!p)
      CsThrowKnownError(c, CsErrGenericError, "audio::playback stopped");

    tool::audio::result_e r;

    if (p->state == audio::paused)
    {
      p->resume();
      return obj;
    }

    if (p->url().starts_with(WCHARS("file:"))) {
      ustring upath = url::file_url_to_path(p->url);
      r = p->play(upath());
    }
    else {
      handle<html::pump::request> prq = new html::pump::request(p->url, html::DATA_SOUND);
      if(!p->pview->load_data(prq, true))
          CsThrowKnownError(c, CsErrGenericError, string::format("audio::sound '%S' not found",p->url.c_str()).c_str());
      r = p->play(prq->data);
    }

    switch (r) {
      case audio::playback_start_error: CsThrowKnownError(c, CsErrGenericError, "audio::playback error"); break;
      case audio::playback_device_error: CsThrowKnownError(c, CsErrGenericError, "audio::playback device error"); break;
      case audio::file_open_error: CsThrowKnownError(c, CsErrGenericError, "audio::file open error"); break;
      default: break;
    }
    return obj;
  }

  value CSF_audio_pause(xvm *c) {
    value  obj;
    CsParseArguments(c, "V=*", &obj, c->audioDispatch);
    if (audio *p = audio_ptr(c, obj)) {
      p->pause();
      return obj;
    }
    CsThrowKnownError(c, CsErrGenericError, "audio::playback stopped");
    return obj;
  }
   
  value CSF_audio_stop(xvm *c) {
    value  obj;
    CsParseArguments(c, "V=*", &obj, c->audioDispatch);
    if (audio *p = audio_ptr(c, obj)) {
      p->stop();
      return obj;
    }
    return obj;
  }

  value CSF_audio_volume(xvm* c) {
    value  obj;
    float  f = -1.0f;
    CsParseArguments(c, "V=*|f", &obj, c->audioDispatch,&f);
    if (audio *p = audio_ptr(c, obj)) {
      if (f >= 0) {
        p->set_volume(f);
        return obj;
      }
      else {
        p->get_volume(f);
        return CsMakeFloat(f);
      }
    }
    return UNDEFINED_VALUE;
  }

  value CSF_audio_status(xvm* c, value obj) {
    if (audio *p = audio_ptr(c, obj)) {
      return CsMakeInteger(p->state);
    }
    return UNDEFINED_VALUE;
  }

  value CSF_audio_progress(xvm* c, value obj) {
    float pos;
    audio *p = audio_ptr(c, obj);
    if (p && p->get_position(pos)) {
      return CsMakeFloat(pos);
    }
    return UNDEFINED_VALUE;
  }

  //VP_METHOD_ENTRY_X("volume", CSF_audio_volume, CSF_audio_set_volume),

  /* constants */
  static constant audio_constants[] = {
    CONSTANT_ENTRY_X("STOPPED", int_value(audio::stopped)),
    CONSTANT_ENTRY_X("PLAYING", int_value(audio::playing)),
    CONSTANT_ENTRY_X("PAUSED", int_value(audio::paused)),
    CONSTANT_ENTRY_X(0, 0) };

  /* methods */
  static c_method audio_methods[] = {

    C_METHOD_ENTRY_X("play", CSF_audio_play),
    C_METHOD_ENTRY_X("pause", CSF_audio_pause),
    C_METHOD_ENTRY_X("stop", CSF_audio_stop),
    C_METHOD_ENTRY_X("volume", CSF_audio_volume),

    C_METHOD_ENTRY_X(0, 0) };

  /* properties */
  static vp_method audio_properties[] = {
    VP_METHOD_ENTRY_X("status", CSF_audio_status, 0),
    VP_METHOD_ENTRY_X("progress", CSF_audio_progress, 0),
    //VP_METHOD_ENTRY_X("url", CSF_audio_url, 0),
    VP_METHOD_ENTRY_X(0, 0, 0) };

  void xvm::init_audio_class() {

    /* create the 'Audio' type */
    dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(this), "Audio", audio_methods, audio_properties, audio_constants);
    if (!pd) CsInsufficientMemory(this);

    audioDispatch = pd;
    audioDispatch->baseType = &CsCObjectDispatch;

    audioDispatch->destroy = AudioRelease;

    audioDispatch->scan = AudioScan;
    audioDispatch->hash = AudioHash;
    audioDispatch->print = AudioPrint;
    audioDispatch->binOp = CsDefaultObjectBinOp;
  }

}

#endif //  AUDIO_SUPPORT
