#pragma once

#ifndef __win_registry_h__
#define __win_registry_h__

#include "tool/tool.h"

namespace mswin {
  namespace registry {
    using namespace tool;
    using namespace gool;

    class key {
      key(const key&) = delete;
      key& operator=(const key&) = delete;
    public:
      key(HKEY hkey = nullptr) : _hkey(hkey) {}
      key(key&& other) : _hkey(other._hkey) { other._hkey = nullptr; }
      key& operator=(key&& other) noexcept
      {
        if ((this != &other) && (_hkey != other._hkey))
        {
          close();
          _hkey = other._hkey;
          other._hkey = nullptr;
        }
        return *this;
      }
      
      ~key() noexcept { close(); }

	    static key open(HKEY hkey_parent, const wchar*key_name, REGSAM access = KEY_READ | KEY_WOW64_64KEY)
	    {
		    HKEY hKey = nullptr;
		    ::RegOpenKeyEx(hkey_parent,key_name,0,access,&hKey);
        //assert(hKey);
		    return key(hKey);
	    }

      static key create(HKEY hkey_parent, const wchar*key_name, wchar* key_class = REG_NONE,
        DWORD options = REG_OPTION_NON_VOLATILE,
        REGSAM access = KEY_READ | KEY_WRITE,
        SECURITY_ATTRIBUTES* securityAttributes = nullptr,
        DWORD* disposition = nullptr ) 
      {
        HKEY hKey = nullptr;
        ::RegCreateKeyEx(hkey_parent,key_name,0,key_class,options,access,securityAttributes,&hKey,disposition);
        //assert(hKey);
        return key(hKey);
      }

	    operator HKEY() const { return _hkey; }

      void close() noexcept 
      {
        if (is_valid())
        {
          ::RegCloseKey(_hkey);
          _hkey = nullptr;
        }
      }

      bool is_valid() const noexcept { return !!_hkey; }
      explicit operator bool() const noexcept { return is_valid(); }

      HKEY detach() noexcept {  HKEY t = _hkey; _hkey = nullptr; return t; }

      void attach(HKEY hkey) noexcept
      {
        if (_hkey != hkey) { close(); _hkey = hkey; }
      }

      bool operator==(const key& b) const { return _hkey == b._hkey; }
      bool operator!=(const key& b) const { return _hkey != b._hkey; }

      uint32 get(const wchar* value, uint32 dw) 
      {
        DWORD data = {0};
        DWORD dataSize = sizeof(data);
        DWORD dataType = REG_DWORD;
        LONG retCode = ::RegQueryValueEx(_hkey,value,NULL,&dataType,(LPBYTE)&data,&dataSize);
        return retCode != ERROR_SUCCESS ? dw : data;
      }
      uint64 get(const wchar* value, uint64 dw)
      {
        ULONGLONG data = { 0 };
        DWORD dataSize = sizeof(data);
        DWORD dataType = REG_QWORD;
        LONG retCode = ::RegQueryValueEx(_hkey,value,NULL,&dataType,(LPBYTE)&data, &dataSize);
        return retCode != ERROR_SUCCESS ? dw : data;
      }
      ustring get(const wchar* value, const wchar* dw)
      {
        DWORD dataSize = 0;
        DWORD dataType = REG_SZ;
        WCHAR data[1024] = {0};
        LONG retCode = ::RegQueryValueEx(_hkey, value, NULL, &dataType, (LPBYTE)&data, &dataSize);
        if(retCode != ERROR_SUCCESS)
          return dw;
        return data;
      }

      //std::vector<std::wstring>   RegGetMultiString(HKEY hKey, const std::wstring& subKey, const std::wstring& value);
	  array<byte> get(const wchar* value, bytes dw)
	  {
		  DWORD dataSize = 0;
      DWORD dataType = REG_BINARY;
		  LONG retCode = ::RegQueryValueEx(_hkey, value, NULL, &dataType, nullptr, &dataSize);
		  if (retCode != ERROR_SUCCESS)
		    return dw;
		  array<byte> data(dataSize);
		  ::RegQueryValueEx(_hkey, value, NULL, &dataType, data.begin(), &dataSize);
		  return data;
	  }

    private:
      HKEY _hkey;
    };


  }

} // namespace mswin

#endif
