#ifndef __html_dom_accessible_h__
#define __html_dom_accessible_h__

#pragma once

#include "tool/tl_scope.h"
#include "html/html-streams.h"
#include "html/html-dom-traversal.h"
#include "html/html-layout.h"
#include "html/html-dom.h"
#include "html/html-document.h"
#include "html/html-view.h"
#include <oleacc.h>

#ifdef USE_UIAUTOMATION

#include <UIAutomation.h>
#include <comdef.h>
#include <comutil.h>
#include <UIAutomationCoreApi.h>
#include "win-delayload.h"
//#include <atlcomcli.h>

EXTERN_DLOADV(_UiaReturnRawElementProvider, UiaReturnRawElementProvider,
              uiautomationcore.dll,
              LRESULT(WINAPI *)(HWND hwnd, WPARAM wParam, LPARAM lParam,
                                IRawElementProviderSimple *el));

EXTERN_DLOADV(_UiaHostProviderFromHwnd, UiaHostProviderFromHwnd,
              uiautomationcore.dll,
              HRESULT(WINAPI *)(HWND                        hwnd,
                                IRawElementProviderSimple **pProvider));

EXTERN_DLOADV(_UiaRaiseAutomationEvent, UiaRaiseAutomationEvent,
              uiautomationcore.dll,
              HRESULT(WINAPI *)(IRawElementProviderSimple *pProvider,
                                EVENTID                    id));

EXTERN_DLOADV(_UiaClientsAreListening, UiaClientsAreListening,
              uiautomationcore.dll, BOOL(WINAPI *)());

#endif

// implementation of IAccessible interface for DOM elements.

namespace html {
  using namespace tool;
  using namespace gool;

#ifdef USE_UIAUTOMATION

  //bool is_like_text(element *b);

  class accessible_root_provider;
  class accessible_sciter_provider;
  class accessible_base;

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class invoke_provider : public IInvokeProvider {
    locked::counter                        _refcount;
    handle<element>                        _element_ptr;
    com::asset<accessible_sciter_provider> _sciter_povider_ptr;

  public:
    invoke_provider(element *element_ptr, uint ref_count = 1)
        : _element_ptr(element_ptr), _refcount(ref_count) {}

    static invoke_provider *make(element *element_ptr, uint ref_count = 1);

    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    //
    // IInvokeProvider
    //
    virtual HRESULT STDMETHODCALLTYPE Invoke();
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class selection_item_provider : public ISelectionItemProvider {
    locked::counter                        _refcount;
    handle<element>                        _element_ptr;
    com::asset<accessible_sciter_provider> _sciter_povider_ptr;

  public:
    selection_item_provider(element *                   element_ptr,
                            accessible_sciter_provider *sciter_ptr,
                            uint                        ref_count = 1)
        : _sciter_povider_ptr(sciter_ptr), _element_ptr(element_ptr),
          _refcount(ref_count) {}

    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    //
    // Make object.
    //
    static selection_item_provider *make(element *element_ptr,
                                         accessible_sciter_provider *sciter_ptr,
                                         uint ref_count = 1);

    //
    // ISelectionItemProvider
    //
    virtual HRESULT STDMETHODCALLTYPE Select();
    virtual HRESULT STDMETHODCALLTYPE AddToSelection();
    virtual HRESULT STDMETHODCALLTYPE RemoveFromSelection();
    virtual HRESULT STDMETHODCALLTYPE get_IsSelected(__RPC__out BOOL *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_SelectionContainer(
        __RPC__deref_out_opt IRawElementProviderSimple **pRetVal);
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class toggle_proivder : public IToggleProvider {
    locked::counter _refcount;
    handle<element> _element_ptr;

  public:
    toggle_proivder(element *element_ptr, uint ref_count = 1)
        : _element_ptr(element_ptr), _refcount(ref_count) {}

    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    template <class T>
    HRESULT STDMETHODCALLTYPE QueryInterface(T **interface_ptr) {
      return QueryInterface(__uuidof(T), (void **)interface_ptr);
    }

    //
    // Make object.
    //
    static toggle_proivder *make(element *element_ptr, uint ref_count = 1);

    //
    // IToggleProvider
    //
    virtual HRESULT STDMETHODCALLTYPE Toggle();
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_ToggleState(__RPC__out enum ToggleState *pRetVal);
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class value_provider : public IValueProvider {
    locked::counter _refcount;
    handle<element> _element_ptr;

  public:
    value_provider(element *element_ptr, uint ref_count = 1)
        : _element_ptr(element_ptr), _refcount(ref_count) {}

    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    template <class T>
    HRESULT STDMETHODCALLTYPE QueryInterface(T **interface_ptr) {
      return QueryInterface(__uuidof(T), (void **)interface_ptr);
    }

    //
    // Make object.
    //
    static value_provider *make(element *element_ptr, uint ref_count = 1);

    //
    // IValueProvider
    //
    virtual HRESULT STDMETHODCALLTYPE SetValue(__RPC__in LPCWSTR val);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_Value(__RPC__deref_out_opt BSTR *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_IsReadOnly(__RPC__out BOOL *pRetVal);
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class accessible_sciter_provider : public IRawElementProviderSimple
                                   , public IRawElementProviderFragment
                                   //, public IAccIdentity
  {
  protected:
    locked::counter                      _refcount;
    handle<element>                      _element_ptr;
    com::asset<accessible_root_provider> _root_ptr;

    accessible_sciter_provider(accessible_root_provider *root_ptr,
                               element *elment_ptr, uint ref_count = 1);

  public:
    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    template <class T>
    HRESULT STDMETHODCALLTYPE QueryInterface(T **interface_ptr) {
      return QueryInterface(__uuidof(T), (void **)interface_ptr);
    }

    //
    // Make object.
    //
    static accessible_sciter_provider *make(accessible_root_provider *root_ptr,
                                            element *element_ptr);

    //
    // IRawElementProviderSimple
    //
    virtual HRESULT STDMETHODCALLTYPE GetPatternProvider(
        PATTERNID patternId, __RPC__deref_out_opt IUnknown **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetPropertyValue(PROPERTYID propertyId, __RPC__out VARIANT *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_HostRawElementProvider(
        __RPC__deref_out_opt IRawElementProviderSimple **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    get_ProviderOptions(__RPC__out enum ProviderOptions *pRetVal);

    //
    // IRawElementProviderFragment
    //
    virtual HRESULT STDMETHODCALLTYPE
                    GetEmbeddedFragmentRoots(__RPC__deref_out_opt SAFEARRAY **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetRuntimeId(__RPC__deref_out_opt SAFEARRAY **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    Navigate(enum NavigateDirection direction,
                             __RPC__deref_out_opt IRawElementProviderFragment **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE SetFocus();
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_BoundingRectangle(__RPC__out struct UiaRect *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_FragmentRoot(
        __RPC__deref_out_opt IRawElementProviderFragmentRoot **pRetVal);
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class accessible_root_provider : public IRawElementProviderSimple,
                                   public IRawElementProviderFragment,
                                   public IRawElementProviderFragmentRoot {
  protected:
    locked::counter _refcount;
    handle<element> _element_ptr;
    HWND            _hwnd;

    accessible_root_provider(HWND hwnd, element *elment_ptr,
                             uint ref_count = 1);

  public:
    static accessible_root_provider *make(HWND hwnd, element *element_ptr);

    void raise_automation_event(EVENTID event_id);

    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    template <class T>
    HRESULT STDMETHODCALLTYPE QueryInterface(T **interface_ptr) {
      return QueryInterface(__uuidof(T), (void **)interface_ptr);
    }

    virtual HRESULT STDMETHODCALLTYPE GetPatternProvider(
        PATTERNID patternId, __RPC__deref_out_opt IUnknown **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetPropertyValue(PROPERTYID propertyId, __RPC__out VARIANT *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_HostRawElementProvider(
        __RPC__deref_out_opt IRawElementProviderSimple **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    get_ProviderOptions(__RPC__out enum ProviderOptions *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetEmbeddedFragmentRoots(__RPC__deref_out_opt SAFEARRAY **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetRuntimeId(__RPC__deref_out_opt SAFEARRAY **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    Navigate(enum NavigateDirection direction,
                             __RPC__deref_out_opt IRawElementProviderFragment **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE SetFocus();
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_BoundingRectangle(__RPC__out struct UiaRect *pRetVal);
    virtual HRESULT STDMETHODCALLTYPE get_FragmentRoot(
        __RPC__deref_out_opt IRawElementProviderFragmentRoot **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE ElementProviderFromPoint(
        double x, double y,
        __RPC__deref_out_opt IRawElementProviderFragment **pRetVal);
    virtual HRESULT STDMETHODCALLTYPE
                    GetFocus(__RPC__deref_out_opt IRawElementProviderFragment **pRetVal);
  };

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  class legacy_iaccessible_provider : public ILegacyIAccessibleProvider {

    locked::counter _refcount;
    handle<element> _element_ptr;

  public:
    legacy_iaccessible_provider(element *element_ptr, uint ref_count = 1);

    //
    // IUnknown
    //
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid,
                                                     void **interface_ptr);

    template <class T>
    HRESULT STDMETHODCALLTYPE QueryInterface(REFIID iid, T **interface_ptr) {
      return QueryInterface(iid, (void **)interface_ptr);
    }

    static legacy_iaccessible_provider *make(element *element_ptr,
                                             uint     ref_count = 1) {
      return new legacy_iaccessible_provider(element_ptr, ref_count);
    }

    //
    // ILegacyIAccessibleProvider
    //
    virtual HRESULT STDMETHODCALLTYPE Select(long flagsSelect);
    virtual HRESULT STDMETHODCALLTYPE DoDefaultAction(void);
    virtual HRESULT STDMETHODCALLTYPE SetValue(__RPC__in LPCWSTR szValue);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      GetIAccessible(__RPC__deref_out_opt IAccessible **ppAccessible);
    virtual HRESULT STDMETHODCALLTYPE get_ChildId(__RPC__out int *pRetVal);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_Name(__RPC__deref_out_opt BSTR *pszName);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_Value(__RPC__deref_out_opt BSTR *pszValue);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_Description(__RPC__deref_out_opt BSTR *pszDescription);
    virtual HRESULT STDMETHODCALLTYPE get_Role(__RPC__out DWORD *pdwRole);
    virtual HRESULT STDMETHODCALLTYPE get_State(__RPC__out DWORD *pdwState);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_Help(__RPC__deref_out_opt BSTR *pszHelp);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_KeyboardShortcut(__RPC__deref_out_opt BSTR *pszKeyboardShortcut);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      GetSelection(__RPC__deref_out_opt SAFEARRAY **pvarSelectedChildren);
    virtual HRESULT                   STDMETHODCALLTYPE
                                      get_DefaultAction(__RPC__deref_out_opt BSTR *pszDefaultAction);
  };

#endif // USE_UIAUTOMATION

  /*inline bool is_like_text(element *b) {
    if (!b->is_of_type<text_block>()) return false;
    if (b->nodes.size() != 1 || !b->nodes[0]->is_text()) return false;
    return true;
  }*/

  class accessible_base : public IAccessible {
  protected:
    enum _ { A_MAGIC = 0xA000FED0 };
    uint            magic;
    locked::counter refcount;

    accessible_base(uint rc = 1) : refcount(rc), magic(uint(A_MAGIC)) {}

  public:
    virtual ~accessible_base() {}

    bool valid() const { return magic != A_MAGIC; }

    // IUnknown stuff

    virtual HRESULT STDMETHODCALLTYPE QueryInterface(
        /* [in] */ REFIID          riid,
        /* [iid_is][out] */ void **ppvObject) {
      IUnknown **ppUnk = (IUnknown **)ppvObject;
      if (riid == IID_IUnknown || riid == IID_IDispatch ||
          riid == IID_IAccessible) {
        *ppUnk = this;
        AddRef();
        return S_OK;
      }
      *ppUnk = NULL;
      return E_NOINTERFACE;
    }

    virtual ULONG STDMETHODCALLTYPE AddRef(void) {
      return locked::inc(refcount);
    }

    virtual ULONG STDMETHODCALLTYPE Release(void) {
      if (locked::dec(refcount) <= 0) {
        delete this;
        return 0;
      }
      return refcount;
    }

    // IDispatch stuff

    virtual HRESULT STDMETHODCALLTYPE GetTypeInfoCount(
        /* [out] */ UINT *pctinfo) {
      if (pctinfo) *pctinfo = 0;
      return S_OK;
    }

    virtual HRESULT STDMETHODCALLTYPE GetTypeInfo(
        /* [in] */ UINT         iTInfo,
        /* [in] */ LCID         lcid,
        /* [out] */ ITypeInfo **ppTInfo) {
      return TYPE_E_ELEMENTNOTFOUND;
    }

    virtual HRESULT STDMETHODCALLTYPE GetIDsOfNames(
        /* [in] */ REFIID             riid,
        /* [size_is][in] */ LPOLESTR *rgszNames,
        /* [in] */ UINT               cNames,
        /* [in] */ LCID               lcid,
        /* [size_is][out] */ DISPID * rgDispId) {
      *rgDispId = DISPID_UNKNOWN;
      return DISP_E_UNKNOWNNAME;
    }

    virtual /* [local] */ HRESULT STDMETHODCALLTYPE Invoke(
        /* [in] */ DISPID           dispIdMember,
        /* [in] */ REFIID           riid,
        /* [in] */ LCID             lcid,
        /* [in] */ WORD             wFlags,
        /* [out][in] */ DISPPARAMS *pDispParams,
        /* [out] */ VARIANT *       pVarResult,
        /* [out] */ EXCEPINFO *     pExcepInfo,
        /* [out] */ UINT *          puArgErr) {
      return DISP_E_MEMBERNOTFOUND;
    }

    // some IAccessible stuff

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accHelp(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszHelp) {
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accHelpTopic(
                                                    /* [out] */ BSTR *           pszHelpFile,
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ long *   pidTopic) {
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accDoDefaultAction(
        /* [optional][in] */ VARIANT varChild) {
      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][propput][hidden] */ HRESULT STDMETHODCALLTYPE put_accName(
        /* [optional][in] */ VARIANT varChild,
        /* [in] */ BSTR              szName) {
      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accName(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszName) {
      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][propput][hidden] */ HRESULT STDMETHODCALLTYPE put_accValue(
        /* [optional][in] */ VARIANT varChild,
        /* [in] */ BSTR              szValue) {
      return DISP_E_MEMBERNOTFOUND;
    }
  };

  class accessible : public accessible_base {
  protected:
    handle<element> blk;

    struct locker {
      html::view *pv;
      locker(element *blk) : pv(0) {
        if (blk) {
          pv = blk->pview();
          if (pv) pv->guard.lock();
        }
      }
      ~locker() {
        if (pv) pv->guard.unlock();
      }
    };

  public:
    bool is_popup;

    accessible(element *b, uint rc = 1) : blk(b), is_popup(false) {}

    static accessible *     factory(element *b, uint rc = 1);
    static accessible_base *factory(node *n, uint rc = 1);

    /*bool is_acc_node(html::node *pn) {
      if (pn->is_comment())
        return false;
      else if (pn->is_text()) {
        return !pn->cast<html::text>()->is_space();
      } else if (pn->is_element()) {
        html::view *pv = pn->cast<element>()->pview();
        if (!pv) return false;
        return pn->is_a11y_visible(*pv);
      }
      return false;
    }

    uint get_acc_node_count() {
      uint cnt = 0;
      for (int n = 0; n < blk->nodes.size(); ++n) {
        if (is_acc_node(blk->nodes[n])) ++cnt;
      }
      return cnt;
    }

    node *get_acc_node(uint n) {
      uint cnt = 0;
      for (int i = 0; i < blk->nodes.size(); ++i) {
        if (!is_acc_node(blk->nodes[i])) continue;
        if (cnt++ == n) return blk->nodes[i];
      }
      return 0;
    }
    */

    // IAccessible per se

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accParent(
        /* [retval][out] */ IDispatch **ppdispParent) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
#endif
      locker l(blk);

      view *pv = blk->pview();
      if (!pv) return S_FALSE;

      /* ?? iwindow *piw = pv->get_iwindow_of(blk);
      if (piw && piw->get_hwnd() && ::IsWindow(piw->get_hwnd())) {
        return AccessibleObjectFromWindow(::GetDesktopWindow(),
                                          DWORD(OBJID_CLIENT), IID_IAccessible,
                                          (void **)ppdispParent);
      }*/

      element *pp = blk->ui_parent(*pv);
      if (pp) {
        *ppdispParent = accessible::factory(pp);
        return S_OK;
      }

      if (blk->is_document()) {
        HWND hwnd = pv->get_hwnd();
        if (::IsWindow(hwnd)) {
          return AccessibleObjectFromWindow(
              ::GetDesktopWindow(), DWORD(OBJID_CLIENT), IID_IAccessible,
              (void **)ppdispParent);
        }
      }

      /*if( blk->hctl)
      {
          HWND hwnd = ::GetParent((HWND)blk->hctl);
          handle<element> parent_root = window_block(hwnd);
          if( parent_root )
          {
            block_visitor bv(parent_root);
            for(element *t = bv.first(); t; t = bv.next())
              if( t->hctl == blk->hctl && t->parent)
              {
                *ppdispParent = accessible::factory(t->parent);
                return S_OK;
              }
          }
          if(::IsWindow(hwnd))
          {
            return AccessibleObjectFromWindow(hwnd, OBJID_CLIENT,
      IID_IAccessible, (void**)ppdispParent);
          }
      }*/

      // dbg_printf("get_accParent \n");

      *ppdispParent = 0;
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accChildCount(
                                                    /* [retval][out] */ long *pcountChildren) {
#ifdef _DEBUG
      if (blk->tag == tag::T_TR) blk = blk;
      if (blk->tag == tag::T_TBODY) blk = blk;
      if (is_popup) is_popup = is_popup;
      if (blk->tag == tag::T_SELECT) blk = blk;
      if (blk->tag == tag::T_BODY) blk = blk;
      if (blk->tag == tag::T_LABEL) blk = blk;

#endif
      locker l(blk);

      view *pv = blk->pview();
      if (!pv) return S_FALSE;

      CTL_TYPE t = blk->ctl_type(*pv);
      /*if( t == CTL_DD_SELECT
            || t == CTL_SELECT_SINGLE
            || t == CTL_SELECT_MULTIPLE
            || t == CTL_LIST
            || t == CTL_NO || t == CTL_UNKNOWN || t == CTL_FRAME || t ==
      CTL_FRAMESET) *pcountChildren = blk->n_ui_children(); else *pcountChildren
      = 0; */

      switch (t) {
      case CTL_IMAGE: *pcountChildren = 0; break;
      case CTL_EDIT:
      case CTL_NUMERIC:
      case CTL_BUTTON:
      case CTL_CHECKBOX:
      case CTL_RADIO:
      case CTL_TEXTAREA:
      case CTL_PASSWORD:
        *pcountChildren = 0;
        break;
        // else fall through
      default:
        //*pcountChildren = blk->n_ui_children();
        {
          array<hnode> children;
          blk->a11y_get_children(children);
          *pcountChildren = children.size();
        }
        break;
      }

#ifdef _DEBUG
      blk->dbg_report("get_accChildCount");
#endif

      return S_OK;
    }

    IAccessible *get_iacc(HWND hwnd) {
      IAccessible *pdispChild = 0;
      HRESULT      hr         = AccessibleObjectFromWindow(
          hwnd, OBJID_WINDOW, IID_IAccessible, (void **)&pdispChild);
      if (SUCCEEDED(hr)) return pdispChild;
      return 0;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accChild(
        /* [in] */ VARIANT              varChild,
        /* [retval][out] */ IDispatch **ppdispChild) {
#ifdef _DEBUG
      if (blk->tag == tag::T_LABEL) blk = blk;
#endif

      if (varChild.vt != VT_I4) return E_INVALIDARG;

      html::hnode child;

      locker l(blk);
      view * pv = blk->pview();
      if (pv) {
        /*if (varChild.lVal == 0x7FFFFFFF) // special case - current element
        {
          child = find_first(*pv, blk, WCHARS(":current"), true);
        } else*/ 
        if (varChild.lVal == CHILDID_SELF) {
          child = blk;
        }
        else if (node::is_node_uid(varChild.lVal)) {
          child = blk->get_node_by_uid(varChild.lVal);
        }
        else {
          array<hnode> children;
          blk->a11y_get_children(children);
          if (varChild.lVal < 1 || varChild.lVal > children.size())
            return E_INVALIDARG;
          child = children[varChild.lVal - 1];
        }
      }

      if (child /*&& (child->is_visible() || child->state.popup() )*/) {
        if (pv && child->is_element()) {
          iwindow *pw = child->cast<element>()->window(*pv);
          if (pw && pw->type == html::CHILD_WINDOW) {
            *ppdispChild = get_iacc(pw->get_hwnd());
            return S_OK;
          }
        }
#ifdef _DEBUG
        blk->dbg_report("get_accChild self");
        child->dbg_report("get_accChild child");
#endif
        *ppdispChild = accessible::factory(child);
        return S_OK;
      } else {
#ifdef _DEBUG
        blk->dbg_report("get_accChild FAILURE");
#endif
        *ppdispChild = 0;
        return S_FALSE;
      }
    }

            
#define ELEMENT(el)                                                            \
  element *el = nullptr;                                                       \
  if (varChild.vt == VT_I4 && node::is_node_uid(varChild.lVal))                \
    { el = blk->get_element_by_uid(varChild.lVal);  assert(el); }              \
  else if (varChild.vt == VT_I4 && varChild.lVal == CHILDID_SELF)              \
    el = blk;                                                                  \
  else if (varChild.vt != VT_I4)                                               \
    return E_INVALIDARG;                                                       \
  else  {                                                                      \
    array<hnode> nodes;                                                        \
    blk->a11y_get_children(nodes);                                             \
    if (varChild.lVal > 0 && varChild.lVal <= nodes.size()) {                  \
      hnode hn = nodes[varChild.lVal - 1];                                     \
      if (!hn->is_element())                                                   \
        return S_FALSE;                                                        \
      el = hn.ptr_of<element>();                                               \
    }                                                                          \
    else                                                                       \
      return E_INVALIDARG;                                                     \
  }                                                                            \
  if (!el) { assert(false); return S_FALSE; }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accValue(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszValue) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF)
      {
        dbg_printf("get_accValue %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif

#ifdef _DEBUG
      blk->dbg_report("get_accValue");
#endif

      ELEMENT(el)

      locker l(blk);

      view *pv = el->pview();
      if (!pv) return S_FALSE;

      int ctl_type = el->ctl_type(*pv);

      if (ctl_type != CTL_NO && ctl_type != CTL_BUTTON) // this is control
      {
        ustring txt;
        html::view *pv = el->pview();
        if (pv && el->a11y_get_value(*pv, txt)) {
          *pszValue   = SysAllocStringLen(txt, (UINT)txt.length());
          return S_OK;
        }
      }

#ifdef _DEBUG
      blk->dbg_report("get_accValue FALSE");
#endif
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accName(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszName) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
      if (blk->tag == tag::T_INPUT) blk = blk;
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF)
      {
        dbg_printf("get_accName %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif

#ifdef _DEBUG
      if (blk->is_id_test())
        blk = blk;
      blk->dbg_report("get_accName");
#endif

      ELEMENT(el)


      locker l(blk);

      ustring name;

      view *pv = el->pview();
      if (pv && el->a11y_get_name(*pv, name)) {

#ifdef _DEBUG
          const wchar_t * xname = name.c_str();
        dbg_printf("a11y_get_name A '%S'\n", name.c_str());
#endif
        *pszName = SysAllocStringLen(name, (UINT)name.length());
        return S_OK;
      }
      // I am not sure about these two, seems like Mozilla does the same:
      /*name = el->atts("alt");
      if( name.length() )
      {
        *pszName = SysAllocStringLen( name, name.length());
        return S_OK;
      }*/
      name = el->attr_title();
      if (name.length()) {
        *pszName = SysAllocStringLen(name, (UINT)name.length());
        return S_OK;
      }

#ifdef _DEBUG
      blk->dbg_report("get_accName FALSE");
#endif

      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accDescription(
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ BSTR *   pszDescription) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF)
      {
        dbg_printf("get_accDescription %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif

#ifdef _DEBUG
      blk->dbg_report("get_accDescription");
#endif

      ELEMENT(el)

      locker l(blk);

      ustring txt;

      view *pv = el->pview();
      if (pv && el->a11y_get_desc(*pv, txt)) {
        *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
        return S_OK;
      }

      txt = el->atts[attr::a_alt];
      if (txt.length()) {
        *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
        return S_OK;
      }
      txt = el->atts[attr::a_title];
      if (txt.length()) {
        *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
        return S_OK;
      }
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accRole(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ VARIANT *pvarRole) {

#ifdef DEBUG
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF) {
        dbg_printf("get_accRole %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif // DEBUG

      ELEMENT(el)

//#ifdef _DEBUG
//        blk->dbg_report("get_accRole");
//#endif

      // if(el != blk)
      //  return S_FALSE;

      pvarRole->vt   = VT_I4;
      pvarRole->lVal = get_role(el);

      // debug_printf( "get_accRole, el=%s role=%x\n",
      // symbol_name(el->type).c_str(), pvarRole->lVal );

      return S_OK;
    }

    static uint get_role(element *blk) {

      // if( blk->state.popup() )
      //  return ROLE_SYSTEM_TOOLTIP;

      locker l(blk);

      view *pv = blk->pview();
      if (!pv) return CTL_NO;

      iwindow *pw = blk->window(*pv);
      if (pw && pw->type == html::CHILD_WINDOW) return ROLE_SYSTEM_WINDOW;

      // 1) first try input elements.

      CTL_TYPE ct = blk->ctl_type(*pv);
      switch (ct) {
      case CTL_EDIT: return ROLE_SYSTEM_TEXT;
      case CTL_NUMERIC:
        return ROLE_SYSTEM_TEXT;
      // case CTL_CLICKABLE        : return ROLE_SYSTEM_PUSHBUTTON;
      case CTL_BUTTON: return ROLE_SYSTEM_PUSHBUTTON;
      case CTL_CHECKBOX: return ROLE_SYSTEM_CHECKBUTTON;
      case CTL_RADIO: return ROLE_SYSTEM_RADIOBUTTON;
      case CTL_SELECT_SINGLE: return ROLE_SYSTEM_LIST;
      case CTL_SELECT_MULTIPLE: return ROLE_SYSTEM_LIST;
      case CTL_DD_SELECT: return ROLE_SYSTEM_COMBOBOX;
      case CTL_TEXTAREA: return ROLE_SYSTEM_TEXT;
      case CTL_HTMLAREA: return ROLE_SYSTEM_TEXT;
      case CTL_PASSWORD: return ROLE_SYSTEM_TEXT;

      case CTL_PROGRESS: return ROLE_SYSTEM_PROGRESSBAR;
      case CTL_SLIDER: return ROLE_SYSTEM_SLIDER;
      case CTL_DECIMAL: return ROLE_SYSTEM_TEXT;
      case CTL_CURRENCY: return ROLE_SYSTEM_TEXT;
      case CTL_SCROLLBAR: return ROLE_SYSTEM_SCROLLBAR;
      case CTL_MENUBAR: return ROLE_SYSTEM_MENUBAR;
      case CTL_MENU: return ROLE_SYSTEM_MENUPOPUP;
      case CTL_HYPERLINK: return ROLE_SYSTEM_LINK;
      case CTL_MENUBUTTON: return ROLE_SYSTEM_BUTTONDROPDOWN;
      case CTL_FRAME: return ROLE_SYSTEM_PANE;
      case CTL_FRAMESET: return ROLE_SYSTEM_PANE;
      case CTL_TOOLTIP: return ROLE_SYSTEM_TOOLTIP;
      default: break;
      }

      // 2) try hyperlink and known role attribute values
      if (blk->atts.exist(attr::a_href)) return ROLE_SYSTEM_LINK;

      if (blk->tag == tag::T_LI && blk->parent &&
          blk->parent->ctl_type(*pv) == CTL_MENU)
        return ROLE_SYSTEM_MENUITEM;

      ustring role = blk->atts("role");
      if (role.length()) {
        if (role == WCHARS("option")) return ROLE_SYSTEM_LISTITEM;
        if (role == WCHARS("statusbar")) return ROLE_SYSTEM_STATUSBAR;
        if (role == WCHARS("toolbar")) return ROLE_SYSTEM_TOOLBAR;
        if (role == WCHARS("panel")) return ROLE_SYSTEM_PANE;
        if (role == WCHARS("group")) return ROLE_SYSTEM_GROUPING;
        if (role == WCHARS("menu-item")) return ROLE_SYSTEM_MENUITEM;
        if (role == WCHARS("dialog")) return ROLE_SYSTEM_DIALOG;
        if (role == WCHARS("page-tab")) return ROLE_SYSTEM_PAGETAB;
        if (role == WCHARS("page-tab-list")) return ROLE_SYSTEM_PAGETABLIST;
        if (role == WCHARS("window-caption")) return ROLE_SYSTEM_TITLEBAR; 
        if (role == WCHARS("window")) return ROLE_SYSTEM_WINDOW; // NOTE: maybe be set this automatically for window-frame="extended" and other
      }

      // ROLE_SYSTEM_PAGETAB
      // The object represents a page tab. The only child of a page tab control
      // is a ROLE_SYSTEM_GROUPING object that contains the contents of the
      // associated page.  ROLE_SYSTEM_PAGETABLIST  The object represents a
      // container of page tab controls.

      if (blk->state.popup()) return ROLE_SYSTEM_MENUPOPUP;

      // 3) try specific tags.
      switch (blk->tag) {
      case tag::T_TFOOT:
      case tag::T_THEAD:
      case tag::T_TBODY: return ROLE_SYSTEM_TABLE;
      case tag::T_TABLE: return ROLE_SYSTEM_PANE;
      case tag::T_TR: return ROLE_SYSTEM_ROW;
      case tag::T_TD:
      case tag::T_TH: return ROLE_SYSTEM_CELL;
      case tag::T_OPTION: return ROLE_SYSTEM_LISTITEM;
      // case T_FIELDSET: return ROLE_SYSTEM_GROUPING; // for some reasons
      // narrator don't go inside GROUPING...
      case tag::T_FIELDSET: return ROLE_SYSTEM_PANE;
      case tag::T_IMG: return ROLE_SYSTEM_GRAPHIC;
      case tag::T_IFRAME: return ROLE_SYSTEM_PANE;
      case tag::T_POPUP: return ROLE_SYSTEM_DIALOG;
      case tag::T_MENU: return ROLE_SYSTEM_MENUPOPUP;
      case tag::T_LI:
        if (blk->parent && blk->parent->tag == tag::T_MENU)
          return ROLE_SYSTEM_MENUITEM;
        else
          return ROLE_SYSTEM_LISTITEM;
        break;
      case tag::T_DD:
      case tag::T_DT: return ROLE_SYSTEM_LISTITEM;

      case tag::T_DL:
      case tag::T_UL:
      case tag::T_OL:
        return ROLE_SYSTEM_LIST;

      case tag::T_BODY: 
        return ROLE_SYSTEM_PANE; // or ROLE_SYSTEM_GROUPING ?
      case tag::T_HTML:
        if(!blk->parent)
          return ROLE_SYSTEM_CLIENT; // root document, spans whole client area of the window.
        else
          return ROLE_SYSTEM_PANE;
      }

      if (blk->a11y_is_text()) 
        return ROLE_SYSTEM_STATICTEXT;

      return ROLE_SYSTEM_GROUPING;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accState(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ VARIANT *pvarState) {

#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF) {
        dbg_printf("get_accState %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif

#ifdef _DEBUG
      blk->dbg_report("get_accState");
#endif


      ELEMENT(el)

      view *pv = el->pview();
      if (!pv) return S_FALSE;

      locker l(blk);

      uint state = 0;

      ui_state ust;
      el->a11y_get_state(*pv, ust);

      if (ust.link()) state |= STATE_SYSTEM_LINKED;
      if (ust.active()) state |= STATE_SYSTEM_PRESSED;
      if (ust.focus()) state |= STATE_SYSTEM_FOCUSED;
      if (ust.visited()) state |= STATE_SYSTEM_TRAVERSED;
      if (ust.current()) state |= STATE_SYSTEM_SELECTED | STATE_SYSTEM_FOCUSED;
      if (ust.checked()) state |= STATE_SYSTEM_CHECKED;
      if (ust.readonly()) state |= STATE_SYSTEM_READONLY;
      if (ust.expanded()) state |= STATE_SYSTEM_EXPANDED;
      if (ust.collapsed()) state |= STATE_SYSTEM_COLLAPSED;

      if (el->is_disabled()) state |= STATE_SYSTEM_UNAVAILABLE; // do we need here deep versions?
      if (el->is_focusable(*pv)) state |= STATE_SYSTEM_FOCUSABLE;

      if (el->ctl_type() == CTL_DD_SELECT ) state |= 0x40000000; // haspopup

      if (!el->is_visible(*pv)) state |= STATE_SYSTEM_INVISIBLE | STATE_SYSTEM_OFFSCREEN;

      element::A11Y_LIVE lt = el->get_a11y_live(*pv);
      switch (lt) {
        case element::A11Y_LIVE_ASSERTIVE: state |= STATE_SYSTEM_ALERT_HIGH; break;
        case element::A11Y_LIVE_POLITE: state |= STATE_SYSTEM_ALERT_MEDIUM; break;
      }

      pvarState->vt   = VT_I4;
      pvarState->lVal = state; //| STATE_SYSTEM_ALERT_HIGH;

      return S_OK;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accKeyboardShortcut(
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ BSTR *   pszKeyboardShortcut) {
#ifdef DEBUG
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF)
      {
        dbg_printf("get_accKeyboardShortcut %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif // DEBUG

      ELEMENT(el)
      locker  l(blk);
      ustring txt = el->atts["accesskey"];
      if (txt.length() > 0) {
        if (txt.like(L"^*")) { txt = ustring(WCHARS("Ctrl+")) + txt(1); }
        else if (txt.like(L"_*")) { txt = ustring(WCHARS("Shift+")) + txt(1); }
        else if (!txt.like(L"!*")) { txt = ustring(WCHARS("Alt+")) + txt(); }
        *pszKeyboardShortcut = SysAllocStringLen(txt, (UINT)txt.length());
        return S_OK;
      }
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accFocus(
        /* [retval][out] */ VARIANT *pvarChild) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
#endif

      view *pv = blk->pview();
      if (pv == 0) return S_FALSE; // element not in the dom;

      if (!pv->focus_element)
        return S_FALSE;

      if (!pv->focus_element->belongs_to(*pv, blk, true))
        return S_FALSE;

      helement focus = pv->focus_element->a11y_get_focus(*pv);
      
      if (focus == this->blk) {
        pvarChild->vt   = VT_I4;
        pvarChild->lVal = CHILDID_SELF;
        return S_OK;
      }

      // <del>TODO: check it with MS Narrator (shall not fall into forever
      // loop)</del> Tested with MS Narrator, seems like OK.

      element *pf = focus;
      if (!pf) return S_FALSE;

      // if(::GetFocus() != (HWND)pf->doc()->hwindow())
      //  return S_FALSE;

      pvarChild->vt       = VT_DISPATCH;
      pvarChild->pdispVal = accessible::factory(pf);

      return S_OK;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accSelection(
                                                    /* [retval][out] */ VARIANT *pvarChildren) {
      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accDefaultAction(
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ BSTR *   pszDefaultAction) {
      /*const wchar* ps = "";

      CTL_TYPE ct = blk->ctl_type();
      switch( ct )
      {
        case CTL_EDIT             : return ROLE_SYSTEM_TEXT;
        case CTL_NUMERIC          : return ROLE_SYSTEM_TEXT;
        case CTL_BUTTON           : return ROLE_SYSTEM_PUSHBUTTON;
        case CTL_CHECKBOX         : return ROLE_SYSTEM_CHECKBUTTON;
        case CTL_RADIO            : return ROLE_SYSTEM_RADIOBUTTON;
        case CTL_SELECT_SINGLE    : return ROLE_SYSTEM_LIST;
        case CTL_SELECT_MULTIPLE  : return ROLE_SYSTEM_LIST;
        case CTL_DD_SELECT        : return ROLE_SYSTEM_COMBOBOX;
        case CTL_TEXTAREA         : return ROLE_SYSTEM_TEXT;
        case CTL_HTMLAREA         : return ROLE_SYSTEM_TEXT;
        case CTL_PASSWORD         : return ROLE_SYSTEM_TEXT;

        case CTL_PROGRESS         : return ROLE_SYSTEM_PROGRESSBAR;
        case CTL_SLIDER           : return ROLE_SYSTEM_SLIDER;
        case CTL_DECIMAL          : return ROLE_SYSTEM_TEXT;
        case CTL_CURRENCY         : return ROLE_SYSTEM_TEXT;
        case CTL_SCROLLBAR        : return ROLE_SYSTEM_SCROLLBAR;
        case CTL_MENUBAR          : return ROLE_SYSTEM_MENUBAR;
        case CTL_MENU             : return ROLE_SYSTEM_MENUPOPUP;
        case CTL_HYPERLINK        : return ROLE_SYSTEM_LINK;
        case CTL_MENUBUTTON       : return ROLE_SYSTEM_BUTTONDROPDOWN;
        case CTL_FRAME            : return ROLE_SYSTEM_PANE;
        case CTL_FRAMESET         : return ROLE_SYSTEM_PANE;
        default: break;
      }
      */
      // action = e.get_attribute("action");
      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accDoDefaultAction(
      /* [optional][in] */ VARIANT varChild)
    {
      ELEMENT(el);
      locker  l(blk);
      view *pv = el->pview();
      if (pv == 0) return S_FALSE; // element not in the dom;
      
      if (el->a11y_do_default_action(*pv))
        return S_OK;
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accSelect(
        /* [in] */ long              flagsSelect,
        /* [optional][in] */ VARIANT varChild) {

      return DISP_E_MEMBERNOTFOUND;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accLocation(
        /* [out] */ long *           pxLeft,
        /* [out] */ long *           pyTop,
        /* [out] */ long *           pcxWidth,
        /* [out] */ long *           pcyHeight,
        /* [optional][in] */ VARIANT varChild) {
#ifdef _DEBUG
      if (is_popup) is_popup = is_popup;
      if (varChild.vt == VT_I4 && !node::is_node_uid(varChild.lVal) && varChild.lVal != CHILDID_SELF)
      {
        dbg_printf("accLocation %d\n", varChild.lVal);
        return S_FALSE;
      }
#endif

      ELEMENT(el)

      locker l(blk);

      view *pv = el->pview();
      if (pv == 0) return S_FALSE; // element not in the dom;

      if (!el->a11y_is_visible(*pv)) return S_FALSE; // element invisible

      // HWND hwnd = (HWND)pv->doc()->hctl;
      // if( !::IsWindow(hwnd) )
      //  return S_FALSE;

      rect rc    = el->rbox(*pv) + pv->client_screen_pos();
      *pxLeft    = rc.left();
      *pyTop     = rc.top();
      *pcxWidth  = rc.width();
      *pcyHeight = rc.height();

      return S_OK;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accNavigate(
        /* [in] */ long              navDir,
        /* [optional][in] */ VARIANT varStart,
        /* [retval][out] */ VARIANT *pvarEndUpAt) 
    {
      if (!blk) return E_INVALIDARG;
      locker   l(blk);

      helement el = blk;

      if (varStart.vt == VT_I4 && node::is_node_uid(varStart.lVal))
        el = blk->get_element_by_uid(varStart.lVal);
      else if (varStart.vt == VT_I4 && varStart.lVal == CHILDID_SELF)
        el = blk;
      else if (varStart.vt != VT_I4)
        return E_INVALIDARG;                                         
      else {
        array<hnode> nodes;
        blk->a11y_get_children(nodes);
        if (varStart.lVal > 0 && varStart.lVal <= nodes.size()) {
          hnode hn = nodes[varStart.lVal - 1];
          if (!hn->is_element())
            return S_FALSE;
          el = hn.ptr_of<element>();
        }
        else
          return E_INVALIDARG;
      }

      element *nxt = el->a11y_navigate((element::NAVDIR)navDir);
      if (nxt) {
        pvarEndUpAt->vt = VT_DISPATCH;
        pvarEndUpAt->pdispVal = accessible::factory(nxt);
        return S_OK;
      }

      // debug_printf( "accNavigate failed on blk=%s navDir=%x\n",
      // symbol_name(blk->type).c_str(), navDir );

      pvarEndUpAt->vt = VT_EMPTY;
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accHitTest(
        /* [in] */ long              xLeft,
        /* [in] */ long              yTop,
        /* [retval][out] */ VARIANT *pvarChild) {
      view *pv = blk->pview();
      if (pv == 0) return S_FALSE; // element not in the dom;

      pvarChild->vt = VT_EMPTY;

#ifdef _DEBUG
      if (blk->tag == tag::T_POPUP) blk = blk;
      if (blk->tag == tag::T_OPTION) blk = blk;
#endif

      if (!blk->is_visible(*pv)) return S_FALSE; // element invisible

      // HWND hwnd = (HWND)pv->doc()->hctl;
      // if( !::IsWindow(hwnd) )
      //  return S_FALSE;

      point ptabs, pt;
      pt.x  = xLeft;
      pt.y  = yTop;
      ptabs = pt;

      pt -= blk->screen_pos(*pv);

      element *cb = blk->a11y_find_element(*pv, pt);
#ifdef _DEBUG
      if (cb) cb->dbg_report("a11y ");
#endif
      // narrator does not understand compound input elements :( ... why?
      // but inspector.exe needs them
      /*for( element *tcb = cb; tcb; tcb = tcb->parent)
      {
        uint t = tcb->ctl_type(*pv);
        if(t == CTL_NO)
          continue;
        if(t == CTL_UNKNOWN || t == CTL_FRAME || t == CTL_FRAMESET)
          break;
        cb = tcb;
        break;
      }*/

      if (cb == 0 || cb == blk) {
        pvarChild->vt   = VT_I4;
        pvarChild->lVal = 0;
      } else {
        pvarChild->vt       = VT_DISPATCH;
        pvarChild->pdispVal = factory(cb);
      }
      return S_OK;
    }


  };

  class accessible_text_node : public accessible_base {
    typedef accessible_base super;
    handle<text>            txt;

  public:
    accessible_text_node(text *n, uint rc = 1) : accessible_base(rc), txt(n) {}

    virtual /* [id][propget][hidden] */
        HRESULT STDMETHODCALLTYPE
                get_accChildCount(/* [retval][out] */ long *pcountChildren) {
      *pcountChildren = 0;
      return S_OK;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accChild(
        /* [in] */ VARIANT              varChild,
        /* [retval][out] */ IDispatch **ppdispChild) {

      if (varChild.vt != VT_I4) return E_INVALIDARG;

      *ppdispChild = 0;
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accRole(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ VARIANT *pvarRole) {

      pvarRole->vt   = VT_I4;
      pvarRole->lVal = ROLE_SYSTEM_STATICTEXT;
      return S_OK;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accName(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszName) {
      tool::wchars t = trim(txt->chars());
      *pszName       = SysAllocStringLen(t.start, (UINT)t.length);
      return S_OK;
      // return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accParent(
        /* [retval][out] */ IDispatch **ppdispParent) {
      element *pp = txt->parent;
      if (pp) {
        *ppdispParent = accessible::factory(pp);
        return S_OK;
      }

      //dbg_printf("accessible_text_node::get_accParent \n");

      *ppdispParent = 0;
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accState(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ VARIANT *pvarState) {

      if (varChild.vt != VT_I4 || varChild.lVal != 0) return S_FALSE;

      pvarState->vt   = VT_I4;
      pvarState->lVal = 0;

      return S_OK;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accLocation(
        /* [out] */ long *           pxLeft,
        /* [out] */ long *           pyTop,
        /* [out] */ long *           pcxWidth,
        /* [out] */ long *           pcyHeight,
        /* [optional][in] */ VARIANT varChild) {
      if (varChild.vt != VT_I4 || varChild.lVal != 0 || !txt || !txt->parent)
        return S_FALSE;

      view *pv = txt->parent->pview();
      if (pv == 0) return S_FALSE; // element not in the dom;

      if (!txt->parent->is_visible(*pv)) return S_FALSE; // element invisible

      rect rc    = txt->rbox(*pv) + pv->client_screen_pos();
      *pxLeft    = rc.left();
      *pyTop     = rc.top();
      *pcxWidth  = rc.width();
      *pcyHeight = rc.height();

      return S_OK;

#pragma TODO("check why this does not work well with comments")
      /*bookmark bm_start = txt->start_caret_pos(*pv);
      bookmark bm_end   = txt->end_caret_pos(*pv);
               bm_end.after_it = false;
               //bm_end.pos = bm_end.pos - 1;

      rect rcs = bm_start.get_caret_rect(*pv);
      rect rce = bm_end.get_caret_rect(*pv);

      rect rc;
      

      if( rcs.y() != rce.y() )
        rc = txt->parent->nearest_box()->rendering_box(*pv,element::TO_SCREEN);
      else {
        point sp = pv->client_screen_pos();
        rc = (rcs | rce) + sp;
      }

      *pxLeft = rc.left();
      *pyTop = rc.top();
      *pcxWidth = rc.width();
      *pcyHeight = rc.height();
      

      return S_OK; */
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accValue(
        /* [optional][in] */ VARIANT varChild,
        /* [retval][out] */ BSTR *   pszValue) 
    {
      //if (varChild.vt != VT_I4 || varChild.lVal != 0 || !txt || !txt->parent)
      return S_FALSE;

      /*tool::wchars t = tool::trim(txt->chars());
      *pszValue = SysAllocStringLen(t.start, (UINT)t.length);
      return S_OK;*/
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accDescription(
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ BSTR *   pszDescription) {
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accKeyboardShortcut(
                                                    /* [optional][in] */ VARIANT varChild,
                                                    /* [retval][out] */ BSTR *   pszKeyboardShortcut) {
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE get_accFocus(
        /* [retval][out] */ __RPC__out VARIANT *pvarChild) {
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accSelection(
                                                    /* [retval][out] */ __RPC__out VARIANT *pvarChildren) {
      return S_FALSE;
    }

    virtual /* [id][propget][hidden] */ HRESULT STDMETHODCALLTYPE
                                                get_accDefaultAction(
                                                    /* [optional][in] */ VARIANT             varChild,
                                                    /* [retval][out] */ __RPC__deref_out_opt BSTR *pszDefaultAction) {
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accSelect(
        /* [in] */ long              flagsSelect,
        /* [optional][in] */ VARIANT varChild) {
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accNavigate(
        /* [in] */ long                navDir,
        /* [optional][in] */ VARIANT   varStart,
        /* [retval][out] */ __RPC__out VARIANT *pvarEndUpAt) {
      return S_FALSE;
    }

    virtual /* [id][hidden] */ HRESULT STDMETHODCALLTYPE accHitTest(
        /* [in] */ long                xLeft,
        /* [in] */ long                yTop,
        /* [retval][out] */ __RPC__out VARIANT *pvarChild) {
      pvarChild->vt   = VT_I4;
      pvarChild->lVal = 0;
      return S_OK;
    }
  };

  inline accessible *accessible::factory(element *b, uint rc) {
    /*int role = get_role(b);
    switch (role) {
    case ROLE_SYSTEM_TOOLTIP:
    case ROLE_SYSTEM_MENUITEM: return new accessible_text(b, rc);
    }
    */
    return new accessible(b, rc);
  }

  inline accessible_base *accessible::factory(node *n, uint rc) {
    if (n->is_element())
      return factory(static_cast<element *>(n));
    else if (n->is_text())
      return new accessible_text_node(static_cast<text *>(n));
    assert(false);
    return 0;
  }
} // namespace html

#pragma comment(lib, "oleacc.lib")

#endif