
// UIAutomation/IAccessible layer provided by Chris Canossi the Great.

#include "win.h"

#include "win-accessible.h"

namespace mswin
{

#if defined(ACCESSIBLE)
  LRESULT window::handle_get_object(WPARAM wParam, LPARAM lParam, BOOL& handled)
  {
    critical_section cs(guard);
    html::helement   el;

    LONG what = (LONG)(DWORD)lParam;

#if defined(USE_UIAUTOMATION)
    if (use_uiautomation && (what == UiaRootObjectId) && _UiaReturnRawElementProvider) {
      LRESULT lr = 0;
      this->_is_uiautomation_active = TRUE;
      el = this->doc();
      if (!el) return 0;
      IRawElementProviderSimple *provider = html::accessible_root_provider::make(get_hwnd(), el);
      lr = _UiaReturnRawElementProvider()(get_hwnd(), wParam, lParam, provider);
      provider->Release();
      handled = TRUE;
      return lr;
    }
#endif
    
    this->_is_iaccessible_active = TRUE;

    if (what == OBJID_CLIENT) {

      el = this->doc();

      if (!el) return 0;

      com::asset<IUnknown> pa = new html::accessible(el);
      if (!pa) return 0;

      //el->dbg_report("GETOBJECT/OBJID_CLIENT");

      LRESULT lr = LresultFromObject(IID_IAccessible, wParam, pa);

      if (lr == 0x800401F0L) // CO_E_NOTINITIALIZED
      {
        // why does this happen? does it create new thread?
        OleInitialize(0);
        _need_OleUninitialize = TRUE;
        lr = LresultFromObject(IID_IAccessible, wParam, pa);
        assert(lr != 0x800401F0L);
      }

      //dbg_printf("WM_GETOBJECT hwnd=%x lr=%x\n", hwnd, lr);
      pa->Release();

      handled = TRUE;
      return lr;
    }
    else {
      //dbg_printf("GETOBJECT %d\n", what);
    }
    return 0;
  }
}

#endif

#ifdef USE_UIAUTOMATION

//#pragma comment(lib, "uiautomationcore.lib")

DLOADV(_UiaReturnRawElementProvider, UiaReturnRawElementProvider,
       uiautomationcore.dll,
       LRESULT(WINAPI *)(HWND hwnd, WPARAM wParam, LPARAM lParam,
                         IRawElementProviderSimple *el));

DLOADV(_UiaHostProviderFromHwnd, UiaHostProviderFromHwnd, uiautomationcore.dll,
       HRESULT(WINAPI *)(HWND hwnd, IRawElementProviderSimple **pProvider));

DLOADV(_UiaRaiseAutomationEvent, UiaRaiseAutomationEvent, uiautomationcore.dll,
       HRESULT(WINAPI *)(IRawElementProviderSimple *pProvider, EVENTID id));

DLOADV(_UiaClientsAreListening, UiaClientsAreListening, uiautomationcore.dll,
       BOOL(WINAPI *)());

namespace {
  using namespace html;

  struct locker {
    html::view *view_ptr;

    locker(html::element *_element_ptr) : view_ptr(0) {
      if (_element_ptr) {
        view_ptr = _element_ptr->pview();

        if (view_ptr) view_ptr->guard.lock();
      }
    }

    ~locker() {
      if (view_ptr) view_ptr->guard.unlock();
    }
  };

  inline void set_variant(VARIANT *variant_ptr, HWND hwnd) {
    variant_ptr->vt     = VT_I4;
    variant_ptr->intVal = (int)(int_ptr)hwnd;
  }

  inline void set_variant(VARIANT *variant_ptr, bool value_boolean) {
    variant_ptr->vt      = VT_BOOL;
    variant_ptr->boolVal = value_boolean ? VARIANT_TRUE : VARIANT_FALSE;
  }

  inline void set_variant(VARIANT *variant_ptr, const ustring &value_str) {
    if (!value_str.is_empty()) {
      variant_ptr->vt = VT_BSTR;
      variant_ptr->bstrVal =
          ::SysAllocStringLen(value_str, (UINT)value_str.length());
    }
  }

  inline void set_variant(VARIANT *variant_ptr, int ival) {
    variant_ptr->vt = VT_I4;
    variant_ptr->intVal = ival;
  }


  inline void set_variant(VARIANT *variant_ptr, LPCWSTR value_str,
                          uint length) {
    if (length) {
      variant_ptr->vt      = VT_BSTR;
      variant_ptr->bstrVal = ::SysAllocStringLen(value_str, length);
    }
  }

  //----------------------------------------------------------------------------------------------
  // A helper function to create a SafeArray Version of an int array of a
  // specified length
  //----------------------------------------------------------------------------------------------
  inline SAFEARRAY *BuildIntSafeArray(/*_In_reads_(length)*/ const int *data,
                                      _In_ int length) {
    auto safe_array_ptr = SafeArrayCreateVector(VT_I4, 0, length);

    if (!safe_array_ptr) return safe_array_ptr;

    for (long index = 0; index < length; index++) {
      if (FAILED(SafeArrayPutElement(safe_array_ptr, &index,
                                     (void *)&(data[index])))) {
        SafeArrayDestroy(safe_array_ptr);
        safe_array_ptr = nullptr;
        break;
      }
    }

    return safe_array_ptr;
  }

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  static uint get_role(html::element *_element_ptr) {
    locker l(_element_ptr);

    if (!l.view_ptr) return html::CTL_NO;

    // 1) first try input elements.

    html::CTL_TYPE ct = _element_ptr->ctl_type(*l.view_ptr);

    switch (ct) {
    case html::CTL_EDIT: return ROLE_SYSTEM_TEXT;
    case html::CTL_NUMERIC:
      return ROLE_SYSTEM_TEXT;
      // case html::CTL_CLICKABLE        : return ROLE_SYSTEM_PUSHBUTTON;
    case html::CTL_BUTTON: return ROLE_SYSTEM_PUSHBUTTON;
    case html::CTL_CHECKBOX: return ROLE_SYSTEM_CHECKBUTTON;
    case html::CTL_RADIO: return ROLE_SYSTEM_RADIOBUTTON;
    case html::CTL_SELECT_SINGLE: return ROLE_SYSTEM_LIST;
    case html::CTL_SELECT_MULTIPLE: return ROLE_SYSTEM_LIST;
    case html::CTL_DD_SELECT: return ROLE_SYSTEM_COMBOBOX;
    case html::CTL_TEXTAREA: return ROLE_SYSTEM_TEXT;
    case html::CTL_HTMLAREA: return ROLE_SYSTEM_TEXT;
    case html::CTL_PASSWORD: return ROLE_SYSTEM_TEXT;

    case html::CTL_PROGRESS: return ROLE_SYSTEM_PROGRESSBAR;
    case html::CTL_SLIDER: return ROLE_SYSTEM_SLIDER;
    case html::CTL_DECIMAL: return ROLE_SYSTEM_TEXT;
    case html::CTL_CURRENCY: return ROLE_SYSTEM_TEXT;
    case html::CTL_SCROLLBAR: return ROLE_SYSTEM_SCROLLBAR;
    case html::CTL_MENUBAR: return ROLE_SYSTEM_MENUBAR;
    case html::CTL_MENU: return ROLE_SYSTEM_MENUPOPUP;
    case html::CTL_HYPERLINK: return ROLE_SYSTEM_LINK;
    case html::CTL_MENUBUTTON: return ROLE_SYSTEM_BUTTONDROPDOWN;
    case html::CTL_FRAME: return ROLE_SYSTEM_PANE;
    case html::CTL_FRAMESET: return ROLE_SYSTEM_PANE;
    case html::CTL_TOOLTIP: return ROLE_SYSTEM_TOOLTIP;
    default: break;
    }

    // 2) try hyperlink and known role attribute values
    if (_element_ptr->atts.exist(attr::a_href)) return ROLE_SYSTEM_LINK;

    if (_element_ptr->tag == tag::T_LI && _element_ptr->parent &&
        _element_ptr->parent->ctl_type(*l.view_ptr) == html::CTL_MENU)
      return ROLE_SYSTEM_MENUITEM;

    ustring role = _element_ptr->atts("role");
    if (role.length()) {
      if (role == WCHARS("option")) return ROLE_SYSTEM_LISTITEM;
      if (role == WCHARS("statusbar")) return ROLE_SYSTEM_STATUSBAR;
      if (role == WCHARS("toolbar")) return ROLE_SYSTEM_TOOLBAR;
      if (role == WCHARS("panel")) return ROLE_SYSTEM_PANE;
      if (role == WCHARS("group")) return ROLE_SYSTEM_GROUPING;
      if (role == WCHARS("menu-item")) return ROLE_SYSTEM_MENUITEM;
      if (role == WCHARS("dialog")) return ROLE_SYSTEM_DIALOG;
      if (role == WCHARS("page-tab")) return ROLE_SYSTEM_PAGETAB;
      if (role == WCHARS("page-tab-list")) return ROLE_SYSTEM_PAGETABLIST;
    }

    // ROLE_SYSTEM_PAGETAB
    // The object represents a page tab. The only child of a page tab control is
    // a ROLE_SYSTEM_GROUPING object that contains the contents of the associated
    // page.  ROLE_SYSTEM_PAGETABLIST  The object represents a container of page
    // tab controls.

    if (_element_ptr->state.popup()) return ROLE_SYSTEM_MENUPOPUP;

    // 3) try specific tags.
    switch (_element_ptr->tag) {
    case tag::T_TFOOT:
    case tag::T_THEAD:
    case tag::T_TBODY: return ROLE_SYSTEM_TABLE;
    case tag::T_TABLE: return ROLE_SYSTEM_PANE;
    case tag::T_TR: return ROLE_SYSTEM_ROW;
    case tag::T_TD:
    case tag::T_TH: return ROLE_SYSTEM_CELL;
    case tag::T_OPTION:
      return ROLE_SYSTEM_LISTITEM;
      // case T_FIELDSET: return ROLE_SYSTEM_GROUPING; // for some reasons
      // narrator don't go inside GROUPING...
    case tag::T_FIELDSET: return ROLE_SYSTEM_PANE;
    case tag::T_IMG: return ROLE_SYSTEM_GRAPHIC;
    case tag::T_IFRAME: return ROLE_SYSTEM_PANE;
    case tag::T_POPUP: return ROLE_SYSTEM_DIALOG;
    case tag::T_MENU: return ROLE_SYSTEM_MENUPOPUP;
    case tag::T_LI:
      if (_element_ptr->parent && _element_ptr->parent->tag == tag::T_MENU)
        return ROLE_SYSTEM_MENUITEM;
      else
        return ROLE_SYSTEM_LISTITEM;
      break;
    case tag::T_DD:
    case tag::T_DT: return ROLE_SYSTEM_LISTITEM;

    case tag::T_DL:
    case tag::T_UL:
    case tag::T_OL:
      return ROLE_SYSTEM_LIST;

      // case tag::T_HTML:
      //  if(!_element_ptr->parent)
      //     return ROLE_SYSTEM_CLIENT;
      //  else
      //    return ROLE_SYSTEM_DOCUMENT;
    }
    
    if (_element_ptr->a11y_is_text()) return ROLE_SYSTEM_STATICTEXT;

    return ROLE_SYSTEM_PANE;
  }

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  static uint get_UIA_control_type(html::element *_element_ptr) {
    locker l(_element_ptr);

    if (!l.view_ptr) return html::CTL_NO;

    // 1) first try input elements.

    html::CTL_TYPE ct = _element_ptr->ctl_type(*l.view_ptr);
    switch (ct) {
    case html::CTL_EDIT: return UIA_EditControlTypeId;
    case html::CTL_NUMERIC: return UIA_EditControlTypeId;
    case html::CTL_BUTTON: return UIA_ButtonControlTypeId;
    case html::CTL_CHECKBOX: return UIA_CheckBoxControlTypeId;
    case html::CTL_RADIO: return UIA_RadioButtonControlTypeId;
    case html::CTL_SELECT_SINGLE: return UIA_ListItemControlTypeId;
    case html::CTL_SELECT_MULTIPLE: return UIA_ListItemControlTypeId;
    case html::CTL_DD_SELECT: return UIA_ComboBoxControlTypeId;
    case html::CTL_TEXTAREA: return UIA_EditControlTypeId;
    case html::CTL_HTMLAREA: return UIA_EditControlTypeId;
    case html::CTL_PASSWORD: return UIA_EditControlTypeId;
    case html::CTL_PROGRESS: return UIA_ProgressBarControlTypeId;
    case html::CTL_SLIDER: return UIA_SliderControlTypeId;
    case html::CTL_DECIMAL: return UIA_TextControlTypeId;
    case html::CTL_CURRENCY: return UIA_TextControlTypeId;
    case html::CTL_SCROLLBAR: return UIA_ScrollBarControlTypeId;
    case html::CTL_MENUBAR: return UIA_MenuBarControlTypeId;
    case html::CTL_MENU: return UIA_MenuControlTypeId;
    case html::CTL_HYPERLINK: return UIA_HyperlinkControlTypeId;
    case html::CTL_MENUBUTTON: return UIA_MenuItemControlTypeId;
    case html::CTL_FRAME: return UIA_PaneControlTypeId;
    case html::CTL_FRAMESET: return UIA_PaneControlTypeId;
    case html::CTL_TOOLTIP: return UIA_ToolTipControlTypeId;
    default: break;
    }

    // 2) try hyperlink and known role attribute values
    if (_element_ptr->atts.exist(attr::a_href))
      return UIA_HyperlinkControlTypeId;

    if (_element_ptr->tag == tag::T_LI && _element_ptr->parent &&
        _element_ptr->parent->ctl_type(*l.view_ptr) == html::CTL_MENU)
      return UIA_MenuItemControlTypeId;

    ustring role = _element_ptr->atts("role");

    if (role.length()) {
      if (role == WCHARS("option")) return UIA_ListItemControlTypeId;
      if (role == WCHARS("statusbar")) return UIA_StatusBarControlTypeId;
      if (role == WCHARS("toolbar")) return UIA_ToolBarControlTypeId;
      if (role == WCHARS("panel")) return UIA_PaneControlTypeId;
      if (role == WCHARS("group")) return UIA_GroupControlTypeId;
      if (role == WCHARS("menu-item")) return UIA_MenuItemControlTypeId;
      if (role == WCHARS("dialog")) return UIA_WindowControlTypeId;
      if (role == WCHARS("page-tab")) return UIA_TabItemControlTypeId;
      if (role == WCHARS("page-tab-list")) return UIA_TabControlTypeId;
    }

    if (_element_ptr->state.popup()) return UIA_MenuControlTypeId;

    // 3) try specific tags.
    switch (_element_ptr->tag) {
    case tag::T_TFOOT:
    case tag::T_THEAD:
    case tag::T_TBODY: return UIA_TableControlTypeId;
    case tag::T_TABLE: return UIA_PaneControlTypeId;
    case tag::T_TR: return UIA_GroupControlTypeId;
    case tag::T_TD:
    case tag::T_TH: return UIA_TextControlTypeId;
    case tag::T_OPTION: return UIA_ListItemControlTypeId;
    case tag::T_FIELDSET: return UIA_PaneControlTypeId;
    case tag::T_IMG: return UIA_ImageControlTypeId;
    case tag::T_IFRAME: return UIA_PaneControlTypeId;
    case tag::T_POPUP: return UIA_WindowControlTypeId;
    case tag::T_MENU: return UIA_MenuControlTypeId;
    case tag::T_LI:
      if (_element_ptr->parent && _element_ptr->parent->tag == tag::T_MENU)
        return UIA_MenuItemControlTypeId;
      else
        return UIA_ListItemControlTypeId;
      break;
    case tag::T_DD:
    case tag::T_DT: return UIA_ListItemControlTypeId;

    case tag::T_DL:
    case tag::T_UL:
    case tag::T_OL:
      return UIA_ListControlTypeId;

      // case tag::T_HTML:
      //  if(!_element_ptr->parent)
      //     return ROLE_SYSTEM_CLIENT;
      //  else
      //    return ROLE_SYSTEM_DOCUMENT;
    }
        
    if (_element_ptr->a11y_is_text()) return UIA_TextControlTypeId;

    return UIA_PaneControlTypeId;
  }

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  void set_value(element *element_ptr, LPCWSTR value_str) {
    do {
      locker l(element_ptr);

      if (!l.view_ptr) break;

      const int control_type = element_ptr->ctl_type(*l.view_ptr);

      if (control_type == html::CTL_NO) break;

      if (control_type == html::CTL_PASSWORD) // this is password
      {
        /* what's this?
          _bstr_t password = L"********";
          value_str =  password.Detach();
         */
        break;
      }

      /*if(control_type == html::CTL_BUTTON) // this is control
      {
          tool::value v;
          html::view* view_ptr = element_ptr->pview();

          if(view_ptr && element_ptr->get_value(*view_ptr,v))
          {
              ustring txt = v.to_string();

              value_str =  SysAllocStringLen(txt, (UINT)txt.length());
              break;
          }
      }*/

      tool::value val(value_str);
      element_ptr->set_value(*l.view_ptr, val);

      /*if(! success)
      {
        const auto success =
      }*/

    } while (false);
  }

  //----------------------------------------------------------------------------------------------
  //----------------------------------------------------------------------------------------------
  BSTR get_value(element *element_ptr) {
    BSTR value_str = nullptr;
    do {
      locker l(element_ptr);
      view * view_ptr = element_ptr->pview();

      if (!view_ptr) break;

      const int control_type = element_ptr->ctl_type(*view_ptr);

      if (control_type == html::CTL_NO) break;

      if (control_type == html::CTL_PASSWORD) // this is password
      {
        const ustring password = L"********";
        value_str = SysAllocStringLen(password, (UINT)password.length());
        break;
      }

      if (control_type != html::CTL_BUTTON) // this is control
      {
        tool::value v;
        if (element_ptr->get_value(*view_ptr, v)) {
          ustring txt = v.to_string();
          value_str   = SysAllocStringLen(txt, (UINT)txt.length());
          break;
        }
        
      }
    } while (false);

    return value_str;
  }

  //---------------------------------------------------------------------------------------------
  //---------------------------------------------------------------------------------------------
  bool is_value_type_control(element *element_ptr) {
    bool is_value_type = false;

    view *view_ptr = element_ptr->pview();

    if (!view_ptr) return false;

    const html::CTL_TYPE control_type = element_ptr->ctl_type(*view_ptr);

    switch (control_type) {
    case html::CTL_EDIT:
    case html::CTL_NUMERIC:
    case html::CTL_CHECKBOX:
    case html::CTL_RADIO:
    case html::CTL_TEXTAREA:
    case html::CTL_HTMLAREA:
    case html::CTL_PASSWORD:
    case html::CTL_SLIDER:
    case html::CTL_DECIMAL:
    case html::CTL_CURRENCY: is_value_type = true; break;
    }

    return is_value_type;
  }

  //---------------------------------------------------------------------------------------------
  //---------------------------------------------------------------------------------------------
  bool is_toggle_type_control(element *element_ptr) {
    bool is_toggle_type = false;

    view *view_ptr = element_ptr->pview();

    if (!view_ptr) return false;

    const html::CTL_TYPE control_type = element_ptr->ctl_type(*view_ptr);

    switch (control_type) {
    case html::CTL_CHECKBOX: is_toggle_type = true; break;
    }

    return is_toggle_type;
  }

  //---------------------------------------------------------------------------------------------
  //---------------------------------------------------------------------------------------------
  bool is_selection_item_type_control(element *element_ptr) {
    bool is_toggle_type = false;

    view *view_ptr = element_ptr->pview();

    if (!view_ptr) return false;

    const html::CTL_TYPE control_type = element_ptr->ctl_type(*view_ptr);

    switch (control_type) {
    case html::CTL_RADIO: is_toggle_type = true; break;
    }

    return is_toggle_type;
  }

  //--------------------------------------------------------------------------------------------------
  //--------------------------------------------------------------------------------------------------
  bool is_invokable_control_type(element *element_ptr) {
    bool is_invokable = false;

    view *view_ptr = element_ptr->pview();

    if (!view_ptr) return false;

    const html::CTL_TYPE control_type = element_ptr->ctl_type(*view_ptr);

    switch (control_type) {
    case html::CTL_BUTTON:
    case html::CTL_CLICKABLE:
    case html::CTL_MENUBUTTON:
    case html::CTL_RADIO:
    case html::CTL_CHECKBOX:
    case html::CTL_HYPERLINK: is_invokable = true; break;
    }

    return is_invokable;
  }

  //--------------------------------------------------------------------------------------------------
  //--------------------------------------------------------------------------------------------------
  /*bool is_like_text( element* b )
  {
      if(b->is_of_type<text_block>())
          return true;

      if(b->nodes.size() == 1 && b->nodes[0]->is_text())
          return true;

      return false;
  }*/
} // namespace

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
html::accessible_sciter_provider::accessible_sciter_provider(
    accessible_root_provider *root_ptr, element *elment_ptr,
    uint ref_count /*= 1*/)
    : _element_ptr(elment_ptr), _root_ptr(root_ptr), _refcount(ref_count) {}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
html::accessible_sciter_provider *
html::accessible_sciter_provider::make(accessible_root_provider *root_ptr,
                                       element *                 element_ptr) {
  return new accessible_sciter_provider(root_ptr, element_ptr);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::accessible_sciter_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::accessible_sciter_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::QueryInterface(
    REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(IRawElementProviderSimple))
    *interface_ptr = static_cast<IRawElementProviderSimple *>(this);
  else if (iid == __uuidof(IRawElementProviderFragment))
    *interface_ptr = static_cast<IRawElementProviderFragment *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr = static_cast<IUnknown *>(static_cast<IRawElementProviderSimple *>(this));
  else if (iid == __uuidof(IRawElementProviderFragmentRoot)) {
    //*interface_ptr = static_cast<IRawElementProviderFragmentRoot *>(this);
    dbg_printf("wants IRawElementProviderFragmentRoot");
    return E_NOINTERFACE;
  }
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::GetPatternProvider(
    PATTERNID patternId, __RPC__deref_out_opt IUnknown **pRetVal) {
  *pRetVal = nullptr;

  switch (patternId) {
  case UIA_InvokePatternId: {
    if (is_invokable_control_type(_element_ptr))
      *pRetVal = invoke_provider::make(_element_ptr);
  } break;

  case UIA_LegacyIAccessiblePatternId: {
    *pRetVal = legacy_iaccessible_provider::make(_element_ptr);
  } break;

  case UIA_ValuePatternId: {
    if (is_value_type_control(_element_ptr))
      *pRetVal = value_provider::make(_element_ptr);
  } break;

  case UIA_TogglePatternId: {
    if (is_toggle_type_control(_element_ptr))
      *pRetVal = toggle_proivder::make(_element_ptr);
  } break;

  case UIA_SelectionItemPatternId: {
    if (is_selection_item_type_control(_element_ptr))
      *pRetVal = selection_item_provider::make(_element_ptr, this);
  } break;
  }

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::GetPropertyValue(
    PROPERTYID propertyId, __RPC__out VARIANT *pRetVal) {
  pRetVal->vt = VT_EMPTY;

  locker l(_element_ptr);

  if (!l.view_ptr) return UIA_E_ELEMENTNOTAVAILABLE;

  switch (propertyId) {
  case UIA_IsInvokePatternAvailablePropertyId: {
    if (is_invokable_control_type(_element_ptr)) set_variant(pRetVal, true);
  } break;

  case UIA_IsControlElementPropertyId:
  case UIA_IsContentElementPropertyId:
  case UIA_IsLegacyIAccessiblePatternAvailablePropertyId: {
    set_variant(pRetVal, true);
  } break;

  case UIA_IsKeyboardFocusablePropertyId: {
    if (_element_ptr->get_state(false).focusable()) set_variant(pRetVal, true);
  } break;

  case UIA_IsSelectionItemPatternAvailablePropertyId: {
    if (is_selection_item_type_control(_element_ptr))
      set_variant(pRetVal, true);
  } break;

  case UIA_IsTogglePatternAvailablePropertyId: {
    if (is_toggle_type_control(_element_ptr)) set_variant(pRetVal, true);
  } break;

  case UIA_IsPasswordPropertyId: {
    const auto control_type = _element_ptr->ctl_type(*l.view_ptr);

    if (html::CTL_PASSWORD == control_type) set_variant(pRetVal, true);
  } break;

  case UIA_IsEnabledPropertyId: {
    const auto control_enabled = !_element_ptr->is_disabled();
    set_variant(pRetVal, control_enabled);
  } break;

  case UIA_IsOffscreenPropertyId: {
    UiaRect boundingRect = {0};
    if (SUCCEEDED(get_BoundingRectangle(&boundingRect)))
      set_variant(pRetVal, UiaRectIsEmpty(boundingRect));
  } break;

  case UIA_IsValuePatternAvailablePropertyId: {
    set_variant(pRetVal, is_value_type_control(_element_ptr));
  } break;

  case UIA_ControlTypePropertyId: {
    pRetVal->vt   = VT_I4;
    pRetVal->lVal = get_UIA_control_type(_element_ptr);
  } break;

  case UIA_NamePropertyId: {
    ustring name;

    if (_element_ptr->a11y_get_name(*l.view_ptr, name)) {
      if (!name.is_empty()) {
        set_variant(pRetVal, name);
        break;
      }
    }

    name = _element_ptr->attr_title();

    if (!name.is_empty()) {
      set_variant(pRetVal, name);
      break;
    }

    if (_element_ptr->a11y_is_text()) {
      array<wchar> text;
      _element_ptr->get_ui_text(*l.view_ptr, text);
      name = trim(text());
      if (!name.is_empty()) {
        set_variant(pRetVal, name);
        break;
      }
    }
  } break;

  case UIA_HasKeyboardFocusPropertyId: {
    const auto has_focus = _element_ptr->get_state(false).focus();
    set_variant(pRetVal, has_focus);
  } break;

  case UIA_HelpTextPropertyId: {
    ustring text = _element_ptr->atts[attr::a_alt];

    if (!text.is_empty()) set_variant(pRetVal, text);
  } break;

  case UIA_AutomationIdPropertyId: {
    ustring name;
    name = _element_ptr->attr_id();

    if (!name.is_empty()) {
      set_variant(pRetVal, name);
      break;
    }

    name = _element_ptr->attr_name();

    if (!name.is_empty()) {
      set_variant(pRetVal, name);
      break;
    }
  } break;

  case UIA_ClassNamePropertyId: {
    ustring class_str;
    class_str = _element_ptr->attr_class();

    if (!class_str.is_empty()) set_variant(pRetVal, class_str);
  } break;

  case UIA_AriaPropertiesPropertyId: {
    // TODO: Return Aria prop-string here...
  } break;

  case UIA_AriaRolePropertyId: {
    ustring name;
    name = _element_ptr->attr_role();

    if (!name.is_empty()) set_variant(pRetVal, name);
  } break;

  case UIA_DescribedByPropertyId: {
    ustring text = _element_ptr->atts[attr::a_aria_describedby];

    if (!text.is_empty()) set_variant(pRetVal, text);
  } break;

  case UIA_LiveSettingPropertyId:
  {
    auto la =_element_ptr->get_a11y_live(*l.view_ptr);
    // Return whichever of Polite or Assertive is most appropriate for your scenario. 
    // Note that a value of zero, (for "Off"), could also be returned, but typically if 
    // an element is ever a LiveRegion, it's always either Assertive or Polite. 

    static_assert(LiveSetting::Assertive == element::A11Y_LIVE_ASSERTIVE,"mismatch");
    set_variant(pRetVal, int(la));
    //dbg_printf("UIA_LiveSettingPropertyId %d\n",la);
  } break;
  };

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_sciter_provider::get_HostRawElementProvider(
    __RPC__deref_out_opt IRawElementProviderSimple **pRetVal) {

  /* WTF?? Breaks operation: if (_UiaHostProviderFromHwnd) {
    if(auto pv = _element_ptr->pview())
      return _UiaHostProviderFromHwnd()(pv->get_hwnd(), pRetVal);
  }*/

  *pRetVal = nullptr;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::get_ProviderOptions(
    __RPC__out enum ProviderOptions *pRetVal) {
  *pRetVal =
      ProviderOptions_ServerSideProvider | ProviderOptions_UseComThreading | ProviderOptions_ProviderOwnsSetFocus;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_sciter_provider::GetEmbeddedFragmentRoots(
    __RPC__deref_out_opt SAFEARRAY **pRetVal) {
  *pRetVal = nullptr;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::GetRuntimeId(
    __RPC__deref_out_opt SAFEARRAY **pRetVal) {
  HRESULT hr = S_OK;

  int id[] = {UiaAppendRuntimeId, _element_ptr->uid};

  *pRetVal = BuildIntSafeArray(id, _countof(id));

  if (!*pRetVal) hr = E_OUTOFMEMORY;

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::Navigate(
    enum NavigateDirection direction,
    __RPC__deref_out_opt IRawElementProviderFragment **pRetVal) {
  *pRetVal = nullptr;

  HRESULT hr = UIA_E_ELEMENTNOTAVAILABLE;

  int dir = -1;

  switch (direction) {
  case NavigateDirection_Parent: 
    if (_root_ptr)
      hr = _root_ptr->QueryInterface(pRetVal);
    else {
      helement p = _element_ptr->parent;
      if (p) {
        HWND hwnd = p->pview()->get_hwnd();
        *pRetVal = accessible_root_provider::make(hwnd, p);
        hr = S_OK;
        return hr;
      }
    }
    break;
  case NavigateDirection_NextSibling: dir = element::DIR_NEXT; break;
  case NavigateDirection_PreviousSibling: dir = element::DIR_PREVIOUS; break;
  case NavigateDirection_FirstChild: dir = element::DIR_FIRSTCHILD; break;
  case NavigateDirection_LastChild: dir = element::DIR_LASTCHILD; break;
  }

  locker   l(_element_ptr);
  element *navigate_ptr = _element_ptr->a11y_navigate((element::NAVDIR)dir);

  if (navigate_ptr) {
    *pRetVal = make(_root_ptr, navigate_ptr);
    hr       = S_OK;
  }
  
  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::SetFocus() {
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_sciter_provider::get_BoundingRectangle(
    __RPC__out struct UiaRect *pRetVal) {
  locker l(_element_ptr);
  // element* el = _element_ptr;

  if (!l.view_ptr) return S_FALSE; // element not in the dom;

  if (!_element_ptr->a11y_is_visible(*l.view_ptr)) {
    static const UiaRect zero_rect = {0};
    *pRetVal                       = zero_rect;
    return S_OK;
  }

  rect ref_count =
      _element_ptr->rbox(*l.view_ptr) + l.view_ptr->client_screen_pos();

  pRetVal->top    = ref_count.top();
  pRetVal->left   = ref_count.left();
  pRetVal->width  = ref_count.width();
  pRetVal->height = ref_count.height();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_sciter_provider::get_FragmentRoot(
    __RPC__deref_out_opt IRawElementProviderFragmentRoot **pRetVal) {
  const HRESULT hr = _root_ptr->QueryInterface(pRetVal);
  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
html::accessible_root_provider::accessible_root_provider(HWND     hwnd,
                                                         element *element_ptr,
                                                         uint ref_count /*= 1*/)
    : _element_ptr(element_ptr), _refcount(ref_count), _hwnd(hwnd) {}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
html::accessible_root_provider *
html::accessible_root_provider::make(HWND hwnd, element *element_ptr) {
  return new accessible_root_provider(hwnd, element_ptr);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::accessible_root_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::accessible_root_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::QueryInterface(
    REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(IRawElementProviderSimple))
    *interface_ptr = static_cast<IRawElementProviderSimple *>(this);
  else if (iid == __uuidof(IRawElementProviderFragment))
    *interface_ptr = static_cast<IRawElementProviderFragment *>(this);
  else if (iid == __uuidof(IRawElementProviderFragmentRoot))
    *interface_ptr = static_cast<IRawElementProviderFragmentRoot *>(this);
  //else if (iid == __uuidof(ILegacyIAccessibleProvider))
  //  *interface_ptr = ------; Narrator does not call this
  //else if (iid == __uuidof(IAccessible))
  //  *interface_ptr = ------; Narrator does not call this
  else if (iid == __uuidof(IUnknown))
    *interface_ptr = static_cast<IUnknown *>(static_cast<IRawElementProviderSimple *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::GetPatternProvider(
    PATTERNID patternId, __RPC__deref_out_opt IUnknown **pRetVal) {
  // Clear out param
  *pRetVal = nullptr;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::GetPropertyValue(
    PROPERTYID propertyId, __RPC__out VARIANT *pRetVal) {
  switch (propertyId) {
  case UIA_ControlTypePropertyId: {
    pRetVal->vt   = VT_I4;
    pRetVal->lVal = UIA_PaneControlTypeId;
  } break;

  case UIA_AutomationIdPropertyId: {
    WCHAR      buffer[255];
    const auto window_text_length =
        GetWindowText(_hwnd, &buffer[0], items_in(buffer));

    if (window_text_length)
      set_variant(pRetVal, &buffer[0], (uint)window_text_length);
  } break;

  case UIA_IsControlElementPropertyId: {
    set_variant(pRetVal, true);
  } break;

  case UIA_HasKeyboardFocusPropertyId: {
    set_variant(pRetVal, true);
  } break;

  case UIA_LiveSettingPropertyId:
  {
    auto la = _element_ptr->get_a11y_live(*this->_element_ptr->pview());
    // Return whichever of Polite or Assertive is most appropriate for your scenario. 
    // Note that a value of zero, (for "Off"), could also be returned, but typically if 
    // an element is ever a LiveRegion, it's always either Assertive or Polite. 

    static_assert(LiveSetting::Assertive == element::A11Y_LIVE_ASSERTIVE, "mismatch");

    set_variant(pRetVal, int(la));
    dbg_printf("UIA_LiveSettingPropertyId %d\n",la);
  } break;

  case UIA_NamePropertyId: {
    ustring name;
    if(_element_ptr->a11y_get_name(*this->_element_ptr->pview(),name))
       set_variant(pRetVal, name.c_str());
    break;
  }
  default: 
    //dbg_printf("accessible_root_provider::GetPropertyValue %d\n", propertyId);
    break;

  }

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::QueryInterface(
    REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(ILegacyIAccessibleProvider))
    *interface_ptr = static_cast<ILegacyIAccessibleProvider *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr = static_cast<IUnknown *>(
        static_cast<ILegacyIAccessibleProvider *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::legacy_iaccessible_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::legacy_iaccessible_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
html::legacy_iaccessible_provider::legacy_iaccessible_provider(
    html::element *element_ptr, uint ref_count /*= 1*/)
    : _refcount(ref_count), _element_ptr(element_ptr) {}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_root_provider::get_HostRawElementProvider(
    __RPC__deref_out_opt IRawElementProviderSimple **pRetVal) {
  if (_UiaHostProviderFromHwnd)
    return _UiaHostProviderFromHwnd()(_hwnd, pRetVal);
  return E_FAIL;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::get_ProviderOptions(
    __RPC__out enum ProviderOptions *pRetVal) {
  *pRetVal = ProviderOptions_ServerSideProvider | ProviderOptions_UseComThreading;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_root_provider::GetEmbeddedFragmentRoots(
    __RPC__deref_out_opt SAFEARRAY **pRetVal) {
  *pRetVal = nullptr;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::GetRuntimeId(
    __RPC__deref_out_opt SAFEARRAY **pRetVal) {
  // Root defers this to host, others must implement it...
  *pRetVal = nullptr;
  
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::Navigate(
    enum NavigateDirection direction,
    __RPC__deref_out_opt IRawElementProviderFragment **pRetVal) {
  *pRetVal   = nullptr;
    
  if (!_element_ptr)
    return S_OK;

/*  helement result;

  switch (direction) {
    case NavigateDirection_Parent:          result = _element_ptr->ui_parent(*pv); break;
    case NavigateDirection_NextSibling:     result = _element_ptr->next_ui_element(); break;
    case NavigateDirection_PreviousSibling: result = _element_ptr->prev_ui_element(); break;
    case NavigateDirection_LastChild:       result = _element_ptr->last_ui_element(); break;
    case NavigateDirection_FirstChild:      result = _element_ptr->first_ui_element(); break;
    default:break;
  }

  if(result)
    *pRetVal = accessible_sciter_provider::make(this, result);
 */

  HRESULT hr = UIA_E_ELEMENTNOTAVAILABLE;

  int dir = -1;

  switch (direction) {
    case NavigateDirection_Parent:
      return hr;
    case NavigateDirection_NextSibling: 
      dir = element::DIR_NEXT; 
      break;
    case NavigateDirection_PreviousSibling: 
      dir = element::DIR_PREVIOUS; 
      break;
    case NavigateDirection_FirstChild: 
      dir = element::DIR_FIRSTCHILD; 
      break;
    case NavigateDirection_LastChild: 
      dir = element::DIR_LASTCHILD; 
      break;
  }

  locker   l(_element_ptr);
  element *navigate_ptr = _element_ptr->a11y_navigate((element::NAVDIR)dir);

  if (navigate_ptr) {
    *pRetVal = accessible_sciter_provider::make(this, navigate_ptr);
    hr = S_OK;
  }
  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::SetFocus() {
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::get_BoundingRectangle(
    __RPC__out struct UiaRect *pRetVal) {
  RECT rc = {0};
  if (!GetWindowRect(_hwnd, &rc)) return HRESULT_FROM_WIN32(GetLastError());

  pRetVal->left   = rc.left;
  pRetVal->top    = rc.top;
  pRetVal->width  = rc.right - rc.left;
  pRetVal->height = rc.bottom - rc.top;

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::get_FragmentRoot(
    __RPC__deref_out_opt IRawElementProviderFragmentRoot **pRetVal) {
  if (!IsWindow(_hwnd)) return UIA_E_ELEMENTNOTAVAILABLE;

  const HRESULT hr = QueryInterface(pRetVal);
  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::accessible_root_provider::ElementProviderFromPoint(
    double x, double y,
    __RPC__deref_out_opt IRawElementProviderFragment **pRetVal) {
  *pRetVal = nullptr;

  if (!_element_ptr) return S_FALSE; // element not in the dom;

  locker l(_element_ptr);
  view * view_ptr = _element_ptr->pview();

  if (!view_ptr) return S_FALSE; // element not in the dom;

  if (!_element_ptr->is_visible(*view_ptr)) return S_FALSE; // element invisible

  gool::point pt;
  pt.x = int(x);
  pt.y = int(y);
  gool::point ptabs(pt);

  pt -= _element_ptr->screen_pos(*view_ptr);

  element *point_element_ptr = _element_ptr->a11y_find_element(*view_ptr, pt);

  if (point_element_ptr && point_element_ptr != _element_ptr)
    *pRetVal = accessible_sciter_provider::make(this, point_element_ptr);

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::accessible_root_provider::GetFocus(
    __RPC__deref_out_opt IRawElementProviderFragment **pRetVal) {
  HRESULT hr = S_FALSE;
  do {
    *pRetVal = nullptr;

    if (!_element_ptr) break;

    locker l(_element_ptr);
    view * view_ptr = _element_ptr->pview();

    if (!view_ptr) break;

    //auto found_element = find_first(*view_ptr, _element_ptr, WCHARS(":focus"), true);
    auto focus_element = view_ptr->get_focus_element();

    if (focus_element) {
      *pRetVal = accessible_sciter_provider::make(this, focus_element);
#ifdef DEBUG
      focus_element->dbg_report("accessible_root_provider::GetFocus");
#endif
      hr = S_OK;
    }
  } while (false);

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
void html::accessible_root_provider::raise_automation_event(EVENTID event_id) {
  if (_UiaRaiseAutomationEvent) {
    HRESULT hr = _UiaRaiseAutomationEvent()(this, event_id);
    assert(SUCCEEDED(hr));
    hr = hr;
  }
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::Select(long flagsSelect) {
  return S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::DoDefaultAction(void) {
  return S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::SetValue(__RPC__in LPCWSTR szValue) {
  return S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::GetIAccessible(
    __RPC__deref_out_opt IAccessible **ppAccessible) {
  *ppAccessible = accessible::factory(_element_ptr);
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::get_ChildId(__RPC__out int *pRetVal) {
  *pRetVal = (int)_element_ptr->uid;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::get_Name(
    __RPC__deref_out_opt BSTR *pszName) {
  *pszName = nullptr;

  HRESULT hr = S_OK;

  do {
    locker l(_element_ptr);
    // element* el = _element_ptr;

    ustring name;

    if (!l.view_ptr) break;

    if (_element_ptr->a11y_get_name(*l.view_ptr, name)) {
      *pszName = SysAllocStringLen(name, (UINT)name.length());
      break;
    }

    // I am not sure about these two, seems like Mozilla does the same:
    name = _element_ptr->attr_title();

    if (name.length()) {
      *pszName = SysAllocStringLen(name, (UINT)name.length());
      break;
    }

    hr = S_FALSE;
  } while (false);

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::get_Value(
    __RPC__deref_out_opt BSTR *pszValue) {
  *pszValue = get_value(_element_ptr);
  return *pszValue ? S_OK : S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::get_Description(
    __RPC__deref_out_opt BSTR *pszDescription) {
  *pszDescription = nullptr;

  HRESULT hr = S_FALSE;

  do {
    locker l(_element_ptr);
    // element* el = _element_ptr;

    ustring txt;

    if (l.view_ptr && _element_ptr->a11y_get_desc(*l.view_ptr, txt)) {
      *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
      hr              = S_OK;
      break;
    }

    txt = _element_ptr->atts[attr::a_alt];

    if (txt.length()) {
      *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
      hr              = S_OK;
      break;
    }

    txt = _element_ptr->atts[attr::a_title];

    if (txt.length()) {
      *pszDescription = SysAllocStringLen(txt, (UINT)txt.length());
      hr              = S_OK;
      break;
    }
  } while (false);

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::get_Role(__RPC__out DWORD *pdwRole) {
  *pdwRole = get_role(_element_ptr);
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::get_State(__RPC__out DWORD *pdwState) {
  view *view_ptr = _element_ptr->pview();

  if (!view_ptr) return S_FALSE;

  locker l(_element_ptr);

  uint state = 0;

  if (_element_ptr->state.link()) state |= STATE_SYSTEM_LINKED;
  if (_element_ptr->state.active()) state |= STATE_SYSTEM_PRESSED;
  if (_element_ptr->state.focus()) state |= STATE_SYSTEM_FOCUSED;
  if (_element_ptr->state.visited()) state |= STATE_SYSTEM_TRAVERSED;
  if (_element_ptr->state.current())
    state |= STATE_SYSTEM_SELECTED | STATE_SYSTEM_FOCUSED;
  if (_element_ptr->state.checked())
    state |= STATE_SYSTEM_SELECTED | STATE_SYSTEM_CHECKED;
  if (_element_ptr->state.readonly()) state |= STATE_SYSTEM_READONLY;
  if (_element_ptr->state.expanded()) state |= STATE_SYSTEM_EXPANDED;
  if (_element_ptr->state.collapsed()) state |= STATE_SYSTEM_COLLAPSED;

  if (_element_ptr->is_disabled())
    state |= STATE_SYSTEM_UNAVAILABLE; // do we need here deep versions?
  if (_element_ptr->is_focusable(*view_ptr)) state |= STATE_SYSTEM_FOCUSABLE;

  if (!_element_ptr->is_visible(*view_ptr)) {
    state |= STATE_SYSTEM_INVISIBLE | STATE_SYSTEM_OFFSCREEN;
  }

  *pdwState = state;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::get_Help(
    __RPC__deref_out_opt BSTR *pszHelp) {
  *pszHelp = nullptr;
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::legacy_iaccessible_provider::get_KeyboardShortcut(
    __RPC__deref_out_opt BSTR *pszKeyboardShortcut) {
  HRESULT hr           = S_FALSE;
  *pszKeyboardShortcut = nullptr;

  locker  l(_element_ptr);
  ustring txt = _element_ptr->atts["accesskey"];

  if (!txt.is_empty()) {
    *pszKeyboardShortcut = SysAllocStringLen(txt, (UINT)txt.length());
    hr                   = S_OK;
  }

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::GetSelection(
    __RPC__deref_out_opt SAFEARRAY **view_ptrarSelectedChildren) {
  *view_ptrarSelectedChildren = nullptr;
  return S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::legacy_iaccessible_provider::get_DefaultAction(
    __RPC__deref_out_opt BSTR *pszDefaultAction) {
  *pszDefaultAction = nullptr;
  return S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::value_provider::SetValue(__RPC__in LPCWSTR val) {
  set_value(_element_ptr, val);
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::value_provider::get_Value(__RPC__deref_out_opt BSTR *pRetVal) {
  *pRetVal = get_value(_element_ptr);
  return *pRetVal ? S_OK : S_FALSE;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::value_provider::get_IsReadOnly(__RPC__out BOOL *pRetVal) {
  *pRetVal = _element_ptr->is_readonly();
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::value_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::value_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::value_provider::QueryInterface(REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(IValueProvider))
    *interface_ptr = static_cast<IValueProvider *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr =
        static_cast<IUnknown *>(static_cast<IValueProvider *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
value_provider *html::value_provider::make(element *element_ptr,
                                           uint     ref_count /*= 1*/) {
  return new value_provider(element_ptr, ref_count);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::toggle_proivder::Toggle(void) {
  HRESULT hr = S_OK;
  do {
    ToggleState toggle_state = ToggleState_Off;

    hr = get_ToggleState(&toggle_state);

    if (S_OK != hr) break;

    locker l(_element_ptr);
    view * view_ptr = _element_ptr->pview();

    if (!view_ptr) break;

    tool::value toggle_value(!(toggle_state));

    _element_ptr->set_value(*view_ptr, toggle_value);
  } while (false);

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::toggle_proivder::get_ToggleState(__RPC__out enum ToggleState *pRetVal) {
  HRESULT hr = S_FALSE;

  do {
    *pRetVal = ToggleState_Off;

    locker l(_element_ptr);
    if (!l.view_ptr) break;

    tool::value toggle_value;
    const bool  success = _element_ptr->get_value(*l.view_ptr, toggle_value);

    if (!success) break;

    if (toggle_value.is_bool()) {
      *pRetVal = toggle_value.get_bool() ? ToggleState_On : ToggleState_Off;
    } else if (toggle_value.is_int()) // TODO: Indeterminate check-boxes - is
                                      // this correct?
    {
      switch (toggle_value.get_int()) {
      case 0: *pRetVal = ToggleState_Off; break;
      case 1: *pRetVal = ToggleState_On; break;
      case 2: *pRetVal = ToggleState_Indeterminate; break;
      }
    }

    hr = S_OK;
  } while (false);

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
toggle_proivder *html::toggle_proivder::make(element *element_ptr,
                                             uint     ref_count /*= 1*/) {
  return new toggle_proivder(element_ptr, ref_count);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::toggle_proivder::QueryInterface(REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(IToggleProvider))
    *interface_ptr = static_cast<IToggleProvider *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr =
        static_cast<IUnknown *>(static_cast<IToggleProvider *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::toggle_proivder::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::toggle_proivder::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::selection_item_provider::Select() {
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::selection_item_provider::AddToSelection() {
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::selection_item_provider::RemoveFromSelection() {
  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::selection_item_provider::get_IsSelected(__RPC__out BOOL *pRetVal) {
  HRESULT hr = S_FALSE;

  do {
    *pRetVal = FALSE;

    locker l(_element_ptr);

    if (!l.view_ptr) break;

    const html::CTL_TYPE control_type = _element_ptr->ctl_type(*l.view_ptr);

    if (html::CTL_RADIO == control_type) {
      tool::value radio_value;
      const bool  success = _element_ptr->get_value(*l.view_ptr, radio_value);

      if (!success) break;

      if (radio_value.is_bool()) {
        *pRetVal = radio_value.get_bool();
        hr       = S_OK;
      }

      break;
    }
  } while (false);

  return hr;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::selection_item_provider::get_SelectionContainer(
    __RPC__deref_out_opt IRawElementProviderSimple **pRetVal) {
  return _sciter_povider_ptr->QueryInterface(pRetVal);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
selection_item_provider *
html::selection_item_provider::make(element *                   element_ptr,
                                    accessible_sciter_provider *sciter_ptr,
                                    uint ref_count /*= 1*/) {
  return new selection_item_provider(element_ptr, sciter_ptr, ref_count);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::selection_item_provider::QueryInterface(
    REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(ISelectionItemProvider))
    *interface_ptr = static_cast<ISelectionItemProvider *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr =
        static_cast<IUnknown *>(static_cast<ISelectionItemProvider *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::selection_item_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::selection_item_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
invoke_provider *html::invoke_provider::make(element *element_ptr,
                                             uint     ref_count /*= 1*/) {
  return new invoke_provider(element_ptr, ref_count);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::invoke_provider::AddRef() {
  return locked::inc(_refcount);
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
ULONG STDMETHODCALLTYPE html::invoke_provider::Release() {
  if (locked::dec(_refcount) <= 0) {
    delete this;
    return 0;
  }

  return _refcount;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE
        html::invoke_provider::QueryInterface(REFIID iid, void **interface_ptr) {
  *interface_ptr = nullptr;

  if (iid == __uuidof(IInvokeProvider))
    *interface_ptr = static_cast<IInvokeProvider *>(this);
  else if (iid == __uuidof(IUnknown))
    *interface_ptr =
        static_cast<IUnknown *>(static_cast<IInvokeProvider *>(this));
  else {
    *interface_ptr = nullptr;
    return E_NOINTERFACE;
  }

  static_cast<IUnknown *>(*interface_ptr)->AddRef();

  return S_OK;
}

//--------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------
HRESULT STDMETHODCALLTYPE html::invoke_provider::Invoke() {
  locker l(_element_ptr);
  view * view_ptr = _element_ptr->pview();

  if (!view_ptr) return S_FALSE;

  method_params params;
  params.method_id = DO_CLICK;
  view_ptr->call_behavior_method(_element_ptr, &params);

  return S_OK;
}

#endif
