//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| t_value, essentially it is a nullable, inheritable int,uint,enum
//|
//|

#ifndef __tl_value_t_h
#define __tl_value_t_h

#include "tl_basic.h"
#include "tl_value.h"
#include "float.h"

namespace tool {

#pragma pack(push, 4) // IT MUST BE 4 bytes aligned!!!!

template <typename T, T default_value, T null_value, T inherit_value>
struct t_value {
  typedef T value_type;

  T _v;

  t_value() : _v(null_value) {}
  t_value(NO_INIT) {}
  t_value(const T &iv) : _v(iv) {}
  t_value(const t_value &c) : _v(c._v) {}

  operator T() const {
    return _v == null_value || _v == inherit_value ? default_value : _v;
  }
  t_value &operator=(const t_value &nv) {
    _v = nv._v;
    return *this;
  }
  t_value &operator=(const T &nv) {
    _v = nv;
    return *this;
  }
  t_value &operator=(const value &nv) {
    clear();
    if (nv.is_undefined())
      return *this;
    else if (nv.is_null())
      _v = null_value;
    else
      _v = T(nv);

    return *this;
  }

  static T null_val() { return null_value; }
  static T inherit_val() { return inherit_value; }
  static T max_val() { return inherit_value - 1; }

  value to_value() const {
    if (is_undefined())
      return value();
    return value(_v);
  }

  bool is_undefined() const { return _v == null_value; }
  bool is_defined() const { return _v != null_value; }
  bool is_inherit() const { return _v == inherit_value; }

  void clear() { _v = null_value; }

  void inherit(const t_value &v) {
    if (v.is_defined())
      _v = v._v;
  }

  T val() const {
    return _v == null_value || _v == inherit_value ? default_value : _v;
  }

  T val(const T &v1) const { return is_defined() ? (T)_v : v1; }
  T val(const t_value &v1) const { return is_defined() ? (T)_v : (T)v1; }

  static T val(const t_value &v1, T defval) {
    return v1.is_defined() ? v1._v : defval;
  }

  static t_value val(const t_value &v1, const t_value &v2) {
    if (v1.is_defined())
      return v1;
    return v2;
  }
  /*static t_value val(const t_value& v1,const t_value& v2, const t_value& v3)
  {
    if(v1.defined()) return v1;
    if(v2.defined()) return v2;
    return v3;
  }
  static T val(const t_value& v1,const t_value& v2,T defval)
  {
    t_value tv = val(v1,v2);
    return tv.defined()? tv._v: defval;
  }
  static T val(const t_value& v1,const t_value& v2, const t_value& v3,T defval)
  {
    t_value tv = val(v1,v2,v3);
    return tv.defined()? tv._v: defval;
  }*/

  unsigned int hash() const { return tool::hash(_v); }
};

typedef t_value<int, 0, int(0x80000000), int(0x80000001)> int_v;
typedef t_value<int64, 0, int64(0x8000000000000000L),    int64(0x8000000000000001L)>
                                                         int64_v;
typedef t_value<uint, 0, 0xFFFFFFFF, 0xFFFFFFFE>         uint_v;
typedef t_value<uint, 0, 0xFF, 0xFFFFFFFF>               tristate_v;
typedef int_v                                            enum_v;

#define UINT_V_MAX uint_v(0xFFFFFFF0)

// milliseconds
typedef t_value<unsigned int, 0, 0xFFFFFFFF, 0xFFFFFFFE> duration_v;

typedef t_value<unsigned int, 1, 0, 0xFFFFFFFE> counter_v;

typedef t_value<uint, 0, 0, 0x80000000> bits_v; // 31 bit flags value

// struct enum_item_def { const char* name; int val; };
// bool   parse(enum_v& v, const char *s, const enum_item_def* pa, int npa);
// string emit(const enum_v& v, const enum_item_def* pa, int npa);

/*struct float_v {
  float _v;

  typedef float value_type;

  float_v() : _v(NAN) {}
  float_v(float iv) : _v(iv) {}

  operator float() const { return _v == NAN ? float(0.0) : _v; }
  float_v &operator=(const float &nv) {
    _v = nv;
    return *this;
  }
  float_v &operator=(const value &nv) {
    clear();
    if (nv.is_undefined())
      return *this;
    else if (nv.is_null())
      _v = NAN;
    else
      _v = float(nv.to_float());
    return *this;
  }

  static float null_val() { return NAN; }
  static float inherit_val() { return INFINITY; }

  value to_value() const {
    if (is_undefined())
      return value();
    return value(_v);
  }

  bool is_undefined() const { return _v == NAN; }
  bool is_defined() const { return _v != NAN; }
  bool is_inherit() const { return _v == INFINITY; }

  void clear() { _v = NAN; }

  void inherit(const float_v &v) {
    if (v.is_defined())
      _v = v._v;
  }

  float val() const {
    return _v == null_val() || _v == inherit_val() ? 0.0f : _v;
  }
  float val(const float_v &v1) const {
    return is_defined() ? (float)_v : (float)v1;
  }
  float val(float v1) const { return is_defined() ? (float)_v : v1; }

  static float val(const float_v &v1, float defval) {
    return v1.is_defined() ? v1._v : defval;
  }
  unsigned int hash() const { return tool::hash(_v); }
};*/

struct float_v {
  float _v;

  typedef float value_type;

  float_v() : _v(null_val()) {}
  float_v(float iv) : _v(iv) {}

  operator float() const { return is_undefined() ? float(0.0) : _v; }
  float_v &operator=(const float &nv) {
    _v = nv;
    return *this;
  }
  float_v &operator=(const value &nv) {
    clear();
    if (nv.is_undefined())
      return *this;
    else if (nv.is_null())
      _v = null_val();
    else
      _v = float(nv.to_float());
    return *this;
  }

  static float null_val() { return INFINITY; } /*cannot use NaN here: NaN values have the odd property that comparisons involving them are always false.*/
  static float inherit_val() { return -INFINITY; }

  value to_value() const {
    if (is_undefined())
      return value();
    return value(_v);
  }

  bool is_undefined() const { return _v == null_val(); }
  bool is_defined() const { return _v != null_val(); }
  bool is_inherit() const { return _v == inherit_val(); }

  void clear() { _v = null_val(); }

  void inherit(const float_v &v) {
    if (v.is_defined())
      _v = v._v;
  }

  float val() const {
    return _v == null_val() || _v == inherit_val() ? 0.0f : _v;
  }
  float val(const float_v &v1) const {
    return is_defined() ? (float)_v : (float)v1;
  }
  float val(float v1) const { return is_defined() ? (float)_v : v1; }

  static float val(const float_v &v1, float defval) {
    return v1.is_defined() ? v1._v : defval;
  }
  unsigned int hash() const { return tool::hash(_v); }
};

struct double_v {
  double _v;

  typedef double value_type;

  double_v() : _v(null_val()) {}
  double_v(double iv) : _v(iv) {}

  operator double() const { return is_undefined() ? double(0.0) : _v; }
  double_v &operator=(const double &nv) {
    _v = nv;
    return *this;
  }
  double_v &operator=(const value &nv) {
    clear();
    if (nv.is_undefined())
      return *this;
    else if (nv.is_null())
      _v = null_val();
    else
      _v = double(nv.to_float());
    return *this;
  }

  static double null_val() { return INFINITY; } /*cannot use NaN here: NaN values have the odd property that comparisons involving them are always false.*/
  static double inherit_val() { return -INFINITY; }

  value to_value() const {
    if (is_undefined())
      return value();
    return value(_v);
  }

  bool is_undefined() const { return _v == null_val(); }
  bool is_defined() const { return _v != null_val(); }
  bool is_inherit() const { return _v == inherit_val(); }

  void clear() { _v = null_val(); }

  void inherit(const double_v &v) {
    if (v.is_defined())
      _v = v._v;
  }

  double val() const {
    return _v == null_val() || _v == inherit_val() ? 0.0f : _v;
  }
  double val(const double_v &v1) const {
    return is_defined() ? (double)_v : (double)v1;
  }
  double val(double v1) const { return is_defined() ? (double)_v : v1; }

  static double val(const double_v &v1, double defval) {
    return v1.is_defined() ? v1._v : defval;
  }
  unsigned int hash() const { return tool::hash(_v); }
};


struct enum_str_v {
  string sv;
  enum_v ev;

  enum_str_v() {}
  enum_str_v(const enum_str_v &iv) : ev(iv.ev), sv(iv.sv) {}

  operator int() const { return ev; }
  enum_str_v &operator=(const enum_str_v &iv) {
    sv = iv.sv;
    ev = iv.ev;
    return *this;
  }
  enum_str_v &operator=(const string &vs) {
    sv = vs;
    ev.clear();
    return *this;
  }
  enum_str_v &operator=(const int &vi) {
    ev = vi;
    sv.clear();
    return *this;
  }

  static enum_str_v null_val() {
    enum_str_v n;
    return n;
  }
  static enum_str_v inherit_val() {
    enum_str_v n;
    n.ev = enum_v::inherit_val();
    return n;
  }

  bool is_undefined() const { return ev.is_undefined() && sv.is_undefined(); }
  bool is_defined() const { return ev.is_defined() || sv.is_defined(); }
  bool is_inherit() const { return ev == enum_v::inherit_val(); }

  bool is_string() const { return sv.is_defined(); }

  void clear() {
    ev.clear();
    sv.clear();
  }

  void inherit(const enum_str_v &v) {
    if (v.is_defined()) {
      ev = v.ev;
      sv = v.sv;
    }
  }

};


#pragma pack(pop)

template <typename V>
inline bool parse_enum(V &v, wchars cs, slice<enum_item_def> idefs) {
  v.clear();
  if (icmp(cs, WCHARS("inherit"))) {
    v = V::inherit_val();
    return true;
  } else
    for (uint i = 0; i < idefs.length; ++i) {
      if (icmp(cs, idefs[i].token)) {
        v = idefs[i].value;
        return true;
      }
    }
  return false;
}
template <typename V>
inline const wchar *enum_to_string(const V &v, slice<enum_item_def> idefs) {
  if (v.is_undefined())
    return W("");
  if (v.is_inherit())
    return W("inherit");
  int iv = v;
  for (uint i = 0; i < idefs.length; ++i)
    if (iv == idefs[i].value)
      return idefs[i].token;
  return W("{unknown}");
}

template <typename V, int size>
inline bool parse_enum(V &v, wchars cs, enum_item_def (&idefs)[size]) {
  return parse_enum(v, cs, slice<enum_item_def>(idefs, size));
}

template <typename V, int size>
inline const wchar *enum_to_string(const V &v, enum_item_def (&idefs)[size]) {
  return enum_to_string(v, slice<enum_item_def>(idefs, size));
}

inline duration_v get_duration_v(const value& v){
  if (v.is_duration())
    return duration_v(uint(v.get_duration() * 1000));
  return duration_v();
}

inline uint_v get_uint_v(const value& v) {
  if (v.is_int())
  {
    int t = v.get(0);
    return t < 0 ? uint_v(uint_v::max_val()) : uint_v((uint)t);
  }
  return uint_v();
}

template <> inline int_v      value::get<int_v>() const { return this->is_number() ? int_v(get(0)) : int_v(); }
template <> inline tristate_v value::get<tristate_v>() const { return this->is_bool() ? tristate_v(get(false)) : tristate_v(); }

inline counter_v get_counter_v(const value& v) {
  if (v.is_string() && v.get_string() == WCHARS("infinite"))
    return counter_v::max_val();
  if (v.is_int())
    return counter_v(v.get(0));
  return counter_v();
}

} // namespace tool

#endif
