//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| value, a.k.a. discriminated union
//|
//|

#ifndef __tl_value_h
#define __tl_value_h

#include "float.h"
#include "tl_array.h"
#include "tl_basic.h"
#include "tl_datetime.h"
#include "tl_dictionary.h"
#include "tl_handle.h"
#include "tl_hash.h"
#include "tl_pool.h"
#include "tl_string.h"
#include "tl_ustring.h"
#include "tl_value_t.h"
#include <initializer_list>
#include <utility>
#include <type_traits>


#pragma warning(push)
#pragma warning(disable : 4311) // 'variable' : pointer truncation from 'type' to 'type'
#pragma warning(disable : 4312) // 'variable' : conversion from 'type' to 'type'
                                // of greater size
#pragma warning(disable : 4100) // 'variable' : unreferenced formal parameter
//#pragma warning (disable :4315) // 'this' pointer for member
//'tool::enum_str_v::sv' may not be aligned 8 as expected by the constructor

struct NATIVE_FUNCTOR_VALUE;
struct som_asset_t;

namespace sciter {
  namespace om {
    long asset_add_ref(som_asset_t *ptr);
    long asset_release(som_asset_t *ptr);
    long asset_get_interface(som_asset_t *ptr, const char* name, void** out);
  }
}

namespace tool {

  class value;

  //using namespace sciter::om;

  extern ustring color_to_string(const value &cv);

  struct enumerator {
    enumerator() {}
    virtual bool doit(const value &val) = 0; // true - continue, false - stop.
    NONCOPYABLE(enumerator)
  };

  enum OBJECT_TYPES {
    UNKNOWN_OBJECT = 0,
    CONDUIT_OBJECT = -1,
  };

  inline int i1000(int f) {
    int v = f / 1000;
    int r = f % 1000;
    if (r < 0)
      v -= 1;
    if (r > 0)
      v += 1;
    // assert(v < 2000);
    return v;
  }
  inline float  f1000(int f) { return float(f) / 1000.f; }
  inline double d1000(int f) { return double(f) / 1000.0; }

  inline int m1000(int f) { return f * 1000; }
  inline int m1000(float f) { return (int) roundf(f * 1000); }
  inline int m1000(double f) { return (int) round(f * 1000); }

  struct object : virtual resource {

    object() {}
    virtual ~object() {}

    virtual bool get_const(wchars name, value &v) { return false; }
    virtual bool get_attr(wchars name, value &v) { return false; }
    virtual bool set_attr(wchars name, const value &v) { return false; }

    virtual bool get_style_attr(wchars name, value &v) { return false; }
    virtual bool set_style_attr(wchars name, const value &v) { return false; }

    virtual bool get_state(wchars name, value &v) { return false; }
    virtual bool set_state(wchars name, const value &v) { return false; }
    virtual bool call(wchars name, uint argn, const value *argv, value &retv) {
      return false;
    }
    virtual bool raise_event(wchars name) { return false; }

    virtual bool get_names(array<wchars> &out_names,
      bool methods /* false - props, true - methods */) {
      return false;
    }

    // resolve em, ex and % units
    virtual bool to_pixels(const value &v, int &px) { return false; }

    virtual void enumerate(enumerator &en) { ; }
    virtual bool to_bool() const { return true; }

    virtual uint type() const { return (uint)type_id(); }

    virtual bool get_user_data(void **ppv) { return false; }
    virtual bool set_user_data(void *pv) { return false; }
  };

  // struct visitor
  //{
  //  virtual bool on( const value& k, const value& v ) = 0; // true to continue
  //  enumeration, false - to stop
  //};
  typedef function<bool(const value &k, const value &v)> kv_visitor;
  typedef function<bool(value &v)> v_mutator;

  struct object_proxy : public resource {
    virtual ustring class_name() const = 0;
    virtual uint    size() const = 0;
    virtual value   get_by_index(uint n) const = 0;
    virtual bool    set_by_index(uint n, const value &v) = 0;

    virtual value get_by_key(const value &key) const = 0;
    virtual bool  set_by_key(const value &key, const value &v) = 0;

    virtual bool get_user_data(void **ppv) const = 0;
    virtual bool set_user_data(void *pv) = 0;

    virtual bool equal(const object_proxy *pv) const = 0;

    virtual value invoke(const value &self, uint argc, const value *argv) = 0;

    virtual bool visit(const kv_visitor &vis) const = 0;
    virtual bool isolate(tool::value &vout) const = 0;
  };

  struct native_functor_holder : public resource_x<native_functor_holder> {
    native_functor_holder() {}
    virtual ~native_functor_holder() {}
    virtual value call(uint argc, const value *argv) = 0;
  };

  typedef handle<object> hobject;

  //struct array_value;
  struct function_value;
  typedef function_value map_value;

  typedef array<byte>::array_data bytes_value;

  struct enum_item_def { int value; const wchar *token; };
  struct enum_def { const wchar* name; slice<enum_item_def> items; };

  template<typename E> enum_def* enum_pdef();

  struct array_value : resource {
    array<value>        elements;
    inline unsigned int hash() const { return elements.hash(); }
  };

#pragma pack(push, 4)

  class value 
  {
    friend ustring color_to_string(const value& cv);
  private:
    //friend class string;
    value(void *obj, uint offset = 0) = delete;
    // void* get(const void* defv) { assert(false); return 0; }
    // void* get(const void* defv) const { assert(false); return 0; }
    value &operator=(void *) = delete;
    value &operator=(uint64) = delete;
    value &operator=(int64) = delete;

  public:
    enum types {
      t_undefined,    // 0
      t_null,         // 1
      t_bool,         // 2
      t_int,          // 3
      t_double,       // 4
      t_string,       // 5
      t_date,         // 6
      t_currency,     // 7
      t_length,       // 8
      t_array,        // 9
      t_map,          // 10
      t_function,     // 11
      t_bytes,        // 12
      t_object_proxy, // 13, tiscript object proxy (pinned value)
      t_object,       // 14 /eval/ object
      t_resource,     // 15 - other thing derived from tool::resource
      t_range,        // 16 - N..M, integer range.
      t_duration,     // 17
      t_angle,        // 18 - radians
      t_color,        // 19
      t_enum,         // 20
      t_asset,        // 21 - sciter::om::iasset
    };

    enum unit_type_length {
      unknown = 0,
      //-------- rel
      em = 1, // height of the element's font.
      ex = 2, // height of letter 'x'
      pr = 3, //%
      sp = 4, //%% "springs"
      rs = 5, // value is -1,1 // smaller larger
      //-------- abs
      as = 6,  // value is x-small, etc 1,2,3,4,5,6,7
      px = 7,  // pixels, device dependent
      in = 8,  // Inches (1 inch = 2.54 centimeters).
      cm = 9,  // Centimeters.
      mm = 10, // Millimeters.
      pt = 11, // Points (1 point = 1/72 inches).
      pc = 12, // Picas (1 pica = 12 points).
      dip = 13, // device independent pixels, 1/96 of inch. Thus on 96 DPI screen
                // these correspond to a single physical pixel
      nm = 14, // Number
      expr = 15, // calc expression
      pr_width = 16, // width(n%)
      pr_height = 17, // height(n%)
      pr_view_width = 18, // vw
      pr_view_height = 19, // vh
      pr_view_min = 20, // vmin
      pr_view_max = 21, // vmax

      rem = 22, // root em
      ppx = 23, // physical px
      ch = 24   // width of '0'

                                          // above-stated should match size_v
                                          // func, // func
                                          // dpi,  // dpi/int
                                          // radians = 20, // angle
    };

    enum length_special_values {
      $none = 0,
      $smaller = -1,
      $larger = 1,

      $xx_small,
      $x_small,
      $small,
      $medium,
      $large,
      $x_large,
      $xx_large,

      $thin,
      $thick,

      $auto,

      $inherit,
      $min_content,
      $max_content,
      $ui_scale,

      $cover,
      $contain,

      $system_metrics_first,
      $scrollbar_height = $system_metrics_first,
      $scrollbar_width,
      $small_icon_height,
      $small_icon_width,
      $border_width,
      $border_3d_width,

      $window_caption_height,
      $window_button_height,
      $window_button_width,
      $window_frame_width,

      $system_metrics_last = $window_frame_width
    };

    enum unit_type_object {
      UT_OBJECT_ARRAY = 0, // type T_OBJECT of type Array
      UT_OBJECT_OBJECT = 1, // type T_OBJECT of type Object
      UT_OBJECT_CLASS = 2, // type T_OBJECT of type Type (class or namespace)
      UT_OBJECT_NATIVE =
      3, // type T_OBJECT of native Type with data slot (LPVOID)
      UT_OBJECT_FUNCTION = 4, // type T_OBJECT of type Function
      UT_OBJECT_ERROR = 5, // type T_OBJECT of type Error
    };

    enum unit_type_string {
      UT_STRING = 0,
      UT_ERROR = 1, // is an error string
      UT_SECURE = 2, // secure string ("wiped" on destroy)
      UT_URL = 3, // url(...)
      UT_SELECTOR = 4, // selector(...)
      UT_FILE_NAME = 0xFFFE,
      UT_SYMBOL = 0xFFFF,
      // aka NMTOKEN
    };
    enum unit_type_array {
      UT_WS_LIST = 1, // white space separated list
      UT_CS_LIST,     // comma separated list
      UT_PAIR,     // slash separated pair, 10px/14px
    };

    enum unit_type_int {
      UT_INTEGER = 0, // white space separated list
      UT_DPI,         // 96dpi
    };

    enum unit_type_angle {
      UT_RAD = 0,
      UT_DEG = 1,
      UT_GRAD = 2,
      UT_TURN = 3,
    };

    enum unit_type_duration {
      UT_S = 0,
      UT_MS = 1,
    };

    enum unit_type_meta {
      UT_NOTHING = 1,
      UT_CANCEL = 0xAFED,
      UT_NONE = 0xAFEE,  // CSS: unset ?
      UT_AUTO = 0xAFEF,  // CSS: initial ?
      UT_INHERIT = 0xFFFF,
    };

    enum unit_type_resource {
      UTR_PATH,
      UTR_IMAGE,
      UTR_BRUSH,
      UTR_TEXT,
      UTR_GRAPHICS,
      UTR_DOM_ELEMENT,
      UTR_DOM_NODE,
    };

    typedef dictionary<value, value>::key_value key_value;

    value() : _type(t_undefined), _units(0) { _i64(0); }

    explicit value(bool b, uint u = 0) {
      _units = u;
      _type = t_bool;
      _b(b);
    }
    explicit value(int i, uint u = 0) {
      _units = u;
      _type = t_int;
      _i(i);
    }
    explicit value(uint i, uint u = 0) {
      _units = u;
      _type = t_int;
      _i(i);
    }
    explicit value(int64 i, uint u = 0) {
      _units = u;
      _type = t_currency;
      _i64(i);
    }
    explicit value(double d, uint u = 0) {
      _units = u;
      _type = t_double;
      _d(d);
    }
    explicit value(float d, uint u = 0) {
      _units = u;
      _type = t_double;
      _d(d);
    }
    // explicit value(const string& s, uint u = 0)    { _units = u; _type =
    // t_string; _s(s.get_data()); }
    explicit value(chars s) {
      _units = UT_SYMBOL;
      _type = t_string;
      _us(ustring(s.start, s.length).get_data());
    }
    explicit value(const char *s) {
      _units = UT_SYMBOL;
      _type = t_string;
      _us(ustring(s).get_data());
    }
    explicit value(const string &s, uint u = UT_SYMBOL) {
      _units = u;
      _type = t_string;
      _us(ustring(s).get_data());
    }
    explicit value(const ustring &us, uint u = 0) {
      _units = u; _type = t_string;
      _us(us.get_data());
    }
    explicit value(wchars us, uint u = 0) {
      _units = u; _type = t_string;
      _us(ustring(us).get_data());
    }
    explicit value(const wchar *us) {
      _units = 0; _type = t_string;
      _us(ustring(us).get_data());
    }
    explicit value(object *obj, uint offset = 0) {
      _units = offset;
      _type = t_object;
      obj->add_ref();
      _obj(obj);
    }
    explicit value(const date_time &dt, uint u = date_time::DT_UTC) {
      _units = u;
      _type = t_date;
      _i64(dt.time());
    }
    explicit value(resource* res, uint un = 0) {
      _type = t_resource;
      _units = un;
      res->add_ref();
      _res(res);
    }

    template<typename T> value(const array<T> &arr) {
      _type = t_array;
      _units = 0;
      array_value *a = new array_value();
      for (auto& el : arr)
        a->elements.push(value(el));
      a->add_ref();
      _a(a);
    }

#ifdef OSX
    explicit value(const wchar_t *us); // { assert(false); }
#endif

    value(const value &cv) : _type(t_undefined), _units(0) {
      _i(0);
      set(cv);
    }

    static value make_array(uint sz = 0);
    static value make_array(slice<value> els, uint units = 0);
    static value make_array(array_value* a, uint units = 0);
    template<typename TI> static value make_array(slice<TI> es, uint units = 0) {
      array_value* pa = new array_value();
      pa->elements.length(es.length);
      for (int n = 0; n < es.size(); ++n)
        pa->elements[n] = value(es[n]);
      return make_array(pa,units);
    }
    static value make_array(std::initializer_list<value> list)
    {
      return value::make_array(slice<value>(list.begin(), list.size()));
    }

    void _set_(int i, uint t, uint u) {
      _units = u;
      _type = t;
      _i(i);
    }

    static value make_packed_color(uint64 packed_color) {
      value t;
      t._type = t_color;
      t._units = 0;
      t._i64(packed_color);
      return t;
    }

    static value make_currency(int64 fixed, uint u = 0) {
      value t;
      t._type = t_currency;
      t._units = u;
      t._i64(fixed);
      return t;
    }

    static value make_dpi(int dpi) {
      value t;
      t._type = t_int;
      t._units = UT_DPI;
      t._i(dpi);
      return t;
    }

    static value make_date(int64 date, /*date_time::type*/ uint u = date_time::DT_UTC) {
      value t;
      t._type = t_date;
      t._units = u;
      t._i64(date);
      return t;
    }
    static value make_function(function_value *f = 0);
    static value make_function(const wchar* name);
    static value make_function(const wchar* name, value p);
    static value make_map(map_value *m = 0);
    static value make_map(std::initializer_list<std::pair<const char*, value>> list)
    {
      value result = value::make_map();
      for (auto& item : list)
        result.set_item(item.first, item.second);
      return result;
    }


    static value make_bytes(bytes bs, uint u = 0) {
      array<byte> data = bs;
      return make_bytes(data, u);
    }
    static value make_bytes(const array<byte> &data, uint u = 0) {
      value t;
      t._type = t_bytes;
      t._units = u;
      array<byte>::array_data *bv = nullptr;
      data.set_ref_to(bv);
      t._bytes(bv);
      return t;
    }

    static value make_proxy(object_proxy *pr, uint u = 0) {
      value t;
      t._type = t_object_proxy;
      t._units = u;
      pr->add_ref();
      t._proxy(pr);
      return t;
    }

    static value make_resource(resource *pr, uint u = 0) {
      value t;
      t._type = t_resource;
      t._units = u;
      pr->add_ref();
      t._res(pr);
      return t;
    }

    static value make_enum(int v, const enum_def* ped) {
      value t;
      t._type = t_enum;
      t._units = v;
      t._ptr(ped);
      return t;
    }

    /*static value make_url(chars s)
    {
    value t(ustring(s),value::uri);
    return t;
    }*/
    static value make_url(const ustring &s) {
      value t(s, value::UT_URL);
      return t;
    }

    static value make_error(const char* s) {
      return make_error(ustring(s));
    }

    static value make_error(const ustring &s) {
      value t(s, value::UT_ERROR);
      return t;
    }

    static value make_range(int s, int e, uint u = 0) {
      value t;
      t._type = t_range;
      t._units = u;
      uint64 d = uint64((uint32)s) << 32 | uint64((uint32)e);
      t._i64(d);
      return t;
    }
    static value wrap_resource(resource *res, uint u = 0) {
      value t;
      t.set_resource(res, u);
      return t;
    }

    static value wrap_asset(som_asset_t *res, uint u = 0) {
      value t;
      t.set_asset(res, u);
      return t;
    }


    static value make_duration(real v, uint u = value::UT_S, bool raw = false) {
      real seconds;
      if (raw)
        seconds = v;
      else switch (u) {
      default:
      case UT_S:  seconds = v; break;
      case UT_MS: seconds = v / 1000.0; break;
      }
      value t;
      t._type = t_duration;
      t._units = u;
      t._d(seconds);
      return t;
    }

    static value make_angle(real v, unit_type_angle u = UT_RAD, bool raw = false) {
      // NOTE: angle always stored in radians
      real vradians;
      if (raw)
        vradians = v;
      else switch (u) {
      default:
      case UT_RAD:  vradians = v; break;
      case UT_DEG:  vradians = v / 57.2957795; break;
      case UT_TURN: vradians = 360 * v / 57.2957795; break;
      case UT_GRAD: vradians = v * 0.015707963; break;
      }
      value t;
      t._type = t_angle;
      t._units = u;
      t._d(vradians);
      return t;
    }

    static double angle_from_radians(real radians, unit_type_angle to) {
      switch (to) {
      default:
      case UT_RAD:  return radians;
      case UT_DEG:  return radians * 57.2957795;
      case UT_TURN: return radians * 57.2957795 / 360;
      case UT_GRAD: return radians / 0.015707963;
      }
    }

    static string angle_to_string(real radians, unit_type_angle units) {
      const char* su[] = { "rad", "deg", "grad", "turn" };
      double v = angle_from_radians(radians, units);
      return string::format("%.2g%s", v, su[units]);
    }

    static double duration_from_seconds(real seconds, unit_type_duration to) {
      switch (to) {
      default:
      case UT_S:
        return seconds;
      case UT_MS:
        return seconds * 1000;
      }
    }

    static string duration_to_string(real seconds, unit_type_duration units) {
      const char* su[] = { "s", "ms" };
      double v = duration_from_seconds(seconds, units);
      return string::format("%.2g%s", v, su[units]);
    }

#if 0    
    static string color_to_string(int64 bits);
    {
      uint type = uint((bits >> 62) & 0x3);
      uint colr = uint(bits & 0xFFFFFFFFull);
      if (type == 0)
        return string::format("#%02x%02x%02x", colr & 0xff, (colr >> 8) & 0xff, (colr >> 16) & 0xff);
      else
        return string("color(...)");
    }
#endif

    void  set(const value &cv);
    bool  set_element(uint idx, const value &v); // set element by index
    value get_element(uint idx) const;           // get element by index

    value operator[](uint idx) const { return get_element(idx); }

    void clear();
    value clone(const hash_table<ustring, value>& subst) const;

    ~value() { clear(); }

    value &operator=(const value &cv) {
      set(cv);
      return *this;
    }

    value &operator=(bool b) {
      *this = value(b);
      return *this;
    }
    value &operator=(int i) {
      *this = value(i);
      return *this;
    }
    value &operator=(double d) {
      *this = value(d);
      return *this;
    }
    value &operator=(const string &s) {
      *this = value(ustring(s), 0xffff);
      return *this;
    }
    value &operator=(const ustring &s) {
      *this = value(s);
      return *this;
    }

    value &set_object(object *obj, uint off = 0) {
      clear();
      _type = t_object;
      _units = off;
      obj->add_ref();
      _obj(obj);
      return *this;
    }
    value &set_resource(resource *res, uint un = 0) {
      clear();
      _type = t_resource;
      _units = un;
      res->add_ref();
      _res(res);
      return *this;
    }

    value &set_asset(som_asset_t* pass, uint un = 0) {
      clear();
      _type = t_asset;
      _units = un;
      sciter::om::asset_add_ref(pass);
      _ass(pass);
      return *this;
    }


    /*value& operator = (const array<value>& va)
    {
    clear();
    _type = t_array;
    v.a = a_impl::create(va);
    return *this;
    }*/

    types type() const { return types(_type); }
    uint units() const { return _units; }
    void units(uint u) { _units = u; }

    unsigned int hash() const;

    // to_string()
    bool is_undefined() const { return _type == t_undefined; }
    bool is_defined() const { return _type != t_undefined; }
    bool is_null() const { return _type == t_null; }
    bool is_cancel() const { return _type == t_null && _units == UT_CANCEL; }
    bool is_nothing() const { return _type == t_undefined && _units == UT_NOTHING; }

    bool is_none() const { return _type == t_null && _units == UT_NONE; }
    bool is_unset() const { return _type == t_null && _units == UT_NONE; }

    bool is_inherit() const { return _type == t_null && _units == UT_INHERIT; }

    bool is_auto() const { return _type == t_null && _units == UT_AUTO; }
    bool is_initial() const { return _type == t_null && _units == UT_AUTO; }

    bool is_color() const { return (_type == t_color) || is_variable_color(); }
    bool is_int() const { return _type == t_int; }
    bool is_int64() const { return _type == t_currency; }
    bool is_strictly_int() const { return _type == t_int && _units == 0; }
    bool is_bool() const { return _type == t_bool; }
    bool is_double() const { return _type == t_double; }
    bool is_number() const { return _type == t_int || _type == t_double || _type == t_currency; }
    bool is_string() const { return _type == t_string; }
    bool is_url() const { return _type == t_string && _units == UT_URL; }
    bool is_selector() const { return _type == t_string && _units == UT_SELECTOR; }
    bool is_string(wchars str) const { return _type == t_string && get_string() == str; }
    bool is_array() const { return _type == t_array; }
    bool is_array(int sz) const { return is_array() && get_array()->elements.size() == sz; }
    bool is_array_like() const { return _type == t_array || is_proxy_of_array(); }

    bool                       is_function() const { return _type == t_function; }
    bool                       is_function(wchars name) const;
    template <typename F> bool is_function(wchars name, F c0) const;


    template <typename T> bool crack_function(wchars name, T& v) const;
    template <typename T0, typename T1> bool crack_function(wchars name, T0& v0, T1& v1) const;


    bool is_variable() const;
    bool is_variable_color() const;
    bool is_variable_length() const;
    bool get_variable_name_and_value(string& name, value& val) const;

    bool is_map() const { return _type == t_map; }
    bool is_map_alike() const { return is_map() || is_proxy_of_object(); }
    bool is_length() const { return _type == t_length || is_variable_length(); }
    bool is_length_or_percent() const { return is_length() || is_percent(); }
    bool is_length_or_percent_or_zero() const { return is_length() || is_percent() || (is_int() && get_int() == 0); }
    bool is_object() const { return _type == t_object; }
    bool is_spring() const { return is_length() && units() == value::sp; }
    bool is_percent() const { return is_length() && units() == value::pr; }
    bool is_currency() const { return _type == t_currency; }
    bool is_date() const { return _type == t_date; }
    bool is_bytes() const { return _type == t_bytes; }
    bool is_proxy() const { return _type == t_object_proxy; }
    bool is_proxy_of_object() const { return _type == t_object_proxy && _units == UT_OBJECT_OBJECT; }
    bool is_proxy_of_class() const { return _type == t_object_proxy && _units == UT_OBJECT_CLASS; }
    bool is_proxy_of_array() const { return _type == t_object_proxy && _units == UT_OBJECT_ARRAY; }
    bool is_proxy_of_function() const { return _type == t_object_proxy && _units == UT_OBJECT_FUNCTION; }
    bool is_resource() const { return _type == t_resource; }
    template<class R> bool is_resource() const { return _type == t_resource && _res()->is_of_type<R>(); }

    bool is_asset() const { return _type == t_asset; }
    //template<class R> bool is_asset() const { return _type == t_resource && _res()->is_of_class<R>(); }

    bool is_range() const { return _type == t_range; }
    bool is_duration() const { return _type == t_duration; }
    bool is_angle() const { return _type == t_angle; }
    bool is_zero() const { return _type == t_int && _int() == 0; }
    bool is_native_functor() const { return is_resource() && get_resource()->is_of_type<native_functor_holder>(); }

    bool is_enum() const { return _type == t_enum; }

    template<typename ED> bool is_enum() const { return _type == t_enum && _ptr() == enum_pdef<ED>(); }

    bool is_string_symbol() const {
      return _type == t_string && _units == UT_SYMBOL;
    }
    bool is_string_symbol(wchars sc) const {
      return _type == t_string && _units == UT_SYMBOL && get(W("")) == sc;
    }
    bool is_pure_string() const { return _type == t_string && _units == UT_STRING; }

    int is_string_of(chars a) const {
      if (!is_string()) return 0;
      string s = to_string();
      if (s == a) return 1;
      return 0;
    }
    int is_string_of(chars a, chars b) const {
      if (!is_string()) return 0;
      string s = to_string();
      if (s == a) return 1;
      if (s == b) return 2;
      return 0;
    }
    int is_string_of(chars a, chars b, chars c) const {
      if (!is_string()) return 0;
      string s = to_string();
      if (s == a) return 1;
      if (s == b) return 2;
      if (s == c) return 3;
      return 0;
    }
    int is_string_of(chars a, chars b, chars c, chars d) const {
      if (!is_string()) return 0;
      string s = to_string();
      if (s == a) return 1;
      if (s == b) return 2;
      if (s == c) return 3;
      if (s == d) return 4;
      return 0;
    }
    int is_string_of(chars a, chars b, chars c, chars d, chars e) const {
      if (!is_string()) return 0;
      string s = to_string();
      if (s == a) return 1;
      if (s == b) return 2;
      if (s == c) return 3;
      if (s == d) return 4;
      if (s == e) return 5;
      return 0;
    }
    
    bool is_error_string() const {
      return _type == t_string && _units == UT_ERROR;
    }
    // evalutable byte codes
    bool is_conduit() const {
      return is_object() && get_object()->type() == uint(CONDUIT_OBJECT);
    }

    // bool  is_text() const { return is_string() || is_ustring(); }

    int get_int() const {
      if (is_enum())
        return get_enum_code();
      if (_type == t_double)
        return int(_d());
      assert(_type == t_int || _type == t_bool);
      return _i();
    }
    int64 get_int64() const {
      assert(_type == t_currency || _type == t_date);
      return _i64();
    }

    string get_url() const {
      if (is_url())
        return ustring(_us());
      if(is_string())
        return ustring(_us());
      return string();
    }
    ustring get_selector() const {
      if (is_selector())
        return ustring(_us());
      if (is_string())
        return ustring(_us());
      return ustring();
    }


    som_asset_t* get_asset() const {
      return _ass();
    }

    bool get_bool() const {
      assert(_type == t_bool);
      return _b();
    }

    int get_enum_code() const {
      assert(_type == t_enum);
      return int(_units);
    }

    const enum_def* get_enum_def() const {
      assert(_type == t_enum);
      assert(_ptr());
      return (enum_def*)_ptr();
    }

    double get_double(double dv = 0.0) const {
      if (_type == t_double)
        return _d();
      else if (_type == t_int)
        return get(0);
      else if (_type == t_length)
        return length_to_float();
      else if (_type == t_duration)
        return _d();
      else if (_type == t_angle)
        return _d();
      return dv;
    }
    float get_duration(float dv = 0) const {
      if (_type == t_double || _type == t_duration)
        return (float)_d();
      else if (_type == t_int)
        return float(get(0)) / 1000.0f;
      return dv;
    }

    float get_angle(float dv = 0) const {
      if(_type == t_angle)
        return (float)get_double();
      if (is_number())
        return (float)(get_double() * 0.0174533); // degree to radian
      return dv;
    }
    float get_float(float dv = 0) const { return (float)get_double(dv); }

    float get_percent(float dv = 0) const { // returns percent normalized to 0.0 ... 1.0 range
      if (is_length() && units() == value::pr)
        return (float)length_to_float() / 100.0f;
      else if (is_double())
        return (float)get_double();
      return dv;
    }

    uint64 get_packed_color() const { 
      assert(!is_variable_color());
      return is_color() ? _i64() : uint64(0); 
    }

    // string    get_string() const { assert(_type == t_string); return
    // string(_s()); }
    ustring get_string() const {
      assert(_type == t_string);
      return (_type == t_string) ? ustring(_us()) : ustring();
    }
    wchars get_chars() const {
      return (_type == t_string) ? wchars(_us()->source()) : wchars();
    }
    function_value *get_function() const {
      assert(_type == t_function);
      return _f();
    }
    map_value *get_map() const {
      assert(_type == t_map);
      return _m();
    }
    array_value *get_array() const {
      assert(_type == t_array);
      return _a();
    }

    template<typename TI>
    array<TI> get_array() const {
      array<TI> rv;
      if (is_array()) {
        if (array_value * av = get_array()) {
          rv.length(av->elements.length());
          for (int n = 0; n < av->elements.size(); ++n)
            rv[n] = av->elements[n].get<TI>();
        }
      }
      return rv;
    }

    object *get_object() const { /*assert(_type == t_object);*/
      if (_type == t_object)
        return _obj();
      return 0;
    }
    object *get_object(uint &off) const {
      assert(_type == t_object);
      if (_type == t_object) {
        off = _units;
        return _obj();
      }
      return 0;
    }

    bytes get_bytes() const {
      return is_bytes() ? bytes_value::get_elements(_bytes()) : bytes();
    }
    array<byte> get_bytes_array() const {
      return is_bytes() ? array<byte>(_bytes()) : array<byte>();
    }

    object_proxy *get_proxy() const {
      assert(_type == t_object_proxy);
      if (_type == t_object_proxy)
        return _proxy();
      return 0;
    }
    date_time get_date() const {
      assert(_type == t_date);
      if (_type != t_date)
        return date_time();
      date_time t = date_time((datetime_t)get_int64());
      if ((_units & date_time::DT_UTC) == 0)
        t.to_utc();
      return t;
    }

    resource *get_resource() const {
      assert(_type == t_resource);
      if (_type == t_resource)
        return _res();
      return 0;
    }

    template <typename T> T *get_resource() const {
      assert(_type == t_resource);
      if (is_resource<T>())
        return static_cast<T*>(_res());
      if (is_proxy_of_object()) {
        void* ppv = nullptr;
        if(get_proxy()->get_user_data(&ppv))
          return static_cast<T*>((resource*)ppv);
      }
      return nullptr;
    }

    void get_range(int &s, int &e) {
      uint64 d = _i64();
      s = int((d >> 32) & uint64(0xFFFFFFFF));
      e = int(d & uint64(0xFFFFFFFF));
    }

    ustring length_to_string() const {
      assert(is_length());
      return length_to_string(_i(), units());
    }

    static ustring length_to_string(int i, int u) {
      switch (u) {
      case value::em:
        return fixedtow(i, 3, 0, W("em"));
      case value::rem:
        return fixedtow(i, 3, 0, W("rem"));
      case value::ex:
        return fixedtow(i, 3, 0, W("ex"));
      case value::ch:
        return fixedtow(i, 3, 0, W("ch"));
      case value::pr:
        return fixedtow(i, 3, 0, W("%"));
      case value::pr_width:
        return fixedtow(i, 3, W("width("), W("%)"));
      case value::pr_height:
        return fixedtow(i, 3, W("height("), W("%)"));
      case value::pr_view_width:
        return fixedtow(i, 3, 0, W("vw"));
      case value::pr_view_height:
        return fixedtow(i, 3, 0, W("vh"));
      case value::pr_view_min:
        return fixedtow(i, 3, 0, W("vmin"));
      case value::pr_view_max:
        return fixedtow(i, 3, 0, W("vmax"));
      case value::sp:
        // return ustring::format(L"%d%%%%",i);
        return fixedtow(i, 3, 0, W("*"));
      case value::px:
        // return ustring::format(L"%dpx",i);
        return fixedtow(i, 3, 0, W("px"));
      case value::ppx:
        // return ustring::format(L"%dpx",i);
        return fixedtow(i, 3, 0, W("ppx"));
      case value::in:
        // if( i % 1000 == 0 ) return ustring::format(L"%din",i/1000);
        // else return ustring::format(L"%fin",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("in"));
      case value::pt: // Points (1 point = 1/72 inches).
                      // if( i % 1000 == 0 ) return ustring::format(L"%dpt",i/1000);
                      // else return ustring::format(L"%fpt",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("pt"));
      case value::dip:
        // if( i % 1000 == 0 ) return ustring::format(L"%ddip",i/1000);
        // else return ustring::format(L"%fdip",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("dip"));
      case value::pc: // Picas (1 pica = 12 points).
                      // if( i % 1000 == 0 ) return ustring::format(L"%dpc",i/1000);
                      // else return ustring::format(L"%fpc",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("pc"));
      case value::cm: // Cm (2.54cm = 1in).
                      // if( i % 1000 == 0 ) return ustring::format(L"%dcm",i/1000);
                      // else return ustring::format(L"%fcm",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("cm"));
      case value::mm:
        // if( i % 1000 == 0 ) return ustring::format(L"%dmm",i/1000);
        // else return ustring::format(L"%fmm",double(i)/1000.0);
        return fixedtow(i, 3, 0, W("mm"));
      case value::as:
        switch (i) {
        case $none:
          return WCHARS("none");
        case $smaller:
          return WCHARS("smaller");
        case $larger:
          return WCHARS("larger");

        case $xx_small:
          return WCHARS("xx-small");
        case $x_small:
          return WCHARS("x-small");
        case $small:
          return WCHARS("small");
        case $medium:
          return WCHARS("medium");
        case $large:
          return WCHARS("large");
        case $x_large:
          return WCHARS("x-large");
        case $xx_large:
          return WCHARS("xx-large");

        case $thin:
          return WCHARS("thin");
        case $thick:
          return WCHARS("thick");

        case $auto:
          return WCHARS("auto");

        case $inherit:
          return WCHARS("inherit");
        case $min_content:
          return WCHARS("min-content");
        case $max_content:
          return WCHARS("max-content");
        case $scrollbar_height:
          return WCHARS("system-scrollbar-height");
        case $scrollbar_width:
          return WCHARS("system-scrollbar-width");
        case $small_icon_height:
          return WCHARS("system-small-icon-height");
        case $small_icon_width:
          return WCHARS("system-small-icon-width");
        case $border_width:
          return WCHARS("system-border-width");
        case $border_3d_width:
          return WCHARS("system-border-3d-width");
        }
        break;

      case value::nm:
        if (i == 0)
          return WCHARS("0");
        // fall through
      }
      return WCHARS("{not a length unit}");
    }

    static ustring length_to_string_fx(int i, int u) {
      switch (u) {
      case value::em:
        return fixedtow(i, 3, 0, W("em"));
      case value::rem:
        return fixedtow(i, 3, 0, W("rem"));
      case value::ex:
        return fixedtow(i, 3, 0, W("ex"));
      case value::ch:
        return fixedtow(i, 3, 0, W("ch"));
      case value::pr:
        return fixedtow(i, 3, 0, W("pr"));
      case value::sp:
        return fixedtow(i, 3, 0, W("fx"));
      case value::px:
        return fixedtow(i, 3, 0, W("px"));
      case value::ppx:
        return fixedtow(i, 3, 0, W("ppx"));
      case value::in:
        return fixedtow(i, 3, 0, W("in"));
      case value::pt: // Points (1 point = 1/72 inches).
        return fixedtow(i, 3, 0, W("pt"));
      case value::dip:
        return fixedtow(i, 3, 0, W("dip"));
      case value::pc: // Picas (1 pica = 12 points).
        return fixedtow(i, 3, 0, W("pc"));
      case value::cm: // Cm (2.54cm = 1in).
        return fixedtow(i, 3, 0, W("cm"));
      case value::mm:
        return fixedtow(i, 3, 0, W("mm"));
      case value::as:
        switch (i) {
        case $none:
          return WCHARS("\"none\"");
        case $smaller:
          return WCHARS("\"smaller\"");
        case $larger:
          return WCHARS("\"larger\"");

        case $xx_small:
          return WCHARS("\"xx-small\"");
        case $x_small:
          return WCHARS("\"x-small\"");
        case $small:
          return WCHARS("\"small\"");
        case $medium:
          return WCHARS("\"medium\"");
        case $large:
          return WCHARS("\"large\"");
        case $x_large:
          return WCHARS("\"x-large\"");
        case $xx_large:
          return WCHARS("\"xx-large\"");

        case $thin:
          return WCHARS("\"thin\"");
        case $thick:
          return WCHARS("\"thick\"");

        case $auto:
          return WCHARS("\"auto\"");

        case $inherit:
          return WCHARS("\"inherit\"");
        case $min_content:
          return WCHARS("\"min-content\"");
        case $max_content:
          return WCHARS("\"max-content\"");
        case $scrollbar_height:
          return WCHARS("\"system-scrollbar-height\"");
        case $scrollbar_width:
          return WCHARS("\"system-scrollbar-width\"");
        case $small_icon_height:
          return WCHARS("\"system-small-icon-height\"");
        case $small_icon_width:
          return WCHARS("\"system-small-icon-width\"");
        case $border_width:
          return WCHARS("\"system-border-width\"");
        case $border_3d_width:
          return WCHARS("\"system-border-3d-width\"");
        }
        break;
      case value::nm:
        if (i == 0)
          return WCHARS("0");
        // fall through
      }
      return WCHARS("{not a length unit}");
    }

    int length_to_int() const {
      assert(is_length());
      return length_to_int(_i(), units());
    }

    static int length_to_int(int i, int u) {
      switch (u) {
      case value::in:
      case value::pt: // Points (1 point = 1/72 inches).
      case value::pc: // Picas (1 pica = 12 points).
      case value::cm: // Cm (2.54cm = 1in).
      case value::mm:
      case value::em:
      case value::dip:
      case value::ex:
      case value::pr:
      case value::pr_width:
      case value::pr_height:
      case value::pr_view_width:
      case value::pr_view_height:
      case value::sp:
      case value::px:
      case value::ppx:
        return i / 1000;
      default:
        return 0;
      }
    }

    static int packed_length(int i, int u) {
      switch (u) {
      case value::in:
      case value::pt: // Points (1 point = 1/72 inches).
      case value::pc: // Picas (1 pica = 12 points).
      case value::cm: // Cm (2.54cm = 1in).
      case value::mm:
      case value::em:
      case value::dip:
      case value::ex:
      case value::pr:
      case value::sp:
      case value::pr_width:
      case value::pr_height:
      case value::pr_view_width:
      case value::pr_view_height:
      case value::px:
      case value::ppx:
        return i * 1000;
      default:
        return 0;
      }
    }

    double length_to_float() const {
      assert(is_length());
      return length_to_float(_i(), units());
    }

    bool is_length_literal() const {
      auto ut = units();
      return !(ut == px || ut == ppx || ut == in || ut == em || ut == rem || ut == cm || ut == mm ||
        ut == pt || ut == pr || ut == pr_view_width ||
        ut == pr_view_height || ut == pr_view_min || ut == pr_view_max ||
        ut == pc || ut == sp || ut == dip);
    }

    static double length_to_float(int i, int u) {
      switch (u) {
      case value::in:
      case value::pt: // Points (1 point = 1/72 inches).
      case value::pc: // Picas (1 pica = 12 points).
      case value::cm: // Cm (2.54cm = 1in).
      case value::mm:
      case value::em:
      case value::dip:
      case value::ex:
      case value::pr:
      case value::pr_width:
      case value::pr_height:
      case value::pr_view_width:
      case value::pr_view_height:
      case value::px:
      case value::ppx:
      case value::sp:
        return double(i) / 1000.0;
      default:
        return 0;
      }
    }
    static int packed_length(double i, int u) {
      switch (u) {
      case value::in:
      case value::pt: // Points (1 point = 1/72 inches).
      case value::pc: // Picas (1 pica = 12 points).
      case value::cm: // Cm (2.54cm = 1in).
      case value::mm:
      case value::em:
      case value::dip:
      case value::ex:
      case value::pr:
      case value::pr_width:
      case value::pr_height:
      case value::pr_view_width:
      case value::pr_view_height:
      case value::sp:
      case value::ppx:
      case value::px:
        return int(i * 1000.0);
      default:
        return 0;
      }
    }

    int _int() const { return _i(); }

    int  to_int() const { return get(0); }
    uint to_uint() const { return (uint)get(0); }
    int  get(int defv) const {
      switch (_type) {
      case t_bool:
      case t_int:
        return _i();
      case t_length:
        return length_to_int();
      case t_double:
        return (int)_d();
      case t_string:
        return wtoi(_us()->chars);
      case t_color:
        return _i();
      case t_enum:
        return (int)_units;
      default:
        return defv;
      }
    }

    double to_float() const { return get(0.0); }
    double get(double defv) const {
      switch (_type) {
      case t_bool:
      case t_int:
        return double(_i());
      case t_length:
        return length_to_float();
      case t_double:
      case t_duration:
      case t_angle:
        return _d();
      case t_string:
        return wtof(_us()->chars);
      case t_color:
        assert(false);
      default:
        return defv;
      }
    }

    float get(float defv) const {
      double d = get(double(defv));
      return float(d);
    }

    /*color_v to_color(color_v defv = color_v()) const
    {
    switch(_type)
    {
    case t_color:
    return color_v( this->get_packed_color() );
    default:
    return defv;
    }
    }*/

    bool to_bool() const { return get(false); }
    bool get(bool defv) const;

    ustring to_string() const { return get(W("")); }

#ifdef OSX
    // OSX thunk
    ustring get(const wchar_t *defv) const {
      assert(false);
      return get((const wchar *)defv);
    }
#endif

    ustring get(const wchar *defv) const;

    template <typename T> T get() const {
      return get_resource<std::remove_pointer_t<T>>(); // by default it tries to treat it as resource
    }
    
    static value parse(const ustring &us) {
      if (us.is_undefined())
        return value();
      double d;
      int    i;
      wchars input = trim(us());

      if (input.length == 0)
        return value(us());

      wchars s = input; if (parse_int(s, i) && !s) return value(i);
             s = input; if (parse_real(s, d) && !s) return value(d);

      if (input == WCHARS("true"))
        return value(true);
      if (input == WCHARS("false"))
        return value(false);
      if (input == WCHARS("null"))
        return value::null_val();
      date_time dt;
      uint      dtype = date_time::DT_UNDEFINED;
      dt = date_time::parse_iso(input, dtype);
      if (dtype & (date_time::DT_HAS_DATE | date_time::DT_HAS_TIME))
        return value::make_date(dt.time(), dtype);
      value t = parse_numeric_units(input);
      return t.is_undefined() ? value(us) : t;
    }

    static value parse_string(const ustring &us) { return value(us); }
    static value parse_integer(const ustring &us) {
      int    i;
      wchars s;
      s = us();
      if (tool::parse_int(s, i) && !s)
        return value(i);
      return value(us);
    }
    static value parse_float(const ustring &us) {
      double d;
      wchars s;
      s = us();
      if (tool::parse_real(s, d) && !s)
        return value(d);
      return value(us);
    }
    static value parse_numeric(const ustring &us) {
      double d;
      int    i;
      wchars s;
      s = us();
      if (tool::parse_int(s, i) && !s)
        return value(i);
      s = us();
      if (tool::parse_real(s, d) && !s)
        return value(d);
      return value(us);
    }

    template<typename>
    static value parse_enum(const ustring& us);

    inline static value none_value() {
      value v;
      v._type = t_null;
      v._units = 0xAFEE;
      return v;
    }
    /*static value parse(const string& us)
    {
    double d;
    int i;
    if( stoi(us,i) ) return value(i);
    if( stof(us,d) ) return value(d);
    if( us == "true" ) return value(true);
    if( us == "false" ) return value(false);
    return value(us);
    }*/

    template <typename CT>
    static value parse_units(double d, slice<CT> &s, bool as_css = true) {

      if (!s)
        return value(d);

      value v;

      switch (s++)
      {
      case 'e':
        if (*s == 'm') { v = value::make_length(d, em); ++s; }
        else if (*s == 'x') { v = value::make_length(d, ex); ++s; }
        break;
      case 'p':
        if (*s == 'x') { v = value::make_length(d, px); ++s; }
        else if (*s == 't') { v = value::make_length(d, pt); ++s; }
        else if (*s == 'c') { v = value::make_length(d, pc); ++s; }
        else if (*s == 'r') { v = value::make_length(d, pr); ++s; }
        break;
      case '%':
        if (!as_css)
          break;
        if (*s == '%') { v = value::make_length(d / 100, sp); ++s; }
        else  v = value::make_length(d, pr);
        break;
      case 'd':
        if (s.starts_with(CHARS("ip"))) { s.prune(2); v = value::make_length(d, dip); }
        else if (s.starts_with(CHARS("eg"))) { s.prune(2); v = make_angle(d, UT_DEG); }
        break;
      case 'i':
        if (*s == 'n') { v = value::make_length(d, in); ++s; }
        break;
      case 'c':
        if (*s == 'm') { v = value::make_length(d, cm); ++s; }
        break;
      case 'm':
        if (*s == 'm') { v = value::make_length(d, mm); ++s; }
        else if (*s == 's') { v = make_duration(d, UT_MS); ++s; }
        break;
      case 'f':
        if (*s == 'x') { v = value::make_length(d, sp); ++s; }
        break;
      case '*':
        if (!as_css)
          break;
        v = value::make_length(d, sp);
        break;
      case 'r':
        if (s.starts_with(CHARS("ad"))) { s.prune(2); v = make_angle(d, UT_RAD); }
        break;
      case 't':
        if (s.starts_with(CHARS("urn"))) { s.prune(3); v = make_angle(d, UT_TURN); }
        break;
      case 'g':
        if (s.starts_with(CHARS("rad"))) { s.prune(3); v = make_angle(d, UT_GRAD); }
        break;
      case 's':
        v = make_duration(d, UT_S);
        break;
      case 'v':
        if (*s == 'w') { v = value::make_length(d, pr_view_width); ++s; }
        else if (*s == 'h') { v = value::make_length(d, pr_view_height); ++s; }
        else if (s.starts_with(CHARS("min"))) { s.prune(3); v = value::make_length(d, pr_view_min); }
        else if (s.starts_with(CHARS("max"))) { s.prune(3); v = value::make_length(d, pr_view_max); }
        break;
      }
      return v;
    }

    // int, double, length , angle, duration
    static value parse_numeric_units(wchars &s, bool as_css = true) {

      double d;
      int    i;
      wchars t = s;
      if (!tool::parse_int(s, i))
        return value();
      if (!s)
        return value(i);
      s = t;
      if (!tool::parse_real(s, d))
        return value();

      return parse_units(d, s, as_css);
    }

    static value parse_numeric_units(const ustring& us)
    {
      wchars s = us();
      value v = parse_numeric_units(s);
      if (s.length == 0)
        return v;
      return value();
    }



    static value make_length(double i, uint ut) {
      assert(ut == px || ut == ppx || ut == in || ut == em || ut == cm || ut == mm ||
        ut == pt || ut == pr || ut == pr_view_width ||
        ut == pr_view_height || ut == pr_view_min || ut == pr_view_max ||
        ut == pc || ut == sp || ut == dip);
      value v;
      v._type = t_length;
      v._units = ut;
      v._i(packed_length(i, ut));
      return v;
    }

    static value make_raw_length(int i1000, uint ut) {
      assert(ut == px || ut == ppx || ut == in || ut == cm || ut == em || ut == mm ||
        ut == pt || ut == pr || ut == pr_view_width ||
        ut == pr_view_height || ut == pr_view_min || ut == pr_view_max ||
        ut == pc || ut == sp || ut == dip);
      value v;
      v._type = t_length;
      v._units = ut;
      v._i(i1000);
      return v;
    }

    static value make_literal_length(length_special_values lit) {
      value v;
      v._type = t_length;
      v._units = as;
      v._i(lit);
      return v;
    }

    static value make_spring_length(int i) {
      value v;
      v._type = t_length;
      v._units = sp;
      v._i(i);
      return v;
    }

    static value make_percent_length(int i) {
      value v;
      v._type = t_length;
      v._units = pr;
      v._i(m1000(i));
      return v;
    }

    static value make_ppx_length(double i) {
      value v;
      v._type = t_length;
      v._units = ppx;
      v._i(m1000(i)); // "micro pixels"
      return v;
    }
    static value make_ppx_length(int i) {
      value v;
      v._type = t_length;
      v._units = ppx;
      v._i(m1000(i)); // "micro pixels"
      return v;
    }

    static value make_functor(native_functor_holder *pf) {
      value v;
      v.set_resource(pf);
      return v;
    }

    native_functor_holder *get_native_functor() {
      if (!is_native_functor())
        return 0;
      return static_cast<native_functor_holder *>(get_resource());
    }

    bool equal(const value &rs) const;

    bool operator==(const value &rs) const { return equal(rs); }
    bool operator!=(const value &rs) const { return !equal(rs); }

    // value& operator [] (uint idx);

    // converts the value to array if needed and append the value to it
    void push(const value &v);

    // converts the value to map if needed and append the named value to it
    void push(const value &k, const value &v);

    // const value   operator [] (const value& k) const;
    //     value&   operator [] (const value& k);

    value get_prop(const value &k) const;
    void  set_prop(const value &k, const value &v);
    value get_prop(const char *k) const { return get_prop(value(k)); }
    void  set_prop(const char *k, const value &v) { set_prop(value(k), v); }
    value get_prop(const wchar *k) const { return get_prop(value(k)); }
    void  set_prop(const wchar *k, const value &v) { set_prop(value(k), v); }

    void  set_item(const char* k, const value& v) { set_prop(k, v); }
    void  set_item(const value &k, const value &v) { set_prop(k, v); }

    value key(uint n) const;

    void inherit(const value &from) {
      if (from.is_defined())
        *this = from;
    }

    static value null_val() {
      value t;
      t._type = t_null;
      t._units = 0;
      return t;
    }
    static value inherit_val() {
      value t;
      t._type = t_null;
      t._units = UT_INHERIT;
      return t;
    }
    static value cancel_val() {
      value t;
      t._type = t_null;
      t._units = UT_CANCEL;
      return t;
    }

    static value auto_val() {
      value t;
      t._type = t_null;
      t._units = UT_AUTO;
      return t;
    }


    //static const value undefined;

    bool visit(const kv_visitor &vis) const;
    bool mutate(const tool::v_mutator &vis);

    uint         size() const;
    slice<value> values() const;
    tslice<value> mutable_values();
    slice<key_value> key_values() const;

    void swap(value &v) {
      tool::swap(_type, v._type);
      tool::swap(_units, v._units);
      tool::swap(_data, v._data);
    }

    value& isolate();

    value call(slice<value> args, value _this = value())
    {
      tool::value v;
      switch (type())
      {
      case tool::value::t_resource: {
        tool::native_functor_holder *pf = get_native_functor();
        if (!pf) return value::make_error("is not a function");
        v = pf->call(uint(args.length), args.start);
        break;
      }
      case tool::value::t_object_proxy:
        v = get_proxy()->invoke(_this, uint(args.length), args.start);
        break;
      default:
        return value::make_error("is not a function");
      }
      return v;
    }

    typedef function<value(unsigned int argc, const value* argv)> native_function_t;
    value(const native_function_t& nfr);
    void set(const native_function_t& nfr);

    // versions of the above but for generic std::function
    template<typename R>
    inline value(std::function<R()> func):value()
    {
      native_function_t tf = [func](unsigned int argc, const value* argv) -> value { R r = func(); return value(r); };
      set(tf);
    }

    template<typename R, typename P0>
    inline value(std::function<R(P0)> func):value()
    {
      native_function_t tf = [func](unsigned int argc, const value* argv) -> value {
        R r = func(argc >= 1 ? argv[0].get<P0>() : P0());
        return value(r);
      };
      set(tf);
    }
    template<typename R, typename P0, typename P1>
    inline value(std::function<R(P0, P1)> func):value()
    {
      native_function_t tf = [func](unsigned int argc, const value* argv) -> value {
        R r = func(argc >= 1 ? argv[0].get<P0>() : P0(),
          argc >= 2 ? argv[1].get<P1>() : P1());
        return value(r);
      };
      set(tf);
    }

    template<typename R, typename P0, typename P1, typename P2>
    inline value(std::function<R(P0, P1, P2)> func):value()
    {
      native_function_t tf = [func](unsigned int argc, const value* argv) -> value {
        R r = func(argc >= 1 ? argv[0].get<P0>() : P0(),
          argc >= 2 ? argv[1].get<P1>() : P1(),
          argc >= 3 ? argv[2].get<P2>() : P2());
        return value(r);
      };
      set(tf);
    }

    template<typename R, typename P0, typename P1, typename P2, typename P3>
    inline value(std::function<R(P0, P1, P2, P3)> func):value() {
      native_function_t tf = [func](unsigned int argc, const value* argv) -> value {
        R r = func(argc >= 1 ? argv[0].get<P0>() : P0(),
          argc >= 2 ? argv[1].get<P1>() : P1(),
          argc >= 3 ? argv[2].get<P2>() : P2(),
          argc >= 4 ? argv[3].get<P3>() : P3());
        return value(r);
      };
      set(tf);
    }


  protected:
    uint _type;
    uint _units;

    union data {
      int64  i;
      double f;
      void * ptr;
    } _data;

    inline bool            _b() const { return _data.i != 0; }
    inline int             _i() const { return (int)(_data.i); }
    inline uint            _ui() const { return (uint)(0xfffffffff & _data.i); }
    inline int64           _i64() const { return _data.i; }
    inline double          _d() const { return _data.f; }
    inline string::data *  _s() const { return (string::data *)_data.ptr; }
    inline ustring::data * _us() const { return (ustring::data *)_data.ptr; }
    inline array_value *   _a() const { return (array_value *)_data.ptr; }
    inline object *        _obj() const { return (object *)_data.ptr; }
    inline function_value *_f() const { return (function_value *)_data.ptr; }
    inline map_value *     _m() const { return (map_value *)_data.ptr; }
    inline bytes_value *   _bytes() const { return (bytes_value *)_data.ptr; }
    inline object_proxy *  _proxy() const { return (object_proxy *)_data.ptr; }
    inline resource *      _res() const { return (resource *)_data.ptr; }
    inline som_asset_t *   _ass() const { return (som_asset_t *)_data.ptr; }
    inline const void *    _ptr() const { return _data.ptr; }

    inline void _b(bool b) { _data.i = b; }
    inline void _i(int i) { _data.i = i; }
    inline void _i64(int64 i) { _data.i = i; }
    inline void _d(double d) { _data.f = d; }
    inline void _s(string::data *s) {
      _data.i = 0;
      _data.ptr = s;
    }
    inline void _us(ustring::data *us) {
      _data.i = 0;
      _data.ptr = us;
    }
    inline void _a(array_value *a) {
      _data.i = 0;
      _data.ptr = a;
    }
    inline void _obj(object *o) {
      _data.i = 0;
      _data.ptr = o;
    }
    inline void _f(function_value *f) {
      _data.i = 0;
      _data.ptr = f;
    }
    inline void _m(map_value *f) {
      _data.i = 0;
      _data.ptr = f;
    }
    inline void _bytes(bytes_value *bv) {
      _data.i = 0;
      _data.ptr = bv;
    }
    inline void _proxy(object_proxy *pr) {
      _data.i = 0;
      _data.ptr = pr;
    }
    inline void _res(resource *pr) {
      _data.i = 0;
      _data.ptr = pr;
    }
    inline void _ass(som_asset_t *pr) {
      _data.i = 0;
      _data.ptr = pr;
    }
    inline void _ptr(const void *pr) {
      _data.i = 0;
      _data.ptr = (void*)pr;
    }
  };

#pragma pack(pop)

  template <> inline int      value::get<int>() const { return get(0); }
  template <> inline unsigned value::get<unsigned>() const { return (unsigned)get(0); }
  template <> inline bool     value::get<bool>() const { return get(false); }
  template <> inline double   value::get<double>() const { return get(0.0); }
  template <> inline float    value::get<float>() const { return (float)get(0.0); }
  template <> inline ustring  value::get<ustring>() const { return get(W("")); }
  template <> inline string   value::get<string>() const { return string(get(W(""))); }
  template <> inline value    value::get<value>() const { return *this; }

  struct named_value {
    tool::string name;
    tool::value  value;
  };

  struct function_value : public resource {
    ustring                  name;
    dictionary<value, value> params;

    function_value() {}
    function_value(const function_value& r) : name(r.name), params(r.params) {}

    bool equal(const function_value &rs) const {
      return name == rs.name && params == rs.params;
    }
    bool operator==(const function_value &rs) const { return equal(rs); }
    bool operator!=(const function_value &rs) const { return !equal(rs); }

    inline unsigned int hash() const {
      unsigned int d = name.hash();
      d ^= rotl(params.hash());
      return d;
    }

    value get(const value &key) const {
      value v;
      params.find(key, v);
      return v;
    }
    value get(const char *name) const { return get(value(name)); }
    value get(const wchar *name) const { return get(value(name)); }
    bool  has(const char *name) const { return params.exists(value(name)); }
    bool  has(const wchar *name) const { return params.exists(value(name)); }
    void  push(const value &k, const value &v) {
      if (k.is_defined())
        params[k] = v;
      else
        params.push(v);
    }

    ustring to_string() const;
  };

  inline value value::make_function(function_value *f) {
    value t;
    t._units = 0;
    t._type = t_function;
    if (!f)
      f = new function_value();
    f->add_ref();
    t._f(f);
    return t;
  }

  inline value value::make_function(const wchar* name) {
    value t;
    t._units = 0;
    t._type = t_function;
    function_value* f = new function_value();
    f->name = name;
    f->add_ref();
    t._f(f);
    return t;
  }

  inline value value::make_function(const wchar* name, value p) {
    value t;
    t._units = 0;
    t._type = t_function;
    function_value* f = new function_value();
    f->name = name;
    f->params.push(p);
    f->add_ref();
    t._f(f);
    return t;
  }

  inline value value::make_map(map_value *m) {
    value t;
    t._units = 0;
    t._type = t_map;
    if (!m)
      m = new map_value();
    m->add_ref();
    t._m(m);
    return t;
  }

  inline value value::make_array(uint sz) {
    value t;
    t._type = t_array;
    t._units = 0;
    array_value *a = new array_value();
    a->elements.size(sz);
    a->add_ref();
    t._a(a);
    return t;
  }

  inline value value::make_array(slice<value> values, uint units) {
    value t;
    t._type = t_array;
    t._units = units;
    array_value *a = new array_value();
    a->elements = values;
    a->add_ref();
    t._a(a);
    return t;
  }

  inline value value::make_array(array_value *a, uint units) {
    value t;
    t._type = t_array;
    t._units = units;
    a->add_ref();
    t._a(a);
    return t;
  }


  inline void value::push(const value &v) {
    if (!is_array())
      *this = make_array();
    get_array()->elements.push(v);
  }

  inline void value::push(const value &k, const value &v) {
    if (is_map())
      get_map()->push(k,v);
    else if (is_function())
      get_function()->push(k,v);
    else {
      *this = make_map();
      get_map()->push(k,v);
    }
  }

  inline value value::get_prop(const value &k) const {
    value v;
    if (is_map())
      get_map()->params.find(k, v);
    else if (is_function())
      get_function()->params.find(k, v);
    else if (is_proxy())
      return _proxy()->get_by_key(k);
    return v;
  }

  inline void value::set_prop(const value &k, const value &v) {
    if (is_undefined()) {
      *this = make_map();
      get_map()->params[k] = v;
    }
    else if (is_map())
      get_map()->params[k] = v;
    else if (is_function())
      get_function()->params[k] = v;
    else if (is_proxy())
      _proxy()->set_by_key(k, v);
    else
      assert(false);
  }

  inline value value::key(uint n) const {
    if (is_map())
      return get_map()->params.key(n);
    else if (is_function())
      return get_function()->params.key(n);
    return value();
  }

  inline void value::set(const value &cv) {
    if (&cv == this)
      return;
    clear();
    _type = cv._type;
    _units = cv._units;
    switch (cv._type) {
    case t_string: {
      ustring::data *p = cv._us();
      p->add_ref();
      _us(p);
    } break;
    case t_array: {
      array_value *p = cv._a();
      p->add_ref();
      _a(p);
    } break;
    case t_function: {
      function_value *p = cv._f();
      p->add_ref();
      _f(p);
    } break;
    case t_map: {
      map_value *p = cv._m();
      p->add_ref();
      _f(p);
    } break;
    case t_object: {
      object *p = cv._obj();
      p->add_ref();
      _obj(p);
    } break;
    case t_bytes: {
      bytes_value *p = cv._bytes();
      bytes_value::add_ref(p);
      _bytes(p);
    } break;
    case t_object_proxy: {
      object_proxy *p = cv._proxy();
      p->add_ref();
      _proxy(p);
    } break;
    case t_resource: {
      resource *p = cv._res();
      p->add_ref();
      _res(p);
    } break;
    case t_asset: {
      som_asset_t *p = cv._ass();
      sciter::om::asset_add_ref(p);
      _ass(p);
    } break;

    case t_undefined:
    case t_null:
    case t_bool:
    case t_int:
    case t_length:
    case t_date:
    case t_currency:
    case t_range:
    case t_duration:
    case t_angle:
    case t_color:
    case t_double:
    case t_enum: {
      _data = cv._data;
    } break;

    default:
      assert(false);
      break;
    }
  }

  inline value value::clone(const hash_table<ustring, value>& subst) const {
    value ov;
    switch (_type) {
    case t_string: {
      value v;
      if ((_units == UT_SYMBOL) && subst.find(this->to_string(), v))
        ov = v;
      else
        ov = *this;
    } break;
    case t_array: {
      ov = make_array(this->values(), _units);
      for (auto& v : ov.mutable_values())
        v = v.clone(subst);
    } break;
    case t_function: {
      ov = make_function(new function_value(*_f()));
      for (auto& e : ov._f()->params.elements())
        e.val = e.val.clone(subst);
    } break;
    case t_map: {
      ov = make_map(new map_value(*_f()));
      for (auto& e : ov._m()->params.elements())
        e.val = e.val.clone(subst);
    } break;
    case t_object:
    case t_bytes:
    case t_object_proxy:
    case t_resource:
    case t_undefined:
    case t_null:
    case t_bool:
    case t_int:
    case t_length:
    case t_date:
    case t_currency:
    case t_range:
    case t_duration:
    case t_angle:
    case t_color:
    case t_double:
    case t_enum: {
      ov = *this;
    } break;

    default:
      assert(false);
      break;
    }
    return ov;
  }


  inline void value::clear() {
    switch (_type) {
    case t_string: {
      ustring::data *p = _us();
      ustring::release_data(p, _units == UT_SECURE);
    } break;
    case t_array: {
      array_value *a = _a();
      a->release();
    } break;
    case t_function: {
      function_value *f = _f();
      f->release();
    } break;
    case t_map: {
      map_value *p = _m();
      p->release();
    } break;
    case t_object: {
      object *o = _obj();
      o->release();
    } break;
    case t_bytes: {
      bytes_value *p = _bytes();
      bytes_value::release(p);
    } break;
    case t_object_proxy: {
      object_proxy *p = _proxy();
      p->release();
    } break;
    case t_resource: {
      resource *p = _res();
      p->release();
    } break;
    case t_asset: {
      som_asset_t *p = _ass();
      sciter::om::asset_release(p);
    } break;
    case t_undefined:
    case t_null:
    case t_bool:
    case t_int:
    case t_length:
    case t_date:
    case t_currency:
    case t_range:
    case t_duration:
    case t_angle:
    case t_color:
    case t_double:
    case t_enum:
      break;
    default:
      assert(false);
      break;
    }
    _type = t_undefined;
    _units = 0;
    _i64(0);
  }

  inline value value::get_element(uint idx) const {
    if (is_array()) {
      if (idx < size())
        return _a()->elements[idx];
    }
    else if (is_map()) {
      map_value *pc = _m();
      if (idx < uint(pc->params.size()))
        return pc->params.value(idx);
    }
    else if (is_function()) {
      function_value *pc = _f();
      if (idx < uint(pc->params.size()))
        return pc->params.value(idx);
    }
    else if (is_proxy())
      return _proxy()->get_by_index(idx);
    return value();
  }

  inline bool value::set_element(uint idx, const value &v) {
    switch (_type) {
    case t_array: {
      array_value *a = _a();
      if (idx >= size())
        a->elements.size(idx + 1);
      a->elements[idx] = v;
      return true;
    }
    case t_object_proxy: {
      object_proxy *p = _proxy();
      return p->set_by_index(idx, v);
    }
    }
    return false;
  }

  inline bool value::get(bool defv) const {
    switch (_type) {
    case t_bool:
      return _b();
    case t_int:
    case t_length:
      return _i() != 0;
    case t_duration:
    case t_angle:
    case t_double:
      return (int)_d() != 0;
    case t_string: {
      ustring s(_us());
      if (s == WCHARS("false"))
        return false;
      return s.length() > 0;
    }
    case t_array:
      return _a()->elements.size() != 0;
    case t_map:
      return _m()->params.size() != 0;
    case t_object:
      return _obj() != 0;
    default:
      return defv;
    }
  }

  inline bool value::equal(const value &rs) const {
    if (_type != rs._type)
      return false;
    if (_i64() != rs._i64()) {
      if (_type == t_string)
        return _us()->length == rs._us()->length &&
        str_cmp(_us()->chars, rs._us()->chars) == 0;
      else if (_type == t_array)
        return _a()->elements == rs._a()->elements;
      else if (_type == t_function || _type == t_map)
        return _f()->equal(*rs._f());
      else if (_type == t_object_proxy)
        return _proxy()->equal(rs._proxy());
      return false;
    }
    return (_units == rs._units);
  }

  inline uint value::size() const {
    switch (_type) {
    case t_array:
      return uint(_a()->elements.size());
    case t_string:
      return uint(_us()->length);
    case t_map:
      return uint(_m()->params.size());
    case t_function:
      return uint(_f()->params.size());
    case t_object_proxy:
      return _proxy()->size();
    }
    assert(false);
    return 0;
  }

  inline slice<value> value::values() const {
    assert(is_array());
    if (is_array())
      return _a()->elements();
    return slice<value>();
  }

  inline tslice<value> value::mutable_values() {
    assert(is_array());
    if (is_array())
      return _a()->elements.target();
    return tslice<value>();
  }

  inline slice<value::key_value> value::key_values() const {
    assert(is_map() || is_function());
    if (is_map() || is_function())
      return _f()->params.elements();
    return slice<key_value>();
  }

  //tslice<value> mutable_values();

  inline unsigned int value::hash() const {
    switch (_type) {
      // case t_null:
      //  return "{undefined}";
    default:
      assert(false);
    case t_null:
      return _type + uint(_units);
    case t_bool:
      return _type + uint(_i());
    case t_int:
      return _type + 1 + uint(_i()) + _units;
    case t_length:
      return _type + 1 + uint(_i()) + _units;
    case t_duration:
    case t_angle:
    case t_double:
      return _type + uint(_i()) + _units;
    case t_color:
      return _type + hidword(_i64()) + lodword(_i64());
    case t_string:
      return ustring(_us()).hash();
    case t_function:
      return _f()->hash();
    case t_array:
      return _a()->hash();
    case t_undefined:
      return 0;
    case t_enum:
      return (uint)(uint_ptr)_ptr();
    }
    // return (unsigned int)_i();
  }

  inline bool enumerate(value &val, enumerator &en) {
    switch (val.type()) {
    case value::t_object: {
      object *obj = val.get_object();
      if (obj)
        obj->enumerate(en);
      return true;
    }
    case value::t_array: {
      slice<value> vals = val.values();
      for (uint n = 0; n < vals.length; ++n) {
        if (en.doit(vals[n]))
          continue;
        break;
      }
      return true;
    }
    }
    return false;
  }

  inline bool value::visit(const kv_visitor &vis) const {
    switch (type()) {
    case t_array: {
      slice<value> vals = values();
      value        k_dummy;
      for (uint n = 0; n < vals.length; ++n) {
        if (vis(k_dummy, vals[n]))
          continue;
        break;
      }
    }
    return true;
    case t_function: {
      function_value *mv = get_function();
      for (int n = 0; n < mv->params.size(); ++n) {
        if (vis(mv->params.key(n), mv->params.value(n)))
          continue;
        break;
      }
    }
    return true;
    case t_map: {
      map_value *mv = get_map();
      for (int n = 0; n < mv->params.size(); ++n) {
        if (vis(mv->params.key(n), mv->params.value(n)))
          continue;
        break;
      }
    }
    return true;
    case value::t_object_proxy: {
      object_proxy *obj = get_proxy();
      return obj->visit(vis);
    }
    }
    return false;
  }


  inline bool value::mutate(const v_mutator &vis) {
    switch (type()) {
    case t_array: 
      {
        array_value *mv = get_array();
        for (int n = 0; n < mv->elements.size(); ++n) {
          if (mv->elements[n].mutate(vis))
            continue;
          break;
        }
        return true;
      }
                  
    case t_function: 
      {
        function_value *mv = get_function();
        for (int n = 0; n < mv->params.size(); ++n) {
          if (mv->params.value(n).mutate(vis))
            continue;
          break;
        }
      }
      return true;
    case t_map: 
      {
        map_value *mv = get_map();
        for (int n = 0; n < mv->params.size(); ++n) {
          if (mv->params.value(n).mutate(vis))
            continue;
          break;
        }
      }
      return true;
    default: 
      return vis(*this);
    }
    return false;
  }


  template<typename F> 
  inline void flatten(const value& v, F f) {
    switch (v.type()) {
    case value::t_array: {
      slice<value> vals = v.values();
      for (auto& vi : vals)
        flatten(vi, f);
    } break;
    case value::t_function: {
      function_value *mv = v.get_function();
      for (int n = 0; n < mv->params.size(); ++n)
        flatten(mv->params.value(n), f);
    } break;
    case value::t_map: {
      map_value *mv = v.get_map();
      for (int n = 0; n < mv->params.size(); ++n)
        flatten(mv->params.value(n), f);
    } break;
    default:
      f(v);
      break;
    }
  }



  inline value& value::isolate() {
    if (is_proxy()) {
      value t;
      get_proxy()->isolate(t);
      *const_cast<value *>(this) = t;
    }
    else if (is_map() || is_function()) {
      function_value *pf = _f();
      for (int n = 0; n < pf->params.size(); ++n)
        pf->params.value(n).isolate();
    }
    else if (is_array()) {
      array_value *pa = _a();
      for (int n = 0; n < pa->elements.size(); ++n)
        pa->elements[n].isolate();
    }
    return *this;
  }

  inline bool value::is_function(wchars name) const {
    return is_function() && get_function()->name == name;
  }

  template <typename F> inline bool value::is_function(wchars name, F c) const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->name != name)
      return false;
    if (fv->params.size() != 1)
      return false;
    return c(fv->params.value(0));
  }

  template <typename TV> inline bool value::crack_function(wchars name, TV& c) const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->name != name)
      return false;
    if (fv->params.size() != 1)
      return false;
    c = fv->params.value(0).get<TV>();
    return true;
  }

  template <typename TV0, typename TV1> inline bool value::crack_function(wchars name, TV0& c0, TV1& c1) const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->name != name)
      return false;
    if (fv->params.size() != 2)
      return false;
    c0 = fv->params.value(0).get<TV0>();
    c1 = fv->params.value(1).get<TV1>();
    return true;
  }


  inline bool value::is_variable() const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->params.size() != 2)
      return false;
    if (fv->name != WCHARS("var"))
      return false;
    if (!fv->params.value(0).is_string())
      return false;
    return true;
  }

  inline bool value::get_variable_name_and_value(string& name, value& def_val) const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->name != WCHARS("var") && fv->name != WCHARS("color") && fv->name != WCHARS("length"))
      return false;
    if (!fv->params.value(0).is_string())
      return false;
    auto uname = fv->params.value(0).get<ustring>();
    name = uname().starts_with(WCHARS("--")) ? string(uname(2)) : string(uname());
    if (fv->params.size() >= 2)
      def_val = fv->params.value(1);
    return true;

  }

  inline bool value::is_variable_color() const {
    if (!is_function())
      return false;
    function_value *fv = get_function();

    //if (fv->name() == WCHARS("var") && fv->params.size() > 1 && fv->params.value(1).is_color())
    //  return true;

    static wchars funcnames[] = {
      WCHARS("rgb"),
      WCHARS("rgba"),
      WCHARS("color"),
      WCHARS("hsl"),
      WCHARS("hsv"),
      WCHARS("tint"),
      WCHARS("morph"),
      WCHARS("currentcolor")
    };
    return items_of(funcnames).contains(fv->name());
  }

  inline bool value::is_variable_length() const {
    if (!is_function())
      return false;
    function_value *fv = get_function();
    if (fv->name == WCHARS("length")) {
      if (fv->params.size() == 1 && fv->params.value(0).is_string())
        return true;
      return true;
    }
    return false;
  }

  namespace arithm
  {
    inline bool try_add(const tool::value &a1, const tool::value &a2, tool::value& r) {
      if (a1.is_pure_string() && a2.is_pure_string()) { r = value(a1.get<ustring>() + a2.get<ustring>()); return true; }
      if (a1.is_int() && a2.is_int()) { r = value(a1.get<int>() + a2.get<int>()); return true; }
      if (a1.is_double() && a2.is_double()) { r = value(a1.get<double>() + a2.get<double>()); return true; }
      if (a1.is_number() && a2.is_number()) { r = value(a1.get<double>() + a2.get<double>()); return true; }
      return false;
    }

    inline bool try_div(const tool::value &a1, const tool::value &a2, tool::value& r) {
      if (a1.is_int() && a2.is_int() && a2.get<int>()) { r = value(a1.get<int>() / a2.get<int>()); return true; }
      if (a1.is_double() && a2.is_double()) { r = value(a1.get<double>() / a2.get<double>()); return true; }
      if (a1.is_number() && a2.is_number()) { r = value(a1.get<double>() / a2.get<double>()); return true; }
      return false;
    }
    inline bool try_mod(const tool::value &a1, const tool::value &a2, tool::value& r) {
      if (a1.is_int() && a2.is_int() && a2.get<int>()) { r = value(a1.get<int>() % a2.get<int>()); return true; }
      return false;
    }


#define TRY_OP(F,OP) \
    inline bool F(const tool::value &a1, const tool::value &a2, tool::value& r) { \
      if (a1.is_int() && a2.is_int()) { r = value(a1.get<int>() OP a2.get<int>()); return true; } \
      if (a1.is_double() && a2.is_double()) { r = value(a1.get<double>() OP a2.get<double>()); return true; } \
      if (a1.is_number() && a2.is_number()) { r = value(a1.get<double>() OP a2.get<double>()); return true; } \
      return false; \
    }

    TRY_OP(try_sub, -)
      TRY_OP(try_mul, *)
      TRY_OP(try_and, &&)
      TRY_OP(try_or, || )

      TRY_OP(try_eq, == )
      TRY_OP(try_ne, != )
      TRY_OP(try_gt, >)
      TRY_OP(try_ge, >= )
      TRY_OP(try_lt, <)
      TRY_OP(try_le, <= )

#define TRY_OP_I(F,OP) \
    inline bool F(const tool::value &a1, const tool::value &a2, tool::value& r) { \
      if (a1.is_int() && a2.is_int()) { r = value(a1.get<int>() OP a2.get<int>()); return true; } \
      return false; \
    }

      TRY_OP_I(try_band, &)
      TRY_OP_I(try_bor, | )
      TRY_OP_I(try_bxor, ^)

      TRY_OP_I(try_lshift, << )
      TRY_OP_I(try_rshift, >> )

#define TRY_UNARY_OP(F,OP) \
    inline bool F(const tool::value &a1, tool::value& r) { \
      if (a1.is_int()) { r = value( OP a1.get<int>()); return true; } \
      if (a1.is_double()) { r = value(OP a1.get<double>()); return true; } \
      return false; \
    }

#define TRY_UNARY_OP_I(F,OP) \
    inline bool F(const tool::value &a1, tool::value& r) { \
      if (a1.is_int()) { r = value( OP a1.get<int>()); return true; } \
      return false; \
    }

      TRY_UNARY_OP(try_neg, -)
      TRY_UNARY_OP_I(try_bnot, ~)

      inline bool try_not(const tool::value &a1, tool::value& r) {
      if (a1.is_bool()) { r = value(!a1.get<bool>()); return true; }
      if (a1.is_int()) { r = value(!a1.get<int>()); return true; }
      if (a1.is_double()) { r = value(!a1.get<double>()); return true; }
      return false;
    }
  }


  template<class EDEF>
    inline value parse_enum(const ustring& us)
  {
    if (icmp(us(), WCHARS("inherit"))) { return value::inherit_val(); }
    else if (icmp(us(), WCHARS("unset"))) { return value::null_val(); }
    else {
      auto pdef = EDEF::pdef();
      for (auto it : pdef->items)
        if (icmp(us(), it.token))
          return value::make_enum(it.value, pdef);
    }
    return value();
  }

  template<class EDEF>
    inline bool coerce(value& v)
  {
    if (!v.is_string()) return false;
    if (v.is_inherit()) return true;
    if (v.is_none()) return true;
    ustring us = v.to_string();
    v = parse_enum<EDEF>(us);
    return v.is_enum();
  }

  inline const wchars enum_to_string(const value &v) {
    assert(v.is_enum());
    int ec = v.get_enum_code();
    const enum_def* pdef = v.get_enum_def();
    for (auto it : pdef->items)
      if (ec == it.value)
        return chars_of(it.token);
    return wchars();
  }

  template<typename T>
  inline value handle<T>::to_value() const {
    if (_ptr) return _ptr->to_value();
    return value();
  }

  template<typename T>
  inline value value_handle<T>::to_value() const {
    if (_ptr) return _ptr->to_value();
    return value();
  }


  template<typename CT, typename ACT>
    inline value string_t<CT, ACT>::to_value(uint ut) const {
      if (is_undefined())
        return value();
      else
        return value(*this, ut);
    }

#if !defined(SCITER) && !defined(SCITERJS)
  inline ustring color_to_string(const value& cv)
  {
    int64 bits = cv._i64();
    uint type = uint((bits >> 62) & 0x3);
    uint colr = uint(bits & 0xFFFFFFFFull);
    if (type == 0)
      return string::format("#%02x%02x%02x", colr & 0xff, (colr >> 8) & 0xff, (colr >> 16) & 0xff);
    else
      return string("color(...)");
  }
#endif

  class native_function : public native_functor_holder
  {
    value::native_function_t func;
  public:
    native_function(const value::native_function_t& f) : func(f) { assert(f); }
    virtual ~native_function() {}

    virtual value call(uint argc, const value *argv) 
    {
      if (func)
        return func(argc, argv);
      else
        return value();
    }
  };

  inline value::value(const native_function_t& nfr): value() {
    native_function* pnf = new native_function(nfr);
    set_resource(pnf);
  }
  inline void value::set(const native_function_t& nfr) {
    clear();
    native_function* pnf = new native_function(nfr);
    set_resource(pnf);
   }


} // namespace tool

inline tool::value operator "" _px(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::px); }
inline tool::value operator "" _ppx(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::ppx); }
inline tool::value operator "" _dip(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::dip); }
inline tool::value operator "" _fx(long double v) { return tool::value::make_length((double)v, tool::value::sp); }
inline tool::value operator "" _fx(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::sp); }
inline tool::value operator "" _pr(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::pr); }
inline tool::value operator "" _em(unsigned long long int v) { return tool::value::make_length((double)v, tool::value::em); }
inline tool::value operator "" _em(long double v) { return tool::value::make_length((double)v, tool::value::em); }

#define length_max_content tool::value::make_literal_length(tool::value::$max_content)
#define length_min_content tool::value::make_literal_length(tool::value::$min_content)
#define length_auto tool::value::make_literal_length(tool::value::$auto)
#define length_contain tool::value::make_literal_length(tool::value::$contain)


#pragma warning(pop)

#include "sdk-headers.h"

#endif
 
