/*
 *
 * cs_url.h
 *
 * Copyright (c) 2001, 2002
 * Andrew Fedoniouk - andrew@terra-informatica.org
 * Portions: Serge Kuznetsov -  kuznetsov@deeptown.org
 *
 * See the file "COPYING" for information on usage
 * and redistribution of this file
 *
 */
#ifndef __cs_url_h
#define __cs_url_h

#include "tl_dictionary.h"
#include "tl_string.h"
#include "tl_ustring.h"

namespace tool {
class url {
  /*
   * protocol://username:password@hostname:port/filename#anchor
   */
public:
  string src;

  string protocol;
  bool   protocol_path; // protocol looks like path - starts from "//"
  string hostname;
  int    port;
  int    dport;
  string filename;
  string anchor;

  string params;

  string data_type;
  string auth_type;

  string username;
  string password;

public:
  url() : port(0), dport(0), protocol_path(false) {}
  url(const char *src) : port(0), dport(0), protocol_path(false) {
    if (!parse(src))
      clear();
  }
  url(const wchar *src) : port(0), dport(0), protocol_path(false) {
    if (!parse(src))
      clear();
  }

  url(const url& other) : 
    src(other.src),
    protocol(other.protocol),
    protocol_path(other.protocol_path),
    hostname(other.hostname),
    port(other.port),
    dport(other.dport),
    filename(other.filename),
    anchor(other.anchor),
    params(other.params),
    data_type(other.data_type),
    auth_type(other.auth_type),
    username(other.username),
    password(other.password) {}

  ~url() {}

  bool parse(const char *src);
  bool parse(const wchar *src) { return parse(escape(chars_of(src))); }

  bool is_defined() const { return src.is_defined(); }

  void clear() {
    src.clear();
    protocol.clear();
    hostname.clear();
    port = dport = 0;
    filename.clear();
    anchor.clear();
    params.clear();
    data_type.clear();
    auth_type.clear();
    username.clear();
    password.clear();
  }

  static string  escape(chars url, bool space_to_plus = false, bool normalize_slash = true);
  static string  escape(wchars url, bool space_to_plus = false, bool normalize_slash = true);
  static string  escape_param(chars param);
  static string  escape_param(wchars param);
  static string  escape_file_path(chars param);
  static string  escape_file_path(wchars param);
  static string  path_to_file_url(ustring path);
  static ustring file_url_to_path(string url); static ustring file_url_to_path(chars url) { return file_url_to_path(string(url)); }  
  static ustring file_url_to_path(ustring url); static ustring file_url_to_path(wchars url) { return file_url_to_path(ustring(url)); }
  static string  file_url_to_u8_path(wchars url) { return u8::cvt(file_url_to_path(ustring(url))); }
  static ustring unescape(const char *src) { return unescape(chars_of(src)); }
  static ustring unescape(const string &src) { return unescape(src()); }
  static ustring unescape(chars src);
  static ustring unescape(const wchar *src) { return unescape(string(src)); }
  static ustring unescape(wchars src) { return unescape(string(src)); }
  static bool    looks_like_encoded(const string &s);
  static bool    looks_like_encoded(const ustring &s) { return looks_like_encoded(string(s)); }
  static bool    need_escapement(const ustring &s);

  bool equals(const url &doc, bool only_doc = false) {
    // case insensitive!
    if (protocol == doc.protocol && hostname() == doc.hostname &&
        (port == doc.port) && filename == doc.filename) {
      if (!only_doc)
        return anchor == doc.anchor;
      return true;
    }
    return false;
  }

  bool anchor_only() {
    return anchor.length() && hostname.is_empty() && filename.is_empty() && !port;
  }

  bool is_local() const {
    return !filename.is_empty() && (protocol.is_empty() || protocol == CHARS("file"));
  }
  bool is_external() const { return !hostname.is_empty() && (port != 0); }
  bool is_http_or_https() const { return !hostname.is_empty() && (protocol().starts_with(CHARS("http"))); }

  bool is_absolute() const {
    if (protocol == CHARS("file") && !filename.is_empty())
      return true;
    return port < 0 || (protocol.length() != 0 && hostname.length() != 0);
  }

  string dir() const;
  string name_ext() const; // file name and ext (without path)
  string name() const;     // file name only (without path)
  string ext() const;      // file ext

  string relative(const url &abspath) const;

  // make this (relative url) absolute one using abs as a base
  void absolute(const url &abs);

  void normalize_path(); // norm path - remove "." and ".."

  string compose(bool only_resource_name = false, bool no_anchor = false) const;
  string compose_host() const;
  string compose_without_anchor() const { return compose(false, true); }
  string compose_object() const;
};

string relpath(const string &absp, const string &basep);
string abspath(const string &absp, const string &relp);

inline ustring abspath(const ustring &absp, const ustring &relp) {
  return abspath(string(absp), string(relp));
}
inline ustring relpath(const ustring &absp, const ustring &basep) {
  return relpath(string(absp), string(basep));
}

bool is_hyperlink_char(wchar uc);
bool is_hyperlink(tool::ustring &text);
bool is_hyperlink(const tool::ustring &text, tool::ustring &out,
                  bool and_no_www = false);

inline string combine_url(string base, const string &rel) {
  if (rel().starts_with(CHARS("data:")))
    return rel;
  url urel(rel);
  if (urel.is_absolute()) {
    if (urel.protocol == CHARS("file") && !rel().starts_with(CHARS("file")))
      return url::path_to_file_url(rel);
    return rel;
  }
  url ubase(base);
  // if(!ubase.is_absolute())
  //  return rel;
  urel.absolute(ubase);
  return urel.compose();
}

inline string combine_url(const url& ubase, const string &rel) {
  if (rel().starts_with(CHARS("data:")))
    return rel;
  url urel(rel);
  if (urel.is_absolute()) {
    if (urel.protocol == CHARS("file") && !rel().starts_with(CHARS("file")))
      return url::path_to_file_url(rel);
    return rel;
  }
  urel.absolute(ubase);
  return urel.compose();
}


}; // namespace tool

#endif
