//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| tokenizer impl.
//|
//|

#include "tool.h"

namespace tool {
  enum parser_states {
    IN_WHITE,
    IN_TOKEN,
    IN_QUOTE,
  };

  tokenz::tokenz(const char *text, size_t text_length, cvt_flag flag)
    : _text(text), _text_end(text + text_length), _pos(text), _prev_pos(text),
    _token_pos(text), _p_flag(flag), _p_state(IN_WHITE), _p_curquote(0),
    _whites(" \t\r"), // blank and tab
    _sbreak(",;=\n"), // comma and carriage return
    _quotes("'\""),   // single and double quote
    _eschar(0),       // "bakslash" is escape
    _break_used(0), _quote_used(0), _skip_cpp_comments(false),
    _skip_c_comments(false) {}

  tokenz::tokenz(const char *text, const char *text_end, cvt_flag flag)
    : _text(text), _text_end(text_end), _pos(text), _prev_pos(text),
    _token_pos(text), _p_flag(flag), _p_state(IN_WHITE), _p_curquote(0),
    _whites(" \t\r"), // blank and tab
    _sbreak(",;=\n"), // comma and carriage return
    _quotes("\'\""),  // single and double quote
    _eschar(0),       // "bakslash" is escape
    _break_used(0), _quote_used(0), _skip_cpp_comments(false),
    _skip_c_comments(false) {}

  // look up character in string
  int tokenz::sindex(char ch, const char *str) {
    const char *cp;
    for (cp = str; *cp; ++cp)
      if (ch == *cp)
        return (int)(cp - str); // return postion of character
    return -1;                  // eol ... no match found
  }

  string tokenz::token_value() {
    if (_p_state == IN_QUOTE)
      return _token;

    switch (_p_flag) {
    case cvt_to_upper: // convert to upper
      return _token.to_upper();
    case cvt_to_lower: // convert to lower
      return _token.to_lower();
    default: // use as is
      return _token;
    }
  }

  // here it is!
  int tokenz::token() {
    _prev_pos = _pos;

    if (_pos >= _text_end)
      return TT_END_OF_TEXT;

    int  qp;
    int  comment_type;
    char c, nc;

    _break_used = 0; // initialize to null
    _quote_used = 0; // assume not quoted

    _token.clear();

    _p_state = IN_WHITE; // initialize state
    _p_curquote = 0;        // initialize previous quote char

    for (; _pos < _text_end; ++_pos) // main loop
    {
      comment_type = is_comment_start();
      if (comment_type) {
        switch (_p_state) {
        case IN_WHITE:
          skip_comment(comment_type);
          continue;
        case IN_TOKEN:
          skip_comment(comment_type);
          return TT_WORD_VALUE;
        case IN_QUOTE:
          // it's plain text here
          break;
        }
      }

      c = *_pos;

      if ((qp = sindex(c, _sbreak)) >= 0) // break
      {
        switch (_p_state) {
        case IN_WHITE:
          _token_pos = _pos;
          ++_pos;
          _break_used = _sbreak[qp];
          return TT_BREAK_CHAR;

        case IN_TOKEN: // ... get out
          _break_used = _sbreak[qp];
          return TT_WORD_VALUE;

        case IN_QUOTE: // keep going
          _token += c;
          break;
        }
      }
      else if ((qp = sindex(c, _quotes)) >= 0) // quote
      {
        switch (_p_state) {
        case IN_WHITE:               // these are identical,
          _p_state = IN_QUOTE;    // change states
          _p_curquote = _quotes[qp]; // save quote char
          _quote_used = _p_curquote; // set it as long as
          _token_pos = _pos;
          break; // something is in quotes

        case IN_QUOTE:
          if (_quotes[qp] == _p_curquote) // same as the beginning quote?
          {
            _p_state = IN_WHITE;
            _p_curquote = 0;
            ++_pos;
            return TT_STRING_VALUE;
          }
          else
            _token += c; // treat as regular char
          break;

        case IN_TOKEN:
          _break_used = c; // uses quote as break char
                           //_pos++;
          _token_pos = _pos;
          return TT_WORD_VALUE;
        }
      }
      else if ((qp = sindex(c, _whites)) >= 0) // white
      {
        switch (_p_state) {
        case IN_WHITE:
          break; // keep going

        case IN_TOKEN:
          ++_pos;
          return TT_WORD_VALUE;

        case IN_QUOTE:
          _token += c; // it's valid here
          break;
        }
      }
      else if (c == _eschar && (_pos < (_text_end - 1))) // escape
      {
        nc = *(_pos + 1);
        switch (_p_state) {
        case IN_WHITE:
          --_pos;
          _token_pos = _pos;
          _p_state = IN_TOKEN;
          break;
        case IN_TOKEN:
        case IN_QUOTE:
          ++_pos;
          _token += nc;
          break;
        }
      }
      else // anything else is just a real character
      {
        switch (_p_state) {
        case IN_WHITE:
          _p_state = IN_TOKEN; // switch states
          _token_pos = _pos;
        case IN_TOKEN: // these too are
        case IN_QUOTE: // identical here
          _token += c;
          break;
        }
      }
    }
    // main loop

    switch (_p_state) {
    case IN_TOKEN:
    case IN_QUOTE:
      return TT_WORD_VALUE;
    }
    return TT_END_OF_TEXT;
  }

  void tokenz::push_back() {
    // assert(_pos != _prev_pos);
    _pos = _prev_pos;
  }

  int tokenz::is_comment_start() {
    if (*_pos == '/') {
      int nc = *(_pos + 1);
      if (_skip_cpp_comments && nc == '/') {
        _pos += 2;
        return 1;
      }
      if (_skip_c_comments && nc == '*') {
        _pos += 2;
        return 2;
      }
    }
    return 0;
  }
  void tokenz::skip_comment(int type) {
    if (type == 1) // cpp comment
      while (_pos < _text_end) {
        if (*_pos++ == '\n')
          break;
      }
    else if (type == 2) // c comment
      while (_pos < _text_end) {
        if (*_pos++ == '*') {
          if (*_pos == '/') {
            ++_pos;
            break;
          }
        }
      }
  }

  bool style_parser::parse_style_def(string &                    name,
    hash_table<string, string> &atts) {
    if (zz.token() != tokenz::TT_WORD_VALUE)
      return false;

    name = zz.token_value();

    if ((zz.token() != tokenz::TT_BREAK_CHAR) || (zz.break_used() != '{'))
      return 0;

    int t = 0;

    string attrname;
    string attrvalue;

    t = zz.token();

    while (true) {

      if ((t == tokenz::TT_BREAK_CHAR) && (zz.break_used() == '}'))
        return true;

      if ((t == tokenz::TT_BREAK_CHAR) && (zz.break_used() == ';')) {
        t = zz.token();
        continue;
      }

      if ((t != tokenz::TT_WORD_VALUE) && (t != tokenz::TT_STRING_VALUE))
        return false;

      attrname = zz.token_value();

      t = zz.token();
      if ((t != tokenz::TT_BREAK_CHAR) || (zz.break_used() != ':'))
        return false;

      t = zz.token();
      if ((t != tokenz::TT_WORD_VALUE) && (t != tokenz::TT_STRING_VALUE))
        return false;

      string val = zz.token_value();

      for (t = zz.token();
        t == tokenz::TT_WORD_VALUE || t == tokenz::TT_STRING_VALUE;
        t = zz.token())
        val += zz.token_value();
      atts[attrname] = val;
    }
    return false;
  }

  bool parse_named_values(const string &txt, hash_table<string, string> &atts) {
    tokenz zz(txt, txt.length());
    zz.break_chars(";:{}");
    zz.escape_char(1);
    zz.quote_chars("");
    zz.white_chars(" \t\r\n");

    string attrname;
    string attrvalue;

    for (int t = zz.token(); t; t = zz.token()) {

      if ((t == tokenz::TT_BREAK_CHAR) && (zz.break_used() == ';')) {
        continue;
      }

      if (t != tokenz::TT_WORD_VALUE)
        return false;

      attrname = zz.token_value();

      t = zz.token();
      if ((t != tokenz::TT_BREAK_CHAR) || (zz.break_used() != ':'))
        return false;

      t = zz.token();
      if ((t != tokenz::TT_WORD_VALUE) && (t != tokenz::TT_STRING_VALUE))
        return false;

      string val = zz.token_value();
      atts[attrname] = val;
    }
    return false;
  }

  bool parse_named_values(const ustring &              utxt,
    hash_table<string, ustring> &atts) {
    string txt(utxt);
    tokenz zz(txt, txt.length());
    zz.break_chars(";:{}");
    zz.escape_char(1);
    zz.quote_chars("'\"");
    zz.white_chars(" \t\r\n");

    string attrname;
    string attrvalue;

    for (int t = zz.token(); t; t = zz.token()) {

      if ((t == tokenz::TT_BREAK_CHAR) && (zz.break_used() == ';')) {
        continue;
      }

      if (t != tokenz::TT_WORD_VALUE)
        return false;

      attrname = zz.token_value();

      t = zz.token();
      if ((t != tokenz::TT_BREAK_CHAR) || (zz.break_used() != ':'))
        return false;

      t = zz.token();
      if ((t != tokenz::TT_WORD_VALUE) && (t != tokenz::TT_STRING_VALUE))
        return false;

      string val = zz.token_value();
      atts[attrname] = val;
    }
    return true;
  }

  namespace xjson {

    scanner::token_t scanner::get_index_token(wchars &value) {
      value = wchars();
      token_t t = get_token();
      if (t == '.') {
        t = get_token();
        if (t == T_NAME) {
          value = get_value();
          t = T_INDEX_NAME;
        }
        else
          t = T_ERROR;
      }
      else if (t == '[') {
        token_t tt = get_token();
        if (tt == T_NUMBER) {
          value = get_value();
          t = T_INDEX_NUMBER;
        }
        else if (tt == T_STRING) {
          value = get_value();
          t = T_INDEX_NAME;
        }
        else
          t = T_ERROR;
        tt = get_token();
        if (tt != ']')
          t = T_ERROR;
      }
      else if (t == T_NAME)
        value = get_value();
      return t;
    }

    void scanner::skip_comment(bool toeol) {
      if (toeol)
        while (pos < end)
          if (*pos == '\n') {
            ++pos;
            break;
          }
          else
            ++pos;
      else
        while (pos < end)
          if (pos < (end - 1) && pos[0] == '*' && pos[1] == '/') {
            pos += 2;
            break;
          }
          else
            ++pos;
    }

    scanner::token_t scanner::get_token() {
      if (saved_token) {
        token_t t = saved_token;
        saved_token = T_END;
        return t;
      }

      token_value.size(0);

      if (!scan_ws())
        return T_END;

      switch (*pos) {
      case '+':
      case '-':
        if (pos < (end - 1) && isdigit(*(pos + 1)))
          return scan_number();
      case '/':
        if (pos < (end - 1)) {
          if (*(pos + 1) == '/') {
            pos += 2;
            skip_comment(true);
            return get_token();
          }
          if (*(pos + 1) == '*') {
            pos += 2;
            skip_comment(false);
            return get_token();
          }
          // else fall through
        }
      case '*':
      case '%':
      case '^':
      case '&':
      case ';':
      case '.':
      case '?':
      case '=':
      case '[':
      case ']':
      case '{':
      case '}':
        return token_t(*pos++);
      case '\"':
      case '\'':
        return scan_string(*pos++);
      case ')':
        return token_t(*pos++);
      case '(':
        if (canonic)
          return token_t(*pos++);
        else {
          ++pos;
          return scan_parenthesis();
        }
      case '#':
        if (canonic)
          return token_t(*pos++);
        else {
          ++pos;
          return scan_color();
        }
      default:
        if (isdigit(*pos))
          return scan_number();
        if (isalpha(*pos) /*WTF? || *pos == '!'*/ || *pos == '_')
          return scan_nmtoken();
        else
          return token_t(*pos++);
      }
      // return T_END;
    }

    scanner::token_t scanner::scan_number() {
      int  n_dots = 0;
      bool is_currency = false;

      if (*pos == '-' || *pos == '+')
        token_value.push(*pos++);
      else if ((*pos == '0') && ((pos + 2) < end)) {
        if ((*(pos + 1) == 'x' || *(pos + 1) == 'X') && isxdigit(*(pos + 2)))
          goto HEX_NUMBER;
        if (!canonic && (*(pos + 1) == 'd' || *(pos + 1) == 'D') &&
          isdigit(*(pos + 2)))
          goto DATE_LITERAL;
      }

      while (pos < end) {
        if (isdigit(*pos))
          token_value.push(*pos);
        else if (*pos == '.') {
          if (++n_dots > 1)
            break;
          token_value.push(*pos);
        }
        else if (!canonic && *pos == '$') {
          if (++n_dots > 1)
            break;
          is_currency = true;
          token_value.push(*pos);
        }
        else if (*pos == 'e' || *pos == 'E') {
          token_value.push(*pos++);
          if ((*pos == '-' || *pos == '+') && isdigit(*(pos + 1)))
            token_value.push(*pos++);
          while (pos < end) {
            if (isdigit(*pos))
              token_value.push(*pos++);
            else
              break;
          }
          break;
        }
        else
          break;
        ++pos;
      }
      return is_currency ? T_CURRENCY : T_NUMBER;
    HEX_NUMBER:
      // skip '0x'
      token_value.push(pos, 2);
      pos += 2;
      while (pos < end) {
        if (isxdigit(*pos))
          token_value.push(*pos);
        else
          break;
        ++pos;
      }
      return T_NUMBER;
    DATE_LITERAL:
      // skip '0d'
      pos += 2;
      while (pos < end) {
        if (wcschr(L"0123456789-+TZtz:", *pos))
          token_value.push(*pos);
        else
          break;
        ++pos;
      }
      return T_DATETIME;
    }

    scanner::token_t scanner::scan_color() {
      while (pos < end) {
        if (isxdigit(*pos))
          token_value.push(*pos);
        else
          break;
        ++pos;
      }
      return T_COLOR;
    }

    wchar scanner::scan_escape() {
      // *pos is '\'
      switch (*(++pos)) {
      case 't':
        ++pos;
        return '\t';
      case 'r':
        ++pos;
        return '\r';
      case 'n':
        ++pos;
        return '\n';
      case 'f':
        ++pos;
        return '\f';
      case 'b':
        ++pos;
        return '\b';
      case '\\':
        ++pos;
        return '\\';
      case 'u':
        ++pos;
        if (pos < (end - 4) && isxdigit(pos[0]) && isxdigit(pos[1]) &&
          isxdigit(pos[2]) && isxdigit(pos[3])) {
          char *e;
          char  bf[5];
          bf[0] = char(pos[0]);
          bf[1] = char(pos[1]);
          bf[2] = char(pos[2]);
          bf[3] = char(pos[3]);
          bf[4] = 0;
          pos += 4;
          return (wchar)strtol(bf, &e, 16);
        }
      default: { return *pos++; }
      }
      // return '?'; // to make compiler happy
    }

    scanner::token_t scanner::scan_string(wchar delimeter) {
      while (pos < end) {
        if (*pos == delimeter) {
          ++pos;
          break;
        }
        else if (*pos == '\\')
          token_value.push(scan_escape());
        else {
          token_value.push(*pos);
          ++pos;
        }
      }
      return T_STRING;
    }

    scanner::token_t scanner::scan_parenthesis() {
      // caller consumed '('
      int level = 0;
      if (!scan_ws())
        return T_END;
      while (pos < end) {
        if (*pos == ')') {
          if (level == 0) {
            ++pos;
            break;
          }
          else {
            --level;
            token_value.push(*pos);
          }
        }
        else if (*pos == '(') {
          ++level;
          token_value.push(*pos);
        }
        else if (*pos == '\\')
          token_value.push(scan_escape());
        else
          token_value.push(*pos);
        ++pos;
      }

      while (token_value.size())
        if (isspace(token_value.last()))
          token_value.pop();
        else
          break;

      return T_STRING;
    }

    scanner::token_t scanner::scan_nmtoken() {
      token_value.push(*pos++);
      while (pos < end) {
        if (isalnum(*pos) || *pos == '_' || *pos == '-')
          token_value.push(*pos);
        else
          break;
        ++pos;
      }
      return T_NAME;
    }

    static value parse_value(scanner &s, scanner::token_t tok);
    static value parse_array(scanner &s);
    static value parse_map(scanner &s);
    static value parse_name(scanner &s);

    struct parse_error {
      ustring errmsg;

      parse_error(scanner &s) {
        int line = 0;
        int pos = 0;
        s.get_parsed_location(line, pos);
        errmsg = ustring::format(W("JSON parsing error in line %d at %d position"),
          line, pos);
      }
    };

    value parse_array(scanner &s) {
      value a = value::make_array();
      scanner::token_t tok;
      // caller consumed '['
      for (tok = s.get_token(); tok; tok = s.get_token()) {
        if (tok == ']')
          goto END;
        value v = parse_value(s, tok);
        a.push(v);
        switch (uint(tok = s.get_token())) {
        case ']':
          goto END;
        case scanner::T_NAME:
        case scanner::T_NUMBER:
        case scanner::T_DATETIME:
        case scanner::T_STRING:
        case scanner::T_COLOR:
        case '{':
        case '[':
          s.push_back(tok);
          continue; // relaxing json rules
        case ',':
        case ';':
          continue;
        default:
          throw parse_error(s);
        }
      }
    END:
      return a;
    }

    value parse_map(scanner &s) {
      value            m = value::make_map();
      scanner::token_t tok;
      // caller consumed '{' /////
      for (tok = s.get_token(); tok; tok = s.get_token()) {
        if (tok == '}')
          return m;
        value k = parse_value(s, tok);
        if (s.get_token() != ':')
          throw parse_error(s);
        tok = s.get_token();
        value v = parse_value(s, tok);
        m.push(k, v);
        tok = s.get_token();
        switch (uint(tok)) {
        case '}':
          return m;
        case tool::xjson::scanner::T_NAME:
        case tool::xjson::scanner::T_NUMBER:
        case tool::xjson::scanner::T_DATETIME:
        case tool::xjson::scanner::T_STRING:
        case tool::xjson::scanner::T_COLOR:
        case '{':
        case '[':
          s.push_back(tok);
          continue; // relaxing json rules
        case ',':
        case ';':
          continue;
        case tool::xjson::scanner::T_END:
          break;
        default:
          throw parse_error(s);
        }
      }
      return m;
    }

    value parse_number(scanner &s) {
      wchars t = trim(s.get_value());
      wchars parsed = t;

      int64 il = to_int64(parsed);

      if (parsed.length == t.length) // not a number
        return value();

      if (parsed.length == 0) { // parsed in full

        if (il > 0x7fffffffll && il <= 0xffffffffll)
          return value(int(il));
        if (il > (int64)limits<uint>::max_value() ||
          il < (int64)limits<int>::min_value())
          return value(double(il));
        return value(int(il));
      }
      double d = 0.0;
      parse_real(t, d);
      return value(d);
    }

    value parse_currency(scanner &s) {
      wchars t = s.get_value();
      int    dp = t.index_of('$');
      if (dp < 0)
        return value();
      wchars tmp(t.start, dp); // You can't bind non-constant references
                               // to anonymous variables!
      int64 dollars = to_int64(tmp);
      t.prune(dp + 1);
      if (t.length > 4)
        t.length = 4;
      wchars tmp2(t);
      int    cents = to_uint(tmp2);
      switch (t.length) {
      case 1:
        cents *= 1000;
        break;
      case 2:
        cents *= 100;
        break;
      case 3:
        cents *= 10;
        break;
      case 4: // cents *= 10;
        break;
      }
      return value::make_currency(dollars * 10000 + cents);
    }

    value parse_date(scanner &s) {
      wchars    t = s.get_value();
      uint      dt;
      date_time d = date_time::parse_iso(t, dt);
      return value::make_date(d.time(), dt);
      // return value();
    }

    value parse_name(scanner &s) {
      wchars t = s.get_value();
      if (t == WCHARS("true"))
        return value(true);
      else if (t == WCHARS("false"))
        return value(false);
      else if (t == WCHARS("null"))
        return value::null_val();
      else if (t == WCHARS("undefined"))
        return value();
      return value(ustring(t), 0xFFFF);
    }

    value parse_color(scanner &s) {
      string text = s.get_value();
      if (text.length() == 0)
        return value();
      uint R = 0, G = 0, B = 0, T = 0;
      switch (text.length()) {
      case 3:
        sscanf(text, "%1x%1x%1x", &R, &G, &B);
        R = R << 4 | R;
        G = G << 4 | G;
        B = B << 4 | B;
        break;
      case 4:
        sscanf(text, "%1x%1x%1x%1x", &R, &G, &B, &T);
        R = R << 4 | R;
        G = G << 4 | G;
        B = B << 4 | B;
        T = T << 4 | T;
        break;
      case 6:
        sscanf(text, "%2x%2x%2x", &R, &G, &B);
        break;
      case 8:
        sscanf(text, "%2x%2x%2x%2x", &R, &G, &B, &T);
        break;
      default:
        return value();
      }
      return value::make_packed_color((T << 24) | (B << 16) | (G << 8) | R);
    }

    value parse_value(scanner &s, scanner::token_t tok) {
      switch (uint(tok)) {
      case '{':
        return parse_map(s);
      case '[':
        return parse_array(s);
      case scanner::T_NUMBER:
        return parse_number(s);
      case scanner::T_CURRENCY:
        return parse_currency(s);
      case scanner::T_DATETIME:
        return parse_date(s);
      case scanner::T_COLOR:
        return parse_color(s);
      case scanner::T_NAME:
        return parse_name(s);
      case scanner::T_STRING: {
        wchars t = s.get_value();
        return value(t);
      }
      // case '[': return parse_map(s);
      default:
        throw parse_error(s);
      }
      // return value();
    }

    value parse(wchars &text, bool open_model, function<void(wchars)> on_error) {
      scanner          s(text);
      scanner::token_t tok;
      try {
        if (open_model)
          return parse_map(s);
        else {
          tok = s.get_token();
          return parse_value(s, tok);
        }
      }
      catch (parse_error &e) {
        text = s.get_nonparsed();
        if (on_error) {
          on_error(e.errmsg);
          return value();
        }
        else {
          return value::make_error(e.errmsg);
        }
      }
      // return value();
    }

    // JSON emitter

    class emitter {
    public:
      emitter(MODE m = XJSON) : mode(m), tabs(0) {}
      emitter(const value &v, MODE m = XJSON, bool open_model = false)
        : mode(m), tabs(0) {
        emit(v, open_model);
      }

      void emit(const value &v, bool open_model) {
        mem_stream_o<wchar> out(out_buf);
        if ((v.type() == value::t_map) && open_model)
          emit_map(v, out);
        else
          emit_value(v, out);
      }

      void emit_value(const value &v, mem_stream_o<wchar> &out) {
        switch (v.type()) {
        case value::t_undefined:
          if (mode == JSON)
            out.put(WCHARS("null"));
          else
            out.put(WCHARS("undefined"));
          break;
        case value::t_null:
        case value::t_bool:
        case value::t_int:
        case value::t_double:
        case value::t_length:
        case value::t_color:
        case value::t_enum:
          out.put(v.to_string());
          break;
        case value::t_currency:
          emit_currency(v, out);
          break;
        case value::t_date:
          emit_date(v, out);
          break;
        case value::t_string:
          emit_string(v, out);
          break;
        case value::t_map:
          out.put('{');
          emit_map(v, out);
          out.put('}');
          break;
        case value::t_function:
          out << v.get_function()->to_string();
          break;
        case value::t_array:
          out.put('[');
          emit_array(v, out);
          out.put(']');
          break;
        case value::t_object_proxy: {
          value iv = v;
          iv.isolate();
          emit_value(iv, out);
        } break;
        }
      }

      void emit_array(const value &v, mem_stream_o<wchar> &out) {
        ++tabs;
        array_value *av = v.get_array();
        for (int n = 0; n < av->elements.size(); ++n) {
          if (n)
            out.put(WCHARS(","));
          emit_value(av->elements[n], out);
        }
        --tabs;
      }

      void emit_map(const value &v, mem_stream_o<wchar> &out) {
        ++tabs;
        map_value *mv = v.get_map();
        for (int n = 0; n < mv->params.size(); ++n) {
          if (mv->params.value(n).is_undefined() && mode == JSON)
            continue;
          if (n)
            out.put(',');
          emit_nl(out);
          emit_value(mv->params.key(n), out);
          out.put(WCHARS("\t:"));
          emit_value(mv->params.value(n), out);
        }
        --tabs;
      }

      void emit_date(const value &v, mem_stream_o<wchar> &out) {
        if (mode == XJSON) {
          out.put(WCHARS("0d"));
          out.put(v.to_string());
        }
        else {
          out.put('"');
          out.put(v.to_string());
          out.put('"');
        }
      }

      void emit_string(const value &v, mem_stream_o<wchar> &out) {
        ustring us = v.get(W(""));
        if (mode == XJSON && v.units() == 0xFFFF && is_nmtoken(us))
          out.put(us);
        else
          emit_string_literal(us, out);
      }

      void emit_nl(mem_stream_o<wchar> &out) {
        out.put(WCHARS("\r\n"));
        for (int t = 0; t < tabs; ++t)
          out.put('\t');
      }

      void emit_string_literal(const ustring &us, mem_stream_o<wchar> &out) {
        out.put('"');

        const wchar *p = us;
        const wchar *end = p + us.length();
        for (; p < end; ++p) {
          switch (*p) {
          case '"':
            out.put('\\');
            out.put('"');
            continue;
          case '\\':
            out.put('\\');
            out.put('\\');
            continue;
          case '\b':
            out.put(WCHARS("\\b"));
            continue;
          case '\f':
            out.put(WCHARS("\\f"));
            continue;
          case '\n':
            out.put(WCHARS("\\n"));
            continue;
          case '\r':
            out.put(WCHARS("\\r"));
            continue;
          case '\t':
            out.put(WCHARS("\\t"));
            continue;
          default:
            if (*p < ' ') {
              out.put(WCHARS("\\u")); // case \u four-hex-digits
              out.put(ustring::format(W("%.4x"), *p));
              continue;
            }
            break;
          }
          out.put(*p);
        }
        out.put('"');
      }

      bool is_nmtoken(const ustring &us) {
        const wchar *p = us;
        const wchar *end = p + us.length();
        if (isdigit(*p) || *p == '-')
          return false;
        for (; p < end; ++p) {
          if (is_alnum(*p))
            continue;
          if (*p == '_' || *p == '-')
            continue;
          return false;
        }
        return true;
      }

      void emit_currency(const value &v, mem_stream_o<wchar> &out) {
        int64 li = v.get_int64();
        int64 dollars = li / 10000;
        uint  cents = uint(uint64(li) % 10000);

        i64tow sd(dollars);
        out.put(sd, sd.length);
        if (mode == XJSON)
          out.put('$');
        if (cents == 0)
          return;
        if (mode != XJSON)
          out.put('.');
        itow   sc(cents, 10, 4);
        wchars scs(sc, sc.length);
        while (scs.last() == '0')
          scs.prune(0, 1);
        out.put(scs);
      }

      MODE         mode;
      array<wchar> out_buf;
      int          tabs;
    };

    ustring emit(const value &v, MODE m, bool open_model) {
      emitter em(m);
      em.emit(v, open_model);
      return em.out_buf();
    }

#if defined(_DEBUG) && 0
    struct unit_test {
      unit_test() {
        {
          value   vcur = value(0x80000000);
          ustring outs = emit(vcur, JSON, false);
          wchars  out = outs;
          assert(out == WCHARS("-2147483648"));
          value vcur1 = parse(out, false);
          assert(vcur == vcur1);
        }
        {
          value   vcur = value::make_currency(12340100L);
          ustring outs = emit(vcur, XJSON, false);
          wchars  out = outs;
          assert(out == WCHARS("1234$01"));
          value vcur1 = parse(out, false);
          assert(vcur == vcur1);
        }
        {
          value arr = value::make_array(3);
          arr.set_element(0, value::make_currency(12340100L));
          arr.set_element(1, value(W("Hello world!")));
          arr.set_element(2, value("hi"));
          ustring outs = emit(arr, XJSON, false);
          wchars  out = outs;
          value   arr1 = parse(out, false);
          assert(arr == arr1);
        }
        {
          value map = value::make_map();
          map.set_prop(value("first"), value::make_currency(12340100L));
          map.set_prop(value("second"), value(W("Hello\"world!")));
          map.set_prop(value("third"), value("hi"));
          map.set_prop(value("fourth"), value(true));
          ustring outs = emit(map, XJSON, false);
          wchars  out = outs;
          value   map1 = parse(out, false);
          assert(map == map1);
          // out.push(0);
          //::MessageBoxW(NULL,buf.head(),L"value emit",MB_OK);
        }
        {
          wchars t = WCHARS("{Cookie:100,Pin:1,Icon:2,Title:\"test\",Content:[{"
            "Text:\"just test\"},{Text:10230},{Process:30}]}");
          value  map = parse(t, false);
          assert(map.is_map());
        }
      }
    } the_test;

#endif

  } // namespace xjson

  ustring value::get(const wchar *defv) const {
    switch (_type) {
    case t_null:
      return WCHARS("null");
    case t_bool:
      return _b() ? WCHARS("true") : WCHARS("false");
    case t_int:
      return ustring::format(W("%d"), _i());
    case t_double: {
      ustring s = ustring::format(W("%f"), _d());
      int     dotpos = s().index_of('.');
      int     n;
      for (n = int(s.length()) - 1; n > dotpos + 1; n--)
        if (s[n] != '0')
          break;
      s.length(n + 1);
      return s;
    }
    case t_string:
      if(_units == UT_URL)
        return ustring::format(W("url(%s)"),ustring(_us()).c_str());
      else if (_units == UT_SELECTOR)
        return ustring::format(W("selector(%s)"), ustring(_us()).c_str());
      else
        return ustring(_us());
    case t_length:
      return length_to_string();
    case t_date:
      return get_date().emit_iso((date_time::type)_units);
    case t_function:
      return this->get_function()->to_string();
    case t_map:
      return xjson::emit(*this, xjson::JSON, false);
    case t_array:
      return xjson::emit(*this, xjson::JSON, true);
    case t_angle:
      return angle_to_string(get<double>(), unit_type_angle(units()));
    case t_duration:
      return duration_to_string(get<double>(), unit_type_duration(units()));
    case t_color:
      return color_to_string(*this);
    case t_enum:
      return enum_to_string(*this);
    default:
      return defv;
    }
  }

  void source_scanner::skip_comment(bool toeol) {
    if (toeol) {
      for (wchar c = get_char(); c; c = get_char()) {
        if (c == '\n')
          break;
        else if (c == wchar(-1)) // end of island
          break;
      }
    }
    else {
      wchar pc = 0;
      while (wchar c = get_char())
        if (pc == '*' && c == '/')
          break;
        else if(c == wchar(-1)) // end of island
          break;
        else
          pc = c;
    }
  }

  source_scanner::token_t source_scanner::get_token() {
    token_t t = _get_token();
    if (token_end_cb)
      token_end_cb(was_push_back() ? -1 : 0);
    return t;
  }

  source_scanner::token_t source_scanner::_get_token() {
    token_value.size(0);

    if (!scan_ws())
      return T_EOF;

    if (token_start_cb)
      token_start_cb(-1);

    switch (wchar c = get_char()) {
      // case '+':
      // case '-':
      //    return scan_number(c);

    case wchar(-1): // end of island
      return T_END_OF_ISLAND;

    case '/':
      if (wchar nc = get_char()) {
        if (nc == '/') {
          skip_comment(true);
          return T_COMMENT;
        }
        if (nc == '*') {
          skip_comment(false);
          return T_COMMENT;
        }
        push_back(nc);
        // else fall through
      }
    case '+':
    case '-':
    case '*':
    case '%':
    case '^':
    case '&':
    case ';':
    case ':':
    case '.':
    case '?':
    case '=':
    case '[':
    case ']':
    case '{':
    case '}':
      return scan_operator(c);
    case '\"':
    case '\'':
      return scan_string(c);
    case '(':
      return T_OPAREN;
    case ')':
      return T_CPAREN;
    case '#':
      return scan_nmtoken(c);
    default:
      if (is_digit(c))
        return scan_number(c);
      if (is_alpha(c) || c == '@' || c == '_' || c == '$')
        return scan_nmtoken(c);
      else
        return token_t(c);
    }
    // return T_END;
  }

  source_scanner::token_t source_scanner::scan_number(wchar c) {
    int n_dots = 0;
    // bool is_currency = false;

    if (c == '0') {

      c = get_char();
      if (c == 'x' || c == 'X') {
        token_value.push(c);
        goto HEX_NUMBER;
      }
    }
    for (; c; c = get_char()) {
      if (is_digit(c))
        token_value.push(c);
      else if (c == '.') {
        if (++n_dots > 1) {
          break;
        }
        token_value.push(c);
      }
      else if (c == 'e' || c == 'E') {
        token_value.push(c);
        c = get_char();
        if (c == '-' || c == '+') {
          token_value.push(c);
          c = get_char();
        }
        for (; c; c = get_char()) {
          if (is_digit(c))
            token_value.push(c);
          else
            break;
        }
        break;
      }
      else
        break;
    }
    if (is_alpha(c)) {
      for (; c; c = get_char())
        if (is_alpha(c))
          token_value.push(c);
        else
          break;
      push_back(c);
      return T_NUMBER_UNIT;
    }
    push_back(c);
    return T_NUMBER;
  HEX_NUMBER:
    // '0x' consumed
    for (; c; c = get_char()) {
      if (is_xdigit(c))
        token_value.push(c);
      else
        break;
    }
    push_back(c);
    return T_NUMBER;
  }

  source_scanner::token_t source_scanner::scan_color(wchar fc) {
    wchar c = fc;
    for (; c && (token_value.length() < 6); c = get_char()) {
      if (is_xdigit(c))
        token_value.push(c);
      else
        break;
    }
    push_back(c);
    return T_COLOR;
  }

  wchar source_scanner::scan_escape() {
    // *pos is '\'
    switch (wchar c = get_char()) {
    case 't':
      return '\t';
    case 'r':
      return '\r';
    case 'n':
      return '\n';
    case 'f':
      return '\f';
    case 'b':
      return '\b';
    case '\\':
      return '\\';
    default:
      return c;
    }
    // return '?'; // to make compiler happy
  }

  source_scanner::token_t source_scanner::scan_string(wchar delimeter) {
    for (wchar c = get_char(); c; c = get_char()) {
      if (c == '\n')
        return T_ERROR; // non closed string
      if (c == delimeter) {
        break;
      }
      else if (c == '\\')
        token_value.push(scan_escape());
      else
        token_value.push(c);
    }
    return T_STRING;
  }

  source_scanner::token_t source_scanner::scan_operator(wchar c) {
    token_value.push(c);
    for (wchar tc = get_char(); tc; tc = get_char())
      switch (tc) {
      case '/':
        tc = tc;
      case '+':
      case '-':
      case '*':
      case '%':
      case '^':
      case '&':
      case ';':
      case ':':
      case '.':
      case '?':
      case '=':
      case '[':
      case ']':
      case '{':
      case '}':
        token_value.push(tc);
        continue;
      default:
        push_back(tc);
        return T_OPERATOR;
      }
    return T_OPERATOR;
  }

  source_scanner::token_t source_scanner::scan_nmtoken(wchar c) {
    token_value.push(c);
    bool hash = c == '#';
    for (c = get_char(); c; c = get_char()) {
      if (is_alnum(c) || c == '_' || c == '$')
        token_value.push(c);
      else if (hash && c == '-')
        token_value.push(c);
      else if (css_like && c == '-')
        token_value.push(c);
      else
        break;
    }
    push_back(c);
    return T_NAME;
  }

}; // namespace tool
