//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| COW ASCII string_t<CT,ACT> class
//|
//|

#ifndef __tl_string_t_h__
#define __tl_string_t_h__

#include "tl_array.h"
#include "tl_basic.h"
#include "tl_config.h"
#include "tl_slice.h"
#include <ctype.h>
#include <string.h>
#include <wchar.h>

namespace tool {

template<typename CT> void to_lower(tslice<CT> t)
{
  const CT *end = t.end();
  for (CT *p = t.start; p < end; ++p)
    *p = to_lower(*p);
}

template<typename CT> void to_upper(tslice<CT> t) {
  const CT *end = t.end();
  for (CT *p = t.start; p < end; ++p)
    *p = to_upper(*p);
}

template<typename CT> void capitalize(tslice<CT> t);


class value;


// this class uses reference counted and copy-on-write semantics to insure
// that it as efficient as possible.

// all indexes are zero-based.  for all functions that accept an index, a
// negative index specifies an index from the right of the string_t<CT,ACT>.  also,
// for all functions that accept a length, a length of -1 specifies the rest
// of the string_t<CT,ACT>.

template <typename CT, typename ACT>
class string_t
{
  friend class value;

public:
  typedef CT char_type;

  string_t();
  string_t(NO_INIT) {}
  string_t(const string_t &s);
  string_t(string_t &&s);
  string_t(const CT *s);
  string_t(const CT *s, size_t count);
  string_t(slice<CT> s);
  string_t(slice<CT> s1, slice<CT> s2);
  string_t(CT c, size_t n = 1);
  string_t(bytes s);

  string_t(const ACT *s) : _data(null_data()) { set(chars_of(s)); }
  string_t(const ACT *s, size_t count) : _data(null_data()) { set(slice<ACT>(s,count)); }
  string_t(slice<ACT> s) : _data(null_data()) { set(s); }
  string_t(const string_t<ACT,CT> &s) : _data(null_data()) { set(s()); }

  void swap(string_t &other) { std::swap(_data, other._data); }

  ~string_t();

  operator const CT *() const;
  operator slice<CT>() const;

  slice<byte> chars_as_bytes() const;
  slice<CT>   chars() const;
  CT*         buffer();

  const CT *begin() const { return head(); }
  const CT *end() const { return head() + length(); }
  const CT *cbegin() const { return head(); }
  const CT *cend() const { return head() + length(); }

  CT&         operator[](index_t index);
  CT          operator[](index_t index) const;

  string_t&   operator=(const string_t<CT, ACT> &s);
  string_t&   operator=(string_t &&s);
  string_t&   operator=(const CT *s);
  string_t&   operator=(slice<CT> s);
  string_t&   operator=(slice<byte> s);

  string_t&   operator=(const string_t<ACT, CT> &s) { set(s()); return *this; }
  string_t&   operator=(CT c);

  string_t operator+(const string_t &s) const;
  string_t operator+(const slice<CT> s) const;
  string_t operator+(CT c) const;

  bool    operator==(const string_t<CT,ACT> &s) const;  bool    operator==(slice<CT> s) const; bool    operator==(const CT *pc) const;
  bool    operator<(const string_t<CT, ACT> &s) const;  bool    operator<(slice<CT> s) const;  bool    operator<(const CT* s) const;
  bool    operator<=(const string_t<CT, ACT> &s) const; bool    operator<=(slice<CT> s) const; bool    operator<=(const CT* s) const;
  bool    operator>(const string_t<CT, ACT> &s) const;  bool    operator>(slice<CT> s) const;  bool    operator>(const CT* s) const;
  bool    operator>=(const string_t<CT, ACT> &s) const; bool    operator>=(slice<CT> s) const; bool    operator>=(const CT* s) const;
  bool    operator!=(const string_t<CT, ACT> &s) const; bool    operator!=(slice<CT> s) const; bool    operator!=(const CT* s) const;

  string_t &append(slice<CT> s);
  string_t &operator+=(const string_t<CT, ACT> &s) { return append(s()); }
  string_t &operator+=(slice<CT> s) { return append(s); }
  string_t &operator+=(CT c) { return append(slice<CT>(c)); }

  explicit operator bool() const { return !is_empty(); }

  size_t    length() const;
  void      length(size_t newlen) { set_length(newlen, true); }
  index_t   size() const { return (index_t)length(); }
  bool      is_empty() const;
  string_t& to_upper() { make_unique(); tool::to_upper(target()); return *this; }
  string_t& to_lower() { make_unique(); tool::to_lower(target()); return *this; }
  string_t& capitalize() { make_unique(); tool::capitalize(target()); return *this; }

  string_t& clear();
  string_t  copy() const;
  string_t& cut(index_t index = 0, index_t len = -1);

  string_t& replace(slice<CT> s, index_t index = 0, index_t len = -1);
  string_t& insert(slice<CT> s, index_t index = 0);
  string_t& insert(CT c, index_t index = 0) { return insert(slice<CT>(c), index); }

  int       replace_all(slice<CT> from, slice<CT> to);
  int       replace_all(CT from, CT to);

  std::basic_string<CT> to_std() const {
    return std::basic_string<CT>(cbegin(), length());
  }

  //
  // pattern slice<CT>:
  //  '*' - any substring
  //  '?' - any one CT
  //  '['CT set']' = any one CT in set
  //    e.g.  [a-z] - all lowercase letters
  //          [a-zA-Z] - all letters
  //          [abd-z] - all lowercase letters except of 'c'
  //          [-a-z] - all lowercase letters and '-'
  // returns:
  //    -1 - no match otherwise start pos of match
  index_t match(const CT *pattern) const;
  bool    like(const CT *pattern) const { return match(pattern) >= 0; }

  //string_t &printf(const CT *fmt, ...);

  static inline string_t format(const CT *fmt, ...) {
    va_list args;
    va_start(args, fmt);
    string_t rv = format_args(fmt, args);
    va_end(args);
    return rv;
  }
  static string_t format_args(const CT *fmt, va_list args);

  // format to roman number
  static string_t roman(uint num, bool ucase);
  // format to abc number
  static string_t alpha(uint num, bool ucase);

  void inherit(const string_t &src) { if (src.length()) *this = src; }
  bool is_defined() const { return _data != null_data(); }
  bool is_undefined() const { return _data == null_data(); }
  bool is_inherit() const { return slice<CT>() == CHARS("inherit"); }

  static string_t val(const string_t &v, const string_t &defval) { return v.is_defined() ? v : defval; }
  static string_t val(const string_t &v, const CT *defval) { return v.is_defined() ? v : string_t(defval); }

  const CT *c_str() const { return head(); }

  uint hash() const;

  tslice<CT> target() { return tslice<CT>(head(), length()); }

  void set(slice<CT> from);
  void set(slice<ACT> from);

  slice<CT> operator()() const { return slice<CT>(head(), length()); }
  slice<CT> operator()(int_ptr s) const {  return slice<CT>(head() + s, length() - s); }
  slice<CT> operator()(int_ptr s, int_ptr e) const {  return slice<CT>(head() + s, min(length(), e) - s); }

  value to_value(uint ut = 0) const;

  array<slice<CT>> tokens(slice<CT> separators/* = " \t\n\v\r\f"*/) const;
  //array<string_t> tokens(CT separator) const;

  struct data {
    data(uint rc = 0) : ref_count(rc), allocated(0), length(0) { chars[0] = '\0'; }
    void            add_ref() { ++ref_count; }
    tslice<CT>      target() { return tslice<CT>(chars, length); }
    slice<CT>       source() { return slice<CT>(chars, length); }
    locked::counter ref_count;
    size_t          allocated;
    size_t          length;
    CT              chars[1];
  };

  const CT *head()  const;

protected:
  string_t(data *dta);
  static data *new_data(size_t length, long ref_count);
  static data *new_extra_data(size_t length, long ref_count);
  bool         set_length(size_t length, bool preserve_content = false);
  void         set_data(data *data);
  void         init(const wchar *us, size_t uslen);
  size_t       buffer_size() const; // sizes of allocated buffer including trailing zero

  // tool::value support
  static void release_data(data *dta, bool wipe = false);
  data *      get_data() const;
  // tool::value support end

  bool make_unique();

  CT *head();

protected:
  data *       _data;
  static data *null_data() {
    static data _null_data(1);
    return &_null_data;
  }
};

// slice<T> that also optionally holds chars buffer
template<typename CT, typename ACT>
struct string_chars_t : public slice<CT> {
  string_t<CT, ACT> buffer;
  string_chars_t(const string_t<CT, ACT>& str) : buffer(str) { this->start = buffer.cbegin(); this->length = buffer.length(); }
  string_chars_t(slice<CT> slc) : buffer(), slice<CT>(slc) {}
  string_chars_t() : buffer(), slice<CT>() {}
  string_chars_t(const string_chars_t<CT, ACT>& strc) : buffer(strc.buffer) {
    if (buffer.is_defined()) {  this->start = buffer.cbegin(); this->length = buffer.length(); }
    else { this->start = strc.start;  this->length = strc.length; }
  }
};

template<typename CT, typename ACT> inline void swap(string_t<CT, ACT> &s1, string_t<CT, ACT> &s2) { s1.swap(s2); }

template<typename CT, typename ACT> inline CT *string_t<CT,ACT>::head() { return _data->chars; }
template<typename CT, typename ACT> inline const CT *string_t<CT, ACT>::head() const { return _data->chars; }

template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t() : _data(null_data()) { }
template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(const string_t &s) : _data(null_data()) { set_data(s._data); }
template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(string_t &&s) : _data(null_data()) { swap(s); }
template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(const CT *s) : _data(null_data()) {
  if (s) {
    size_t len = str_len(s);
    if (set_length(len))
      target().copy(s, len);
  }
}

template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(const CT *s, size_t count) : _data(null_data()) {
  if (set_length(count))
    target().copy(s, count);
}

template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(slice<CT> s) : _data(null_data()) {
  if (set_length(s.size()))
    target().copy(s);
}

template<typename CT, typename ACT> inline string_t<CT, ACT>::string_t(slice<CT> s1, slice<CT> s2) : _data(null_data()) {
  if (set_length(s1.length + s2.length))
    target().copy(s1).copy(s2);
}

/*template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(slice<byte> s) : _data(null_data()) {
  if (set_length(s.length))
    target().copy((const CT *)s.start, s.length);
}*/

template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(CT c, size_t n) : _data(null_data()) {
  if (set_length(n))
    target().set(c);
}

template<typename CT, typename ACT> inline string_t<CT,ACT>::~string_t() { release_data(_data); }

template<typename CT, typename ACT> inline size_t string_t<CT, ACT>::length() const { return _data == null_data() ? 0u : _data->length; }
template<typename CT, typename ACT> inline size_t string_t<CT, ACT>::buffer_size() const { if (_data == null_data()) return 0; return _data->allocated + 1; }

template<typename CT, typename ACT> inline string_t<CT,ACT>::operator const CT *() const { return _data->chars; }

template<typename CT, typename ACT> inline slice<CT> string_t<CT,ACT>::chars() const { return slice<CT>(_data->chars, _data->length); }
template<typename CT, typename ACT> inline string_t<CT,ACT>::operator slice<CT>() const { return this->chars(); }

template<typename CT, typename ACT> inline CT *string_t<CT,ACT>::buffer() { make_unique(); return _data->chars; }

template<typename CT, typename ACT> slice<byte> string_t<CT, ACT>::chars_as_bytes() const { return slice<byte>((const byte *)head(), length() * sizeof(CT)); }


template<typename CT, typename ACT> inline CT &string_t<CT,ACT>::operator[](index_t index) {
  if (index < 0) index += size();
  assert(index >= 0 && index < size());
  make_unique();
  return head()[index];
}

template<typename CT, typename ACT> inline CT string_t<CT,ACT>::operator[](index_t index) const {
  if (index < 0)
    index += size();
  assert(index >= 0 && index < size());
  return head()[index];
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::operator=(const string_t<CT, ACT> &s) {
  if (&s != this)
    set_data(s._data);
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::operator=(string_t<CT, ACT> &&s) {
  if (&s != this) {
    release_data(_data);
    _data = null_data();
    swap(s);
  }
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::operator=(const CT *s) {
  set(chars_of(s));
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::operator=(slice<CT> s) {
  set(s);
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::operator=(CT c) {
  if (set_length(1))
    target().copy(c);
  return *this;
}

template<typename CT, typename ACT> inline bool operator == (slice<CT> s1, const string_t<CT, ACT> &s2) { return s1 == s2(); }
template<typename CT, typename ACT> inline bool operator < (const slice<CT> s1, const string_t<CT, ACT> &s2) { return s1.cmp(s2) < 0; }
template<typename CT, typename ACT> inline bool operator <= (const slice<CT> s1, const string_t<CT, ACT> &s2) { return s1.cmp(s2) <= 0; }
template<typename CT, typename ACT> inline bool operator > (const slice<CT> s1, const string_t<CT, ACT> &s2) { return s1.cmp(s2) > 0; }
template<typename CT, typename ACT> inline bool operator >= (const slice<CT> s1, const string_t<CT, ACT> &s2) { return s1.cmp(s2) >= 0; }
template<typename CT, typename ACT> inline bool operator != (const slice<CT> s1, const string_t<CT, ACT> &s2) { return s1 != s2(); }

template<typename CT, typename ACT> inline bool operator == (const CT* s1, const string_t<CT, ACT> &s2) { return str_cmp(s1,s2.cbegin()) == 0;}
template<typename CT, typename ACT> inline bool operator < (const CT*  s1, const string_t<CT, ACT> &s2) { return str_cmp(s1, s2.cbegin()) < 0; }
template<typename CT, typename ACT> inline bool operator <= (const CT*  s1, const string_t<CT, ACT> &s2) { return str_cmp(s1, s2.cbegin()) <= 0; }
template<typename CT, typename ACT> inline bool operator > (const CT*  s1, const string_t<CT, ACT> &s2) { return str_cmp(s1, s2.cbegin()) > 0; }
template<typename CT, typename ACT> inline bool operator >= (const CT*  s1, const string_t<CT, ACT> &s2) { return str_cmp(s1, s2.cbegin()) >= 0; }
template<typename CT, typename ACT> inline bool operator != (const CT*  s1, const string_t<CT, ACT> &s2) { return str_cmp(s1, s2.cbegin()) != 0; }

template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator==(const string_t &s) const { if (_data == s._data) return true; return this->chars() == s.chars(); }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator==(slice<CT> s) const { return chars() == s; }

template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator < (const string_t &s) const { return chars().cmp(s()) < 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator < (slice<CT> s) const { return chars().cmp(s) < 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator <= (const string_t &s) const { return chars().cmp(s()) <= 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator <= (slice<CT> s) const { return chars().cmp(s) <= 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator > (const string_t &s) const { return chars().cmp(s()) > 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator > (slice<CT> s) const { return chars().cmp(s) > 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator >= (const string_t &s) const { return chars().cmp(s()) >= 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator >= (slice<CT> s) const { return chars().cmp(s) >= 0; }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator != (const string_t &s) const { if (_data == s._data) return false; return this->chars() != s.chars(); }
template<typename CT, typename ACT> inline bool string_t<CT,ACT>::operator!=(slice<CT> s) const { return  chars() != s; }

template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator==(const CT* s) const { return str_cmp(cbegin(), s) == 0; }
template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator < (const CT* s) const { return str_cmp(cbegin(), s) < 0; }
template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator <= (const CT* s) const { return str_cmp(cbegin(), s) <= 0; }
template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator > (const CT* s) const { return str_cmp(cbegin(), s) > 0; }
template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator >= (const CT* s) const { return str_cmp(cbegin(), s) >= 0; }
template<typename CT, typename ACT> inline bool string_t<CT, ACT>::operator!=(const CT* s) const { return  str_cmp(cbegin(), s) != 0; }


template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::append(slice<CT> s) {
  if (s.length) {
    size_t old_length = length();
    if (set_length(old_length + s.length, true))
      target().copy(old_length, s);
  }
  return *this;
}

template<typename CT, typename ACT> inline bool string_t<CT,ACT>::is_empty() const { return _data == null_data(); }

template<typename CT, typename ACT> inline string_t<CT, ACT> string_t<CT,ACT>::copy() const {
  string_t<CT,ACT> newstring(*this);
  return newstring;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::clear() {
  release_data(_data);
  _data = null_data();
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::replace(slice<CT> s, index_t index, index_t len)
{
    // a negative index specifies an index from the right of the string_t<CT,ACT>.
    if (index < 0)
      index += size();

    // a length of -1 specifies the rest of the string_t<CT,ACT>.
    if (len < 0)
      len = size() - index;

    assert(index >= 0 && index < size() && len >= 0 || len < (size() - index));

    if (!make_unique())
      return *this;

    if (len == index_t(s.length))
      target().copy(index, s);
    else {
      index_t prev_len = size();
      if (s.length > size_t(len)) {
        if (set_length(prev_len - len + s.length, true))
          target().move(index + index_t(s.length), index + len, prev_len - len - index);
      }
      else {
        target().move(index + index_t(s.length), index + len, prev_len - len - index);
        set_length(prev_len - len + s.length, true);
      }
      if (s.length > 0)
        target().copy(index, s);
    }
    return *this;
  }

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT,ACT>::insert(slice<CT> s, index_t index)
{
  if (index < 0)
    index += size();

  assert(index >= 0 && index < size());

  if (s.length > 0) {
    make_unique();
    size_t prev_len = length();
    if (set_length(prev_len + s.length, true)) {
      target().move(index + index_t(s.length), index, prev_len - index);
      target().copy(index, s);
    }
  }
  return *this;
}

template<typename CT, typename ACT> inline string_t<CT, ACT>& string_t<CT, ACT>::cut(index_t index, index_t len) {
  if (len == 0)
    return *this;

  // a negative index specifies an index from the right of the string.
  if (index < 0)
    index += size();

  // a length of -1 specifies the rest of the string.
  if (len < 0)
    len = size() - index;

  make_unique();

  assert((index >= 0) && (index < size()) && (len > 0) &&
    (len <= (size() - index)));

  assert(index + _data->length - index - len <= _data->allocated);

  target().move(index, index + len, length() - index - len);

  set_length(_data->length - len);

  return *this;
}

template<typename CT, typename ACT> inline array<slice<CT>> string_t<CT, ACT>::tokens(slice<CT> separators) const {
  array<slice<CT>> list;
  tool::tokens<CT> t(chars(), separators);
  slice<CT> item;
  while (t.next(item))
    list.push(item);
  return list;
}

template<typename CT, typename ACT> inline uint string_t<CT,ACT>::hash() const
{
  unsigned int h  = 0, g;
  const CT * pc = head();
  while (*pc) {
    h = (h << 4) + *pc++;
    if ((g = h & 0xF0000000) != 0)
      h ^= g >> 24;
    h &= ~g;
  }
  return h;
}


template<typename CT, typename ACT> inline typename string_t<CT, ACT>::data * string_t<CT,ACT>::new_data(size_t len, long refcount)
{
  if (len > 0) {
    byte* p = new byte[sizeof(string_t<CT, ACT>::data) + len * sizeof(CT)];
    string_t<CT,ACT>::data *dt = (string_t<CT,ACT>::data *) p;
    if (!dt)
      return null_data();
    dt->ref_count = refcount;
    dt->length = len;
    dt->allocated = len;
    dt->chars[len] = '\0';
    return dt;
  }
  else
    return null_data();
}

template<typename CT, typename ACT> inline typename string_t<CT, ACT>::data * string_t<CT, ACT>::new_extra_data(size_t len, long refcount)
{
  if (len > 0) {
    size_t to_allocate = (len * 3) / 2; //( len + 0x10 ) & ~0xF;
    if (to_allocate < len)
      to_allocate = len;
    assert(to_allocate >= len);
    string_t<CT, ACT>::data *dt = (string_t<CT, ACT>::data *) new byte[sizeof(string_t<CT, ACT>::data) + to_allocate * sizeof(CT)];
    if (!dt)
      return null_data();
    dt->ref_count = refcount;
    dt->length = len;
    dt->allocated = to_allocate;
    dt->chars[len] = '\0';
    return dt;
  }
  else
    return null_data();
}

template<typename CT, typename ACT> inline bool string_t<CT,ACT>::set_length(size_t len, bool preserve_content) {

  if (len == 0) {
    release_data(_data);
    _data = null_data();
    return false;
  }

  if ((len <= _data->allocated) && (_data->ref_count <= 1)) {
    _data->length = len;
    _data->chars[len] = '\0';
    return true;
  }

  data *dt = _data == null_data() ? new_data(len, 1) : new_extra_data(len,1);
  if (dt != null_data()) {

    if (preserve_content)
      dt->target().copy(head(), length());

    release_data(_data);
    _data = dt;
    return true;
  }
  return false;
}

template<typename CT, typename ACT> inline string_t<CT, ACT> operator+(slice<CT> s1, string_t<CT, ACT> s2) {
  if (s1.length == 0)
    return s2;
  else
    return string_t<CT, ACT>(s1,s2());
}

template<typename CT, typename ACT> inline string_t<CT, ACT> string_t<CT, ACT>::operator+(const string_t<CT, ACT> &s) const {
  if (length() == 0)
    return s;
  else if (s.length() == 0)
    return *this;
  else
    return string_t<CT, ACT>(chars(), s());
}

template<typename CT, typename ACT> inline string_t<CT, ACT> string_t<CT, ACT>::operator+(slice<CT> s) const {
  if (length() == 0)
    return string_t<CT, ACT>(s);
  else if (s.length == 0)
    return *this;
  else
    return string_t<CT, ACT>(chars(), s);
}

template<typename CT, typename ACT> inline string_t<CT, ACT> string_t<CT, ACT>::operator+(CT c) const {
  return string_t<CT, ACT>(chars(), slice<CT>(c));
}

template<typename CT, typename ACT> inline void string_t<CT, ACT>::set(slice<CT> from) {
  if (from.start >= head() && from.end() <= end())
  {
    if (_data->ref_count <= 1) {
      target().move(0, index_t(from.start - head()), from.length);
      set_length(from.length);
    }
    else {
      data *dt = new_data(from.length, 1);
      if (dt != null_data()) {
        dt->target().copy(from);
        release_data(_data);
        _data = dt;
      }
    }
  }
  else if (set_length(from.length))
    target().copy(from);
}

template<typename CT, typename ACT> inline int string_t<CT, ACT>::replace_all(CT from, CT to) {
  int cnt = 0;
  if (make_unique())
    for (CT *p = head(); *p; p++)
      if (*p == from) {
        *p = to;
        ++cnt;
      }
  return cnt;
}

template<typename CT, typename ACT> inline int string_t<CT, ACT>::replace_all(slice<CT> from, slice<CT> to) {
  int to_length = to.size();
  int from_length = from.size();

  if (from_length == 0)
    return 0;

  int       count = 0, idx = 0;
  array<CT> out;

  while (true) {
    int next_idx = chars().index_of(from, idx); // + to_length
    if (next_idx < 0) {
      out.push(head() + idx, length() - idx);
      break;
    }
    else
      out.push(head() + idx, next_idx - idx);

    if (to_length)
      out.push(to);
    ++count;
    idx = next_idx + from_length;
  }
  if (count)
    set(out());
  return count;
}

template<typename CT, typename ACT> inline index_t string_t<CT, ACT>::match(const CT *pattern) const {
  return tool::match(chars(), pattern);
}

// https://www.securecoding.cert.org/confluence/display/seccode/MSC06-C.+Be+aware+of+compiler+optimization+when+dealing+with+sensitive+data
#pragma optimize("", off)
template <typename CT>
inline void s_wipe(tslice<CT> s) { memset(s.start, 0, s.length * sizeof(CT)); }
#pragma optimize("", on)

template<typename CT, typename ACT> inline void string_t<CT,ACT>::release_data(typename string_t<CT,ACT>::data *dta, bool wipe) {
  if (dta && (dta != null_data()) && (--dta->ref_count == 0)) {
    if (wipe)
      s_wipe(dta->target());
    delete[](byte *) dta;
  }
}

template<typename CT, typename ACT> inline void string_t<CT,ACT>::set_data(typename string_t<CT, ACT>::data *data) {
  if (_data == data)
    return;
  release_data(_data);
  _data = data;
  ++_data->ref_count;
}

template<typename CT, typename ACT> inline string_t<CT,ACT>::string_t(typename string_t<CT,ACT>::data *dta) {
  _data = dta;
  ++_data->ref_count;
}

template<typename CT, typename ACT> inline typename string_t<CT,ACT>::data *string_t<CT,ACT>::get_data() const {
  ++_data->ref_count;
  return _data;
}

template<typename CT, typename ACT> inline bool string_t<CT,ACT>::make_unique() {
  if (_data->ref_count > 1) {
    data *data = new_data(length(), 1);
    if (!data)
      return false;
    data->target().copy(head(), length());
    --_data->ref_count;
    _data = data;
  }
  return true;
}

/****************************************************************************/

template<> inline slice<byte> string_t<char,wchar>::chars_as_bytes() const {
  return slice<byte>((byte *)_data->chars, _data->length);
}

template<> inline string_t<char, wchar>::string_t(tool::bytes s) : _data(null_data()) {
  if (set_length(s.length))
    target().copy((const char*)s.start,s.length);
}

template<typename CT>
uint levenshtein_distance( slice<CT> s1, slice<CT> s2)
{
  array<uint> col(s2.length + 1), prev_col(s2.length + 1);

  for (uint i = 0; i < prev_col.length(); ++i)
    prev_col[i] = i;
  for (uint i = 0; i < s1.length; ++i) {
    col[0] = i + 1;
    for (uint j = 0; j < s2.length; j++)
      col[j + 1] = std::min({ prev_col[1 + j] + 1, col[j] + 1, prev_col[j] + (s1[i] == s2[j] ? 0 : 1) });
    col.swap(prev_col);
  }
  return prev_col[index_t(s2.length)];
}

  inline array<char> xml_escape_seq(chars src)
  {
    tool::array<char> buf;
    while (src)
    {
      switch (*src)
      {
      case '<': buf.push(CHARS("&lt;")); break;
      case '>': buf.push(CHARS("&gt;")); break;
      case '&': buf.push(CHARS("&amp;")); break;
      case '"': buf.push(CHARS("&quot;")); break;
      case '\'': buf.push(CHARS("&apos;")); break;
      default:
        if (*src < ' ') {
          buf.push(CHARS("&#"));
          itostr<char, int> sn(*src);
          buf.push(sn);
          buf.push(';');
        }
        else
          buf.push(*src);
        break;
      }
      ++src;
    }
    return buf;
  }

  inline array<wchar> xml_escape_seq(slice<wchar> src)
  {
    tool::array<wchar> buf;
    while (src)
    {
      switch (*src)
      {
      case '<': buf.push(WCHARS("&lt;")); break;
      case '>': buf.push(WCHARS("&gt;")); break;
      case '&': buf.push(WCHARS("&amp;")); break;
      case '"': buf.push(WCHARS("&quot;")); break;
      case '\'': buf.push(WCHARS("&apos;")); break;
      default:
        if (*src < ' ') {
          buf.push(WCHARS("&#"));
          itostr<wchar, int> sn(*src);
          buf.push(sn);
          buf.push(';');
        }
        else
          buf.push(*src);
        break;
      }
      ++src;
    }
    return buf;
  }



} // namespace tool

#endif /* string_defined */
