//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| dynamic array
//|
//|

#ifndef __tl_streams_h
#define __tl_streams_h

#include "tl_array.h"
#include "tl_basic.h"
#include "tl_slice.h"
#include <stdio.h>

namespace tool {

// abstract stream
struct stream {
  enum constants {
    EOS     = -1,
    TIMEOUT = -2,
  };
  virtual int  read() { return EOS; }
  virtual bool write(int /*ch*/) { return false; }
  virtual void rewind() {}
  virtual ~stream() {}
};

template <typename T> struct stream_o {
  virtual bool put(const T *buf, size_t buflen) = 0;
  bool         put(T t) { return put(&t, 1); }
  bool         put(slice<T> st) { return put(st.start, st.length); }

  stream_o<T> &operator<<(slice<T> st) {
    put(st);
    return *this;
  }
  stream_o<T> &operator<<(T t) {
    put(t);
    return *this;
  }
};

template <typename T> struct stream_i {
  virtual uint get(T *buf, size_t buflen) = 0;
  bool         get(T &t) { return get(&t, 1) == 1; }
  uint         get(slice<T> st) { return get(st.start, st.length); }
};

template <typename T> struct mem_stream_o : public stream_o<T> {
  array<T> &at;
  mem_stream_o(array<T> &buf) : at(buf) {}
  using stream_o<T>::put;
  virtual bool put(const T *buf, size_t buflen) override {
    at.push(buf, buflen);
    return true;
  }

private:
  mem_stream_o(const mem_stream_o &rs);
  mem_stream_o &operator=(const mem_stream_o &);
};

template <typename T> struct mem_stream_i : public stream_i<T> {
  slice<T> it;
  uint     pos;
  mem_stream_i(slice<T> buf) : it(buf) {}

  using stream_i<T>::get;

  virtual uint get(T *buf, size_t buflen) override {
    uint cnt = min(it.length - pos, buflen);
    if (cnt) {
      copy<T>(buf, buflen);
      pos += cnt;
    }
    return cnt;
  }
};

template <typename T> struct file_stream_o : public stream_o<T> {
  FILE *fh;
#if defined(WINDOWS)
  file_stream_o(const wchar *filename) { fh = wfopen(filename, L"wb"); }
#endif
  file_stream_o(const char *filename) { fh = fopen(filename, "wb"); }

  bool is_open() { return fh != 0; }
  void close() {
    if (fh)
      fclose(fh);
    fh = 0;
  }

  virtual ~file_stream_o() { close(); }
  using stream_o<T>::put;
  virtual bool put(const T *buf, size_t buflen) override {
    if (!fh)
      return false;
    return fwrite(buf, sizeof(T), buflen, fh) == buflen;
  }
};

template <typename T> struct file_stream_i : public stream_i<T> {
  FILE *fh;
#if defined(WINDOWS)
  file_stream_i(const wchar *filename) { fh = wfopen(filename, L"rb"); }
#endif
  file_stream_i(const char *filename) { fh = fopen(filename, "rb"); }
  bool is_open() { return fh != 0; }
  void close() {
    if (fh)
      fclose(fh);
    fh = 0;
  }

  using stream_i<T>::get;
  virtual uint get(T *buf, size_t buflen) override {
    if (feof(fh))
      return 0;
    return fread(buf, sizeof(buf), buflen, fh);
  }
};

template <typename T> struct text_stream_o : public stream_o<T> {
  array<T> buf;
  text_stream_o(uint reserve = 16) {
    if (reserve)
      buf.reserve(reserve);
  }

  using stream_o<T>::put;
  virtual bool put(const T *data, size_t datalen) override {
    // return fwrite(buf,sizeof(T), buflen, fh) == buflen;
    buf.push(data, datalen);
    return true;
  }

  slice<T> text() {
    buf.push(0);
    buf.pop();
    return buf();
  }
};

typedef text_stream_o<wchar> wchar_stream_o;
typedef text_stream_o<char>  char_stream_o;

class utf8_ostream {
  array<byte> buf;

public:
  utf8_ostream() {
    // utf8 byte order mark
    static byte BOM[] = {0xEF, 0xBB, 0xBF};
    buf.push(BOM, sizeof(BOM));
  }

  // intended to handle only ascii-7 strings
  // use this for markup output
  utf8_ostream &operator<<(const char *str) {
    buf.push((const byte *)str, str_len(str));
    return *this;
  }

  utf8_ostream &operator<<(char c) {
    buf.push((byte)c);
    return *this;
  }

  // use UNICODE chars for value output
  utf8_ostream &operator<<(const wchar *wstr) {
    const wchar *pc = wstr;
    for (unsigned int c = *pc; c; c = *(++pc)) {
      switch (c) {
      case '<':
        *this << "&lt;";
        continue;
      case '>':
        *this << "&gt;";
        continue;
      case '&':
        *this << "&amp;";
        continue;
      case '"':
        *this << "&quot;";
        continue;
      case '\'':
        *this << "&apos;";
        continue;
      }
      if (c < (1 << 7)) {
        buf.push(byte(c));
      } else if (c < (1 << 11)) {
        buf.push(byte((c >> 6) | 0xc0));
        buf.push(byte((c & 0x3f) | 0x80));
      } else if (c < (1 << 16)) {
        buf.push(byte((c >> 12) | 0xe0));
        buf.push(byte(((c >> 6) & 0x3f) | 0x80));
        buf.push(byte((c & 0x3f) | 0x80));
      } else if (c < (1 << 21)) {
        buf.push(byte((c >> 18) | 0xf0));
        buf.push(byte(((c >> 12) & 0x3f) | 0x80));
        buf.push(byte(((c >> 6) & 0x3f) | 0x80));
        buf.push(byte((c & 0x3f) | 0x80));
      }
    }
    return *this;
  }

  bytes utf8() {
    buf.push(0);
    buf.pop();
    return buf();
  }
};

} // namespace tool

#endif
