//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| slices, array/string fragments
//|
//|

#ifndef __tl_slice_h__
#define __tl_slice_h__

#include "tl_basic.h"
#include "tl_config.h"
#include <algorithm> // std::reverse
#include <ctype.h>
#include <wctype.h>

#include "assert.h"
#include "ucdata/ucdata_lt.h"

#define BREAK_CHAR wchar('\r')
#define NEWLINE_CHAR wchar('\n')
#define SOFT_HYPHEN wchar(0xAD)
#define NBSP_CHAR wchar(0xA0)

namespace tool {

inline void hash_combine(uint_ptr &seed, uint_ptr v);

template <typename T> struct tslice;

template <typename T> struct slice {
  typedef T element_type;
  const T * start;
  size_t    length;

  slice() : start(0), length(0) {}

  slice(const T *start_, size_t length_) {
    start  = start_;
    length = length_;
  }
  slice(T &single) {
    start  = &single;
    length = 1;
  }

  slice(const slice &src) : start(src.start), length(src.length) {}
  slice(const tslice<T> &src);

  static slice range(const T *start_, const T *end_) {
    return slice(start_, end_ >= start_ ? uint_ptr(end_ - start_) : 0);
  }
  // slice(const T* start_, const T* end_): start(start_), length(
  // end_>=start_?uint_ptr(end_-start_):0) {}

  slice &operator=(const slice &src) {
    start  = src.start;
    length = src.length;
    return *this;
  }
  slice &operator=(const tslice<T> &src);

  slice    end(uint_ptr n) const {
    if (length < n)
      return *this;
    return slice(start + length - n, n);
  }
  int size() const { return static_cast<int>(length); }
  int byte_size() const { return static_cast<int>(length * sizeof(T)); }

  const T *begin() const { return start; }
  const T *end() const { return start + length; }
  const T *cbegin() const { return start; }
  const T *cend() const { return start + length;}

  //bool operator!() const { return length == 0; }
  explicit operator bool() const { return length > 0; }

  template <class Y> bool operator==(const slice<Y> &r) const {
    if (length != r.length)
      return false;
    const T *p1 = end();
    const Y *p2 = r.end();
    while (p1 > start) {
      if (*--p1 != *--p2)
        return false;
    }
    return true;
  }
  bool operator==(const slice &r) const {
    if (length != r.length)
      return false;
    const T *p1 = end();
    const T *p2 = r.end();
    while (p1 > start) {
      if (*--p1 != *--p2)
        return false;
    }
    return true;
  }

  template<typename CMP> 
  bool equal(const slice &r, CMP cmp) const {
    if (length != r.length)
      return false;
    const T *p1 = end();
    const T *p2 = r.end();
    while (p1 > start) {
      if (!cmp(*--p1,*--p2))
        return false;
    }
    return true;
  }


  template <typename F> bool each(F f) {
    for (uint_ptr i = 0; i < length; ++i)
      if (f(start[i]))
        return true;
    return false;
  }
  template <typename F> bool each_backward(F f) {
    for (uint_ptr i = length; i-- > 0;)
      if (f(start[i]))
        return true;
    return false;
  }
  template <typename Y> Y accumulate(Y init) {
    for (uint_ptr i = 0; i < length; ++i)
      init += start[i];
    return init;
  }

  template <typename Tv> slice find(const Tv &v) {
    for (uint_ptr i = 0; i < length; ++i)
      if (start[i] == v)
        return slice(start + i, length - i);
    return slice(end(), 0u);
  }

  template <typename F> slice find_if(F f) {
    for (uint_ptr i = 0; i < length; ++i)
      if (f(start[i]))
        return slice(start + i, length - i);
    return slice(end(), 0u);
  }

  template <class Y> bool operator!=(const slice<Y> &r) const {
    return !operator==(r);
  }

  bool operator!=(const slice &r) const { return !operator==(r); }

  const T &operator[](uint_ptr idx) const {
    if (idx < length)
      return start[idx];
    assert(false);
    return black_hole();
  }
  const T &at(uint_ptr idx) const // NOTE: does not assert if out of range
  {
    if (idx < length)
      return start[idx];
    return black_hole();
  }

  // returns first element or null
  const T operator*() const {
    if (length)
      return *start;
    // assert(false);
    return T();
  }

  const T *operator->() const {
    if (length)
      return start;
    return 0;
  }

  T operator++() {
    if (length) {
      ++start;
      if (--length)
        return *start;
    }
    return T();
  }
  T operator++(int) {
    if (length) {
      --length;
      ++start;
      return *(start - 1);
    }
    return T();
  }

  const T &last() const {
    if (length)
      return start[length - 1];
    assert(false);
    return black_hole();
  }
  const T &last(const T &dv) const {
    if (length)
      return start[length - 1];
    return dv;
  }

  const T &first() const {
    if (length)
      return start[0];
    assert(false);
    return black_hole();
  }

  const T &first(const T &dv) const {
    if (length)
      return start[0];
    return dv;
  }

  static const T &black_hole() {
    static T z = T();
    return z;
  }

  int first_index() const { return 0; }
  int last_index() const { return int(length) - 1; }

  // [idx1..length)
  slice operator()(index_t idx1) const {
    assert(idx1 <= size());
    idx1 = limit(idx1, 0, size());
    return slice(start + idx1, size() - idx1);
  }
  // [idx1..idx2)
  slice operator()(index_t idx1, index_t idx2) const {
    idx1 = limit(idx1, 0, size());
    idx2 = limit(idx2, idx1, size());
    return slice(start + idx1, idx2 - idx1);
  }

  // substr  
  slice sub(index_t index, index_t len = -1) const {
    // a negative index specifies an index from the right of the string.
    if (index < 0)
      index += size();

    // length of -1 specifies the rest of the string.
    if (len < 0)
      len = size() - index;

    if (index < 0 || index >= size() || len < 0 || len > size() - index)
      return slice(); // error in parameters;

    return slice(start + index, len);
  }

  template <typename Tv> int index_of(Tv e, uint_ptr from = 0) const {
    for (uint_ptr i = from; i < length; ++i)
      if (start[i] == T(e))
        return int(i);
    return -1;
  }

  int index_of_one_of(slice set, T &found_element, uint_ptr from = 0) const {
    for (uint_ptr i = from; i < length; ++i) {
      int found = set.index_of(start[i]);
      if (found >= 0) {
        found_element = set[found];
        return int(i);
      }
    }
    return -1;
  }

  template <typename Tv> int last_index_of(const Tv &e) const {
    for (uint_ptr i = uint_ptr(length); i > 0;)
      if (start[--i] == e)
        return int(i);
    return -1;
  }

  int index_of(const slice &s, uint_ptr from = 0) const {
    if (s.length > length)
      return -1;
    if (s.length == 0)
      return -1;
    uint_ptr l = unsigned(length - s.length);
    for (uint_ptr i = from; i <= l; ++i)
      if (start[i] == *s.start) {
        const T *p    = s.start;
        uint_ptr last = i + s.length;
        for (uint_ptr j = i + 1; j < last; ++j)
          if (*(++p) != start[j])
            goto next_i;
        return int(i);
      next_i:
        continue;
      }
    return -1;
  }

  int last_index_of(const slice &s) const {
    if (s.length > length)
      return -1;
    if (s.length == 0)
      return -1;
    const T *ps = s.end() - 1;
    for (uint_ptr i = length; i > 0;)
      if (start[--i] == *ps) {
        const T *p = ps;
        uint_ptr j, first = i - s.length + 1;
        for (j = i; j > first;)
          if (*(--p) != start[--j])
            goto next_i;
        return int(j);
      next_i:
        continue;
      }
    return -1;
  }

  int index_by(const function<bool(const T &proto)> &cmp,
               uint_ptr                              from = 0) const {
    for (uint_ptr i = from; i < length; ++i)
      if (cmp(start[i]))
        return int(i);
    return -1;
  }

  template <class Y>
  bool starts_with(const slice<Y> &s) const {
    if (length < s.length)
      return false;
    slice t(start, s.length);
    return t == s;
  }

  bool starts_with(T c) const {
    if (length < 1)
      return false;
    return start[0] == c;
  }

  bool ends_with(const slice &s) const {
    if (length < s.length)
      return false;
    slice t(start + length - s.length, s.length);
    return t == s;
  }

  bool ends_with(T c) const {
    if (length < 1)
      return false;
    return last() == c;
  }

  bool contains(T v) const { return index_of(v) >= 0; }
  bool contains(slice v) const { return index_of(v) >= 0; }
  bool contains_one_of(slice v) const {
    T dummy;
    return index_of_one_of(v, dummy) >= 0;
  }

  int cmp(slice other) const { // -1 - this smaller , +1 - this larger
    uint n = 0;
    for (;; ++n)
    {
      if (n == length) return n == other.length ? 0 : -1;
      if (n == other.length) return n == length ? 0 : 1;
      if (start[n] != other.start[n]) break;
    }
    return int(start[n]) - int(other.start[n]);
  }

  bool overlaps(slice with) const {
    return max(start, with.start) < min(cend(), with.cend());
  }
  

  void prune(uint_ptr from_start, uint_ptr from_end = 0) {
    uint_ptr s = from_start >= length ? length : from_start;
    uint_ptr e = length - (from_end >= length ? length : from_end);
    start += s;
    if (s < e)
      length = e - s;
    else
      length = 0;
  }

  bool like(const T *pattern) const;

  slice chop(const slice &delimeter, slice &head) const {
    int d = index_of(delimeter);
    if (d < 0) {
      head = *this;
      return slice();
    }
    const T *s = start;
    size_t   l = length;
    head       = slice(s, d);
    return slice(s + d + delimeter.length, l - d - delimeter.length);
  }

  slice chop(const slice &delimeter) {
    int   d = index_of(delimeter);
    slice head;
    if (d < 0) {
      head = *this;
      start += length;
      length = 0;
    } else {
      head = slice(start, d);
      start += d + delimeter.length;
      length -= d + delimeter.length;
    }
    return head;
  }

  slice chop(const slice &delimeter, bool &found) {
    int   d = index_of(delimeter);
    slice head;
    if (d < 0) {
      head = *this;
      start += length;
      length = 0;
      found  = false;
    } else {
      head = slice(start, d);
      start += d + delimeter.length;
      length -= d + delimeter.length;
      found = true;
    }
    return head;
  }

  slice chop(T delimeter) {
    int   d = index_of(delimeter);
    slice head;
    if (d < 0) {
      head = *this;
      start += length;
      length = 0;
    } else {
      head = slice(start, d);
      start += d + 1;
      length -= d + 1;
    }
    return head;
  }

  slice chop_one_of(slice delimeters, T &used_delimeter) {
    int   d = index_of_one_of(delimeters, used_delimeter);
    slice head;
    if (d < 0) {
      head = *this;
      start += length;
      length = 0;
    } else {
      head = slice(start, d);
      start += d + 1;
      length -= d + 1;
    }
    return head;
  }

  bool split(T delimeter, slice &head, slice &tail) const {
    int d = index_of(delimeter);
    if (d < 0)
      return false;
    const T *s = start;
    size_t   l = length;
    head       = slice(s, d);
    tail       = slice(s + d + 1, l - d - 1);
    return true;
  }

  bool split(const slice &delimeter, slice &head, slice &tail) const {
    int d = index_of(delimeter);
    if (d < 0)
      return false;
    const T *s = start;
    size_t   l = length;
    head       = slice(s, d);
    tail       = slice(s + d + delimeter.length, l - d - delimeter.length);
    return true;
  }

  slice head(const slice &s) const {
    int d = index_of(s);
    if (d < 0)
      return *this;
    return slice(start, d);
  }
  slice tail(const slice &s) const {
    int d = index_of(s);
    if (d < 0)
      return slice();
    return slice(start + d + s.length, length - d - s.length);
  }
  slice r_head(const slice &s) const {
    int d = last_index_of(s);
    if (d < 0)
      return *this;
    return slice(start, d);
  }
  slice r_tail(const slice &s) const {
    int d = last_index_of(s);
    if (d < 0)
      return slice();
    return slice(start + d + s.length, length - d - s.length);
  }

  slice head(T c) const {
    int d = index_of(c);
    if (d < 0)
      return *this;
    return slice(start, d);
  }
  slice tail(T c, bool self_if_not_found = false) const {
    int d = index_of(c);
    if (d < 0)
      return self_if_not_found ? *this : slice();
    return slice(start + d + 1, length - d - 1);
  }
  slice r_head(T c) const {
    int d = last_index_of(c);
    if (d < 0)
      return *this;
    return slice(start, d);
  }
  slice r_tail(T c, bool self_if_not_found = false) const {
    int d = last_index_of(c);
    if (d < 0)
      return self_if_not_found ? *this : slice();
    return slice(start + d + 1, length - d - 1);
  }

  slice read(size_t nelements) {
    slice out = *this;
    if (out.length > nelements)
      out.length = nelements;
    prune(out.length);
    return out;
  }

  unsigned int hash() const {
    unsigned int r   = (uint)length;
    const T *    p   = start;
    const T *    end = this->cend();
    while (p < end)
      r = tool::hash(*p++) + 0x9e3779b9 + (r << 6) + (r >> 2);

    return r;
  }

  // copy content to 'to' and prune this one by count of bytes copied
  tslice<T> pull(tslice<T> out);
};

#define MAKE_SLICE(T, D) slice<T>(D, sizeof(D) / sizeof(D[0]))
#define CONST_SLICE(ARR)                                                       \
  slice<std::remove_extent<decltype(ARR)>::type>(ARR,                          \
                                                 sizeof(ARR) / sizeof(ARR[0]))

template <typename T, size_t N>
inline tool::slice<T> source(const T (&dst)[N]) {
  return tool::slice<T>(dst, N);
}

// target slice, writeable thing
template <typename T> struct tslice {
  typedef T element_type;
  T *       start;
  size_t    length;

  tslice() : start(0), length(0) {}

  // template<class C>
  //  tslice(C& cont) : start(c.begin()), length(c.length()) {}

  tslice(T *start_, size_t length_) {
    start  = start_;
    length = start ? length_ : 0;
  }
  tslice(T &single) {
    start  = &single;
    length = 1;
  }

  tslice(const tslice &src) : start(src.start), length(src.length) {}

  static tslice range(T *start_, T *end_) {
    return tslice(start_, end_ >= start_ ? uint_ptr(end_ - start_) : 0);
  }

  tslice &operator=(const tslice &src) {
    start  = src.start;
    length = src.length;
    return *this;
  }

  T *begin() const { return start; }
  T *end() const { return start + length; }

  int size() const { return static_cast<int>(length); }

  //bool operator!() const { return length == 0; }
  explicit operator bool() const { return length > 0; }

  static T &black_hole() {
    static T z = T();
    return z;
  }

  T &operator[](uint_ptr idx) {
    if (idx < length)
      return start[idx];
    assert(false);
    return black_hole();
  }
  
  // [idx1..length)
  tslice operator()(index_t idx1) const {
    assert(idx1 <= size());
    if (idx1 < size())
      return tslice(start + idx1, size() - idx1);
    return tslice(end(), 0u);
  }
  // [idx1..idx2)
  tslice operator()(index_t idx1, index_t idx2) const {
    assert(idx1 <= size());
    assert(idx2 <= size());
    assert(idx1 <= idx2);
    if (idx1 <= size() && idx2 <= size()) {
      if (idx1 < idx2)
        return tslice(start + idx1, idx2 - idx1);
      else
        return tslice(start + idx2, 0u);
    }
    return tslice(end(), 0u);
  }

  tslice copy(slice<T> src) const {
    size_t n;
    if (max(start, src.start) >= min(end(), src.end()))
      n = tool::copy(start, length, src.start, src.length);
    else
      n = tool::move(start, length, src.start - start, 0, src.length);
    assert(n >= src.length);
    return tslice(start + n, length - n);
  }
  tslice copy(index_t at, slice<T> src) { return operator()(at).copy(src); }
  tslice copy(size_t at, slice<T> src) { return operator()(index_t(at)).copy(src); }
  tslice copy(const T *src, size_t srclen) { return copy(slice<T>(src, srclen)); }
  tslice copy(index_t at, const T *src, size_t srclen) { return copy(at, slice<T>(src, srclen)); }
  tslice copy(size_t at, const T *src, size_t srclen) { return copy(at, slice<T>(src, srclen)); }
  tslice copy(const T *src) { return copy(slice<T>(src, length)); }
  tslice copy(const T &src) { return copy(&src, 1); }

  void move(index_t to, index_t from, size_t n) { tool::move(start, length, size_t(to), size_t(from), n); }

  tslice& set(T el) {
    for (size_t i = 0; i < length; ++i)
      start[i] = el;
    return *this;
  }

  tslice xcopy(const void *src, size_t srclen) {
    assert((srclen % sizeof(T)) == 0);
    tslice<byte> t((byte *)start, length * sizeof(T));
    t = t.copy((const byte *)src, srclen);
    return tslice((T *)t.start, t.length / sizeof(T));
  }

  void prune(uint_ptr from_start, uint_ptr from_end = 0) {
    uint_ptr s = from_start >= length ? length : from_start;
    uint_ptr e = length - (from_end >= length ? length : from_end);
    start += s;
    if (s < e)
      length = e - s;
    else
      length = 0;
  }

  template <typename F> void each(F f) {
    uint_ptr l = length;
    for (uint_ptr i = 0; i < l; ++i)
      f(start[i]);
  }
};

template <typename T>
slice<T>::slice(const tslice<T> &src) : start(src.start), length(src.length) {}

template <typename T> slice<T> &slice<T>::operator=(const tslice<T> &src) {
  start  = src.start;
  length = src.length;
  return *this;
}

template <typename T> inline tslice<T> target(T *dst, size_t dstlen) {
  return tslice<T>(dst, dstlen);
}

inline tslice<byte> target(void *dst, size_t dstlen) {
  return tslice<byte>((byte *)dst, dstlen);
}

template <typename T, size_t N> inline tslice<T> target(T (&dst)[N]) {
  return tslice<T>(dst, N);
}

// copy content to 'to' and prune this one by count of bytes copied
template <typename T> inline tslice<T> slice<T>::pull(tslice<T> out) {
  size_t   n = min(length, out.length);
  const T *s = start;
  prune(n);
  return out.copy(s, n);
}

typedef tslice<byte> target_bytes;

template <typename T> class tokens {
  slice<T>   delimeters;
  const T *  p;
  const T *  tail;
  const T *  start;
  const T *  end;
  const bool is_delimeter(T el) {
    //for (const T *t = delimeters; t && *t; ++t)
    //  if (el == *t)
    //    return true;
    //return false;
    return delimeters.contains(el);
  }
  const T *tok() {
    for (; p < tail; ++p)
      if (is_delimeter(*p))
        return p++;
    return p;
  }

public:
  /*tokens(const T *text, uint_ptr text_length, const T *separators)
      : delimeters(separators) {
    start = p = text;
    tail      = p + text_length;
    end       = tok();
  }*/

  tokens(slice<T> s, slice<T> separators) : delimeters(separators) {
    start = p = s.start;
    tail      = p + s.length;
    end       = tok();
  }

  bool next(slice<T> &v) {
    if (start < tail) {
      v.start  = start;
      v.length = uint_ptr(end - start);
      start    = p;
      end      = tok();
      return true;
    }
    return false;
  }
  bool has_more() const { return start < tail; }
};

typedef tokens<char>  atokens;
typedef tokens<wchar> wtokens;

template <typename T> class spans {
  function<bool(T c)>  filter;
  const T *  p;
  const T *  tail;
  const T *  start;
  const T *  end;

  const T *scan_start() {
    for (; p < tail; ++p)
      if (filter(*p))
        return p;
    return tail;
  }

  const T *scan_end() {
    for (; p < tail; ++p)
      if (!filter(*p))
        return p++;
    return p;
  }


public:
    
  spans(slice<T> s, function<bool(T c)>  f) : filter(f) {
    start = p = s.start;
    tail = p + s.length;
    start = scan_start();
    end = scan_end();
  }

  bool next(slice<T> &v) {
    if (start < tail) {
      v.start = start;
      v.length = uint_ptr(end - start);
      start = scan_start();
      end = scan_end();
      return true;
    }
    return false;
  }
  bool has_more() const { return start < tail; }
};

typedef spans<char>  aspans;
typedef spans<wchar> wspans;


typedef slice<char>    chars;
typedef slice<wchar>   wchars;
typedef slice<wchar16> wchars16;
typedef slice<byte>    bytes;
inline slice<byte>     to_bytes(chars cs) {
  return slice<byte>((const byte *)cs.start, cs.length);
}

// template <typename T, typename N>
//  inline slice<T> items_of( typename T (&s)[typename N] ) { return
//  slice<T>(&s[0],N); }

template <typename T, int size> inline slice<T> items_of(const T (&arr)[size]) {
  return slice<T>(&arr[0], size);
}

template <typename T> inline slice<T> items_of(std::initializer_list<T> list) {
  return slice<T>(list.begin(), list.size());
}



  // Note: CS here is a string literal!

#define CHARS(CS) tool::slice<char>(CS, chars_in(CS))
#define WCHARS(CS) tool::slice<wchar>(W(CS), chars_in(WTEXT(CS)))

#define CHAR_BYTES(CS) tool::slice<byte>((const byte *)CS, chars_in(CS))
#define BYTES(BA) tool::slice<byte>(BA, items_in(BA))

template <typename T> inline slice<T> trim_left(slice<T> str) {
  uint_ptr n = str.length;
  for (uint_ptr i = 0; i < n; ++i)
    if (is_space(str[0])) {
      ++str.start;
      --str.length;
    } else
      break;
  return str;
}

template <typename T> inline slice<T> trim_right(slice<T> str) {
  for (int_ptr j = int_ptr(str.length) - 1; j >= 0; --j)
    if (is_space(str[j]))
      --str.length;
    else
      break;
  return str;
}

template <typename T> inline slice<T> trim(slice<T> str) {
  return trim_right(trim_left(str));
}

template <typename T, typename F> inline slice<T> trim_left(slice<T> str, F f) {
  uint_ptr n = str.length;
  for (uint_ptr i = 0; i < n; ++i)
    if (f(str[0])) {
      ++str.start;
      --str.length;
    }
    else
      break;
  return str;
}

template <typename T, typename F> inline slice<T> trim_right(slice<T> str, F f) {
  for (int_ptr j = int_ptr(str.length) - 1; j >= 0; --j)
    if (f(str[j]))
      --str.length;
    else
      break;
  return str;
}

template <typename T, typename F> inline slice<T> trim(slice<T> str, F f) {
  return trim_right(trim_left(str, f), f);
}

inline wchars chars_of(const wchar *t) {
  return t ? wchars(t, (unsigned int)str_len(t)) : wchars();
}
inline chars chars_of(const char *t) {
  return t ? chars(t, (unsigned int)str_len(t)) : chars();
}

template<typename T>
slice<T> chars_of(const std::basic_string<T> &s) { 
  return slice<T>(s.c_str(), s.length()); 
}

//inline chars chars_of(const array<byte>& utf8data) {
//  return chars((const char*)utf8data.cbegin(), utf8data.length());
//}

inline chars chars_of(slice<byte> utf8data) {
  return chars((const char*)utf8data.start, utf8data.length);
}


template <typename C> inline bool icmp(const slice<C> &s1, const slice<C> &s2) {
  if (s1.length != s2.length)
    return false;
  for (uint_ptr i = 0; i < s1.length; ++i)
    if (to_lower(s1[i]) != to_lower(s2[i]))
      return false;
  return true;
}

template <typename C> inline bool icmp(const slice<C> &s1, const C *s2) {
  uint_ptr i = 0;
  for (; i < s1.length; ++i)
    if (to_lower(s1[i]) != to_lower(s2[i]))
      return false;
  return s2[i] == 0;
}

int compare_strings(slice<char> a, slice<char> b, bool case_insensitive,
                    slice<char> lang = slice<char>());
int compare_strings(slice<wchar> a, slice<wchar> b, bool case_insensitive,
                    slice<wchar> lang = slice<wchar>());

namespace lexical {
namespace ci // case insensitive stuff
{
template <typename CT>
inline bool eq(slice<CT> s1, slice<CT> s2) {
  if (s1.length != s2.length)
    return false;
  for (uint i = 0; i < s1.length; ++i)
    if (to_lower(s1[i]) != to_lower(s2[i]))
      return false;
  return true;
}

template <typename CT> inline bool eq(slice<CT> s1, const CT *s2) {
  uint i = 0;
  for (; i < s1.length; ++i) {
    if (s2[i] == 0)
      return false;
    if (to_lower(s1[i]) != to_lower(s2[i]))
      return false;
  }
  return s2[i] == 0;
}

template <typename CT> inline bool eq(const CT *s1, slice<CT> s2) {
  return eq(s2, s1);
}

template <typename CT> inline int cmp(slice<CT> s1, slice<CT> s2) {
  return compare_strings(s1, s2, true);
}
template <typename CT>
inline int cmp(slice<CT> s1, slice<CT> s2, slice<CT> lang) {
  return compare_strings(s1, s2, true, lang);
}

template <typename CT> inline int cmp(const slice<CT> &s1, const CT *s2) {
  return cmp(s1, chars_of(s2));
}
template <typename CT> inline int cmp(const CT *s1, const slice<CT> &s2) {
  return cmp(chars_of(s1), s2);
}
template <typename CT> inline int cmp(const CT *s1, const CT *s2) {
  return cmp(chars_of(s1), chars_of(s2));
}
} // namespace ci

namespace cs // case sensitive stuff
{
template <typename CT>
inline int cmp(const slice<CT> &s1, const slice<CT> &s2) {
  return compare_strings(s1, s2, false);
}
template <typename CT>
inline int cmp(const slice<CT> &s1, const slice<CT> &s2, slice<CT> lang) {
  return compare_strings(s1, s2, false, lang);
}
template <typename CT> inline int cmp(const slice<CT> &s1, const CT *s2) {
  return cmp(s1, chars_of(s2));
}
template <typename CT> inline int cmp(const CT *s1, const slice<CT> &s2) {
  return cmp(chars_of(s1), s2);
}
template <typename CT> inline int cmp(const CT *s1, const CT *s2) {
  return cmp(chars_of(s1), chars_of(s2));
}
} // namespace cs
} // namespace lexical

// lst="screen,desktop" val="screen"

template <typename T>
bool list_contains(slice<T> lst, const T *delim, slice<T> val) {
  if (!val.length || !lst.length)
    return false;
  tokens<T> z(lst, delim);
  slice<T>  t, v = val;
  while (z.next(t)) {
    if (icmp(t, v))
      return true;
  }
  return false;
}

// lst="screen,desktop" val_lst="screen,print"

template <typename T>
bool list_contains_one_of(slice<T> lst, const T *delim, slice<T> val_lst) {
  tokens<T> z(val_lst, delim);
  slice<T>  t;
  while (z.next(t)) {
    if (list_contains<T>(lst, delim, t))
      return true;
  }
  return false;
}

template <typename T> inline bool only_spaces(slice<T> s) {
  const T *p   = s.start;
  const T *end = s.end();
  while (p < end)
    if (*p == NBSP_CHAR || !is_space(*p++))
      return false;
  return true;
}

// int match ( chars cr, const char *pattern );
// int match ( wchars cr, const wchar *pattern );

/****************************************************************************/
//
// idea was taken from Konstantin Knizhnik's FastDB
// see http://www.garret.ru/
// extended by [] operations
//

template <typename CT, CT sep = '-', CT end = ']'> struct charset {
  enum { SET_SIZE = (1 << ((sizeof(CT) > 2 ? 2 : sizeof(CT)) * 8)) };
  unsigned char codes[SET_SIZE >> 3];

  unsigned charcode(CT c) { return (SET_SIZE - 1) & unsigned(c); }

private:
  void set(CT from, CT to, bool v) {
    for (unsigned i = charcode(from); i <= charcode(to); ++i) {
      unsigned int bit   = i & 7;
      unsigned int octet = i >> 3;
      if (v)
        codes[octet] |= 1 << bit;
      else
        codes[octet] &= ~(1 << bit);
    }
  }
  void init(unsigned char v) { memset(codes, v, (SET_SIZE >> 3)); }

public:
  void parse(const CT *&pp) {
    // assert( sizeof(codes) == sizeof(CT) * sizeof(bool));
    const CT *    p   = (const CT *)pp;
    unsigned char inv = *p == '^' ? 0xff : 0;
    if (inv) {
      ++p;
    }
    init(inv);
    if (*p == sep)
      set(sep, sep, inv == 0);
    while (*p) {
      if (p[0] == end) {
        p++;
        break;
      }
      if (p[1] == sep && p[2] != 0) {
        set(p[0], p[2], inv == 0);
        p += 3;
      } else if (p[0] == '\\') {
        CT t = *++p;
        set(t, t, inv == 0);
        ++p;
      } else {
        CT t = *p++;
        set(t, t, inv == 0);
      }
    }
    pp = (const CT *)p;
  }

  bool valid(CT c) {
    unsigned int bit   = charcode(c) & 7;
    unsigned int octet = charcode(c) >> 3;
    return (codes[octet] & (1 << bit)) != 0;
  }
};

template <typename CT> inline int match(slice<CT> cr, const CT *pattern) {
  // if( !cr.length || !pattern )
  //  return -1;
  // static CT zero = 0;
  // if(cr.length == 0 && !cr.start ) cr.start = &zero;

  const CT AnySubstring = '*';
  const CT AnyOneChar   = '?';
  const CT AnyOneDigit  = '#';

  const CT *  str      = cr.start;
  const CT *  wildcard = 0;
  const CT *  strpos   = 0;
  const CT *  matchpos = 0;
  charset<CT> cset;

  for (;;) {
    if (*pattern == '\\')
      ++pattern;
    else if (*pattern == AnySubstring) {
      wildcard = ++pattern;
      strpos   = str;
      if (!matchpos)
        matchpos = str;
    } else if (str >= cr.end() || *str == '\0') {
      return (*pattern == '\0') ? int(matchpos - cr.start) : -1;
    } else if (*pattern == '[' && *(pattern + 1) != ']') {
      pattern++;
      cset.parse(pattern);
      if (!cset.valid(*str))
        return -1;
      if (!matchpos)
        matchpos = str;
      str += 1;
    } else if (*str == *pattern || *pattern == AnyOneChar) {
      if (!matchpos)
        matchpos = str;
      str += 1;
      pattern += 1;
    } else if (*pattern == AnyOneDigit) {
      if (isdigit(*str)) {
        if (!matchpos)
          matchpos = str;
        str++;
        pattern++;
      } else if (wildcard) {
        str     = ++strpos;
        pattern = wildcard;
      } else
        return -1;
    } else if (wildcard) {
      str     = ++strpos;
      pattern = wildcard;
    } else
      break;
  }
  return -1;
}

inline bool is_like(chars cr, const char *pattern) {
  return match<char>(cr, pattern) >= 0;
}
inline bool is_like(wchars cr, const wchar *pattern) {
  return match<wchar>(cr, pattern) >= 0;
}

template <typename T> bool slice<T>::like(const T *pattern) const {
  return is_like(*this, pattern);
}

template <typename CT>
inline bool match_list(slice<CT> val, slice<CT> str, slice<CT> sep) {
  if (!str.length)
    return false;
  tokens<CT> z(str, sep);
  slice<CT>  t, v = val;
  while (z.next(t)) {
    if (t == v)
      return true;
  }
  return false;
}

template <typename CT>
inline bool match_list_ci(slice<CT> val, slice<CT> str, slice<CT> sep) {
  if (!str.length)
    return false;
  tokens<CT> z(str, sep);
  slice<CT>  t, v = val;
  while (z.next(t)) {
    if (icmp(t, v))
      return true;
  }
  return false;
}

template <typename CT>
inline bool match_lists(slice<CT> val, slice<CT> str, slice<CT> sep) {
  if (!str.length)
    return false;

  tokens<CT> z(val, sep);
  slice<CT>  vali;
  while (z.next(vali)) {
    if (!vali)
      continue;
    if (!match_list(vali, str, sep))
      return false;
  }
  return true;
}

// chars to uint_ptr
// chars to int

template <typename T, typename V>
inline bool parse_uint(slice<T> &span, V &rv, unsigned int base = 10) {
  V        result = 0, value;
  const T *cp     = span.start;
  const T *pend   = span.end();

  while (cp < pend && is_space(*cp))
    ++cp;

  int ndigits = 0;

  if (!base) {
    base = 10;
    if (*cp == '0') {
      base = 8;
      cp++;
      ++ndigits;
      if ((to_upper(cp[0]) == 'X') && is_xdigit(cp[1])) {
        cp++;
        base = 16;
      }
    }
  } else if (base == 16) {
    if (cp[0] == '0' && to_upper(cp[1]) == 'X')
      cp += 2;
  }

  if (base == 16) {
    while (cp < pend && is_xdigit(*cp) &&
           ((value = is_digit(*cp) ? *cp - '0' : to_upper(*cp) - 'A' + 10) <
            static_cast<V>(base))) {
      result = result * base + value;
      ++ndigits;
      cp++;
    }
  } else {
    while (cp < pend && is_digit(*cp) &&
           ((value = (*cp - '0')) < static_cast<V>(base))) {
      result = result * base + value;
      ++ndigits;
      cp++;
    }
  }

  span.prune(cp - span.start);
  // span.length = cp - span.start;
  if (ndigits) {
    rv = result;
    return true;
  }
  return false;
}

template <typename T>
inline unsigned int to_uint(slice<T> &span, unsigned int base = 10) {
  uint r = 0;
  parse_uint(span, r, base);
  return r;
}

template <typename T>
inline uint64 to_uint64(slice<T> &span, unsigned int base = 10) {
  uint64 r = 0;
  parse_uint(span, r, base);
  return r;
}

template <typename T, typename V>
inline uint parse_uint(slice<T> span, unsigned int base = 10) {
  return to_uint(span, base);
}

template <typename T, typename V>
inline uint parse_uint64(slice<T> span, unsigned int base = 10) {
  return to_uint64(span, base);
}

template <typename T, typename V> // lefts in span non-parsed reminder
inline bool parse_int(slice<T>& span, V &rv, unsigned int base = 10) {
  /*while (span.length > 0 && is_space(span[0])) {
    ++span.start;
    --span.length;
  }*/
  if (span.length == 0)
    return false;
  typename unsigned_t<V>::type uv = 0;
  if (span[0] == '-') {
    ++span.start;
    --span.length;
    if (!parse_uint(span, uv, base))
      return false;
    rv = -static_cast<V>(uv);
    return true;
  }
  if (span[0] == '+') {
    ++span.start;
    --span.length;
  }
  if (!parse_uint(span, uv, base))
    return false;
  rv = static_cast<V>(uv);
  return true;
}

template <typename T, typename V> // true if the span is parsed in full
inline bool try_parse_int(slice<T> span, V &rv, unsigned int base = 10) {
  slice<T> t = trim(span);
  return parse_int(t,rv,base) && t.length == 0;
}

template <typename T>
inline int to_int(slice<T> &span, unsigned int base = 10) {
  int rv = 0;
  parse_int(span, rv, base);
  return rv;
}

template <typename T>
inline int64 to_int64(slice<T> &span, unsigned int base = 10) {
  int64 rv = 0;
  parse_int(span, rv, base);
  return rv;
}

template <typename T>
inline int parse_int(slice<T> span, unsigned int base = 10) {
  return to_int(span, base);
}

template <typename T>
inline int parse_uint(slice<T> span, unsigned int base = 10) {
  return to_uint(span, base);
}

template <typename T>
inline int64 parse_int64(slice<T> span, unsigned int base = 10) {
  return to_int64(span, base);
}

template <typename TC, typename TV> class itostr : public slice<TC> {
  TC buffer[86];

public:
  itostr(TV n, uint radix = 10, uint_ptr width = 0, TC padding_char = '0') {
    buffer[0] = 0;
    if (radix < 2 || radix > 36)
      return;

    static char digits[] = "0123456789abcdefghijklmnopqrstuvwxyz";
    uint_ptr    i        = 0;
    TV          sign     = n;

    if (sign < 0)
      n = -n;

    do
      buffer[i++] = TC(digits[n % radix]);
    while ((n /= radix) > 0);

    if (width && i < width) {
      while (i < width)
        buffer[i++] = padding_char;
    }
    if (sign < 0)
      buffer[i++] = TC('-');
    buffer[i] = TC('\0');

    TC *p1 = &buffer[0];
    TC *p2 = &buffer[i - 1];
    while (p1 < p2) {
      swap(*p1, *p2);
      ++p1;
      --p2;
    }
    slice<TC>::start  = buffer;
    slice<TC>::length = i;
  }

  operator const TC *() const { return buffer; }
};

typedef itostr<char, int>    itoa;
typedef itostr<wchar, int>   itow;
typedef itostr<char, int64>  i64toa;
typedef itostr<wchar, int64> i64tow;

// fixed number to string:
template <typename TC, typename TV> class fixedtostr : public slice<TC> {
  TC buffer[256];

public:
  operator const TC *() const { return buffer; }
  // const TC* elements() const { return buffer; }
  // uint_ptr      length() const { return buffer_length; }

  // fixedtostr(TV i, int fd = 3, const TC* pu = 0) { format(i,fd,0,pu); }
  fixedtostr(TV i, int fd = 3, const TC *preffix = 0, const TC *suffix = 0) {
    wchar *p = buffer;
    if (preffix)
      while (*preffix)
        *p++ = *preffix++;
    wchar *pnum_start = p;
    bool   gotnz      = false;
    bool   neg        = false;
    if (i < 0) {
      neg = true;
      i   = -i;
    }
    static const char *digits = "0123456789";
    for (int k = 0; k < fd; ++k) {
      int r = i % 10;
      if (gotnz)
        *p++ = digits[r];
      else if (r) {
        *p++  = digits[r];
        gotnz = true;
      }
      i /= 10;
    }
    if (gotnz)
      *p++ = '.';
    do {
      int r = i % 10;
      *p++  = digits[r];
      i /= 10;
    } while (i > 0);
    if (neg)
      *p++ = '-';
    *p = 0;
    std::reverse(pnum_start, p);
    if (suffix)
      while (*suffix)
        *p++ = *suffix++;
    *p                = 0;
    slice<TC>::start  = buffer;
    slice<TC>::length = uint_ptr(p - buffer);
  }
};

typedef fixedtostr<char, int>    fixedtoa;
typedef fixedtostr<wchar, int>   fixedtow;
typedef fixedtostr<char, int64>  fixed64toa;
typedef fixedtostr<wchar, int64> fixed64tow;

/** Float to string converter.
    Use it as ostream << ftoa(234.1); or
    Use it as ostream << ftoa(234.1,"pt"); or
**/
class ftoa : public chars {
  char buffer[64];

public:
  ftoa(double d, const char *units = "", int fractional_digits = 1) {
    //_snprintf(buffer, 64, "%.*f%s", fractional_digits, d, units );
    do_snprintf(buffer, 64, "%.*f%s", fractional_digits, d, units);
    buffer[63] = 0;
    start      = buffer;
    length     = (uint_ptr)strlen(buffer);
  }
  operator const char *() { return buffer; }
};

/** Float to wstring converter.
    Use it as wostream << ftow(234.1); or
    Use it as wostream << ftow(234.1,"pt"); or
**/
class ftow : public wchars {
  wchar buffer[64];

public:
  ftow(double d, const wchar *units = W(""), int fractional_digits = 1) {
    do_w_snprintf(buffer, 64, W("%.*f%s"), fractional_digits, d, units);
    //_snwprintf(buffer, 64, L"%.*f%s", fractional_digits, d, units );
    buffer[63] = 0;
    start      = buffer;
    length     = (uint_ptr)str_len(buffer);
  }
  operator const wchar *() { return buffer; }
};

template <typename T, typename TI> // lefts in span non-parsed reminder
inline int str_to_i(tool::slice<T> &span, TI dv) {
  int n = 0;
  return parse_int(span, n, 10) ? n : dv;
}

template <typename T, typename TI> // lefts in span non-parsed reminder
inline int str2i(tool::slice<T> span, TI dv) {
  return str_to_i(span, dv);
}

template <typename TC, typename TF> // lefts non-parsed reminder in span
inline TF str_to_f(slice<TC> &span, TF dv) {
  TF  number;
  int exponent;
  int negative;
  TF  p10;
  int n;
  int num_digits;
  int num_decimals;
  
  //// Skip leading whitespace
  //while (is_space(*span))
  //  ++span;

  // Handle optional sign
  negative = 0;
  switch (*span) {
  case '-':
    negative = 1; // Fall through to increment position
  case '+':
    ++span;
  }

  number       = 0.f;
  exponent     = 0;
  num_digits   = 0;
  num_decimals = 0;

  // Process string of digits
  while (is_digit(*span)) {
    number = number * 10.f + (*span - '0');
    ++span;
    ++num_digits;
  }

  // Process decimal part
  if (*span == '.' && span.length == 1) 
    ++span;
  else if (*span == '.' && span.length > 1 && span[1] != '.') {
    ++span;
    while (is_digit(*span)) {
      number = number * 10.f + (*span - '0');
      ++span;
      num_digits++;
      num_decimals++;
    }
    exponent -= num_decimals;
  }

  if (num_digits == 0) {
    return dv;
  }

  // Correct for sign
  if (negative)
    number = -number;

  // Process an exponent string
  if (*span == 'e' || *span == 'E') {
    // Handle optional sign
    negative = 0;
    switch (++span) {
    case '-':
      negative = 1; // Fall through to increment pos
    case '+':
      ++span;
    default:
      if (!is_digit(*span)) {
        --span.start;
        ++span.length;
        goto NO_EXPONENT;
      }
    }

    // Process string of digits
    n = 0;
    while (is_digit(*span)) {
      n = n * 10 + (*span - '0');
      ++span;
    }

    if (negative)
      exponent -= n;
    else
      exponent += n;
  }

  if (exponent < DBL_MIN_EXP || exponent > DBL_MAX_EXP) {
    return TF(HUGE_VAL);
  }
NO_EXPONENT:
  // Scale the result
  p10 = 10;
  n   = exponent;
  if (n < 0)
    n = -n;
  while (n) {
    if (n & 1) {
      if (exponent < 0)
        number /= p10;
      else
        number *= p10;
    }
    n >>= 1;
    p10 *= p10;
  }
  return number;
}

template <typename TC, typename TF> // lefts non-parsed reminder in span
inline bool parse_real(slice<TC>& span, TF &r) {
  slice<TC> t = span;
  r = str_to_f(span, TF(0));
  return span != t;
}

template <typename TC, typename TF> // lefts non-parsed reminder in span
inline bool try_parse_real(slice<TC> span, TF &r) {
  slice<TC> t = trim(span);
  return parse_real(t, r) && t.length == 0;
}

template <typename TC, typename TF> // lefts non-parsed reminder in span
inline TF str2f(slice<TC> span, TF dv) {
  TF r = str_to_f(span, dv);
  assert(span.length == 0);
  return span.length ? dv : r;
}

inline int wtoi(const wchar *strz, uint base = 0) {
  //   wchar_t *endptr;
  //   return (int)wcstol(strz, &endptr, base);
  wchars str = chars_of(strz);
  int    r   = 0;
  parse_int(str, r, base);
  return r;
}

inline double wtof(const wchar *strz) {
  wchars str = chars_of(strz);
  double r   = 0;
  parse_real(str, r);
  return r;
}

inline double atof(const char *strz) {
  chars  str = chars_of(strz);
  double r   = 0;
  parse_real(str, r);
  return r;
}

bool chopline(wchars &text,wchars& line);
bool chopline(wchars &text, wchars& line, bool& crlf_seen);

/*  template<typename T, typename V>
    STR join(const V& seq, STR::char_type s = ' ')
    {
      array<STR::char_type> out;
      for (int n = 0; n < seq.size(); ++n) {
        if (n)
          out.push(s);
        out.push(seq[n]);
      }
      return out();
    }*/

#ifdef _DEBUG

inline void slice_unittest() {
  int v1[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9};
  int v2[] = {3, 4, 5};
  int v3[] = {0, 1, 2};
  int v4[] = {0, 1, 2, 4};
  int v5[] = {1, 1, 2, 3};

  slice<int> s1 = MAKE_SLICE(int, v1);
  slice<int> s2 = MAKE_SLICE(int, v2);
  slice<int> s3 = MAKE_SLICE(int, v3);
  slice<int> s4 = MAKE_SLICE(int, v4);
  slice<int> s5 = MAKE_SLICE(int, v5);

  assert(s1 != s2);
  assert(s1(3, 6) == s2);
  assert(s1.index_of(3) == 3);
  assert(s1.index_of(s2) == 3);
  assert(s1.last_index_of(3) == 3);
  assert(s1.last_index_of(s2) == 3);

  assert(s1.index_of(s3) == 0);
  assert(s1.last_index_of(s3) == 0);

  assert(s1.index_of(s4) == -1);
  assert(s1.last_index_of(s4) == -1);

  assert(s1.index_of(s5) == -1);
  assert(s1.last_index_of(s5) == -1);
}

#endif

} // namespace tool

#endif
