#ifndef __tl_shared_memory_h__
#define __tl_shared_memory_h__

#include "config.h"
#include "shmem/shmem.h"
#include "tl_array.h"
#include "tl_config.h"
#include "tl_slice.h"

namespace tool {

//|
//| pipe for interpocess communications on the same machine
//| messages fifo
//|

class fifo {
protected:
  class item {
  public:
#ifdef USE_BASED_POINTERS
    REF(item) next;
#else
    item *next;
#endif
    uint length;
    byte buffer[1];

    void *operator new(size_t fixed_size, shared_memory &shmem,
                       size_t varying_size) {
      return shmem.allocate(fixed_size + varying_size - 4);
    }
    void operator delete(void *p) { shared_memory::deallocate(p); }

    void store(bytes data) {
      length = uint(data.length);
      next   = nullptr;
      target(buffer, length).copy(data);
    }
  };

  class header {
  private:
#ifdef USE_BASED_POINTERS
    REF(item) tail;
    REF(item) head;
#else
    item *tail;
    item *head;
#endif

  public:
    void *operator new(size_t size, shared_memory &shmem) {
      return shmem.allocate(size);
    }
    void operator delete(void *p) { shared_memory::deallocate(p); }

    header() {
      tail = nullptr;
      head = nullptr;
    }

    bool enqueue(shared_memory &shmem, event &not_full, bytes message) {
      exclusive_lock xlock(shmem);
      // create object of varying size
      item *ip = new (shmem, message.length) item;
      if (ip == nullptr) {
        not_full.reset();
        return false; // not enough space in storage
      }
      ip->store(message);
      if (head == nullptr) {
        head = tail = (REF(item))ip;
      } else {
        tail = tail->next = (REF(item))ip;
      }
      return true;
    }
    bool dequeue(shared_memory &shmem, array<byte> &data) {
      exclusive_lock xlock(shmem);
      item *         ip = head;
      if (ip == nullptr) { // queue is empty
        return false;
      }
      head = head->next;
      data.push(ip->buffer, ip->length);
      delete ip;
      return true;
    }
  };

  header *      root;
  semaphore     not_empty;
  event         not_full;
  shared_memory shmem;
  char *        name;
  size_t        max_size;

public:
  bool open(const char *name, size_t max_size) {
    shared_memory::status rc = shmem.open(nullptr, name, max_size);
    if (rc != shared_memory::ok) {
      return false;
    }
    root = (header *)shmem.get_root_object();
    if (root == nullptr) {
      exclusive_lock xlock(shmem);
      root = new (shmem) header;
      shmem.set_root_object(root);
    }
    size_t len         = strlen(name);
    char * global_name = new char[len + 5];
    strcpy(global_name, name);
    strcpy(global_name + len, ".put");
    if (!not_full.open(global_name)) {
      delete[] global_name;
      return false;
    }
    strcpy(global_name + len, ".get");
    if (!not_empty.open(global_name)) {
      delete[] global_name;
      return false;
    }
    delete[] global_name;
    return true;
  }
  void close() {
    shmem.close();
    not_empty.close();
    not_full.close();
  }

  void put(bytes message) {
    while (!root->enqueue(shmem, not_full, message)) {
      not_full.wait();
    }
    not_empty.signal();
  }
  bool get(array<byte> &buf, uint msec) {
    if (not_empty.wait(msec)) {
      int len = root->dequeue(shmem, buf);
      assert(len >= 0);
      not_full.signal();
      return true;
    }
    return false;
  }
};

} // namespace tool

#endif //__tl_shared_memory_h__
