#include "config.h"
#include "tool.h"
#include "tl_lzf.h"

#ifndef WINDOWS
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#endif

// Sciter ARchive support

namespace tool {

int each_file(wchars dirpath, function<void(wchars)> cb);

struct item_def : resource {
  ustring     path;
  array<byte> data;    // can be compressed
  uint        ulength; // original (non-compressed) length
};

void sar::pack(wchars path, wchars inclusions, wchars exclusions, function<uint(bytes)> out) {
  array<handle<item_def>> items;

  array<ustring> i;
  array<ustring> x;

  while (!!inclusions) {
    wchars filter = inclusions.chop(';');
    i.push(filter);
  }

  while (!!exclusions) {
    wchars filter = exclusions.chop(';');
    x.push(filter);
  }


  each_file(path, [&](wchars abspath) {
    ustring relpath = abspath(path.size() + 1);
    relpath.replace_all('\\', '/');

    if (i.length()) {
      bool found = false;
      for (int n = 0; n < i.size(); ++n)
        if (relpath.like(i[n])) { found = true; break; }
      if (!found)
        return;
    }

    for (int n = 0; n < x.size(); ++n)
      if (relpath.like(x[n]))
        return;

    mm_file f;
    if (!f.open(abspath.start))
      return;

    wchars ext = relpath().r_tail('.');

    handle<item_def> id = new item_def();
    items.push(id);
    id->data = bytes((const byte *)f.data(), f.size());
    id->path = relpath;
    // id->path.replace('\\','/');
    id->ulength = 0;

    if (ext == WCHARS("jpeg") || ext == WCHARS("jpg") || ext == WCHARS("png") ||
        ext == WCHARS("gif")) {
#if defined(WINDOWS)
      fwprintf(stdout, W("%s %d\n"), id->path.c_str(), id->data.size());
#else
        fprintf(stdout,"%s %d\n", u8::cvt(id->path).c_str(),id->data.size());
#endif
      return; // compressed already
    }

    array<byte> outdata(id->data.size() + id->data.size() / 2);

    size_t compressed_len =
        lzf_compress(id->data.cbegin(), uint(id->data.length()),
                     outdata.begin(), uint(outdata.length()));
    if (compressed_len >= (id->data.length() * 8 / 10)) {
#if defined(WINDOWS)
      fwprintf(stdout, WTEXT("%s %d\n"), id->path.c_str(), id->data.size());
#else
        fprintf(stdout,"%s %d\n", u8::cvt(id->path).c_str(),id->data.size());
#endif
      return; // compressed enough
    }

    outdata.length(compressed_len);

    id->ulength = uint(id->data.length());
    id->data.swap(outdata);
#if defined(WINDOWS)
    fwprintf(stdout, WTEXT("%s %d compressed to %d (%d%%)\n"), id->path.c_str(),
             id->ulength, id->data.size(),
             (id->data.size() * 100) / id->ulength);
#else
      fprintf(stdout,"%s %d compressed to %d (%d%%)\n", u8::cvt(id->path).c_str(),id->ulength, id->data.size(), (id->data.size() * 100) / id->ulength);
#endif

  });

  item_loc cloc = {0};
  for (int n = 0; n < items.size(); ++n) {
    auto it                                  = items[n];
    cloc.length                              = uint(it->data.length());
    cloc.ulength                             = it->ulength;
    ltbl.data(ltbl.insert(it->path.c_str())) = cloc;
    cloc.offset += cloc.length;
  }

  byte marker[] = {'S', 'A', 'r', 0};

  auto out32 = [&](uint32 n) -> uint {
    bytes pb((byte *)&n, sizeof(n));
    return out(pb);
  };
  auto out16 = [&](uint16 n) -> uint {
    bytes pb((byte *)&n, sizeof(n));
    return out(pb);
  };

  uint offset = 0;

  offset += out(items_of(marker));
  offset += out32(uint(ltbl.nodes.length()));

  for (int n = 0; n < ltbl.nodes.size(); ++n) {
    auto nd = ltbl.nodes[n];
    offset += out16(nd.splitchar);
    offset += out16(nd.lokid);
    offset += out16(nd.eqkid);
    offset += out16(nd.hikid);
  }

  offset += out32(uint(ltbl.items.length()));
  uint off = offset + ltbl.items.size() * sizeof(uint) * 3;
  for (int n = 0; n < ltbl.items.size(); ++n) {
    auto it = ltbl.items[n];
    offset += out32(it.data.offset + off);
    offset += out32(it.data.length);
    offset += out32(it.data.ulength);
  }

  // pack the data
  for (int n = 0; n < items.size(); ++n)
    offset += out(items[n]->data());

  printf("-----------------------------\ncompressed size %d bytes\n", offset);
}

bool sar::unpack(bytes data) {

  // lookup_tbl<ustring,false,item_loc>& ltbl

  bytes stream = data;

  auto read   = [&](tslice<byte> t) { stream.pull(t); };
  auto read32 = [&]() -> uint32 {
    uint32       n;
    tslice<byte> s((byte *)&n, sizeof(n));
    read(s);
    return n;
  };
  auto read16 = [&]() -> uint16 {
    uint16       n;
    tslice<byte> s((byte *)&n, sizeof(n));
    read(s);
    return n;
  };

  byte marker[] = {'S', 'A', 'r', 0};
  byte tmarker[sizeof(marker)];

  read(target(tmarker));
  if (items_of(marker) != items_of(tmarker))
    return false;

  uint32 nnodes = read32();
  ltbl.nodes.length(nnodes);
  for (uint32 n = 0; n < nnodes; ++n) {
    auto &node     = ltbl.nodes[n];
    node.splitchar = read16();
    node.lokid     = read16();
    node.eqkid     = read16();
    node.hikid     = read16();
  }
  uint32 nitems = read32();
  ltbl.items.length(nitems);
  for (uint32 n = 0; n < nitems; ++n) {
    auto &it        = ltbl.items[n];
    it.data.offset  = read32();
    it.data.length  = read32();
    it.data.ulength = read32();
  }

  udata.length(nitems);

  return true;
}

bytes sar::get(const wchar *path) {
  array<wchar> relpath = normalize_path(chars_of(path));
  relpath.push(0);
  relpath.pop();
  uint sym = this->ltbl.search(relpath.cbegin());
  if (!sym)
    return bytes();
  auto d = ltbl.data(sym);
  if (d.ulength == 0) // raw data
    return mem(d.offset, d.offset + d.length);
  array<byte> &ud = udata[sym - 1];
  if (!ud.length()) {
    bytes cd = mem(d.offset, d.offset + d.length);
    ud.length(d.ulength);
    uint r = lzf_decompress(cd.start, uint(cd.length), ud.begin(),
                            uint(ud.length()));
    assert(r == uint(ud.length()));
    r = r;
  }
  return ud();
}

#if defined(WINDOWS)
int each_file(wchars dirpath, function<void(wchars)> cb) {
  ustring          root = dirpath;
  WIN32_FIND_DATAW file_data;
  HANDLE           hSearch = FindFirstFileW(root + L"\\*.*", &file_data);
  if (hSearch == INVALID_HANDLE_VALUE)
    return 0;
  int counter = 0;
  for (;;) {
    if ((file_data.dwFileAttributes &
         (FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_HIDDEN)) == 0 &&
        file_data.cFileName[0] != '.') {
      ustring name =
          ustring::format(WTEXT("%s\\%s"), root.c_str(), file_data.cFileName);
      if (file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
        counter += each_file(name, cb);
      else {
        ++counter;
        cb(name);
      }
    }
    if (!FindNextFileW(hSearch, &file_data))
      break;
  }
  FindClose(hSearch);
  return counter;
}

#else

/*    DIR *dp;
    struct dirent *dirp;
    struct stat filestat;
    



    std::string dir(path);
    



    dp = opendir( path );
    if (dp == nullptr)
    {
        std::cerr << "error(" << errno << ") opening " << dir << " folder" <<
   std::endl; return;
        }
        



        while ((dirp = readdir( dp )))
        {
            std::string filename = dirp->d_name;
            if(filename[0] == '.')
                continue;
            std::string filepath = dir + "/" + filename;
            



            // If the file is a directory (or is in some way invalid) we'll skip
   it if (stat( filepath.c_str(), &filestat )) continue; if (S_ISDIR(
   filestat.st_mode ))         continue;
            



            // Endeavor to read a single number from the file and display it
            FILE* fin = fopen(filepath.c_str(), "rb");
            if( !fin ) {
                std::cerr << "error(" << errno << ") reading " << filepath << "
   file" << std::endl; continue;
            }
            



            convert(fin, fout, filename.c_str(), ++no, id);
            



            std::cout << "processed:" << filepath << std::endl;
            



            fclose(fin);
        }
        closedir( dp );
 */

int each_file(wchars dirpath, function<void(wchars)> cb) {
  DIR *          dp;
  struct dirent *dirp;
  struct stat    filestat;
  tool::string   root = dirpath;

  dp = opendir(root);
  if (dp == nullptr)
    return 0;
  int counter = 0;
  while ((dirp = readdir(dp))) {
    string filename = dirp->d_name;
    if (filename[0] == '.')
      continue;
    string path = string::format("%s/%s", root.c_str(), filename.c_str());

    if (stat(path.c_str(), &filestat))
      continue;

    if (S_ISDIR(filestat.st_mode))
      counter += each_file(ustring(path), cb);
    else {
      ++counter;
      cb(ustring(path));
    }
  }
  closedir(dp);
  return counter;
}

#endif

} // namespace tool
