//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| hash table backed by array
//|
//|

#ifndef __cs_HASH_TABLE_H
#define __cs_HASH_TABLE_H

#include "tl_array.h"
#include "tl_basic.h"
#include "tl_hash.h"
#include "tl_string.h"
#include <string.h>

namespace tool {

// template <typename c_key> inline unsigned int hash ( const c_key &the_key ) {
// return the_key.hash();  }

template <typename c_key> inline bool eq(const c_key &key1, const c_key &key2) {
  return key1 == key2;
}

//{ return key1 == key2; }

template <> inline bool eq(char const *const &key1, char const *const &key2) {
  return strcmp(key1, key2) == 0;
}

template <typename c_key, typename c_element> class hash_table {
public:
  hash_table(uint hash_size = 0x20 /*32*/) {
    _hash_size = hash_size;
    _table     = new array<hash_item>[hash_size];
  }

  hash_table(const hash_table &c) {
    _hash_size = c._hash_size;
    _table     = new array<hash_item>[_hash_size];
    copy(c);
  }

  hash_table(hash_table<c_key, c_element> &&src) : _hash_size(0), _table(0) {
    swap(_hash_size, src._hash_size);
    swap(_table, src._table);
    _array.swap(src._array);
  }

  virtual ~hash_table() {
    clear();
    delete[] _table;
  }

  bool exists(const c_key &the_key) const;

  hash_table &operator=(const hash_table &c) {
    copy(c);
    return *this;
  }

  hash_table &operator=(hash_table &&src) {
    if (this == &src)
      return *this;
    _array.destroy();
    delete[] _table;
    _table = 0;
    swap(_hash_size, src._hash_size);
    swap(_table, src._table);
    _array.swap(src._array);
    return *this;
  }

  //
  // Add to hash table association of object with specified name.
  //
  c_element &      operator[](const c_key &the_key);
  c_element        operator[](const c_key &the_key) const;
  c_element &      operator()(index_t the_index);
  const c_element &operator()(index_t the_index) const;

  //
  // Search for object with specified name in hash table.
  // If object is not found false is returned.
  //
  bool find(const c_key &the_key, c_element &the_element) const;
  bool find(const c_key &the_key, c_element *&the_element_ptr) const;

  //
  // Remove object with specified name from hash table.
  //
  void remove(const c_key &the_key);
  void remove_at(index_t the_index);

  index_t get_index(const c_key &the_key, bool create = false);

  // get reference to existing or newly created element (created == true in this
  // case)
  c_element &get_ref(const c_key &the_key, bool &created);
  // get reference to existing or newly created element (created == true in this
  // case)
  c_element &get_ref(const c_key &the_key, function<void(c_element &)> initialzer);

  void clear();

  index_t size() const { return _array.size(); }
  size_t  length() const { return _array.length(); }
  bool    is_empty() const { return _array.size() == 0; }

  // will return list of all keys. key allkeys[n] correspond to element in n'th
  // position in elements()
  void keys(array<c_key> &allkeys) const;

protected:
  struct hash_item {
    unsigned int _key_hash = 0;
    c_key        _key;
    index_t      _index = 0;

    hash_item() {}

    hash_item(uint kh, const c_key &k, index_t i)
        : _key_hash(kh), _key(k), _index(i) {}
  };
  size_t            _hash_size;
  array<hash_item> *_table;

  array<c_element> _array;
  c_element *      _get(const c_key &the_key, bool create);

  void copy(const hash_table &c) {
    if (&c == this)
      return;
    clear();
    array<c_key> k;
    c.keys(k);
    for (index_t i = 0; i < k.size(); i++)
      (*this)[k[i]] = c[k[i]];
  }

public:
  void transfer_from(hash_table &src) {
    swap(_hash_size, src._hash_size);
    swap(_table, src._table);
    _array.transfer_from(src._array);
    src.clear();
  }

  array<c_element> &elements() { return _array; }

#ifdef _DEBUG
  void report() {
    dbg_printf("hash table: size=%d\n", _hash_size);
    dbg_printf("\tbuckets:\n");
    for (unsigned i = 0; i < _hash_size; i++) {
        dbg_printf("%d\t", _table[i].size());
        if ((i + 1) % 10 == 0) dbg_printf("\n");
    }
  }
#endif
};

template <class c_key, class c_element>
inline c_element &hash_table<c_key, c_element>::
                  operator[](const c_key &the_key) {
  return *_get(the_key, true);
}

template <class c_key, class c_element>
inline c_element hash_table<c_key, c_element>::
                 operator[](const c_key &the_key) const {
  c_element *pel =
      const_cast<hash_table<c_key, c_element> *>(this)->_get(the_key, false);
  if (pel)
    return *pel;
  // assert(false);
  return c_element();
}

template <class c_key, class c_element>
inline bool hash_table<c_key, c_element>::find(const c_key &the_key,
                                               c_element &  the_element) const {
  c_element *pe =
      const_cast<hash_table<c_key, c_element> *>(this)->_get(the_key, false);
  if (pe) {
    the_element = *pe;
    return true;
  }
  return false;
}

template <class c_key, class c_element>
inline bool
hash_table<c_key, c_element>::find(const c_key &the_key,
                                   c_element *& the_element_ptr) const {
  c_element *pe =
      const_cast<hash_table<c_key, c_element> *>(this)->_get(the_key, false);
  if (pe) {
    the_element_ptr = pe;
    return true;
  }
  return false;
}

template <class c_key, class c_element>
inline c_element *hash_table<c_key, c_element>::_get(const c_key &the_key,
                                                     bool         create) {
  index_t idx = get_index(the_key, create);
  if (idx < 0)
    return 0;
  return &_array[idx];
}

template <class c_key, class c_element>
inline c_element &hash_table<c_key, c_element>::get_ref(const c_key &the_key,
                                                        bool &       created) {
  index_t prev_size = _array.size();
  index_t idx       = get_index(the_key, true);
  created           = prev_size != _array.size();
  return _array[idx];
}

template <class c_key, class c_element>
inline c_element &
hash_table<c_key, c_element>::get_ref(const c_key &               the_key,
                                      function<void(c_element &)> initializer) {
  index_t prev_size = _array.size();
  index_t idx       = get_index(the_key, true);
  if (prev_size < _array.size())
    initializer(_array[idx]);
  return _array[idx];
}

template <class c_key, class c_element>
inline index_t hash_table<c_key, c_element>::get_index(const c_key &the_key,
                                                       bool         create) {
  unsigned int hk = hash<c_key>(the_key);
  uint         h;
#ifdef UNDER_CE
  switch (_hash_size) {
  case 0x10:
    h = hk & 0x0F;
    break;
  case 0x20:
    h = hk & 0x1F;
    break;
  case 0x40:
    h = hk & 0x3F;
    break;
  case 0x80:
    h = hk & 0x7F;
    break;
  default:
    h = hk % _hash_size;
    break;
  }
#else
  h = hk % _hash_size;
#endif

  index_t           i;
  array<hash_item> &bucket = _table[h];

  for (i = 0; i < bucket.size(); i++) {
    const hash_item &it = bucket[i];
    if ((hk == it._key_hash) && eq(it._key, the_key))
      return it._index;
  }

  if (create) {
    index_t ni = _array.size();
    _array.size(ni + 1);
    bucket.push(hash_item(hk, the_key, ni));
    return ni;
  }
  return -1;
}

template <class c_key, class c_element>
inline void hash_table<c_key, c_element>::keys(array<c_key> &allkeys) const {
  allkeys.size(size());
  for (size_t i = 0; i < _hash_size; ++i) {
    array<hash_item> &bucket = _table[i];
    for (index_t k = 0; k < bucket.size(); k++) {
      hash_item &hi      = bucket[k];
      allkeys[hi._index] = hi._key;
    }
  }
}

template <class c_key, class c_element>
inline void hash_table<c_key, c_element>::remove(const c_key &the_key) {
  uint              h = hash<c_key>(the_key) % _hash_size;
  index_t           i;
  array<hash_item> &bucket = _table[h];

  for (i = 0; i < bucket.size(); i++) {
    const hash_item &it = bucket[i];
    if (it._key == the_key) {
      index_t index = it._index;
      _array.remove(index);
      bucket.remove(i);
      // adjust other references
      for (h = 0; h < _hash_size; h++) {
        array<hash_item> &bucket = _table[h];
        for (i = 0; i < bucket.size(); i++) {
          hash_item &it = bucket[i];
          if (it._index > index)
            it._index--;
        }
      }
      return;
    }
  }
}

template <class c_key, class c_element>
inline void hash_table<c_key, c_element>::remove_at(index_t the_index) {
    for (size_t h = 0; h < _hash_size; h++) {
      array<hash_item> &bucket = _table[h];
      for (index_t i = bucket.last_index(); i >= 0; --i) {
        hash_item &it = bucket[i];
        if (it._index == the_index) {
          _array.remove(the_index);
          bucket.remove(i);
        }
        else if (it._index > the_index)
          it._index--;
      }
    }
}

template <class c_key, class c_element>
inline c_element &hash_table<c_key, c_element>::operator()(index_t the_index) {
  return _array[the_index];
}

template <class c_key, class c_element>
inline const c_element &hash_table<c_key, c_element>::
                        operator()(index_t the_index) const {
  return _array[the_index];
}

template <class c_key, class c_element>
inline bool hash_table<c_key, c_element>::exists(const c_key &the_key) const {
  return const_cast<hash_table<c_key, c_element> *>(this)->_get(the_key,
                                                                false) != 0;
}

template <class c_key, class c_element>
inline void hash_table<c_key, c_element>::clear() {
  if (_array.size()) {
    for (uint i = 0; i < _hash_size; ++i)
      _table[i].clear();
    _array.clear();
  }
}

}; // namespace tool

#endif
