//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terra-informatica.org
//|
//| pool of unique instances, e.g. strings
//|
//|

#ifndef __tl_handle_pool_h
#define __tl_handle_pool_h

#include "tl_basic.h"

namespace tool {

//#define POOL_TRAITS pool_traits<c_element>

template <typename c_element>
inline bool handle_pool_equal(const c_element &l, const c_element &r) {
  return l == r;
}
template <typename c_element> inline uint handle_pool_hash(const c_element &l) {
  return l.hash();
}

template <typename c_element> class handle_pool {

  handle_pool &operator=(const handle_pool &c) {
    copy(c);
    return *this;
  }

  handle_pool(const handle_pool &c) {
    _hash_size = c._hash_size;
    _table     = new array<hash_item>[_hash_size];
    copy(c);
  }

public:
  handle_pool(uint hash_size = 32) {
    _hash_size = hash_size;
    _table     = new array<hash_item>[hash_size];
  }

  virtual ~handle_pool() {
    clear();
    delete[] _table;
  }

  bool exists(const c_element &the_key);

  //
  // Add to hash table association of object with specified name.
  //
  index_t          operator[](const c_element &the_key);
  index_t          operator[](const c_element &the_key) const;
  c_element *      operator()(index_t the_index);
  const c_element *operator()(index_t the_index) const;

  inline c_element *intern(const c_element &elem) {
    return _array[get_index(elem, true)];
  }

  inline c_element *intern(const c_element *elem) {
    return _array[get_index(elem, true)];
  }

  //
  // Remove object with specified name from hash table.
  //
  void remove(const c_element &the_key);

  index_t get_index(const c_element &the_key, bool create = false);
  index_t get_index(const c_element *the_key, bool create = false);

  void clear();

  index_t size() const { return _array.size(); }
  size_t  length() const { return _array.length(); }
  bool    is_empty() const { return _array.size() == 0; }

protected:

  void gc();

  struct hash_item {
    uint _key_hash = 0;
    uint _index = 0;

    hash_item() {}

    hash_item(uint kh, uint i) : _key_hash(kh), _index(i) {}
  };
  uint                     _hash_size;
  array<hash_item>*        _table;
  array<handle<c_element>> _array;
  uint                     _expansion_counter = 0;

public:
  array<handle<c_element>> &elements() { return _array; }

#ifdef _DEBUG
  void report() {
    //FILE *f = fopen("d:\\handle_pool.txt", "wt+");
    //if (!f)
    //  return;
    dbg_printf( "pool size: %d\n", _array.size());
    dbg_printf( "hash table: size=%d\n", _hash_size);
    dbg_printf( "\tbuckets:\n");
    for (unsigned i = 0; i < _hash_size; i++) {
      dbg_printf( " %d", _table[i].size());
      //if (_table[i].size() == 1)
      //  dbg_printf( " *");
      /*else if (_table[i].size() > 1) {
        for (int k = 0; k < _table[i].size(); ++k) {
          dbg_printf( " %X(%d)", _table[i][k]._key_hash, _table[i][k]._index);
          c_element *elm = _array[_table[i][k]._index];
          elm            = elm;
        }
        // dbg_printf(" !!!!");
      }*/
      if( (i % 10) == 9 )
        dbg_printf( "\n");
    }
    //fclose(f);
  }
#endif
};

template <typename c_element>
inline index_t handle_pool<c_element>::operator[](const c_element &the_key) {
  return get_index(the_key, true);
}

template <typename c_element>
inline index_t handle_pool<c_element>::
               operator[](const c_element &the_key) const {
  return const_cast<handle_pool<c_element> *>(this)->get_index(the_key, false);
}

template <typename c_element>
inline index_t handle_pool<c_element>::get_index(const c_element &the_key,
                                                 bool             create) {
  uint hk = handle_pool_hash(the_key);
  uint h;
  h = hk % _hash_size;
  array<hash_item> &bucket = _table[h];
  index_t           i, itotal = bucket.size();

  assert(itotal < 100); // too long collision chain, something wrong with hash function?

  for (i = 0; i < itotal; i++) {
    const hash_item &it = bucket[i];
    if (hk != it._key_hash)
      continue;

    if (handle_pool_equal(the_key, *(_array[it._index].ptr())))
      // if ( the_key == *(_array[it._index].ptr()) )
      return it._index;
  }

  if (create) {
    if (++_expansion_counter > 100) {
      _expansion_counter = 0;
      gc();
    }
    uint ni = (uint)_array.length();
    _array.push(new c_element(the_key));
    bucket.push(hash_item(hk, ni));
    return ni;
  }
  return -1;
}

template <typename c_element>
inline index_t handle_pool<c_element>::get_index(const c_element *the_key,
                                                 bool             create) {
  uint hk = the_key->hash();
  uint h  = hk % _hash_size;
  index_t           i;
  array<hash_item> &bucket = _table[h];

  for (i = 0; i < bucket.size(); i++) {
    const hash_item &it = bucket[i];
    if ((hk == it._key_hash) && (*the_key == *(_array[it._index].ptr())))
      return it._index;
  }

  if (create) {
    if (++_expansion_counter > 100) {
      _expansion_counter = 0;
      gc();
    }
    uint ni = _array.size();
    _array.push(the_key);
    bucket.push(hash_item(hk, ni));
#ifdef _DEBUG
    assert(bucket.size() < 20);
#endif 
    return ni;
  }
  return -1;
}

template <typename c_element>
inline void handle_pool<c_element>::remove(const c_element &the_key) {
  uint              h = hash<c_element>(the_key) % _hash_size;
  index_t           i;
  array<hash_item> &bucket = _table[h];
  for (i = 0; i < bucket.size(); i++) {
    const hash_item &it = bucket[i];
    if (it._key == the_key) {
      unsigned int index = it._index;
      _array.remove(index);
      bucket.remove(i);
      // adjust other references
      for (h = 0; h < _hash_size; h++) {
        array<hash_item> &bucket = _table[h];
        for (i = 0; i < bucket.size(); i++) {
          hash_item &it = bucket[i];
          if (it._index > index)
            it._index--;
        }
      }
      return;
    }
  }
}

template <typename c_element>
inline c_element *handle_pool<c_element>::operator()(index_t the_index) {
  return _array[the_index];
}

template <typename c_element>
inline const c_element *handle_pool<c_element>::
                        operator()(index_t the_index) const {
  return _array[the_index];
}

template <typename c_element>
inline bool handle_pool<c_element>::exists(const c_element &the_key) {
  return get_index(the_key, false) != -1;
}

template <typename c_element> inline void handle_pool<c_element>::clear() {
  for (int i = 0; i < int(_hash_size); ++i)
    _table[i].clear();
  _array.clear();
}

template <typename c_element> inline void handle_pool<c_element>::gc() {
  for (auto& el : _array)
    if (el->get_ref_count() == 1)
      goto CLEAN_DEAD;
  return;
CLEAN_DEAD:
  for (int i = 0; i < int(_hash_size); ++i) _table[i].clear();
  array<handle<c_element>> list; list.transfer_from(_array);
  for (auto& el : list) {
    if (el->get_ref_count() == 1) continue;
    uint hk = el->hash();
    uint h = hk % _hash_size;
    array<hash_item> &bucket = _table[h];
    uint ni = _array.size();
    _array.push(el);
    bucket.push(hash_item(hk, ni));
#ifdef _DEBUG
    assert(bucket.size() < 20);
#endif 
   }
  }

} // namespace tool

#endif
