//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|

#ifndef __tl_handle_h_
#define __tl_handle_h_

#include "tl_basic.h"
#include "tl_slice.h"

namespace tool {

// nocopy thing
template <typename T = int> struct NO_COPY {
private:
  NO_COPY(const NO_COPY &);
  NO_COPY &operator=(const NO_COPY &);

protected:
  NO_COPY() {}
};

// turn_to<B>(A* obj) changes class of the object
// that means it just replaces VTBL of the object to new class:
/*template <typename TO_T, typename FROM_T>
  inline TO_T* turn_to(FROM_T* p)
  {
    assert_static( sizeof(FROM_T) == sizeof(TO_T));
    p->transforming(); // notify that the object is about to loose old vtbl
    TO_T* pt = ::new(p) TO_T(NO_INIT());
    pt->transformed(); // variant of ctor, the object just got new class
    return pt;
  }*/

/****************************************************************************/

template <class T> class weak_handle;

class value;

template <class T> class handle {
public:
  handle() { _ptr = nullptr; }
  handle(NO_INIT) {}

  handle(const T *p) {
    _ptr = nullptr;
    _set(const_cast<T *>(p));
  }

  handle(const handle<T> &p) {
    _ptr = nullptr;
    _set(p._ptr);
  }

  template<class DT>
  handle(const handle<DT> &p) {
    _ptr = nullptr;
    _set(static_cast<T*>(p.ptr()));
  }

  handle(const weak_handle<T> &p);

  ~handle() { 
    if (_ptr) {
      _ptr->release();
      _ptr = nullptr;
    }
  }

  handle<T> &operator=(T *p) {
    _set(p);
    return *this;
  }

  handle<T> &operator=(const handle<T> &p) {
    _set(p._ptr);
    return *this;
  }
    
  handle<T> &operator=(const weak_handle<T> &p);
    
  template<class DT>
  handle<T> &operator=(const handle<DT> &p) {
    _set(static_cast<T*>(p.ptr()));
    return *this;
  }

  T *operator->() const { return _ptr; }

  T *ptr() const { return _ptr; }

  template <typename Y> Y *ptr_of() const { return static_cast<Y *>(_ptr); }

  operator T *() const { return _ptr; }

  bool is_null() const { return _ptr == nullptr; }

  bool is_identical(const handle& other) const { 
    if (_ptr == other._ptr) return true;
    if (is_undefined() || other.is_undefined()) return false;
    return *_ptr == *other._ptr;
  }

  T *detach() {
    T *t = _ptr;
    _ptr = 0;
    return t;
  }

  bool is_undefined() const { return _ptr == 0; }
  bool is_defined() const { return _ptr != 0; }

  void clear() { _set(nullptr); }
  void inherit(const handle &v) {
    if (v.is_defined())
      _set(v._ptr);
  }

  unsigned int hash() const {
    /*if(is_proxy)
    return _ptr? tool::hash(*ptr):0;
    else*/
    return (unsigned int)(uint_ptr)_ptr;
  }

  /*bool get_interface( resource* r ) {
  _set(0);
  return r->get_interface(T::interface_name(),&_ptr);
  }*/

  value to_value() const;

private:
  T *  _ptr;
  void _set(T *p) {
    if (_ptr == p)
      return;

    if (_ptr)
      _ptr->release();

    _ptr = p;

    if (_ptr)
      _ptr->add_ref();
  }
};

// similar to handle<T> but has value semantic of operator== and hash() 
template <class T> class value_handle {
public:
  value_handle() { _ptr = nullptr; }
  value_handle(NO_INIT) {}

  value_handle(const T *p) {
    _ptr = nullptr;
    _set(const_cast<T *>(p));
  }

  value_handle(const value_handle<T> &p) {
    _ptr = nullptr;
    _set(p._ptr);
  }

  template<class DT>
  value_handle(const handle<DT> &p) {
    _ptr = nullptr;
    _set(static_cast<T*>(p.ptr()));
  }

  ~value_handle() {
    if (_ptr) {
      _ptr->release();
      _ptr = nullptr;
    }
  }

  value_handle<T> &operator=(T *p) {
    _set(p);
    return *this;
  }

  value_handle<T> &operator=(const value_handle<T> &p) {
    _set(p._ptr);
    return *this;
  }

  template<class DT>
  value_handle<T> &operator=(const value_handle<DT> &p) {
    _set(static_cast<T*>(p.ptr()));
    return *this;
  }

  T *operator->() const { return _ptr; }

  T *ptr() const { return _ptr; }

  template <typename Y> Y *ptr_of() const { return static_cast<Y *>(_ptr); }

  operator T *() const { return _ptr; }

  bool is_null() const { return _ptr == nullptr; }

  bool is_identical(const T* other) const {
    if (_ptr == other) return true;
    if (!_ptr || !other) return false;
    return *_ptr == *other;
  }

  bool operator == ( const T *p ) const { return is_identical(p); }
  bool operator != ( const T *p ) const { return !is_identical(p); }
  bool operator == ( const value_handle<T>& p ) const { return is_identical(p.ptr()); }
  bool operator != ( const value_handle<T>& p ) const { return !is_identical(p.ptr()); }

  T *detach() {
    T *t = _ptr;
    _ptr = 0;
    return t;
  }

  bool is_undefined() const { return _ptr == 0; }
  bool is_defined() const { return _ptr != 0; }

  void clear() { _set(nullptr); }
  void inherit(const value_handle &v) {
    if (v.is_defined())
      _set(v._ptr);
  }

  unsigned int hash() const {
    return _ptr ? _ptr->hash() : 0;
  }

  value to_value() const;

private:
  T *  _ptr;
  void _set(T *p) {
    if (_ptr == p)
      return;

    if (_ptr)
      _ptr->release();

    _ptr = p;

    if (_ptr)
      _ptr->add_ref();
  }
};


class weakable;

namespace weak {
  class proxy {
    locked::counter _ref_cntr;
    weakable *_wptr;
    friend class resource;
    template<class> friend class handle;
    template<class> friend class weak_handle;
  public:
    proxy() :_ref_cntr(0), _wptr(nullptr) {}
    proxy(weakable *ptr) :_ref_cntr(0), _wptr(ptr) {}
    virtual ~proxy() {
      assert(_wptr == nullptr);
      assert(_ref_cntr == 0);
    }
    weakable* ptr() { return _wptr; }
    void      ptr(weakable* p) { _wptr = p; }
    virtual void finalize() {
      delete this;
    }
    virtual long release() {
      assert(_ref_cntr > 0);
      long t = --_ref_cntr;
      if (t == 0)
        finalize();
      return t;
    }
    virtual long add_ref() { 
      return ++_ref_cntr; 
    }
  };
}

class resource {

  template<class> friend class handle;

  locked::counter       _ref_cntr;

  resource &operator = (const resource & /*r*/) = delete;

  typedef void super;

public:
  resource(NO_INIT ni) : _ref_cntr(ni) {}
  resource() : _ref_cntr(0) {}
  resource(const resource & /*r*/) : _ref_cntr(0) {}

  virtual ~resource() { 
    if(_ref_cntr != 0)
      assert(_ref_cntr == 0); 
  }
  long get_ref_count() const { return _ref_cntr; }

  virtual long release() {
    assert(_ref_cntr > 0);
    long t = --_ref_cntr;
    if (t == 0)
      finalize();
    return t;
  }
  virtual long add_ref() { return ++_ref_cntr; }

  virtual void final_release() {}

  virtual void finalize() {
    final_release();
    delete this;
  }

  uint_ptr                    class_id() const { return 0; }
  virtual uint_ptr            type_id() const { return 0; }
  template <typename OT> bool is_of_type() const { return is_of_type_id(OT::class_id()); }

  virtual bool is_of_type_id(uint_ptr tid) const { return tid == class_id(); }
  virtual bool is_of_type_name(tool::chars tn) const { return tn == resource_class_name(); }

  virtual tool::chars resource_class_name() const { return CHARS("[native resource]"); }

  virtual bool get_interface(tool::chars name, void **pp) { 
    if (name == resource_class_name()) {
      if(pp) *pp = (resource*)this;
      return true;
    }
    return false; 
  }

  //som_passport_t* get_passport() const { return nullptr; }

  template <typename Y> Y *get_interface() {
    void *p = nullptr;
    if (get_interface(Y::interface_name(), &p))
      return static_cast<Y*>(p);
    return nullptr;
  }

  template <typename T> T *cast() {
    if(this->is_of_type<T>())
      return static_cast<T*>(this);
    return nullptr;
  }
  

};

// class that needs to support weak_handle (weak pointer) shall be derived from this:
class weakable {
  template<class> friend class weak_handle;
  handle<weak::proxy>   _p_ptr;
public:
  weakable() {} 
  weakable(NO_INIT ni) : _p_ptr(ni) {}
  ~weakable() { 
    if(_p_ptr)
      _p_ptr->ptr(nullptr);
    _p_ptr = nullptr; 
  }
};

template <class T>
class weak_handle {
  handle<weak::proxy> _wptr;
public:
  weak_handle() : _wptr() {}
  weak_handle(T *ptr) : _wptr(proxy_of(ptr)) {}
  weak_handle(const handle<T> &ptr) : _wptr(proxy_of(ptr.ptr())) {}
  weak_handle(NO_INIT ni) : _wptr(ni) { }
  weak_handle<T>& operator=(const handle<T>& ptr) {
    _wptr = proxy_of(ptr.ptr());
    return *this;
  }
  weak_handle<T>& operator=(T *ptr) {
    _wptr = proxy_of(ptr);
    return *this;
  }
  //operator const handle<T>() const { return handle<T>(ptr()); }
  
  T* ptr() const { return _wptr ? (T*)_wptr->ptr() : nullptr; }
  template <typename Y> Y *ptr_of() const { return static_cast<Y *>(ptr()); }

  bool is_null() const { return ptr() == nullptr; }

  operator T*()const { return (T*)ptr(); }
  T* operator->()const { return ptr(); }

  void clear() { _wptr = nullptr; }

  void inherit(const weak_handle &v) {
    if (!v.is_null())
      operator=(v.ptr());
  }


private:
  static handle<weak::proxy> proxy_of(T *ptr) {
    weakable *o = static_cast<weakable*>(ptr);
    if (!o) { return handle<weak::proxy>(); }
    else if (!o->_p_ptr) { o->_p_ptr = new weak::proxy(o); }
    assert(o->_p_ptr->ptr() == o);
    return o->_p_ptr;
  }
};

/*template<class DT>
bool operator == (const DT* p) const { return ptr() == static_cast<const T*>(p); }
template<class DT>
bool operator != (const DT* p) const { return ptr() != static_cast<const T*>(p); }*/

template<typename T>
  inline handle<T>::handle(const weak_handle<T> &p) {
    _ptr = nullptr;
    _set(static_cast<T*>(p.ptr()));
  }
    
template<typename T>
    inline handle<T>& handle<T>::operator=(const weak_handle<T> &p) {
        _set(p.ptr());
        return *this;
    }


#define TOOL_INTERFACE_NAME(ns, n)                                             \
  static chars interface_name() { return CHARS(#n "." #ns); }

#define TOOL_INTERFACE(ns, n)                                                  \
  static chars interface_name() { return CHARS(#n "." #ns); }                  \
  virtual bool get_interface(chars name, void **pp) override {                 \
    if (name == resource_class_name()) {                                       \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    if (name == interface_name()) {                                            \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    return false;                                                              \
  }

#define TOOL_INTERFACE_1(ns, n, pn)                                            \
  static chars interface_name() { return CHARS(#n "." #ns); }                  \
  virtual bool get_interface(chars name, void **pp) override {                 \
    if (name == resource_class_name()) {                                       \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    if (name == interface_name()) {                                            \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    if (pn::get_interface(name, pp))                                           \
      return true;                                                             \
    return false;                                                              \
  }

#define TOOL_INTERFACE_2(ns, n, pn1, pn2)                                      \
  static chars interface_name() { return CHARS(#n "." #ns); }                  \
  /*virtual tool::chars resource_class_name() const override { return interface_name(); }*/ \
  virtual bool get_interface(chars name, void **pp) override {                 \
    if (name == resource_class_name()) {                                       \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    if (name == interface_name()) {                                            \
      *pp = this;                                                              \
      return true;                                                             \
    }                                                                          \
    if (pn1::get_interface(name, pp))                                          \
      return true;                                                             \
    if (pn2::get_interface(name, pp))                                          \
      return true;                                                             \
    return false;                                                              \
  }

template <typename T> class resource_x : public resource {
public:
  resource_x() : resource() {}
  resource_x(NO_INIT ni) : resource(ni) {}
  static uint_ptr class_id() {
    static int d = 0;
    return (uint_ptr)&d;
  }
  virtual uint_ptr    type_id() const override { return class_id(); }
  virtual tool::chars resource_class_name() const override {
    return CHARS("[native x-object]");
  }
  virtual bool is_of_type_id(uint_ptr tid) const { return tid == class_id(); }
};

#if 0
class resource_asset : public virtual resource, public sciter::om::iasset {
public:
  resource_asset() : resource() {}

  virtual const char *resource_class_name() const override { return "resource/asset"; }

  virtual bool get_interface(chars name, void **pp) override {
    if (name == CHARS("asset"))
      *pp = static_cast<iasset*>(this);
    return resource::get_interface(name, pp);
  }

  sciter::om::iasset* as_asset() { return static_cast<sciter::om::iasset*>(this); }

  virtual long asset_add_ref() override { 
    return resource::add_ref(); 
  }
  virtual long asset_release() override { 
    return resource::release(); 
  }

};
#endif


#define DEFINE_TYPE_ID(classname)                                              \
  virtual tool::chars resource_class_name() const override {                   \
    return CHARS(#classname);                                                  \
  }                                                                            \
  static uint_ptr class_id() {                                                 \
    static int d = 0;                                                          \
    return (uint_ptr)&d;                                                       \
  }                                                                            \
  virtual uint_ptr type_id() const override { return class_id(); }             \
  virtual bool     is_of_type_id(uint_ptr tid) const override {                \
    return tid == class_id(); }                                                \
  virtual bool     is_of_type_name(tool::chars tn) const override {            \
    return tn == resource_class_name(); }

#define DEFINE_TYPE_ID_DERIVED(classname, parentclassname)                     \
  typedef parentclassname super;                                               \
  virtual tool::chars resource_class_name() const override {                   \
    return CHARS(#classname);                                                  \
  }                                                                            \
  static uint_ptr class_id() {                                                 \
    static int d = 0;                                                          \
    return (uint_ptr)&d;                                                       \
  }                                                                            \
  virtual uint_ptr type_id() const override{ return class_id(); }              \
  virtual bool     is_of_type_id(uint_ptr tid) const override {                \
    return tid == class_id() || parentclassname::is_of_type_id(tid);           \
  }

// turn_resource_to<RB>(RA* obj) changes class of the object that is a resource
template <typename TO_T, typename FROM_T>
inline bool turn_object_to(FROM_T *p) {
  assert_static(sizeof(FROM_T) == sizeof(TO_T));
  if (p->type_id() == TO_T::class_id())
    return false;    // no need to change
  p->transforming(); // notify that the object is about to loose old vtbl
  TO_T *pt = ::new (p) TO_T(NO_INIT());
  pt->transformed(); // variant of ctor, the object just got new class
  return true;
}

class ext_resource {
  locked::counter _ext_ref_cntr;

public:
  ext_resource(NO_INIT ni) : _ext_ref_cntr(ni) {}
  ext_resource() : _ext_ref_cntr(0) {}
  virtual ~ext_resource() { assert(_ext_ref_cntr == 0); }
  long         ext_get_ref_count() const { return _ext_ref_cntr; }
  virtual long ext_release() {
    assert(_ext_ref_cntr > 0);
    long t = --_ext_ref_cntr;
    if (t == 0)
      ext_finalize();
    return t;
  }
  virtual void ext_add_ref() { ++_ext_ref_cntr; }

  virtual void ext_finalize() = 0;
};

struct functor : virtual public resource {
  functor() {}
  virtual ~functor() {}
  virtual bool operator()() = 0; // returns 'true' if complete
  NONCOPYABLE(functor)
};

struct function_functor : public functor {
  function<bool()> f;
  locked::counter  processed;
  function_functor(function<bool()> pf) : f(pf) {}
  virtual ~function_functor() {}
  virtual bool operator()() override {
    bool r    = f();
    processed = 1;
    return r;
  }
  NONCOPYABLE(function_functor)
};

} // namespace tool

#endif
