#pragma once

#ifndef __tl_audio_h
#define __tl_audio_h

//#include <atomic>

#include "tl_handle.h"
#include "tl_url.h"

#define DR_FLAC_IMPLEMENTATION
#include "external/audio/extras/dr_flac.h"  // Enables FLAC decoding.
#define DR_MP3_IMPLEMENTATION
#include "external/audio/extras/dr_mp3.h"   // Enables MP3 decoding.
#define DR_WAV_IMPLEMENTATION
#include "external/audio/extras/dr_wav.h"   // Enables WAV decoding.

#define MINIAUDIO_IMPLEMENTATION
#include "external/audio/miniaudio.h"

namespace tool {

  struct audio : public resource
  {
    enum result_e {
      playback_start_error = -3,
      playback_device_error = -2,
      file_open_error = -1,
      success = 0
    };

    enum state_e {
      stopped,
      playing,
      paused,
    };

    volatile state_e state = stopped; // note, we don't need strict atomics here
    
    uint64 passed_frames = 0;
    uint64 total_frames = 0;
    bool   stopping = false;
    float  volume = -1.0;
    

    array<byte> data; // if not from file
    
    ma_decoder decoder = {};
    ma_device  device = {};

    audio() {}
    virtual ~audio() { 
      cleanup();
    }

    void cleanup() {
      if(state != stopped) {
        ma_device_stop(&device);
        ma_decoder_uninit(&decoder);
        ma_device_uninit(&device);
        state = stopped;
      }
    }

    virtual void on_heartbit() {}
    virtual void on_stop() { }

    result_e play(array<byte> data_to_play) {

      data = data_to_play;

      ma_result result = ma_decoder_init_memory(data.cbegin(),data.length(), nullptr, &decoder);

      if (result != MA_SUCCESS)
        return file_open_error;

      return perform();
    }

    result_e play(wchars path) {
#ifdef WINDOWS
      ma_result result = ma_decoder_init_file_w(ustring(path), nullptr, &decoder);
#else
      ma_result result = ma_decoder_init_file(string(path), nullptr, &decoder);
#endif // WINDOWS
      if (result != MA_SUCCESS)
        return file_open_error;
      
      return perform();
    }

    result_e perform() 
    {
      if(!total_frames)
        total_frames = ma_decoder_get_length_in_pcm_frames(&decoder);
        
      ma_device_config config = ma_device_config_init(ma_device_type_playback);
      config.playback.format = decoder.outputFormat;
      config.playback.channels = decoder.outputChannels;
      config.sampleRate = decoder.outputSampleRate;
      config.dataCallback = data_callback;
      config.stopCallback = stop_callback;
      config.pUserData = this;

      if (ma_device_init(nullptr, &config, &device) != MA_SUCCESS) {
        //printf("Failed to open playback device.\n");
        ma_decoder_uninit(&decoder);
        return playback_device_error;
      }

      if (volume >= 0)
        ma_device_set_master_volume(&device, volume);

      state = playing;

      if (ma_device_start(&device) != MA_SUCCESS) {
        //printf("Failed to start playback device.\n");
        ma_device_uninit(&device);
        ma_decoder_uninit(&decoder);
        return playback_start_error;
      }
      
      //this->add_ref();
        
      return success;
    }

    virtual bool stop() {
      cleanup();
      return true;
    }

    virtual bool pause() {
      state = paused;
      return true;
    }

    virtual bool resume() {
      state = playing;
      return true;
    }


    bool set_volume(float max1) {
      if (state)
        return ma_device_set_master_volume(&device, volume = max1) == MA_SUCCESS;
      volume = max1;
      return true;
    }

    bool get_volume(float& max1) {
      if (state && ma_device_get_master_volume(&device, &volume) == MA_SUCCESS) {
        max1 = volume;
        return true;
      }
      return false;
    }

    bool get_position(float& max1) {
      if (total_frames) {
        max1 = float(double(passed_frames) / total_frames);
        return true;
      }
      return false;
    }

    static void data_callback(ma_device* pDevice, void* pOutput, const void* pInput, ma_uint32 frameCount)
    {
      audio* pap = (audio*)pDevice->pUserData;
      if (!pap)
        return;
      if (pap->state == paused || pap->state == stopped)
        return;
      if (pap->stopping)
        return;
      ma_decoder* pDecoder = &pap->decoder;
      if (pDecoder == nullptr)
        return;
      pap->passed_frames += frameCount;
      pap->on_heartbit();
      uint64 frames_read = ma_decoder_read_pcm_frames(pDecoder, pOutput, frameCount);
      if (frames_read == 0) {
        pap->stopping = true;
        pap->on_stop();
        //pap->state = stopped;
      }
      (void)pInput;
    }

    static void stop_callback(ma_device* pDevice) {
      audio* pap = (audio*)pDevice->pUserData;
      if (!pap) return;
      if (pap->stopping) return;
      pap->on_stop();
    }
  };

}

#endif
