//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| dynamic array
//|
//|

#ifndef __tl_array_h
#define __tl_array_h

#include "tl_basic.h"
#include "tl_slice.h"
#include <stdio.h>

namespace tool {
// assumes element has a default constructor and operator=().

void hash_combine(uint &seed, uint v);

template <typename element> class array {
public:
  struct array_data {
    locked::counter ref_count;
    size_t          size;
    size_t          allocated_size;
    element         elements[1];

    static slice<element> get_elements(const array_data *pd) {
      return pd ? slice<element>(pd->elements, pd->size) : slice<element>();
    }

    static array_data *alloc(size_t alloc_num) {
      assert(alloc_num);
      array_data *pd = (array_data *)calloc(
          sizeof(array_data) + sizeof(element) * (alloc_num - 1), 1);
      if (!pd)
        return nullptr;
      pd->allocated_size = alloc_num;
      pd->ref_count      = 1;
      return pd;
    }
    /*static void        free(array_data* &pd)  {
                                                if(!pd) return;
                                                assert(pd->ref_count >= 0);
                                                erase(pd->elements,pd->size);
                                                pd->size = 0;
                                                if(--pd->ref_count == 0)
                                                  ::free(pd);
                                                pd = nullptr;
                                              }*/
    static size_t get_size(array_data *pd) { return pd ? pd->size : 0; }
    static size_t get_allocated_size(array_data *pd) {
      return pd ? pd->allocated_size : 0;
    }
    static void add_ref(array_data *pd) {
      if (pd)
        ++pd->ref_count;
    }
    static void release(array_data *&pd) {
      if (!pd)
        return;
      if (--pd->ref_count == 0) {
        erase(pd->elements, pd->size);
        pd->size = 0;
        ::free(pd);
      }
      pd = nullptr;
    }
  };

protected:
  array_data *data;

  struct data_holder {
    array_data *data;
    data_holder(array *p) {
      data = p->data;
      array_data::add_ref(data);
    }
    ~data_holder() { array_data::release(data); }
  };

public:
  typedef element element_t;

  array();
  array(size_t sz);
  array(size_t sz, const element &init_element);
  array(size_t sz, std::function<void(int, element &)> f);
  array(const array<element> &the_array);
  array(const slice<element> &d);
  array(NO_INIT) {} // explicitly does nothing!
  array(array<element> &&the_array);
  array(array_data *pad) : data(pad) { array_data::add_ref(data); }

  ~array() { destroy(); }

  array(std::initializer_list<element> list);
  
  void destroy();
  void release_data() { array_data::release(data); }
  void attach_data(const array &from) {
    release_data();
    data = from.data;
    array_data::add_ref(data);
  }
  array_data *detach_data() {
    array_data *p = data;
    data          = 0;
    return p;
  }

  index_t size() const { return (int)length(); }
  size_t  length() const;
  bool    is_empty() const;

  void size(index_t new_size) { length(max(0, new_size)); }
  void length(size_t new_size);
  void length(size_t new_size, const element &init_value);
  void clear();
  void reserve(size_t new_size) {
    size_t old_size = length();
    length(new_size);
    length(old_size);
  }

  array copy() const { return array(operator()()); }

  void set_ref_to(array_data *&ad) const {
    array_data::release(ad);
    ad = data;
    array_data::add_ref(ad);
  }

  array<element> &operator=(const array<element> &the_array);
  array<element> &operator=(const slice<element> &the_range);
  array<element> &operator=(array<element> &&the_array);

  inline const element &operator[](index_t index) const;
  inline element &      operator[](index_t index);

  // inline const element &operator[] ( uint index ) const { return
  // operator[](int(index)); }  inline element &operator[] ( uint index ) {
  // return operator[](int(index)); }

  element remove(index_t index);
  void    remove(index_t index, size_t length);
  index_t insert(index_t index, const element &elem);
  void    insert(index_t index, const element *elems, size_t count);
  void    insert(index_t index, slice<element> elems) {
    insert(index, elems.start, elems.length);
  }
  void insert(index_t index, const element &elem, size_t count);

  index_t                       get_index(const element &e) const;
  template <typename E> index_t get_index(const E &e) const {
    index_t n = size();

    const element *arr = data->elements;
    index_t        i   = 0;
    while (i + 7 < n) {
      if (arr[i + 0] == e)
        return i + 0;
      if (arr[i + 1] == e)
        return i + 1;
      if (arr[i + 2] == e)
        return i + 2;
      if (arr[i + 3] == e)
        return i + 3;
      if (arr[i + 4] == e)
        return i + 4;
      if (arr[i + 5] == e)
        return i + 5;
      if (arr[i + 6] == e)
        return i + 6;
      if (arr[i + 7] == e)
        return i + 7;
      i += 8;
    }
    while (i < n) {
      if (arr[i] == e)
        return i;
      ++i;
    }
    // while( --i >= 0 )
    //  if(data->elements[i] == e) return i;
    return -1;
  }

  bool remove_by_value(const element &el);

  bool operator==(const array<element> &rs) const;
  bool operator!=(const array<element> &rs) const;
  bool operator<(const array<element> &rs) const;
  bool operator<=(const array<element> &rs) const;
  bool operator>(const array<element> &rs) const;
  bool operator>=(const array<element> &rs) const;

  index_t  push(const element &elem);
  void     push(const element *elems, size_t count);
  void     push(slice<element> elems);
  void     push(const element &elem, size_t count);
  element &push();
  element  pop();
  bool     pop(element &el);
  element  pop_front();
  void     drop(int n);

  void reverse();

  bool is_valid_index(index_t i) const { return i >= 0 && i < size(); }

  array<element> &operator+=(const element &e) {
    push(e);
    return *this;
  }
  array<element> &operator+=(slice<element> elems) {
    push(elems);
    return *this;
  }

  element &      first();
  const element &first() const;
  element &      last();

  inline const element &last() const;
  inline const element  last(const element &dv) const;
  inline index_t        last_index() const { return size() - 1; }

  // will create elements buffer if it does not exist
  element *elements() {
    if (!data) {
      size(4);
      size(0);
    }
    return data->elements;
  }

  element *head() const { return data ? data->elements : nullptr; }
  element *tail() const { return data ? data->elements + data->size : nullptr; }
  element *begin() const { return head(); }
  element *end() const { return tail(); }
  const element *cbegin() const { return head(); }
  const element *cend() const { return tail(); }

  operator slice<element>() const { 
    return length() ? slice<element>(head(), size()) : slice<element>();
  }

  slice<element> operator()() const {
    return length() ? slice<element>(head(), size()) : slice<element>();
  }
  slice<element> operator()(index_t s) const {
    slice<element> t = operator()();
    return t(s, size());
  }
  slice<element> operator()(index_t s, index_t e) const {
    slice<element> t = operator()();
    return t(s, e);
  }

  tslice<element> target() { return tslice<element>(begin(), length()); }

  void set(index_t from, index_t to, const element &v) {
    int s = limit(from, 0, size());
    int e = limit(to, 0, size());
    for (; s < e; ++s)
      (*this)[s] = v;
  }

  void set(index_t from, slice<element> by) {
    int s = limit(from, 0, size());
    int e = limit(s + int(by.length), 0, size());
    for (int i = 0; s < e; ++s)
      (*this)[s] = by[i++];
  }

  void set_all_to(const element &element_val);

  // ensure that it exists and get reference
  element &get(index_t idx) {
    if (idx >= size())
      size(idx + 1);
    return operator[](idx);
  };

  void swap(array<element> &the_array) {
    if (this == &the_array)
      return;
    std::swap(data, the_array.data);
  }

  void transfer_from(array<element> &the_array) {
    destroy();
    data           = the_array.data;
    the_array.data = nullptr;
  }

  unsigned int hash() const {
    unsigned int r   = (unsigned int)length();
    element *    p   = head();
    element *    end = tail();
    while (p < end)
      hash_combine(r, tool::hash(*p++));
    return r;
  }

  template <typename F> bool each(F f) {
    data_holder dh(this);
    if (dh.data)
      for (size_t i = 0; i < dh.data->size; ++i) {
        element el = dh.data->elements[i];
        if (f(el))
          return true;
      }
    return false;
  }

  template <typename F, typename AF> bool each(F f, AF af) {
    data_holder dh(this);
    if (dh.data)
      for (size_t i = 0; i < dh.data->size; ++i) {
        element el = dh.data->elements[i];
        if (af(el) && f(el))
          return true;
      }
    return false;
  }

  template <typename F> bool each_backward(F f) {
    data_holder dh(this);
    if (dh.data)
      for (int i = int(dh.data->size) - 1; i >= 0; --i) {
        if (i >= int(dh.data->size)) // that may happen if f() will modify the
                                     // array itself
        {
          i = int(dh.data->size);
          continue;
        }
        element el = dh.data->elements[i];
        if (f(el))
          return true;
      }
    return false;
  }

  // element * _elements;

  /*inline void set_size(size_t new_size)
  {
    assert( _elements );
    *(((size_t*)_elements) - 1) = new_size;
  }
  INLINE size_t get_size() const
  {
    assert( _elements );
    return *(((size_t*)_elements) - 1);
  }
  inline void set_allocated_size(size_t new_size)
  {
    assert( _elements );
    *(((size_t*)_elements) - 2) = new_size;
  }
  inline size_t allocated_size() const
  {
    if(_elements) return *(((size_t*)_elements) - 2);
    return 0;
  }*/
protected:
  static element &black_hole() {
    static element _black_hole;
    return _black_hole;
  }
};

template <typename element> inline array<element>::array() : data(nullptr) {
  ; /* do nothing */
}

template <typename element>
inline array<element>::array(size_t sz) : data(nullptr) {
  length(sz);
}

template <typename element>
inline array<element>::array(size_t sz, const element &init_element)
    : data(nullptr) {
  length(sz, init_element);
}

template <typename element>
inline array<element>::array(size_t sz, std::function<void(int, element &)> f)
    : data(nullptr) {
  length(sz);
  data_holder _(this);
  for (size_t n = 0; n < length(); ++n) {
    f(int(n), operator[](int(n)));
  }
}

template <typename element>
inline array<element>::array(const slice<element> &d) : data(nullptr) {
  length(d.length);
  target().copy(d);
}

template <typename element>
inline array<element>::array(std::initializer_list<element> list) : data(nullptr) {
  *this = items_of(list);
}


template <typename element>
inline array<element>::array(const array<element> &the_array) : data(nullptr) {
  operator=(the_array);
}

template <typename element>
inline array<element>::array(array<element> &&the_array) : data(nullptr) {
  data           = the_array.data;
  the_array.data = 0;
}

template <typename element> inline void array<element>::destroy() {
  array_data::release(data);
}

template <typename element> inline void array<element>::clear() { size(0); }

template <typename element>
inline array<element> &array<element>::
                       operator=(const array<element> &the_array) {
  if (this == &the_array)
    return *this;
  size(the_array.size());

  if (data && the_array.data)
    target().copy(the_array.data->elements, the_array.data->size);

  return *this;
}

template <typename element>
inline array<element> &array<element>::operator=(array<element> &&the_array) {
  if (this == &the_array)
    return *this;
  destroy();
  data           = the_array.data;
  the_array.data = 0;
  return *this;
}

template <typename element>
inline array<element> &array<element>::
                       operator=(const slice<element> &the_range) {
  size(int(the_range.length));
  if (data && the_range.length)
    target().copy(the_range);
  return *this;
}

template <typename element>
inline index_t array<element>::push(const element &elem) {
  // assert( _elements == 0 || &elem < _elements || &elem >= (_elements +
  // size()) );
  index_t tpos = size();
  size(tpos + 1);
  data->elements[tpos] = elem;
  return tpos;
}

template <typename element> inline element &array<element>::push() {
  index_t tpos = size();
  size(tpos + 1);
  return data->elements[tpos];
}

template <typename element> inline element array<element>::pop() {
  assert(size() > 0);
  if (size() > 0) {
    element e = data->elements[size() - 1];
    size(size() - 1);
    return e;
  }
  return element();
}

template <typename element> inline bool array<element>::pop(element &el) {
  assert(size() > 0);
  if (size() > 0) {
    el = data->elements[size() - 1];
    size(size() - 1);
    return true;
  }
  return false;
}

template <typename element> inline element array<element>::pop_front() {
  assert(size() > 0);
  if (size() > 0) {
    element r;
#pragma TODO("investigate better options for this, e.g. ++_elements")
    tool::swap(r, data->elements[0]);
    remove(0);
    return r;
  }
  return element();
}

template <typename element> inline void array<element>::drop(int n) {
  assert(n >= 0 && n <= size());
  n = min(n, size());
  size(size() - n);
}

template <typename element>
inline element array<element>::remove(index_t index) {
  assert(index >= 0 && index < size());
  if (!data)
    return element();
  element *dst = data->elements + index;
  element  r   = *dst;
  // size--;
  data->size = length() - 1;
  if (index < index_t(data->size))
    move(dst, dst + 1, size_t(data->size - index));
  erase(data->elements + data->size, 1);
  return r;
}

template <typename element>
inline void array<element>::remove(index_t index, size_t count) {
  assert(index >= 0 && ((index + count) <= length()));
  if (!data)
    return;

  if (index + int(count) > size()) {
    int n = size() - index;
    if (n <= 0)
      return;
    count = size_t(n);
  }
  element *dst = data->elements + index;
  data->size -= count;
  if (index < size())
    move(dst, dst + count, data->size - index);
  erase(data->elements + data->size, count);
}

template <typename element>
inline bool array<element>::remove_by_value(const element &el) {
  index_t idx = get_index(el);
  if (idx < 0)
    return false;
  (void)remove(idx);
  return true;
}

template <typename element>
inline index_t array<element>::insert(index_t index, const element &elem) {
  if (index < 0)
    index = 0;
  if (index >= size()) {
    push(elem);
    return size() - 1;
  } else {
    length(length() + 1);

    move(data->elements + index + 1, data->elements + index,
         data->size - index - 1);

    data->elements[index] = elem;
    return index;
  }
}

template <typename element>
inline void array<element>::insert(index_t index, const element *elems,
                                   size_t count) {
  assert(count > 0);
  if (count <= 0)
    return;
  // assert( _elements == 0 || elems < _elements || elems >= (_elements +
  // size()) );
  if (index < 0)
    index = 0;
  if (index >= size())
    push(elems, count);
  else {
    size_t _old_size = length();
    length(length() + count);
    element *dst = data->elements + data->size - 1;
    element *src = data->elements + _old_size - 1;
    for (size_t i = 0; i < _old_size - index; i++)
      *dst-- = *src--;
    // move<element>(_elements + index + count,_elements + index,size() - index
    // - count);

    element *p = data->elements + index;
    for (size_t j = 0; j < count; j++)
      *p++ = *elems++;
  }
}

template <typename element>
inline void array<element>::insert(index_t index, const element &elem,
                                   size_t count) {
  // assert(count > 0);
  if (count <= 0)
    return;
  if (index < 0)
    index = 0;
  if (index >= size())
    push(elem, count);
  else {
    int _old_size = size();
    length(length() + count);
    element *dst = data->elements + length() - 1;
    element *src = data->elements + _old_size - 1;
    for (int i = 0; i < _old_size - index; i++)
      *dst-- = *src--;
    // move<element>(_elements + index + count,_elements + index,size() - index
    // - count);

    element *p = data->elements + index;
    for (size_t j = 0; j < count; j++)
      *p++ = elem;
  }
}

template <typename element>
inline element &array<element>::operator[](index_t index) {
  index_t sz = size();
  assert(index >= 0 && index < sz);
  if (index >= 0 && index < sz)
    return data->elements[index];
  else
    return black_hole();
}

template <typename element>
inline const element &array<element>::operator[](index_t index) const {
  assert(index >= 0 && index < size());
  if (index >= 0 && index < size())
    return data->elements[index];
  else
    return black_hole();
}

template <typename element>
inline const element array<element>::last(const element &dv) const {
  if (size() > 0)
    return data->elements[data->size - 1];
  return dv;
}

template <typename element> inline element &array<element>::last() {
  assert(size() > 0);
  if (size() > 0)
    return data->elements[data->size - 1];
  return black_hole();
}

template <typename element> inline const element &array<element>::last() const {
  assert(size() > 0);
  if (size() > 0)
    return data->elements[data->size - 1];
  return black_hole();
}

template <typename element> inline element &array<element>::first() {
  assert(size() > 0);
  if (size() > 0)
    return data->elements[0];
  return black_hole();
}

template <typename element>
inline const element &array<element>::first() const {
  if (size() > 0)
    return data->elements[0];
  else
    assert(false);
  return black_hole();
}

template <typename element> INLINE size_t array<element>::length() const {
  return data ? data->size : 0;
}

template <typename element> inline bool array<element>::is_empty() const {
  return (length() == 0);
}

template <typename element>
inline void array<element>::length(size_t new_size) {
  size_t old_size = length();

  if (old_size == new_size)
    return;

  if (new_size > old_size) {
    size_t allocated = array_data::get_allocated_size(data);
    if (new_size > allocated) {
      size_t toallocate = allocated ? ((allocated * 3) / 2) : max(4, new_size);

      if (toallocate < new_size)
        toallocate = new_size;

      array_data *d = array_data::alloc(toallocate);
      assert(d);
      if (!d)
        return; // Hmm... throw error? But who will handle it?

      // element *new_space = (element *) ( d + 2);

      init(d->elements, new_size);
      d->size = new_size;

      if (data) {
        transfer(d->elements, d->size, data->elements, old_size);
        // erase(data->elements,old_size);
        array_data::release(data);
      }
      data = d;

      return;

    } else // if ( new_size <= allocated ), simply init the tail
      init(data->elements + old_size, new_size - old_size);
  } else if (data) // and yet new_size < old_size
    erase(data->elements + new_size, old_size - new_size);

  if (data)
    data->size = new_size;
}

/* Something wrong with realloc here:
template <typename element>
inline void
  array<element>::size ( int new_size )
{
  assert(new_size >= 0);
  new_size = max(0,new_size);
  int old_size = size();

  if ( old_size == new_size )
    return;

  if ( new_size > old_size )
  {
    int allocated = allocated_size();
    if ( new_size > allocated )
    {
      int toallocate = allocated?
          ((allocated * 3) / 2) :
          max(4,new_size);

      if( toallocate < new_size ) toallocate = new_size;

      int *d = 0;
      int* old_d = 0;

      if(_elements)
      {
         old_d = (((int *)_elements) - 2);
         d = (int *)realloc( old_d, sizeof(int) * 2 + toallocate *
sizeof(element)); if( d != old_d ) // reallocated inplace _elements = (element
*) ( d + 2 ); init(_elements + old_size, new_size - old_size);
      }
      else
      {
         d = (int *)malloc(sizeof(int) * 2 + toallocate * sizeof(element));
         _elements = (element *) ( d + 2 );
         init(_elements,new_size);
      }
      set_size(new_size);
      set_allocated_size(toallocate);

    }
    else //if ( new_size <= allocated ), simply init the tail
      init(_elements + old_size, new_size - old_size);
  }
  else if( _elements ) // and yet new_size < old_size
    erase(_elements + new_size, old_size - new_size);

  set_size(new_size);
}

*/

template <typename element>
inline void array<element>::length(size_t new_size, const element &init_value) {
  assert(new_size >= 0);
  size_t oldlen = length();
  length(new_size);
  for (uint_ptr i = oldlen; i < new_size; i++)
    data->elements[i] = init_value;
}

template <typename element>
inline void array<element>::set_all_to(const element &the_element) {
  for (int i = 0; i < size(); i++)
    data->elements[i] = the_element;
}

template <typename element>
inline index_t array<element>::get_index(const element &e) const {
  for (int i = 0; i < size(); i++)
    if (data->elements[i] == e)
      return i;
  return -1;
}

template <typename element>
inline void array<element>::push(const element *elems, size_t count) {
  size_t psz = length();
  length(psz + count);
  // element* pdst = head() + psz;
  // copy(pdst,elems,count);
  target().copy(psz, elems, count);
}

template <typename element>
inline void array<element>::push(const element &elem, size_t count) {
  size_t psz = length();
  length(psz + count);
  element *pdst = head() + psz;
  for (size_t i = 0; i < count; i++)
    pdst[i] = elem;
}

template <typename element>
inline bool array<element>::operator==(const array<element> &rs) const {
  if (size() != rs.size())
    return false;
  for (index_t i = size() - 1; i >= 0; --i)
    if (!(data->elements[i] == rs.data->elements[i]))
      return false;
  return true;
}

template <typename element>
inline bool array<element>::operator!=(const array<element> &rs) const {
  if (size() != rs.size())
    return true;
  for (index_t i = 0; i < size(); i++)
    if (data->elements[i] != rs.data->elements[i])
      return true;
  return false;
}

template <typename element>
inline bool array<element>::operator>(const array<element> &rs) const {
  assert(size() && rs.size());

  index_t mi = min(size(), rs.size());
  for (index_t i = 0; i < mi; i++) {
    element el = data->elements[i];
    element er = rs.data->elements[i];
    if (el < er)
      return false;
    if (el > er)
      return true;
  }
  // equal so far
  if (rs.size() > size())
    return rs.data->elements[mi] < 0;
  else if (rs.size() < size())
    return data->elements[mi] >= 0;

  return false;

  /*if(size() != rs.size())
    return _elements[mi] >= rs._elements[mi];
  else
    return _elements[mi] > rs._elements[mi];*/
}

template <typename element>
inline bool array<element>::operator>=(const array<element> &rs) const {
  assert(size() && rs.size());
  index_t mi = min(size(), rs.size());
  for (index_t i = 0; i < mi; i++) {
    element el = data->elements[i];
    element er = rs.data->elements[i];
    if (el < er)
      return false;
    if (el > er)
      return true;
  }
  // equal so far
  if (rs.size() > size())
    return rs.data->elements[mi] < 0;
  else if (rs.size() < size())
    return data->elements[mi] >= 0;

  return true;
}

template <typename element>
inline bool array<element>::operator<(const array<element> &rs) const {
  return !operator>=(rs);
}

template <typename element>
inline bool array<element>::operator<=(const array<element> &rs) const {
  return !operator>(rs);
}

template <typename element> inline void array<element>::reverse() {
  index_t i = 0;
  index_t k = size() - 1;
  while (i < k) {
    swop(data->elements[i], data->elements[k]);
    ++i;
    --k;
  }
}

template <typename element>
inline void array<element>::push(slice<element> elems) {
  push(elems.start, int(elems.length));
}

#define FOREACH(I, A)                                                          \
  for (index_t I = A.size() - 1; I >= 0 && I < A.size(); --I)

//#define ceach(S,pel) for( auto pel = S.cbegin(), auto end = S.cend(); pel <
// end; ++pel ) #define each(S,pel) for( auto pel = S.begin(), auto end =
// S.end(); pel < end; ++pel )

template <typename T, int CAPACITY = 256> class circular_buffer : private array<T> {
private:
  typedef array<T> super;

public:
  // ctor
  circular_buffer(size_t capacity = CAPACITY, const T &def = T())
      : _size(0), _full(false) {
    super::length(capacity, def);
    _begin = _end = super::head();
  }

  ~circular_buffer() {}

  // get the buffer capacity (maximim size)
  size_t capacity() const { return super::length(); }

  // get number of items stored
  int    size() const { return (int)_size; }
  size_t length() const { return _size; }

  // is the buffer empty?
  bool is_empty() const { return !_full && _begin == _end; }

  // is the buffer full?
  bool full() const { return _full; }

  // push an item to the front of the buffer, \param x is item to push to the
  // front.
  // because the buffer is circular, the last item will be overwritten if the
  // buffer is already full.
  void push_front(const T &x) {
    if (super::size()) {
      decrement(_begin);
      *_begin = x;
      if (_full)
        decrement(_end);
      else {
        if (++_size == super::length())
          _full = true;
      }
    }
  }

  // push an item to the back of the buffer, \param x is item to push to the
  // back. because the buffer is circular, the first item will be overwritten if
  // the buffer is already full.
  void push(const T &x) {
    if (super::size()) {
      *_end = x;
      increment(_end);
      if (_full)
        increment(_begin);
      else {
        if (++_size == super::length())
          _full = true;
      }
    }
  }

  // pop item from the front of the buffer, returns the item
  T pop_front() {
    assert(super::size());
    assert(_size > 0);
    if (_size > 0) {
      T t;
      tool::swap(t, *_begin);
      increment(_begin);
      _size--;
      _full = false;
      return t;
    }
    return T();
  }

  // pop item from the back of the buffer, returns the item
  T pop() {
    assert(super::size());
    assert(_size > 0);
    if (_size > 0) {
      T t;
      tool::swap(t, *_end);
      decrement(_end);
      _size--;
      _full = false;
      return t;
    }
    return T();
  }

  // access the front item
  T &first() {
    assert(super::size());
    assert(_size > 0);
    return *_begin;
  }

  // access the front item
  const T &first() const {
    assert(super::size());
    assert(_size > 0);
    return *_begin;
  }

  // access the back item
  T &last() {
    assert(super::size());
    assert(_size > 0);
    T *pe = _end;
    decrement(pe);
    return *pe;
  }

  // access the back item
  const T &last() const {
    assert(super::size());
    assert(_size > 0);
    T *pe = _end;
    decrement(pe);
    return *pe;
  }

  // access the i'th item in the buffer
  T &operator[](index_t i) {
    if (_begin < _end)
      return *(_begin + i);
    else {
      const size_t s1 = super::head() + super::length() - _begin;
      if (i < index_t(s1))
        return *(_begin + i);
      else
        return *(super::head() + i - s1);
    }
  }

  // access the i'th item in the buffer
  const T &operator[](index_t i) const {
    if (_begin < _end)
      return *(_begin + i);
    else {
      const size_t s1 = super::head() + super::length() - _begin;
      if (i < index_t(s1))
        return *(_begin + i);
      else
        return *(super::head() + i - s1);
    }
  }

  // shift the contents left
  void left_shift() {
    decrement(_begin);
    decrement(_end);
  }

  // shift the contents right
  void right_shift() {
    decrement(_begin);
    decrement(_end);
  }

  // clear the buffer
  void clear() {
    //tool::erase(super::head(), super::length());
    size_t cap = capacity();
    super::length(0);
    super::length(cap);
    _begin = _end = super::head();
    _size         = 0;
    _full         = false;
  }

  // swap the contents of this buffer with another
  void swap(circular_buffer<T> &buffer) {
    super::swap(buffer);
    tool::swap(_begin, buffer._begin);
    tool::swap(_end, buffer._end);
    tool::swap(_size, buffer._size);
    tool::swap(_full, buffer._full);
  }

  // resize the (capacity) of the buffer, \param capacity new capacity.
  // if the new buffer capacity is too small for the existing contents, only the
  // front-most items will remain.
  void size(const int cap) {
    if (cap == capacity())
      return;

    circular_buffer<T> tmp(cap);
    swap(tmp);

    T * i = tmp._begin;
    T * j = _begin;
    int k = 0;

    for (; k < cap && k < capacity() && k < tmp.size(); j++, k++) {
      *j = *i;
      tmp.increment(i);
    }
    _end  = j;
    _size = k;
    _full = (_size == cap);
  }

private:
  T *    _begin;
  T *    _end;
  size_t _size;
  bool   _full;

  // increment an item ptr, checking for wrap-around
  void increment(const T *&i) const {
    i++;
    if (i == super::tail())
      i = super::head();
  }

  // decrement an item ptr, checking for wrap-around
  void decrement(const T *&i) const {
    if (i == super::head())
      i += super::size() - 1;
    else
      i--;
  }

  /// Increment an iterator, checking for wrap-around
  void increment(T *&i) const {
    i++;
    if (i == super::tail())
      i = super::head();
  }

  /// Decrement an iterator, checking for wrap-around
  void decrement(T *&i) const {
    if (i == super::head())
      i += super::size() - 1;
    else
      i--;
  }
};

// buffer that is created either on stack or in the heap

template <typename T, unsigned N> class buffer {
  T        data[N];
  array<T> dyn_data;
  slice<T> vdata;

public:
  buffer(size_t n) { reset(n); }
  buffer(const buffer &src) {
    reset(src.length());
    target().copy(src.cbegin(), src.length());
  }
  buffer(slice<T> src) {
    reset(src.length);
    target().copy(src.start, src.length);
  }
  buffer &operator=(const buffer &src) {
    reset(src.length());
    target().copy(src.cbegin(), src.length());
    return *this;
  }
  buffer &operator=(slice<T> src) {
    reset(src.length);
    target().copy(src);
    return *this;
  }

  void reset(size_t n) {
    if (n > N) {
      dyn_data.length(n);
      vdata = dyn_data();
    } else {
      dyn_data.length(0);
      vdata.start  = data;
      vdata.length = n;
    }
  }
  const T *cbegin() const { return vdata.start; }
  const T *cend() const { return vdata.end(); }
  T *      begin() const { return const_cast<T *>(vdata.start); }
  const T *end() const { return vdata.end(); }
  size_t   length() const { return vdata.length; }
  int      size() const { return vdata.size(); }
  slice<T> operator()() { return vdata; }

  tslice<T> target() { return tslice<T>(begin(), length()); }

  T &      operator[](index_t i) { return const_cast<T *>(vdata.start)[i]; }
  const T &operator[](index_t i) const { return vdata.start[i]; }
};

} // namespace tool

#endif
