#include "config.h"

#include "tool.h"

#include "tl_spell_checker.h"

#ifdef SPELL_CHECK_SUPPORT
#include "engine/external/hunspell/hunspell.hxx"
#endif

namespace tool {

#ifdef SPELL_CHECK_SUPPORT

struct spell_checker_impl : public spell_checker {
  Hunspell spell;

  spell_checker_impl(const string& path_aff, const string& path_dic)
     : spell(path_aff.c_str(), path_dic.c_str()) {}

  virtual bool check_spelling(const ustring & text,
                              spell_check_callback_t cb) override {

    bool spell_ok = true;

    wchars chars = text();

    //tool::tokens<wchar> ts(chars,WCHARS(" \u00A0:;!@#$%^&*()_+=-\"'|\\/?<>,.`~\t"));
    tool::spans<wchar> ts(chars, [](wchar c)-> bool { return ucisalpha(c); } );

    wchars word_chars;
    while(ts.next(word_chars)) {
      string u8word = u8::cvt(word_chars);
      std::string word(u8word.c_str(),u8word.length());
      if(!spell.spell(word))
        cb(SPELL_CORRECTIVE_ACTION_NONE,uint(word_chars.start-chars.start),uint(word_chars.length),ustring());
    }
    return spell_ok;
  }

  virtual array<ustring> get_suggestions_for(const ustring &text) {
    array<ustring> list;

    string u8word = u8::cvt(text);

    std::string word(u8word.c_str(),u8word.length());

    std::vector<std::string> suggestions = spell.suggest(word);

    for(auto s: suggestions )
      list.push(u8::cvt(chars_of(s)));

    return list;
  }
};

struct spell_checker_factory_impl : public spell_checker_factory
{

  virtual spell_checker * create(const ustring &lang) override {

    string alang = u8::cvt(lang);
    chars  alang_chars = alang();
    chars  language, country;
    alang_chars.split('-',language,country);

    string slanguage = language;
    string scountry = country;
    to_upper(scountry.target());

    string path_aff, path_dic;

    path_aff = string::format("/usr/share/hunspell/%s_%s.aff",slanguage.c_str(),scountry.c_str());
    path_dic = string::format("/usr/share/hunspell/%s_%s.dic",slanguage.c_str(),scountry.c_str());

    if(tool::filesystem::is_file(path_aff.c_str())) {
      //Hunspell spell()
      return new spell_checker_impl(path_aff,path_dic);
    } else {

      ustring path_mask = ustring::format(W("/usr/share/hunspell/%S_*.*"),slanguage.c_str());

      path_aff.clear();
      path_dic.clear();

      tool::filesystem::scan(path_mask,[&](wchars name,wchars caption,uint flags)->bool {
        if( name.like(W("*.aff")) )
          path_aff = CHARS("/usr/share/hunspell/") + string(name);
        if( name.like(W("*.dic")) )
          path_dic = CHARS("/usr/share/hunspell/") + string(name);
        return true;
      });

      if( path_aff.length() && path_dic.length() ) {
        return new spell_checker_impl(path_aff,path_dic);
      }
    }
    return nullptr;
  }
};

spell_checker_factory *spell_checker_factory::create_factory() {

  spell_checker_factory_impl *pimpl = new spell_checker_factory_impl;
  return pimpl;
}
#else
spell_checker_factory *spell_checker_factory::create_factory() {
  return nullptr;
}

#endif
}
