#include "tool.h"
#include "../snprintf.h"
#include "tl_eval.h"
#include <math.h>

namespace tool {
namespace eval {
static pool<ustring> string_pool;

enum KNOWN_SYMBOLS {
  sym_length = 1,
  sym_toUpper,
  sym_toLower,
  sym_int,
  sym_float,
  sym_min,
  sym_max,
  sym_limit,
  sym_substr,
  sym_rgb,
  sym_rgba,
  sym_pixels,
  sym_compare,
  sym_toUrl,
};

void init_symbols() {
  if (string_pool.size() != 0)
    return;
  string_pool[WCHARS("unknown")];
  string_pool[WCHARS("length")];
  string_pool[WCHARS("toUpper")];
  string_pool[WCHARS("toLower")];
  string_pool[WCHARS("int")];
  string_pool[WCHARS("float")];
  string_pool[WCHARS("min")];
  string_pool[WCHARS("max")];
  string_pool[WCHARS("limit")];
  string_pool[WCHARS("substr")];
  string_pool[WCHARS("rgb")];
  string_pool[WCHARS("rgba")];
  string_pool[WCHARS("pixels")];
  string_pool[WCHARS("compare")];
  string_pool[WCHARS("toUrl")];
}

uint symbol_id(wchars name) {
  init_symbols();
  return (uint)string_pool[name];
}

ustring symbol_name(uint id) {
  init_symbols();
  return string_pool(id);
}

bool set_string_attribute(uint /*attr_sym*/, const value & /*obj*/,
                          const value & /*v*/) {
  return false;
}
bool get_string_attribute(uint attr_sym, const value &obj, value &v) {
  switch (attr_sym) {
  case sym_length:
    v = value((int)obj.size());
    return true;
  }
  return false;
}
bool call_string_function(uint attr_sym, const value &obj, uint argn,
                          const value *argv, value &retval) {
  switch (attr_sym) {
  case sym_toUpper: {
    ustring us = obj.to_string();
    retval     = value(us.to_upper());
    return true;
  }
  case sym_toLower: {
    ustring us = obj.to_string();
    retval     = value(us.to_lower());
    return true;
  }
  case sym_toUrl: {
    ustring us = obj.to_string();
    retval     = value::make_url(us);
    return true;
  }
  case sym_substr: {
    ustring us    = obj.to_string();
    int     index = 0;
    if (argn >= 1)
      index = argv[0].get(0);
    int len = -1;
    if (argn >= 2)
      len = argv[1].get(-1);
    retval = value(us().sub(index, len));
    return true;
  }
  case sym_compare: {
    ustring us = obj.to_string();
    ustring us_other;
    if (argn >= 1 && argv[0].is_string())
      us_other = argv[0].to_string();
    else
      return false;
    // retval = value(us.to_lower());
    bool case_insensitive = false;
    if (argn >= 2 && argv[1].is_bool())
      case_insensitive = argv[1].to_bool();
    if (case_insensitive)
      retval = lexical::ci::cmp(us(), us_other());
    else
      retval = lexical::cs::cmp(us(), us_other());
    return true;
  }
  }
  return false;
}

bool is_less(const value &a1, const value &a2) {
  if (a1.is_double() || a2.is_double())
    return (a1.to_float() < a2.to_float());
  if (a1.is_int() || a2.is_int())
    return (a1.to_int() < a2.to_int());
  if (a1.is_string() || a2.is_string())
    return (a1.to_string() < a2.to_string());
  return (a1.to_int() < a2.to_int());
}

bool call_intrinsic_function(vm &vm, uint attr_sym, const value & /*obj*/,
                             uint argn, const value *argv, value &retval) {
  switch (attr_sym) {
  case sym_pixels:
    if (argn >= 1) {
      int px;
      if (vm.to_pixels(*argv, px))
        retval = value(px);
      else
        retval = value(0);
    } else
      retval = value(0);
    return true;
  case sym_int: {
    if (argn >= 1) {
      int px;
      if (argv->is_length() && vm.to_pixels(*argv, px))
        retval = value(px);
      else
        retval = value(argv->to_int());
    } else
      retval = value(0);
    return true;
  }
  case sym_float: {
    if (argn >= 1)
      retval = value(argv->to_float());
    else
      retval = value(0.0);
    return true;
  }
  case sym_length: {
    if (argn >= 1) {
      if (argv->is_string())
        retval = value::parse_numeric_units(argv->get(L""));
      else
        retval = value::make_ppx_length(argv->to_int());
    } else
      retval = value::make_ppx_length(0);
    return true;
  }

  case sym_min:
    if (argn >= 1) {
      value mv = argv[0];
      for (uint i = 1; i < argn; ++i)
        if (is_less(argv[i], mv))
          mv = argv[i];
      retval = mv;
    } else
      retval = value();
    return true;
  case sym_max:
    if (argn >= 1) {
      value mv = argv[0];
      for (uint i = 1; i < argn; ++i)
        if (is_less(mv, argv[i]))
          mv = argv[i];
      retval = mv;
    } else
      retval = value();
    return true;
  case sym_limit:
    if (argn == 2) {
      if (is_less(argv[0], argv[1]))
        retval = argv[1];
      else
        retval = argv[0];
    } else if (argn == 3) {
      retval = argv[0];
      if (is_less(argv[2], retval))
        retval = argv[2];
      if (is_less(retval, argv[1]))
        retval = argv[1];
    } else {
      vm.raise_error(ERR_WRONG_ARGC);
      // retval = value();
    }
    return true;
  case sym_rgb: {
    if (argn != 3)
      vm.raise_error(ERR_WRONG_ARGC);
    int rgb = ((argv[2].to_int() & 0xff) << 16) |
              ((argv[1].to_int() & 0xff) << 8) | (argv[0].to_int() & 0xff);
    retval = value::make_packed_color(uint(rgb));
    return true;
  }
  case sym_rgba: {
    if (argn != 4)
      vm.raise_error(ERR_WRONG_ARGC);
    int rgba = ((argv[2].to_int() & 0xff) << 16) |
               ((argv[1].to_int() & 0xff) << 8) | (argv[0].to_int() & 0xff);
    int alfa = 255 - int(argv[3].to_float() * 255);
    rgba |= alfa << 24;
    retval = value::make_packed_color(uint(rgba));
    return true;
  }
  }
  return false;
}

// in CSS: 1px == 1/96in

bool vm::to_pixels(const value &v, int &retv) {
  if (v.is_int()) {
    retv = v.get_int();
    return true;
  }
  if (v.is_double()) {
    retv = int(v.get_double());
    return true;
  }
  if (v.is_length())
    return globals->to_pixels(v, retv);

  retv = 0;
  return true;
}

tool::value vm::add(const tool::value &a1, const tool::value &a2) {
  if (a1.is_string() || a2.is_string())
    return tool::value(a1.to_string() + a2.to_string());
  if (a1.is_double() || a2.is_double())
    return tool::value(a1.to_float() + a2.to_float());
  if (a1.is_length() || a2.is_length()) {
    //if (a1.is_spring() && a2.is_spring())
    //  return tool::value::make_spring_length(a1._int() + a2._int());
    //if (a1.is_percent() && a2.is_percent())
    //  return tool::value::make_percent_length(a1._int() + a2._int());
    int p1 = 0, p2 = 0;
    if (!to_pixels(a1, p1) || !to_pixels(a2, p2))
      raise_error(ERR_LENGTH_CONVERSION);
    return tool::value::make_ppx_length(p1 + p2);
  }
  if (a1.is_int() || a2.is_int())
    return tool::value(a1.to_int() + a2.to_int());
  return tool::value(a1.to_string() + a2.to_string());
}

tool::value vm::cmp(uint code, const tool::value &a1, const tool::value &a2) {
  if (a1.is_undefined() || a2.is_undefined())
    return tool::value();
  switch (code) {
  case BC_LT:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() < a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() < a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() < a2.to_string());
    return tool::value(a1.to_int() < a2.to_int());
  case BC_LE:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() <= a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() <= a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() <= a2.to_string());
    return tool::value(a1.to_int() <= a2.to_int());
  case BC_GT:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() > a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() > a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() > a2.to_string());
    return tool::value(a1.to_int() > a2.to_int());
  case BC_GE:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() >= a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() >= a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() >= a2.to_string());
    return tool::value(a1.to_int() >= a2.to_int());
  case BC_EQ:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() == a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() == a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() == a2.to_string());
    return tool::value(a1 == a2);
  case BC_NE:
    if (a1.is_double() || a2.is_double())
      return tool::value(a1.to_float() != a2.to_float());
    if (a1.is_int() || a2.is_int())
      return tool::value(a1.to_int() != a2.to_int());
    if (a1.is_string() || a2.is_string())
      return tool::value(a1.to_string() != a2.to_string());
    return tool::value(a1 != a2);
  }
  return value();
}

tool::value vm::sub(const tool::value &a1, const tool::value &a2) {
  if (a1.is_length() || a2.is_length()) {
    //if (a1.is_spring() && a2.is_spring())
    //  return tool::value::make_spring_length(a1._int() - a2._int());
    //if (a1.is_percent() && a2.is_percent())
    //  return tool::value::make_percent_length(a1._int() - a2._int());
    int p1 = 0, p2 = 0;
    if (!to_pixels(a1, p1) || !to_pixels(a2, p2))
      raise_error(ERR_LENGTH_CONVERSION);
    return tool::value::make_ppx_length(p1 - p2);
  }
  if (a1.is_double() || a2.is_double())
    return tool::value(a1.to_float() - a2.to_float());
  if (a1.is_int() || a2.is_int())
    return tool::value(a1.to_int() - a2.to_int());
  raise_error(ERR_IS_NOT_NUMBER);
  return tool::value();
}

int rounded_int(double d) { return fmod(d, 1.0) >= 0.5 ? int(d) + 1 : int(d); }

tool::value vm::mul(const tool::value &a1, const tool::value &a2) {
  if (a1.is_length() || a2.is_length()) {
    /*if (a1.is_percent()) {
      value vp(real(a1._int()) / real(100));
      return mul(vp, a2);
    } else if (a2.is_percent()) {
      value vp(real(a2._int()) / real(100));
      return mul(a1, vp);
    }
    if (a1.is_spring()) {
      if (a2.is_int())
        return tool::value::make_spring_length(a1._int() * a2._int());
      if (a2.is_double())
        return tool::value::make_percent_length(
            rounded_int(a1._int() * a2.get_double()));
    }*/
    int p1 = 0, p2 = 0;
    if (!to_pixels(a1, p1) || !to_pixels(a2, p2))
      raise_error(ERR_LENGTH_CONVERSION);
    return tool::value::make_ppx_length(p1 * p2);
  }
  if (a1.is_double() || a2.is_double())
    return tool::value(a1.to_float() * a2.to_float());
  if (a1.is_int() || a2.is_int())
    return tool::value(a1.to_int() * a2.to_int());
  raise_error(ERR_IS_NOT_NUMBER);
  return tool::value();
}
tool::value vm::div(const tool::value &a1, const tool::value &a2) {
  if (a1.is_length()) {
    /*if (a1.is_spring()) {
      if (a2.is_int() && (a2._int() != 0))
        return tool::value::make_spring_length(a1._int() / a2._int());
      if (a2.is_double() && (a2.get_double() != 0.0))
        return tool::value::make_spring_length(
            int(a1._int() / a2.get_double()));
    }*/
    int p1;
    if (!to_pixels(a1, p1))
      raise_error(ERR_LENGTH_CONVERSION);
    double t = a2.to_float();
    if (t == 0.0)
      return tool::value();
    return tool::value::make_ppx_length(p1 / t);
  }
  if (a1.is_double() || a2.is_double()) {
    double t = a2.to_float();
    if (t == 0.0)
      return tool::value();
    return tool::value(a1.to_float() / t);
  }
  if (a1.is_int() || a2.is_int()) {
    int t = a2.to_int();
    if (t == 0)
      return tool::value();
    return tool::value(a1.to_int() / t);
  }
  raise_error(ERR_IS_NOT_NUMBER);
  return tool::value();
}
tool::value vm::mod(const tool::value &a1, const tool::value &a2) {
  if (a1.is_length()) {
    int p1;
    if (!to_pixels(a1, p1))
      raise_error(ERR_LENGTH_CONVERSION);
    int t = a2.to_int();
    if (t == 0)
      return tool::value();
    return tool::value::make_ppx_length(int(p1 % t));
  }
  if (a1.is_double() || a2.is_double()) {
    double t = a2.to_float();
    if (t == 0.0)
      return tool::value();
    return tool::value(fmod(a1.to_float(), t));
  }
  if (a1.is_int() || a2.is_int()) {
    int t = a2.to_int();
    if (t == 0)
      return tool::value();
    return tool::value(a1.to_int() % t);
  }
  raise_error(ERR_IS_NOT_NUMBER);
  return tool::value();
}

tool::value vm::power(const tool::value &a1, const tool::value &a2) {
  if (a1.is_double() || a2.is_double())
    return tool::value(pow(a1.to_float(), a2.to_float()));
  if (a1.is_int() || a2.is_int())
    return tool::value(int(pow(a1.to_float(), a2.to_float())));
  raise_error(ERR_IS_NOT_NUMBER);
  return tool::value();
}
tool::value vm::neg(const tool::value &a1) {
  if (a1.is_double())
    return tool::value(-a1.get_double());
  if (a1.is_int())
    return tool::value(-a1.get_int());
  // raise_error( ERR_IS_NOT_NUMBER );
  return tool::value();
}
tool::value vm::bnot(const tool::value &a1) {
  if (a1.is_bool())
    return tool::value(a1.get_bool() == false);
  if (a1.is_double())
    return tool::value(a1.get_double() == 0.0);
  if (a1.is_int())
    return tool::value(a1.get_int() == 0);
  if (a1.is_string())
    return tool::value(a1.get_string().length() == 0);
  return tool::value(a1.to_bool() == false);
}

bool vm::to_bool(const tool::value &a1) {
  if (a1.is_undefined())
    return false;
  if (a1.is_double())
    return a1.get_double() != 0.0;
  if (a1.is_int())
    return a1.get_int() != 0;
  if (a1.is_bool())
    return a1.get_bool();
  if (a1.is_object()) {
    object *obj = a1.get_object();
    return obj && obj->to_bool();
  }
  if (a1.is_null())
    return false;
  return true;
}

inline uint get_uint(byte *&bc) {
  uint  t;
  byte *b = (byte *)&t;
  b[0]    = *bc++;
  b[1]    = *bc++;
  b[2]    = *bc++;
  b[3]    = *bc++;
  return t;
}
inline uint read_uint(byte *bc) {
  uint  t;
  byte *b = (byte *)&t;
  b[0]    = *bc++;
  b[1]    = *bc++;
  b[2]    = *bc++;
  b[3]    = *bc++;
  return t;
}

bool vm::eval() {
  stack_clear();
  val.clear();
  if (prg->codes.size() == 0)
    return false;

  byte *bc = prg->codes.head();

  closure *clos  = new closure();
  clos->func_off = 0;
  val.set_resource(clos, 0xf000);
  argv = 0;
  argn = 0;
  next = prg->codes.tail();

  return _eval(bc);
}

bool vm::_eval(byte *bc) {
  index_t call_level = calls.size();

  byte *start = prg->codes.head();
  byte *end   = prg->codes.tail();
  uint  off, n;
  byte  code;
  // debug_printf(".....................starting!");
  while (bc < end) {
    code = *bc++;
    // debug_printf("code=%d",code);
    switch (code) {
    case BC_LINENO:
      line_no = get_uint(bc);
      break;

    case BC_FRAME: {
      // uint func_off = (bc - 1) - start;
      n          = get_uint(bc);
      frame *pf  = new frame();
      pf->bc_off = uint(next - start);
      pf->clos   = val; // it is a closure
      assert(is_func_ref(pf->clos));
      calls.push(pf);
      if (n) {
        pf->vars = value::make_array(n);
        assert((argn - 1) <= int(pf->vars.size()));
        for (int i = 1; i < argn; ++i)
          pf->vars.set_element(i - 1, argv[i]); // first arg is 'this', skip it
      }
      // else
      //  vars = value();
      _stack.drop(argn);
    } break;
    case BC_RETURN: {
      handle<frame> f = calls.pop();
      if (f) {
        bc = start + f->bc_off;
        // vars = f->vars;
      }
      if (calls.size() == call_level) {
        // assert(stack_size() == 0);
        return true;
      }
    } break;
    case BC_CLOSE:
      n   = get_uint(bc);
      val = make_func_ref(n);
      break;
    case BC_PUSH:
      stack_push(val);
      break;
    case BC_POP:
      stack_pop(val);
      break;
    case BC_ATTR_GET: {
      value obj;
      obj.swap(val);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_ATTR_ON_NULL, symbol_name(sym).c_str());
      if (!get_attribute(sym, obj, val))
        raise_error(ERR_ATTR_UNKNOWN, symbol_name(sym).c_str());
    } break;
    case BC_ATTR_SET: {
      value obj;
      stack_pop(obj);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_ATTR_ON_NULL, symbol_name(sym).c_str());
      if (!set_attribute(sym, obj, val))
        raise_error(ERR_ATTR_IS_READ_ONLY, symbol_name(sym).c_str());
    } break;

    case BC_SATTR_GET: {
      value obj;
      obj.swap(val);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_SATTR_ON_NULL, symbol_name(sym).c_str());
      if (!get_style_attribute(sym, obj, val))
        raise_error(ERR_ATTR_UNKNOWN, symbol_name(sym).c_str());
    } break;
    case BC_SATTR_SET: {
      value obj;
      stack_pop(obj);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_SATTR_ON_NULL, symbol_name(sym).c_str());
      if (!set_style_attribute(sym, obj, val))
        raise_error(ERR_ATTR_IS_READ_ONLY, symbol_name(sym).c_str());
    } break;

    case BC_CONST_GET: {
      value obj;
      obj.swap(val);
      uint sym = get_uint(bc);
      if (!get_const(sym, val))
        raise_error(ERR_CONST_UNKNOWN, symbol_name(sym).c_str());
    } break;

    case BC_STATE_GET: {
      value obj;
      obj.swap(val);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_STATE_ON_NULL, symbol_name(sym).c_str());
      if (!get_state(sym, obj, val))
        raise_error(ERR_STATE_UNKNOWN, symbol_name(sym).c_str());
    } break;
    case BC_STATE_SET: {
      value obj;
      stack_pop(obj);
      uint sym = get_uint(bc);
      if (obj.is_undefined() || obj.is_null())
        raise_error(ERR_STATE_ON_NULL, symbol_name(sym).c_str());
      if (!set_state(sym, obj, val))
        raise_error(ERR_STATE_IS_READ_ONLY, symbol_name(sym).c_str());
    } break;

    case BC_ADD:
      val = add(stack_pop(), val);
      break;
    case BC_SUB:
      val = sub(stack_pop(), val);
      break;
    case BC_MUL:
      val = mul(stack_pop(), val);
      break;
    case BC_DIV:
      val = div(stack_pop(), val);
      break;
    case BC_MOD:
      val = mod(stack_pop(), val);
      break;
    case BC_POWER:
      val = power(stack_pop(), val);
      break;
    case BC_NEG:
      val = neg(val);
      break;
    case BC_NOT:
      val = bnot(val);
      break;
    case BC_AND:
      val = value(stack_pop().to_int() & val.to_int());
      break;
    case BC_OR:
      val = value(stack_pop().to_int() | val.to_int());
      break;
    case BC_LIKE: 
      val = value(stack_pop().to_string().like(val.to_string()));
      break;
    case BC_LITERAL:
      val = prg->get_literal(get_uint(bc));
      break;
    case BC_TRUE:
      val = value(true);
      break;
    case BC_FALSE:
      val = value(false);
      break;
    case BC_NULL:
      val = value();
      break;
    case BC_CANCEL:
      val = value::cancel_val();
      break;

    case BC_GLOBAL:
      val.set_object(globals);
      break;
    case BC_SELF:
      val.set_object(self);
      break;
    case BC_CALL: {
      uint sym    = get_uint(bc);
      argn        = get_uint(bc);
      uint t_argn = argn;
      argv        = _stack.tail() - argn;
      val         = value::null_val();

      if (is_func_ref(argv[0])) {
        next = bc; // next bc after the call
        // that is locally defined function
        val             = argv[0];
        closure *clos   = (closure *)val.get_resource();
        uint     offset = clos->func_off;
        bc              = start + offset;
        argn            = t_argn;
        continue;
      } else if (!call_function(sym, argv[0], argn - 1, argv + 1, val))
        raise_error(ERR_FUNC_UNKNOWN, symbol_name(sym).c_str());
      _stack.drop(t_argn);
    } break;
    case BC_INVOKE: {
      value obj;
      stack_pop(obj);
      if (!is_func_ref(val))
        raise_error(ERR_IS_NOT_FUNCTION);
      if (obj.is_object() /*&& !obj.is_range()*/) {
        struct t : enumerator {
          vm &  mv;
          value f;
          t(vm &v, const value &m) : mv(v), f(m) {}
          virtual bool doit(const value &val) {
            value rv = mv.call(f, 1, &val);
            if (rv.equal(value::cancel_val()))
              return false;
            return true;
          }
        } en(*this, val);
        next = bc;
        val  = value(enumerate(obj, en));
      } else if (obj.is_string()) {
        value   f  = val;
        ustring us = obj.to_string();
        for (int n = 0; n < us.size(); ++n) {
          value p;
          p   = value(int(us[n]));
          val = call(f, 1, &p);
          if (val == value::cancel_val())
            break;
        }
      } else if (obj.is_range()) {
        value f = val;
        int   s, e;
        obj.get_range(s, e);
        for (int n = s; n <= e; ++n) {
          value p;
          p   = value(int(n));
          val = call(f, 1, &p);
          if (val == value::cancel_val())
            break;
        }
      } else
        raise_error(ERR_IS_NOT_ENUMERABLE);
    } break;
    case BC_BRF:
      off = get_uint(bc);
      if (to_bool(val) == false)
        bc = prg->codes.head() + off;
      break;
    case BC_BRT:
      off = get_uint(bc);
      if (to_bool(val) == true)
        bc = prg->codes.head() + off;
      break;
    case BC_BR:
      off = get_uint(bc);
      bc  = prg->codes.head() + off;
      break;
    case BC_VAR_GET:
      n   = get_uint(bc);
      off = n >> 16;
      n &= 0xffff;
      {
        frame *frm = calls.last();
        while (off) {
          frm = ((closure *)frm->clos.get_resource())->context;
          --off;
        }
        assert(n < frm->vars.size());
        val = frm->vars.get_element(n);
      }
      break;
    case BC_EXT_CONST: {
      uint sym = get_uint(bc);
      if (!get_const(sym, val))
        val.clear();
      // raise_error(ERR_CONST_UNKNOWN, symbol_name(sym).c_str());
    } break;

    case BC_VAR_SET:
      n   = get_uint(bc);
      off = n >> 16;
      n &= 0xffff;
      {
        frame *frm = calls.last();
        while (off) {
          frm = ((closure *)frm->clos.get_resource())->context;
          --off;
        }
        assert(n < frm->vars.size());
        frm->vars.set_element(n, val);
      }

      /*if( off )
        calls[ calls.size() - off ]->vars[n] = val;
      else
      {
        assert( n < vars.size() );
        vars[n] = val;
      }*/

      break;

    case BC_LE:
    case BC_LT:
    case BC_GT:
    case BC_GE:
    case BC_EQ:
    case BC_NE:
      val = cmp(code, stack_pop(), val);
      break;

    case BC_RANGE:
      val = value::make_range(stack_pop().get(0), val.get(0));
      break;

    default:
      printf("unknown bytecode=%d\n", code);
      assert(0);

      break;
    }
  }
  return false;
}

uint vm::func_num_vars(const value &func_ref) {
  if (!is_func_ref(func_ref))
    return 0;
  uint func_start = ((closure *)func_ref.get_resource())->func_off;
  return read_uint(prg->codes.head() + func_start + 1);
}

value vm::call(const value &func_ref, uint carg, const value *varg) {
  assert(varg < _stack.head() ||
         varg > _stack.tail()); // must not be on the VM's stack

  assert(is_func_ref(func_ref));
  val             = func_ref;
  uint func_start = ((closure *)func_ref.get_resource())->func_off;
  // func_start here is an idx of BC_FRAME instruction.
  uint func_varc = read_uint(prg->codes.head() + func_start + 1);
  assert(carg <= func_varc);

  uint used_argc = min(carg, func_varc);

  stack_push(value(globals.ptr())); // always 'this'
  for (uint n = 0; n < used_argc; ++n)
    stack_push(varg[n]);
  ++used_argc;

  argv = _stack.tail() - used_argc;
  argn = used_argc;

  bool r = _eval(prg->codes.head() + func_start);
  assert(r);
  r = r;
  return val;
}

void vm::raise_error(runtime_error_e n, ...) {
  va_list args;
  va_start(args, n);
  char          buf[1024];
  eval_runtime_error er;
  er.no      = n;
  er.line_no = 1 + line_no;
  er.url     = prg->url;
  switch (n) {
  case ERR_IS_NOT_NUMBER:
    // er.msg = string::format("is not a number");
    do_vsnprintf(buf, 1023, "is not a number", args);
    break;
  case ERR_ATTR_IS_READ_ONLY:
    do_vsnprintf(buf, 1023, "attribute '%S' not found or is read only", args);
    break;
  case ERR_ATTR_UNKNOWN:
    do_vsnprintf(buf, 1023, "attribute '%S' not found", args);
    break;
  case ERR_FUNC_UNKNOWN:
    do_vsnprintf(buf, 1023, "function '%S' not found", args);
    break;
  case ERR_STATE_UNKNOWN:
    do_vsnprintf(buf, 1023, "state flag '%S' not found", args);
    break;
  case ERR_STATE_IS_READ_ONLY:
    do_vsnprintf(buf, 1023, "state flag '%S' not found or is read only", args);
    break;
  case ERR_LENGTH_CONVERSION:
    do_vsnprintf(buf, 1023, "bad length unit value", args);
    break;
  case ERR_EVENT_UNKNOWN:
    do_vsnprintf(buf, 1023, "event '%S' not found", args);
    break;
  case ERR_IS_NOT_ENUMERABLE:
    do_vsnprintf(buf, 1023, "left side is not enumerable", args);
    break;
  case ERR_IS_NOT_FUNCTION:
    do_vsnprintf(buf, 1023, "is not a function", args);
    break;
  case ERR_CONST_UNKNOWN:
    do_vsnprintf(buf, 1023, "constant '%S' not found", args);
    break;
  case ERR_WRONG_ARGC:
    do_vsnprintf(buf, 1023, "wrong number of arguments", args);
    break;

  case ERR_ATTR_ON_NULL:
    do_vsnprintf(buf, 1023, "attempt to get/set attribute '%S' on null", args);
    break;
  case ERR_SATTR_ON_NULL:
    do_vsnprintf(buf, 1023, "attempt to get/set property '%S' on null", args);
    break;
  case ERR_STATE_ON_NULL:
    do_vsnprintf(buf, 1023, "attempt to get/set state '%S' on null", args);
    break;
  }
  va_end(args);
  er.msg = buf;
  throw er;
}

} // namespace eval

} // namespace tool
