/* vector.c - 'Array' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <string.h>
#include "cs.h"

namespace tis {

  /* method handlers */
  static value CSF_ctor(VM *c);
  static value CSF_clone(VM *c);
  static value CSF_push(VM *c);
  static value CSF_pushFront(VM *c);
  static value CSF_pop(VM *c);
  static value CSF_popFront(VM *c);
  static value CSF_concat(VM *c);
  static value CSF_join(VM *c);
  static value CSF_reverse(VM *c);
  static value CSF_slice(VM *c);
  static value CSF_splice(VM *c);
  static value CSF_sort(VM *c);
  static value CSF_indexOf(VM *c);
  static value CSF_lastIndexOf(VM *c);
  static value CSF_remove(VM *c);
  static value CSF_removeByValue(VM *c);
  static value CSF_map(VM *c);
  static value CSF_reduce(VM *c);
  static value CSF_reduceRight(VM *c);
  static value CSF_filter(VM *c);
  static value CSF_find(VM *c);
  static value CSF_findLast(VM *c);
  static value CSF_every(VM *c);
  static value CSF_some(VM *c);
  static value CSF_LCS(VM *c);

  value  CSF_addObserver(VM *c);
  value  CSF_removeObserver(VM *c);
  value  CSF_eachObserver(VM *c);

#if defined(TISCRIPT_USE_PERSISTENCE)
#define FETCH(c, obj)                                                          \
  if (_CsIsPersistent(obj)) obj = CsFetchVectorData(c, obj);
#define FETCH_P(c, obj, p1)                                                    \
  if (_CsIsPersistent(obj)) {                                                  \
    CsPush(c, p1);                                                             \
    obj = CsFetchVectorData(c, obj);                                           \
    p1  = CsPop(c);                                                            \
  }
#define FETCH_PP(c, obj, p1, p2)                                               \
  if (_CsIsPersistent(obj)) {                                                  \
    CsPush(c, p2);                                                             \
    CsPush(c, p1);                                                             \
    obj = CsFetchVectorData(c, obj);                                           \
    p1  = CsPop(c);                                                            \
    p2  = CsPop(c);                                                            \
  }
#else
#define FETCH(c, obj)
#define FETCH_P(c, obj, p1)
#define FETCH_PP(c, obj, p1, p2)
#endif

  /* virtual property methods */
  static value CSF_length(VM *c, value obj);
  static void  CSF_set_length(VM *c, value obj, value value);

  static value CSF_first(VM *c, value obj);
  static void  CSF_set_first(VM *c, value obj, value value);
  static value CSF_last(VM *c, value obj);
  static void  CSF_set_last(VM *c, value obj, value value);

  /* Vector methods */
  static c_method methods[] = {
      C_METHOD_ENTRY("this", CSF_ctor),
      C_METHOD_ENTRY("toLocaleString", CSF_std_toLocaleString),
      C_METHOD_ENTRY("toString", CSF_join),
      C_METHOD_ENTRY("toCssString", CSF_join),
      C_METHOD_ENTRY("valueOf", CSF_join),
      C_METHOD_ENTRY("clone", CSF_clone),
      C_METHOD_ENTRY("push", CSF_push),
      C_METHOD_ENTRY("unshift", CSF_pushFront),
      C_METHOD_ENTRY("pop", CSF_pop),
      C_METHOD_ENTRY("shift", CSF_popFront),
      C_METHOD_ENTRY("concat", CSF_concat),
      C_METHOD_ENTRY("join", CSF_join),
      C_METHOD_ENTRY("reverse", CSF_reverse),
      C_METHOD_ENTRY("slice", CSF_slice),
      C_METHOD_ENTRY("splice", CSF_splice),
      C_METHOD_ENTRY("sort", CSF_sort),
      C_METHOD_ENTRY("indexOf", CSF_indexOf),
      C_METHOD_ENTRY("lastIndexOf", CSF_lastIndexOf),
      C_METHOD_ENTRY("remove", CSF_remove),
      C_METHOD_ENTRY("removeByValue", CSF_removeByValue),
      C_METHOD_ENTRY("find", CSF_find),
      C_METHOD_ENTRY("findLast", CSF_findLast),
      C_METHOD_ENTRY("some", CSF_some),
      C_METHOD_ENTRY("every", CSF_every),
      C_METHOD_ENTRY("LCS", CSF_LCS),
      
      C_METHOD_ENTRY("map", CSF_map),
      C_METHOD_ENTRY("reduce", CSF_reduce),
      C_METHOD_ENTRY("reduceRight", CSF_reduceRight),
      C_METHOD_ENTRY("filter", CSF_filter),

      C_METHOD_ENTRY("addObserver", CSF_addObserver),
      C_METHOD_ENTRY("removeObserver", CSF_removeObserver),
      C_METHOD_ENTRY("eachObserver", CSF_eachObserver),

      C_METHOD_ENTRY(0, 0)};

  /* Vector properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY("length", CSF_length, CSF_set_length),
      VP_METHOD_ENTRY("first", CSF_first, CSF_set_first),
      VP_METHOD_ENTRY("last", CSF_last, CSF_set_last),
      VP_METHOD_ENTRY(0, 0, 0)};

  /* prototypes */

  /* CsInitVector - initialize the 'Vector' obj */
  void CsInitVector(VM *c) {
    c->vectorObject = CsEnterType(CsGlobalScope(c), "Array", &CsVectorDispatch);
    CsEnterMethods(c, c->vectorObject, methods);
    CsEnterVPMethods(c, c->vectorObject, properties);
  }

  /* CSF_ctor - built-in method 'initialize' */
  static value CSF_ctor(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsVectorDispatch);
    int n = c->argc - 3 + 1;
    if (n == 1 && CsIntegerP(CsGetArg(c, 3))) {
      int size = CsIntegerValue(CsGetArg(c, 3));
      if( size < 0 )
        CsThrowKnownError(c, CsErrValueError, CsGetArg(c, 3));
      obj = CsResizeVector(c, obj, size);
    } else if (n) {
      obj = CsResizeVector(c, obj, n);
      for (int i = 3; i <= c->argc; ++i)
        CsSetVectorElementI(obj, i - 3, CsGetArg(c, i));
    }
    CsCtorRes(c) = obj;
    return obj;
  }

  /* CSF_Clone - built-in method 'Clone' */
  static value CSF_clone(VM *c) {
    value obj, deep = 0;
    CsParseArguments(c, "V=*|V", &obj, &CsVectorDispatch, &deep);
    FETCH(c, obj);
    return CsCloneVector(c, obj, deep == TRUE_VALUE);
  }

  void CsVectorPush(VM *c, value obj, value v) {
    FETCH(c, obj);
    PROTECT(obj, v);
    CsSetModified(obj, true);
    int size    = CsVectorSize(c, obj);
    obj         = CsResizeVector(c, obj, size + 1);
    value *parr = CsVectorAddressI(obj);
    parr[size]  = v;
  }

  void CsVectorRemove(VM *c, value obj, int idx) {
    FETCH(c, obj);
    CsVectorRemoveI(obj, idx);
  }

  void CsVectorRemoveI(value obj, int idx) {
    CsSetModified(obj, true);
    int size = CsVectorSizeI(obj);
    assert(idx >= 0 && idx < size);
    // obj = CsResizeVector(c,obj,size + 1);
    value *p = CsVectorAddressI(obj);
    value *plast = p + size - 1;
    for (p += idx; p < plast; ++p)
      *p = *(p + 1);
    *plast = UNDEFINED_VALUE;
    CsSetVectorSize(obj, size - 1);
  }


  /*value CsUnspreadVector(VM* c, value vec)
  {
    PROTECT(vec);
    GC_vector expanded(c);
    for (int i = 0; i < CsVectorSize(c, vec); ++i)
    {
      value el = CsVectorAddressI(vec)[i];
      if (CsSpreadP(el)) {
        value other = CsFixedVectorElement(el, 0);
        if (CsVectorP(other))
          expanded.elements.push(CsVectorElements(c, other));
        else if (CsTupleP(other))
          expanded.elements.push(CsTupleElements(other));
        else if (CsGeneratorP(c, other))
        {
          PROTECT(other);
          int n = 0;
          for (; n < 10000; ++n) {
            value next = CsExecGenerator(c, other);
            if (next == NOTHING_VALUE)
              break;
            expanded.elements.push(next);
          }
          if (n == 10000)
            CsThrowKnownError(c, CsErrGenericError, "too many elements");
        }
        else
          expanded.elements.push(other);
      }
      else
        expanded.elements.push(el);
    }
    vec = CsMakeVector(c, expanded.elements.size());
    CsVectorTargetElements(vec).copy(expanded.elements());
    return vec;
  }

  value CsUnspreadTuple(VM* c, value tup)
  {
    value tname = CsTupleName(tup);
    PROTECT(tup,tname);
    GC_vector expanded(c);
    for (int i = 0; i < CsTupleSize(tup); ++i)
    {
      value el = CsTupleElement(tup, i);
      if (CsSpreadP(el)) {
        value other = CsFixedVectorElement(el, 0);
        if (CsVectorP(other))
          expanded.elements.push(CsVectorElements(c, other));
        else if (CsTupleP(other))
          expanded.elements.push(CsTupleElements(other));
        else if (CsGeneratorP(c, other))
        {
          PROTECT(other);
          int n = 0;
          for (; n < 10000; ++n) {
            value next = CsExecGenerator(c, other);
            if (next == NOTHING_VALUE)
              break;
            expanded.elements.push(next);
          }
          if(n == 10000)
            CsThrowKnownError(c, CsErrGenericError, "too many elements");
        }
        else
          expanded.elements.push(other);
      }
      else
        expanded.elements.push(el);
    }
    PROTECT(tname);
    tup = CsMakeTuple(c, expanded.elements.size());
    CsTupleTargetElements(tup).copy(expanded.elements());
    CsSetTupleName(tup, tname);
    return tup;
  }*/


  /* CSF_Push - built-in method 'Push' */
  static value CSF_push(VM *c) {
    value obj;
    int_t size;
    int_t argc = CsArgCnt(c);
    if (argc < 3) {
      // CsTooFewArguments(c); make it to accept zero real args - see https ://sciter.com/forums/topic/array-push-apply-fires-an-exception/
      return UNDEFINED_VALUE;
    }
    CsCheckType(c, 1, CsVectorP);
    obj = CsGetArg(c, 1);
    FETCH(c, obj);
    CsSetModified(obj, true);
    size         = CsVectorSize(c, obj);
    int new_size = size + argc - 2;

    value  v = UNDEFINED_VALUE;
    PROTECT(obj,v);
    obj          = CsResizeVector(c, obj, new_size);
    value *parr  = CsVectorAddressI(obj);
    for (int n = 3; n <= argc; ++n)
      parr[size + n - 3] = (v = CsGetArg(c, n));

    if (value observer = CsVectorObserver(obj)) {
      CsEnqueueNotification(c, observer, obj, CsMakeInteger(size),
                            CsMakeInteger(size + argc - 2), UNDEFINED_VALUE,
                            ADD_RANGE);
    }
    return v;
  }

  value ThisVector(VM *c) {
    CsCheckType(c, 1, CsVectorP);
    return CsGetArg(c, 1);
  }

  /* CSF_PushFront - built-in method 'pushFront' */
  static value CSF_pushFront(VM *c) {
    value obj;
    int_t size;
    int_t argc = CsArgCnt(c);
    //if (argc < 3) CsTooFewArguments(c);
    if (argc < 3) {
      // CsTooFewArguments(c); make it to accept zero real args - see https ://sciter.com/forums/topic/array-push-apply-fires-an-exception/
      return UNDEFINED_VALUE;
    }
    CsCheckType(c, 1, CsVectorP);
    obj = CsGetArg(c, 1);
    FETCH(c, obj);
    CsSetModified(obj, true);
    size         = CsVectorSize(c, obj);
    int new_size = size + argc - 2;
    value v = UNDEFINED_VALUE;
    PROTECT(obj, v);
    obj          = CsResizeVector(c, obj, new_size);
    value *parr  = CsVectorAddressI(obj);
    value *dst   = parr + new_size;
    value *src   = parr + size;
    while (--size >= 0)
      *--dst = *--src;
    for (int n = 3; n <= argc; ++n)
      parr[n - 3] = (v = CsGetArg(c, n));

    if (value observer = CsVectorObserver(obj)) {
      CsEnqueueNotification(c, observer, obj, CsMakeInteger(0),
                            CsMakeInteger(argc - 2), UNDEFINED_VALUE,
                            ADD_RANGE);
    }
    return v;
  }

  /* CSF_Pop - built-in method 'Pop' */
  static value CSF_pop(VM *c) {
    value vector = ThisVector(c), val;
    FETCH(c, vector);
    CsSetModified(vector, true);

    int_t size = CsVectorSize(c, vector);
    // if (size <= 0)
    //    CsThrowKnownError(c,CsErrStackEmpty,obj);
    if (size <= 0) return NOTHING_VALUE;
    val = CsVectorElement(c, vector, --size);
    CsSetVectorSize(vector, size);

    if (value observer = CsVectorObserver(vector)) {
      PROTECT(val);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(size),
                            CsMakeInteger(size + 1), UNDEFINED_VALUE,
                            DELETE_RANGE);
    }

    return val;
  }

  /* CSF_PopFront - built-in method 'PopFront' */
  static value CSF_popFront(VM *c) {
    value vector = ThisVector(c), val, *p;
    FETCH(c, vector);
    CsSetModified(vector, true);
    int_t size;
    size = CsVectorSize(c, vector);
    if (size <= 0) return NOTHING_VALUE;
    val = CsVectorElement(c, vector, 0);
    CsSetVectorSize(vector, --size);
    for (p = CsVectorAddress(c, vector); --size >= 0; ++p)
      *p = p[1];
    if (value observer = CsVectorObserver(vector)) {
      PROTECT(val);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(0),
                            CsMakeInteger(1), UNDEFINED_VALUE, DELETE_RANGE);
    }
    return val;
  }

  static value CSF_remove(VM *c) {
    value vector, val, *p;
    int_t pos;
    CsParseArguments(c, "V=*i", &vector, &CsVectorDispatch, &pos);
    FETCH(c, vector);
    CsSetModified(vector, true);

    int_t size = CsVectorSize(c, vector);

    if (pos < 0 || pos >= size) return NOTHING_VALUE;

    val = CsVectorElement(c, vector, pos);
    CsSetVectorSize(vector, --size);
    for (p = CsVectorAddress(c, vector) + pos; --size >= pos; ++p)
      *p = p[1];

    if (value observer = CsVectorObserver(vector)) {
      PROTECT(val);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(pos),
                            CsMakeInteger(pos + 1), UNDEFINED_VALUE,
                            DELETE_RANGE);
    }

    return val;
  }

  static bool RemoveItem(VM *c, value vector, value idx) {
    if (!CsIntegerP(idx))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, idx,
                        "only integer as an index");
    int_t pos = CsIntegerValue(idx);
    FETCH(c, vector);
    CsSetModified(vector, true);
    int_t size = CsVectorSize(c, vector);
    if (pos < 0 || pos >= size) return false;
    // value rval = CsVectorElementI(vector,pos);
    CsSetVectorSize(vector, --size);
    for (value *p = CsVectorAddress(c, vector) + pos; --size >= pos; ++p)
      p[0] = p[1];
    if (value observer = CsVectorObserver(vector))
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(pos),
                            CsMakeInteger(pos + 1), UNDEFINED_VALUE,
                            DELETE_RANGE);
    return true;
  }

  static value CSF_removeByValue(VM *c) {
    value vector = 0, element = 0;
    PROTECT(vector, element);

    CsParseArguments(c, "V=*V", &vector, &CsVectorDispatch, &element);
    FETCH(c, vector);
    CsSetModified(vector, true);

    int_t size = CsVectorSize(c, vector);
    int   n    = 0;
    for (; n < size; ++n) {
      if (CsEqualOp(c, CsVectorElement(c, vector, n), element)) {
        element = CsVectorElement(c, vector, n);
        break;
      }
    }
    if (n >= size) return NOTHING_VALUE;
    CsSetVectorSize(vector, --size);
    value *p = CsVectorAddress(c, vector);
    for (p += n; --size >= n; ++p)
      p[0] = p[1];
    if (value observer = CsVectorObserver(vector)) {
      PROTECT(element);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(n),
                            CsMakeInteger(n + 1), UNDEFINED_VALUE,
                            DELETE_RANGE);
    }

    return element;
  }

  /* CSF_length - built-in property 'length' */
  static value CSF_length(VM *c, value obj) {
    FETCH(c, obj);
    return CsMakeInteger(CsVectorSize(c, obj));
  }

  /* CSF_set_length - built-in property 'size' */
  static void CSF_set_length(VM *c, value obj, value val) {
    PROTECT(obj, val);
    FETCH(c, obj);
    if (!CsIntegerP(val)) CsTypeError(c, val);
    CsSetModified(obj, true);

    int_t size  = CsVectorSize(c, obj);
    int_t nsize = CsIntegerValue(val);

    CsResizeVector(c, obj, CsIntegerValue(val));

    if (value observer = CsVectorObserver(obj)) {
      if (size < nsize)
        CsEnqueueNotification(c, observer, obj, CsMakeInteger(size),
                              CsMakeInteger(nsize), UNDEFINED_VALUE, ADD_RANGE);
      else if (size > nsize)
        CsEnqueueNotification(c, observer, obj, CsMakeInteger(nsize),
                              CsMakeInteger(size), UNDEFINED_VALUE,
                              DELETE_RANGE);
    }
  }

  /* CSF_first - built-in property 'first' */
  static value CSF_first(VM *c, value obj) {
    FETCH(c, obj);
    if (CsVectorSize(c, obj)) return CsVectorElement(c, obj, 0);
    return NOTHING_VALUE;
  }

  /* CSF_set_first - built-in property 'first' */
  static void CSF_set_first(VM *c, value obj, value val) {
    PROTECT(obj,val);
    FETCH(c, obj);
    CsSetModified(obj, true);
    if (CsVectorSize(c, obj) == 0) obj = CsResizeVector(c, obj, 1);
    CsSetVectorElement(c, obj, 0, val);
    if (value observer = CsVectorObserver(obj))
      CsEnqueueNotification(c, observer, obj, CsMakeInteger(0),
                            CsMakeInteger(1), UNDEFINED_VALUE, UPDATE_RANGE);
  }

  /* CSF_last - built-in property 'last' */
  static value CSF_last(VM *c, value obj) {
    FETCH(c, obj);
    if (CsVectorSize(c, obj))
      return CsVectorElement(c, obj, CsVectorSize(c, obj) - 1);
    return NOTHING_VALUE;
  }

  /* CSF_set_last - built-in property 'last' */
  static void CSF_set_last(VM *c, value obj, value val) {
    PROTECT(obj,val);
    FETCH(c, obj);
    CsSetModified(obj, true);
    if (CsVectorSize(c, obj) == 0) obj = CsResizeVector(c, obj, 1);
    CsSetVectorElement(c, obj, CsVectorSize(c, obj) - 1, val);
    if (value observer = CsVectorObserver(obj))
      CsEnqueueNotification(
          c, observer, obj, CsMakeInteger(CsVectorSize(c, obj) - 1),
          CsMakeInteger(CsVectorSize(c, obj)), UNDEFINED_VALUE, UPDATE_RANGE);
  }

  static value CSF_concat(VM *c) {
    value vector  = ThisVector(c);
    value nvector = 0;
    PROTECT(vector, nvector);

    FETCH(c, vector);
    int n = c->argc - 3 + 1;
    if (n == 0) return vector;

    int_t d = CsVectorSize(c, vector);
    int   i;

    int_t extra = 0;
    for (i = 3; i <= c->argc; ++i) {
      value t = CsGetArg(c, i);
      if (CsVectorP(t))
        extra += CsVectorSize(c, t);
      else
        extra++;
    }
    nvector = CsMakeVector(c, d + extra, CsVectorClass(vector));

    n = d;
    for (i = 0; i < d; ++i)
      CsSetVectorElement(c, nvector, i, CsVectorElement(c, vector, i));

    for (i = 3; i <= c->argc; ++i) {
      value t = CsGetArg(c, i);
      if (CsVectorP(t)) {
        for (int j = 0; j < CsVectorSize(c, t); ++j)
          CsSetVectorElement(c, nvector, n++, CsVectorElement(c, t, j));
      } else
        CsSetVectorElement(c, nvector, n++, t);
    }
    return nvector;
  }

  static value CSF_join(VM *c) {
    value vector = 0;
    PROTECT(vector);
    wchar *str = 0;
    CsParseArguments(c, "V=*|S", &vector, &CsVectorDispatch, &str);
    tool::ustring dlm = str ? str : W(",");

    FETCH(c, vector);

    int_t n = CsVectorSize(c, vector);

    value         r;
    string_stream s(20);

    int i;
    for (i = 0; i < n - 1; ++i) {
      CsToString(c, CsVectorElement(c, vector, i), s);
      s.put_str(dlm);
    }
    if (i < n) CsToString(c, CsVectorElement(c, vector, i), s);

    r = s.string_o(c);
    s.close();
    return r;
  }

  static value CSF_reverse(VM *c) {
    value vector = ThisVector(c);

    FETCH(c, vector);
    CsSetModified(vector, true);
    int_t  d  = CsVectorSize(c, vector);
    value *p1 = CsVectorAddress(c, vector);
    value *p2 = p1 + d - 1;

    while (p1 < p2) {
      tool::swap(*p1, *p2);
      ++p1;
      --p2;
    }
    if (value observer = CsVectorObserver(vector)) {
      PROTECT(vector);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(0),
                            CsMakeInteger(CsVectorSize(c, vector)),
                            UNDEFINED_VALUE, UPDATE_RANGE);
    }

    return vector;
  }

  value CsVectorSlice(VM *c, value vector, int start, int end) {
    FETCH(c, vector);
    int len = CsVectorSize(c, vector);

    /* handle indexing from the left */
    if (start > 0) {
      if (start > len) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (start < 0) {
      if ((start = len + start) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (end < 0)
      end = len + end + 1;
    else if (end > len)
      end = len;

    if (start > end) return CsMakeVector(c, 0, CsVectorClass(vector));

    /* return the slice */
    PROTECT(vector);
    value  vdst = CsMakeVector(c, end - start, CsVectorClass(vector));
    value *src  = CsVectorAddress(c, vector) + start;
    CsVectorTargetElements(vdst).copy(src, end - start);
    return vdst;
  }

  static value CSF_slice(VM *c) {
    int   len, start = 0, end = -1;
    value vector;

    /* parse the arguments */
    CsParseArguments(c, "V=*|ii", &vector, &CsVectorDispatch, &start, &end);
    FETCH(c, vector);

    len = CsVectorSize(c, vector);

    /* handle indexing from the left */
    if (start >= len) 
      return CsMakeVector(c, 0, CsVectorClass(vector));
    
    /* handle indexing from the right */
    else if (start < 0)
      start = max(0, len + start);

    /* handle the count */
    if (end < 0)
      end = max(0,len + end + 1);
    else if (end > len)
      end = len;

    if (start > end) return CsMakeVector(c, 0, CsVectorClass(vector));

    /* return the slice */
    PROTECT(vector);
    value  vdst = CsMakeVector(c, end - start, CsVectorClass(vector));
    value *src  = CsVectorAddress(c, vector) + start;
    CsVectorTargetElements(vdst).copy(src, end - start);
    return vdst;
  }

  value CsVectorInsertReplace(VM *c, value vector, int at, value otherVector)
  {
    if( !CsVectorP(vector) )
      return CsCloneVector(c, otherVector);
    int original_size = CsVectorSize(c, vector);
    int other_size = CsVectorSize(c, otherVector);
    PROTECT(vector, otherVector);
    value newVector = CsMakeVector(c, original_size + other_size - 1);
    auto  target = CsVectorTargetElements(newVector);
    target.copy(CsVectorElements(c, vector)(0, at));
    target.copy(CsVectorElements(c, otherVector));
    target.copy(CsVectorElements(c, vector)(at+1));
    return newVector;
  }

  static value CSF_splice(VM *c) {
    int   len, start, cnt = -1;
    value vector;

    /* parse the arguments */
    CsParseArguments(c, "V=*i|i|", &vector, &CsVectorDispatch, &start, &cnt);
    FETCH(c, vector);

    len = CsVectorSize(c, vector);

    /* handle indexing from the left */
    //WTF? if (start > 0) {
    //  if (start > len) return UNDEFINED_VALUE;
    //}

    if (start > len) /* handle indexing from the left */ 
      start = len;
    else if (start < 0) /* handle indexing from the right */ 
      start = max(0, len + start);

    /* handle the count */
    if (cnt < 0)
      cnt = len - start;
    else if (start + cnt > len)
      cnt = len - start;

    if (cnt < 0) return UNDEFINED_VALUE;

    int cnt_to_insert = CsArgCnt(c) - 4;

    /* return the slice */
    PROTECT(vector);
    value vdst = CsMakeVector(c, cnt, CsVectorClass(vector));

    CsVectorTargetElements(vdst).copy(CsVectorAddress(c, vector) + start, cnt);

    CsSetModified(vector, true);

    if (cnt_to_insert) {
      int nlen = len - cnt + cnt_to_insert;
      size_t cnt_copy = len - start - cnt;
      if (nlen < len) {
        auto   target = CsVectorTargetElements(vector);
        target.move(start + cnt_to_insert, start + cnt, cnt_copy);
        target.prune(start);
        for (int n = 5; n <= CsArgCnt(c); ++n) {
          value v = CsGetArg(c, n);
          target = target.copy(v);
        }
        CsResizeVector(c, vector, nlen);
      }
      else {
        CsResizeVector(c, vector, nlen);
        auto   target = CsVectorTargetElements(vector);
        target.move(start + cnt_to_insert, start + cnt, cnt_copy);
        target.prune(start);
        for (int n = 5; n <= CsArgCnt(c); ++n) {
          value v = CsGetArg(c, n);
          target = target.copy(v);
        }
      }

      if (value observer = CsVectorObserver(vector)) {
        PROTECT(vdst);
        if (len < nlen)
          CsEnqueueNotification(c, observer, vector, CsMakeInteger(start),
                                CsMakeInteger(start + nlen - len),
                                UNDEFINED_VALUE, ADD_RANGE);
        else if (len > nlen)
          CsEnqueueNotification(c, observer, vector, CsMakeInteger(start),
                                CsMakeInteger(start + len - nlen),
                                UNDEFINED_VALUE, DELETE_RANGE);
        else
          CsEnqueueNotification(c, observer, vector, CsMakeInteger(start),
                                CsMakeInteger(start + len), UNDEFINED_VALUE,
                                UPDATE_RANGE);
      }
    } else if(cnt) {
      CsVectorTargetElements(vector).move(start, start + cnt,
                                          len - start - cnt);
      CsResizeVector(c, vector, len - cnt);
      if (value observer = CsVectorObserver(vector)) {
        PROTECT(vdst);
        CsEnqueueNotification(c, observer, vector, CsMakeInteger(start),
                              CsMakeInteger(start + cnt), UNDEFINED_VALUE,
                              DELETE_RANGE);
      }
    }
    return vdst;
  }

  struct cmpValues {
    VM *vm;
    cmpValues(VM *pvm) : vm(pvm) {}
    bool operator()(const value &v1, const value &v2) {
      return CsCompareObjects(vm, v1, v2, true) < 0;
    }
  };

  // value CsCallFunction(CsScope *scope,value fun,int argc,...)

  /*struct cmpValuesProxy
  {
    pvalue fun;
    cmpValuesProxy( VM *pvm, value f ):fun(pvm,f) {}
    bool operator()( const value& v1, const value& v2)
    {
      value r = CsCallFunction(CsCurrentScope(fun.pvm),fun,2,v1,v2);
      if(CsIntegerP(r))
        return CsIntegerValue(r) < 0;
      return false;
    }
  };*/

  static value CSF_sort(VM *c) {
    value vector = 0, cmpf = 0;
    PROTECT(vector, cmpf);

    CsParseArguments(c, "V=*|V", &vector, &CsVectorDispatch, &cmpf);

    FETCH(c, vector);
    CsSetModified(vector, true);

    int_t  d = CsVectorSize(c, vector);
    value *p = CsVectorAddress(c, vector);
    if (!cmpf) {
      tool::sort<value, cmpValues>(p, d, cmpValues(c));
    } else if (CsMethodP(cmpf)) {
      auto comparator = [&](const value &v1, const value &v2) -> bool {
        value r = CsCallFunction(CsCurrentScope(c), cmpf, 2, v1, v2);
        if (CsIntegerP(r)) return CsIntegerValue(r) < 0;
        if (CsFloatP(r)) return CsFloatValue(r) < 0;
        return false;
      };

      GC_vector arr(c);
      arr.elements = tool::slice<value>(p, d);
      tool::sort(arr.elements.head(), arr.elements.length(), comparator);
      // tool::copy(CsVectorAddress(c,vector),arr.elements.head(),arr.elements.size());
      CsVectorTargetElements(vector).copy(arr.elements());
    } else
      CsTypeError(c, cmpf);

    if (value observer = CsVectorObserver(vector)) {
      PROTECT(vector);
      CsEnqueueNotification(c, observer, vector, CsMakeInteger(0),
                            CsMakeInteger(CsVectorSize(c, vector)),
                            UNDEFINED_VALUE, UPDATE_RANGE);
    }

    return vector;
  }

  int_t CsVectorIndexOf(VM *c, value vector, value val) {
    PROTECT(vector, val);
    FETCH(c, vector);
    int_t d = CsVectorSize(c, vector);
    for (int n = 0; n < d; ++n) {
      if (CsStrongEql(CsVectorElement(c, vector, n), val)) return n;
    }
    return -1;
  }

  static value CSF_indexOf(VM *c) {
    value vector = 0, v = 0, dv = CsMakeInteger(-1);
    PROTECT(vector, v, dv);

    CsParseArguments(c, "V=*V|V", &vector, &CsVectorDispatch, &v, &dv);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);
    for (int n = 0; n < d; ++n) {
      if (CsStrongEql(CsVectorElement(c, vector, n), v))
        return CsMakeInteger(n);
    }
    return dv;
  }

  static value CSF_lastIndexOf(VM *c) {
    value vector = 0, v = 0, dv = CsMakeInteger(-1);
    PROTECT(vector, v, dv);

    CsParseArguments(c, "V=*V|V", &vector, &CsVectorDispatch, &v, &dv);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);
    for (int n = d - 1; n >= 0; --n) {
      if (CsStrongEql(CsVectorElement(c, vector, n), v))
        return CsMakeInteger(n);
    }
    return dv;
  }

  static value CSF_map(VM *c) {
    value vector = 0, cmpf = 0, cmpf_this = 0, r_vector = 0;
    PROTECT(vector, cmpf, cmpf_this, r_vector);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    r_vector = CsMakeVector(c, d, CsVectorClass(vector));

    int_t ndst = 0;
    // value *p = CsVectorAddress(c,vector);
    if (cmpf_this) {
      for (int n = 0; n < d; ++n) {
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3,
                               CsVectorElement(c, vector, n), CsMakeInteger(n),
                               vector);
        if (r == NOTHING_VALUE) continue;
        CsSetVectorElement(c, r_vector, ndst++, r);
      }
    } else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = 0; n < d; ++n) {
        value r = CsCallFunction(scope, cmpf, 3, CsVectorElement(c, vector, n),
                                 CsMakeInteger(n), vector);
        if (r == NOTHING_VALUE) continue;
        CsSetVectorElement(c, r_vector, ndst++, r);
      }
    }
    if (ndst != d) return CsResizeVector(c, r_vector, ndst);
    return r_vector;
  }

  static value CSF_find(VM *c) {
    value vector = 0, cmpf = 0, cmpf_this = 0, element = 0;
    PROTECT(vector, cmpf, cmpf_this, element);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    // value *p = CsVectorAddress(c,vector);
    if (cmpf_this) {
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c,r) == TRUE_VALUE) 
          CS_RETURN2(c, CsMakeInteger(n), element);
      }
    }
    else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallFunction(scope, cmpf, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          CS_RETURN2(c, CsMakeInteger(n), element);
      }
    }
    CS_RETURN2(c, CsMakeInteger(-1), UNDEFINED_VALUE);
  }

  static value CSF_findLast(VM *c) 
  {
    value vector = 0, cmpf = 0, cmpf_this = 0, element = 0;
    PROTECT(vector, cmpf, cmpf_this, element);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    if (cmpf_this) {
      for (int n = d - 1; n >= 0; --n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          CS_RETURN2(c, CsMakeInteger(n), element);
      }
    }
    else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = d - 1; n >= 0; --n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallFunction(scope, cmpf, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          CS_RETURN2(c, CsMakeInteger(n), element);
      }
    }
    CS_RETURN2(c, CsMakeInteger(-1), UNDEFINED_VALUE);
  }

  static value CSF_every(VM *c) 
  {
    value vector = 0, cmpf = 0, cmpf_this = 0, element = 0;
    PROTECT(vector, cmpf, cmpf_this, element);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    // value *p = CsVectorAddress(c,vector);
    if (cmpf_this) {
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == FALSE_VALUE)
          return FALSE_VALUE;
      }
    }
    else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallFunction(scope, cmpf, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == FALSE_VALUE)
          return FALSE_VALUE;
      }
    }
    return TRUE_VALUE;
  }

  static value CSF_some(VM *c)
  {
    value vector = 0, cmpf = 0, cmpf_this = 0, element = 0;
    PROTECT(vector, cmpf, cmpf_this, element);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    // value *p = CsVectorAddress(c,vector);
    if (cmpf_this) {
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          return TRUE_VALUE;
      }
    }
    else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = 0; n < d; ++n) {
        element = CsVectorElement(c, vector, n);
        value r = CsCallFunction(scope, cmpf, 3, element, CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          return TRUE_VALUE;
      }
    }
    return FALSE_VALUE;
  }

  static value CSF_reduce(VM *c) {
    value vector = 0, cmpf = 0, last = 0;
    PROTECT(vector, cmpf, last);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf,&last);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    CsScope *scope = CsCurrentScope(c);
    if (last)
      for (int n = 0; n < d; ++n) {
        last = CsCallFunction(
            scope, cmpf, 4, // previousValue, currentValue, index, array
            last, CsVectorElement(c, vector, n), CsMakeInteger(n), vector);
      }
    else if (d) {
      last = CsVectorElement(c, vector, 0);
      for (int n = 1; n < d; ++n) {
        last = CsCallFunction(
            scope, cmpf, 4, // previousValue, currentValue, index, array
            last, CsVectorElement(c, vector, n), CsMakeInteger(n), vector);
      }
    }
    if( !last )
      CsThrowKnownError(c, CsErrGenericError, "reduce() called with empty input but no initialValue");
    return last;
  }

  static value CSF_reduceRight(VM *c) {
    value vector = 0, cmpf = 0, last = 0;
    PROTECT(vector, cmpf, last);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &last);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    CsScope *scope = CsCurrentScope(c);
    if (last)
      for (int n = d - 1; n >= 0; --n) {
        last = CsCallFunction(
            scope, cmpf, 4, // previousValue, currentValue, index, array
            last, CsVectorElement(c, vector, n), CsMakeInteger(n), vector);
      }
    else if (d) {
      last = CsVectorElement(c, vector, d - 1);
      for (int n = d - 2; n >= 0; --n) {
        last = CsCallFunction(
            scope, cmpf, 4, // previousValue, currentValue, index, array
            last, CsVectorElement(c, vector, n), CsMakeInteger(n), vector);
      }
    }
    return last;
  }

  static value CSF_filter(VM *c) {
    value vector = 0, cmpf = 0, cmpf_this = 0, r_vector = 0;
    PROTECT(vector, cmpf, cmpf_this, r_vector);

    CsParseArguments(c, "V=*M|V", &vector, &CsVectorDispatch, &cmpf, &cmpf_this);

    FETCH(c, vector);

    int_t d = CsVectorSize(c, vector);

    r_vector = CsMakeVector(c, d, CsVectorClass(vector));
    int ndst = 0;
    // value *p = CsVectorAddress(c,vector);
    if (cmpf_this) {
      for (int n = 0; n < d; ++n) {
        value r = CsCallMethod(c, cmpf_this, cmpf, cmpf_this, 3,
                               CsVectorElement(c, vector, n), CsMakeInteger(n),
                               vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          CsSetVectorElement(c, r_vector, ndst++,
                             CsVectorElement(c, vector, n));
      }
    } else {
      CsScope *scope = CsCurrentScope(c);
      for (int n = 0; n < d; ++n) {
        value r = CsCallFunction(scope, cmpf, 3, CsVectorElement(c, vector, n),
                                 CsMakeInteger(n), vector);
        if (CsToBoolean(c, r) == TRUE_VALUE)
          CsSetVectorElement(c, r_vector, ndst++,
                             CsVectorElement(c, vector, n));
      }
    }
    return CsResizeVector(c, r_vector, ndst);
  }

  // longest common sequence
  static value CSF_LCS(VM *c) 
  {
    value a = 0, b = 0, cmpf = 0;
    PROTECT(a, b, cmpf);
    CsParseArguments(c, "**V=V=M", &a, &CsVectorDispatch, &b, &CsVectorDispatch, &cmpf);

    int_t a_length = CsVectorSize(c, a);
    int_t b_length = CsVectorSize(c, b);
    
    array<uint> L((a_length + 1) * (b_length + 1),0);

    auto A = [&](int i) -> value { return CsVectorElementI(a, i); };
    auto B = [&](int i) -> value { return CsVectorElementI(b, i); };

    auto LEL = [&](int r, int c) -> uint& { return L[r * b_length + c];  };

    auto CMP = [&](value a, value b) -> bool {
      CsScope *scope = CsCurrentScope(c);
      value r = CsCallFunction(scope, cmpf, 2, a, b);
      return r == TRUE_VALUE;
    };

    for(int i = 0; i < a_length; ++i)
      for (int j = 0; j < b_length; ++j)
        LEL(i + 1,j + 1) = (CMP(A(i), B(j))) 
        ? (1 + LEL(i,j)) 
        : max(LEL(i + 1,j), LEL(i,j + 1));

    GC_vector result(c);
    /*for (int i = a_length, j = b_length; i > 0 && j > 0; ) {
      if (CMP(A(i - 1),B(j - 1))) {
        result.elements.push(A(i - 1));
        i--;
        j--;
      }
      else
        if (LEL(i,j - 1) < LEL(i - 1,j))
          i--;
        else
          j--;
    }*/

    for (int i = a_length, j = b_length; i > 0 && j > 0; ) {
        if (LEL(i, j) == LEL(i - 1, j))
          i--;
        else if (LEL(i, j) == LEL(i, j - 1))
          j--;
        else {
          //assert(CMP(A(i - 1),B(j - 1)));
          result.elements.push(A(i - 1));
          j--;
          i--;
        }
    }

    result.elements.reverse();
    value r = CsMakeVector(c,result.elements.size());
    CsVectorTargetElements(r).copy(result.elements());
    return r;
  }
      
  // static value CSF_prototype(VM *c,value obj);
  // static void CSF_set_prototype(VM *c,value obj, value pro);

  bool CsVectorsEqual(VM *c, value v1, value v2, array<value> &stack) {
    if (v1 == v2) return true;
    if (CsVectorSize(c, v1) != CsVectorSize(c, v2)) return false;

    if (stack.get_index(v1) >= 0 || stack.get_index(v2) >= 0)
      CsThrowKnownError(c, CsErrGenericError,
                        "comparison of vectors with loops");
    // return false; // recursive data graphs are non-comparable

    stack.push(v1);
    stack.push(v2);

    PROTECT(v1, v2);
    FETCH(c, v1);
    FETCH(c, v2);
    int  d = CsVectorSize(c, v1);
    bool r = false;
    for (int n = 0; n < d; ++n) {
      if (!CsEqualOp(c, CsVectorElement(c, v1, n), CsVectorElement(c, v2, n), stack))
        goto DONE;
    }
    // return CsEqualOp(c,CsVectorTag(c,v1),CsVectorTag(c,v2));
    r = true;
  DONE:
    stack.pop();
    stack.pop();
    return r;
  }

  int CsCompareVectors(VM *c, value v1, value v2, bool suppressError) {
    PROTECT(v1, v2);
    FETCH(c, v1);
    FETCH(c, v2);
    int d = CsVectorSize(c, v1);
    for (int n = 0; n < d; ++n) {
      int r = CsCompareObjects(c, CsVectorElement(c, v1, n),
                               CsVectorElement(c, v2, n), suppressError);
      if (r) return r;
    }
    return tool::limit(int(CsVectorSize(c, v1) - CsVectorSize(c, v2)), -1, 1);
  }

  /* VECTOR */

  /* Vector handlers */
  static bool  GetVectorProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetVectorProperty(VM *c, value obj, value tag, value value);
  static value VectorNewInstance(VM *c, value proto);
  static long  VectorSize(value obj);
  static void  VectorScan(VM *c, value obj);

  static value CsVectorGetItem(VM *c, value obj, value tag);
  static void  CsVectorSetItem(VM *c, value obj, value tag, value value);

  bool VectorPrint(VM *c, value vector, stream *s, bool toLocale) {
    int_t n = CsVectorSize(c, vector);

    int i;
    for (i = 0; i < n - 1; ++i) {
      if (!CsPrintValue(c, CsVectorElement(c, vector, i), s, toLocale))
        return false;
      if (!s->put_str(",")) return false;
    }
    if (i < n)
      return CsPrintValue(c, CsVectorElement(c, vector, i), s, toLocale);

    return true;
  }

  /*inline value FindFirstMember(VM *c, value &index, value collection) {
    if (CsVectorP(collection)) {
      if (CsVectorSize(c, collection)) {
        index = CsMakeInteger(0);
        return CsVectorElement(c, collection, 0);
      }
    }
    // else if(CsObjectOrMethodP(obj))
    //{
    //  return CsFindFirstSymbol(c,obj);
    //}
    return NOTHING_VALUE;
  }*/

  value VectorNextElement(VM *c, value *index, value collection, int nr) {
    if (*index == NOTHING_VALUE) // first
    {
      FETCH(c, collection);
      if (CsVectorSize(c, collection)) {
        *index = CsMakeInteger(0);
        //CsSetRVal(c, 1, *index);
        //return CsVectorElement(c, collection, 0);
        CS_RETURN2(c, *index, CsVectorElement(c, collection, 0));
      }
    } else if (CsIntegerP(*index)) {
      int_t i = CsIntegerValue(*index) + 1;
      *index  = CsMakeInteger(i);
      if (i < CsVectorSize(c, collection)) {
        //CsSetRVal(c, 1, *index);
        //return CsVectorElement(c, collection, i);
        CS_RETURN2(c, *index, CsVectorElement(c, collection, i));
      }
    } else
      assert(false);

    //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
    return NOTHING_VALUE;
  }

  /* Vector pdispatch */
  dispatch CsVectorDispatch = {
      "Array",           &CsVectorDispatch,
      GetVectorProperty, SetVectorProperty,
      VectorNewInstance, VectorPrint,
      VectorSize,        CsDefaultCopy,
      VectorScan,        CsDefaultHash,
      CsVectorGetItem,   CsVectorSetItem,
      VectorNextElement, 0,
      RemoveItem,
  };

  static value CsVectorGetItem(VM *c, value obj, value tag) {
    FETCH_P(c, obj, tag);
    if (CsIntegerP(tag)) {
      int_t i;
      if ((i = CsIntegerValue(tag)) < 0 || i >= CsVectorSizeI(obj))
        //CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
        return NOTHING_VALUE;
      return CsVectorElementI(obj, i);
    }
    return UNDEFINED_VALUE;
  }
  static void CsVectorSetItem(VM *c, value obj, value tag, value val) {
    FETCH_PP(c, obj, tag, val);
    CsSetModified(obj, true);
    if (CsIntegerP(tag)) {
      int_t i;
      if ((i = CsIntegerValue(tag)) < 0)
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      else if (i >= CsVectorSizeI(obj)) {
        PROTECT(tag,val);
        obj = CsResizeVector(c, obj, i + 1);
      }
      value pval = CsVectorElementI(obj, i);
      if (pval != val) {
        CsSetVectorElementI(obj, i, val);
        if (value observer = CsVectorObserver(obj))
          CsEnqueueNotification(c, observer, obj, CsMakeInteger(i),
                                CsMakeInteger(i + 1), UNDEFINED_VALUE,
                                UPDATE_RANGE);
      }
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag,
                        "only integer as an index");
  }

  /* GetVectorProperty - Vector get property handler */
  static bool GetVectorProperty(VM *c, value &obj, value tag, value *pValue) {
    if (tag == PROTOTYPE_SYM) {
      *pValue = CsVectorClass(obj);
      return true;
    }
    value proto = CsVectorClass(obj);
    if (CsEntityP(proto)) {
      value tobj = obj;
      if (CsGetVirtualProperty(c, tobj, proto, tag, pValue)) return true;
    }
    return CsGetVirtualProperty(c, obj, c->vectorObject, tag, pValue);
  }

  /* SetVectorProperty - Vector set property handler */
  static bool SetVectorProperty(VM *c, value obj, value tag, value val) {
    if (tag == PROTOTYPE_SYM) {
      if (!CsEntityP(val) && val != UNDEFINED_VALUE)
        CsUnexpectedTypeError(c, val, "instance of Object or undefined");
      CsSetVectorClass(obj, val);
      return true;
    }

    value proto = CsVectorClass(obj);
    if (CsEntityP(proto)) {
      value tobj = obj;
      if (CsSetVirtualProperty(c, tobj, proto, tag, val)) return true;
    }
    return CsSetVirtualProperty(c, obj, c->vectorObject, tag, val);
  }

  /* VectorNewInstance - create a new vector */
  static value VectorNewInstance(VM *c, value proto) {
    return CsMakeVector(c, 0);
  }

  /* VectorSize - Vector size handler */
  static long VectorSize(value obj) {
    // assert(!CsMovedVectorP(obj));
    return sizeof(vector);
  }

  inline value _MakeFixedVectorCopy(VM *c, int nelements, value otherFW) {
    long  allocSize = sizeof(CsFixedVector) + nelements * sizeof(value);
    value newo; // = CsAllocate(c, allocSize);
    assert(c->newSpace->free + allocSize < c->newSpace->top);

    {
        memset(c->newSpace->free, 0, allocSize);
        newo = ptr_value((header *)c->newSpace->free);
        c->newSpace->free += allocSize;
    }

    CsSetDispatch(newo, &CsFixedVectorDispatch);
    CsSetFixedVectorSize(newo, nelements);
    if (nelements) {
      value *pdst = CsFixedVectorAddress(newo);
      value *psrc = CsFixedVectorAddress(otherFW);
      target(pdst, nelements).copy(psrc);
    }
    return newo;
  }

  /* VectorScan - Vector scan handler */
  static void VectorScan(VM *c, value obj) {
    ptr<persistent_header>(obj)->vstorage = CsCopyValue(c, ptr<persistent_header>(obj)->vstorage);
    vector *pv = ptr<vector>(obj);
    if (pv->data) {
      if (!CsIsNewObjectP(c, pv->data)) // compact the vector
        pv->data = _MakeFixedVectorCopy(c, pv->size, pv->data);
    }
    if (pv->observer) pv->observer = CsCopyValue(c, pv->observer);
    pv->proto = CsCopyValue(c, pv->proto);
    if (CsEntityObserver(obj))
      CsSetEntityObserver(obj, CsCopyValue(c, CsEntityObserver(obj)));
  }

  /* CsMakeFixedVectorValue - make a new vector value */
  value CsMakeFixedVectorValue(VM *c, dispatch *type, int size) {
    size = max(1,size); // Fixed vector MUST have at least ONE value, see CsBrokenHeart
    long  allocSize = sizeof(CsFixedVector) + size * sizeof(value);
    value newo      = CsAllocate(c, allocSize);
    CsSetDispatch(newo, type);
    CsSetFixedVectorSize(newo, size);
    value *p = CsFixedVectorAddress(newo);
    while (--size >= 0)
      *p++ = UNDEFINED_VALUE;
    // assert(allocSize == ValueSize(newo));
    return newo;
  }

  /* CsFixedVector - construct a fixed vector */
  value CsMakeFixedVector(VM *c, dispatch *type, int argc, value *argv) {
    assert(argc < 16); // this function used only for small arrays
    value newo;
    int   i;
    CsCheck(c, argc);
    for (i = argc - 1; i >= 0; --i)
      CsPush(c, argv[i]);
    newo = CsMakeFixedVectorValue(c, type, argc);
    for (i = 0; i < argc; ++i)
      CsSetFixedVectorElement(newo, i, CsPop(c));
    return newo;
  }

  /* Property handlers */
  long        FixedVectorSize(value obj);
  void        FixedVectorScan(VM *c, value obj);
  static bool FixedVectorPrint(VM *c, value obj, stream *s, bool toLocale);

  /* FixedVector pdispatch */
  dispatch CsFixedVectorDispatch = {
      "FixedVector",        &CsFixedVectorDispatch, CsDefaultGetProperty,
      CsDefaultSetProperty, CsDefaultNewInstance,   FixedVectorPrint,
      FixedVectorSize,      CsDefaultCopy,          FixedVectorScan,
      CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem};

  dispatch CsSpreadDispatch = {
    "SpreadProxy",        &CsSpreadDispatch,      CsDefaultGetProperty,
    CsDefaultSetProperty, CsDefaultNewInstance,   FixedVectorPrint,
    FixedVectorSize,      CsDefaultCopy,          FixedVectorScan,
    CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem };

  dispatch CsVirtualPropertyDispatch = {
    "VirtualProperty",    &CsVirtualPropertyDispatch,      CsDefaultGetProperty,
    CsDefaultSetProperty, CsDefaultNewInstance,   FixedVectorPrint,
    FixedVectorSize,      CsDefaultCopy,          FixedVectorScan,
    CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem };

  dispatch CsValueListDispatch = {
    "ValueList",          &CsValueListDispatch,   CsDefaultGetProperty,
    CsDefaultSetProperty, CsDefaultNewInstance,   FixedVectorPrint,
    FixedVectorSize,      CsDefaultCopy,          FixedVectorScan,
    CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem };


  /* FixedVectorSize - FixedVector size handler */
  long FixedVectorSize(value obj) {
    long len = CsFixedVectorSize(obj);
    return sizeof(CsFixedVector) + len * sizeof(value);
  }

  /* FixedVectorScan - FixedVector scan handler */
  void FixedVectorScan(VM *c, value obj) {
    long i, len = CsFixedVectorSize(obj);
    // dispatch *d = CsQuickGetDispatch(obj);
    //assert(len < 20);
    for (i = 0; i < len; ++i)
      CsSetFixedVectorElement(obj, i, CsCopyValue(c, CsFixedVectorElement(obj, i)));
  }

  /* FixedVectorPrint - print FV */
  bool FixedVectorPrint(VM *c, value obj, stream *s, bool toLocale) {
    long i;
    long len = CsFixedVectorSize(obj);

    s->put('[');
    s->put_str(CsTypeName(obj));
    s->put(' ');

    for (i = 0; i < len - 1; ++i) {
      CsDisplay(c, CsFixedVectorElement(obj, i), s);
      s->put(',');
    }
    if (len) CsDisplay(c, CsFixedVectorElement(obj, len - 1), s);

    s->put(']');

    return true;
  }

  /* CsMakeVector - make a new vector value */
  value CsMakeVector(VM *c, int_t size, value proto) {
    value obj = 0;
    PROTECT(proto, obj);
    obj = CsAllocate(c, sizeof(vector));
    CsSetDispatch(obj, &CsVectorDispatch);
    CsSetVectorSize(obj, size);
    CsSetVectorClass(obj, proto);
    // VectorScan knows about 0 val, so this is not needed: CsSetVectorData(obj,
    // UNDEFINED_VALUE ); // to make happy the GC that can happen int the next
    // line:
    value data = CsMakeFixedVectorValue(c, &CsFixedVectorDispatch, size);
    CsSetVectorData(obj, data);

    // value *p = CsVectorAddressI(obj);
    // while (--size >= 0)
    //    *p++ = UNDEFINED_VALUE;
#if defined(TISCRIPT_USE_PERSISTENCE)
    _CsInitPersistent(obj);
#endif
    return obj;
  }

  /* CsCloneVector - clone an existing vector */
  value CsCloneVector(VM *c, value obj, bool deep) {
    PROTECT(obj);
    FETCH(c, obj);
    int_t size = CsVectorSize(c, obj);
    value nobj = CsMakeVector(c, size, CsVectorClass(obj));
    if (deep) {
      value val = 0;
      PROTECT(nobj, val);
      while (--size >= 0) {
        val = CsVectorElementI(obj, size);
        val = CsClone(c, val, true);
        CsSetVectorElementI(nobj, size, val);
      }
    } else {
      value *src = CsVectorAddress(c, obj);
      value *dst = CsVectorAddress(c, nobj);
      while (--size >= 0)
        *dst++ = *src++;
    }
    return nobj;
  }

  value CsResizeVector(VM *c, value obj, int_t newSize) {
    FETCH(c, obj);
    return CsResizeVectorNoLoad(c, obj, newSize);
  }

  /* CsResizeVector - resize a vector */
  value CsResizeVectorNoLoad(VM *c, value obj, int_t newSize) {
    int_t size;
    // value resizeVector = obj;

    /* make sure the size is really changing */
    if ((size = CsVectorSizeI(obj)) != newSize) {

      /* check for extra existing space */
      if (newSize <= CsVectorMaxSize(obj)) {

        /* fill the extra space with nil */
        if (newSize > size) {
          value *dst = CsVectorAddressI(obj) + size;
          while (++size <= newSize)
            *dst++ = UNDEFINED_VALUE;
        }

        /* store the new vector size */
        CsSetVectorSize(obj, newSize);
      }

      /* expand the vector */
      else {
        value *src, *dst;
        int_t  allocSize;

        /* try expanding by a fraction of the current size */
        allocSize = size / CsVectorExpandDivisor;

        /* but make sure we expand by at least CsVectorExpandMinimum */
        if (allocSize < CsVectorExpandMinimum)
          allocSize = CsVectorExpandMinimum;

        /* and at most CsVectorExpandMaximum */
        if (allocSize > CsVectorExpandMaximum)
          allocSize = CsVectorExpandMaximum;

        /* but at least what we need */
        if ((allocSize += size) < newSize) allocSize = newSize;

        /* make a new vector data */
        CsCheck(c, 1);
        CsPush(c, obj);
        value new_data =
            CsMakeFixedVectorValue(c, &CsFixedVectorDispatch, allocSize);
        obj            = CsPop(c);
        value old_data = CsVectorData(obj);
        CsSetVectorData(obj, new_data);
        CsSetVectorSize(obj, newSize);
        // CsSetVectorMaxSize(obj,allocSize);

        /* copy the data from the old to the new vector */
        src = CsFixedVectorAddress(old_data);
        dst = CsFixedVectorAddress(new_data);
        target(dst, size).copy(src);
      }
    }

    /* return the resized vector */
    return obj;
  }

  /* CsVectorSize - get the size of a vector */
  int_t CsVectorSize(VM *c, value obj) {
    FETCH(c, obj);
    return CsVectorSizeI(obj);
  }

  /* CsVectorSize - get the size of a vector */
  int_t CsVectorSizeNoLoad(VM *c, value obj) { return CsVectorSizeI(obj); }

  /* CsVectorAddress - get the address of the vector data */
  value *CsVectorAddress(VM *c, value obj) {
    FETCH(c, obj);
    return CsVectorAddressI(obj);
  }

  /* CsVectorAddress - get the address of the vector data */
  value *CsVectorAddressNoLoad(VM *c, value obj) {
    return CsVectorAddressI(obj);
  }

  /* CsVectorElement - get a vector element */
  value CsVectorElement(VM *c, value obj, int_t i) {
    FETCH(c, obj);
    assert(i < CsVectorSize(c, obj));
    return CsVectorElementI(obj, i);
  }

  /* CsVectorElement - get a vector element */
  value CsVectorElementNoLoad(VM *c, value obj, int_t i) {
    return CsVectorElementI(obj, i);
  }

  /* CsSetVectorElement - set a vector element */
  void CsSetVectorElement(VM *c, value obj, int_t i, value val) {
    FETCH(c, obj);
    CsSetModified(obj, true);
    assert(i < CsVectorSize(c, obj));
    CsSetVectorElementI(obj, i, val);
  }

  void CsSetVectorElementNoLoad(VM *c, value obj, int_t i, value val) {
    CsSetVectorElementI(obj, i, val);
  }

    /* BASIC VECTOR */

#undef FETCH

  /* CsTupleSizeHandler - Tuple size handler */
  long CsTupleSizeHandler(value obj) {
    return sizeof(tuple) + CsTupleSize(obj) * sizeof(value);
  }

  value& CsTupleMeta(value tup) {
    return ptr<tuple>(tup)->meta;
  }

  /* CsTupleScanHandler - Tuple scan handler */
  void CsTupleScanHandler(VM *c, value obj) {
    //#ifdef _DEBUG
    //  string s = value_to_string(CsTupleName(obj));
    //  s = s;
    //#endif
    long sz = CsTupleSize(obj);
    for (long i = 0; i < sz; ++i)
      CsSetTupleElement(obj, i, CsCopyValue(c, CsTupleElement(obj, i)));
    CsSetTupleName(obj, CsCopyValue(c,CsTupleName(obj)));
    if (CsTupleMeta(obj)) CsTupleMeta(obj) = CsCopyValue(c, CsTupleMeta(obj));
  }

  /* CsMakeTuple - make a new vector value */
  value CsMakeTuple(VM *c, dispatch *type, int_t size) {
    //#ifdef _DEBUG
    //  if (size == 21)
    //    size = size;
    //#endif
    long   allocSize = sizeof(tuple) + size * sizeof(value);
    value  newo      = CsAllocate(c, allocSize);
    value *p         = CsTupleAddress(newo);
    CsSetDispatch(newo, type);
    CsSetTupleSize(newo, size);
    while (--size >= 0)
      *p++ = UNDEFINED_VALUE;
    assert(allocSize == ValueSize(newo));
    CsSetTupleName(newo, UNDEFINED_VALUE);
    return newo;
  }

  value CsMakeTuple(VM *c, int_t size) {
    return CsMakeTuple(c, &CsTupleDispatch, size);
  }

  bool CsTuplesEqual(VM* c, value o1, value o2, array<value> &stack) {
    assert(CsTupleP(o1) && CsTupleP(o2));
    if (!CsTupleP(o1) || !CsTupleP(o2)) return false;
    if (!CsEqualOp(c, CsTupleName(o1), CsTupleName(o2),stack)) return false;
    return CsTupleElements(o1).equal(CsTupleElements(o2), [&](value a, value b) {return CsEqualOp(c, a, b,stack); });
  }

  /* VectorNewInstance - create a new vector */
  static value TupleNewInstance(VM *c, value proto) {
    return CsMakeTuple(c, &CsTupleDispatch, 0);
  }

  /* GetTupleProperty - Tuple get property handler */
  static bool GetTupleProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->tupleObject, tag, pValue);
  }

  /* SetTupleProperty - Tuple set property handler */
  static bool SetTupleProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->tupleObject, tag, value);
  }

  static value TupleGetItem(VM *c, value obj, value tag) {
    if (CsIntegerP(tag)) {
      int_t i;
      if ((i = CsIntegerValue(tag)) < 0 || i >= CsTupleSize(obj))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      return CsTupleElement(obj, i);
    }
    return UNDEFINED_VALUE;
  }
  static void TupleSetItem(VM *c, value obj, value tag, value value) {
    if (CsIntegerP(tag)) {
      int_t i;
      if ((i = CsIntegerValue(tag)) < 0 || i >= CsTupleSize(obj))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      CsSetTupleElement(obj, i, value);
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag,
                        "only integer as an index");
  }

  bool TuplePrint(VM *c, value obj, stream *s, bool toLocale) {
    long    i, n;
    ustring name;
    if (CsTupleName(obj) != UNDEFINED_VALUE)
      name = value_to_string(CsTupleName(obj));

    s->put('[');
    s->put_str(name);
    s->put(':');
    s->put(' ');

    n = CsTupleSize(obj);
    if (n) {
      for (i = 0; i < n - 1; ++i) {
        CsDisplay(c, CsTupleElement(obj, i), s);
        s->put(',');
      }
      CsDisplay(c, CsTupleElement(obj, i), s);
    }
    s->put(']');
    return true;
  }

  value TupleNextElement(VM *c, value *index, value collection, int nr) {
    if (*index == NOTHING_VALUE) // first
    {
      // FETCH(c,collection);
      if (CsTupleSize(collection)) {
        *index = CsMakeInteger(0);
        //CsSetRVal(c, 1, *index);
        //return CsTupleElement(collection, 0);
        CS_RETURN2(c, *index, CsTupleElement(collection, 0));
      }
    } else if (CsIntegerP(*index)) {
      int_t i = CsIntegerValue(*index) + 1;
      *index  = CsMakeInteger(i);
      if (i < CsTupleSize(collection)) {
        //CsSetRVal(c, 1, *index);
        //return CsTupleElement(collection, i);
        CS_RETURN2(c, *index, CsTupleElement(collection, i));
      }
    } else
      assert(false);

    //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
    return NOTHING_VALUE;
  }

  /* Tuple pdispatch - fixed vector here */
  dispatch CsTupleDispatch = {"Tuple",
                              &CsTupleDispatch,
                              GetTupleProperty,
                              SetTupleProperty,
                              TupleNewInstance,
                              TuplePrint,
                              CsTupleSizeHandler,
                              CsDefaultCopy,
                              CsTupleScanHandler,
                              CsDefaultHash,
                              TupleGetItem,
                              TupleSetItem,
                              TupleNextElement};

  //metaAccessor

  /* CSF_tuple_name - built-in property 'name' */
  static value CSF_tuple_name(VM *c, value obj) { return CsTupleName(obj); }

  /* CSF_tuple_set_name - built-in property 'name' */
  static void CSF_tuple_set_name(VM *c, value obj, value val) {
    CsSetTupleName(obj, val);
  }

  /* CSF_tuple_length - built-in property 'length' */
  static value CSF_tuple_length(VM *c, value obj) {
    return CsMakeInteger(CsTupleSize(obj));
  }

  static value CSF_tuple_clone(VM *c) {
    value tuple = 0;
    PROTECT(tuple);

    CsParseArguments(c, "V=*", &tuple, &CsTupleDispatch);

    int_t d = CsTupleSize(tuple);

    value new_tuple = CsMakeTuple(c, d);

    for (int n = 0; n < d; ++n)
      CsSetTupleElement(new_tuple, n, CsTupleElement(tuple, n));
    CsSetTupleName(new_tuple, CsTupleName(tuple));

    return new_tuple;
  }

  static value _MakeTuple(VM *c, value tag, int_t n) {
    value t = CsMakeTuple(c, &CsTupleDispatch, n);
    CsSetTupleName(t, tag);
    return t;
  }

  /* CSF_tuple_ctor - built-in method 'this' */
  static value CSF_tuple_ctor(VM *c) {
    value tag, vec;
    CsParseArguments(c, "**VV", &tag, &vec);

    if (CsSymbolP(tag))
      ;
    else if (CsStringP(tag))
      tag = CsMakeSymbol(c, CsStringChars(tag));
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag, "symbol or string");

    value obj = NULL_VALUE;

    if (CsIntegerP(vec)) {
      obj = _MakeTuple(c, tag, CsIntegerValue(vec));
    } else if (CsVectorP(vec)) {
      PROTECT(obj, vec);
      int_t n = CsVectorSize(c, vec);
      obj     = _MakeTuple(c, tag, n);
      for (int_t i = 0; i < n; ++i) {
        value el = CsVectorElement(c, vec, i);
        CsSetTupleElement(obj, i, el);
      }
    } else if (CsTupleP(vec)) {
      PROTECT(obj, vec);
      int_t n = CsTupleSize(vec);
      obj     = _MakeTuple(c, tag, n);
      for (int_t i = 0; i < n; ++i) {
        value el = CsTupleElement(vec, i);
        CsSetTupleElement(obj, i, el);
      }
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag,
                        "integer, vector or tuple");

    CsCtorRes(c) = obj;
    return obj;
  }

  /*
  string_stream s;
      int i, argc = CsArgCnt(c);
      for( i = 3; i <= argc; ++i )
         if(i & 1)
           CsToString(c,CsGetArg(c,i),s);
         else
           CsToHtmlString(c,CsGetArg(c,i),s);

     


      bool r = pv->set_element_html(self, s.buf(), where);
  */

  static value CSF_tuple_toString(VM *c) {
    value tuple = 0;
    PROTECT(tuple);

    CsParseArguments(c, "V=*", &tuple, &CsTupleDispatch);

    int_t d = CsTupleSize(tuple);

    string_stream s;

    s.put_str(W("["));
    if (CsTupleName(tuple) != UNDEFINED_VALUE)
      CsToString(c, CsTupleName(tuple), s);
    s.put_str(W(":"));
    for (int n = 0; n < d; ++n) {
      if (n) s.put_str(W(","));
      CsToString(c, CsTupleElement(tuple, n), s);
    }
    s.put_str(W("]"));
    return s.string_o(c);
  }

  static value CSF_tuple_toHtmlString(VM *c) {
    value tuple = 0;
    PROTECT(tuple);

    CsParseArguments(c, "V=*", &tuple, &CsTupleDispatch);

    int_t d = CsTupleSize(tuple);

    string_stream s;

    s.put_str(W("["));
    if (CsTupleName(tuple) != UNDEFINED_VALUE)
      CsToHtmlString(c, CsTupleName(tuple), s);
    s.put_str(W(":"));
    for (int n = 0; n < d; ++n) {
      if (n) s.put_str(W(","));
      CsToHtmlString(c, CsTupleElement(tuple, n), s);
    }
    s.put_str(W("]"));
    return s.string_o(c);
  }

  static value CSF_tuple_toCssString(VM *c) {
    value tuple = 0;
    PROTECT(tuple);

    CsParseArguments(c, "V=*", &tuple, &CsTupleDispatch);

    if (CsTupleName(tuple) == UNDEFINED_VALUE || !CsSymbolP(CsTupleName(tuple)))
      return CSF_tuple_toString(c);

    int_t d = CsTupleSize(tuple);

    string_stream s;

    if (CsTupleName(tuple) != UNDEFINED_VALUE)
      CsToString(c, CsTupleName(tuple), s);
    s.put_str(W("("));
    for (int n = 0; n < d; ++n) {
      if (n) s.put_str(W(","));
      value element = CsTupleElement(tuple, n);
      if (CsVectorP(element)) {
        int i, n = CsVectorSize(c, element); 
        for (i = 0; i < n - 1; ++i) {
          CsToString(c, CsVectorElement(c, element, i), s);
          s.put_str(" ");
        }
        if (i < n) CsToString(c, CsVectorElement(c, element, i), s);
      }
      else
        CsToString(c, CsTupleElement(tuple, n), s);
    }
    s.put_str(W(")"));
    return s.string_o(c);
  }

  /* Tuple methods */
  static c_method tuple_methods[] = {
      C_METHOD_ENTRY("this", CSF_tuple_ctor),
      C_METHOD_ENTRY("clone", CSF_tuple_clone),
      C_METHOD_ENTRY("toString", CSF_tuple_toString),
      C_METHOD_ENTRY("toHtmlString", CSF_tuple_toHtmlString),
      C_METHOD_ENTRY("toCssString", CSF_tuple_toCssString),
      C_METHOD_ENTRY(0, 0)};

  /* Tuple properties */
  static vp_method tuple_properties[] = {
      VP_METHOD_ENTRY("length", CSF_tuple_length, 0),
      VP_METHOD_ENTRY("tag", CSF_tuple_name, CSF_tuple_set_name),
      VP_METHOD_ENTRY(0, 0, 0)};

  // value CsMakeTuple(VM *c, int_t size)
  //{
  //  return CsMakeTuple(c, &CsTupleDispatch,size);
  //}

  value CsMakeTuple(VM *c, const char *tag, value v1) {
    PROTECT(v1);
    value t = CsMakeTuple(c, &CsTupleDispatch, 1);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    return t;
  }
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2) {
    PROTECT(v1, v2);
    value t = CsMakeTuple(c, &CsTupleDispatch, 2);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    CsSetTupleElement(t, 1, v2);
    return t;
  }
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3) {
    PROTECT(v1, v2, v3);
    value t = CsMakeTuple(c, &CsTupleDispatch, 3);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    CsSetTupleElement(t, 1, v2);
    CsSetTupleElement(t, 2, v3);
    return t;
  }
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4) {
    PROTECT(v1, v2, v3, v4);
    value t = CsMakeTuple(c, &CsTupleDispatch, 4);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    CsSetTupleElement(t, 1, v2);
    CsSetTupleElement(t, 2, v3);
    CsSetTupleElement(t, 3, v4);
    return t;
  }
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4, value v5) {
    PROTECT(v1, v2, v3, v4, v5);
    value t = CsMakeTuple(c, &CsTupleDispatch, 5);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    CsSetTupleElement(t, 1, v2);
    CsSetTupleElement(t, 2, v3);
    CsSetTupleElement(t, 3, v4);
    CsSetTupleElement(t, 4, v5);
    return t;
  }
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4, value v5, value v6) {
    PROTECT(v1, v2, v3, v4, v5, v6);
    value t = CsMakeTuple(c, &CsTupleDispatch, 6);
    CsSetTupleName(t, CsSymbolOf(tag));
    CsSetTupleElement(t, 0, v1);
    CsSetTupleElement(t, 1, v2);
    CsSetTupleElement(t, 2, v3);
    CsSetTupleElement(t, 3, v4);
    CsSetTupleElement(t, 4, v5);
    CsSetTupleElement(t, 5, v6);
    return t;
  }

  /* CsInitTuple - initialize the 'Tuple' obj */
  void CsInitTuple(VM *c) {
    CsTupleDispatch.metaAccessor = CsTupleMeta;
    c->tupleObject = CsEnterType(CsGlobalScope(c), "Tuple", &CsTupleDispatch);
    CsEnterMethods(c, c->tupleObject, tuple_methods);
    CsEnterVPMethods(c, c->tupleObject, tuple_properties);
  }

} // namespace tis
