/*
        Copyright (c) 2001-2017 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"

#include <stdlib.h>
#include <stdio.h>

#if defined(ANDROID)
  
#elif !defined(WINDOWS)
  #include <spawn.h>
#endif

#include "cs_async.h"

namespace tis {

  /* 'System' pdispatch */

  void checkFsError(VM *c, int r)
  {
    if (r < 0)
      CsThrowKnownError(c, CsErrGenericError, tool::string::format("file I/O:%s", uv_strerror(r)).c_str());
  }

  ustring getFsError(int r)
  {
    if (r < 0)
      return uv_strerror(r);
    return ustring();
  }


  /* method handlers */
  static value CSF_scanFiles(VM *c);
  static value CSF_dir(VM *c);
  static value CSF_home(VM *c);
  static value CSF_path(VM *c);
  static value CSF_exec(VM *c);
  static value CSF_watch(VM *c);
  static value CSF_unlink(VM *c); static value CSF_unlinkSync(VM *c);
  static value CSF_rmdir(VM *c); static value CSF_rmdirSync(VM *c);
  static value CSF_mkdir(VM *c); static value CSF_mkdirSync(VM *c);
  static value CSF_rename(VM *c); static value CSF_renameSync(VM *c);
  static value CSF_copyFile(VM *c); static value CSF_copyFileSync(VM *c);
  static value CSF_stat(VM *c); static value CSF_statSync(VM *c);
  static value CSF_chmod(VM *c); static value CSF_chmodSync(VM *c);
  static value CSF_utimes(VM *c); static value CSF_utimesSync(VM *c);
  static value CSF_md5(VM *c);
  static value CSF_parameters(VM *c);

  static value CSF_userLang(VM *c, value obj) {
    tool::ustring lang;
    tool::ustring country;
    if (tool::environment::used_lang_country(lang, country, true))
      return CsSymbolOf(lang);
    return UNDEFINED_VALUE;
  }

  static value CSF_userCountry(VM *c, value obj) {
    tool::ustring lang;
    tool::ustring country;
    if (tool::environment::used_lang_country(lang, country, true))
      return CsSymbolOf(country);
    return UNDEFINED_VALUE;
  }

  static value CSF_ticks(VM *c, value obj) {
    return CsMakeInteger(tool::get_ticks());
  }
  
  /* System methods */
  static c_method methods[] = {C_METHOD_ENTRY("scanFiles", CSF_scanFiles),
                               C_METHOD_ENTRY("dir", CSF_dir),
                               C_METHOD_ENTRY("home", CSF_home),
                               C_METHOD_ENTRY("path", CSF_path),
                               C_METHOD_ENTRY("exec", CSF_exec),
                               C_METHOD_ENTRY("watch", CSF_watch),
                               C_METHOD_ENTRY("unlink", CSF_unlink),
                               C_METHOD_ENTRY("rmdir", CSF_rmdir),
                               C_METHOD_ENTRY("mkdir", CSF_mkdir),
                               C_METHOD_ENTRY("rename", CSF_rename),
                               C_METHOD_ENTRY("copyFile", CSF_copyFile),
                               C_METHOD_ENTRY("stat", CSF_stat),
                               C_METHOD_ENTRY("chmod", CSF_chmod),
                               C_METHOD_ENTRY("utimes", CSF_utimes),
                               C_METHOD_ENTRY("md5", CSF_md5),
                               C_METHOD_ENTRY("parameters", CSF_parameters),
                               
                               C_METHOD_ENTRY(0, 0)};

  static c_method sync_methods[] = { 
    //C_METHOD_ENTRY("scanFiles", CSF_scanFiles),
    //C_METHOD_ENTRY("dir", CSF_dir),
    //C_METHOD_ENTRY("home", CSF_home),
    //C_METHOD_ENTRY("path", CSF_path),
    //C_METHOD_ENTRY("exec", CSF_exec),
    //C_METHOD_ENTRY("watch", CSF_watch),
    C_METHOD_ENTRY("unlink", CSF_unlinkSync),
    C_METHOD_ENTRY("rmdir", CSF_rmdirSync),
    C_METHOD_ENTRY("mkdir", CSF_mkdirSync),
    C_METHOD_ENTRY("rename", CSF_renameSync),
    C_METHOD_ENTRY("copyFile", CSF_copyFileSync),
    C_METHOD_ENTRY("stat", CSF_statSync),
    C_METHOD_ENTRY("chmod", CSF_chmodSync),
    C_METHOD_ENTRY("utimes", CSF_utimesSync),
    C_METHOD_ENTRY("md5", CSF_md5),

    C_METHOD_ENTRY(0, 0) };


  /* System properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY("language", CSF_userLang, 0),
      VP_METHOD_ENTRY("country", CSF_userCountry, 0),
      VP_METHOD_ENTRY("ticks", CSF_ticks, 0), VP_METHOD_ENTRY(0, 0, 0)};

#if defined(WINDOWS)
#define platform "Windows"
#elif defined(OSX)
#define platform "OSX"
#elif defined(LINUX)
#define platform "Linux"
#elif defined(ANDROID)
#define platform "Android"
#elif defined(WINDOWLESS)
#define platform "IoT"
#else
#define platform "undefined"
#endif

  static constant constants[] = {
      CONSTANT_ENTRY("IS_READONLY", int_value(tool::filesystem::FA_READONLY)),
      CONSTANT_ENTRY("IS_DIR", int_value(tool::filesystem::FA_DIR)),
      CONSTANT_ENTRY("IS_HIDDEN", int_value(tool::filesystem::FA_HIDDEN)),
      CONSTANT_ENTRY("IS_SYSTEM", int_value(tool::filesystem::FA_SYSTEM)),

#ifdef PLATFORM_HANDHELD
      CONSTANT_ENTRY("MOBILE_OS", TRUE_VALUE),
      CONSTANT_ENTRY("DESKTOP_OS", FALSE_VALUE),
#else
      CONSTANT_ENTRY("MOBILE_OS", FALSE_VALUE),
      CONSTANT_ENTRY("DESKTOP_OS", TRUE_VALUE),
#endif
      CONSTANT_ENTRY("OS", CsSymbolOf(tool::environment::get_os_version_name())),
      CONSTANT_ENTRY("PLATFORM", CsSymbolOf(platform)),

      CONSTANT_ENTRY(0, 0)};

#undef platform


  struct folder_entry {
    uint          flags;
    tool::ustring name;
    tool::ustring caption;

    int  is_file() const { return (flags & tool::filesystem::FA_DIR) ? 0 : 1; }
    bool operator<(const folder_entry &rs) const {
      if (is_file() < rs.is_file()) return true;
      if (is_file() > rs.is_file()) return false;
      return name < rs.name;
    }
  };


  static value CSF_scanFiles(VM *c) 
  {
    if (!c->can_use_feature(FEATURE_FILE_IO))
    {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
      // return UNDEFINED_VALUE;
    }

    wchars path;
    value  ef   = 0;

    CsParseArguments(c, "**S#|m", &path.start,&path.length, &ef);

    int counter = 0;
    if (ef) {
      // pvalue

      tool::array<folder_entry> entries;

      auto account_item = [&entries](wchars name, wchars caption,
                                     uint flags) -> bool {
        folder_entry fe;
        fe.flags   = flags;
        fe.name    = name;
        fe.caption = caption;
        // fe.name.replace('\\','/');
        entries.push(fe);
        return true;
      };

      // folder_content_list fcl(entries);
      // scanEntryProxy sep( c, ef );
      value name = UNDEFINED_VALUE, caption = UNDEFINED_VALUE;

      PROTECT(ef, name, caption);

      counter = tool::filesystem::scan(ustring(path), account_item);
      tool::sort(entries.head(), entries.size());
      for (int n = 0; n < entries.size(); ++n) {
        const folder_entry &fe = entries[n];
        name                   = CsMakeCString(c, fe.name);
        caption = fe.caption.length() ? CsMakeCString(c, fe.caption) : name;
        value r = CsCallFunction(CsCurrentScope(c), ef, 3, name,
                                 CsMakeInteger(fe.flags), caption);
        if (CsToBoolean(c, r) != TRUE_VALUE) break;
      }
    } else {
      auto dummy = [](wchars name, wchars caption, uint flags) -> bool {
        return true;
      };
      counter = tool::filesystem::scan(ustring(path), dummy);
    }
    return CsMakeInteger(counter);
  }

  tool::value stat_to_value(const uv_stat_t& st)
  {
    tool::value stats = tool::value::make_map();
    stats.set_prop("dev", tool::value((int)st.st_dev));
    stats.set_prop("ino", tool::value((int)st.st_ino));
    stats.set_prop("mode", tool::value((int)st.st_mode));
    stats.set_prop("nlink", tool::value((int)st.st_nlink));
    stats.set_prop("uid", tool::value((int)st.st_uid));
    stats.set_prop("gid", tool::value((int)st.st_gid));
    stats.set_prop("rdev", tool::value((int)st.st_rdev));
    stats.set_prop("size", tool::value((double)st.st_size));
    stats.set_prop("blksize", tool::value((int)st.st_blksize));
    stats.set_prop("blocks", tool::value((double)st.st_blocks));
    stats.set_prop("atime", tool::value(tool::filesystem::uv_time_cvt(st.st_atim)));
    stats.set_prop("mtime", tool::value(tool::filesystem::uv_time_cvt(st.st_mtim)));
    stats.set_prop("ctime", tool::value(tool::filesystem::uv_time_cvt(st.st_ctim)));
    stats.set_prop("birthtime", tool::value(tool::filesystem::uv_time_cvt(st.st_birthtim)));
    return stats;
  }

  static value CSF_dir(VM *c)
  {
    if (!c->can_use_feature(FEATURE_FILE_IO))
    {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
      // return UNDEFINED_VALUE;
    }

    static value dirent_sym_type[] = {
      CsSymbolOf("unknown"),
      CsSymbolOf("file"),
      CsSymbolOf("dir"),
      CsSymbolOf("link"),
      CsSymbolOf("fifo"),
      CsSymbolOf("socket"),
      CsSymbolOf("char"),
      CsSymbolOf("block") };
      //return syms[type];

    wchars path;
    value  ef = 0;

    CsParseArguments(c, "**S#m", &path.start, &path.length, &ef);

    bool wants_stats = CsMethodArgCnt(ef) >= 3;

    value vname = 0;
    value vtype = 0;
    value vstat = 0;

    PROTECT(ef,vname);

    auto_scope as(c, CsMethodNamespace(ef));

    auto callback = [&](const tool::filesystem::dir_entry& de) {
      vname = CsMakeString(c, de.name);
      vtype = dirent_sym_type[de.type];
      if (wants_stats) 
      {
        tool::value tvstat = stat_to_value(de.stat);
        if (de.caption.is_defined())
          tvstat.set_prop("caption", tool::value(de.caption));
        vstat = value_to_value(c, tvstat);
        CsCallFunction(&as, ef, 3, vname, vtype, vstat);
      }
      else 
        CsCallFunction(&as, ef, 2, vname, vtype);
    };

    //string utf8 = url::file_url_to_u8_path(path);
    tool::filesystem::result_t r = tool::filesystem::scan_dir(path,callback,wants_stats);
    checkFsError(c, r);
    return TRUE_VALUE;
  }


  static value CSF_home(VM *c) {
    if ((c->features & FEATURE_SYSINFO) == 0) {
      CsThrowKnownError(c, CsErrNotAllowed, "SYS INFO");
      // return UNDEFINED_VALUE;
    }

    wchar *path = 0;
    CsParseArguments(c, "**|S", &path);

    tool::ustring p = tool::get_home_dir();

    if (path) {
      if (*path == '\\' || *path == '/') path += 1;
      p += path;
    }

    p.replace_all('\\', '/');

    return CsMakeCString(c, p);
  }

  static value CSF_path(VM *c) {
    if ((c->features & FEATURE_SYSINFO) == 0) {
      CsThrowKnownError(c, CsErrNotAllowed, "SYS INFO");
      // return UNDEFINED_VALUE;
    }

    symbol_t sym  = 0;
    wchar *  path = 0;
    CsParseArguments(c, "**L|S", &sym, &path);

    tool::string s = symbol_name(sym);
    auto sd = tool::parse_standard_dir(s);

    if (!sd) {
      tool::string msg = tool::string::format("#%s is not a valid name", s.c_str());
      CsThrowKnownError(c, CsErrGenericError, msg.c_str());
    }

    tool::ustring p = tool::get_standard_dir(sd);

    if (path) {
      if (*path != '\\' && *path != '/') p += '/';
      p += path;
    }

    p.replace_all('\\', '/');

    if (path)
      return CsMakeString(c, normalize_path(p())());
    else
      return CsMakeString(c, p());
  }

  static value CSF_exec(VM *c) 
  {

//#if defined(PLATFORM_DESKTOP) 

    if ((c->features & FEATURE_SYSINFO) == 0) {
      CsThrowKnownError(c, CsErrNotAllowed, "SYS INFO");
      // return UNDEFINED_VALUE;
    }

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);

    int argc = CsArgCnt(c);

#ifdef WINDOWS
    ustring        spath = path;
    array<ustring> args;
#else
    string        spath = u8::cvt(path);
    array<string> args;
#endif

    auto wrap_spaced_string = [](ustring in) -> ustring {
      if (in().contains_one_of(WCHARS(" \r\n\t")) && in[0] != '"')
        return ustring::format(W("\"%s\""), in.c_str());
      //      else if (in().contains('\"'))
      //        in.replace(W("\""), W("\\\""));
      return in;
    };

    args.push(wrap_spaced_string(spath));

    for (int narg = 4; narg <= argc; ++narg) {
      value arg = CsGetArg(c, narg);
      if (!CsStringP(arg))
        CsThrowKnownError(c, CsErrUnexpectedTypeError, arg, "string only");
      tool::ustring us = value_to_string(arg);
      args.push(wrap_spaced_string(us));
    }
    int r = 0;
#ifdef WINDOWS
    array<const wchar *> argv;
    for (int i = 0; i < args.size(); ++i) {
      const wchar *parg = args[i].c_str();
      argv.push(parg);
    }
    argv.push(nullptr);
    r = (int)_wspawnvp(_P_NOWAIT, spath.c_str(), argv.cbegin());
#else
    array<char *> argv;
    for (int i = 0; i < args.size(); ++i)
      argv.push(args[i].buffer());
    argv.push(nullptr);
    pid_t processID = 0;
    r = posix_spawn(&processID, spath.c_str(), nullptr, nullptr, argv.begin(),
                    nullptr);
#endif
    if (r == -1)
      r = errno;
    else
      r = 0;
    return CsMakeInteger(r);
//#else 
//    CsThrowKnownError(c, CsErrNotAllowed, "SYS INFO");
//    return UNDEFINED_VALUE;
//#endif
  }

  class fs_operation: 
    public async::fs_req,
    public thenable_async_object<fs_operation,VM,false>
  {
  public:

    fs_operation() {}

    uv_loop_t* loop() { return req.loop; }
    uv_fs_t*   preq() { return &req; }

    virtual void close() {
      //uv.req ???
    }

    virtual void on_done(const uv_fs_t& req)
    {
      switch (req.fs_type) {
        case UV_FS_REALPATH: notify_completion(tool::value(u8::cvt(chars_of(req.path))), true); break;
        case UV_FS_STAT: {
          tool::value stats = stat_to_value(req.statbuf);
          notify_completion(stats, true); 
        } break;

        default: notify_completion(tool::value(true), true); break;
      }
    }
    virtual void on_error(const uv_fs_t& req) 
    {
      notify_completion(tool::value(getFsError(int(req.result))),false);
    }
    
    static dispatch*   this_dispatch(VM* c) { return c->fileOperationDispatch; }
    static const char* type_name() { return "FileOperation"; }

  };
      
  static value CSF_unlink(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();
    
    int r = uv_fs_unlink(fso->loop(), fso->preq(), utf8, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c,fso);
  }

  static value CSF_unlinkSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_unlink(fso->loop(), fso->preq(), utf8, nullptr);
    //checkFsError(c, r);
    //return fs_operation::this_object(c, fso);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }


  static value CSF_rmdir(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_rmdir(fso->loop(), fso->preq(), utf8, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_rmdirSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_rmdir(fso->loop(), fso->preq(), utf8, nullptr);
    //checkFsError(c, r);
    //return fs_operation::this_object(c, fso);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_rename(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path, topath;
    CsParseArguments(c, "**S#S#", &path.start, &path.length,&topath.start, &topath.length);
    string utf8 = url::file_url_to_u8_path(path);
    string utf8to = url::file_url_to_u8_path(topath);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_rename(fso->loop(), fso->preq(), utf8, utf8to, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_renameSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path, topath;
    CsParseArguments(c, "**S#S#", &path.start, &path.length, &topath.start, &topath.length);
    string utf8 = url::file_url_to_u8_path(path);
    string utf8to = url::file_url_to_u8_path(topath);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_rename(fso->loop(), fso->preq(), utf8, utf8to, nullptr);
    //checkFsError(c, r);
    //return fs_operation::this_object(c, fso);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }
    
  static value CSF_copyFile(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path, topath;
    value  excl = 0;
    CsParseArguments(c, "**S#S#|V", &path.start, &path.length, &topath.start, &topath.length, &excl);
    string utf8 = url::file_url_to_u8_path(path);
    string utf8to = url::file_url_to_u8_path(topath);

    handle<fs_operation> fso = new fs_operation();

    int flags = 0;
    if (excl == CsSymbolOf("no-overwrite"))
      flags = UV_FS_COPYFILE_EXCL;

    int r = uv_fs_copyfile(fso->loop(), fso->preq(), utf8, utf8to, flags, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_copyFileSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path, topath;
    value  excl = 0;
    CsParseArguments(c, "**S#S#|V", &path.start, &path.length, &topath.start, &topath.length, &excl);
    string utf8 = url::file_url_to_u8_path(path);
    string utf8to = url::file_url_to_u8_path(topath);

    handle<fs_operation> fso = new fs_operation();

    int flags = 0;
    if (excl == CsSymbolOf("no-overwrite"))
      flags = UV_FS_COPYFILE_EXCL;

    int r = uv_fs_copyfile(fso->loop(), fso->preq(), utf8, utf8to, flags, nullptr);
    //checkFsError(c, r);
    //return fs_operation::this_object(c, fso);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }


  static value CSF_chmod(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    int    mode;
    CsParseArguments(c, "**S#|i", &path.start, &path.length,&mode);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_chmod(fso->loop(), fso->preq(), utf8, mode, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_chmodSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    int    mode;
    CsParseArguments(c, "**S#|i", &path.start, &path.length, &mode);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_chmod(fso->loop(), fso->preq(), utf8, mode, nullptr);
    //checkFsError(c, r);
    //return fs_operation::this_object(c, fso);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }


  static value CSF_stat(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_stat(fso->loop(), fso->preq(), utf8, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_statSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_stat(fso->loop(), fso->preq(), utf8, nullptr);
    if (r >= 0) {
      auto tv = stat_to_value(fso->preq()->statbuf);
      return value_to_value(c, tv);
    }
    return NULL_VALUE;
  }


  static value CSF_mkdir(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    int    mode = 0777;
    CsParseArguments(c, "**S#|i", &path.start, &path.length, &mode);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    int r = uv_fs_mkdir(fso->loop(), fso->preq(), utf8, mode, fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_mkdirSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    int    mode = 0777;
    CsParseArguments(c, "**S#|i", &path.start, &path.length, &mode);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();
    int r = uv_fs_mkdir(fso->loop(), fso->preq(), utf8, mode, nullptr);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }


  static value CSF_utimes(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    value  vatime;
    value  vmtime;
    CsParseArguments(c, "**S#V=V=", &path.start, &path.length, &vatime, c->dateDispatch, &vmtime, c->dateDispatch);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    tool::date_time atime = CsDateValue(c,vatime);
    tool::date_time mtime = CsDateValue(c, vmtime);

    int r = uv_fs_utime(fso->loop(), fso->preq(), utf8, atime.seconds_since_1970(), mtime.seconds_since_1970(), fso->fs_cb);
    checkFsError(c, r);

    return fs_operation::this_object(c, fso);
  }

  static value CSF_utimesSync(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    value  vatime;
    value  vmtime;
    CsParseArguments(c, "**S#V=V=", &path.start, &path.length, &vatime, c->dateDispatch, &vmtime, c->dateDispatch);
    string utf8 = url::file_url_to_u8_path(path);

    handle<fs_operation> fso = new fs_operation();

    tool::date_time atime = CsDateValue(c, vatime);
    tool::date_time mtime = CsDateValue(c, vmtime);

    int r = uv_fs_utime(fso->loop(), fso->preq(), utf8, atime.seconds_since_1970(), mtime.seconds_since_1970(), nullptr);
    return r >= 0 ? TRUE_VALUE : FALSE_VALUE;
  }


  static value CSF_md5(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");

    wchars path;
    CsParseArguments(c, "**S#", &path.start, &path.length);
    ustring upath = url::file_url_to_path(path);

    tool::mm_file f;
    if (f.open(upath)) {
      tool::string str = md5(f.bytes()).to_string();
      return string_to_value(c, str);
    }
    return NULL_VALUE;
  }


  static value CSF_parameters(VM *c) {
    if (!c->can_use_feature(FEATURE_SYSINFO))
      CsThrowKnownError(c, CsErrNotAllowed, "SYSINFO");

#if defined(WINDOWS)
    //SystemParametersInfo()
    value op = 0;
    value val = 0;
    bool toset = false;
    wchars path;
    CsParseArguments(c, "**V=S#|V",&op,&CsSymbolDispatch,&path.start, &path.length,&val);
    toset = op == CsSymbolOf(WCHARS("set"));
    //ustring upath = url::file_url_to_path(path);
    if (path == WCHARS("desktop.wallpaper")) {
      if (toset) {
        if(!CsStringP(val)) CsThrowKnownError(c, CsErrUnexpectedTypeError, val, "string");
        SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, (LPVOID)value_to_string(c,val).c_str(), 0);
        return TRUE_VALUE;
      }
      else {
        WCHAR path[MAX_PATH];
        if (SystemParametersInfo(SPI_GETDESKWALLPAPER, MAX_PATH, path, 0))
          return CsMakeCString(c, path);
        return NULL_VALUE;
      }

    }



#else 
#endif

    return NULL_VALUE;
  }


// System.FileMonitor
  
  class monitor : public async_object<monitor,VM>, public tool::filesystem::monitor {
    typedef tool::filesystem::monitor super;
  public:

    monitor(): super( tool::async::dispatch::current() ) {}
      
    virtual ~monitor() {
        close();
    }

    virtual void stop() { 
      close(); 
      detach(); 
    }

    virtual void handle_change(const ustring& filename, bool changed, bool renamed)
    {
      if (changed)
        notify(WCHARS("change"), string_to_value(get_vm(), filename));
      if (renamed)
        notify(WCHARS("rename"), string_to_value(get_vm(), filename));
    }
    virtual void handle_error(int status)
    {
       notify(WCHARS("error"), CsMakeInteger(status));
    }

    static value CSF_close(VM *c) {
      value obj;
      CsParseArguments(c, "V=*", &obj, c->fileMonitorDispatch);
      monitor *mo = monitor::object_ptr(c, obj);
      if(mo)
        mo->stop();
      return UNDEFINED_VALUE;
    }

    static value CSF_path(VM *c, value obj) {
      monitor *mo = monitor::object_ptr(c, obj);
      return string_to_value(c,mo->get_path());
    }

    //traits:
    static dispatch*   this_dispatch(VM* c) { return c->fileMonitorDispatch; }
    static const char* type_name() { return "FileMonitor"; }

    static c_method  methods[];
    static vp_method properties[];
    //static constant  constants[];
  };

  c_method  monitor::methods[] = {
    C_METHOD_ENTRY("close", monitor::CSF_close),
    C_METHOD_ENTRY(0, 0)
  };

  vp_method  monitor::properties[] = {
    VP_METHOD_ENTRY("path", monitor::CSF_path, 0),
    VP_METHOD_ENTRY(0, 0, 0)
  };


  static value CSF_watch(VM *c) 
  {
    if (!c->can_use_feature(FEATURE_FILE_IO))
    {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
    }
    wchars path;

    CsParseArguments(c, "**S#", &path.start, &path.length);

    handle<monitor> mon = new monitor();
    string utf8 = url::file_url_to_u8_path(path);
    int r = mon->watch(utf8);
    checkFsError(c, r);
    return monitor::this_object(c, mon);
  }


// System.Process 

#if defined(PROCESS_SUPPORT)

  struct process : public tool::process, public thenable_async_object<process, VM>
  {
    typedef tool::process super;

    process() : super(tool::async::dispatch::current()) {}

    TOOL_INTERFACE_NAME(tis, process)

    bool send(value data) {
      ustring us = value_to_string(data);
      if (!us.length()) return false;
      return super::send(us());
    }

    virtual void on_stdout(wchars text) override {
      notify(WCHARS("stdout"), CsMakeString(get_vm(), text));
    }
    virtual void on_stderr(wchars text) override {
      notify(WCHARS("stderr"), CsMakeString(get_vm(), text));
    }
    virtual void on_stdin(int status) override {
      notify(WCHARS("stdin"), CsMakeInteger(status));
    }

    virtual void on_terminate(int status) override {
      notify(WCHARS("terminate"), CsMakeInteger(status));
      notify_completion(tool::value(status), status >= 0);
      detach();
    }

    static value CSF_running(VM *c, value obj) {
      process *pr = process::object_ptr(c, obj);
      if (!pr) return FALSE_VALUE;
      return TRUE_VALUE;
    }

    static value CSF_status(VM *c, value obj) {
      process *pr = process::object_ptr(c, obj);
      if (!pr) return UNDEFINED_VALUE;
      return CsMakeInteger(pr->status);
    }

    static value CSF_terminate(VM *c) {
      value obj;
      CsParseArguments(c, "V=*", &obj, c->processDispatch);
      process *pr = process::object_ptr(c, obj);
      if (!pr)
        CsThrowKnownError(c, CsErrGenericError, "inactive process");
      else
        pr->terminate();
      return obj;
    }

    static value CSF_send(VM *c) {
      value data;
      value obj;
      CsParseArguments(c, "V=*V", &obj, c->processDispatch, &data);
      process *pr = process::object_ptr(c, obj);
      if (!pr)
        CsThrowKnownError(c, CsErrGenericError, "inactive process");
      else {
        PROTECT(obj);

        if (pr->is_sending())
          CsThrowKnownError(c, CsErrGenericError, "previous send is pending");
        
        if(!pr->send(data))
          CsThrowKnownError(c, CsErrGenericError, "send failure");
      }
      return obj;
    }

    static value CSF_exec(VM *c) {

      if (!c->can_use_feature(FEATURE_FILE_IO))
      {
        CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
      }

      bool   detached = false;

      value mode = 0;

      value  vargs = 0;
      wchars path;

      CsParseArguments(c, "**S#|V=|V=", &path.start, &path.length, &vargs, &CsVectorDispatch, &mode, &CsSymbolDispatch);
      if(mode && mode == CsSymbolOf("detached"))
        detached = true;

      handle<process> pr = new process();

      PROTECT(vargs);
        
      //if (!pr) CsThrowKnownError(c, CsErrGenericError, "inactive process");

      array<ustring> args;

      if(vargs)
        for (int narg = 0; narg < CsVectorSize(c,vargs); ++narg) {
          value arg = CsVectorElement(c, vargs, narg);
          if (!CsStringP(arg))
            CsThrowKnownError(c, CsErrUnexpectedTypeError, arg, "string only");
          tool::ustring us = value_to_string(arg);
          args.push(us);
        }

      //if (!pr->exec(path, args(), detached))
      //  CsThrowKnownError(c, CsErrGenericError, "exec failure");
      if (!pr->exec(path, args(), detached))
          return NULL_VALUE;

      return process::this_object(c, pr);
    }


    //traits:
    static dispatch*   this_dispatch(VM* c) { return c->processDispatch; }
    static const char* type_name() { return "Process"; }

    static c_method  methods[];
    static vp_method properties[];

  };
    
  /* element methods */
  c_method process::methods[] = { 
    C_METHOD_ENTRY("exec", CSF_exec),
    C_METHOD_ENTRY("send", CSF_send),
    C_METHOD_ENTRY("terminate", CSF_terminate),
    C_METHOD_ENTRY(0, 0) };

  /* Element properties */
  vp_method process::properties[] = { 
    VP_METHOD_ENTRY("running", CSF_running, 0),
    VP_METHOD_ENTRY("status", CSF_status, 0),
    VP_METHOD_ENTRY(0, 0, 0) };
#endif



  /* CsInitSystem - initialize the 'System' obj */
  void CsInitSystem(VM *c) {
    /* create the 'System' type */
    c->systemDispatch = CsEnterCPtrObjectType(CsGlobalScope(c), "System",methods, properties);
    if (!c->systemDispatch)
      CsInsufficientMemory(c);
    else {
      CsEnterConstants(c, c->systemDispatch->obj, constants);
      CsAddConstant(c, c->systemDispatch->obj, CsSymbolOf("EOL"),
        CsMakeString(c, WCHARS(PLATFORM_EOL)));

      auto_scope as(c, c->systemDispatch->obj);
      c->fileMonitorDispatch = monitor::init_class(c, monitor::methods, monitor::properties, nullptr);
#if defined(PROCESS_SUPPORT)
      c->processDispatch = process::init_class(c, process::methods, process::properties, nullptr);
#endif
      c->fileOperationDispatch = fs_operation::init_class(c, nullptr, nullptr, nullptr);

      value SyncNS = CsNewNamespaceInstance(c, UNDEFINED_VALUE, CsSymbolOf("sync"));
      CsAddConstant(c, c->systemDispatch->obj, CsSymbolOf("sync"), SyncNS);
      CsEnterMethods(c, SyncNS, sync_methods);


    }
  }

  
} // namespace tis
