/* symbol.c - 'CsSymbol' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <string.h>
#include "cs.h"

namespace tis {

  static const char *well_known_symbols[] = {
      // CsTypeOf set of values
      "void",      // S_NOTHING = 1, 
      "undefined", // S_UNDEFINED,
      "null",      // S_NULL,
      "true",      // S_TRUE_,
      "false",     // S_FALSE_,
      "prototype", // S_PROTOTYPE,
      "toString",  // S_TO_STRING,
      "valueOf",   // S_VALUE_OF,
      "this",      // S_THIS,
      "any",       // S_ANY,
      "#NaN",      // S_NAN,
      "#Inf",      // S_INF,
      "constructor", // S_CONSTRUCTOR
      "boolean",   // S_BOOLEAN,
      "integer",   // S_INTEGER,
      "float",     // S_FLOAT,
      "string",    // S_STRING,
      "date",      // S_DATE,
      "array",     // S_ARRAY,
      "object",    // S_OBJECT,
      "class",     // S_CLASS,
      "symbol",    // S_SYMBOL,
      "function",  // S_FUNCTION,
      "color",     // S_COLOR,
      "length",    // S_LENGTH,
      "duration",  // S_DURATION,
      "angle",     // S_ANGLE,
      "storage",   // S_STORAGE,
      "index",     // S_INDEX,
      "tuple",     // S_TUPLE,
      "namespace", // S_NAMESPACE,
  };

  tool::mutex &symbol_table_guard() {
    static tool::mutex _symbol_table_guard;
    return _symbol_table_guard;
  }

  static ustring get_symbol(uint idx) {
    tool::critical_section cs(symbol_table_guard());
    return symbol_table().name(idx);
  }

  symtab &symbol_table() {
    static symtab          _symbol_table;
    tool::critical_section cs(symbol_table_guard());
    if (_symbol_table.size() == 0) {
      for (int n = 0; n < (int)items_in(well_known_symbols); ++n) {
        uint i = _symbol_table[well_known_symbols[n]];
        i = i;
        //const wchar* str = _symbol_table( i );
        //dbg_printf("%d %S\n", i, str);
      }

#ifdef _DEBUG
      assert( get_symbol(S_FUNCTION) == WCHARS("function"));
#endif 

    }
    return _symbol_table;
  }


  static uint get_symbol_hash(uint idx) {
    tool::critical_section cs(symbol_table_guard());
    uint &                 hv = symbol_table().data(idx);
    if (!hv) hv = CsHashString(symbol_table().name(idx)());
    return hv;
  }

  static uint intern(const wchar *s) {
    tool::critical_section cs(symbol_table_guard());
    return symbol_table()[s];
  }

  static uint intern(const ustring &s) {
    tool::critical_section cs(symbol_table_guard());
    return symbol_table()[s];
  }

  tool::ustring CsSymbolName(value o) {
    symbol_t idx = to_symbol(o);
    return symbol_name(idx);
  }

  tool::ustring symbol_name(symbol_t idx) {
    assert(idx && idx <= symbol_table().size());
    if (idx && idx <= symbol_table().size()) return get_symbol(idx);
    return tool::ustring();
  }

  /*int CsSymbolPrintNameLength(value o)
  {
    symbol_t idx = to_symbol(o);
    assert(idx && idx <= symbol_table().size());
    return (int)strlen(get_symbol( idx ));
  }*/

  static value CSF_toString(VM *c);

  void CsDumpSymbolTable(VM *c) {
    /*#ifdef _DEBUG
      int total = symbol_table().size();
      dbg_printf("Symbol table, total %d:\n",total);
      for( int n = 1; n <= total; ++n )
        dbg_printf("%d - %S\n", n, get_symbol( n ).c_str());
    #endif*/
  }

  /* property handlers */
  // static value CSF_printName(VM *c,value obj);

  /* Vector methods */
  static c_method methods[] = {C_METHOD_ENTRY("toString", CSF_toString),
                               C_METHOD_ENTRY("toHtmlString", CSF_toString),
                               C_METHOD_ENTRY("toCssString", CSF_toString),
                               C_METHOD_ENTRY(0, 0)};

  /* Vector properties */
  static vp_method properties[] = {
      // VP_METHOD_ENTRY( "printName",      CSF_printName,      0 ),
      VP_METHOD_ENTRY(0, 0, 0)};

  /* CsInitSymbol - initialize the 'CsSymbol' obj */
  void CsInitSymbol(VM *c) {
    c->symbolObject =
        CsEnterType(CsGlobalScope(c), "Symbol", &CsSymbolDispatch);
    CsEnterMethods(c, c->symbolObject, methods);
    CsEnterVPMethods(c, c->symbolObject, properties);
  }

  /* CSF_printName - built-in property 'printName' */

  value CSF_toString(VM *c) {
    // value obj, r;
    int_t sym;

    value vsym = CsGetArg(c, 1);
#ifdef _DEBUG
    dispatch *pd = CsGetDispatch(vsym);
    pd           = pd;
#endif
    assert(CsSymbolP(vsym));
    /* parse the arguments */
    // CsParseArguments(c,"L*",&sym);
    sym             = to_symbol(vsym);
    tool::ustring s = get_symbol(sym);
    return string_to_value(c, s);
  }

  /*
  static value CSF_printName(VM *c,value obj)
  {
    symbol_t idx = to_symbol(o);
    assert(idx < symbol_table().size());
    tool::string s = symbol_table()( idx );
    tool::ustring us = tool::ustring::utf8(s,s.length());
    return string_to_value(c,us);
    //return CsMakeString(c,(unsigned char*)(const char*)s,s.length());
  } */

  /* SYMBOL */

  //#define SymbolHashValue(o)              (((CsSymbol *)o)->hashValue)
  //#define SetSymbolHashValue(o,v)         (((CsSymbol *)o)->hashValue = (v))
  //#define SetSymbolPrintNameLength(o,v)   (((CsSymbol *)o)->printNameLength =
  //(v))

  static bool  GetSymbolProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetSymbolProperty(VM *c, value obj, value tag, value value);
  static bool  SymbolPrint(VM *c, value val, stream *s, bool toLocale);
  static long  SymbolSize(value obj);
  static value SymbolCopy(VM *c, value obj);
  static int_t SymbolHash(value obj);

  // static value MakeSymbol(VM *c,unsigned char *printName,int length,int_t
  // hashValue);  static value AllocateSymbolSpace(VM *c,long size);

  dispatch CsSymbolDispatch = {
      "Symbol",          &CsSymbolDispatch,    GetSymbolProperty,
      SetSymbolProperty, CsDefaultNewInstance, SymbolPrint,
      SymbolSize,        SymbolCopy,           CsDefaultScan,
      SymbolHash,        CsDefaultGetItem,     CsDefaultSetItem};

  /* GetSymbolProperty - CsSymbol get property handler */
  static bool GetSymbolProperty(VM *c, value &obj, value tag, value *pValue) {
    // return CsGetVirtualProperty(c,obj,c->symbolObject,tag,pValue);
    return CsGetNonObjectProperty(c, obj, c->symbolObject, tag, pValue);
  }

  /* SetSymbolProperty - CsSymbol set property handler */
  static bool SetSymbolProperty(VM *c, value obj, value tag, value val) {
    // return CsSetVirtualProperty(c,obj,c->symbolObject,tag,value);
    return CsSetNonObjectProperty(c, obj, c->symbolObject, tag, val);
  }

  /* SymbolPrint - CsSymbol print handler */
  static bool SymbolPrint(VM *c, value val, stream *s, bool toLocale) {
    symbol_t idx = to_symbol(val);
    assert(idx <= symbol_table().size());
    tool::ustring str = get_symbol(idx);
    return s->put_str(str);

    // long size = str.size();
    // const wchar* p = str;
    // while (--size >= 0)
    //    if (!s->put(*p++))
    //      return false;
    // return true;
  }

  /* SymbolSize - CsSymbol size handler */
  static long SymbolSize(value obj) {
    // CsSymbol *sym = (CsSymbol *)obj;
    // return sizeof(CsSymbol) + CsRoundSize(sym->printNameLength - 1);
    return sizeof(value);
  }

  /* SymbolCopy - CsSymbol copy handler */
  static value SymbolCopy(VM *c, value obj) { return obj; }

  /* SymbolHash - CsSymbol hash handler */
  static int_t SymbolHash(value obj) {
    symbol_t idx = to_symbol(obj);
    return get_symbol_hash(idx);
    // return (int_t)obj; // each symbol has a unique value
  }

  /* CsMakeSymbol - make a new symbol */
  value CsMakeSymbol(VM *c, const char *printName, int length) {
    tool::string s(printName, length ? length : strlen(printName));
    uint         idx = intern(ustring(s));
    value        v   = CsSymbol(idx);
    return v;
  }

  /* CsMakeSymbol - make a new symbol */
  value CsMakeSymbol(VM *c, const wchar *printName, int length) {
    tool::ustring us(printName, length ? length : str_len(printName));
    uint          idx = intern(us);
    return CsSymbol(idx);
  }
  value CsMakeSymbol(VM *c, wchars name) {
    tool::ustring us(name);
    uint          idx = intern(us);
    return CsSymbol(idx);
  }

  value CsSymbolOf(const char *printName) {
    tool::ustring s(printName);
    uint          idx = intern(s);
    return CsSymbol(idx);
  }

  value CsSymbolOf(const wchar *printName) {
    tool::ustring s(printName);
    uint          idx = intern(s);
    return CsSymbol(idx);
  }

  value CsSymbolOf(chars printName) {
    tool::ustring s(printName);
    uint          idx = intern(s);
    return CsSymbol(idx);
  }

  value CsSymbolOf(wchars printName) {
    tool::ustring s(printName);
    uint          idx = intern(s);
    return CsSymbol(idx);
  }

  value CsSymbolOf(const ustring &printName) {
    uint idx = intern(printName);
    return CsSymbol(idx);
  }

  uint CsSymbolIdx(wchars printName) { return intern(printName); }

  uint CsSymbolIdx(const wchar *printName) { return intern(printName); }

  /* CsIntern - intern a symbol given its print name as a string */
  value CsIntern(VM *c, value printName) {
    // needs some optimization
    // ustring us could be avoided.
    tool::ustring us = value_to_string(printName);
    return CsSymbolOf(us);
  }

  /* CsInternCString - intern a symbol given its print name as a c string */
  /*value CsInternCString(VM *c,const char *printName)
  {
      return CsInternString(c,printName,(int)strlen(printName));
  }*/

  /* CsInternString - intern a symbol given its print name as a string/length */
  value CsInternString(VM *c, const char *printName, int length) {
    return CsMakeSymbol(c, printName, length);
  }

  /* CsGlobalValue - get the value of a global symbol
  bool CsGlobalValue(CsScope *scope,value sym,value *pValue)
  {
    VM *c = scope->c;
    value obj,property;
    for (int sn = CsScope* ps = scope ; ps ; ps = ps->next)
    {
          obj = CsScopeObject(ps);
          if ((property = CsFindProperty(c,obj,sym,NULL,NULL)) != 0) {
              *pValue = CsPropertyValue(property);
              return true;
          }
    }
    return false;
  }*/

  bool CsGetGlobalValue(VM *c, value sym, value *pValue) {
    value obj, property;
    bool  silent_unknowns = false;
    value globals         = 0;
    for (int sn = c->scopes.last_index(); sn >= 0; --sn) {
      CsScope *ps = c->scopes[sn];

      if ((ps->c == c) && ps->getValue(sym, pValue))
          return true;
      
      if (ps->silent_unknowns) silent_unknowns = true;
      obj = CsScopeObject(ps);
      if (globals == obj) // note scopes can be doubled in the list
        continue;
      if (!CsObjectOrMethodP(obj)) // note scopes can be doubled in the list
        continue;
      globals = obj;
      if ((property = CsFindProperty(c, obj, sym, NULL, NULL)) != 0) {
        *pValue = CsPropertyValue(property);
        return true;
      }

      if (CsClassP(obj)) // special case, class establishes namespace too
      {
        obj = CsObjectClass(obj);
        while (CsObjectP(obj) || CsCObjectP(obj) || CsClassP(obj)) {
          property = CsFindProperty(c, obj, sym, 0, 0);
          if (property) {
            *pValue = CsPropertyValue(property);
            return true;
          }
          obj = CsObjectClass(obj);
        }
      }
    }
    // not found
    if (silent_unknowns) {
      *pValue = NOTHING_VALUE;
      return true;
    } else
      return false;
  }

  value CsGetGlobalValueByPath(VM *c, value ns, const char *path) {
    tool::atokens tz(tool::chars_of(path), CHARS("."));
    tool::chars   s;
    value         obj = ns;
    while (tz.next(s)) {
      value v   = 0;
      value sym = CsMakeSymbol(c, s.start, s.size());

      if (obj == 0 ? CsGetGlobalValue(c, sym, &v): CsGetProperty1(c, obj, sym, &v)) 
      {
        if (v /*&& CsObjectP(v)*/) {
          obj = v;
          continue;
        }
      }
      obj = UNDEFINED_VALUE;
      break;
    }
    return obj == ns ? UNDEFINED_VALUE : obj;
  }

  value CsGetGlobalValueByPath(VM *c, const char *path) {
    return CsGetGlobalValueByPath(c,0,path);
  }

#if 0
  bool CsSetGlobalValueByPath(VM *c, value ns, const char *path, value val) {
    tool::atokens tz(tool::chars_of(path), CHARS("."));
    tool::chars   s;
    value         obj = ns;
    while (tz.next(s)) {
      value v = 0;
      value sym = CsMakeSymbol(c, s.start, s.size());

      if (obj == 0 ? CsGetGlobalValue(c, sym, &v) : CsGetProperty1(c, obj, sym, &v))
      {
        if (v /*&& CsObjectP(v)*/) {
          obj = v;
          continue;
        }
      }
      obj = UNDEFINED_VALUE;
      break;
    }
    return obj == ns ? UNDEFINED_VALUE : obj;
  }
#endif


  /* CsSetGlobalValue - set the value of a global symbol */
  bool CsSetGlobalOrNamespaceValue(VM *c, value tag, value val, bool create) {
    // CsSetProperty(scope->c,CsScopeObject(scope),sym,value);
    value obj = c->getCurrentNS();

    value p;

    // bool silent_unknowns = false;

    while (CsObjectOrMethodP(obj)) {
#if defined(TISCRIPT_USE_PERSISTENCE)
      obj = CsFetchObject(c, obj);
#endif
      p = CsFindProperty(c, obj, tag, NULL, NULL);
      if (p) {
        value propValue = CsPropertyValue(p);
        if (CsVPMethodP(propValue)) {
          vp_method *method = ptr<vp_method>(propValue);
          return method->set(c, tag, obj, val);
        } else if (CsPropertyMethodP(propValue)) {
          // CsStackOverflowGuard _(c,obj,propValue, true);
          CsSendMessage(c, obj, propValue, 1, val);
        }
        else if (CsVirtualPropertyP(propValue)) {
          value fun = CsGetVirtualPropertySetter(propValue);
          if (CsAnyMethodP(fun))
            CsSendMessage(c, obj, fun, 1, val);
          else
            CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
        }
        else if (CsPropertyIsConst(p))
          CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
        else if (propValue != val) {
          CsSetPropertyValue(p, val);
          if (value observer = CsEntityObserver(obj))
            CsEnqueueNotification(c, observer, obj, tag, val, propValue,
                                  UPDATE_PROPERTY);
        }
        //  CsSetPropertyValue(p,val);
        return true;
      }
      //obj = CsObjectClass(obj);
      if (CsClassP(obj))
        obj = CsClassNamespace(obj);
      else
        obj = CsObjectClass(obj);
    }
    /* add it as a property of globals */
    CsSetGlobalValue(CsCurrentScope(c), tag, val, create);
    return true;
  }

  bool CsDelGlobalOrNamespaceValue(VM *c, value sym) {
    value obj = c->getCurrentNS();

    while (CsObjectOrMethodP(obj)) {
      if (CsFindProperty(c, obj, sym, NULL, NULL)) {
        CsRemoveObjectProperty(c, obj, sym);
        return true;
      }
      obj = CsObjectClass(obj);
    }
    CsRemoveObjectProperty(c, CsCurrentScope(c)->globals, sym);
    return true;
  }

  bool CsGetGlobalOrNamespaceValue(VM *c, value tag, value *pValue) {
    value obj = c->getCurrentNS();
    value p;

    bool silent_unknowns = !CsNamespaceP(obj);

    while (CsObjectOrMethodP(obj)) {
#if defined(TISCRIPT_USE_PERSISTENCE)
      obj = CsFetchObject(c, obj);
#endif
      p = CsFindProperty(c, obj, tag, NULL, NULL);
      if (p) {
        value propValue = CsPropertyValue(p);
        if (CsVPMethodP(propValue)) {
          vp_method *method = ptr<vp_method>(propValue);
          if (method->get(c, obj, *pValue))
            return true;
          else
            CsThrowKnownError(c, CsErrWriteOnlyProperty, tag);
        }
        else if (CsPropertyMethodP(propValue)) {
          *pValue = CsSendMessage(c, obj, propValue, 1, NOTHING_VALUE);
        }
        else if (CsVirtualPropertyP(propValue)) {
          PROTECT(obj);
          value fun = CsGetVirtualPropertyGetter(propValue);
          if (CsAnyMethodP(fun))
            *pValue = CsSendMessage(c, obj, fun, 0);
          else
            CsThrowKnownError(c, CsErrWriteOnlyProperty, tag);
        }
        else
          *pValue = propValue;
        return true;
      }
      if (CsClassP(obj))
        obj = CsClassNamespace(obj);
      else
        obj = CsObjectClass(obj);
    }

    /* try to get it from globals */
    bool r = CsGetGlobalValue(c, tag, pValue);
    if (!r && silent_unknowns) {
      *pValue = UNDEFINED_VALUE;
      return true;
    }
    return r;
  }

  void CsSetGlobalValue(CsScope *scope, value tag, value val, bool create) {
    
    if (scope->setValue(tag, val))
        return;

    value obj = CsScopeObject(scope);

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(scope->c, CsErrIsFrozen, obj);

    int_t hashValue = 0, i = 0;
    value p = CsFindProperty(scope->c, obj, tag, &hashValue, &i);

    if (p) {
      if (create) CsAlreadyDefined(scope->c, tag);
      value propValue = CsPropertyValue(p);
      if (CsPropertyMethodP(propValue))
        CsSendMessage(scope->c, obj, propValue, 1, val);
      else if (CsVirtualPropertyP(propValue)) {
        value fun = CsGetVirtualPropertySetter(propValue);
        if (CsAnyMethodP(fun))
          CsSendMessage(scope->c, obj, fun, 1, val);
        else
          CsThrowKnownError(scope->c, CsErrReadOnlyProperty, tag);
      }
      else if (CsPropertyIsConst(p))
        CsThrowKnownError(scope->c, CsErrReadOnlyProperty, tag);
      else if (propValue != val) {
        if (value observer = CsEntityObserver(obj))
          CsEnqueueNotification(scope->c, observer, obj, tag, val, propValue,
                                UPDATE_PROPERTY);
        CsSetPropertyValue(p, val);
      }
      return;
    }

    if (!create && !scope->silent_unknowns)
      CsThrowKnownError(scope->c, CsErrUnboundVariable, tag);
    CsAddProperty(scope->c, CsScopeObject(scope), tag, val, hashValue, i);
    if (value observer = CsEntityObserver(obj))
      CsEnqueueNotification(scope->c, observer, CsScopeObject(scope), tag, val,
                            UNDEFINED_VALUE, ADD_PROPERTY);

    // return true;
  }

  void CsSetNamespaceValue(VM *c, value sym, value val, bool create, bool force) {
    value obj = c->getCurrentNS();

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    int_t hashValue = 0, i = 0;
    value p = CsFindProperty(c, obj, sym, &hashValue, &i);
    if (p) {
      if(create) CsAlreadyDefined(c, sym);
      value propValue = CsPropertyValue(p);
      if (CsPropertyMethodP(propValue)) {
        CsSendMessage(c, obj, propValue, 1, val);
      }
      else if (CsVirtualPropertyP(propValue)) {
        value fun = CsGetVirtualPropertySetter(propValue);
        if (CsAnyMethodP(fun))
          CsSendMessage(c, obj, fun, 1, val);
        else
          CsThrowKnownError(c, CsErrReadOnlyProperty, sym);
      }
      else if (!force && CsPropertyIsConst(p))
        CsThrowKnownError(c, CsErrReadOnlyProperty, sym);
      else if (propValue != val) {
        if (value observer = CsEntityObserver(obj))
          CsEnqueueNotification(c, observer, obj, sym, val, propValue, UPDATE_PROPERTY);
        CsSetPropertyValue(p, val);
      }
      return;
    }
    if (sym == UNDEFINED_VALUE && CsClassP(obj) && CsPropertyMethodP(val))
      CsSetClassUndefinedPropHandler(obj, val);
    else {
      PROTECT(obj);
      CsAddProperty(c, obj, sym, val, hashValue, i);
      if (value observer = CsEntityObserver(obj)) {
        CsEnqueueNotification(c, observer, obj, sym, val, UNDEFINED_VALUE,
                              ADD_PROPERTY);
      }
    }
  }

  void CsSetNamespaceGetterSetter(VM *c, value sym, value val, bool setter) {
    value obj = c->getCurrentNS();

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    int_t hashValue = 0, i = 0;
    value p = CsFindProperty(c, obj, sym, &hashValue, &i);
    if (p) {
      value propValue = CsPropertyValue(p);
      if (CsVirtualPropertyP(propValue)) {
        assert(CsAnyMethodP(val));
        if (setter)
          CsSetVirtualPropertySetter(propValue,val);
        else 
          CsSetVirtualPropertyGetter(propValue, val);
      }
      else 
        CsAlreadyDefined(c, sym);
      return;
    }
    PROTECT(obj,val);
    value propValue = CsMakeVirtualProperty(c);
    if (setter)
      CsSetVirtualPropertySetter(propValue, val);
    else
      CsSetVirtualPropertyGetter(propValue, val);
    CsAddProperty(c, obj, sym, propValue, hashValue, i);
  }

  void CsSetNamespaceConst(VM *c, value sym, value val) {
    // printf("CsSetGlobalValue step 0 %x %x\n", scope, scope->c );
    if (!CsSymbolP(sym)) 
      CsThrowKnownError(c, CsErrImpossible);

    // dispatch* pd = CsGetDispatch(c->getCurrentNS());

    CsAddConstant(c, c->getCurrentNS(), sym, val);
  }

  /* AllocateSymbolSpace - allocate symbol space
  static value AllocateSymbolSpace(VM *c,long size)
  {
      value p;
      if (!c->symbolSpace || c->symbolSpace->bytesRemaining < size) {
          CsSymbolBlock *b;
          if (!(b = (CsSymbolBlock *)CsAlloc(c,sizeof(CsSymbolBlock))))
              CsInsufficientMemory(c);
          b->bytesRemaining = CsSBSize;
          b->nextByte = b->data;
          b->next = c->symbolSpace;
          c->symbolSpace = b;
      }
      c->symbolSpace->bytesRemaining -= size;
      p = (value)c->symbolSpace->nextByte;
      c->symbolSpace->nextByte += size;
    return p;
  } */

} // namespace tis
