/* parse.c - argument parsing function */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"

namespace tis {

  /*
  value CsGetNthArg(VM *c, int n)
  {
    return CsGetArg(c,n);
  }

  int CsGetArgCount(VM *c)
  {
    return CsArgCnt(c);
  }*/

  /* CsParseArguments - parse the argument list of a method */
  int CsParseArguments(VM *c, const char *fmt, ...) {
    int     spec;
    bool    optionalP = false; /* no optional specifier seen yet */
    value * argv      = c->argv;
    int     argc      = c->argc;
    value   arg;
    va_list ap;

    bool    throw_error = true;
    if (*fmt == '|') {
      throw_error = false;
      ++fmt;
    }

#define REPORT_ERROR(vm,arg,txt) { if (throw_error) CsUnexpectedTypeError(vm, arg, txt); else return 0; }

    /* get the variable argument list */
    va_start(ap, fmt);

    /* handle each argument specifier */
    while (*fmt) {

      /* check for the optional specifier */
      if ((spec = *fmt++) == '|') 
        optionalP = true;

      /* handle argument specifiers */
      else {

        /* check for another argument */
        if (--argc < 0) break;

        /* get the argument */
        arg = *--argv;

        /* pdispatch on the specifier */
        switch (spec) {
        case '*': /* skip */ break;
        case 'c': /* char */
        {
          char *p = va_arg(ap, char *);
          if (!CsIntegerP(arg))
            // CsTypeError(c,arg);
            REPORT_ERROR(c, arg, "integer");
          *p = (char)CsIntegerValue(arg);
        } break;
        case 's': /* short */
        {
          short *p = va_arg(ap, short *);
          if (!CsIntegerP(arg)) CsTypeError(c, arg);
          *p = (short)CsIntegerValue(arg);
        } break;
        case 'i': /* int */
        {
          int *p = va_arg(ap, int *);
          if (CsIntegerP(arg))
            *p = CsIntegerValue(arg);
          else if (CsLengthP(arg))
            *p = (int)CsLengthPixels(c, arg);
          else
            REPORT_ERROR(c, arg, "integer");
        } break;
        case 'T': /* int */
        {
          int *p = va_arg(ap, int *);
          if (CsIntegerP(arg))
            *p = CsIntegerValue(arg);
          else if (CsDurationP(arg))
            *p = int(1000 * CsDurationSeconds(arg));
          else
            REPORT_ERROR(c, arg, "integer or duration");
        } break;
        case 'I': /* int */
        {
          int *p = va_arg(ap, int *);
          if (CsIntegerP(arg))
            *p = (int)CsIntegerValue(arg);
          else if (CsFloatP(arg))
            *p = (int)CsFloatValue(arg);
          else if (CsLengthP(arg))
            *p = (int)CsLengthPixels(c, arg);
          else if (arg == TRUE_VALUE)
            *p = 1;
          else if (arg == FALSE_VALUE)
            *p = 0;
          else
            REPORT_ERROR(c, arg, "integer");
        } break;
        case 'l': /* long */
        {
          long *p = va_arg(ap, long *);
          if (CsIntegerP(arg))
            *p = (long)CsIntegerValue(arg);
          else if (CsLengthP(arg))
            *p = (long)CsLengthPixels(c, arg);
          else
            REPORT_ERROR(c, arg, "integer");
        } break;
        case 'F': /* float */
        {
          float *p = va_arg(ap, float *);
          if (CsIntegerP(arg))
            *p = (float)CsIntegerValue(arg);
          else if (CsFloatP(arg))
            *p = (float)CsFloatValue(arg);
          else
            REPORT_ERROR(c, arg, "float");
        } break;
        case 'f': /* float */
        {
          float *p = va_arg(ap, float *);
          if (CsIntegerP(arg))
            *p = (float)CsIntegerValue(arg);
          else if (CsFloatP(arg))
            *p = (float)CsFloatValue(arg);
          else if (CsLengthP(arg))
            *p = (float)CsLengthPixels(c, arg);
          else if (CsAngleP(arg))
            *p = (float)CsAngleRadians(arg);
          else
            REPORT_ERROR(c, arg, "float");
        } break;
        case 'D': /* double */
        {
          double *p = va_arg(ap, double *);
          if (CsIntegerP(arg))
            *p = (double)CsIntegerValue(arg);
          else if (CsFloatP(arg))
            *p = (double)CsFloatValue(arg);
          else
            REPORT_ERROR(c, arg, "float");
        } break;
        case 'd': /* double */
        {
          double *p = va_arg(ap, double *);
          if (CsIntegerP(arg))
            *p = (double)CsIntegerValue(arg);
          else if (CsFloatP(arg))
            *p = (double)CsFloatValue(arg);
          else if (CsLengthP(arg))
            *p = CsLengthPixels(c, arg);
          else if (CsAngleP(arg))
            *p = CsAngleRadians(arg);
          else
            // CsTypeError(c,arg);
            REPORT_ERROR(c, arg, "float");
        } break;

        case 'g': /* double or angle (degree) "gradus" */
        {
          float *p = va_arg(ap, float *);
          if (CsIntegerP(arg))
            *p = float(CsIntegerValue(arg));
          else if (CsFloatP(arg))
            *p = float(CsFloatValue(arg));
          else if (CsAngleP(arg))
            *p = float(CsAngleRadians(arg) * 57.2957795);
          else
            // CsTypeError(c,arg);
            REPORT_ERROR(c, arg, "float or angle");
        } break;
        case 'G': /* double or angle (degree) "gradus" */
        {
          double *p = va_arg(ap, double *);
          if (CsIntegerP(arg))
            *p = double(CsIntegerValue(arg));
          else if (CsFloatP(arg))
            *p = double(CsFloatValue(arg));
          else if (CsAngleP(arg))
            *p = double(CsAngleRadians(arg) * 57.2957795);
          else
            // CsTypeError(c,arg);
            REPORT_ERROR(c, arg, "float or angle");
        } break;

        case 'S': /* string */
        {
          wchar **p           = va_arg(ap, wchar **);
          bool    nilAllowedP = false;
          if (*fmt == '?') {
            nilAllowedP = true;
            ++fmt;
          }
          if (nilAllowedP && arg == UNDEFINED_VALUE)
            *p = NULL;
          else if (CsStringP(arg))
            *p = CsStringAddress(arg);
          else if (CsSymbolP(arg))
            *p = (wchar*)CsSymbolName(arg).c_str();
          else
            CsTypeError(c, arg);
          if (*fmt == '#') {
            int *p = va_arg(ap, int *);
            if(CsStringP(arg))
              *p = CsStringSize(arg);
            else if (CsSymbolP(arg))
              *p = CsSymbolName(arg).size();
            ++fmt;
          }
        } break;
        case 'V': /* value */
        {
          value *p           = va_arg(ap, value *);
          bool   nilAllowedP = false;
          if (*fmt == '?') {
            nilAllowedP = true;
            ++fmt;
          }
          if (nilAllowedP && arg == UNDEFINED_VALUE)
            *p = 0;
          else {
            if (*fmt == '=') {
              dispatch *desiredType = va_arg(ap, dispatch *);
              dispatch *type        = CsGetDispatch(arg);
              if (type != desiredType && type->baseType != desiredType) {
                if (optionalP)
                  arg = *p; // don't change its value
                else
                  REPORT_ERROR(c, arg, desiredType->typeName);
              }
              ++fmt;
            }
            *p = arg;
          }
        } break;
        case 'P': /* foreign pointer */
        {
          void **p           = va_arg(ap, void **);
          bool   nilAllowedP = false;
          if (*fmt == '?') {
            nilAllowedP = true;
            ++fmt;
          }
          if (nilAllowedP && arg == UNDEFINED_VALUE)
            *p = NULL;
          else {
            if (*fmt == '=') {
              dispatch *desiredType = va_arg(ap, dispatch *);
              dispatch *type        = CsGetDispatch(arg);
              if (type != desiredType && type->baseType != desiredType) 
                REPORT_ERROR(c, arg, desiredType->typeName);
              ++fmt;
            }
            *p = CsCObjectValue(arg);
          }
        } break;
        case 'b':
        case 'B': /* boolean */
        {
          bool *p = va_arg(ap, bool *);
          *p = CsTrueP(CsToBoolean(c,arg));
        } break;
        case 'L': /* symbol */
        {
          symbol_t *p = va_arg(ap, symbol_t *);
          if (!CsSymbolP(arg)) 
            REPORT_ERROR(c, arg, "symbol");
          *p = (int)to_symbol(arg);
        } break;

        case 'C': /* color */
        {
          uint *p = va_arg(ap, uint *);
          if (CsColorP(arg))
            *p = (uint)CsColorValue(arg);
          else if (CsIntegerP(arg))
            *p = CsIntegerValue(arg);
          else if (CsTupleP(arg)) {
            if(!c->resolveNamedColor(arg,p))
              REPORT_ERROR(c, arg, "color");
          }
          else 
            REPORT_ERROR(c, arg, "color");
        } break;

        case 'M': /* any method (function) */
        {
          value *p = va_arg(ap, value *);
          if (CsAnyMethodP(arg))
            *p = arg;
          else
            REPORT_ERROR(c, arg, "function");
        } break;

        case 'm': /* scripting method (function) */
        {
          value *p = va_arg(ap, value *);
          if (CsMethodP(arg))
            *p = arg;
          else
            REPORT_ERROR(c, arg, "script function");
        } break;


        default: CsThrowKnownError(c, CsErrBadParseCode); break;
        }
      }
    }

    /* finished with the variable arguments */
    va_end(ap);

    /* check for too many arguments */
    // if (argc > 0)
    //    CsTooManyArguments(c);
    // else
    if (argc < 0 && !optionalP) {
      if(throw_error)
        CsTooFewArguments(c);
      else
        return 0;
    }

    return c->argc - argc;
  }

} // namespace tis
