/* cs_length.c - 'Length' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include "cs_int.h"
#include <limits.h>

namespace tis {

  /*static uint value_length_units[] = {
      0,
      tool::value::em,
      tool::value::ex,
      tool::value::pr,
      tool::value::sp,
      tool::value::px,
      tool::value::in,
      tool::value::cm,
      tool::value::mm,
      tool::value::pt,
      tool::value::pc,
      tool::value::dip,
      tool::value::pr_width,
      tool::value::pr_height,
      tool::value::pr_view_width,
      tool::value::pr_view_height,
      tool::value::pr_view_min,
      tool::value::pr_view_max,
  };*/

  /*uint lu2tool(UNIT_TYPE ut) {
#ifdef WINDOWS
//  assert(IS_LENGTH_UNIT(ut));
#endif
    return IS_LENGTH_UNIT(ut) ? value_length_units[ut] : 0;
  }
  UNIT_TYPE tool2lu(uint ut) {
    int n = items_of(value_length_units).index_of(ut);
    if (IS_LENGTH_UNIT(n)) return UNIT_TYPE(n);
    // assert(false);
    return ut_none;
  }*/

  /* method handlers */
  static value CSF_parse(VM *c);
  static value CSF_toFloat(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);
  value        CSF_make_length(VM *c);
  static value CSF_morph(VM *c);

  /* Integer methods */
  static c_method methods[] = {C_METHOD_ENTRY("make", CSF_make_length),
                               C_METHOD_ENTRY("parse", CSF_parse),
                               C_METHOD_ENTRY("toFloat", CSF_toFloat),
                               C_METHOD_ENTRY("toInteger", CSF_toInteger),
                               C_METHOD_ENTRY("toString", CSF_toString),
                               C_METHOD_ENTRY("toHtmlString", CSF_toString),
                               C_METHOD_ENTRY("toUrlString", CSF_toString),
                               C_METHOD_ENTRY("toCssString", CSF_toString),
                               C_METHOD_ENTRY("morph", CSF_morph),

                               C_METHOD_ENTRY(0, 0)};

  static value CSF_units(VM *c, value obj);

  /* Integer properties */
  static vp_method properties[] = {VP_METHOD_ENTRY("units", CSF_units, 0),
                                   VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {
      // CONSTANT_ENTRY("MAX"    , int_value(INT_MAX     )),
      // CONSTANT_ENTRY("MIN"    , int_value(INT_MIN     )),
      CONSTANT_ENTRY(0, 0)};

  /* CsInitLength - initialize the 'Length' obj */
  void CsInitLength(VM *c) {
    c->lengthObject =
        CsEnterType(CsGlobalScope(c), "Length", &CsLengthDispatch);
    CsEnterMethods(c, c->lengthObject, methods);
    CsEnterVPMethods(c, c->lengthObject, properties);
    CsEnterConstants(c, c->lengthObject, constants);
  }

  /* CSF_parse - built-in method 'parse' */
  static value CSF_parse(VM *c) {
    wchar *str    = 0;
    value  defval = UNDEFINED_VALUE;
    CsParseArguments(c, "**S|V", &str, &defval);
    tool::value vv = tool::value::parse_numeric_units(str);
    if (vv.is_length())
      return pack_value(PT_LENGTH, vv.units(), vv._int());
    else if (vv.is_int())
      return pack_value(PT_LENGTH, LENGTH_UNIT_TYPE::px, vv._int());
    return defval;
  }

  static value sym_em  = 0;
  static value sym_ex  = 0;
  static value sym_ch = 0;
  static value sym_pr  = 0;
  static value sym_fx  = 0;
  static value sym_px  = 0;
  static value sym_in  = 0;
  static value sym_cm  = 0;
  static value sym_mm  = 0;
  static value sym_pt  = 0;
  static value sym_pc  = 0;
  static value sym_dip = 0;
  static value sym_as  = 0;
  static value sym_rem = 0;
  static value sym_ppx = 0;

  static void init_symbols() {
    if (sym_em == 0) {
      sym_em  = CsSymbolOf("em");
      sym_rem = CsSymbolOf("rem");
      sym_ex  = CsSymbolOf("ex");
      sym_ch = CsSymbolOf("ch");
      sym_pr  = CsSymbolOf("pr");
      sym_fx  = CsSymbolOf("fx");
      sym_px  = CsSymbolOf("px");
      sym_in  = CsSymbolOf("in");
      sym_cm  = CsSymbolOf("cm");
      sym_mm  = CsSymbolOf("mm");
      sym_pt  = CsSymbolOf("pt");
      sym_pc  = CsSymbolOf("pc");
      sym_dip = CsSymbolOf("dip");
      sym_ppx = CsSymbolOf("ppx");
    }
  }

  static value CSF_units(VM *c, value obj) {
    init_symbols();
    LENGTH_UNIT_TYPE u = get_unit<LENGTH_UNIT_TYPE>(obj);
    // to_unit(obj,u);
    switch (u) {
    case LENGTH_UNIT_TYPE::em: return sym_em;
    case LENGTH_UNIT_TYPE::rem: return sym_rem;
    case LENGTH_UNIT_TYPE::ex: return sym_ex;
    case LENGTH_UNIT_TYPE::pr: return sym_pr;
    case LENGTH_UNIT_TYPE::sp: return sym_fx;
    case LENGTH_UNIT_TYPE::px: return sym_px;
    case LENGTH_UNIT_TYPE::in: return sym_in;
    case LENGTH_UNIT_TYPE::cm: return sym_cm;
    case LENGTH_UNIT_TYPE::mm: return sym_mm;
    case LENGTH_UNIT_TYPE::pt: return sym_pt;
    case LENGTH_UNIT_TYPE::pc: return sym_pc;
    case LENGTH_UNIT_TYPE::dip: return sym_dip;
    case LENGTH_UNIT_TYPE::ppx: return sym_ppx;
    }
    return UNDEFINED_VALUE;
  }

  value length_value(VM *c, LENGTH_UNIT_TYPE type) {
    value v;
    CsParseArguments(c, "**V", &v);

    if (CsIntegerP(v))
      return CsMakeLength(CsIntegerValue(v), type);
    else if (CsFloatP(v))
      return CsMakeLength(CsFloatValue(v), type);
    else {
      CsThrowKnownError(c, CsErrUnexpectedTypeError, v,
                        "only integer or float");
    }
    return CsMakeLength(0, type);
  }

  value flex_value(VM *c) {
    value v,minv = 0,maxv = 0;
    CsParseArguments(c, "**V|V|V", &v);

    value flex = UNDEFINED_VALUE;
    if (CsIntegerP(v))
      flex = CsMakeLength(CsIntegerValue(v), LENGTH_UNIT_TYPE::sp);
    else if (CsFloatP(v))
      flex = CsMakeLength(CsFloatValue(v), LENGTH_UNIT_TYPE::sp);
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, v, "only integer or float");
    
    if (minv && maxv)
      return CsMakeTuple(c, "fx", flex, minv, maxv);
    else if (minv)
      return CsMakeTuple(c, "fx", flex, minv);
    else
      return flex;
    
  }


  LENGTH_UNIT_TYPE sym2ut(VM *c, value sym) {
    init_symbols();

    LENGTH_UNIT_TYPE type = LENGTH_UNIT_TYPE::unknown;
    if (sym == sym_em)
      type = LENGTH_UNIT_TYPE::em;
    else if (sym == sym_ex)
      type = LENGTH_UNIT_TYPE::ex;
    else if (sym == sym_pr)
      type = LENGTH_UNIT_TYPE::pr;
    else if (sym == sym_fx)
      type = LENGTH_UNIT_TYPE::sp;
    else if (sym == sym_px)
      type = LENGTH_UNIT_TYPE::px;
    else if (sym == sym_in)
      type = LENGTH_UNIT_TYPE::in;
    else if (sym == sym_cm)
      type = LENGTH_UNIT_TYPE::cm;
    else if (sym == sym_mm)
      type = LENGTH_UNIT_TYPE::mm;
    else if (sym == sym_pt)
      type = LENGTH_UNIT_TYPE::pt;
    else if (sym == sym_pc)
      type = LENGTH_UNIT_TYPE::pc;
    else if (sym == sym_dip)
      type = LENGTH_UNIT_TYPE::dip;
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, sym, "only unit symbol");

    return type;
  }

  value CSF_make_length(VM *c) {
    init_symbols();
    value val;
    value sym;
    CsParseArguments(c, "**VV=", &val, &sym, &CsSymbolDispatch);

    LENGTH_UNIT_TYPE type = sym2ut(c, sym);

    return length_value(c, type);
  }

  bool coerce_length_data(VM *c, int &i1000, LENGTH_UNIT_TYPE from, LENGTH_UNIT_TYPE to);

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    value sym = 0;
    value def = 0;
    CsParseArguments(c, "V=*|V|V", &obj, &CsLengthDispatch, &sym, &def);
    LENGTH_UNIT_TYPE u = get_unit<LENGTH_UNIT_TYPE>(obj);
    int  v = unpack_uint32(obj);
    if (sym) {
      int vd = v;
      LENGTH_UNIT_TYPE ud = sym2ut(c, sym);
      if (!coerce_length_data(c, vd, u, ud)) {
        if (def) return def;
      }
      return CsMakeFloat(vd / 1000.0);
    }
    return CsMakeFloat(v / 1000.0);
    // return CsMakeFloat((float_t) tool::value::length_to_float(v,lu2tool(u)));
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsLengthDispatch);
    // int v;
    // UNIT_TYPE u;
    // v = to_unit(obj,u);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(obj);
    int v = unpack_uint32(obj);
    return CsMakeInteger(tool::value::length_to_int(v, u));
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value obj;
    CsParseArguments(c, "V=*|i", &obj, &CsLengthDispatch);
    // int v;
    // UNIT_TYPE u;
    // v = to_unit(obj,u);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(obj);
    int       v = unpack_uint32(obj);
    return CsMakeCString(c, tool::value::length_to_string(v, u));
  }

  static value CSF_morph(VM *c) {
    value  from;
    value  to;
    double ratio = 0;

    CsParseArguments(c, "**V=V=d", &from, &CsLengthDispatch, &to,
                     &CsLengthDispatch, &ratio);

    int       vfrom = unpack_int32(from);
    LENGTH_UNIT_TYPE ufrom = unpack_unit<LENGTH_UNIT_TYPE>(from);
    int       vto   = unpack_int32(to);
    LENGTH_UNIT_TYPE uto   = unpack_unit<LENGTH_UNIT_TYPE>(to);

    if (ufrom != uto) {
      if (vfrom == 0)
        ufrom = uto;
      else if (vto == 0)
        uto = ufrom;
      else
        CsThrowKnownError(c, CsErrGenericError,
                          "Length.morph - incompatible values");
    }

    int r = int(vfrom + (vto - vfrom) * ratio);

    return pack_value(PT_LENGTH, ufrom, r);
  }

  value CSF_em(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::em); }
  value CSF_rem(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::rem); }
  value CSF_ex(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::ex); }
  value CSF_ch(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::ch); }
  value CSF_pr(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::pr); }
  value CSF_fx(VM *c) { return flex_value(c); }
  value CSF_px(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::px); }
  value CSF_in(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::in); }
  value CSF_cm(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::cm); }
  value CSF_mm(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::mm); }
  value CSF_pt(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::pt); }
  value CSF_pc(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::pc); }
  value CSF_dip(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::dip); }
  value CSF_ppx(VM *c) { return length_value(c, LENGTH_UNIT_TYPE::ppx); }

  static bool  GetLengthProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetLengthProperty(VM *c, value obj, value tag, value value);
  static bool  LengthPrint(VM *c, value obj, stream *s, bool toLocale);
  static long  LengthSize(value obj);
  static value LengthCopy(VM *c, value obj);
  static int_t LengthHash(value obj);

  dispatch CsLengthDispatch = {
      "Length",          &CsLengthDispatch,    GetLengthProperty,
      SetLengthProperty, CsDefaultNewInstance, LengthPrint,
      LengthSize,        LengthCopy,           CsDefaultScan,
      LengthHash,        CsDefaultGetItem,     CsDefaultSetItem};

  /* GetLengthProperty - Length get property handler */
  static bool GetLengthProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->lengthObject, tag, pValue);
  }

  /* SetLengthProperty - Length set property handler */
  static bool SetLengthProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->lengthObject, tag, value);
  }

  /* LengthPrint - Length print handler */
  static bool LengthPrint(VM *c, value obj, stream *s, bool toLocale) {
    int       v = unpack_int32(obj);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(obj);
    return s->put_str(tool::value::length_to_string_fx(v, u));
  }

  bool CsLengthPrintFx(VM *c, value obj, stream *s, bool toLocale) {
    int       v = unpack_int32(obj);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(obj);
    return s->put_str(tool::value::length_to_string_fx(v, u));
  }

  /* LengthSize - Length size handler */
  static long LengthSize(value obj) {
    // return sizeof(CsLength);
    return sizeof(value);
  }

  /* LengthCopy - Length copy handler */
  static value LengthCopy(VM *c, value obj) {
    // return CsPointerP(obj) ? CsDefaultCopy(c,obj) : obj;
    return obj;
  }

  /* LengthHash - Length hash handler */
  static int_t LengthHash(value obj) { return (int_t)obj; }

  /* CsMakeLength - make a new length value */
  value CsMakeLength(float_t v, LENGTH_UNIT_TYPE ut) {
    // assert(IS_LENGTH_UNIT(ut));
    return pack_value(PT_LENGTH, ut, int(v * 1000.0));
  }

  value CsMakeLengthFromTool(const tool::value& v) {
    int i1000 = v._int();
    if (v.units() == tool::value::nm && i1000 == 0) return CsMakeInteger(0);
    return pack_value(PT_LENGTH, v.units(), i1000);
  }

  tool::value CsLengthToolValue(value lv) {
    int       i = unpack_int32(lv);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(lv);
    return tool::value::make_raw_length(i, u);
  }

  bool coerce_length_data(VM *c, int &i1000, LENGTH_UNIT_TYPE from, LENGTH_UNIT_TYPE to) {
    int kp = 0; // "kilopoints"
    switch (from) {
      case LENGTH_UNIT_TYPE::px: kp = muldiv(i1000, 72, c->pixels_per_inch()); break;
      case LENGTH_UNIT_TYPE::in:
        kp = i1000 * 72;
        break; // Points (1 point = 1/72 inches).
      case LENGTH_UNIT_TYPE::pt: kp = i1000; break;
      case LENGTH_UNIT_TYPE::dip: kp = muldiv(i1000, 72, 96); break;
      case LENGTH_UNIT_TYPE::pc: kp = i1000 * 12; break; // Picas (1 pica = 12 points).
      case LENGTH_UNIT_TYPE::cm: kp = muldiv(i1000, 72 * 100, 254); break;
      case LENGTH_UNIT_TYPE::mm: kp = muldiv(i1000, 72 * 100, 2540); break;
      default: return false;
    }

    switch (to) {
      case LENGTH_UNIT_TYPE::px: i1000 = muldiv(kp, c->pixels_per_inch(), 72); break;
      case LENGTH_UNIT_TYPE::in:
        i1000 = kp / 72;
        break; // Points (1 point = 1/72 inches).
      case LENGTH_UNIT_TYPE::pt: i1000 = kp; break;
      case LENGTH_UNIT_TYPE::dip: i1000 = muldiv(kp, 96, 72); break;
      case LENGTH_UNIT_TYPE::pc: i1000 = kp / 12; break; // Picas (1 pica = 12 points).
      case LENGTH_UNIT_TYPE::cm: i1000 = muldiv(kp, 254, 72 * 100); break;
      case LENGTH_UNIT_TYPE::mm: i1000 = muldiv(kp, 2540, 72 * 100); break;
      default: return false;
    }
    return true;
  }

  value CsLengthBinaryOp(VM *c, int op, value p1, value p2) {
    // note: p1 is always the length, p2 can be any, op is everything but not
    // BC_SUB

    int       v = unpack_int32(p1);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(p1);

    if (CsIntegerP(p2)) {
      int vr = CsIntegerValue(p2); // fixed(3)
      switch (op) {
      case BC_ADD: v += 1000 * vr; break;
      case BC_SUB: v -= 1000 * vr; break;
      case BC_MUL: v *= vr; break;
      case BC_DIV:
        if (vr != 0)
          v /= vr;
        else
          v = 0;
        break;
      case BC_REM:
        if (vr != 0)
          v %= (1000 * vr);
        else
          v = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        v = 0; /* never reached */
        break;
      }
    } else if (CsFloatP(p2)) {
      double vr = CsFloatValue(p2);
      double fv = v / 1000.0; // fixed(3)
      switch (op) {
      case BC_ADD: fv += vr; break;
      case BC_SUB: fv -= vr; break;
      case BC_MUL: fv *= vr; break;
      case BC_DIV:
        if (vr != 0)
          fv /= vr;
        else
          fv = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        v = 0; /* never reached */
        break;
      }
      v = int(fv * 1000.0);
    } else if (CsLengthP(p2)) {
      int       v2 = unpack_int32(p2);
      LENGTH_UNIT_TYPE u2 = unpack_unit<LENGTH_UNIT_TYPE>(p2);

      if (u == u2) {
      DOIT:
        switch (op) {
        case BC_ADD: v += v2; break;
        case BC_SUB: v -= v2; break;
        default:
          CsUnsupportedBinaryOp(c, op, p1, p2);
          v = 0; /* never reached */
          break;
        }
      } else {
        if (coerce_length_data(c, v2, u2, u))
          goto DOIT;
        else
          CsThrowKnownError(c, CsErrUnexpectedTypeError, v2,
                            "length conversion is unavailable");
      }
    } else
      CsUnsupportedBinaryOp(c, op, p1, p2);

    return pack_value(PT_LENGTH, u, v);
  }

  int CsLengthsCompare(VM *c, int op, value obj1, value obj2) {
    int       v1 = unpack_int32(obj1);
    LENGTH_UNIT_TYPE u1 = unpack_unit<LENGTH_UNIT_TYPE>(obj1);
    int       v2 = unpack_int32(obj2);
    LENGTH_UNIT_TYPE u2 = unpack_unit<LENGTH_UNIT_TYPE>(obj2);
    if (u2 != u1) {
      if (!coerce_length_data(c, v2, u2, u1))
        CsUnsupportedBinaryOp(c, op, obj1, obj2);
    }
    return v1 - v2;
  }

  double CsLengthPixels(VM *c, value lv) {
    int       v = unpack_int32(lv);
    LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(lv);
    if (u != LENGTH_UNIT_TYPE::px) coerce_length_data(c, v, u, LENGTH_UNIT_TYPE::px);
    return v / 1000.0;
  }

} // namespace tis
