/* integer.c - 'Integer' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include <limits.h>

namespace tis {

  /* method handlers */
  static value CSF_toFloat(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);
  static value CSF_min(VM *c);
  static value CSF_max(VM *c);
  static value CSF_limit(VM *c);
  static value CSF_parse(VM *c) {
    wchars s;
    value  dv    = UNDEFINED_VALUE;
    int_t  radix = 10;
    // wchar *pend;
    CsParseArguments(c, "**S#|V|i", &s.start, &s.length, &dv, &radix);
    s       = tool::trim(s);
    int_t i = 0;
    if (!tool::parse_int(s, i, radix)) return dv;
    return CsMakeInteger(i);
  }

  /* Integer methods */
  static c_method methods[] = {C_METHOD_ENTRY("toFloat", CSF_toFloat),
                               C_METHOD_ENTRY("toInteger", CSF_toInteger),
                               C_METHOD_ENTRY("toString", CSF_toString),
                               C_METHOD_ENTRY("toHtmlString", CSF_toString),
                               C_METHOD_ENTRY("toUrlString", CSF_toString),
                               C_METHOD_ENTRY("toCssString", CSF_toString),
                               C_METHOD_ENTRY("min", CSF_min),
                               C_METHOD_ENTRY("max", CSF_max),
                               C_METHOD_ENTRY("limit", CSF_limit),
                               C_METHOD_ENTRY("parse", CSF_parse),
                               C_METHOD_ENTRY(0, 0)};

  /* Integer properties */
  static vp_method properties[] = {VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {CONSTANT_ENTRY("MAX", int_value(INT_MAX)),
                                 CONSTANT_ENTRY("MIN", int_value(INT_MIN)),
                                 CONSTANT_ENTRY(0, 0)};

  /* CsInitInteger - initialize the 'Integer' obj */
  void CsInitInteger(VM *c) {
    c->integerObject =
        CsEnterType(CsGlobalScope(c), "Integer", &CsIntegerDispatch);
    CsEnterMethods(c, c->integerObject, methods);
    CsEnterVPMethods(c, c->integerObject, properties);
    CsEnterConstants(c, c->integerObject, constants);
  }

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsIntegerDispatch);
    return CsMakeFloat((float_t)CsIntegerValue(obj));
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsIntegerDispatch);
    return obj;
  }

  static value minimum(VM *c, value *argv, int argc) {
    value arg;
    int_t r      = INT_MAX;
    bool  gotone = false;
    for (int i = 0; i < argc; ++i) {
      arg = argv[i];
      if (CsVectorP(arg)) {
        arg = minimum(c, CsVectorAddress(c, arg), CsVectorSize(c, arg));
      }
      if (!CsIntegerP(arg)) CsUnexpectedTypeError(c, arg, "integer");
      gotone  = true;
      int_t t = CsIntegerValue(arg);
      if (t < r) r = t;
    }
    return gotone ? CsMakeInteger(r) : UNDEFINED_VALUE;
  }

  static value maximum(VM *c, value *argv, int argc) {
    value arg;
    int_t r      = INT_MIN;
    bool  gotone = false;
    for (int i = 0; i < argc; ++i) {
      arg = argv[i];
      if (CsVectorP(arg)) {
        arg = maximum(c, CsVectorAddress(c, arg), CsVectorSize(c, arg));
      }
      if (!CsIntegerP(arg)) CsUnexpectedTypeError(c, arg, "integer");
      gotone  = true;
      int_t t = CsIntegerValue(arg);
      if (t > r) r = t;
    }
    return gotone ? CsMakeInteger(r) : UNDEFINED_VALUE;
  }

  static value CSF_min(VM *c) {
    value *argv = c->argv - 2;
    int    argc = c->argc - 2;

    return minimum(c, argv - argc, argc);
  }
  static value CSF_max(VM *c) {
    value *argv = c->argv - 2;
    int    argc = c->argc - 2;

    return maximum(c, argv - argc, argc);
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    int         radix = 10;
    char        buf[100];
    const char *fmt;
    value       obj;
    CsParseArguments(c, "V=*|i", &obj, &CsIntegerDispatch, &radix);
    switch (radix) {
    case 8: fmt = "%o"; break;
    case 10: fmt = "%d"; break;
    case 16: fmt = "%x"; break;
    default: return UNDEFINED_VALUE;
    }
    sprintf(buf, fmt, (int)CsIntegerValue(obj));
    return CsMakeCString(c, buf);
  }

  /* CSF_limit - built-in method 'limit' */
  static value CSF_limit(VM *c) {
    int v, minv, maxv;
    CsParseArguments(c, "i*ii", &v, &minv, &maxv);
    if (minv > maxv) return UNDEFINED_VALUE;
    if (v < minv) return CsMakeInteger(minv);
    if (v > maxv) return CsMakeInteger(maxv);
    return CsMakeInteger(v);
  }

  static bool  GetIntegerProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetIntegerProperty(VM *c, value obj, value tag, value value);
  static bool  IntegerPrint(VM *c, value obj, stream *s, bool toLocale);
  static long  IntegerSize(value obj);
  static value IntegerCopy(VM *c, value obj);
  static int_t IntegerHash(value obj);
  static value IntegerNextElement(VM *c, value *index, value obj, int nr) {
    int_t idx = 0;
    if (*index != NOTHING_VALUE) idx = CsIntegerValue(*index) + 1; // not first
    if (idx >= CsIntegerValue(obj)) return NOTHING_VALUE;
    return (*index = CsMakeInteger(idx));
  }

  // returns true/false if bit is set
  value IntegerGetItem(VM *c, value obj, value tag) {
    if (!CsIntegerP(tag)) return UNDEFINED_VALUE;

    int_t bit = CsIntegerValue(tag);
    if (bit < 0 || bit >= 32) return UNDEFINED_VALUE;

    return tool::get_bit((uint32)(1 << bit), (uint)CsIntegerValue(obj))
               ? TRUE_VALUE
               : FALSE_VALUE;
  }

  dispatch CsIntegerDispatch = {
      "Integer",          &CsIntegerDispatch,   GetIntegerProperty,
      SetIntegerProperty, CsDefaultNewInstance, IntegerPrint,
      IntegerSize,        IntegerCopy,          CsDefaultScan,
      IntegerHash,        IntegerGetItem,       CsDefaultSetItem,
      IntegerNextElement,
  };

  /* GetIntegerProperty - Integer get property handler */
  static bool GetIntegerProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->integerObject, tag, pValue);
  }

  /* SetIntegerProperty - Integer set property handler */
  static bool SetIntegerProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->integerObject, tag, value);
  }

  /* IntegerPrint - Integer print handler */
  static bool IntegerPrint(VM *c, value obj, stream *s, bool toLocale) {
    return s->put_str(itow(CsIntegerValue(obj)));
  }

  /* IntegerSize - Integer size handler */
  static long IntegerSize(value obj) {
    // return sizeof(CsInteger);
    return sizeof(value);
  }

  /* IntegerCopy - Integer copy handler */
  static value IntegerCopy(VM *c, value obj) {
    // return CsPointerP(obj) ? CsDefaultCopy(c,obj) : obj;
    return obj;
  }

  /* IntegerHash - Integer hash handler */
  static int_t IntegerHash(value obj) {
    return tool::hash(CsIntegerValue(obj));
  }

  /* CsMakeInteger - make a new integer value
  value CsMakeInteger(VM *c,int_t val)
  {
      if (CsSmallIntegerValueP(val))
          return CsMakeSmallInteger(val);
      else {
          value newo = CsAllocate(c,sizeof(CsInteger));
          CsSetDispatch(newo,&CsIntegerDispatch);
          CsSetHeapIntegerValue(newo,val);
          return newo;
      }
  }
  */

} // namespace tis
