/* int.c - bytecode interpreter */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include "cs.h"
#include "cs_int.h"
#include "cs_com.h"

#pragma optimize("t", on)

namespace tis {

  /* types */
  typedef struct FrameDispatch FrameDispatch;

  /* frame */
  struct CsFrame {
    FrameDispatch *pdispatch;
    // CsFrame *next;
    stack_ptr<CsFrame> next;
  };

  /* frame pdispatch structure */
  struct FrameDispatch {
    bool (*restore)(VM *c); // true - execution compelted
    value *(*copy)(VM *c, CsFrame *frame);
  };

  /* call frame pdispatch */
  static bool   CallRestore(VM *c);
  static value *CallCopy(VM *c, CsFrame *frame);
  FrameDispatch CsCallCDispatch = {CallRestore, CallCopy};

  /* call frame */
  typedef struct CallFrame CallFrame;
  struct CallFrame : CsFrame {
    value env;
    //value globals;
    value ns;
    value code;
    value method;
    int   pcOffset;
    int   argc;

    CsEnvironment stackEnv;
  };

  /* top frame pdispatch */
  static bool   TopRestore(VM *c);
  FrameDispatch CsTopCDispatch = {TopRestore, CallCopy};

  /* block frame pdispatch */
  static bool   BlockRestore(VM *c);
  static value *BlockCopy(VM *c, CsFrame *frame);
  FrameDispatch CsBlockCDispatch = {BlockRestore, BlockCopy};

  /* block frame */
  typedef struct BlockFrame BlockFrame;
  struct BlockFrame : CsFrame {
    // value *fp;
    // void* align;
    value         env;
    CsEnvironment stackEnv;
  };

  /* macro to convert a byte size to a stack entry size */
  inline uint WordSize(size_t n) {
    assert((n % sizeof(value)) == 0);
    return uint(n / sizeof(value)); // + (n % sizeof(value))?1:0;
  }

  /* prototypes */
  static value ExecuteCall(CsScope *scope, value fun, int argc, va_list ap);
  int          Send(VM *c, FrameDispatch *d, int argc);
  void         UnaryOp(VM *c, int op);
  inline void  BinaryOp(VM *c, int op) {
    value p1  = value_to_set(CsPop(c));
    value p2  = value_to_set(c->val);
    c->val = CsBinaryOp(c, op, p1, p2);
  }

  static bool  Call(VM *c, FrameDispatch *d, int argc);
  static void  PushFrame(VM *c, int size, value names = UNDEFINED_VALUE);
  static value UnstackEnv(VM *c, value env);
  static void  BadOpcode(VM *c, int opcode);
  static int   CompareStrings(value str1, value str2);
  // static value ConcatenateStrings(VM *c,value str1,value str2);

  static value CsGetRange(VM *c, value col, value start, value end);

  int CsGetLineNumber(VM *c, value ccode, int pc);

  bool CsAttachTaskToPromise(VM *c, value promise, value taskInstance);

  // static int cnt = 0;

  CsSavedState::CsSavedState(VM *vm) : next(0), catch_pcoff(-1) {
    //++cnt;
    store(vm);
    assert(vm->exec);
    next             = vm->exec->states;
    vm->exec->states = this;
  }

  CsSavedState::~CsSavedState() {
    if (!vm || !vm->exec) return;
    //assert(vm->exec->states == nullptr || vm->exec->states == this);
    if (vm->exec->states != this) return;
    vm->exec->states = next;
    //--cnt;
    // dbg_printf("~CsSavedState, total=%d\n",cnt);
  }

  /* CsSaveInterpreterState - save the current interpreter state */
  void CsSavedState::store(VM *c) {
    vm      = c;
    globals = c->currentScope()->globals;
    ns      = c->currentNS;
    sp      = c->sp;
    fp      = c->fp;
    env     = c->env;
    if ((code = c->code) != 0) pcoff = long(c->pc - c->cbase);
    stack    = c->stack;
    stackTop = c->stackTop;
    // currentTaskInstance = c->currentTaskInstance;
  }

  void CsSavedState::scan(VM *c) {
    globals = CsCopyValue(c, globals);
    ns      = CsCopyValue(c, ns);
    env     = CsCopyValue(c, env);
    if (code != 0) code = CsCopyValue(c, code);
    if (this->stackTop && (this->stackTop != c->stackTop)) {
      CsFrame *fp = this->fp;
      value *  sp = this->sp;
      while (sp < this->stackTop) {
        if (sp >= (value *)fp) {
          sp = (*fp->pdispatch->copy)(c, fp);
          fp = fp->next.get(this->stackTop);
        } else {
          *sp = CsCopyValue(c, *sp);
          ++sp;
        }
      }
    }
  }

  /* CsRestoreInterpreterState - restore the previous interpreter state */
  void CsSavedState::restore() {
    // if(!isDetached) {
    vm->currentNS = ns;
    // vm->scopes->globals = globals;
    vm->sp  = sp;
    vm->fp  = fp;
    vm->env = env;

    if (code != 0) {
      value t   = vm->code;
      vm->code  = code;
      vm->cbase = CsByteVectorAddress(CsCompiledCodeBytecodes(vm->code));
      vm->pc    = vm->cbase + pcoff;
#if defined(TISCRIPT_DEBUGGER)
      if ((t != code) && vm->pdebug.is_defined()) {
        int firstLine = CsGetLineNumber(vm, vm->code, int(vm->pc - vm->cbase));
        int lastLine  = CsGetLineNumber(
            vm, vm->code,
            int(vm->pc - vm->cbase) +
                (int)CsByteVectorSize(CsCompiledCodeBytecodes(vm->code)) - 2);
        vm->pdebug->enter_function(vm, code, firstLine, lastLine);
      }
#endif
    } else {
      vm->code  = 0;
      vm->cbase = 0;
      vm->pc    = 0;
    }
    vm->stack                   = stack;
    vm->stackTop                = stackTop;
    vm->currentScope()->globals = globals;
    // c->savedState = next; -- delegate it to destructor?
  }

  /* CsCallFunction - call a function */
  value CsCallFunction(CsScope *scope, value fun, int argc, ...) {
    value   result;
    va_list ap;

    /* call the function */
    va_start(ap, argc);
    result = ExecuteCall(scope, fun, argc, ap);
    va_end(ap);

    /* return the result */
    return result;
  }

  /* CsCallFunctionByName - call a function by name
  value CsCallFunctionByName(CsScope *scope,char *fname,int argc,...)
  {
      VM *c = scope->c;
      value fun,result;
      va_list ap;

    // get the symbol value
    CsCPush(c,CsSymbolOf(fname));
    if (!CsGlobalValue(scope,CsTop(c),&fun))
        CsThrowKnownError(c,CsErrUnboundVariable,CsTop(c));
    CsDrop(c,1);

      // call the function
      va_start(ap,argc);
      result = ExecuteCall(scope,fun,argc,ap);
      va_end(ap);

      // return the result
      return result;
  }*/

  /* ExecuteCall - execute a function call */
  static value ExecuteCall(CsScope *scope, value fun, int argc, va_list ap) {
    VM *c = scope->c;

    int n;

    /* save the interpreter state */
    CsSavedState state(c);
    // auto_scope as(c,c->scopes->globals);

    TRY {

      /* push the function */
      CsCheck(c, argc + 3);
      CsPush(c, fun);
      CsPush(c, scope->globals);
      CsPush(c, scope->globals);

      /* push the arguments */
      for (n = argc; --n >= 0;)
        CsPush(c, va_arg(ap, value));
      va_end(ap);

      /* setup the call */
      if (Call(c, &CsTopCDispatch, argc + 2)) { return c->val; }
      /* execute the function code */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  /* ExecuteCall - execute a function call */
  value CsCallFunction(CsScope *scope, value fun, vargs &args) {
    VM *c = scope->c;

    int n;

    /* save the interpreter state */
    CsSavedState state(c);
    // auto_scope as(c,c->scopes->globals);

    TRY {
      /* push the function */
      CsCheck(c, args.count() + 3);
      CsPush(c, fun);
      CsPush(c, scope->globals);
      CsPush(c, scope->globals);

      /* push the arguments */
      int argc = args.count();

      for (n = 0; n < argc; ++n) // for (n = argc; --n >= 0; )
        CsPush(c, args.nth(n));

      /* setup the call */
      if (Call(c, &CsTopCDispatch, argc + 2)) { return c->val; }
      /* execute the function code */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  value CsSendMessage(VM *c, value obj, value selector, int argc, ...) {
    va_list ap;
    int     n;

    /* save the interpreter state */
    CsSavedState state(c);
    // auto_scope as(c,c->scopes->globals);

    TRY {
      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
      CsPush(c, selector);
      CsPush(c, obj); /* _next */

      /* push the arguments */
      va_start(ap, argc);
      for (n = argc; --n >= 0;)
        CsPush(c, va_arg(ap, value));
      va_end(ap);

      /* setup the call */
      if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }
        /* execute the method */

#ifdef _DEBUG
      if (!c->code) c->code = c->code;
#endif

      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  /* CsSendMessage - send a message to an obj */
  value CsCallMethod(VM *c, value obj, value method, value ofClass, int argc,...) {

    va_list ap;
    int     n;

    /* save the interpreter state */
    CsSavedState state(c);

    TRY {
      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
#ifdef DEBUG
      //dispatch* pd = CsGetDispatch(method);
      //assert(CsMethodP(method));
#endif // DEBUG

      CsPush(c, method);
      CsPush(c, ofClass); /* _next */

      /* push the arguments */
      va_start(ap, argc);
      for (n = argc; --n >= 0;)
        CsPush(c, va_arg(ap, value));
      va_end(ap);

      /* setup the call */
      if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }
      /* execute the method */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

#if 0
  /* CsSendMessage - send a message to an obj */
  value CsCallMethodV(VM *c, value obj, value method, value ofClass, int argc, va_list ap) {

    int     n;

    /* save the interpreter state */
    CsSavedState state(c);

    TRY{
      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

    /* push the obj, selector and _next argument */
    CsPush(c, obj);
    assert(CsMethodP(method));
    CsPush(c, method);
    CsPush(c, ofClass); /* _next */

                        /* push the arguments */
    for (n = argc; --n >= 0;)
      CsPush(c, va_arg(ap, value));

    /* setup the call */
    if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }
    /* execute the method */
    c->exec->run(c);
    }
      CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  value CsCallMethod(VM *c, value obj, value method, value ofClass, int argc, ...)
  {
    va_list args;
    va_start(args, argc);
    value mns = CsMethodNamespace(method);
    value r;
    if (c->currentScope()->globals != mns) {
      auto_scope _(c, mns);
      r = CsCallMethodV(c, obj, method, ofClass, argc, args);
    }
    else
      r = CsCallMethodV(c, obj, method, ofClass, argc, args);
    va_end(args);
    return r;
  }

#endif

  value CsSendMessage(CsScope *scope, value obj, value selector, int argc,  ...) {
    VM *c = scope->c;

    va_list ap;
    int     n;

    /* save the interpreter state */
    CsSavedState state(c);

    TRY {
      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
      CsPush(c, selector);
      CsPush(c, obj); /* _next */

      /* push the arguments */
      va_start(ap, argc);
      for (n = argc; --n >= 0;)
        CsPush(c, va_arg(ap, value));
      va_end(ap);

      /* setup the call */
      if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }
      /* execute the method */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  value CsSendMessage(VM *c, value obj, value selector, const value *argv,
                      int argc) {
    int n;

    if (obj == NULL_VALUE) obj = CsCurrentScope(c)->globals;

    /* save the interpreter state */
    CsSavedState state(c);

    TRY {
      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
      CsPush(c, selector);
      CsPush(c, obj); /* _next */

      /* push the arguments */
      for (n = argc; --n >= 0;)
        CsPush(c, *argv++);

      /* setup the call */
      if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }
      /* execute the method */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }



  /* CsSendMessageByName - send a message to an obj by name */
  value CsSendMessageByName(VM *c, value obj, char *sname, int argc, ...) {
    va_list ap;
    int     n;

    /* save the interpreter state */
    CsSavedState state(c);
    // auto_scope as(c,c->scopes->globals);

    TRY {

      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
      CsPush(c, UNDEFINED_VALUE); /* filled in below */
      CsPush(c, obj);             /* _next */

      /* push the arguments */
      va_start(ap, argc);
      for (n = argc; --n >= 0;)
        CsPush(c, va_arg(ap, value));
      va_end(ap);

      /* fill in the selector (because interning can cons) */
      c->sp[argc + 1] = CsSymbolOf(sname);

      /* setup the call */
      if (Send(c, &CsTopCDispatch, argc + 2)) { return c->val; }

      /* execute the method */
      c->exec->run(c);
    }
    CATCH_ERROR(e) {
      state.restore();
      RETHROW(e);
    }
    return c->val;
  }

  /*inline void StreamOut(VM *c)
  {
      static value print_sym = 0;
      if(!print_sym)
        print_sym = CsMakeSymbol(c,"print",5);
      value strm = CsPop(c);
      CsSendMessage(c,strm,print_sym,1,c->val);
      c->val = strm;
  }*/

  static int UnspreadArgs(VM* c, int argc);
  value CsCurrentCodeLocation(VM* c);

  INLINE unsigned int getcword(VM *c) {
    uint off = *c->pc++;
         off |= *c->pc++ << 8;
    return off;
  }

  /* Execute - execute code */
  bool Exec::run(VM *c, bool riseError) {
    value        p1, p2, *p;
    unsigned int off = 0;
    int n;
    int i;
#ifdef TISCRIPT_DEBUGGER
    debug_env dbg(c);
#endif
    // int throwLevel = 0;
    awaiting = false;
  START:
    TRY {

      if (riseError) {
        // val[0] contains the error;
        riseError = false;
        THROW_ERROR(CsErrThrown);
      }

      for (;;) {

        /*DecodeInstruction(c,c->code,c->pc - c->cbase,c->standardOutput);*/
        switch (*c->pc++) {
          case BC_NOP:
            break;
#ifdef TISCRIPT_DEBUGGER
        case BC_LINENO:
          off = getcword(c);
          // dbg.check_line_no(off);
          if (c->pdebug) c->pdebug->check_line_no(c, off);
          break;
#endif
        case BC_CALL: Call(c, &CsCallCDispatch, *c->pc++); break;
        case BC_CALL_WITH_SPREADS: { int argc = UnspreadArgs(c,*c->pc++); Call(c, &CsCallCDispatch, argc); break; }
        case BC_SEND: Send(c, &CsCallCDispatch, *c->pc++); break;
        case BC_SEND_WITH_SPREADS: { int argc = UnspreadArgs(c, *c->pc++); Send(c, &CsCallCDispatch, argc); break; }
        case BC_UNFRAME:
          if ((*c->fp->pdispatch->restore)(c)) return true;
          break;
        case BC_FRAME:
          i   = *c->pc++;
          off = getcword(c);
          CsCheck(c, i);
          for (n = i; --n >= 0;)
            CsPush(c, UNDEFINED_VALUE);
          PushFrame(c, i, CsCompiledCodeLiteral(c->code, off));
          break;
        case BC_AFRAME: /* handled by BC_CALL */
        case BC_AFRAMER: BadOpcode(c, c->pc[-1]); break;
        case BC_ARGSGE:
          i = *c->pc++;
          c->val = CsMakeBoolean(c->argc >= i);
          break;
        case BC_CLOSE: {
          c->env            = UnstackEnv(c, c->env);
          FUNCTION_TYPE fct = (FUNCTION_TYPE)*c->pc++;
          switch (fct) {
          case EVENT: {
            c->val = CsMakeMethod(c, c->val, c->env, c->getCurrentNS());
            value nameAggregate = CsPop(c);
            value methodCode    = CsMethodCode(c->val);
            assert(CsCompiledCodeP(methodCode));
            if (CsCompiledCodeP(methodCode))
              CsSetCompiledCodeName(methodCode, nameAggregate);
            else
              CsThrowKnownError(c, CsErrBadParseCode);
          } break;
          case ASYNC:
          case GENERATOR:
          case FUNCTION:
          case FAT_ARROW:
            c->val = CsMakeMethod(c, c->val, c->env, c->getCurrentNS());
            break;
          case PROPERTY:
          case UNDEFINED_PROPERTY:
            c->val = CsMakePropertyMethod(c, c->val, c->env, c->getCurrentNS());
            break;
          default: assert(false); break;
          }
        } break;
        case BC_RETURN:
          assert(c->fp->pdispatch == &CsTopCDispatch ||
                 c->fp->pdispatch == &CsCallCDispatch);
          if ((*c->fp->pdispatch->restore)(c)) return true;
          break;

        case BC_AWAIT:
          // check if the <expr> returned promise
          // if it is a promise we need to do:
          // promise.then(fullfilTaskInstance,rejectTaskInstance) persist
          // current registers and do return
          awaiting = true;
          CsPersistCurrentCoroutine(c);

          if (!CsAttachTaskToPromise(c, c->val, c->exec->taskInstance)) {
            c->pending_tasks.push(c->exec->taskInstance);
            VM *vm = c;
            if(c->pending_tasks.size() == 1)
              c->post([vm]() { return vm->deliver_notifications(); });
          }
          return false; // to indicate that we are not complete yet
                        // break;

        case BC_YIELD:
          awaiting = true;
          CsPersistCurrentCoroutine(c);
          return false; // to indicate that we are not complete yet
                        // val registers contain yielded value(s)

        case BC_EREF:
          i = *c->pc++;
          for (p2 = c->env; --i >= 0;)
            p2 = CsEnvNextFrame(p2);
          i = CsEnvSize(p2) - *c->pc++;
          c->val = CsEnvElement(p2, i);
          break;
        case BC_ESET:
          i = *c->pc++;
          for (p2 = c->env; --i >= 0;)
            p2 = CsEnvNextFrame(p2);
          i = CsEnvSize(p2) - *c->pc++;
          //if (c->val != NOTHING_VALUE) breaks generator
          CsSetEnvElement(p2, i, value_to_set(c->val));
          break;

        case BC_BRT:
          off = getcword(c);
          if (CsToBoolean(c, c->val) == TRUE_VALUE) c->pc = c->cbase + off;
          break;
        case BC_BRDEF: // branch if val != nothingValue
          off = getcword(c);
          if (c->val != NOTHING_VALUE) c->pc = c->cbase + off;
          break;
        case BC_BRUNDEF:
          // c->val = ( CsGetArg(c,3) != NOTHING_VALUE )? TRUE_VALUE :
          // FALSE_VALUE;
          off = getcword(c);
          if (c->val == NOTHING_VALUE) c->pc = c->cbase + off;
          break;
        case BC_BR_UNDEFINED:
          off = getcword(c);
          if (c->val == NOTHING_VALUE || c->val == UNDEFINED_VALUE || c->val == NULL_VALUE) c->pc = c->cbase + off;
          break;
        case BC_BRF:
          off = getcword(c);
          if (CsToBoolean(c, c->val) == FALSE_VALUE) c->pc = c->cbase + off;
          break;
        case BC_BR:
          off = getcword(c);
          c->pc = c->cbase + off;
          break;
        case BC_SWITCH:
          i = (int)getcword(c);
          while (--i >= 0) {
            off = *c->pc++;
            off |= *c->pc++ << 8;
            if (CsEqualOp(c, c->val, CsCompiledCodeLiteral(c->code, off)))
              break;
            c->pc += 2;
          }
          off = getcword(c);
          c->pc = c->cbase + off;
          break;
        case BC_T: c->val = TRUE_VALUE; break;
        case BC_F: c->val = FALSE_VALUE; break;
        case BC_NULL: c->val = NULL_VALUE; break;
        case BC_UNDEFINED: c->val = UNDEFINED_VALUE; break;
        case BC_NOTHING: c->val = NOTHING_VALUE; break;
        case BC_PUSH: CsCPush(c, c->val); break;
        case BC_PUSH_VALUE: CsCPush(c, scalar_value(c->val)); break;

        case BC_SPREAD:
             c->val = CsMakeFixedVector(c, &CsSpreadDispatch, 1, &c->val);
             break;

        case BC_NOT:
          c->val = CsToBoolean(c, c->val) == TRUE_VALUE ? FALSE_VALUE
                                                              : TRUE_VALUE;
          break;
        case BC_NEG: UnaryOp(c, '-'); break;
        case BC_ADD_EQ:
          if (CsStringP(c->val)) {
            tool::swap(CsTop(c), c->val);
            if (!CsStringP(c->val)) c->val = CsToString(c, c->val);
            c->val = CsConcatStrings(c, c->val, CsPop(c), true);
          } else if (CsStringP(CsTop(c))) {
            if (!CsStringP(c->val)) c->val = CsToString(c, c->val);
            c->val = CsConcatStrings(c, CsPop(c), c->val, false);
          } else
            BinaryOp(c, BC_ADD);
          break;
        case BC_ADD:
          if (CsStringP(c->val)) {
            tool::swap(CsTop(c), c->val);
            if (!CsStringP(c->val)) c->val = CsToString(c, c->val);
            c->val = CsConcatStrings(c, c->val, CsPop(c), false);
          } else if (CsStringP(CsTop(c))) {
            if (!CsStringP(c->val)) c->val = CsToString(c, c->val);
            c->val = CsConcatStrings(c, CsPop(c), c->val, false);
          } else
            BinaryOp(c, BC_ADD);
          break;
        case BC_SUB: BinaryOp(c, BC_SUB); break;
        case BC_MUL: BinaryOp(c, BC_MUL); break;
        case BC_DIV: BinaryOp(c, BC_DIV); break;
        case BC_REM: BinaryOp(c, BC_REM /*'%'*/); break;
        case BC_INC: UnaryOp(c, 'I'); break;
        case BC_DEC: UnaryOp(c, 'D'); break;
        case BC_BAND: BinaryOp(c, BC_BAND /*'&'*/); break;
        case BC_BOR: BinaryOp(c, BC_BOR /*'|'*/); break;
        case BC_XOR: BinaryOp(c, BC_XOR /*'^'*/); break;
        case BC_BNOT: UnaryOp(c, '~'); break;
        case BC_SHL: {
          dispatch *pd = CsGetDispatch(CsTop(c));
          if (pd->binOp) {
            value obj = CsPop(c);
            c->val = pd->binOp(c, BC_SHL, obj, c->val);
          }
          else
            BinaryOp(c, BC_SHL /*'L'*/);
        } break;
        case BC_SHR: {
          dispatch *pd = CsGetDispatch(CsTop(c));
          if (pd->binOp) {
            value obj = CsPop(c);
            c->val = pd->binOp(c, BC_SHR, obj, c->val);
          } else
            BinaryOp(c, BC_SHR /*'R'*/);
        } break;
        case BC_USHL: BinaryOp(c, BC_USHL /*'l'*/); break;
        case BC_USHR: BinaryOp(c, BC_USHR /*'r'*/); break;
        case BC_LT:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsCompareObjects(c, p1, c->val) < 0);
          break;
        case BC_LE:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsCompareObjects(c, p1, c->val) <= 0);
          break;
        case BC_EQ:
          p1 = CsPop(c);
          c->val =
              CsMakeBoolean(CsEqualOp(c, p1, c->val)); // symbol == string
          break;
        case BC_NE:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(!CsEqualOp(c, p1, c->val)); //
          break;

        case BC_EQ_STRONG:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsStrongEql(p1, c->val));
          break;
        case BC_NE_STRONG:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(!CsStrongEql(p1, c->val));
          break;

        case BC_GE:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsCompareObjects(c, p1, c->val) >= 0);
          break;
        case BC_GT:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsCompareObjects(c, p1, c->val) > 0);
          break;

        case BC_MATCH: // 'like' in 'switch'
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsMatch(c, c->val, p1));
          /*
            switch(val) {
              like {one:any} : expr1; break;
              like {two:2} : expr2; break;
              case "some": expr3; break;
              default: expr4; break;
            }
          */
          break;

        case BC_LIT:
          off = getcword(c);
          c->val = CsCompiledCodeLiteral(c->code, off);
          break;

        case BC_GREF:
          off = getcword(c);
          if (!CsGetGlobalOrNamespaceValue(c, CsCompiledCodeLiteral(c->code, off), &c->val))
            //CsThrowKnownError(c, CsErrUnboundVariable,CsCompiledCodeLiteral(c->code, off));
            c->val = NOTHING_VALUE;
          break;
        case BC_GSET:
          off = getcword(c);
          //if(c->val != NOTHING_VALUE) breaks generator
          CsSetGlobalOrNamespaceValue(c, CsCompiledCodeLiteral(c->code, off), value_to_set(c->val), false);
          break;
        case BC_GSETNEW: // create new global variable
          off = getcword(c);
          CsSetGlobalOrNamespaceValue(c, CsCompiledCodeLiteral(c->code, off),
                                      value_to_set(c->val), true);
          break;

        case BC_GSETNS:
          off = getcword(c);
          //if (c->val != NOTHING_VALUE) breaks generator
          CsSetNamespaceValue(c, CsCompiledCodeLiteral(c->code, off), value_to_set(c->val), false, true);
          break;

        case BC_GSETNS_NEW:
          off = getcword(c);
          CsSetNamespaceValue(c, CsCompiledCodeLiteral(c->code, off), value_to_set(c->val), true);
          break;
        case BC_GSETNS_NEW_CONST:
          off = getcword(c);
          CsSetNamespaceConst(c, CsCompiledCodeLiteral(c->code, off), value_to_set(c->val));
          break;

        case BC_GSETNS_GETTER:
          off = getcword(c);
          CsSetNamespaceGetterSetter(c, CsCompiledCodeLiteral(c->code, off), c->val, false);
          break;
        case BC_GSETNS_SETTER:
          off = getcword(c);
          CsSetNamespaceGetterSetter(c, CsCompiledCodeLiteral(c->code, off), c->val, true);
          break;

        case BC_MTSET:
          off = getcword(c);
          CsSetClassMemberTemplateVar(c, CsCompiledCodeLiteral(c->code, off),value_to_set(c->val));
          break;

        case BC_ADD_EVENT:
          p1 = CsPop(c); // event method
          CsEventObjectAddEF(c, c->val, p1);
          break;
        case BC_GETP:
          p1 = CsPop(c);
          if (!CsGetProperty(c, p1, c->val, &c->val))
            c->val = UNDEFINED_VALUE;
          break;
        case BC_GETP_NTH:
          {
            n = getcword(c);
            if(!CsDerivedFromObjectP(c->val) || (n >= CsObjectPropertyCount(c->val)) )
              c->val = NOTHING_VALUE;
            else {
              each_property iter(c, c->val);
              for (value k, v; iter(k, v);) {
                if (iter.ordinal == uint(n)) {
                  c->val = v;
                  break;
                }
              }
            }
          } break;

        case BC_SETP:
          p2 = CsPop(c);
          p1 = CsPop(c);
          if (!CsSetProperty(c, p1, p2, value_to_set(c->val))) {
            // string msg = CsSymbolName(p2);
            // const char* pname = msg;
            CsThrowKnownError(c, CsErrNoProperty, p1, p2);
          }
          break;
        case BC_SETPM: // set method, used in class declarations only
          p2 = CsPop(c);
          p1 = CsPop(c);
          if (CsObjectOrMethodP(p1) || CsIsType(p1, c->typeDispatch)) {
            if (CsMethodP(c->val) && (CsClassP(p1) || CsNamespaceP(p1))) {
                CsSetMethodNamespace(c->val, p1);
            }
            CsSetProperty1(c, p1, p2, c->val);
          }
          else {
            dispatch *pd = CsGetDispatch(p1);
            pd           = pd;
            CsThrowKnownError(c, CsErrUnexpectedTypeError, p1,
                              "either <class> or <object>");
          }
          break;

        case BC_VREF:
          p1     = CsPop(c);
          c->val = CsGetItem(c, p1, c->val);
          break;
        case BC_VSET:
          p2 = CsPop(c);
          p1 = CsPop(c);
          CsSetItem(c, p1, p2, value_to_set(c->val));
          break;

        case BC_GET_TOP_VECTOR_ELEMENT:
        {
          int i = *c->pc++;
          value vec = CsTop(c);
          if (CsVectorP(vec))
            c->val = i < CsVectorSize(c, vec) ? CsVectorElement(c, vec, i) : NOTHING_VALUE;
          else if (CsTupleP(vec))
            c->val = i < CsTupleSize(vec) ? CsTupleElement(vec, i) : NOTHING_VALUE;
          else
            c->val = NOTHING_VALUE;
        } break;

        case BC_GET_TOP_VECTOR_REST_ELEMENTS:
        {
          int i = *c->pc++;
          value vec = CsTop(c);
          if (CsVectorP(vec)) {
            c->val = CsMakeVector(c, max(0, CsVectorSize(c, vec) - i));
            vec = CsTop(c);
            CsVectorTargetElements(c->val).copy(CsVectorElements(c,vec)(i));
          }
          else if (CsTupleP(vec)) {
            c->val = CsMakeVector(c, max(0, CsTupleSize(vec) - i));
            vec = CsTop(c);
            CsVectorTargetElements(c->val).copy(CsTupleElements(vec)(i));
          }
          else
            c->val = UNDEFINED_VALUE;
        } break;

        case BC_GET_TOP_MAP_ELEMENT:
        {
          off = getcword(c);
          value obj = CsTop(c);
          if (CsDerivedFromObjectP(obj)) {
            c->val = CsCompiledCodeLiteral(c->code, off);
            if (!CsGetProperty(c, CsTop(c), c->val, &c->val))
              c->val = NOTHING_VALUE;
          }
          else
            c->val = NOTHING_VALUE;
        } break;

        case BC_CLONE_TOP_MAP_EXCEPT: {
          uint exceptions_cnt = *c->pc++;
          array<value> ex_list(exceptions_cnt);
          for (uint n = 0; n < exceptions_cnt; ++n) {
            ex_list[n] = CsCompiledCodeLiteral(c->code, getcword(c));
            assert(CsSymbolP(ex_list[n]));
          }
          value obj = CsTop(c);
          if (CsDerivedFromObjectP(obj)) {
            c->val = CsCloneObject(c, obj, slice<value>(), ex_list());
          }

        } break;

        case BC_EXTEND_TOP_MAP: {
          if(CsObjectP(c->val))
            CsExtendObject(c, CsTop(c), c->val, false);
        } break;

        case BC_DELP:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsDelProperty(c, p1, c->val));
          break;
        case BC_GDEL:
          off = *c->pc++;
          off |= *c->pc++ << 8;
          c->val = CsMakeBoolean(CsDelGlobalOrNamespaceValue(
              c, CsCompiledCodeLiteral(c->code, off)));
          // CsSetGlobalOrNamespaceValue(c,CsCompiledCodeLiteral(c->code,off),UNDEFINED_VALUE,false);
          break;
        case BC_VDEL:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsDelItem(c, p1, c->val));
          break;
        case BC_DUP2:
          CsCheck(c, 2);
          c->sp -= 2;
          c->sp[1] = c->val;
          CsSetTop(c, c->sp[2]);
          break;
        case BC_DROP: c->val = CsPop(c); break;
        case BC_DUP:
          CsCheck(c, 1);
          c->sp -= 1;
          CsSetTop(c, c->sp[1]);
          break;
        case BC_OVER:
          CsCheck(c, 1);
          c->sp -= 1;
          CsSetTop(c, c->sp[2]);
          break;
        case BC_PUSHSCOPE:
          CsCheck(c, 2);
          CsPush(c, c->getCurrentNS());
          CsPush(c, CsObjectClass(c->getCurrentNS()));
          // CsPush(c,c->currentScope()->globals);
          // CsPush(c,c->currentScope()->globals);
          break;
        case BC_PUSH_NS: // this used in class construction
          CsCheck(c, 1);
          // CsPush(c,c->currentScope()->globals);
          // c->currentScope()->globals = c->val;
          CsPush(c, c->currentNS);
          c->currentNS = c->val;
          break;
        case BC_POP_NS:
          // c->currentScope()->globals = CsPop(c);
          c->currentNS = CsPop(c);
          break;
        case BC_NS: c->val = c->getCurrentNS(); break;
        case BC_ROOT_NS: c->val = CsCurrentScope(c)->globals; break;

        case BC_DEBUG:
          // c->val = c->val;
          // c->standardOutput->put_str("|");
          n = *c->pc++;
          switch (n) {
          case 0:
            c->standardOutput->put_str("\ndebug namespaces:\n");
            CsDumpScopes(c);
            break;
          case 1:
            c->standardOutput->put_str("\ndebug stacktrace:\n");
            CsStackTrace(c);
            break;
          case 2:
            off = *c->pc++;
            off |= *c->pc++ << 8;
            if (c->pdebug) c->pdebug->handle_breakpoint_hit(c, off);
            break;
          }
          break;
        case BC_TRACE: c->val = CsMakeStackTrace(c); break;
        case BC_NEWOBJECT:
          if (*c->pc++) {
            CsCheck(c, 4);
            if (CsMethodP(c->val)) {
              value proto = CsGetMethodPrototype(c, c->val);
              p1 = CsMakeObject(c, proto);
              CsPush(c, p1);        // sp[3] - obj
              CsPush(c, p1);        // sp[2] - obj
              CsPush(c, c->val);    // sp[1] - method
              CsPush(c, c->val);    // sp[0] - method
            }
            else {
              p1 = CsNewInstance(c, c->val);
              CsPush(c, p1);        // sp[3] - obj
              CsPush(c, p1);        // sp[2] - obj
              CsPush(c, THIS_SYM);  // sp[1] - #this
              CsPush(c, c->val);    // sp[0] - class
            }
          } else {
            // literal creation case
            c->val = CsNewInstance(c, c->val);
          }
          break;
        case BC_NEWCLASS: {
          bool  is_class    = *c->pc++ != 0;
          value parentClass = CsPop(c);
          if (parentClass == UNDEFINED_VALUE)
            parentClass = c->currentScope()->globals;
          value classNameSymbol = c->val;
          c->val =
              is_class
                  ? CsNewClassInstance(c, parentClass, classNameSymbol)
                  : CsNewNamespaceInstance(c, parentClass, classNameSymbol);
        } break;

        case BC_NEWVECTOR:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          c->val = CsMakeVector(c, n);
          //p = CsVectorAddressI(c->val) + n;
          //while (--n >= 0)
          //  *--p = value_to_set(CsPop(c));
          break;
        }
        /*case BC_NEWVECTOR_WITH_SPREADS:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          c->val = CsMakeVector(c, n);
          p = CsVectorAddressI(c->val) + n;
          while (--n >= 0)
            *--p = value_to_set(CsPop(c));
          c->val = CsUnspreadVector(c, c->val);
          break;
        }*/

        case BC_NEWTUPLE:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          c->val = CsMakeTuple(c, n);
          CsSetTupleName(c->val, CsPop(c));
          break;
        }

        case BC_NEWTUPLE_STACK:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          c->val = CsMakeTuple(c, n);
          p = CsTupleAddress(c->val) + n;
          while (--n >= 0)
            *--p = value_to_set(CsPop(c));
          CsSetTupleName(c->val, CsPop(c));
        } break;

        case BC_VSET_TOP_NTH_ELEMENT_SPREAD:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;

          if (CsVectorP(c->val)) { // spread
            size_t src_len = CsVectorSize(c,c->val);
            size_t dst_len = CsVectorSize(c, CsTop(c));
            if (src_len) // insert elements
            {
              CsResizeVector(c, CsTop(c), int(dst_len + src_len - 1));
              auto dst = CsVectorTargetElements(CsTop(c));
              auto src = CsVectorElements(c, c->val);
              dst.move(n + index_t(src_len) - 1, n, dst_len - n);
              dst.copy(n, slice<value>(src));
            }
            else { // element is an empty array, just remove it from the result, note reduces output size by 1
              auto dst = CsVectorTargetElements(CsTop(c));
              dst.move(n, n + 1, dst_len - n - 1);
              CsResizeVector(c, CsTop(c), int(dst_len - 1));
            }
          }
          else if (CsGeneratorP(c, c->val)) {
            GC_vector expanded(c);
            value gen = c->val;
            PROTECT(gen);
            int i = 0;
            for (; i < 10000; ++i) {
              value next = CsExecGenerator(c, gen);
              if (next == NOTHING_VALUE)
                break;
              expanded.elements.push(next);
            }
            if (i == 10000)
              CsThrowKnownError(c, CsErrGenericError, "too many elements");
            size_t src_len = expanded.elements.length();
            size_t dst_len = CsVectorSize(c, CsTop(c));
            CsResizeVector(c, CsTop(c), int(dst_len + src_len - 1));
            auto dst = CsVectorTargetElements(CsTop(c));
            auto src = expanded.elements();
            dst.move(n + index_t(src_len) - 1, n, dst_len - n);
            dst.copy(n, slice<value>(src));
          }
          else

            CsSetVectorElement(c, CsTop(c),n,c->val);

          break;
        }

        case BC_VSET_TOP_NTH_ELEMENT:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          CsSetVectorElement(c,CsTop(c),n,c->val);
          break;
        }

        case BC_TSET_TOP_NTH_ELEMENT:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;
          CsSetTupleElement(CsTop(c), n, c->val);
          break;
        }

        case BC_TSET_TOP_NTH_ELEMENT_SPREAD:
        {
          n = *c->pc++;
          n |= *c->pc++ << 8;

          assert(CsTupleP(CsTop(c)));

          auto insert_at = [](VM* c, value tuple, int_t pos, slice<value> src) -> value {
            GC_vector dst(c);
            dst.elements = CsTupleElements(tuple);
            dst.elements.remove(pos);
            dst.elements.insert(pos, src);
            value tag = CsTupleName(tuple);
            PROTECT(tag);
            tuple = CsMakeTuple(c, dst.elements.size());
            CsSetTupleName(tuple, tag);
            CsTupleTargetElements(tuple).copy(dst.elements());
            return tuple;
          };

          if (CsVectorP(c->val)) { // spread
            auto src = CsVectorElements(c, c->val);
            CsTop(c) = insert_at(c, CsTop(c), n, src);
          }
          else if (CsTupleP(c->val)) {
            auto src = CsTupleElements(c->val);
            CsTop(c) = insert_at(c, CsTop(c), n, src);
          }
          else if (CsGeneratorP(c, c->val)) {
            GC_vector expanded(c);
            value gen = c->val;
            PROTECT(gen);
            int i = 0;
            for (; i < 10000; ++i) {
              value next = CsExecGenerator(c, gen);
              if (next == NOTHING_VALUE)
                break;
              expanded.elements.push(next);
            }
            if (i == 10000)
              CsThrowKnownError(c, CsErrGenericError, "too many elements");
            auto src = expanded.elements();
            CsTop(c) = insert_at(c, CsTop(c), n, src);
          }
          else {
            CsSetTupleElement(CsTop(c), n, c->val);
          }

          break;
        }


        case BC_THROW: THROW_ERROR(CsErrThrown); break;
        case BC_INSTANCEOF:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsInstanceOf(c, p1, c->val));
          // CsTypeError(c,c->val);
          break;
        case BC_LIKE:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsMatch(c, c->val, p1));
          // CsTypeError(c,c->val);
          break;
        case BC_IN:
          p1        = CsPop(c);
          c->val = CsMakeBoolean(CsHasMember(c, c->val, p1));
          // CsTypeError(c,c->val);
          break;
        case BC_TYPEOF:
          //"number," "string," "boolean," "obj," "function," and "undefined".
          c->val = CsTypeOf(c, c->val);
          break;

        case BC_EH_PUSH:
          off = *c->pc++;
          off |= *c->pc++ << 8;
          {
            // save the interpreter state
            // let's borrow C++ stack for that...
            CsSavedState *pstate = new CsSavedState(c);
            pstate->catch_pcoff  = off;
            ++this->tryLevel;
          }
          break;
        case BC_EH_POP:
          /* } try end successfully reached */
          if (this->states && this->states->isTryState()) {
            assert(this->tryLevel);
            --this->tryLevel;
            delete states;
          }
          break;

        case BC_NEXT:
          n = *c->pc++;
          n |= *c->pc++ << 8;
          c->val = CsGetNextMember(c, &c->sp[0], c->sp[1], n);
          break;

        case BC_THIS_FUNCTION: {
          c->val   = NULL_VALUE;
          CsFrame *fp = c->fp;
          while (fp) {
            if (fp->pdispatch == &CsTopCDispatch) {
              c->val = ((CallFrame *)fp)->method;
              break;
            }
            if (fp->pdispatch == &CsCallCDispatch) {
              c->val = ((CallFrame *)fp)->method;
              break;
            }
            fp = fp->next.get(c);
          }
        } break;

        case BC_OUTPUT: CsDisplay(c, c->val, c->standardOutput); break;
        case BC_ASSERT:
          n = *c->pc++;
          if (CsToBoolean(c, CsTop(c,n - 1)) != TRUE_VALUE) // expr result
          {
            assert(n >= 2);
            //p1 = CsPop(c);                              // where literal
            if (n == 2) {
              p1 = CsPop(c);                              // where literal
              CsPop(c);
              CsThrowKnownError(c, CsErrAssertion, p1);
            }
            else {
              c->val = CsMakeVector(c, n - 2);
              for (int i = n - 2 - 1; i >= 0 ; --i)
                CsSetVectorElementI(c->val,i, CsPop(c));
              p1 = CsPop(c);                              // where literal
              CsPop(c);
              CsThrowKnownError(c, CsErrAssertion2, p1, c->val);
            }
          }
          else
            CsDrop(c, n);
          break;

        case BC_LOG:
          {
            c->standardOutput->flush();
            value location = CsCurrentCodeLocation(c);
            uint flags = uint(*c->pc++);
            n = *c->pc++;
            buffer<value,16> tt(n);
            for (int i = 0; i < n; ++i)
              tt[n - i - 1] = CsPop(c);
            slice<value> list = tt();
            value mode = NULL_VALUE;
            if (flags & 0x01) { mode = list[0]; list.prune(1); }
            bool as_stringizer = (flags & 0x02) != 0;
            c->log_values(list, mode, location, as_stringizer);
          }
          break;

        case BC_GETRANGE:
          p1        = CsPop(c);
          p2        = CsPop(c);
          c->val = CsGetRange(c, p2, p1, c->val);
          break;

        case BC_IMPORT:
        {
          assert(CsStringP(c->val) || CsFileP(c, c->val) || CsByteVectorP(c->val));
          c->val = CsImport(c, c->val);
        }
        break;

        case BC_INCLUDE:
          {
            auto_scope s(c, c->currentNS);
            assert(CsStringP(c->val) || CsFileP(c, c->val) || CsByteVectorP(c->val));
            c->val = CsInclude(CsCurrentScope(c), c->val);
          }
          break;
        case BC_INCLUDE_LIBRARY:
          assert(CsStringP(c->val));
          c->val = CsIncludeLibrary(CsCurrentScope(c), value_to_string(c->val));
          break;
        case BC_S_CALL:
          CsCheck(c, 1);
          off = unsigned(c->pc - c->cbase + 2);
          CsPush(c, int_value(off));
          off = c->pc[0];
          off |= c->pc[1] << 8;
          c->pc = c->cbase + off;
          break;

        case BC_S_RETURN:
          assert(CsIntegerP(CsTop(c)));
          off   = to_int(CsPop(c));
          c->pc = c->cbase + off;
          break;

        case BC_CAR:
          p1 = CsPop(c);
          if (CsStringP(p1))
            c->val = CsStringHead(c, p1, c->val);
          else
            CsUnexpectedTypeError(c, p1, "string");
          break;
        case BC_CDR:
          p1 = CsPop(c);
          if (CsStringP(p1))
            c->val = CsStringTail(c, p1, c->val);
          else
            CsUnexpectedTypeError(c, p1, "string");
          break;
        case BC_RCAR:
          p1 = CsPop(c);
          if (CsStringP(p1))
            c->val = CsStringHeadR(c, p1, c->val);
          else
            CsUnexpectedTypeError(c, p1, "string");
          break;
        case BC_RCDR:
          p1 = CsPop(c);
          if (CsStringP(p1))
            c->val = CsStringTailR(c, p1, c->val);
          else
            CsUnexpectedTypeError(c, p1, "string");
          break;

        case BC_ROTATE: // rotate stack. Used in do_decorator() to move last
                        // param (that is the function) to start of params list.
          n = *c->pc++;
          if (n > 1) {
            i  = 0;
            p  = c->sp;
            p1 = p[0];
            while (++i < n) {
              p[0] = p[1];
              ++p;
            }
            *p = p1;
          }
          break;

        case BC_GET_RVAL:
          {
            value list = CsTop(c);
            int n = *c->pc++;

            if (list == NOTHING_VALUE)
              break;
            //dispatch* pd = CsGetDispatch(list);
            if (!CsValueListP(list)) {
              c->val = list;
              //??? CsTop(c) = NOTHING_VALUE;
              break;
            }
            value *vs = CsFixedVectorAddress(list);
            int    sz = CsFixedVectorSize(list);
            c->val = n < sz ? vs[n] : UNDEFINED_VALUE;
          }
          break;

        case BC_LIST_RVALS:
          n = *c->pc++;
          c->val = CsMakeFixedVectorValue(c, &CsValueListDispatch, n);
          for (int i = 0; i < n; ++i)
            CsSetFixedVectorElement(c->val, n-i-1, CsPop(c));
          break;
        
        case BC_STRINGIFY:
          if (CsVectorP(c->val))
            break;
          else if (CsTupleP(c->val))
            break;
          else if (CsObjectP(c->val))
            break;
          else if (CsStringP(c->val))
            break;
          else if (CsMethodP(c->val))
            break;
          else
            c->val = CsToString(c, c->val);
          break;
        default:
          BadOpcode(c, c->pc[-1]); break;
        }
      }
    }
    CATCH_ERROR(e) {
      if (e.number == 0) {
        this->states->restore();
        throw e;
      }
      // if( this->states == &state ) {
      //}

      if (awaiting) {
        assert(!this->states->isTryState());
        awaiting = false;
        delete this->states; // this is a state created by last BC_AWAIT, remove it
      }

      if (this->tryLevel > 0 && this->states->isTryState()) {
        assert(this->states->isTryState());
        this->states->pcoff = this->states->catch_pcoff;
        this->states->restore(); // it will restore to the pcoff of catch block
        delete this->states;
        --this->tryLevel;
        goto START;
      } else
        throw e; // propagate it further
    }
    return false;
  }

  void CoroutineDestroy(VM *c, value obj);

  void CsExecTask(VM *c, value taskInstance, bool error) {
    assert(CsTaskP(c, taskInstance));

    // assert( c->exec == nullptr ); // this shall run on pristine machine

    CsSavedState state(c);

    PROTECT(taskInstance);

    bool gotError = false;
    bool finsihed = false;

    CsTask *exec = CsCoroutineValue(c, taskInstance);
    if (!exec) return; // destroyed already

    {
      auto_state<Exec *> _1(c->exec, static_cast<Exec *>(exec));
      auto_scope         _2(c, exec->ns);

      if (exec->states) exec->states->restore();

      if (exec->current) {
        assert(exec->states == exec->current);
        delete exec->current;
        exec->current = nullptr;
      }

      finsihed = false;

      if (exec->argc >= 0) {
        c->argc = exec->argc;
        c->argv = &c->sp[exec->argc];
        exec->argc = -1;
      }

      TRY {
        /* execute the function code */
        finsihed = c->exec->run(c, error);
      }
      CATCH_ERROR(e) {
        e = e;

        if (exec->callbacks == NULL_VALUE) {
          // RETHROW(e);
          CsWarning(c, "Task ended with unhandled error:");
          CsHandleUnhandledError(c);
        }
        finsihed = true;
        gotError = true;
      }
    }
    state.restore();

    if (finsihed) {

      CsScope* psc = c->currentScope();
      psc = psc;

     CsTaskNotifyCompletion(c, gotError, exec->callbacks);
      if(exec->isTask)
        c->active_tasks.remove_by_value(taskInstance);
      exec->taskInstance = 0;
      // assert(exec->states == &exec->initial);
      assert(exec->states == nullptr);
      // c->sp = c->stackTop;
      CoroutineDestroy(c, taskInstance);
    }
  }

  value CsExecGenerator(VM *c, value genInstance) {
    assert(CsGeneratorP(c, genInstance));

    // assert( c->exec == nullptr ); // this shall run on pristine machine

    CsSavedState state(c);

    bool finsihed = false;

    PROTECT(genInstance);

    {
      CsTask *exec = CsCoroutineValue(c, genInstance);
      if (!exec)
        return NOTHING_VALUE; // destroyed already

      auto_state<Exec *> _1(c->exec, static_cast<Exec *>(exec));

      if (exec->states) exec->states->restore();

      if (exec->current) {
        assert(exec->states == exec->current);
        delete exec->current;
        exec->current = nullptr;
      }

      finsihed = false;

      if (exec->argc >= 0) {
        c->argc = exec->argc;
        c->argv = &c->sp[exec->argc];
        exec->argc = -1;
      }

      TRY {
        /* execute the function code */
        finsihed = c->exec->run(c, false);
        state.restore();
        if (finsihed) {
          c->active_tasks.remove_by_value(exec->taskInstance);
          exec->taskInstance = 0;
          CoroutineDestroy(c, genInstance);
          // return NOTHING_VALUE;
        }
        return c->val;
      }
      CATCH_ERROR(e) {
        state.restore();
        c->active_tasks.remove_by_value(exec->taskInstance);
        exec->taskInstance = 0;
        CoroutineDestroy(c, genInstance);
        RETHROW(e);
      }
    }
  }

  /* UnaryOp - handle unary opcodes */
  void UnaryOp(VM *c, int op) {
    value p1 = value_to_set(c->val);

    if (CsIntegerP(p1)) {
      int_t i1 = CsIntegerValue(p1);
      int_t ival;
      switch (op) {
      case '+': ival = i1; break;
      case '-': ival = -i1; break;
      case '~': ival = ~i1; break;
      case 'I': ival = i1 + 1; break;
      case 'D': ival = i1 - 1; break;
      default:
        ival = 0; /* never reached */
        break;
      }
      c->val = CsMakeInteger(ival);
    } else if (CsFloatP(p1)) {
      float_t f1 = CsFloatValue(p1);
      float_t fval;
      switch (op) {
      case '+': fval = f1; break;
      case '-': fval = -f1; break;
      case 'I': fval = f1 + 1; break;
      case 'D': fval = f1 - 1; break;
      case '~':
        CsTypeError(c, p1);
        /* fall through */
      default:
        fval = 0.0; /* never reached */
        break;
      }
      c->val = CsMakeFloat(fval);
    } else if (CsLengthP(p1)) {
      PRIMITIVE_TYPE pt   = primitive_type(p1);
      LENGTH_UNIT_TYPE u = unpack_unit<LENGTH_UNIT_TYPE>(p1);
      int_t          i1   = unpack_int32(p1);
      int_t          ival = 0;
      switch (op) {
      case '+': ival = i1; break;
      case '-': ival = -i1; break;
      case '~':
        // ival = ~i1;
        CsUnexpectedTypeError(c, p1, "integer");
        break;
      case 'I': ival = i1 + 1000; break;
      case 'D': ival = i1 - 1000; break;
      default:
        ival = 0; /* never reached */
        break;
      }
      c->val = pack_value(pt, u, ival);
    } else if (CsAngleP(p1)) {
      PRIMITIVE_TYPE pt   = primitive_type(p1);
      ANGLE_UNIT_TYPE u = unpack_unit<ANGLE_UNIT_TYPE>(p1);
      int_t          i1   = unpack_int32(p1);
      int_t          ival = 0;
      switch (op) {
      case '+': ival = i1; break;
      case '-': ival = -i1; break;
      case '~': CsUnexpectedTypeError(c, p1, "integer"); break;
      case 'I': ival = i1 + 10000; break;
      case 'D': ival = i1 - 10000; break;
      default:
        ival = 0; /* never reached */
        break;
      }
      c->val = pack_value(pt, u, ival);
    }
    else if (CsDurationP(p1)) {
      PRIMITIVE_TYPE pt = primitive_type(p1);
      DURATION_UNIT_TYPE u = unpack_unit<DURATION_UNIT_TYPE>(p1);
      int_t          i1 = unpack_int32(p1);
      int_t          ival = 0;
      switch (op) {
      case '+': ival = i1; break;
      case '-': ival = -i1; break;
      case '~': CsUnexpectedTypeError(c, p1, "integer"); break;
      case 'I': ival = i1 + 10000; break;
      case 'D': ival = i1 - 10000; break;
      default:
        ival = 0; /* never reached */
        break;
      }
      c->val = pack_value(pt, u, ival);
    }

    else
      CsUnexpectedTypeError(c, p1, "number");
  }

  value CsAngleBinaryOp(VM *c, int op, value p1, value p2);
  value CsDurationBinaryOp(VM *c, int op, value p1, value p2);
  value CsLengthBinaryOp(VM *c, int op, value p1, value p2);

  // either p1 or p2 are units
  value CsBinaryUnitsOp(VM *c, int op, value p1, value p2) {

    if (CsUnitsP(p1)) {
    COMPUTE:
      if (CsLengthP(p1))
        return CsLengthBinaryOp(c, op, p1, p2);
      else if (CsAngleP(p1))
        return CsAngleBinaryOp(c, op, p1, p2);
      else if (CsDurationP(p1))
        return CsDurationBinaryOp(c, op, p1, p2);
      CsUnexpectedTypeError(c, p1, "length");
    } else {
      if (op == BC_SUB) {
        PRIMITIVE_TYPE pt = primitive_type(p2);
        uint           u2 = unpack_unit<uint>(p2);
        int_t          v2 = unpack_int32(p2);
        v2                = -v2;
        p2                = pack_value(pt, u2, v2);
        op                = BC_ADD;
      }
      swap(p1, p2);
      goto COMPUTE;
    }
    return UNDEFINED_VALUE;
  }

  /* BinaryOp - handle binary opcodes */
  value CsBinaryOp(VM *c, int op, value p1, value p2) {
    if (CsIntegerP(p1) && CsIntegerP(p2)) {
      int_t i1 = CsIntegerValue(p1);
      int_t i2 = CsIntegerValue(p2);
      int_t ival;
      switch (op) {
      case BC_ADD: ival = i1 + i2; break;
      case BC_SUB: ival = i1 - i2; break;
      case BC_MUL: ival = i1 * i2; break;
      case BC_DIV:
        if (i2 == 0) CsThrowKnownError(c, CsErrValueError, p2);
        ival = i1 / i2;
        break;
      case BC_REM:
        if (i2 == 0) CsThrowKnownError(c, CsErrValueError, p2);
        ival = i1 % i2;
        break;
      case BC_BAND: ival = i1 & i2; break;
      case BC_BOR: ival = i1 | i2; break;
      case BC_XOR: ival = i1 ^ i2; break;
      case BC_SHL: ival = i1 << i2; break;
      case BC_SHR: ival = i1 >> i2; break;
      case BC_USHL: ival = uint(i1) << i2; break;
      case BC_USHR: ival = uint(i1) >> i2; break;
      default:
        ival = 0; /* never reached */
        assert(false);
        break;
      }
      return CsMakeInteger(ival);
    } else if (CsUnitsP(p1) || CsUnitsP(p2)) {
      return CsBinaryUnitsOp(c, op, p1, p2);
    } else {
      float_t f1, f2, fval;
      if (CsFloatP(p1))
        f1 = CsFloatValue(p1);
      else if (CsIntegerP(p1))
        f1 = (float_t)CsIntegerValue(p1);
      else {
        dispatch *pd1 = CsGetDispatch(p1);
        if (pd1->binOp) return pd1->binOp(c, op, p1, p2);
        // CsTypeError(c,p1);
        CsUnexpectedTypeError(c, p1, "float,integer");
        f1 = 0.0; /* never reached */
      }
      if (CsFloatP(p2))
        f2 = CsFloatValue(p2);
      else if (CsIntegerP(p2))
        f2 = (float_t)CsIntegerValue(p2);
      else {
        // CsTypeError(c,p2);
        CsUnexpectedTypeError(c, p1, "float,integer");
        f2 = 0.0; /* never reached */
      }
      switch (op) {
      case BC_ADD: fval = f1 + f2; break;
      case BC_SUB: fval = f1 - f2; break;
      case BC_MUL: fval = f1 * f2; break;
      case BC_DIV: fval = f1 / f2; break;
      default:
        CsTypeError(c, p1);
        fval = 0.0; /* never reached */
        break;
      }
      //if (isnan(fval) || isinf(fval))
      //  return NAN_VALUE;
      return CsMakeFloat(fval);
    }
  }

  /* CsInternalSend - internal function to send a message */
  value CsInternalSend(VM *c, int argc) {
    /* setup the unwind target */

    CsSavedState _(c);

    // TRY {

    /* setup the call */
    if (Send(c, &CsTopCDispatch, argc)) { return c->val; }

    /* execute the function code */
    c->exec->run(c);

    //}
    // CATCH_ERROR(e)
    //{
    //  RETHROW(e);
    //}
    return c->val;
  }

  /* Send - setup to call a method */
  int Send(VM *c, FrameDispatch *d, int argc) {
    value _this = c->sp[argc];

    //c->vals = 1;

    value next     = c->sp[argc - 2];
    value selector = c->sp[argc - 1];
    // value method = UNDEFINED_VALUE;

    if (_this == UNDEFINED_VALUE || _this == NOTHING_VALUE ||
        _this == NULL_VALUE)
      CsThrowKnownError(c, CsErrNoMethod, "", _this, selector);

    //#ifdef _DEBUG
    //    dispatch* pdd = CsGetDispatch(next);
    //    dispatch* pdt = CsGetDispatch(_this);
    //    dispatch* pds = CsGetDispatch(selector);
    //#endif

    /* setup the 'this' parameter */
    c->sp[argc - 1] = c->sp[argc];

    /* set the method */
    c->sp[argc] = selector;

    if (CsAnyMethodP(selector) || CsPropertyMethodP(selector)) {
      ; // nothing to lookup, it is a method already
    } else {
      /* find the method */
      if (!CsGetProperty1(c, next, selector, &c->sp[argc])) {
        // try to call direct expando methods:
        dispatch *pd = CsGetDispatch(_this);
        if (pd->handleCall) {
          c->argv = &c->sp[argc];
          c->argc = argc;
          if (pd->handleCall(c, _this, selector, argc, &c->val)) {
            CsDrop(c, argc + 1);
            return true;
          }
        }
        // no luck
        CsThrowKnownError(c, CsErrNoMethod, CsTypeName(_this), _this, selector);
      }
    }

    if (!CsObjectOrMethodP(next) ||
        (c->sp[argc - 2] = CsObjectClass(next)) == 0)
      c->sp[argc - 2] = UNDEFINED_VALUE;

    /* call the method */
    return Call(c, d, argc);
  }

  void check_thrown_error(VM *c) {
    if (c->nativeThrowValue.length()) {
      tool::ustring er;
      tool::swap(c->nativeThrowValue, er);
      CsThrowKnownError(c, CsErrGenericErrorW, er.c_str());
    }
  }

  void CsObjectSetItemNoLoad(VM *c, value obj, value tag, value val);

  static void push_reversed(array<value>& arr, slice<value> elements) {
    arr.reserve(arr.length() + elements.length);
    for (int n = elements.size() - 1; n >= 0; --n)
      arr.push(elements[n]);
  }

  static int UnspreadArgs(VM* c, int argc) {
    //value method = c->sp[argc];

#pragma TODO("UnspreadArgs")

    GC_vector expanded(c);
    for (int i = 0; i < argc - 2; ++i)
    {
      value el = CsPop(c);
      if (CsSpreadP(el)) {
        value other = CsFixedVectorElement(el, 0);
        if (CsVectorP(other))
          push_reversed(expanded.elements,CsVectorElements(c, other));
        else if (CsTupleP(other))
          push_reversed(expanded.elements,CsTupleElements(other));
        else if (CsGeneratorP(c, other))
        {
          PROTECT(other);
          int n = 0;
          int idx = expanded.elements.size();
          for (; n < 10000; ++n) {
            value next = CsExecGenerator(c, other);
            if (next == NOTHING_VALUE)
              break;
            expanded.elements.insert(idx,next);
          }
          if (n == 10000)
            CsThrowKnownError(c, CsErrGenericError, "too many elements");
        }
        else
          expanded.elements.push(other);
      }
      else
        expanded.elements.push(el);
    }

    for (int n = expanded.elements.size() - 1; n >= 0; --n)
      CsCPush(c, expanded.elements[n]);

    return expanded.elements.size() + 2;
  }

  /* Call - setup to call a function */
  static bool Call(VM *c, FrameDispatch *d, int argc)
  {
    STATIC_ASSERT(sizeof(CallFrame) % sizeof(value) == 0);

    int        rflag, rargc, oargc, targc, n;
    value      method = c->sp[argc];
    byte *     cbase, *pc;
    int        old_argc = c->argc;
    value *    old_argv = c->argv;
    CallFrame *frame;
    value      code = UNDEFINED_VALUE;


    /* setup the argument list */
    c->argv = &c->sp[argc];
    c->argc = argc;

    // PROTECT(code,method);

    /* handle built-in methods */
    if (CsCMethodP(method)) {
      c->val = CsCMethodPtr(method)->call(c, CsGetArg(c, 1));
      CsDrop(c, argc + 1);
      c->argv = old_argv;
      c->argc = old_argc;
      check_thrown_error(c);
      return true;
    } else if (CsCFunctorP(method)) {
      c->val = CsCFunctorCall(c, method);
      CsDrop(c, argc + 1);
      c->argv = old_argv;
      c->argc = old_argc;
      check_thrown_error(c);
      return true;
    }

    else if (CsClassP(method)) {
      // Special case, "Literal Ctor":
      // var obj = ClassName { x:1, y:2 };
      // obj instanceof ClassName -> true
      // and ClassName.this() is called against that object.

      value cls  = method;
      value tcls = cls;

      // PROTECT(tcls, cls);

      if (!CsGetObjectProperty(c, tcls, THIS_SYM, &method))
        method = UNDEFINED_VALUE;
      c->val = CsNewInstance(c, cls);
      if (CsArgCnt(c) == 3 && CsObjectP(c->val) &&
          CsObjectP(CsGetArg(c, 3)) &&
          CsObjectClass(CsGetArg(c, 3)) == c->objectObject) {
        // dispatch* pd = CsGetDispatch(c->val);
        // CsSetObjectProperties(c->val, CsObjectProperties(CsGetArg(c,3)));
        // CsSetObjectProperties(CsGetArg(c,3),UNDEFINED_VALUE);

        // mixin properties from the object passed as literal object:
        each_property gen(c, CsGetArg(c, 3));
        for (value key, val; gen(key, val);)
          CsObjectSetItemNoLoad(c, c->val, key, val);

        // pd = CsGetDispatch(c->val);
        CsDrop(c, argc + 1);
        c->argv = old_argv;
        c->argc = old_argc;

        // call ctor if any
        CsPush(c, c->val);
        if (method != UNDEFINED_VALUE) CsSendMessage(c, c->val, method, 0);
        // pd = CsGetDispatch(c->val);
        c->val = CsPop(c);
        // pd = CsGetDispatch(c->val);
        return true;
      }

    } else if (CsNamespaceP(method)) {
      value obj = method;
      if (!CsGetProperty1(c, obj, THIS_SYM, &method))
        CsUnexpectedTypeError(c, method, "this namespace is not callable");
    }
    else if (CsVNodeFunctionP(method)) {
      // SSX construct [func:atts,kids,states]
      method = CsCallFunction(c->currentScope(), CsVNodeTag(method),3,
                                                 CsVNodeAtts(method),
                                                 CsVNodeKids(method),
                                                 CsVNodeStates(method));
    }

    /* otherwise, it had better be a bytecode method */
    if (!CsMethodP(method) && !CsPropertyMethodP(method))
      CsUnexpectedTypeError(c, method, "function");

    GC_vector rarguments(c); // this vector is used as temp storage for args..
                             // to be converted to Array (Vector)
    value *prarguments = 0;  // after setup of call frame

    /* get the code obj */
    value newcode = CsMethodCode(method);
    assert(newcode);
    if (!newcode) return false;
    if (!CsCompiledCodeP(newcode)) return false;

    auto setup_function_call = [&]() {

      code = newcode;

      cbase = pc = CsByteVectorAddress(CsCompiledCodeBytecodes(code));

      /* parse the argument frame instruction */

#ifdef TISCRIPT_DEBUGGER
      if (*pc == BC_LINENO) { pc += 3; }
#endif
      rflag = *pc++ == BC_AFRAMER;

      rargc = *pc++;
      oargc = *pc++;
      targc = rargc + oargc;

      /* check the argument count */
      if (argc < rargc) CsTooFewArguments(c);
      // else if (!rflag && argc > rargc + oargc)
      //    CsTooManyArguments(c);
      else if (!rflag && argc > rargc + oargc) {
        // drop rest of arguments.
        int delta = c->argc - (rargc + oargc);
        CsDrop(c, delta);
        c->argc -= delta;
      }

      /* fill out the optional arguments */
      if ((n = targc - argc) > 0) {
        CsCheck(c, n);
        while (--n >= 0)
          CsPush(c, UNDEFINED_VALUE);
      }

      /* build the rest argument */
      if (rflag) {
        // value val,*p;
        int rcnt;
        if ((rcnt = argc - targc) < 0) rcnt = 0;

        while (--rcnt >= 0)
          rarguments.elements.push(CsPop(c));
        CsPush(c, NULL_VALUE);
        prarguments = c->sp;

        ++targc;
      }

      /* reserve space for the call frame */
      CsCheck(c, WordSize(sizeof(CallFrame)) + CsFirstEnvElement);

      /* complete the environment frame */
      CsPush(c, CsCompiledCodeArgNames(code)); /* names */
      CsPush(c, CsMethodEnv(method));          /* nextFrame */

      /* initialize the frame */
      frame = (CallFrame *)(c->sp - WordSize(sizeof(CallFrame)));

      frame->pdispatch = d;
      frame->next.set(c, c->fp);
      //frame->globals  = c->currentScope()->globals;
      frame->ns       = c->getCurrentNS();
      frame->env      = c->env;
      frame->code     = c->code;
      frame->pcOffset = int(c->pc - c->cbase);
      frame->argc     = old_argc;
      frame->method   = method;

      CsSetDispatch(ptr_value(&frame->stackEnv), &CsStackEnvironmentDispatch);
      CsSetEnvSize(ptr_value(&frame->stackEnv), CsFirstEnvElement + targc);

      /* establish the new frame */
      c->env = ptr_value(&frame->stackEnv);
      c->fp  = (CsFrame *)frame;
      c->sp  = (value *)frame;

      /* setup the new method */
      //??? c->currentScope()->globals = CsMethodGlobals(method);
      c->currentNS               = CsMethodNamespace(method);
      c->code                    = code;
      c->cbase                   = cbase;
      c->pc                      = pc;

      // c->vals = 1; - cannot do that due to nested functions.

#if defined(TISCRIPT_DEBUGGER)
      if (c->pdebug.is_defined()) {
        int firstLine = CsGetLineNumber(c, code, int(pc - cbase));
        int lastLine  = CsGetLineNumber(
            c, code,
            int(pc - cbase) +
                (int)CsByteVectorSize(CsCompiledCodeBytecodes(code)) - 2);
        c->pdebug->enter_function(c, code, firstLine, lastLine);
      }
#endif

      if (prarguments) {
        value  arguments = CsMakeVector(c, rarguments.elements.size());
        value *p = CsVectorAddressI(arguments) + rarguments.elements.size();
        for (int i = 0; i < rarguments.elements.size(); ++i)
          *--p = rarguments.elements[i];
        *prarguments = arguments;
      }
    };

    if (CsCompiledCodeIsCoroutine(newcode)) {
      CsSavedState pre_task_rgs(c);
      value        taskInstance;

      {
        bool is_async_task = CsCompiledCodeIsTask(newcode);
        if (is_async_task && c->active_tasks.size() >= 32)
          CsThrowKnownError(c, CsErrGenericError, "too many running async tasks");

        PROTECT(method, newcode, code); // new CsTask below allocates objects in heap

        CsTask *           task = new CsTask(c, is_async_task);
        auto_state<Exec *> _1(c->exec, task);

        d = &CsTopCDispatch; // cause - Execute()'d

        setup_function_call();

        task->method = method;

        // task->initial.store(c);
        task->current = new CsSavedState(c);

        taskInstance = task->taskInstance = CsMakeCoroutine(c, task);

        if (is_async_task)
          c->active_tasks.push(taskInstance);
      }

      pre_task_rgs.restore();

      // return task instance (that is a promise)
      c->val = taskInstance;

      CsDrop(c, argc + 1);
      c->argv = old_argv;
      c->argc = old_argc;

      if (CsCompiledCodeIsTask(newcode)) {
        c->pending_tasks.push(taskInstance);
        c->post([c]() { return c->deliver_notifications(); });
      } else {
        // CsExecGenerator(c,taskInstance);
      }

      return true;

    } else {

      setup_function_call();
      /* didn't complete the call */
      return false;
    }
  }

  /* TopRestore - restore a top continuation */
  static bool TopRestore(VM *c) {
    CallRestore(c);
    return true;
  }

  /* CallRestore - restore a call continuation */
  static bool CallRestore(VM *c) {
    CallFrame *frame = (CallFrame *)c->fp;
    value      env   = c->env;

    /* restore the previous frame */
    c->fp                      = frame->next.get(c);
    // c->currentScope()->globals = frame->globals;
    c->currentNS               = frame->ns;
    c->env                     = frame->env;
    if ((c->code = frame->code) != 0) {
      c->cbase = CsByteVectorAddress(CsCompiledCodeBytecodes(c->code));
      c->pc    = c->cbase + frame->pcOffset;
    }
    c->argc = frame->argc;

    /* fixup moved environments */
    if (c->env && CsMovedEnvironmentP(c->env))
      c->env = CsMovedEnvForwardingAddr(c->env);

    /* reset the stack pointer */
    c->sp = (value *)frame;
    CsDrop(c, WordSize(sizeof(CallFrame)) + CsEnvSize(env) + 1);

#if defined(TISCRIPT_DEBUGGER)
    if (c->pdebug && c->code) {
      int     callerLineNo = CsGetLineNumber(c, c->code, int(c->pc - c->cbase));
      ustring fn           = CsSymbolName(CsCompiledCodeFileName(c->code));
      c->pdebug->leave_function(c, c->code, callerLineNo);
    }
#endif

    return false;
  }

  /* CallCopy - copy a call continuation during garbage collection */
  static value *CallCopy(VM *c, CsFrame *frame) {
    CallFrame *call = (CallFrame *)frame;
    call->ns        = CsCopyValue(c, call->ns);
    //call->globals   = CsCopyValue(c, call->globals);
    call->method    = CsCopyValue(c, call->method);
    if (call->code) call->code = CsCopyValue(c, call->code);

    value  env   = ptr_value(&call->stackEnv);
    value *data  = CsEnvAddress(env);
    int_t  count = CsEnvSize(env);
    call->env    = CsCopyValue(c, call->env);

    if (CsStackEnvironmentP(env)) {
      while (--count >= 0) {
        *data = CsCopyValue(c, *data);
        ++data;
      }
    } else
      data += count;

    return data;
  }

  /* PushFrame - push a frame on the stack */
  static void PushFrame(VM *c, int size, value names) {
    BlockFrame *frame;

    /* reserve space for the frame */
    CsCheck(c, WordSize(sizeof(BlockFrame)) + CsFirstEnvElement);

    /* complete the environment frame */
    CsPush(c, names);  /* names */
    CsPush(c, c->env); /* nextFrame */

    /* initialize the frame */
    frame            = (BlockFrame *)(c->sp - WordSize(sizeof(BlockFrame)));
    frame->pdispatch = &CsBlockCDispatch;
    frame->next.set(c, c->fp);
    frame->env = c->env;
    CsSetDispatch(ptr_value(&frame->stackEnv), &CsStackEnvironmentDispatch);
    CsSetEnvSize(ptr_value(&frame->stackEnv), CsFirstEnvElement + size);

    /* establish the new frame */
    c->env = ptr_value(&frame->stackEnv);
    // dispatch* pd = CsGetDispatch(c->env);
    c->fp = (CsFrame *)frame;
    c->sp = (value *)frame;
  }

  /* BlockRestore - restore a frame continuation */
  static bool BlockRestore(VM *c) {
    BlockFrame *frame = (BlockFrame *)c->fp;

    /* restore the previous frame */
    c->fp  = frame->next.get(c);
    c->env = frame->env;

    /* fixup moved environments */
    if (c->env && CsMovedEnvironmentP(c->env))
      c->env = CsMovedEnvForwardingAddr(c->env);

    // dispatch* pd = CsGetDispatch(c->env);

    /* reset the stack pointer */
    c->sp = (value *)frame;
    CsDrop(c, WordSize(sizeof(BlockFrame)) +
                  CsEnvSize(ptr_value(&frame->stackEnv)));
    return false;
  }

  /* BlockCopy - copy a frame continuation during garbage collection */
  static value *BlockCopy(VM *c, CsFrame *frame) {
    BlockFrame *block = (BlockFrame *)frame;
    value       env   = ptr_value(&block->stackEnv);
    value *     data  = CsEnvAddress(env);
    int_t       count = CsEnvSize(env);
    block->env        = CsCopyValue(c, block->env);
    if (CsStackEnvironmentP(env)) {
      while (--count >= 0) {
        *data = CsCopyValue(c, *data);
        ++data;
      }
    } else
      data += count;
    return data;
  }

  /* UnstackEnv - unstack the environment */
  static value UnstackEnv(VM *c, value env) {
    value newo, *src, *dst;
    long  size;

    /* initialize */
    CsCheck(c, 3);
    CsPush(c, UNDEFINED_VALUE);
    CsPush(c, UNDEFINED_VALUE);

    /* copy each stack environment frame to the heap */
    while (CsStackEnvironmentP(env)) {

      /* allocate a new frame */
      CsPush(c, env);
      size = CsEnvSize(env);
      newo = CsMakeEnvironment(c, size);
      env  = CsPop(c);

      /* copy the data */
      src = CsEnvAddress(env);
      dst = CsEnvAddress(newo);
      while (--size >= 0)
        *dst++ = *src++;

      /* link the newo frame into the newo environment */
      if (CsTop(c) == UNDEFINED_VALUE)
        c->sp[1] = newo;
      else
        CsSetEnvNextFrame(CsTop(c), newo);
      CsSetTop(c, newo);

      /* get next frame */
      CsPush(c, CsEnvNextFrame(env));

      /* store the forwarding address */
      CsSetDispatch(env, &CsMovedEnvironmentDispatch);
      CsSetMovedEnvForwardingAddr(env, newo);

      /* move ahead to the next frame */
      env = CsPop(c);
    }

    /* link the first heap frame into the newo environment */
    if (CsTop(c) == UNDEFINED_VALUE)
      c->sp[1] = env;
    else
      CsSetEnvNextFrame(CsTop(c), env);
    CsDrop(c, 1);

    /* return the newo environment */
    return CsPop(c);
  }

  /* CsTypeError - signal a 'type' error */
  void CsTypeError(VM *c, value v) { CsThrowKnownError(c, CsErrTypeError, v); }

  void CsUnexpectedTypeError(VM *c, value v, const char *expectedTypes) {
    CsThrowKnownError(c, CsErrUnexpectedTypeError, v, expectedTypes);
  }

  /* CsStackOverflow - signal a 'stack overflow' error */
  void CsStackOverflow(VM *c) { CsThrowKnownError(c, CsErrStackOverflow, 0); }

  /* CsTooManyArguments - signal a 'too many arguments' error */
  void CsTooManyArguments(VM *c) {
    CsThrowKnownError(c, CsErrTooManyArguments, c->sp[c->argc]);
  }

  /* CsAlreadyDefined - signal a 'already defined' error */
  void CsAlreadyDefined(VM *c, value tag) {
    CsThrowKnownError(c, CsErrAlreadyDefined,
                      string(CsSymbolName(tag)).c_str());
  }

  /* CsTooFewArguments - signal a 'too few arguments' error */
  void CsTooFewArguments(VM *c) {
    CsThrowKnownError(c, CsErrTooFewArguments, c->sp[c->argc]);
  }

  /* CsWrongNumberOfArguments - signal a 'bad number of arguments' error */
  void CsWrongNumberOfArguments(VM *c) {
    CsThrowKnownError(c, CsErrArguments, c->sp[c->argc]);
  }

  void CsUnsupportedBinaryOp(VM *c, int op, value p1, value p2) {
    const char *s = 0;

    switch (op) {
    case BC_ADD: s = "+"; break;
    case BC_SUB: s = "-"; break;
    case BC_MUL: s = "*"; break;
    case BC_DIV: s = "/"; break;
    case BC_REM: s = "%"; break;
    case BC_BAND: s = "&"; break;
    case BC_BOR: s = "|"; break;
    case BC_XOR: s = "^"; break;
    case BC_BNOT: s = "~"; break;
    case BC_SHL: s = "<<"; break;
    case BC_SHR: s = ">>"; break;
    case BC_LT: s = "<"; break;
    case BC_LE: s = "<="; break;
    case BC_EQ: s = "=="; break;
    case BC_NE: s = "!="; break;
    case BC_GE: s = ">="; break;
    case BC_GT: s = ">"; break;
    default:
      s = "?";
      assert(false);
      break;
    }

    CsThrowKnownError(c, CsErrUnsupportedBinaryOp, s, p1, p2);
  }

  void CsBadValue(VM *c, value v) { CsThrowKnownError(c, CsErrValueError, v); }

  /* BadOpcode - signal a 'bad opcode' error */
  static void BadOpcode(VM *c, int opcode) {
    CsThrowKnownError(c, CsErrBadOpcode, opcode);
  }

  void CsThrowExit(VM *c, value v) {
    c->val = v;
    THROW_ERROR(0);
  }

  /* CsEql - compare two objects for equality, used for key comparison in
   * objects */
  bool CsKeysAreEqual(VM *c, value obj1, value obj2) {
    if (obj1 == obj2) return true;
    /*if (CsIntegerP(obj1)) {
        if (CsIntegerP(obj2))
            return CsIntegerValue(obj1) == CsIntegerValue(obj2);
        else if (CsFloatP(obj2))
            return (float_t)CsIntegerValue(obj1) == CsFloatValue(obj2);
        else
            return false;
    }
    else*/
    if (CsFloatP(obj1)) {
      if (CsFloatP(obj2))
        return CsFloatValue(obj1) == CsFloatValue(obj2);
      else if (CsIntegerP(obj2))
        return CsFloatValue(obj1) == (float_t)CsIntegerValue(obj2);
      else
        return false;
    } else if (CsStringP(obj1)) {
      if (CsSymbolP(obj2)) {
        ustring str = CsSymbolName(obj2);
        return str() == CsStringChars(obj1);
      } else if (CsStringP(obj2))
        return CompareStrings(obj1, obj2) == 0;
    } else if (CsSymbolP(obj1)) {
      if (CsStringP(obj2)) {
        ustring str = CsSymbolName(obj1);
        return str() == CsStringChars(obj2);
      }
      // else if( CsSymbolP(obj2) )
      //  return obj1 == obj2;
    }
    return false;
  }

  bool CsStrongEql(value obj1, value obj2) {
    if (obj1 == obj2)
      return true;
    else if (CsStringP(obj1) && CsStringP(obj2) &&
             CompareStrings(obj1, obj2) == 0)
      return true;
    return false;
  }

  // From JS spec:
  // ToBoolean
  //     The operator ToBoolean converts its argument to a value of type Boolean
  //     according to the following table:
  //  Undefined - false
  //  Null      - false
  //  Boolean   - The result equals the input argument (no conversion).
  //  Number    - The result is false if the argument is +0, -0, or NaN;
  //  otherwise the result is true. String    - The result is false if the
  //  argument is the empty string (its length is zero); otherwise
  //              the result is true.
  //  Object    - true

  value CsToBoolean(VM *c, value v) {
    // START:
    if (CsSymbolP(v)) {
      if (v == FALSE_VALUE) return v;
      if (v == UNDEFINED_VALUE || v == NULL_VALUE || v == NOTHING_VALUE)
        return FALSE_VALUE;
      return TRUE_VALUE;
    }
    if (CsIntegerP(v)) return CsIntegerValue(v) != 0 ? TRUE_VALUE : FALSE_VALUE;
    if (CsFloatP(v)) return CsFloatValue(v) != 0.0 ? TRUE_VALUE : FALSE_VALUE;
    if (CsStringP(v)) return CsStringSize(v) != 0 ? TRUE_VALUE : FALSE_VALUE;

    // everything else is simply TRUE
    return TRUE_VALUE;

    /*
    if (CsVectorP(v))
      return TRUE_VALUE;
    if (CsByteVectorP(v))
      return TRUE_VALUE;

    if( CsObjectP(v) )
    {
      value r;
      value obj = v;
      if(CsGetProperty1(c, obj, VALUE_OF_SYM, &r) && (CsMethodP(r) ||
    CsCMethodP(r)))
      {
        v = CsSendMessage(CsCurrentScope(c),v,r, 0);
        if( !CsObjectP(v) )
          goto START;
      }
    }*/
  }

  int CsLengthsCompare(VM *c, int op, value obj1, value obj2);
  int CsAnglesCompare(VM *c, int op, value obj1, value obj2);
  int CsDurationsCompare(VM *c, int op, value obj1, value obj2);
  int CsDatesCompare(VM *c, int op, value obj1, value obj2);

  bool CsEqualOp(VM *c, value obj1, value obj2) {
    GC_vector stack(c);
    return CsEqualOp(c, value_to_set(obj1), value_to_set(obj2), stack.elements);
  }

  /* CsEql - compare two objects for equality */
  bool CsEqualOp(VM *c, value obj1, value obj2, array<value> &stack) {
  //// Too restrictive?
  //// if (obj1 == UNDEFINED_VALUE || obj2 == UNDEFINED_VALUE)
  ////  CsWarning(c,"comparison with undefined value!");

  // if (CsFloatP(obj1) || CsFloatP(obj2))
  //    return CsToFloat(c,obj1) == CsToFloat(c,obj2);

#ifdef _DEBUG
  // dispatch *pd1 = CsGetDispatch(obj1);
  // dispatch *pd2 = CsGetDispatch(obj2);
#endif

    /* ECMA spec for this:

    1. If Type(x) is different from Type(y), go to step 14.
    2. If Type(x) is Undefined, return true.
    3. If Type(x) is Null, return true.
    4. If Type(x) is not Number, go to step 11.
    5. If x is NaN, return false.
    6. If y is NaN, return false.
    7. If x is the same number value as y, return true.
    8. If x is +0 and y is ??í0, return true.
    9. If x is ??í0 and y is +0, return true.
    10. Return false.
    11.If Type(x) is String, then return true if x and y are exactly the same
    sequence of characters (same length and same characters in corresponding
    positions). Otherwise, return false.
    12. If Type(x) is Boolean, return true if x and y are both true or both
    false. Otherwise, return false. 13.Return true if x and y refer to the same
    object or if they refer to objects joined to each other (see 13.1.2).
    Otherwise, return false.
    14. If x is null and y is undefined, return true.
    15. If x is undefined and y is null, return true.
    - page 56 -
    16.If Type(x) is Number and Type(y) is String,
    return the result of the comparison x == ToNumber(y).
    17.If Type(x) is String and Type(y) is Number,
    return the result of the comparison ToNumber(x) == y.
    18. If Type(x) is Boolean, return the result of the comparison ToNumber(x)
    == y.
    19. If Type(y) is Boolean, return the result of the comparison x ==
    ToNumber(y). 20.If Type(x) is either String or Number and Type(y) is Object,
    return the result of the comparison x == ToPrimitive(y).
    21.If Type(x) is Object and Type(y) is either String or Number,
    return the result of the comparison ToPrimitive(x) == y.
    22. Return false.
    */

    // this implementation is close to the canonic text above but not exact.

    if (CsFloatP(obj1)) {
      CsPush(c, obj1);
      obj2 = CsToFloat(c, obj2, false);
      return CsPop(c) == obj2;
    } else if (CsFloatP(obj2)) {
      CsPush(c, obj2);
      obj1 = CsToFloat(c, obj1, false);
      return obj1 == CsPop(c);
    }

    if (CsIntegerP(obj1)) {
      CsPush(c, obj1);
      obj2 = CsToInteger(c, obj2, false);
      return CsPop(c) == obj2;
    } else if (CsIntegerP(obj2)) {
      CsPush(c, obj2);
      obj1 = CsToInteger(c, obj1, false);
      return obj1 == CsPop(c);
    }

    if (CsStringP(obj1)) {
      CsPush(c, obj1);
      obj2 = CsToString(c, obj2);
      return CompareStrings(CsPop(c), obj2) == 0;
    } else if (CsStringP(obj2)) {
      CsPush(c, obj2);
      obj1 = CsToString(c, obj1);
      return CompareStrings(obj1, CsPop(c)) == 0;
    }

    if (CsBooleanP(obj1)) {
      CsPush(c, obj1);
      obj2 = CsSymbolP(obj2) ? obj2 : CsToBoolean(c, obj2);
      return CsPop(c) == obj2;
    } else if (CsBooleanP(obj2)) {
      CsPush(c, obj2);
      obj1 = CsSymbolP(obj1) ? obj1 : CsToBoolean(c, obj1);
      return obj1 == CsPop(c);
    }

    if (obj1 == UNDEFINED_VALUE || obj1 == NULL_VALUE) {
      if (obj2 == UNDEFINED_VALUE || obj2 == NULL_VALUE) return true;
      return false;
    }
    if (obj2 == UNDEFINED_VALUE || obj2 == NULL_VALUE) { return false; }

    obj1 = value_to_set(obj1);
    obj2 = value_to_set(obj2);

    if (CsVectorP(obj1) && CsVectorP(obj2))
      return CsVectorsEqual(c, obj1, obj2, stack);
    else if (CsObjectP(obj1) && CsObjectP(obj2))
      return CsObjectsEqual(c, obj1, obj2, stack);
    // else if (CsMethodP(obj1) && CsMethodP(obj2))
    //    return obj1 == obj2;
    else if (CsTupleP(obj1) && CsTupleP(obj2))
      return CsTuplesEqual(c,obj1, obj2,stack);
    else if (CsLengthP(obj1) && CsLengthP(obj2))
      return CsLengthsCompare(c, BC_EQ, obj1, obj2) == 0;
    else if (CsDurationP(obj1) && CsDurationP(obj2))
      return CsDurationsCompare(c, BC_EQ, obj1, obj2) == 0;
    else if (CsDateP(c,obj1) && CsDateP(c, obj2))
      return CsDatesCompare(c, BC_EQ, obj1, obj2) == 0;
    else if (CsAngleP(obj1) && CsAngleP(obj2))
      return CsAnglesCompare(c, BC_EQ, obj1, obj2) == 0;
    else if (CsByteVectorP(obj1) && CsByteVectorP(obj2))
      return CsByteVectorBytes(obj1) == CsByteVectorBytes(obj2);
    else
      return obj1 == obj2;
  }

  static const float_t epsilon_minus = -0.00000000000000088;
  static const float_t epsilon_plus  = 0.00000000000000088;

  /* CsCompareObjects - compare two objects */
  int CsCompareObjects(VM *c, value obj1, value obj2, bool suppressError) {
    obj1 = value_to_set(obj1);
    obj2 = value_to_set(obj2);

    if (CsFloatP(obj1) || CsFloatP(obj2)) {
      float_t diff =
          to_float(CsToFloat(c, obj1)) - to_float(CsToFloat(c, obj2));
      return diff < epsilon_minus ? -1 : (diff > epsilon_plus ? 1 : 0);
      // return diff < 0 ? -1 : (diff > 0 ? 1 : 0);
    } else if (CsIntegerP(obj1) || CsIntegerP(obj2))
      return to_int(CsToInteger(c, obj1)) - to_int(CsToInteger(c, obj2));
    else if (CsStringP(obj1) && CsStringP(obj2))
      return CompareStrings(obj1, obj2);
    else if (CsSymbolP(obj1) && CsSymbolP(obj2))
      return str_cmp(CsSymbolName(obj1), CsSymbolName(obj2));
    else if (CsVectorP(obj1) && CsVectorP(obj2))
      return CsCompareVectors(c, obj1, obj2, suppressError);
    else if (CsLengthP(obj1) && CsLengthP(obj2))
      return CsLengthsCompare(c, BC_LE, obj1, obj2);
    else if (CsAngleP(obj1) && CsAngleP(obj2))
      return CsAnglesCompare(c, BC_LE, obj1, obj2);
    else if (CsDurationP(obj1) && CsDurationP(obj2))
      return CsDurationsCompare(c, BC_LE, obj1, obj2);
    else if (CsDateP(c,obj1) && CsDateP(c, obj2))
      return CsDatesCompare(c, BC_LE, obj1, obj2);
    else if (suppressError) {
      int_t diff = CsHashValue(obj1) - CsHashValue(obj2);
      return diff < 0 ? -1 : diff == 0 ? 0 : 1;
    } else {
      CsTypeError(c, obj1);
      return 0; /* never reached */
    }
  }

  /* CompareStrings - compare two strings */
  static int CompareStrings(value str1, value str2) {
    const wchar *p1   = CsStringAddress(str1);
    long         len1 = CsStringSize(str1);
    const wchar *p2   = CsStringAddress(str2);
    long         len2 = CsStringSize(str2);
    while (len1 > 0 && len2 > 0 && *p1++ == *p2++) {
      --len1;
      --len2;
    }
    if (len1 == 0) return len2 == 0 ? 0 : -1;
    if (len2 == 0) return 1;
    return (int(*--p1) - int(*--p2)) < 0 ? -1 : ((*p1 == *p2) ? 0 : 1);
  }

  bool CsMatchObject(VM *c, value tmpl, value obj) {
    assert(CsObjectP(tmpl));
    assert(CsObjectP(obj));
    each_property props(c, tmpl);
    for (value key, tval; props(key, tval);) {
      value rval = NOTHING_VALUE;
      if (!CsGetProperty(c, obj, key, &rval)) return false;
      if (!CsMatch(c, tval, rval)) return false;
    }
    return true;
  }

  bool CsMatchVector(VM *c, value tmpl, value obj) {
    assert(CsVectorP(tmpl));
    assert(CsVectorP(obj));
    int_t tsize = CsVectorSize(c, tmpl);
    int_t osize = CsVectorSize(c, obj);
    if (tsize > osize) return false;
    for (int_t n = 0; n < tsize; ++n)
      if (!CsMatch(c, CsVectorElement(c, tmpl, n), CsVectorElement(c, obj, n)))
        return false;
    return true;
  }

  /* match obj against the template */
  bool CsMatch(VM *c, value tmpl, value obj) {
    if (tmpl == ANY_SYM) return obj != NOTHING_VALUE;

    if (CsStringP(tmpl) || CsRegExpP(c, tmpl)) {
      if (!CsStringP(obj)) return 0;
      return CsIsLike(c, obj, tmpl);
    }

    if (CsClassP(tmpl)) return CsInstanceOf(c, obj, tmpl);
    if (CsTypeP(c, tmpl)) return CsIsBaseType(obj, (tis::dispatch *)tis::CsCObjectValue(tmpl));

    if (CsObjectP(tmpl)) {
      if (!CsObjectP(obj)) return false;
      return CsMatchObject(c, tmpl, obj);
    }

    if (CsVectorP(tmpl)) {
      if (!CsVectorP(obj)) return false;
      return CsMatchVector(c, tmpl, obj);
    }

    if (CsGetDispatch(tmpl) == CsGetDispatch(obj))
      return CsEqualOp(c, tmpl, obj); // case: alike compare

    return false;
  }

#if 0 // replaced by CsAppendString

/* ConcatenateStrings - concatenate two strings */
static value ConcatenateStrings(VM *c,value str1,value str2)
{
  //tool::ustring s = value_to_string(str1) + value_to_string(str2);
  //return CsMakeCharString(c,s,s.length());
    wchar *src,*dst;
    long len1 = CsStringSize(str1);
    long len2 = CsStringSize(str2);
    long len = len1 + len2;
    value newo;
    CsCheck(c,2);
    CsPush(c,str1);
    CsPush(c,str2);
    newo = CsMakeCharString(c,NULL,len);
    dst = CsStringAddress(newo);
    src = CsStringAddress(c->sp[1]);
    while (--len1 >= 0)
        *dst++ = *src++;
    src = CsStringAddress(CsTop(c));
    while (--len2 >= 0)
        *dst++ = *src++;
    CsDrop(c,2);
    return newo;
}
#endif

  /* CsCopyStack - copy the stack for the garbage collector */
  // void CsCopyStack(VM *c)
  void CsCopyStack(VM *c) {
    // copy the stack
    CsFrame *fp = c->fp;
    value *  sp = c->sp;
    while (sp < c->stackTop) {
      if (sp >= (value *)fp) {
        sp = (*fp->pdispatch->copy)(c, fp);
        fp = fp->next.get(c);
      } else {
        *sp = CsCopyValue(c, *sp);
        ++sp;
      }
    }
  }

  int CsGetLineNumber(VM *c, value ccode, int pc) {
    value lna = CsCompiledCodeLineNumbers(ccode);
    if (lna == UNDEFINED_VALUE) return 0;
    LineNumberEntry *plne = (LineNumberEntry *)CsByteVectorAddress(lna);
    int              n = int(CsByteVectorSize(lna) / sizeof(LineNumberEntry));

    if (n >= 2)
      for (int i = n - 2; i >= 0; --i)
        if (pc >= plne[i].pc && pc <= plne[i + 1].pc) return plne[i].line;
    return 0;

    /*
      int last_ln = -1;

      for( int i = n - 2; i >= 0 ; --i )
        if( pc >= plne[i].pc && pc <= plne[i+1].pc )
          return plne[i].line;
        else if( last_ln < 0 )
          last_ln = plne[i].line;
      return last_ln;
    */
  }

  void CsStreamStackTrace(VM *c, stream *s) {
    CsFrame *fp = c->fp;
    if (c->code) {
      value name = CsCompiledCodeName(c->code);

      int ln = CsGetLineNumber(
          c, c->code,
          int(c->pc - c->cbase - 1)); // -1 as pc already incremented
      if (ln) {
        value         fn       = CsCompiledCodeFileName(c->code);
        tool::ustring filename = tool::url::unescape(CsSymbolName(fn).c_str());
        s->printf(W("\tat %s (%s(%d))\n"), value_to_string(name).c_str(),
                  filename.c_str(), ln);
      } else if (name == UNDEFINED_VALUE && c->script_url.length())
        s->printf(W("\tat %s\n"), c->script_url.c_str());
      else if (CsSymbolP(name))
        s->printf(W("\tat %s\n"), CsSymbolName(name).c_str());
      else if (CsStringP(name))
        s->printf(W("\tat %s\n"), CsStringAddress(name));
      // s->put_str("'\n");
    }
    while (fp && fp < (CsFrame *)c->stackTop) {
      CallFrame *frame = (CallFrame *)fp;
      if (frame->pdispatch == &CsCallCDispatch && frame->code) {
        value name = CsCompiledCodeName(frame->code);
        value fn   = CsCompiledCodeFileName(frame->code);
        int   ln   = CsGetLineNumber(c, frame->code, frame->pcOffset);
        if (ln) {
          // CsDisplay();
          if (fn != name)
            s->printf(W("\tat %s (%s(%d))\n"), value_to_string(name).c_str(),
                      tool::url::unescape(CsSymbolName(fn).c_str()).c_str(),
                      ln);
          else
            s->printf(W("\tat (%s(%d))\n"),
                      tool::url::unescape(CsSymbolName(fn).c_str()).c_str(),
                      ln);
        } else {
          s->printf(W("\tat %s\n"), value_to_string(name).c_str());
        }
      }
      fp = fp->next.get(c);
    }
  }

  // make stack trace as a vector of triplets:
  //  [0] - lineNo
  //  [1] - functionName
  //  [2] - fileName
  value CsMakeStackTrace(VM *c) {
    pvalue vec(c, CsMakeVector(c, 256));
    pvalue triple(c, CsMakeVector(c, 3));

    int n = 0;

    // bool calledFromP = false;
    CsFrame *fp = c->fp;
    if (c->code) {
      value name      = CsCompiledCodeName(c->code);
      value file_name = UNDEFINED_VALUE;
      int   ln        = CsGetLineNumber(
          c, c->code,
          int(c->pc - c->cbase - 1)); // -1 as pc already incremented
      if (ln) file_name = CsCompiledCodeFileName(c->code);
      CsSetVectorElement(c, triple, 0, CsMakeInteger(ln));
      CsSetVectorElement(c, triple, 1, name);
      CsSetVectorElement(c, triple, 2, file_name);
      CsSetVectorElement(c, vec, n++, triple);
    }
    while (fp && fp < (CsFrame *)c->stackTop) {
      if (fp->pdispatch == &CsCallCDispatch &&
          static_cast<CallFrame *>(fp)->code) {
        if (n >= 256) break;

        CallFrame *frame = static_cast<CallFrame *>(fp);

        triple          = CsMakeVector(c, 3);
        value name      = CsCompiledCodeName(frame->code);
        value file_name = CsCompiledCodeFileName(frame->code);
        int   ln        = CsGetLineNumber(c, frame->code, frame->pcOffset);
        // if(!ln) //  file_name = UNDEFINED_VALUE;
        CsSetVectorElement(c, triple, 0, CsMakeInteger(ln));
        CsSetVectorElement(c, triple, 1, name);
        CsSetVectorElement(c, triple, 2, file_name);
        CsSetVectorElement(c, vec, n++, triple);
      }
      fp = fp->next.get(c);
    }
    return CsResizeVector(c, vec, n);
  }

  value CsCurrentCodeLocation(VM* c) {
    value triple = CsMakeTuple(c, 3);
    PROTECT(triple);
    if (c->code) {
      value name = CsCompiledCodeName(c->code);
      value file_name = UNDEFINED_VALUE;
      int ln = CsGetLineNumber(c, c->code,int(c->pc - c->cbase - 1)); // -1 as pc already incremented
      if (ln) file_name = CsCompiledCodeFileName(c->code);
      CsSetTupleElement(triple, 0, CsMakeInteger(ln));
      CsSetTupleElement(triple, 1, name);
      CsSetTupleElement(triple, 2, file_name);
    }
    return triple;
  }

  // (debugger) make stack trace as a vector of triplets:
  //  [0] - lineNo
  //  [1] - functionName
  //  [2] - fileName

  tool::value CsMakeStackTraceV(VM *c) {
    tool::array<tool::value> vec;

    int n = 0;

    // bool calledFromP = false;
    CsFrame *fp = c->fp;
    if (c->code) {
      value name      = CsCompiledCodeName(c->code);
      value file_name = UNDEFINED_VALUE;
      int   ln        = CsGetLineNumber(
          c, c->code,
          int(c->pc - c->cbase - 2)); // -1 as pc already incremented
      if (ln) file_name = CsCompiledCodeFileName(c->code);

      tool::value triplet[3];

      triplet[0] = tool::value(ln);
      triplet[1] = tool::value(value_to_string(name));
      triplet[2] = tool::value(value_to_string(file_name));
      vec.push(tool::value::make_array(tool::items_of(triplet)));
      ++n;
    }
    while (fp && fp < (CsFrame *)c->stackTop) {
      if (fp->pdispatch == &CsCallCDispatch &&
          static_cast<CallFrame *>(fp)->code) {
        if (n >= 256) break;

        CallFrame *frame = static_cast<CallFrame *>(fp);

        value name      = CsCompiledCodeName(frame->code);
        value file_name = CsCompiledCodeFileName(frame->code);
        int   ln        = CsGetLineNumber(c, frame->code, frame->pcOffset);

        tool::value triplet[3];

        triplet[0] = tool::value(ln);
        triplet[1] = tool::value(value_to_string(name));
        triplet[2] = tool::value(value_to_string(file_name));
        vec.push(tool::value::make_array(tool::items_of(triplet)));
        ++n;
      }

      fp = fp->next.get(c);
    }
    return tool::value::make_array(vec());
  }

#if 0
tool::value CsLocalVarsV(VM *c)
{
    tool::array<tool::value> vec;


    //bool calledFromP = false;
    CsFrame *fp = c->fp;
    value code = c->code;
    int ffstart = 0;

    string this_name = "this";

    while (fp && fp < (CsFrame *)c->stackTop)
    {
        if(vec.size() >= 256)
          break;
        if ((fp->pdispatch == &CsCallCDispatch || fp->pdispatch == &CsTopCDispatch) && code)
        {
          CallFrame *frame = static_cast<CallFrame*>(fp);
          //value name = CsCompiledCodeName(code);

          //tool::ustring nn = value_to_string(name);
          //value file_name = CsCompiledCodeFileName(frame->code);
          //int ln = CsGetLineNumber(c, frame->code, frame->pcOffset);

          //value names = CsCompiledCodeArgNames(code);
          //dispatch* pd = CsGetDispatch(names);

          value env = ptr_value(&frame->stackEnv);

          do {

            int n = CsEnvSize(env);
            value names = CsEnvNames(env);

            tool::value map = tool::value::make_map();

            //map.set_prop("!name",value_to_value(c,name));

            if(CsVectorP(names)) {
              tool::value _this = value_to_value_def(c,CsEnvElement(env,n - 1));
              map.set_prop(this_name,_this);
              this_name += " super";
              for( int i = 0; i < CsVectorSize(c,names); ++i ) {
                tool::string name = value_to_string( CsVectorElement(c,names,i));
                tool::value val = value_to_value_def(c,CsEnvElement(env,n - 3 - i));
                map.set_prop(name,val);
              }
            }
            else if(CsFixedVectorP(names)) {
              tool::value map = tool::value::make_map();
              int n = CsFixedVectorSize(names);
              for(int i = 0; i < n; ++i) {
                tool::string name = value_to_string(CsFixedVectorElement(names, i));
                tool::value val = value_to_value_def(c,CsEnvElement(env,CsFirstEnvElement + n - 1 - i));
                map.set_prop(name,val);
              }
            } else {
              tool::value _this = value_to_value_def(c,CsEnvElement(env,n - 1));
              map.set_prop(this_name,_this);
              this_name += " super";
            }

            vec.push(map);
            std::reverse(&vec[ffstart], vec.end());
            ffstart = vec.size();

            env = CsEnvNextFrame(env);

          } while(CsEnvironmentP(env) || CsMovedEnvironmentP(env));

          break;
          //code = frame->code;
        }
        else if (fp->pdispatch == &CsBlockCDispatch)
        {
          BlockFrame *block = static_cast<BlockFrame *>(fp);
          value env = ptr_value(&block->stackEnv);
          value names = CsEnvNames(env);
          if(CsFixedVectorP(names)) {
            tool::value map = tool::value::make_map();
            int n = CsFixedVectorSize(names);
            for(int i = 0; i < n; ++i) {
              tool::string name = value_to_string(CsFixedVectorElement(names, i));
              tool::value val = value_to_value_def(c,CsEnvElement(env,CsFirstEnvElement + n - 1 - i));
              map.set_prop(name,val);
            }
            vec.push(map);
          }
        }

        fp = fp->next.get(c);
    }


    return tool::value::make_array(vec());
}

#endif

  tool::value CsLocalVarsV(VM *c) {
    tool::array<tool::value> vec;

    // bool calledFromP = false;
    CsFrame *fp      = c->fp;
    value    code    = c->code;
    int      ffstart = 0;

    string this_name = "this";

    while (fp && fp < (CsFrame *)c->stackTop) {
      if (vec.size() >= 256) break;
      if ((fp->pdispatch == &CsCallCDispatch ||
           fp->pdispatch == &CsTopCDispatch) &&
          code) {
        CallFrame *frame = static_cast<CallFrame *>(fp);
        // value name = CsCompiledCodeName(code);
        // tool::ustring nn = value_to_string(name);
        // value file_name = CsCompiledCodeFileName(frame->code);
        // int ln = CsGetLineNumber(c, frame->code, frame->pcOffset);

        // value names = CsCompiledCodeArgNames(code);

        value env = ptr_value(&frame->stackEnv);

        do {

          int   envSize = CsEnvSize(env);
          value names   = CsEnvNames(env);

          // dispatch* pd = CsGetDispatch(names);

          tool::value map = tool::value::make_map();

          // map.set_prop("!name",value_to_value(c,name));

          if (CsTupleP(names)) {

            int n = CsTupleSize(names);

            static value args_marker = CsSymbolOf(W("args"));

            int loc_start = CsFirstEnvElement + n - 1;

            if (CsTupleName(names) == args_marker) {
              loc_start = envSize - 3;
              tool::value _this =
                  value_to_value_def(c, CsEnvElement(env, envSize - 1));
              map.set_prop(this_name, _this);
              this_name += " super";
            }

            for (int i = 0; i < n; ++i) {
              tool::string name = value_to_string(CsTupleElement(names, i));
              tool::value  val =
                  value_to_value_def(c, CsEnvElement(env, loc_start - i));
              map.set_prop(name, val);
            }
          } else {
            tool::value _this =
                value_to_value_def(c, CsEnvElement(env, envSize - 1));
            map.set_prop(this_name, _this);
            this_name += " super";
          }

          vec.push(map);
          std::reverse(&vec[ffstart], vec.end());
          ffstart = vec.size();

          env = CsEnvNextFrame(env);

        } while (CsEnvironmentP(env) || CsMovedEnvironmentP(env));

        break;
        // code = frame->code;
      } else if (fp->pdispatch == &CsBlockCDispatch) {
        BlockFrame *block = static_cast<BlockFrame *>(fp);
        value       env   = ptr_value(&block->stackEnv);
        value       names = CsEnvNames(env);
        if (CsTupleP(names)) {
          tool::value map = tool::value::make_map();
          int         n   = CsTupleSize(names);
          for (int i = 0; i < n; ++i) {
            tool::string name = value_to_string(CsTupleElement(names, i));
            tool::value  val  = value_to_value_def(
                c, CsEnvElement(env, CsFirstEnvElement + n - 1 - i));
            map.set_prop(name, val);
          }
          vec.push(map);
        }
      }

      fp = fp->next.get(c);
    }

    return tool::value::make_array(vec());
  }

  // this function is used by VM.debugEval to get values from call frames
  value* DebugLocalVarRef(VM *c, value name) {

    if (!c->pdebug || !c->pdebug->breakpointFrame.is_set())
      return nullptr;
    CsFrame *fp =
      c->pdebug->breakpointFrame.get(c);

    while (fp && fp < (CsFrame *)c->stackTop)
    {
      if ((fp->pdispatch == &CsCallCDispatch || fp->pdispatch == &CsTopCDispatch))
      {
        CallFrame *frame = static_cast<CallFrame *>(fp);

        value env = ptr_value(&frame->stackEnv);

        do {

          int   envSize = CsEnvSize(env);
          value names = CsEnvNames(env);

          if (CsTupleP(names)) {

            int n = CsTupleSize(names);

            static value args_marker = CsSymbolOf(W("args"));

            int loc_start = CsFirstEnvElement + n - 1;

            if (CsTupleName(names) == args_marker) {
              loc_start = envSize - 3;
              if ( name == THIS_SYM)
                return CsEnvElementRef(env, envSize - 1);
            }

            for (int i = 0; i < n; ++i) {
              //dbg_printf("name %S\n", value_to_string(CsTupleElement(names, i)).c_str());
              if ( name == CsTupleElement(names, i) )
                return CsEnvElementRef(env, loc_start - i);
            }
          }
          env = CsEnvNextFrame(env);

        } while (CsEnvironmentP(env) || CsMovedEnvironmentP(env));

        break;
      }
      else if (fp->pdispatch == &CsBlockCDispatch) {
        BlockFrame *block = static_cast<BlockFrame *>(fp);
        value       env = ptr_value(&block->stackEnv);
        value       names = CsEnvNames(env);
        if (CsTupleP(names)) {
          //tool::value map = tool::value::make_map();
          int         n = CsTupleSize(names);
          for (int i = 0; i < n; ++i) {
            //dbg_printf("name %S\n", value_to_string(CsTupleElement(names, i)).c_str());
            if (name == CsTupleElement(names, i))
              return CsEnvElementRef(env, CsFirstEnvElement + n - 1 - i);
          }
        }
      }

      fp = fp->next.get(c);
    }

    return nullptr;
  }


  /* CsStackTrace - display a stack trace */
  void CsStackTrace(VM *c) { CsStreamStackTrace(c, c->standardOutput); }

  value CsGetNextMember(VM *c, value *index, value collection, int nr) {
    dispatch *pd = CsGetDispatch(collection);
    if (pd->getNextElement)
      return (*(pd->getNextElement))(c, index, collection, nr);
    else if (collection == NOTHING_VALUE || collection == UNDEFINED_VALUE ||
             collection == NULL_VALUE)
      return NOTHING_VALUE;
    CsThrowKnownError(c, CsErrNoSuchFeature, collection, "enumeration");
    return NOTHING_VALUE;
  }

  value CsGetRange(VM *c, value col, value start, value end) {
    if (CsStringP(col)) {
      int iStart = CsIntegerP(start) ? CsIntegerValue(start) : 0;
      int iEnd   = CsIntegerP(end) ? CsIntegerValue(end) : -1;
      return CsStringSlice(c, col, iStart, iEnd);
    } else if (CsVectorP(col)) {
      int iStart = CsIntegerP(start) ? CsIntegerValue(start) : 0;
      int iEnd   = CsIntegerP(end) ? CsIntegerValue(end) : -1;
      return CsVectorSlice(c, col, iStart, iEnd);
    }
#if defined(TISCRIPT_USE_PERSISTENCE)
    else if (CsDbIndexP(c, col)) {
      return CsDbIndexSlice(c, col, start, end, true /* ascent */,
                            true /*startIncluded*/, true /*endIncluded*/);
    }
#endif
    else
      CsThrowKnownError(c, CsErrNoSuchFeature, col, "range operation");
    return UNDEFINED_VALUE;
  }

  void CsSilentPropertySetter(VM *c, value obj, value value) {
    assert(false);
  }

  bool vp_method::set(VM *c, value prop, value obj, value val) {
    if (!set_handler)
      CsThrowKnownError(c, CsErrReadOnlyProperty, prop);
    if (set_handler == CsSilentPropertySetter)
      return false;
    if (tag)
      (*((vp_set_ext_t)set_handler))(c, obj, val, tag);
    else
      (*set_handler)(c, obj, val);
    check_thrown_error(c);
    return true;
  }


} // namespace tis
