/* heap.c - heap management routines */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <stdlib.h>
#include <string.h>
#include "cs.h"

#if defined(WINDOWS) && defined(TISCRIPT_USE_VIRTUAL_MEMORY)
#define USE_VIRTUAL_MEMORY_ON_WINDOWS
#endif

// Below is an implementation of Cheney's moving GC
// see: http://en.wikipedia.org/wiki/Cheney's_algorithm

#define CHECK_ALIGNMENT(ptr) assert((((uint_ptr)ptr) % (sizeof(void*)) == 0));

namespace tis {
  unsigned int VM::default_features = 0;

  void CsSetFeatures(VM * pvmOrNull,
                     uint features /*FEATURE_*** combination */) {

    if (pvmOrNull)
      pvmOrNull->features = features;
    else
      VM::default_features = features;
  }

  /* VALUE */

  inline void ScanValue(VM *c, value o) { CsQuickGetDispatch(o)->scan(c, o); }

  /* prototypes */
  static void InitInterpreter(VM *c);

  static CsMemorySpace *NewMemorySpace(VM *c, size_t size);

  /* create/destroy - make/destroy a virtual machine */

  void CopyStackRefValues(VM *c) {
    value **p   = c->values_on_c_stack.head();
    value **end = c->values_on_c_stack.tail();
    for (; p < end; ++p)
      if (**p) **p = CsCopyValue(c, **p);
  }

  /*
  VM* VM::create(unsigned int features, long size,long expandSize,long
  stackSize)
  {
    VM* pvm = (VM*)::malloc(sizeof(VM));
    if( pvm )
    {
      if(pvm->init(features,size,expandSize,stackSize))
        return pvm;
      ::free(pvm);
    }
    return 0;
  }

  void VM::destroy(VM* pvm)
  {
    if( pvm )
    {
      pvm->finalize();
      ::free(pvm);
    }
  }
  */

  void finalize() { symbol_table().clear(); }

  /* CsMalloc - allocate memory */
  void *CsMalloc(VM *c, size_t size) {
    size_t         totalSize = sizeof(unsigned long) + size;
    unsigned long *p         = (unsigned long *)malloc(totalSize);
    if (!p) {
#ifdef WINDOWS
      alert("not enough memory");
#endif
      abort();
    }

    *p++ = (unsigned long)(totalSize);
    c->totalMemory += (unsigned long)(totalSize);
    ++c->allocCount;
    return (void *)p;
  }
  /* CsFree - free memory */
  void CsFree(VM *c, void *p) {
    if (!p) return;
    unsigned long *p1        = (unsigned long *)p;
    unsigned long  totalSize = *--p1;
    if (c) {
      c->totalMemory -= totalSize;
      --c->allocCount;
    }
    free(p1);
  }

  static tool::thread_context<VM*> thread_context_VM;

  extern void CsInitDuration(VM *c);

  // void finalize();
  VM::VM(unsigned int features, long size, long expandSize, long stackSize)
      : ploader(nullptr), queue(0) {
    /* initialize */

    exec = &main_exec;

    //for(value& v: val)
    //  v = NOTHING_VALUE;
    val = NOTHING_VALUE;
        
    this->expandSize = expandSize;
    this->ploader    = this;
    this->pdebug     = 0;

    this->pins.next = &this->pins;
    this->pins.prev = &this->pins;

    if (features == 0xFFFFFFFF)
      this->features = default_features;
    else
      this->features = features;

    this->queue = tool::async::dispatch::current();

    // make the initial old memory space
    this->oldSpace = NewMemorySpace(this, size);
    // make the initial new memory space
    this->newSpace = NewMemorySpace(this, size);

    // this->pNextNew = &this->newSpace->next;

    /* make the stack */
    stack = (value *)CsMalloc(this, (size_t)(stackSize * sizeof(value)));

    /* initialize the stack */
    this->stackTop = this->stack + stackSize;
    this->fp       = (CsFrame *)this->stackTop;
    this->sp       = this->stackTop;

    /* initialize the global scope */
    this->globalScope.c           = this;
    this->globalScope.globals     = CsNewNamespaceInstance(this, UNDEFINED_VALUE, NULL_VALUE);
    this->currentNS               = this->globalScope.globals;
    this->globalScope.previous_ns = 0;

    this->scopes.push(&this->globalScope);

    notificationList          = UNDEFINED_VALUE;
    unhandledExceptionHandler = UNDEFINED_VALUE;
    // breakpointHitHandler = UNDEFINED_VALUE;

    /* enter undefined into the symbol table */
    CsSetGlobalValue(CsCurrentScope(this), UNDEFINED_VALUE, UNDEFINED_VALUE);
    CsSetGlobalValue(CsCurrentScope(this), NOTHING_VALUE, NOTHING_VALUE);

    /* make the true and false symbols */
    CsSetGlobalValue(CsCurrentScope(this), TRUE_VALUE, TRUE_VALUE);
    CsSetGlobalValue(CsCurrentScope(this), FALSE_VALUE, FALSE_VALUE);

    /* initialize the interpreter */
    InitInterpreter(this);

    /* create the base of the obj heirarchy */
    CsInitObject(this);

    /* initialize the built-in types */
    CsInitTypes(this);
    CsInitMethod(this);
    CsInitVector(this);
    CsInitTuple(this);
    CsInitSymbol(this);
    CsInitString(this);
    CsInitInteger(this);
    CsInitColor(this);
    CsInitLength(this);
    CsInitAngle(this);
    CsInitDuration(this);
    CsInitByteVector(this);
    CsInitFloat(this);

    CsAddTypeSymbols(this);

    CsInitErrorType(this);

    /* initialize the "external" types */
    CsInitFile(this);
    CsInitRegExp(this);
    CsInitDate(this);
    CsInitAsset(this);
#if defined(TISCRIPT_USE_PERSISTENCE)
    CsInitStorage(this);
    CsInitDbIndex(this);
#endif
    CsInitXmlScanner(this);

    CsInitSystem(this);
    CsInitVM(this);
    CsInitCoroutine(this); // enables await and yield
    CsInitNet(this);

    /* initialize the standard i/o streams */
    this->standardInput  = &null_stream;
    this->standardOutput = &null_stream;
    this->standardError  = &null_stream;

    /* add the library functions to the symbol table */
    // if(features & FEATURE_EXT_1)
    CsEnterLibrarySymbols(this);

    /* use the math routines */
    // if(features & FEATURE_MATH)
    CsUseMath(this);

    /* use the eval package */
    //if (this->features & FEATURE_EVAL) 
    CsUseEval(this);

    if (!thread_context_VM.get()) thread_context_VM.set(this);

    /* return successfully */
  }

  /* NewMemorySpace - make a new semi-space */
  static CsMemorySpace *NewMemorySpace(VM *c, size_t size) {
#ifndef USE_VIRTUAL_MEMORY_ON_WINDOWS
    CsMemorySpace *space =
        (CsMemorySpace *)CsMalloc(c, sizeof(CsMemorySpace) + size);

    space->base     = (byte *)space + sizeof(CsMemorySpace);
    space->free     = space->base;
    space->top      = space->base + size;
    space->cObjects = 0;
    space->bObjects = 0;
    // space->next = 0;
#else
    CsMemorySpace *space = (CsMemorySpace *)::VirtualAlloc(NULL, TISCRIPT_VIRTUAL_MEMORY_SIZE, MEM_RESERVE, PAGE_READWRITE);
    if (!space) {
      alert("Cannot reserve virtual memory");
      abort();
    }
    if (!::VirtualAlloc(space, size, MEM_COMMIT, PAGE_READWRITE)) {
      alert("Cannot commit virtual memory");
      abort();
    }
    space->base = (byte *)space + sizeof(CsMemorySpace);
    space->free = space->base;
    space->top = (byte *)space + size;
    space->cObjects = 0;
    space->bObjects = 0;
#endif
    return space;
  }

  static void FreeMemorySpace(VM *c, CsMemorySpace *addr) { CsFree(c, addr); }

  /* CsFreeInterpreter - free an interpreter structure */
  VM::~VM() {

    tool::critical_section cs(guard);
        

    // this has to be done first as it uses gc_callback
    detach_all_gc_callbacks();

    if (thread_context_VM.get() == this) thread_context_VM.set(nullptr);

    if (this->standardInput) this->standardInput->close();
    if (this->standardOutput) this->standardOutput->close();
    if (this->standardError) this->standardError->close();

    dispatch *d, *nextd;

    //if (this->features & FEATURE_EVAL) 
    CsUnuseEval(this);

    exec = nullptr;

    /* destroy cobjects */
    CsDestroyAllCObjects(this);
    CsDestroyAllByteVectors(this);

    pvalue *t = pins.next;

    while (t && t != &pins) {
      pvalue *tn = t->next;
      t->prune();
      t = tn;
    }

    assert(t == &pins);
    /* free memory spaces */
    FreeMemorySpace(this, oldSpace);
    FreeMemorySpace(this, newSpace);

    /* free the stack */
    CsFree(this, this->stack);

    /* free the type list */
    for (d = this->types; d != nullptr; d = nextd) {
      nextd = d->next;
      CsFreeDispatch(this, d);
    }
  }

  VM *VM::get_current() { return thread_context_VM.get(); }

  /*static bool get_stream(const pvalue& tv, tool::handle<async_stream>& as)
  {
    if( tv.pvm && tv.val)
    {
      //??? tool::critical_section cs(tv.pvm->guard);
      stream* s = CsFileStream(tv.val);
      if(!s->is_async_stream())
        return false;
      as = static_cast<async_stream*>(s);
    }
    return true;
  }*/

  // ATTN: must be called in context(thread of) VM* c
  /*
  bool CsConnect(VM* c, pvalue& input, pvalue& output, pvalue& error)
  {
    tool::handle<async_stream> sin,sout,serr;
    if( get_stream(input, sin) &&
        get_stream(output, sout) &&
        get_stream(error, serr))
    {
      if(sin)   { sin->add_ref();  c->standardInput->close(); c->standardInput =
  sin.ptr(); } if(sout)  { sout->add_ref(); c->standardOutput->close();
  c->standardOutput = sout.ptr(); } if(serr)  { serr->add_ref();
  c->standardError->close(); c->standardError = serr.ptr(); } return true;
    }
    return false;
  }
  */

  /* CsMakeScope - make a variable scope structure */
  /*CsScope *CsMakeScope(VM *c,CsScope *proto)
  {
      CsScope *scope = (CsScope *)CsAlloc(c,sizeof(CsScope));
      if (scope) {
          scope->c = c;
          scope->globals = CsMakeObject(c,CsScopeObject(proto));
          scope->ns = UNDEFINED_VALUE;
          scope->next = c->scopes;
          c->scopes = scope;
      }
      return scope;
  }

  CsScope *CsMakeScopeFromObject(VM *c,value gobject)
  {
      CsScope *scope = (CsScope *)CsAlloc(c,sizeof(CsScope));
      if (scope) {
          scope->c = c;
          scope->globals = gobject;
          scope->next = c->scopes;
          scope->ns = UNDEFINED_VALUE;
          c->scopes = scope;
          CsSetObjectClass(scope->globals,CsGlobalScope(c)->globals);
      }
      return scope;
  }*/

  /* CsFreeScope - free a variable scope structure
  void CsFreeScope(CsScope *scope)
  {
      VM *c = scope->c;
      CsScope **pNext,*s;
      if (scope != CsGlobalScope(c) && scope != CsCurrentScope(c)) {
          for (pNext = &c->scopes; (s = *pNext) != nullptr; pNext = &s->next)
              if (s == scope) {
                  *pNext = s->next;
                  break;
              }
          CsFree(c,scope);
      }
  }*/

  /* CsInitScope - remove all variables from a scope
  void CsInitScope(CsScope *scope)
  {
      value obj = CsScopeObject(scope);
      CsSetObjectProperties(obj,UNDEFINED_VALUE);
      CsSetObjectPropertyCount(obj,0);
  }*/

  /* InitInterpreter - initialize an interpreter structure */
  static void InitInterpreter(VM *c) {
    /* initialize the stack */
    c->fp = (CsFrame *)c->stackTop;
    c->sp = c->stackTop;

    /* initialize the registers */
    c->val       = UNDEFINED_VALUE;
    c->env       = UNDEFINED_VALUE;
    c->currentNS = UNDEFINED_VALUE;
    c->code      = 0;
  }

  /* CsAllocate - allocate memory for a value */
  value CsAllocate(VM *c, size_t size) {

    assert(size >= sizeof(CsBrokenHeart));

  // CsMemorySpace *newSpace;
    long           expandSize;
#ifndef USE_VIRTUAL_MEMORY_ON_WINDOWS
    CsMemorySpace *oldSpace;
#endif
    value val;

    tool::critical_section cs(c->guard);

    // todo: we can try to align to 8 on 64 bit platforms to improve cache
    // access but it will require more space
    // size = ((size + 3) / 4) * 4;
    assert((size % sizeof(void*)) == 0);

    /* look for free space
    for (newSpace = c->newSpace; newSpace != nullptr; newSpace = newSpace->next)
        if (newSpace->free + size <= newSpace->top) {
            memset(newSpace->free, 0, size);
            val = ptr_value((header*)newSpace->free);
            newSpace->free += size;
            return val;
        }
    */

    if (c->newSpace->free + size < c->newSpace->top) {
      memset(c->newSpace->free, 0, size);
      val = ptr_value((header *)c->newSpace->free);
      c->newSpace->free += size;
      //CHECK_ALIGNMENT(val);
      return val;
    }

    /* collect garbage */
    CsCollectGarbage(c);

    /* look again */
    if (c->newSpace->free + size < c->newSpace->top) {
      bool need_allocate = false;
      //  using allocation threashold to avoid frequent allocations:
      if (c->newSpace->top - c->newSpace->free <
          (c->newSpace->free - c->newSpace->base) / 4)
        need_allocate = true;
      memset(c->newSpace->free, 0, size);
      val = ptr_value((header *)c->newSpace->free);
      c->newSpace->free += size;
      if (!need_allocate) {
        //CHECK_ALIGNMENT(val);
        return val;
      }
    }

    /* No luck, need to reallocate memory spaces */
    /* check for being able to expand memory */
    if ((expandSize = c->expandSize) == 0) CsInsufficientMemory(c);

#ifndef USE_VIRTUAL_MEMORY_ON_WINDOWS
    size_t neededSize =
        (c->newSpace->free - c->newSpace->base) // allocated already
        + size;                                 // plus size requested

    /* make sure we allocate at least as much as needed */
    // if (size > expandSize)
    //    expandSize = size;

    // expandSize = (size / expandSize) * expandSize + (size % expandSize?
    // expandSize : 0);
    neededSize = (neededSize / expandSize) * expandSize + (neededSize % expandSize ? expandSize : 0);

    /* allocate more old space, it will be used as newSpace in CsCollectGarbage
     */
    oldSpace = NewMemorySpace(c, neededSize);
    if (!oldSpace) CsInsufficientMemory(c);

    tool::swap(oldSpace, c->oldSpace);
    FreeMemorySpace(c, oldSpace);

    /* allocate more new space
    if (!(newSpace = NewMemorySpace(c,expandSize))) {
        CsFree(c,oldSpace);
        CsInsufficientMemory(c);
    }
    see below
    */

    // collect garbage - copy stuff to our brand new space
    CsCollectGarbage(c);

    /* reallocate another half */
    oldSpace = NewMemorySpace(c, neededSize);
    if (!oldSpace) CsInsufficientMemory(c);

    tool::swap(oldSpace, c->oldSpace);
    FreeMemorySpace(c, oldSpace);
#else
    long allocated = c->newSpace->top - (byte *)c->newSpace;
#ifdef UNDER_CE
    long extra     = size + allocated / 3;
#else
    // long extra = size + allocated / 2;
    //long extra = max(long(size), allocated); // more agressive than above.
    long extra = 0;
    while (extra < long(size))
        extra += expandSize;
#endif
    // lets commit more pages
    if (!::VirtualAlloc((byte *)c->newSpace + allocated, extra, MEM_COMMIT,
                        PAGE_READWRITE))
      CsInsufficientMemory(c);
    c->newSpace->top = (byte *)c->newSpace + allocated + extra;

    if (!::VirtualAlloc((byte *)c->oldSpace + allocated, extra, MEM_COMMIT,
                        PAGE_READWRITE))
      CsInsufficientMemory(c);
    c->oldSpace->top = (byte *)c->oldSpace + allocated + extra;

    c->totalMemory += size;
#endif

    /* return some of the newly allocated space */
    memset(c->newSpace->free, 0, size);
    val = ptr_value((header *)c->newSpace->free);
    c->newSpace->free += size;
    assert(c->newSpace->free <= c->newSpace->top);
    //CHECK_ALIGNMENT(val);
    return val;
  }

  /* CsCollectGarbageAndReclaim - does GC and tries to reduce heap halves */
  CsHeapStats CsCollectGarbageAndReclaim(VM *c, int factor) {

    // CsMemorySpace *newSpace;
    CsMemorySpace *oldSpace;

    tool::critical_section cs(c->guard);
    
    // collect garbage
    auto gcstats = CsCollectGarbage(c);
    if (gcstats.free * factor < gcstats.used)
      return gcstats;

    size_t neededSize = gcstats.used + gcstats.used * factor;

    /* allocate more old space, it will be used as newSpace in CsCollectGarbage  */
    oldSpace = NewMemorySpace(c, neededSize);
    if (!oldSpace) CsInsufficientMemory(c);

    tool::swap(oldSpace, c->oldSpace);
    FreeMemorySpace(c, oldSpace);

    // collect garbage - copy stuff to our brand new space
    gcstats = CsCollectGarbage(c);

    /* reallocate another half */
    oldSpace = NewMemorySpace(c, neededSize);
    if (!oldSpace) CsInsufficientMemory(c);

    tool::swap(oldSpace, c->oldSpace);
    FreeMemorySpace(c, oldSpace);

    return gcstats;

  }


  /*tslice<byte> CsAllocate(VM *c, size_t size, value &r) {
    r = CsAllocate(c, size);
    return target(ptr<byte>(r), size);
  }*/

  /* CsMakeDispatch - make a new type pdispatch */
  dispatch *CsMakeDispatch(VM *c, const char *typeName, dispatch *prototype) {
    size_t    typeNameLen = strlen(typeName);
    size_t    totalSize   = sizeof(dispatch) + typeNameLen + 1;
    dispatch *d;

    /* allocate a new pdispatch structure */
    if ((d = (dispatch *)CsAlloc(c, (unsigned long)totalSize)) == nullptr) {
      CsDrop(c, 1);
      return nullptr;
    }
    memset(d, 0, totalSize);

    /* initialize */
    *d          = *prototype;
    d->baseType = d;
    d->typeName = (char *)d + sizeof(dispatch);
#ifdef HAS_STR_S
    strcpy_s((char *)d->typeName, typeNameLen + 1, typeName);
#else
    strcpy((char *)d->typeName, typeName);
#endif

    /* add the type to the type list */
    d->next  = c->types;
    c->types = d;

    /* return the new type */
    return d;
  }




  /* CsFreeDispatch - free a type pdispatch structure */
  void CsFreeDispatch(VM *c, dispatch *d) { CsFree(c, d); }

  uint CsFreeSpace(VM *c) {
    CsMemorySpace *space = c->newSpace;
    return uint(space->top - space->free);
  }

#ifdef _DEBUG
  void printPins(VM *c, const char *msg) {
    /*    c->standardError->printf(L"pins:%S\n", msg);
        pvalue *t = c->pins.next;
        while(t != &c->pins)
        {
          if(t->is_set())
          {
            dispatch* pd = CsGetDispatch(t->val);
            c->standardError->printf(L"pinned %x %S\n", t, pd->typeName);
          }
          else
            t = t;
          t = t->next;
        } */
  }

  void checkIfPinned(VM *c, pvalue *pv) {
    pvalue *t = c->pins.next;
    while (t != &c->pins) {
      if (t == pv) {
        return;
      } else
        t = t;
      t = t->next;
    }
    assert(false);
  }

#endif

  /* CsCollectGarbage - garbage collect a heap */
  CsHeapStats CsCollectGarbage(VM *c) {
    // CsProtectedPtrs *ppb;
    byte *         scan;
    CsMemorySpace *ms;
    dispatch *     d;
    value          obj;

    CsHeapStats r = {0};

#ifdef _DEBUG
    c->standardOutput->put_str("[GC");
#endif

    if (c->collecting_garbage) {
#ifdef _DEBUG
      c->standardOutput->put_str(" nested call, skiping]\n");
#endif
      return r;
    }

    tool::critical_section _1(c->guard);
    tool::auto_state<bool> _2(c->collecting_garbage, true);

    ++c->gc_generation;

#if defined(USE_VIRTUAL_MEMORY_ON_WINDOWS) && defined(UNDER_CE)
    {
      // This has to be more smart, e.g. to be based on some statistic about
      // last three GC cycles or so.

      int free_space = c->newSpace->top - c->newSpace->free;
      int allocated  = c->newSpace->free - (byte *)c->newSpace;
      if (free_space > (allocated / 3)) {
        // too much memory allocated, lets decommit
        int extra = allocated / 3;
        ::VirtualAlloc(c->newSpace->free + extra + c->page_size,
                       free_space - extra, MEM_DECOMMIT, PAGE_READWRITE);
        c->newSpace->top = (byte *)c->newSpace + allocated + extra;

        ::VirtualAlloc(c->newSpace->free + extra + c->page_size,
                       free_space - extra, MEM_DECOMMIT, PAGE_READWRITE);
        c->oldSpace->top = (byte *)c->oldSpace + allocated + extra;
      }
    }
#endif

#if defined(TISCRIPT_USE_PERSISTENCE)
    /* run prepare opened storages for garbage collection */
    for (int i = c->storages.size() - 1; i >= 0; i--)
      StoragePreGC(c, c->storages[i]);
#endif

    /* reverse the memory spaces */
    ms          = c->oldSpace;
    c->oldSpace = c->newSpace;
    c->newSpace = ms;

    /* reset the new space pointers */
    // for (; ms != nullptr; ms = ms->next)
    //    ms->free = ms->base;
    ms->free = ms->base;

    c->globalScope.globals = CsCopyValue(c, c->globalScope.globals);

    // copy the root objects

    c->notificationList          = CsCopyValue(c, c->notificationList);
    c->unhandledExceptionHandler = CsCopyValue(c, c->unhandledExceptionHandler);
    // c->breakpointHitHandler = CsCopyValue(c,c->breakpointHitHandler);

    c->objectObject = CsCopyValue(c, c->objectObject);
    c->classObject  = CsCopyValue(c, c->classObject);

    /* copy basic types */
    c->vectorObject   = CsCopyValue(c, c->vectorObject);
    c->tupleObject    = CsCopyValue(c, c->tupleObject);
    c->methodObject   = CsCopyValue(c, c->methodObject);
    c->propertyObject = CsCopyValue(c, c->propertyObject);

    c->symbolObject   = CsCopyValue(c, c->symbolObject);
    c->stringObject   = CsCopyValue(c, c->stringObject);
    c->errorObject    = CsCopyValue(c, c->errorObject);
    c->integerObject  = CsCopyValue(c, c->integerObject);
    c->colorObject    = CsCopyValue(c, c->colorObject);
    c->lengthObject   = CsCopyValue(c, c->lengthObject);
    c->angleObject    = CsCopyValue(c, c->angleObject);
    c->durationObject = CsCopyValue(c, c->durationObject);

    c->floatObject      = CsCopyValue(c, c->floatObject);
    c->byteVectorObject = CsCopyValue(c, c->byteVectorObject);

    /* copy the type list */
    for (d = c->types; d; d = d->next) {
      if (d->obj) d->obj = CsCopyValue(c, d->obj);
    }

    /* copy values referred from C stack */
    CopyStackRefValues(c);

    /* copy "pinned" values */
    pvalue *t = c->pins.next;
    while (t && (t != &c->pins)) {
      if (t->is_set())
        t->val = CsCopyValue(c, t->val);
      t = t->next;
    }

    /* copy the stack */
    CsCopyStack(c);

    // copy the current code obj
    if (c->code) c->code = CsCopyValue(c, c->code);

    // copy the value register
    c->val = CsCopyValue(c, c->val);

    // copy the environment
    c->env = CsCopyValue(c, c->env);

    // copy the currentNS
    c->currentNS = CsCopyValue(c, c->currentNS);

    // if( c->currentTaskInstance )
    //  c->currentTaskInstance = CsCopyValue(c,c->currentTaskInstance);

    for (int sn = c->scopes.last_index(); sn > 0;
         --sn) // sn > 0, note globalScope is always at index 0
    {
      CsScope *scope     = c->scopes[sn];
      scope->globals     = CsCopyValue(c, scope->globals);
      scope->previous_ns = CsCopyValue(c, scope->previous_ns);
    }

    c->exec->scan(c);
    if (c->exec != &c->main_exec) c->main_exec.scan(c);

    /* pending observers */
    FOREACH(n, c->pending_observers) {
      c->pending_observers[n] = CsCopyValue(c, c->pending_observers[n]);
    }

    FOREACH(n, c->pending_tasks) {
      if(c->pending_tasks[n])
        c->pending_tasks[n] = CsCopyValue(c, c->pending_tasks[n]);
    }

    FOREACH(n, c->active_tasks) {
      c->active_tasks[n] = CsCopyValue(c, c->active_tasks[n]);
    }

    //FOREACH(n, c->generators) {
    //  c->generators[n] = CsCopyValue(c, c->generators[n]);
    //}

    /* copy any user objects */
    if (c->protectHandler) (*c->protectHandler)(c, c->protectData);

    if (auto async_dispatch = async::dispatch::current(false)) {
      async_dispatch->each([c](tool::async::entity *tsk) {
        gcable *pg = tsk->get_interface<tis::gcable>();
        if (pg) pg->on_gc(c);
      });
    }

    c->GC_started();

    FOREACH(igc, c->gc_callbacks) {
      gc_callback *gccb = c->gc_callbacks[igc];
      gccb->on_GC(c);
    }

    /* scan and copy until all accessible objects have been copied */

    scan = c->newSpace->base;
    while (scan < c->newSpace->free) {
      obj = ptr_value((header *)scan);
      scan += ValueSize(obj);
      ScanValue(c, obj);
    }

    // fixup cbase and pc
    if (c->code) {
      size_t pcoff = c->pc - c->cbase;
      c->cbase     = CsByteVectorAddress(CsCompiledCodeBytecodes(c->code));
      c->pc        = c->cbase + pcoff;
    }

    r.total = uint(c->newSpace->top - c->newSpace->base);
    r.free  = uint(c->newSpace->top - c->newSpace->free);
    r.used  = r.total - r.free;

#ifdef _DEBUG

    c->standardOutput->printf(
        W(" - %d bytes free out of %d, total memory %u, allocations %u]\n"),
        c->newSpace->top - c->newSpace->free,
        c->newSpace->top - c->newSpace->base, c->totalMemory, c->allocCount);
#endif

#if defined(TISCRIPT_USE_PERSISTENCE)
    /* run CollectGarbage() through all opened storages */
    for (int i = c->storages.size() - 1; i >= 0; i--) {
      if (c->storages[i] && CsBrokenHeartP(c->storages[i])) {
        value newStorage = CsBrokenHeartForwardingAddr(c->storages[i]);
        StoragePostGC(c, newStorage);
        c->storages[i] = newStorage;
      }
    }
#endif

    /* destroy any unreachable cobjects */
    CsDestroyUnreachableCObjects(c);
    CsDestroyUnreachableByteVectors(c);

    c->GC_ended();

    return r;
  }

  void Exec::scan(VM *c) {
    taskInstance = CsCopyValue(c, taskInstance);
    // copy the saved interpreter states
    for (CsSavedState *ss = states; ss != nullptr; ss = ss->next) {
      ss->scan(c);
    }
  }

  /* CsDumpHeap - dump the contents of the heap */
  void CsDumpHeap(VM *c) {
    byte *scan;

    /* first collect the garbage */
    CsCollectGarbage(c);

    /* dump each heap obj */
    scan = c->newSpace->base;
    while (scan < c->newSpace->free) {
      value val = ptr_value((header *)scan);
      scan += ValueSize(val);
      // if (CsCObjectP(val)) {
      CsPrint(c, val, c->standardOutput);
      c->standardOutput->put('\n');
      //}
    }
  }

  /* CsDumpScopes - dump the contents of the heap */
  void CsDumpScopes(VM *c) {
    CsScope *scope; // = &c->currentScope;
    // CsScope *cscope = &c->currentScope;
    // scope = cscope;

    int t = 0;

    value last_globals = 0;
    for (int sn = c->scopes.last_index(); sn >= 0; --sn) {
      scope = c->scopes[sn];

      if (last_globals == scope->globals) continue;

      for (int n = 0; n < t; ++n)
        c->standardOutput->put_str("\t");
      c->standardOutput->printf(W("scope[%d]:\n"), t);

      last_globals = scope->globals;
      each_property gen(c, last_globals);
      value         key = 0, val = 0;
      for (; gen(key, val);) {
        for (int n = 0; n <= t; ++n)
          c->standardOutput->put_str("\t");
        CsPrint(c, key, c->standardOutput);
        c->standardOutput->put_str(" : ");
        CsPrint(c, val, c->standardOutput);
        c->standardOutput->put('\n');
      }
      ++t;
    }
    c->standardOutput->put_str("\n\v");
  }

  void CsDumpObject(VM *c, value obj) {
    value props = CsObjectProperties(obj);
    if (CsHashTableP(props)) {
      int i;
      for (i = 0; i < CsHashTableSize(props); ++i) {
        value p;
        for (p = CsHashTableElement(props, i); p != UNDEFINED_VALUE;
             p = CsPropertyNext(p)) {
          CsPrint(c, CsPropertyTag(p), c->standardOutput);
          c->standardOutput->put_str(" = ");
          CsPrint(c, CsPropertyValue(p), c->standardOutput);
          c->standardOutput->put('\n');
        }
      }
    } else {
      value p;
      for (p = props; p != UNDEFINED_VALUE; p = CsPropertyNext(p)) {
        CsPrint(c, CsPropertyTag(p), c->standardOutput);
        c->standardOutput->put_str(" = ");
        CsPrint(c, CsPropertyValue(p), c->standardOutput);
        c->standardOutput->put('\n');
      }
    }
  }

  /* default handlers */

  /* CsDefaultGetProperty - get the value of a property */
  bool CsDefaultGetProperty(VM *c, value &obj, value tag, value *pValue) {
    CsThrowKnownError(c, CsErrNoProperty, obj, tag);
    return false;
  }

  /* CsDefaultSetProperty - set the value of a property */
  bool CsDefaultSetProperty(VM *c, value obj, value tag, value value) {
    CsThrowKnownError(c, CsErrNoProperty, obj, tag);
    return false;
  }

  value CsDefaultGetItem(VM *c, value obj, value tag) {
    CsThrowKnownError(c, CsErrNoProperty, obj, tag);
    return UNDEFINED_VALUE;
  }
  void CsDefaultSetItem(VM *c, value obj, value tag, value value) {
    CsThrowKnownError(c, CsErrNoProperty, obj, tag);
    /*do nothing*/
  }

  /* CsDefaultNewInstance - create a new instance */
  value CsDefaultNewInstance(VM *c, value proto) {
    CsThrowKnownError(c, CsErrNewInstance, proto);
    return UNDEFINED_VALUE; /* never reached */
  }

  /* CsDefaultPrint - print an obj */
  bool CsDefaultPrint(VM *c, value obj, stream *s, bool toLocale) {
    // char buf[64];
    // sprintf(buf,"%08lx",(long)obj);
    return s->put_str(W("[object ")) &&
           s->put_str(CsTypeName(obj))
           //&& s->put('-')
           //&& s->put_str(buf)
           && s->put(']');
  }

  value CsDefaultCopy(VM *c, value obj) {
    /* don't copy an obj that is already in new space */
    if (CsIsNewObjectP(c, obj)) return obj;

    /* CsMemorySpace *space = c->newSpace;
    while(space && (space->free + sz >= space->top))
    space = space->next;
    assert(space); */

    size_t sz = ValueSize(obj);
    value  newObj;

    /* find a place to put the new obj */
    newObj = ptr_value((header *)c->newSpace->free);

    /* copy the obj */

    target(ptr<byte>(newObj), (size_t)sz).copy(ptr<byte>(obj));

    c->newSpace->free += sz;

#if defined(TISCRIPT_USE_PERSISTENCE)
    if (CsIsPersistent(c, obj)) {
      CsStorageOfPersistent(newObj) =
        CsCopyValue(c, CsStorageOfPersistent(obj));
      assert(ptr<persistent_header>(newObj)->oid);
    }
#endif

    // value* pa = CsFixedVectorAddress(newObj);

    /* store a forwarding address in the old obj */
    CsSetDispatch(obj, &CsBrokenHeartDispatch);
    CsBrokenHeartSetForwardingAddr(obj, newObj);

    /* return the new obj */
    return newObj;
  }


  /* CsCsDefaultScan - scan an obj without embedded pointers */
  void CsDefaultScan(VM *c, value obj) {}

  /* CsDefaultHash - default obj hash routine */
  int_t CsDefaultHash(value obj) { return (int_t)obj; }

  /* BROKEN HEART */

  static long  BrokenHeartSize(value obj);
  static value BrokenHeartCopy(VM *c, value obj);

  /* CsDefaultGetProperty - get the value of a property */
  static bool BrokenHeartGetProperty(VM *c, value &obj, value tag,
                                     value *pValue) {

    CsThrowKnownError(c, CsErrNoProperty, obj, tag);
    // string msg = string::format("attempt to get property '%s' of GCed
    // object",);  CsThrowKnownError(c, CsErrGenericError, "attempt to get
    // property of GCed object");
    return false;
  }

  /* CsDefaultSetProperty - set the value of a property */
  bool BrokenHeartSetProperty(VM *c, value obj, value tag, value value) {
    string propName = value_to_string(tag);

    string msg =
        string::format("object, attempt to set property \"%s\" on GCed object",
                       propName.c_str());
    CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, msg.c_str());
    return true;
  }

  dispatch CsBrokenHeartDispatch = {
      "BrokenHeart",          &CsBrokenHeartDispatch, BrokenHeartGetProperty,
      BrokenHeartSetProperty, CsDefaultNewInstance,   CsDefaultPrint,
      BrokenHeartSize,        BrokenHeartCopy,        CsDefaultScan,
      CsDefaultHash,          CsDefaultGetItem,       CsDefaultSetItem};

  static long BrokenHeartSize(value obj) { return sizeof(CsBrokenHeart); }

  static value BrokenHeartCopy(VM *c, value obj) {
    return CsBrokenHeartForwardingAddr(obj);
  }

  // nullptr value, cannot appear normally ...

  static long  NilSize(value obj);
  static value NilCopy(VM *c, value obj);

  /* CsDefaultGetProperty - get the value of a property */
  static bool NilGetProperty(VM *c, value &obj, value tag, value *pValue) {
    string propName = value_to_string(tag);
    string msg      = string::format("attempt to get property '%s' of nullptr",
                                propName.c_str());
    CsThrowKnownError(c, CsErrGenericError, msg.c_str());
    return false;
  }

  /* CsDefaultSetProperty - set the value of a property */
  bool NilSetProperty(VM *c, value obj, value tag, value value) {
    string propName = value_to_string(tag);
    string msg      = string::format(
        "object, attempt to set property \"%s\" on nullptr", propName.c_str());
    CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, msg.c_str());
    return true;
  }

  dispatch CsNilDispatch = {"nullptr",
                            &CsNilDispatch,
                            NilGetProperty,
                            NilSetProperty,
                            CsDefaultNewInstance,
                            CsDefaultPrint,
                            NilSize,
                            NilCopy,
                            CsDefaultScan,
                            CsDefaultHash,
                            CsDefaultGetItem,
                            CsDefaultSetItem};

  static long NilSize(value obj) { return 0; }

  static value NilCopy(VM *c, value obj) { return obj; }

  void pvalue::pin(VM *c, value v) {
    unpin();
    if (c) {
      tool::critical_section cs(c->guard);
      pvm = c;
      (next = c->pins.next)->prev = this;
      (prev = &c->pins)->next     = this;
      val                         = v;
    } else {
      assert(v == 0);
    }
  }
  void pvalue::unpin() {
    if (!pvm || !next || !prev) return;

    tool::critical_section cs(pvm->guard);

    assert(next->prev == this);
    assert(prev->next == this);

    next->prev = prev;
    prev->next = next;

    next = 0;
    prev = 0;
    pvm  = 0;
    val  = 0;
  }

  /* CsAlloc - allocate memory and initialize it to 0*/
  void *CsAlloc(VM *c, unsigned long size) {
    unsigned long  totalSize = sizeof(unsigned long) + size;
    unsigned long *p         = (unsigned long *)calloc(totalSize, 1);
    if (p) {
      *p++ = totalSize;
      c->totalMemory += totalSize;
      ++c->allocCount;
    } else
      throw std::bad_alloc();
    return (void *)p;
  }

  /* CsInsufficientMemory - report an insufficient memory error */
  void CsInsufficientMemory(VM *c) {
    CsThrowKnownError(c, CsErrInsufficientMemory, 0);
  }

  bool is_valid_ptr_value(VM *c, value v) {
    void *p = ptr<void>(v);
    return p >= c->newSpace->base && p < c->newSpace->free;
  }

  enum _ { NEXT_PTR = 5 };

  static value notificationSymbols[] = {
      CsSymbolOf("add"),          CsSymbolOf("update"),
      CsSymbolOf("delete"),       CsSymbolOf("add-range"),
      CsSymbolOf("update-range"), CsSymbolOf("delete-range")};

  bool CsMerge1Notification(VM *c, value notificationList, value obj, value v0,
                            value v1, value v2, OBJECT_MUTATION_OP mop) {
    if (mop < ADD_RANGE || mop > DELETE_RANGE)
      return false; // we can merge only array operations

    int start = CsIntegerValue(v0);
    int end   = CsIntegerValue(v1);

    for (; notificationList != UNDEFINED_VALUE;
         notificationList = CsTupleElement(notificationList, NEXT_PTR)) {

      if (CsTupleElement(notificationList, 1) != obj) continue;

      int istart = CsIntegerValue(CsTupleElement(notificationList, 2));
      int iend   = CsIntegerValue(CsTupleElement(notificationList, 3));

      value imutop = CsTupleElement(notificationList, 0);

      if (mop == UPDATE_RANGE && imutop == notificationSymbols[ADD_RANGE]) {
        if (start >= istart && start < iend && end > istart && end <= iend)
          return true; // it fully contained inside ADD_RANGE - nothiong to do, merged already
      }

      if (imutop != notificationSymbols[mop])
        break; // only last op can be merged

      if (mop == ADD_RANGE) {
        if (iend == start) { // next to old one
          // update range end
          CsSetTupleElement(notificationList, 3, CsMakeInteger(end));
          return true;                    // merged
        } else if ((end - 1) == istart) { // prior to old one
          CsSetTupleElement(notificationList, 2, CsMakeInteger(start));
          CsSetTupleElement(notificationList, 3,
                            CsMakeInteger(iend + end - start));
          return true; // merged
        }
      } else if (mop == UPDATE_RANGE) {
        if (iend == start) { // immediately after old one
          // update range end
          CsSetTupleElement(notificationList, 3, CsMakeInteger(end));
          return true;              // merged
        } else if (istart == end) { // immediately before old one
          // update range start
          CsSetTupleElement(notificationList, 2, CsMakeInteger(start));
          return true; // merged
        } else if (start >= istart && end <= iend)
          return true; // it is already covered , do nothing

      } else if (mop == DELETE_RANGE) {
        if (istart == end) { // previous
          CsSetTupleElement(notificationList, 2,
                            CsMakeInteger(istart - (end - start)));
          return true;
        } else if (istart == start) { // next
          CsSetTupleElement(
              notificationList, 3,
              CsMakeInteger(start + (end - start) + (iend - istart)));
          return true;
        }
      }
      break;
    }
    return false;
  }

  void CsEnqueueNotification1(VM *c, value &observer, value &obj, value &tag,
                              value &newval, value &oldval,
                              OBJECT_MUTATION_OP mop) {

    if (!CsMethodP(observer)) {
      assert(false);
      return;
    }

    static_assert(MAX_OBJECT_MUTATION_CODE == items_in(notificationSymbols),
                  "OBJECT_MUTATION_OP does not match notificationSymbols");

    value mutop = notificationSymbols[mop];

    value nl = CsMethodNotificationList(observer);

    if (nl != UNDEFINED_VALUE &&
        CsMerge1Notification(c, nl, obj, tag, newval, oldval, mop))
      return;

    PROTECT(observer);

    value notificationRecord =
        CsMakeTuple(c, "notification", mutop, obj, tag, newval, oldval, nl);
    CsSetMethodNotificationList(observer, notificationRecord);
    if (nl == UNDEFINED_VALUE) {
      int no = c->pending_observers.size();
      c->pending_observers.push(observer);
      if (no == 0) // very first observer
        c->post([c]() { return c->deliver_notifications(); });
    }
  }

  void CsEnqueueNotification(VM *c, value observer, value obj, value tag,
                             value newval, value oldval,
                             OBJECT_MUTATION_OP mop) {
    value nth_observer = 0;
    PROTECT(nth_observer, observer, obj, tag, newval, oldval);

    if (CsMethodP(observer))
      CsEnqueueNotification1(c, observer, obj, tag, newval, oldval, mop);
    else if (CsVectorP(observer)) {
      int cnt = CsVectorSize(c, observer);
      for (int n = 0; n < cnt; ++n) {
        nth_observer = CsVectorElement(c, observer, n);
        CsEnqueueNotification1(c, nth_observer, obj, tag, newval, oldval, mop);
      }
    } else {
      assert(false);
      return;
    }
  }

  static value reverseNotificationList(value head) {
    value cursor = UNDEFINED_VALUE;
    value next;
    while (head != UNDEFINED_VALUE) {
      next = CsTupleElement(head, NEXT_PTR);
      CsSetTupleElement(head, NEXT_PTR, cursor);
      cursor = head;
      head   = next;
    }
    return cursor;
  }

  value CsRemoveObserver(VM *c, value obj, value fun);


  bool VM::deliver_notifications() {

    if (delivering_notifications)
      return false; // not reentrant

    auto_state<bool> _(delivering_notifications, true);

    value     list = UNDEFINED_VALUE, observer = UNDEFINED_VALUE;
    
    PROTECT_THIS(list, observer);

    //protected_vector
    GC_vector observers(this);
    observers.elements.swap(pending_observers);

    for (int n = 0; n < observers.elements.size(); ++n) {
      observer = observers.elements[n];
      if (!CsMethodP(observer))
        continue;
      list = reverseNotificationList(CsMethodNotificationList(observer));
      CsSetMethodNotificationList(observer, UNDEFINED_VALUE);
      TRY {
        tis::auto_scope as(this, CsMethodNamespace(observer));
        while (list && CsTupleP(list)) {
          CsCallFunction(&as, observer, 1, list);
          list = CsTupleElement(list, NEXT_PTR);
        }
      }
      CATCH_ERROR_NE {
        if (this->val != UNDEFINED_VALUE) tis::CsHandleUnhandledError(this);
        value obj = CsTupleElement(list, 1);
        CsRemoveObserver(this, obj, observer);
      }
    }

    GC_vector tasks(this);
    tasks.elements.swap(pending_tasks);

    for (int n = 0; n < tasks.elements.size(); ++n) {
      value task = tasks.elements[n];
      if(task)
        CsExecTask(this, task, false);
    }

    if (pending_observers.size() != 0 || pending_tasks.size() != 0)
      this->post([this]() { return this->deliver_notifications(); });

    return true;
  }

} // namespace tis
