/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include "cs_int.h"

namespace tis {

  /* 'File' pdispatch */

  /* method handlers */
  // static value CSF_ctor(VM *c);
  static value CSF_close(VM *c);
  static value CSF_print(VM *c);
  static value CSF_println(VM *c);
  static value CSF_$print(VM *c);
  static value CSF_$println(VM *c);
  static value CSF_printf(VM *c);
  static value CSF_scanf(VM *c);
  static value CSF_getc(VM *c);
  static value CSF_putc(VM *c);
  static value CSF_readln(VM *c);
  // static value CSF_send(VM *c);
  // static value CSF_post(VM *c);
  static value CSF_openFile(VM *c);
  static value CSF_openPipe(VM *c);
  static value CSF_openSocket(VM *c);
  static value CSF_openString(VM *c);
  static value CSF_toString(VM *c);

  static value CSF_isInput(VM *c, value obj);
  static value CSF_isOutput(VM *c, value obj);
  static value CSF_isPipe(VM *c, value obj);
  // static value CSF_proxy(VM *c,value obj);
  // static void  CSF_set_proxy(VM *c,value obj,value val);
  // static value CSF_pending(VM *c,value obj);

  static value CSF_encoding(VM *c, value obj);
  static void  CSF_set_encoding(VM *c, value obj, value val);

  static value CSF_name(VM *c, value obj);

  /* file methods */
  static c_method methods[] = {
      // C_METHOD_ENTRY( "this",      CSF_ctor            ),
      C_METHOD_ENTRY("openFile", CSF_openFile),
      C_METHOD_ENTRY("openSocket", CSF_openSocket),
      C_METHOD_ENTRY("openString", CSF_openString),
      // C_METHOD_ENTRY( "openPipe",         CSF_openPipe        ),
      C_METHOD_ENTRY("close", CSF_close), C_METHOD_ENTRY("print", CSF_print),
      C_METHOD_ENTRY("println", CSF_println), C_METHOD_ENTRY("$", CSF_$print),
      C_METHOD_ENTRY("$n", CSF_$println), C_METHOD_ENTRY("printf", CSF_printf),
      C_METHOD_ENTRY("scanf", CSF_scanf),
      C_METHOD_ENTRY("toString", CSF_toString),
      C_METHOD_ENTRY("getc", CSF_getc), C_METHOD_ENTRY("putc", CSF_putc),
      // C_METHOD_ENTRY( "send",             CSF_send            ),
      // C_METHOD_ENTRY( "post",             CSF_post            ),
      C_METHOD_ENTRY("readln", CSF_readln), C_METHOD_ENTRY(0, 0)};

  /* file properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY("isInput", CSF_isInput, 0),
      VP_METHOD_ENTRY("isOutput", CSF_isOutput, 0),
      // VP_METHOD_ENTRY( "isPipe",    CSF_isPipe,            0  ),
      // VP_METHOD_ENTRY( "proxy",     CSF_proxy,             CSF_set_proxy ),
      // VP_METHOD_ENTRY( "pending",   CSF_pending,           0 ),
      VP_METHOD_ENTRY("encoding", CSF_encoding, CSF_set_encoding),
      VP_METHOD_ENTRY("name", CSF_name, 0),

      // set_request_handler

      // VP_METHOD_ENTRY( "onRead",    CSF_onRead,            CSF_set_onRead  ),
      VP_METHOD_ENTRY(0, 0, 0)};

  /* prototypes */
  static void EnterPort(VM *c, const char *name, stream **pStream);
  static void DestroyFile(VM *c, value obj);

  value FileBinOp(VM *c, int op, value obj, value operand) {
    switch (op) {
    case BC_SHL:
      if (!CsFileStream(obj)->put(c, operand)) {
        tool::string n = CsFileStream(obj)->stream_name();
        CsThrowKnownError(c, CsErrIOError, n.c_str());
      }
      break;
    case BC_SHR: {
      tool::string n = CsFileStream(obj)->stream_name();
      CsThrowKnownError(c, CsErrIOError, n.c_str());
    } break;
    }
    return obj;
  }

  /* CsInitFile - initialize the 'File' obj */
  void CsInitFile(VM *c) {
    /* create the 'File' type */
    c->fileDispatch = CsEnterCPtrObjectType(CsGlobalScope(c), "Stream", methods, properties);
    if (!c->fileDispatch)
      CsInsufficientMemory(c);
    else {
      /* setup alternate handlers */
      c->fileDispatch->destroy = DestroyFile;
      c->fileDispatch->binOp   = FileBinOp;
    }

    /* enter the built-in ports */
    EnterPort(c, "stdin", &c->standardInput);
    EnterPort(c, "stdout", &c->standardOutput);
    EnterPort(c, "stderr", &c->standardError);
  }

  bool CsFileP(VM *c, value obj) { return CsIsType(obj, c->fileDispatch); }

  /* EnterPort - add a built-in port to the symbol table */
  static void EnterPort(VM *c, const char *name, stream **pStream) {
    stream *s = CsMakeIndirectStream(c, pStream);
    if (!s) CsInsufficientMemory(c);
    CsCheck(c, 2);
    CsPush(c, CsMakeFile(c, s));
    CsPush(c, CsSymbolOf(name));
    CsSetNamespaceConst(c, CsTop(c), CsTop(c, 1));
    CsDrop(c, 2);
  }

  /* CsMakeFile - make a 'File' obj */
  value CsMakeFile(VM *c, stream *s) {
    return CsMakeCPtrObject(c, c->fileDispatch, s);
  }

  /* CSF_ctor - built-in method 'initialize' */
  /*
  static value CSF_ctor(VM *c)
  {
      wchar *fname,*mode;
      stream *s;
      value val;
      CsParseArguments(c,"V=*SS",&val,c->fileDispatch,&fname,&mode);
      if (!(s = OpenFileStream(c,fname,mode)))
          return UNDEFINED_VALUE;
      CsSetCObjectValue(val,s);
      CsCtorRes(c) = val;
      return val;
  }*/

  static void CsTimeout(VM *c) { CsThrowKnownError(c, CsErrIOTimeout); }

  static value CSF_openFile(VM *c) {
    const wchar *fmode = W("");
    wchar *      fname, *mode = (wchar *)&fmode;
    stream *     s;
    CsParseArguments(c, "**S|S", &fname, &mode);
    s = OpenFileStream(c, fname, mode);
    if (!s) return NULL_VALUE;
    return CsMakeFile(c, s);
  }

  static value CSF_openPipe(VM *c) {
    /*
    value received = NULL_VALUE, proxy = NULL_VALUE;
    CsParseArguments(c,"**V|V",&received, &proxy);
    if( (received != NULL_VALUE) && !CsMethodP(received) )
      CsThrowKnownError(c,CsErrUnexpectedTypeError,received, "either function or
    null" );
    //if( (sent != NULL_VALUE) && !CsMethodP(sent) )
    //  CsThrowKnownError(c,CsErrUnexpectedTypeError,sent, "either function or
    null" ); if( (proxy != NULL_VALUE) && !CsObjectP(proxy) )
      CsThrowKnownError(c,CsErrUnexpectedTypeError,proxy, "either object or
    null" ); async_stream *s = new async_stream(c,received, proxy); if( !s )
      return NULL_VALUE;
    s->add_ref();
    return CsMakeFile(c,s);
    */
    return NULL_VALUE;
  }

  static value CSF_openSocket(VM *c) {
    /*wchar * address;
    int     tout        = 10;
    int     maxattempts = 1;
    stream *s;
    CsParseArguments(c, "**S|i|i", &address, &tout, &maxattempts);
    s = OpenSocketStream(c, address, tout, maxattempts, true);
    if (!s) return NULL_VALUE;
    return CsMakeFile(c, s); */
    return NULL_VALUE;
  }

  static value CSF_openString(VM *c) {
    value   initial = 0;
    stream *s       = 0;
    CsParseArguments(c, "**|V", &initial);
    if (initial) {
      if (CsStringP(initial))
        s = new string_stream_sd(CsStringAddress(initial),
                                 CsStringSize(initial));
      else if (CsIntegerP(initial))
        s = new string_stream_sd(min(0, CsIntegerValue(initial)));
    } else
      s = new string_stream_sd(64);
    if (!s) return NULL_VALUE;
    return CsMakeFile(c, s);
  }

  static value CSF_isInput(VM *c, value obj) {
    stream *s = CsFileStream(obj);
    return s && s->is_input_stream() ? TRUE_VALUE : FALSE_VALUE;
  }
  static value CSF_isOutput(VM *c, value obj) {
    stream *s = CsFileStream(obj);
    return s && s->is_output_stream() ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_isPipe(VM *c, value obj) {
    stream *s = CsFileStream(obj);
    return s && s->is_async_stream() ? TRUE_VALUE : FALSE_VALUE;
  }

  /*static value CSF_pending(VM *c,value obj)
  {
    stream *s = (stream *)CsCObjectValue(obj);
    if(!s || !s->is_async_stream()) return int_value(0);
    async_stream *as = static_cast<async_stream *>(s);
    return int_value(as->pending);
  }

  static value CSF_proxy(VM *c,value obj)
  {
    stream *s = (stream *)CsCObjectValue(obj);
    if(!s || !s->is_async_stream()) return UNDEFINED_VALUE;
    async_stream *as = static_cast<async_stream *>(s);
    if( !as->request_cb.is_alive() )
      return NULL_VALUE;
    if( as->request_cb.pvm == c )
      return as->request_cb;
    else
      return TRUE_VALUE;
  }
  static void CSF_set_proxy(VM *c,value obj,value val)
  {
    stream *s = (stream *)CsCObjectValue(obj);
    if(!s || !s->is_async_stream())
      CsThrowKnownError(c,CsErrUnexpectedTypeError,obj, "is not a pipe stream"
  ); if( (val != NULL_VALUE) && !CsObjectP(val) )
      CsThrowKnownError(c,CsErrUnexpectedTypeError,val, "either object or null"
  ); async_stream *as = static_cast<async_stream *>(s); if(
  as->request_cb.is_alive() && as->request_cb.pvm != c)
      CsThrowKnownError(c,CsErrUnexpectedTypeError,val, "cannot reset remote
  proxy" ); if(val == NULL_VALUE) as->request_cb.unpin(); else
      as->request_cb.pin(c,val);
  }
  */

  static value CSF_encoding(VM *c, value obj) {
    stream *s = (stream *)CsCObjectValue(obj);
    if (!s) return UNDEFINED_VALUE;
    return CsSymbolOf(s->get_encoder()->name());
  }

  static value CSF_name(VM *c, value obj) {
    stream *s = (stream *)CsCObjectValue(obj);
    if (!s) return UNDEFINED_VALUE;
    return string_to_value(c, s->stream_name());
  }

  static void CSF_set_encoding(VM *c, value obj, value val) {
    stream *      s    = (stream *)CsCObjectValue(obj);
    tool::ustring name = value_to_string(val);
    if (name == WCHARS("raw"))
      s->set_encoder(stream::null_encoder());
    else if (name == WCHARS("utf-8"))
      s->set_encoder(stream::utf8_encoder());
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, val,"unsupported encoding");
  }

  /* DestroyFile - destroy a file obj */
  static void DestroyFile(VM *c, value obj) {
    stream *s = (stream *)CsCObjectValue(obj);
    if (s) s->close();
    CsSetCObjectValue(obj, 0);
  }

  /* CSF_Close - built-in method 'Close' */
  static value CSF_close(VM *c) {
    pvalue  val(c);
    stream *s;
    bool    sts;
    bool    return_string = false;
    value   retval        = TRUE_VALUE;

    CsParseArguments(c, "V=*|B", &val.val, c->fileDispatch, &return_string);
    s = (stream *)CsCObjectValue(val);
    if (!s) return FALSE_VALUE;
    if (return_string) {
      if (s->is_string_stream()) {
        string_stream_sd *ps = static_cast<string_stream_sd *>(s);
        retval               = ps->string_o(c);
      } else
        retval = CsMakeCString(c, s->stream_name());
    }
    sts = s->close();
    CsSetCObjectValue(val, 0);
    return sts ? retval : FALSE_VALUE;
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value   val;
    stream *s;
    CsParseArguments(c, "V=*", &val, c->fileDispatch);
    s = (stream *)CsCObjectValue(val);
    if (!s) return UNDEFINED_VALUE;
    if (s->is_string_stream()) {
      string_stream_sd *ps = static_cast<string_stream_sd *>(s);
      return ps->string_o(c);
    }
    return CsMakeCString(c, s->stream_name());
  }

  /* CSF_print - built-in function 'Print' */
  static value CSF_print(VM *c) {
    int_t i;
    CsCheckArgMin(c, 2);
    CsCheckArgType(c, 1, c->fileDispatch);
    stream *s = CsFileStream(CsGetArg(c, 1));
    if (!s) return FALSE_VALUE;
    for (i = 3; i <= CsArgCnt(c); ++i) {
      if (i > 3) s->put_str(" ");
      CsDisplay(c, CsGetArg(c, i), s);
    }
    return TRUE_VALUE;
  }

  /* CSF_println - built-in function 'Display' */
  static value CSF_println(VM *c) {
    int_t i;
    CsCheckArgMin(c, 2);
    CsCheckArgType(c, 1, c->fileDispatch);
    stream *s = CsFileStream(CsGetArg(c, 1));
    if (!s) return FALSE_VALUE;
    for (i = 3; i <= CsArgCnt(c); ++i) {
      if (i > 3) s->put_str(" ");
      CsDisplay(c, CsGetArg(c, i), s);
    }
    s->put('\n');
    s->flush();
    return TRUE_VALUE;
  }

  /* CSF_println - built-in function '$n' */
  static value CSF_$println(VM *c) {
    CsCheckArgType(c, 1, c->fileDispatch);

    stream *s = CsFileStream(CsGetArg(c, 1));
    if (!s) return FALSE_VALUE;

    int i, argc = CsArgCnt(c);
    for (i = 3; i <= argc; ++i)
      CsToString(c, CsGetArg(c, i), *s);

    // r = CSF_print(c);
    return s->put_str("\n") ? TRUE_VALUE : FALSE_VALUE;
  }

  /* CSF_println - built-in function '$' */
  static value CSF_$print(VM *c) {
    CsCheckArgType(c, 1, c->fileDispatch);

    stream *s = CsFileStream(CsGetArg(c, 1));
    if (!s) return FALSE_VALUE;

    int i, argc = CsArgCnt(c);
    for (i = 3; i <= argc; ++i)
      CsToString(c, CsGetArg(c, i), *s);

    return TRUE_VALUE;
  }

  /* CSF_printf - built-in function 'printf' */
  static value CSF_printf(VM *c) {
    CsCheckArgMin(c, 3);
    // bool r = CsFileP(c, CsGetArg(c,1));
    CsCheckArgType(c, 1, c->fileDispatch);

    stream *s = CsFileStream(CsGetArg(c, 1));
    if (!s) return FALSE_VALUE;
    s->printf_args(c);
    return TRUE_VALUE;
  }

  /* CSF_scanf - built-in function 'scanf' */
  static value CSF_scanf(VM *c) {
    stream *s;
    wchar * fmt;
    CsParseArguments(c, "P=*S", &s, c->fileDispatch, &fmt);
    if (!s) return FALSE_VALUE;
    return s->scanf(c, fmt);
  }

  /* CSF_GetC - built-in method 'GetC' */
  static value CSF_getc(VM *c) {
    stream *s;
    int     ch;
    CsParseArguments(c, "P=*", &s, c->fileDispatch);
    if (!s) return UNDEFINED_VALUE;
    if ((ch = s->get()) == stream::EOS) return UNDEFINED_VALUE;
    if (ch == stream::TIMEOUT) CsTimeout(c);
    return CsMakeInteger(ch);
  }

  /* CSF_PutC - built-in method 'PutC' */
  static value CSF_putc(VM *c) {
    stream *s;
    int     ch;
    CsParseArguments(c, "P=*i", &s, c->fileDispatch, &ch);
    if (!s) return FALSE_VALUE;
    return CsMakeBoolean(s->put(ch));
  }

  /* CSF_readln - reads line from the string, returned string does not include
   * \r\n at the end */
  static value CSF_readln(VM *c) {
    stream *s;
    int     ch;
    CsParseArguments(c, "P=*", &s, c->fileDispatch);
    if (!s) return UNDEFINED_VALUE;

    if ((ch = s->get()) == stream::EOS) return UNDEFINED_VALUE;

    if (ch == stream::TIMEOUT) CsTimeout(c);

    tool::array<wchar> buf(10);
    buf.size(0);

    while (ch != stream::EOS) {
      if (ch == '\n') break;
      if (ch == stream::TIMEOUT) CsTimeout(c);
      buf.push(wchar(ch));
      ch = s->get();
    }

    if (buf.size() && buf[0] == 0xFEFF) // BOM handling.
      buf.remove(0);
    if (ch == '\n' && buf.size() && buf.last() == '\r')
      buf.size(buf.size() - 1);

    return CsMakeCharString(c, buf.head(), buf.size());
  }

  /* CSF_send - invokes method on other end of the pipe and waits for returned
   * result */
  /*static value CSF_send(VM *c)
  {
      stream *s;
      value   v = NOTHING_VALUE;
      CsParseArguments(c,"P=*|",&s,c->fileDispatch);
      if(!s || !s->is_async_stream())
        CsThrowKnownError(c,CsErrUnexpectedTypeError,CsGetArg(c,1), "is not a
  pipe stream" ); async_stream* as = static_cast<async_stream*>(s);
      if(as->send(c,v))
        return v;
      v = CsToString(c,v);
      CsThrowKnownError(c,CsErrGenericErrorW,value_to_string(v).c_str());
      //optional_return(v);
  }*/

  /* CSF_post - invokes method on other end of the pipe without waiting */
  /*static value CSF_post(VM *c)
  {
      stream *s;
      CsParseArguments(c,"P=*|",&s,c->fileDispatch);
      if(!s || !s->is_async_stream())
        CsThrowKnownError(c,CsErrUnexpectedTypeError,CsGetArg(c,1), "is not a
  pipe stream" ); async_stream* as = static_cast<async_stream*>(s);
      if(as->post(c))
        return TRUE_VALUE;
      CsThrowKnownError(c,CsErrGenericErrorW,"failure of remote side");
      //optional_return(FALSE_VALUE);
  }*/

} // namespace tis
