/* fcn.c - built-in functions */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits>
#include "cs.h"

namespace tis {

  /* prototypes */
  static value CSF_type(VM *c);
  static value CSF_hash(VM *c);
  // static value CSF_toString(VM *c);
  static value CSF_rand(VM *c);
  value CSF_gc(VM *c);
  static value CSF_LoadObjectFile(VM *c);
  static value CSF_quit(VM *c);
  static value CSF_symbol(VM *c);
  static value CSF_missed(VM *c);
  static value CSF_dumpScopes(VM *c);

  static value CSF_toInteger(VM *c);
  static value CSF_toFloat(VM *c);
  static value CSF_URL_parse(VM *c);
  static value CSF_membersOf(VM *c);

  static value CSF_$url(VM *c);

  value CSF_em(VM *c);
  value CSF_rem(VM *c);
  value CSF_ex(VM *c);
  value CSF_ch(VM *c);
  value CSF_pr(VM *c);
  value CSF_fx(VM *c);
  value CSF_px(VM *c);
  value CSF_in(VM *c);
  value CSF_cm(VM *c);
  value CSF_mm(VM *c);
  value CSF_pt(VM *c);
  value CSF_pc(VM *c);
  value CSF_dip(VM *c);
  value CSF_ppx(VM *c);
  value CSF_color(VM *c);
  value CSF_make_length(VM *c);
  // value CSF_handheld(VM *c);
  value CSF_deg(VM *c);
  value CSF_rad(VM *c);

  extern dispatch color_method_dispatch;

  /* function table */
  static c_method functionTable[] = {
      // C_METHOD_ENTRY( "type",             CSF_type            ),
      C_METHOD_ENTRY("hash", CSF_hash), C_METHOD_ENTRY("symbol", CSF_symbol),
      // C_METHOD_ENTRY( "toString",         CSF_toString        ),
      C_METHOD_ENTRY("rand", CSF_rand), C_METHOD_ENTRY("gc", CSF_gc),
      C_METHOD_ENTRY("toInteger", CSF_toInteger),
      C_METHOD_ENTRY("toFloat", CSF_toFloat),
      C_METHOD_ENTRY("dumpScopes", CSF_dumpScopes),
      C_METHOD_ENTRY("missed", CSF_missed),
      C_METHOD_ENTRY("crackUrl", CSF_URL_parse),
      //C_METHOD_ENTRY("pathToUrl", CSF_pathToUrl),
      C_METHOD_ENTRY("membersOf", CSF_membersOf),

      C_METHOD_ENTRY("px", CSF_px), C_METHOD_ENTRY("pt", CSF_pt),
      C_METHOD_ENTRY("pc", CSF_pc), C_METHOD_ENTRY("in", CSF_in),
      C_METHOD_ENTRY("em", CSF_em), C_METHOD_ENTRY("ex", CSF_ex), C_METHOD_ENTRY("ch", CSF_ch),
      C_METHOD_ENTRY("pr", CSF_pr), C_METHOD_ENTRY("fx", CSF_fx),
      C_METHOD_ENTRY("mm", CSF_mm), C_METHOD_ENTRY("cm", CSF_cm),
      C_METHOD_ENTRY("rem", CSF_rem), C_METHOD_ENTRY("dip", CSF_dip), 
      C_METHOD_ENTRY("ppx", CSF_ppx),

      // special case to support color #ffa; constructs
      c_method("color", CSF_color, &color_method_dispatch),

      C_METHOD_ENTRY("rgb", CSF_color), C_METHOD_ENTRY("rgba", CSF_color),
      C_METHOD_ENTRY("length", CSF_make_length), C_METHOD_ENTRY("rad", CSF_rad),
      C_METHOD_ENTRY("deg", CSF_deg),

      C_METHOD_ENTRY("$url", CSF_$url),

      // C_METHOD_ENTRY( "quit",             CSF_quit            ),
      C_METHOD_ENTRY(0, 0)};

  value CSF_JSON_stringify(VM *c);
  value CSF_JSON_parse(VM *c);

  static c_method JSON_methods[] = {
      C_METHOD_ENTRY("parse", CSF_JSON_parse),
      C_METHOD_ENTRY("stringify", CSF_JSON_stringify), C_METHOD_ENTRY(0, 0)};

  static value CSF_URL_relative(VM *c);
  static value CSF_URL_absolute(VM *c);
  static value CSF_URL_encode(VM *c);
  static value CSF_URL_encodeParam(VM *c);
  static value CSF_URL_decode(VM *c);
  static value CSF_URL_toPath(VM *c);
  static value CSF_URL_fromPath(VM *c);

  static c_method URL_methods[] = {
      C_METHOD_ENTRY("parse", CSF_URL_parse),
      C_METHOD_ENTRY("relative", CSF_URL_relative),
      C_METHOD_ENTRY("absolute", CSF_URL_absolute),
      C_METHOD_ENTRY("encode", CSF_URL_encode),
      C_METHOD_ENTRY("encodeParam", CSF_URL_encodeParam),
      C_METHOD_ENTRY("decode", CSF_URL_decode),
      C_METHOD_ENTRY("toPath", CSF_URL_toPath),
      C_METHOD_ENTRY("fromPath", CSF_URL_fromPath),
      C_METHOD_ENTRY(0, 0)};

  /* CsEnterLibrarySymbols - enter the built-in functions and symbols */
  void CsEnterLibrarySymbols(VM *c) {
    CsEnterFunctions(CsGlobalScope(c), functionTable);
    value JSON = CsNewNamespaceInstance(c, UNDEFINED_VALUE, CsSymbolOf("JSON"));
    CsAddConstant(c, CsGlobalScope(c)->globals, CsSymbolOf("JSON"), JSON);
    CsEnterMethods(c, JSON, JSON_methods);
    value URL = CsNewNamespaceInstance(c, UNDEFINED_VALUE, CsSymbolOf("URL"));
    CsAddConstant(c, CsGlobalScope(c)->globals, CsSymbolOf("URL"), URL);
    CsEnterMethods(c, URL, URL_methods);
  }

#if 0
  /* CSF_type - built-in function 'Type' */
  static value CSF_type(VM *c) {
    CsCheckArgCnt(c, 3);
    return CsSymbolOf(CsGetDispatch(CsGetArg(c, 3))->typeName);
  }
#endif

  /* CSF_Hash - built-in function 'Hash' */
  static value CSF_hash(VM *c) {
    CsCheckArgCnt(c, 3);
    return CsMakeInteger(CsHashValue(CsGetArg(c, 3)));
  }

  static value CSF_symbol(VM *c) {
    CsCheckArgCnt(c, 3);
    value v = CsGetArg(c, 3);
    if (CsStringP(v)) return CsIntern(c, v);
    if (CsSymbolP(v)) return v;
    CsTypeError(c, v);
    return UNDEFINED_VALUE;
  }

  static value CSF_missed(VM *c) {
    CsCheckArgCnt(c, 3);
    value v = CsGetArg(c, 3);
    return (v == NOTHING_VALUE) ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_dumpScopes(VM *c) {
    CsDumpScopes(c);
    return UNDEFINED_VALUE;
  }

  static value CSF_$url(VM *c) {
    string_stream s(20);
    int           i, argc = CsArgCnt(c);
    for (i = 3; i <= argc; ++i)
      if (i & 1)
        CsToString(c, CsGetArg(c, i), s); // as is
      else
        CsToUrlString(c, CsGetArg(c, i), s); // escape
    tool::ustring rel = s.to_ustring();
    s.close();
        
    tool::ustring base = value_to_string(CsCompiledCodeFileName(c->code));
    tool::url u(rel);
    tool::url b(base);

    u.absolute(b);

    return string_to_value(c, u.compose());
  }


#if 0
  /* CSF_toString - built-in function 'toString'
  static value CSF_toString(VM *c) {

    int   width,maxWidth,ch;
    bool  rightP;
    CsStringOutputStream s;
    char buf[1024],*start;

    /* check the argument count */
    CsCheckArgRange(c,3,5);

    /* get the field width and fill character */
    switch (CsArgCnt(c)) {
    case 3:
        width = 0;
        ch = ' ';
        break;
    case 4:
        CsCheckType(c,4,CsIntegerP);
        width = (int)CsIntegerValue(CsGetArg(c,4));
        ch = ' ';
        break;
    case 5:
        CsCheckType(c,4,CsIntegerP);
        CsCheckType(c,5,CsIntegerP);
        width = (int)CsIntegerValue(CsGetArg(c,4));
        ch = (int)CsIntegerValue(CsGetArg(c,5));
        break;
    default:
        /* never reached */
        width = 0;
        ch = 0;
        break;
    }

    /* check for right fill */
    if (width <= 0) {
        width = -width;
        rightP = true;
        maxWidth = sizeof(buf);
        start = buf;
    }
    else {
        rightP = false;
        maxWidth = sizeof(buf) / 2;
        start = buf + maxWidth;
    }

    /* make sure the width is within range */
    if (width > maxWidth)
        width = maxWidth;

    /* print the value to the buffer */
    CsInitStringOutputStream(c,&s,start,maxWidth);
    CsDisplay(c,CsGetArg(c,3),(CsStream *)&s);

    /* fill if necessary */
    if (width > 0 && s.len < width) {
        int fill = width - s.len;
        if (rightP) {
            while (--fill >= 0)
                *s.ptr++ = ch;
        }
        else {
            while (--fill >= 0)
                *--start = ch;
        }
        s.len = width;
    }

    /* return the resulting string */
    return CsMakeString(c,start,s.len);
  }
#endif

  /* CsToString - coerce the val to string */
  value CsToString(VM *c, value val) {
    if (CsStringP(val)) return val;
    if (CsSymbolP(val)) return CsMakeString(c, CsSymbolName(val));

    /*value r;
    value obj = val;
    if(CsGetProperty1(c, obj, TO_STRING_SYM, &r) && (CsMethodP(r) ||
    CsCMethodP(r)))
    {
      //TRY
      //{
      return CsSendMessage(CsCurrentScope(c),val,r, 0);
      //}
      //CATCH_ERROR(e) {}
    }*/

    string_stream s(64);

    CsDisplay(c, val, &s);

    value r = s.string_o(c);

    s.close();

    // return the resulting string
    return r;
  }

  void CsToString(VM *c, value val, stream &s) {
    if (CsStringP(val)) {
      s.put_str(CsStringAddress(val));
      return;
    }
    if (CsSymbolP(val)) {
      s.put_str(CsSymbolName(val));
      return;
    }

      /*value r;
      value obj = val;
      if(CsGetProperty1(c,obj,TO_STRING_SYM, &r) && (CsMethodP(r) ||
      CsCMethodP(r)))
      {
        val = CsSendMessage(c,val,r);
        if(CsStringP(val))
        {
          s.put_str(CsStringAddress(val));
          return;
        }
      }*/
#ifdef _DEBUG
    dispatch *pd = CsGetDispatch(val);
    pd           = pd;
#endif

    CsDisplay(c, val, &s);
  }

  void CsToHtmlString(VM *c, value val, stream &s) {
    value r;
    value obj = val;

    static value TO_HTML_STRING = 0;
    if (!TO_HTML_STRING) TO_HTML_STRING = CsSymbolOf("toHtmlString");

    if (!CsGetProperty1(c, obj, TO_HTML_STRING, &r)) {
      obj = val;
      if (!CsGetProperty1(c, obj, TO_STRING_SYM, &r))
        CsThrowKnownError(c, CsErrNoSuchFeature, val, "toHtmlString() method");
    }
    if (CsMethodP(r) || CsCMethodP(r)) {
      val = CsSendMessage(c, val, r);
      if (CsStringP(val)) {
        s.put_str(CsStringAddress(val));
        return;
      }
    } else
      CsThrowKnownError(c, CsErrNoSuchFeature, val, "toHtmlString() method");
  }

  void CsToCssString(VM *c, value val, stream &s) {
    value r;
    value obj = val;

    static value TO_CSS_STRING = 0;
    if (!TO_CSS_STRING) TO_CSS_STRING = CsSymbolOf("toCssString");

    if (!CsGetProperty1(c, obj, TO_CSS_STRING, &r)) {
      obj = val;
      if (!CsGetProperty1(c, obj, TO_STRING_SYM, &r))
        CsThrowKnownError(c, CsErrNoSuchFeature, val, "toCssString() method");
    }
    if (CsAnyMethodP(r)) {
      val = CsSendMessage(c, val, r);
      if (CsStringP(val)) {
        s.put_str(CsStringAddress(val));
        return;
      }
    }
    else
      CsThrowKnownError(c, CsErrNoSuchFeature, val, "toCssString() method");
  }


  void CsToUrlString(VM *c, value val, stream &s) {
    value r;
    value obj = val;

    static value TO_CSS_STRING = CsSymbolOf("toUrlString");

    if (!CsGetProperty1(c, obj, TO_CSS_STRING, &r)) {
      obj = val;
      if (!CsGetProperty1(c, obj, TO_STRING_SYM, &r))
        CsThrowKnownError(c, CsErrNoSuchFeature, val, "toUrlString() method");
    }
    if (CsAnyMethodP(r)) {
      val = CsSendMessage(c, val, r);
      if (CsStringP(val)) {
        s.put_str(CsStringAddress(val));
        return;
      }
    }
    else
      CsThrowKnownError(c, CsErrNoSuchFeature, val, "toUrlString() method");
  }


  /* CSF_rand - built-in function 'rand' */
  static value CSF_rand(VM *c) {
    static time_t rseed = time(0);
    time_t        k1, i;

    /* parse the arguments */
    CsCheckArgCnt(c, 3);
    CsCheckType(c, 3, CsIntegerP);
    i = (long)CsIntegerValue(CsGetArg(c, 3));

    /* make sure we don't get stuck at zero */
    if (rseed == 0) rseed = 1;

    /* algorithm taken from Dr. Dobbs Journal, November 1985, page 91 */
    k1 = rseed / 127773L;
    if ((rseed = 16807L * (rseed - k1 * 127773L) - k1 * 2836L) < 0L)
      rseed += 2147483647L;

    /* return a random number between 0 and n-1 */
    return CsMakeInteger((int_t)(i ? (rseed % i) : 0));
  }

  value CsToInteger(VM *c, value v, bool throw_error) {
    int_t i = 0;
    // wchar *pend;
    // START:
    if (CsIntegerP(v))
      return v;
    else if (CsFloatP(v))
      i = (int_t)CsFloatValue(v);
    else if (v == TRUE_VALUE)
      return CsMakeInteger(1);
    else if (v == FALSE_VALUE)
      return CsMakeInteger(0);
    else if (v == UNDEFINED_VALUE || v == NULL_VALUE)
      return CsMakeInteger(0);
    else if (CsStringP(v)) {
      int_t  n   = 0;
      wchars str = CsStringChars(v);
      if (parse_int(str, n)) return CsMakeInteger(n);
    }
    if (throw_error)
      CsThrowKnownError(c, CsErrUnexpectedTypeError, v, "number");

    /*else if( CsObjectP(v) )
    {
      value r;
      value obj = v;
      if(CsGetProperty1(c, obj, VALUE_OF_SYM, &r) && (CsMethodP(r) ||
    CsCMethodP(r)))
      {
        v = CsSendMessage(CsCurrentScope(c),v,r, 0);
        if( !CsObjectP(v) )
          goto START;
      }
    }*/
    return NULL_VALUE; // CsMakeInteger(i);
  }

  value CsToFloat(VM *c, value v, bool throw_error) {
    // wchar *pend = 0;
    // START:
    if (CsFloatP(v))
      return v;
    else if (CsIntegerP(v))
      return CsMakeFloat((float_t)CsIntegerValue(v));
    else if (v == TRUE_VALUE)
      return CsMakeFloat(1);
    else if (v == FALSE_VALUE)
      return CsMakeFloat(0);
    else if (v == UNDEFINED_VALUE || v == NULL_VALUE)
      return CsMakeInteger(0);
    else if (CsStringP(v)) {
      float_t n   = 0;
      wchars  str = CsStringChars(v);
      if (parse_real(str, n)) return CsMakeFloat(n);
    }
    /*else if( CsObjectP(v) )
    {
      value r;
      value obj = v;
      if(CsGetProperty1(c, obj, VALUE_OF_SYM, &r) && (CsMethodP(r) ||
    CsCMethodP(r)))
      {
        v = CsSendMessage(CsCurrentScope(c),v,r, 0);
        if( !CsObjectP(v) )
          goto START;
      }
    }*/
    if (throw_error)
      CsThrowKnownError(c, CsErrUnexpectedTypeError, v, "number");

    return CsMakeFloat(std::numeric_limits<double>::quiet_NaN());
  }

  static value CSF_toInteger(VM *c) {
    value dv = CsMakeInteger(0);
    value v;
    CsParseArguments(c, "**V|V", &v, &dv);
    if (CsIntegerP(v))
      return v;
    else if (CsFloatP(v))
      return CsMakeInteger((int)CsFloatValue(v));
    else if (v == TRUE_VALUE)
      return CsMakeInteger(1);
    else if (v == FALSE_VALUE)
      return CsMakeInteger(0);
    else if (v == UNDEFINED_VALUE || v == NULL_VALUE)
      return CsMakeInteger(0);
    else if (CsStringP(v)) {
      ustring text = value_to_string(v);
      int_t   n    = 0;
      if (parse_int(text(), n)) return CsMakeInteger(n);
      // wchar *pend;
      // int_t n = wcstol(CsStringAddress(v),&pend,0);
      // if( CsStringAddress(v) != pend )
      //  return CsMakeInteger(n);
    }
    return dv;
  }

  static value CSF_toFloat(VM *c) {
    value v, dv = CsMakeFloat(0.0);
    // wchar *pend;
    CsParseArguments(c, "**V|V", &v, &dv);
    if (CsIntegerP(v))
      return CsMakeFloat(CsIntegerValue(v));
    else if (CsFloatP(v))
      return v;
    else if (v == TRUE_VALUE)
      return CsMakeFloat(1.0);
    else if (v == FALSE_VALUE)
      return CsMakeFloat(0.0);
    else if (v == UNDEFINED_VALUE || v == NULL_VALUE)
      return CsMakeFloat(0.0);
    else if (CsStringP(v)) {
      float_t n   = 0.0;
      wchars  str = CsStringChars(v);
      if (parse_real(str, n)) return CsMakeFloat(n);
      // float_t n = wcstod(CsStringAddress(v),&pend);
      // if( CsStringAddress(v) != pend )
      //  return CsMakeFloat(n);
    }
    return dv;
  }

  /* CSF_gc - built-in function 'gc' */
  value CSF_gc(VM *c) {

    int_t factor = -1;

    CsParseArguments(c, "**|i", &factor);
    
    auto r = factor >= 0 ? CsCollectGarbageAndReclaim(c,factor) : CsCollectGarbage(c);
    CS_RETURN3(c, CsMakeInteger(r.total), CsMakeInteger(r.free), CsMakeInteger(r.used));
  }

  /* CSF_quit - built-in function 'Quit' */
  static value CSF_quit(VM *c) {
    CsCheckArgCnt(c, 2);
    CsThrowKnownError(c, CsErrExit, 0);
    return 0; /* never reached */
  }

  value CsTypeOf(VM *c, value val) {
    // char *str = "undefined";
    dispatch *d;

    if (val == UNDEFINED_VALUE) return symbol_value(S_UNDEFINED);
    if (val == NOTHING_VALUE) return symbol_value(S_NOTHING);

    if (val == TRUE_VALUE || val == FALSE_VALUE) return symbol_value(S_BOOLEAN);

    if (val == NULL_VALUE) return symbol_value(S_OBJECT); // #11.4.3

    d = CsGetDispatch(val);
    if (d == &CsIntegerDispatch) return symbol_value(S_INTEGER);
    if (d == &CsFloatDispatch) return symbol_value(S_FLOAT);
    if (d == &CsStringDispatch) return symbol_value(S_STRING);
    if (d == &CsVectorDispatch) return symbol_value(S_ARRAY);
    if (d == &CsObjectDispatch || d == &CsCObjectDispatch) return symbol_value(S_OBJECT);
    if (d == &CsSymbolDispatch) return symbol_value(S_SYMBOL);
    if (d == &CsMethodDispatch || d == &CsCMethodDispatch || d->baseType == &CsCMethodDispatch) return symbol_value(S_FUNCTION);
    if (d == c->dateDispatch) return symbol_value(S_DATE);
    if (d == &CsColorDispatch) return symbol_value(S_COLOR);
    if (d == &CsLengthDispatch) return symbol_value(S_LENGTH);
    if (d == &CsClassDispatch) return symbol_value(S_CLASS);
    if (d == &CsNamespaceDispatch) return symbol_value(S_NAMESPACE);
    if (d == &CsTupleDispatch) return symbol_value(S_TUPLE);
    if (d == &CsAngleDispatch) return symbol_value(S_ANGLE);
    if (d == &CsDurationDispatch) return symbol_value(S_DURATION);
    return CsSymbolOf(d->typeName);
  }

  static value CSF_crackUrl(VM *c) {
    wchar *str = 0;
    int    len = 0;
    CsParseArguments(c, "**S#", &str, &len);
    if (str && len) {
      tool::url u;
      if (u.parse(str)) {
        pvalue pobj(c, CsMakeObject(c, c->objectObject));
        pvalue pkey(c);
        pvalue pval(c);

        pkey = CsSymbolOf("port");
        pval = CsMakeInteger(u.port);
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("protocol");
        pval = CsMakeCString(c, tool::url::unescape(u.protocol));
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("hostname");
        pval = CsMakeCString(c, tool::url::unescape(u.hostname));
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("anchor");
        pval = CsMakeCString(c, tool::url::unescape(u.anchor));
        CsObjectSetItem(c, pobj, pkey, pval);

        if (!u.is_local()) {
          pkey = CsSymbolOf("username");
          pval = CsMakeCString(c, tool::url::unescape(u.username));
          CsObjectSetItem(c, pobj, pkey, pval);

          pkey = CsSymbolOf("password");
          pval = CsMakeCString(c, tool::url::unescape(u.password));
          CsObjectSetItem(c, pobj, pkey, pval);
        }

        pkey = CsSymbolOf("params");
        pval = CsMakeCString(c, tool::url::unescape(u.params));
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring dir = tool::url::unescape(u.dir());
        pkey              = CsSymbolOf("dir");
        pval              = CsMakeCString(c, dir);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring name = tool::url::unescape(u.name());
        pkey               = CsSymbolOf("name");
        pval               = CsMakeCString(c, name);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring ext = tool::url::unescape(u.ext());
        pkey              = CsSymbolOf("ext");
        pval              = CsMakeCString(c, ext);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring name_ext = tool::url::unescape(u.name_ext());
        pkey                   = CsSymbolOf("name_ext");
        pval                   = CsMakeCString(c, name_ext);
        CsObjectSetItem(c, pobj, pkey, pval);
        return pobj;
      }
    }
    return NULL_VALUE;
  }

  static value CSF_URL_parse(VM *c) {
    wchar *str = 0;
    int    len = 0;
    CsParseArguments(c, "**S#", &str, &len);
    if (str && len) {
      tool::url u;
      if (u.parse(str)) {
        value pobj = CsMakeObject(c, c->objectObject);
        value pkey = 0;
        value pval = 0;
        PROTECT(pobj, pkey, pval);

        pkey = CsSymbolOf("port");
        pval = CsMakeInteger(u.port);
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("protocol");
        pval = CsMakeCString(c, tool::url::unescape(u.protocol));
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("hostname");
        pval = CsMakeCString(c, tool::url::unescape(u.hostname));
        CsObjectSetItem(c, pobj, pkey, pval);

        pkey = CsSymbolOf("anchor");
        pval = CsMakeCString(c, tool::url::unescape(u.anchor));
        CsObjectSetItem(c, pobj, pkey, pval);

        if (!u.is_local()) {
          pkey = CsSymbolOf("username");
          pval = CsMakeCString(c, tool::url::unescape(u.username));
          CsObjectSetItem(c, pobj, pkey, pval);

          pkey = CsSymbolOf("password");
          pval = CsMakeCString(c, tool::url::unescape(u.password));
          CsObjectSetItem(c, pobj, pkey, pval);
        }

        pkey = CsSymbolOf("params");
        pval = CsMakeCString(c, tool::url::unescape(u.params));
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring dir = tool::url::unescape(u.dir());
        pkey              = CsSymbolOf("dir");
        pval              = CsMakeCString(c, dir);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring name = tool::url::unescape(u.name());
        pkey               = CsSymbolOf("name");
        pval               = CsMakeCString(c, name);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring ext = tool::url::unescape(u.ext());
        pkey              = CsSymbolOf("ext");
        pval              = CsMakeCString(c, ext);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring name_ext = tool::url::unescape(u.name_ext());
        pkey                   = CsSymbolOf("name_ext");
        pval                   = CsMakeCString(c, name_ext);
        CsObjectSetItem(c, pobj, pkey, pval);

        tool::ustring fn = tool::url::unescape(u.filename);
        pkey = CsSymbolOf("filename");
        pval = CsMakeCString(c, fn);
        CsObjectSetItem(c, pobj, pkey, pval);


        return pobj;
      }
    }
    return NULL_VALUE;
  }

  static value CSF_membersOf(VM *c) {
    CsCheckArgCnt(c, 3);
    CsCheckType(c, 3, CsObjectOrMethodP);
    pvalue pobj(c, CsMakeObject(c, UNDEFINED_VALUE));

    CsSetObjectProperties(pobj, CsObjectProperties(CsGetArg(c, 3)));

    /*each_property gen(c, CsGetArg(c,3));
    

    for(value key,val; gen(key,val); )
      CsObjectSetItem(c,pobj,key,val);*/

    return pobj;
  }

  static value CSF_URL_relative(VM *c) {
    wchars what;
    wchars base;
    CsParseArguments(c, "**S#S#", &what.start, &what.length, &base.start,
                     &base.length);
    tool::url uw;
    uw.parse(string(what));
    tool::url ub;
    ub.parse(string(base));
    tool::ustring r = ub.relative(uw);
    return CsMakeString(c, r);
  }
  static value CSF_URL_absolute(VM *c) {
    wchars what;
    wchars base;
    CsParseArguments(c, "**S#S#", &what.start, &what.length, &base.start,
                     &base.length);
    tool::url uw;
    uw.parse(string(what));
    tool::url ub;
    ub.parse(string(base));
    uw.absolute(ub);
    return CsMakeString(c, uw.compose());
  }
  static value CSF_URL_encode(VM *c) {
    wchars what;
    CsParseArguments(c, "**S#", &what.start, &what.length);
    return CsMakeString(c, tool::url::escape(what));
  }
  static value CSF_URL_encodeParam(VM *c) {
    wchars what;
    CsParseArguments(c, "**S#", &what.start, &what.length);
    return CsMakeString(c, tool::url::escape_param(what));
  }

  static value CSF_URL_decode(VM *c) {
    wchars what;
    CsParseArguments(c, "**S#", &what.start, &what.length);
    return CsMakeString(c, tool::url::unescape(what.start));
  }

  static value CSF_URL_toPath(VM *c) {
    wchars what;
    CsParseArguments(c, "**S#", &what.start, &what.length);
    return CsMakeString(c, tool::url::file_url_to_path(what));
  }

  static value CSF_URL_fromPath(VM *c) {
    wchars what;
    CsParseArguments(c, "**S#", &what.start, &what.length);
    return CsMakeString(c, tool::url::path_to_file_url(what));
  }



} // namespace tis
