/* env.c - environment types */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"

namespace tis {

  /* ENVIRONMENT */

  /* Environment pdispatch */
  dispatch CsEnvironmentDispatch = {"Environment",
                                    &CsEnvironmentDispatch,
                                    CsDefaultGetProperty,
                                    CsDefaultSetProperty,
                                    CsDefaultNewInstance,
                                    CsDefaultPrint,
                                    CsTupleSizeHandler,
                                    CsDefaultCopy,
                                    CsTupleScanHandler,
                                    CsDefaultHash,
                                    CsDefaultGetItem,
                                    CsDefaultSetItem

  };

  /* CsMakeEnvironment - make an environment value */
  value CsMakeEnvironment(VM *c, long size) {
    return CsMakeTuple(c, &CsEnvironmentDispatch, size);
  }

  /* STACK ENVIRONMENT */

  /* StackEnvironment handlers */
  static value StackEnvironmentCopy(VM *c, value obj);

  /* StackEnvironment pdispatch */
  dispatch CsStackEnvironmentDispatch = {
      "StackEnvironment",   &CsEnvironmentDispatch, CsDefaultGetProperty,
      CsDefaultSetProperty, CsDefaultNewInstance,   CsDefaultPrint,
      CsTupleSizeHandler,   StackEnvironmentCopy,   CsTupleScanHandler,
      CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem};

  /* StackEnvironmentCopy - StackEnvironment copy handler */
  static value StackEnvironmentCopy(VM *c, value obj) {
    /* copying the stack will copy the elements */
    return obj;
  }

  /* MOVED ENVIRONMENT */

  /* MovedEnvironment handlers */
  static value MovedEnvironmentCopy(VM *c, value obj);

  /* MovedEnvironment pdispatch */
  dispatch CsMovedEnvironmentDispatch = {
      "MovedEnvironment",   &CsEnvironmentDispatch, CsDefaultGetProperty,
      CsDefaultSetProperty, CsDefaultNewInstance,   CsDefaultPrint,
      CsTupleSizeHandler,   MovedEnvironmentCopy,   CsTupleScanHandler,
      CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem};

  /* MovedEnvironmentCopy - MovedEnvironment copy handler */
  static value MovedEnvironmentCopy(VM *c, value obj) {
    return CsCopyValue(c, CsMovedEnvForwardingAddr(obj));
  }

  ///////////////////////////////////////////////////////
  ///////////////////////////////////////////////////////

  value CSF_fulfillTask(VM *c) {
    value method = *c->argv;
    assert(CsCMethodP(method));
    value taskInstance = CsCMethodPayload(method);
    assert(CsTaskP(c, taskInstance));
    value retval = TRUE_VALUE;
    int argc = CsArgCnt(c);
    if (argc > 3) {
      PROTECT(taskInstance);
      retval = CsMakeVector(c,argc - 2);
      for (int n = 3; n <= argc; ++n)
        CsSetVectorElementI(retval, n - 3, CsGetArg(c, n));
    } else 
      CsParseArguments(c, "**|V", &retval);
    c->val = retval;
    CsExecTask(c, taskInstance, false);
    return NULL_VALUE;
  }

  value CSF_rejectTask(VM *c) {
    value method = *c->argv;
    assert(CsCMethodP(method));
    value taskInstance = CsCMethodPayload(method);
    assert(CsTaskP(c, taskInstance));
    value retval = TRUE_VALUE;
    int argc = CsArgCnt(c);
    if (argc > 3) {
      PROTECT(taskInstance);
      retval = CsMakeVector(c, argc - 2);
      for (int n = 3; n <= argc; ++n)
        CsSetVectorElementI(retval, n - 3, CsGetArg(c, n));
    }
    else
      CsParseArguments(c, "**|V", &retval);
    c->val = retval;
    CsExecTask(c, taskInstance, true);
    return NULL_VALUE;
  }

  bool CsAttachTaskToPromise(VM *c, value promise, value taskInstance) {
    static value symThen  = CsSymbolOf("then");
    value        thenFunc = 0;

    value okFunc   = 0;
    value failFunc = 0;

    if (value_to_set(promise) == UNDEFINED_VALUE) return false;

    PROTECT(thenFunc, promise, taskInstance, okFunc, failFunc);

    if (!CsGetProperty(c, promise, symThen, &thenFunc)) return false;

    if (!CsAnyMethodP(thenFunc)) return false;

    // looks like a promise

    okFunc   = CsMakeCMethod(c, "taskFulfill", CSF_fulfillTask);
    failFunc = CsMakeCMethod(c, "taskReject", CSF_rejectTask);

    CsSetCMethodPayload(okFunc, taskInstance);
    CsSetCMethodPayload(failFunc, taskInstance);

#ifdef _DEBUG
    // dispatch* pd = CsGetDispatch(promise);
#endif

    CsSendMessage(c, promise, thenFunc, 2, okFunc, failFunc);

    return true;
  }

  //////////////////////////////////////

  // CsTask

  //////////////////////////////////////

  static value CSF_task_then(VM *c) {
    value obj;
    value onFulfilled, onRejected = NULL_VALUE;
    // wchar *pend;
    CsParseArguments(c, "V=*V|V", &obj, c->coroutineDispatch, &onFulfilled,
                     &onRejected);
    if (!CsTaskP(c, obj)) CsUnexpectedTypeError(c, obj, "Task");

    CsTask *exec = CsCoroutineValue(c, obj);

    if (!CsAnyMethodP(onFulfilled)) onFulfilled = NULL_VALUE;
    if (!CsAnyMethodP(onRejected)) onRejected = NULL_VALUE;

    PROTECT(obj, onFulfilled, onRejected);
    exec->callbacks =
        CsMakeTuple(c, "sub", onFulfilled, onRejected, exec->callbacks);
    return obj;
  }

  static value CSF_task_catch(VM *c) {
    value obj;
    value onRejected = NULL_VALUE;
    // wchar *pend;
    CsParseArguments(c, "V=*V", &obj, c->coroutineDispatch, &onRejected);
    if (!CsTaskP(c, obj)) CsUnexpectedTypeError(c, obj, "Task");

    CsTask *exec = CsCoroutineValue(c, obj);

    if (!CsAnyMethodP(onRejected)) onRejected = NULL_VALUE;

    PROTECT(obj, onRejected);
    exec->callbacks =
        CsMakeTuple(c, "sub", NULL_VALUE, onRejected, exec->callbacks);
    return obj;
  }

  static value CSF_task_finally(VM *c) {
    value obj;
    value onFinally;
    // wchar *pend;
    CsParseArguments(c, "V=*V", &obj, c->coroutineDispatch, &onFinally);
    if (!CsTaskP(c, obj)) CsUnexpectedTypeError(c, obj, "Task");

    CsTask *exec = CsCoroutineValue(c, obj);

    if (!CsAnyMethodP(onFinally)) onFinally = NULL_VALUE;

    PROTECT(obj, onFinally);
    exec->callbacks =
        CsMakeTuple(c, "sub", onFinally, onFinally, exec->callbacks);
    return obj;
  }

  static value CSF_task_next(VM *c) {
    value obj;
    value nextVal = 0;
    // wchar *pend;
    CsParseArguments(c, "V=*V|V", &obj, c->coroutineDispatch, &nextVal);

    if (!CsGeneratorP(c, obj)) 
      return NOTHING_VALUE;

    c->val = nextVal ? nextVal : UNDEFINED_VALUE;

    return CsExecGenerator(c, obj);
  }

  static value CSF_task_isGenerator(VM *c, value obj) {
    return CsGeneratorP(c, obj) ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_task_isAsync(VM *c, value obj) {
    return CsTaskP(c, obj) ? TRUE_VALUE : FALSE_VALUE;
  }

  static value CSF_task_function(VM *c, value obj) {
    if (!CsTaskP(c, obj)) return NULL_VALUE;
    CsTask *exec = CsCoroutineValue(c, obj);
    if(!exec) return NULL_VALUE;
    return exec->method;
  }

  void CsTaskNotifyCompletion(VM *c, bool reject, value list) {
    value onFulfilled = NULL_VALUE;
    value onRejected  = NULL_VALUE;

    PROTECT(list, onFulfilled, onRejected);

    while (CsTupleP(list)) {
      onFulfilled = CsTupleElement(list, 0);
      onRejected  = CsTupleElement(list, 1);
      list        = CsTupleElement(list, 2);
      value func  = reject ? onRejected : onFulfilled;
      if (CsAnyMethodP(func)) 
        CsCallCompletion(c, func, c->val);
    }
  }

  /* Task methods */
  static c_method task_instance_methods[] = {
      C_METHOD_ENTRY("then", CSF_task_then),
      C_METHOD_ENTRY("catch", CSF_task_catch),
      C_METHOD_ENTRY("finally", CSF_task_finally),
      C_METHOD_ENTRY("next", CSF_task_next), C_METHOD_ENTRY(0, 0)};

  /* Task properties */
  static vp_method task_instance_properties[] = {
      VP_METHOD_ENTRY("isAsync", CSF_task_isAsync, 0),
      VP_METHOD_ENTRY("isGenerator", CSF_task_isGenerator, 0),
      VP_METHOD_ENTRY("function", CSF_task_function, 0),
      VP_METHOD_ENTRY(0, 0, 0)};

  static void CoroutineScan(VM *c, value obj) {
    CsTask *exec = CsCoroutineValue(c, obj);
    if (exec) {
      exec->scan(c);
      exec->callbacks = CsCopyValue(c, exec->callbacks);
      exec->ns        = CsCopyValue(c, exec->ns);
      exec->method    = CsCopyValue(c, exec->method);
      //exec->stackPlaceholder = CsCopyValue(c, exec->stackPlaceholder);
      if (exec->current) {
        CsSavedState that(c);
        exec->current->restore();
        CsCopyStack(c);
        that.restore();
      }
    }
    CsCObjectScan(c, obj);
  }

  void CoroutineDestroy(VM *c, value obj) {
    CsTask *exec = CsCoroutineValue(c, obj);
    
    int pidx = c->pending_tasks.get_index(obj);
    if( pidx >= 0) c->pending_tasks[pidx] = 0;
    c->active_tasks.remove_by_value(obj);
    CsSetCObjectValue(obj, 0);
    delete exec;
  }

  static value CoroutineNextElement(VM *c, value *index, value obj, int nr) {
    if (!CsGeneratorP(c, obj))
      //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
      return NOTHING_VALUE;
    c->val = UNDEFINED_VALUE;
    return CsExecGenerator(c, obj);
  }

  /* CsInitTaskInstance - initialize the 'TaskInstance' class */
  void CsInitCoroutine(VM *c) {
    /* create the 'Task' type */
    c->coroutineDispatch = CsEnterCPtrObjectType(CsGlobalScope(c), "Task",
                                                 task_instance_methods,
                                                 task_instance_properties, nullptr);
    if (!c->coroutineDispatch)
      CsInsufficientMemory(c);
    else {
      c->coroutineDispatch->scan           = CoroutineScan;
      c->coroutineDispatch->destroy        = CoroutineDestroy;
      c->coroutineDispatch->getNextElement = CoroutineNextElement;
    }
  }

  bool CsCoroutineP(VM *c, value obj) {
    return obj && CsIsType(obj, c->coroutineDispatch);
  }

  bool CsTaskP(VM *c, value obj) {
    if (obj && CsIsType(obj, c->coroutineDispatch)) {
      CsTask *p = CsCoroutineValue(c, obj);
      return p && p->isTask;
    }
    return false;
  }

  bool CsGeneratorP(VM *c, value obj) {
    if (obj && CsIsType(obj, c->coroutineDispatch)) {
      CsTask *p = CsCoroutineValue(c, obj);
      return p && !p->isTask;
    }
    return false;
  }

  CsTask *CsCoroutineValue(VM *c, value obj) {
#ifdef _DEBUG
    dispatch *pd = CsGetDispatch(obj);
#endif
    assert(CsCoroutineP(c, obj));
    return (CsTask *)CsCObjectValue(obj);
  }

  /* CsMakeCoroutine - make a 'TaskInstance' obj */
  value CsMakeCoroutine(VM *c, CsTask *prg) {
    value newo = CsMakeCPtrObject(c, c->coroutineDispatch, prg);
    return newo;
  }

  CsTask::CsTask(VM *c, bool task)
      : current(nullptr), callbacks(NULL_VALUE), isTask(task), ns(NULL_VALUE), method(NULL_VALUE) {

    //stackPlaceholder = CsMakeByteVector(c, nullptr, sizeof(CsTask)); // Note: stackPlaceholder is not used as a memory storage, just a memory holder to triger GC

    ns = c->currentNS;
    
    memzero(stack);

    CsSavedState initial(c);

    initial.stack = stack;
    /* initialize the stack */
    initial.stackTop = initial.stack + items_in(stack);
    initial.fp       = (CsFrame *)initial.stackTop;
    initial.sp       = initial.stackTop;

    // copy the function itself and its arguments to the begining of the Task's
    // stack
    for (int n = 0; n <= c->argc; ++n)
      *(--initial.sp) = CsGetArg(c, n);

    assert(CsMethodP(initial.stackTop[-1])); // task's stack shall start from
                                             // the task's function reference
    initial.env = UNDEFINED_VALUE;
    initial.vm  = c;

    initial.restore();

    argc = c->argc;
  }

  CsTask::~CsTask() { delete current; }

  void CsPersistCurrentCoroutine(VM *c) {
    assert(CsCompiledCodeIsCoroutine(c->code));
    CsTask *exec = CsCoroutineValue(c, c->exec->taskInstance);
    assert(CsCoroutineP(c, c->exec->taskInstance));
    assert(exec);
    // WRONG : if(exec) exec->states->store(c); - has to be chained
    if (exec) exec->current = new CsSavedState(c);
  }

} // namespace tis
