/* cs_angle.c - 'Duration' type */
/*
        Copyright (c) 2001-2013 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include "cs_int.h"
#include <limits.h>

namespace tis {

  /* method handlers */
  static value CSF_parse(VM *c);
  static value CSF_toFloat(VM *c);
  // static value CSF_toDegrees(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);
  static value CSF_morph(VM *c);

  /* Integer methods */
  static c_method methods[] = {C_METHOD_ENTRY("parse", CSF_parse),
                               C_METHOD_ENTRY("toFloat", CSF_toFloat),
                               C_METHOD_ENTRY("toInteger", CSF_toInteger),
                               C_METHOD_ENTRY("toString", CSF_toString),
                               C_METHOD_ENTRY("toCssString", CSF_toString),
                               C_METHOD_ENTRY("toHtmlString", CSF_toString),
                               C_METHOD_ENTRY("toUrlString", CSF_toString),
                               C_METHOD_ENTRY("morph", CSF_morph),
                               C_METHOD_ENTRY(0, 0)};

  static value CSF_seconds(VM *c, value obj);
  static value CSF_milliseconds(VM *c, value obj);

  /* Integer properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY("seconds", CSF_seconds, 0),
      VP_METHOD_ENTRY("milliseconds", CSF_milliseconds, 0),
      VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {
      // CONSTANT_ENTRY("MAX"    , int_value(INT_MAX     )),
      // CONSTANT_ENTRY("MIN"    , int_value(INT_MIN     )),
      CONSTANT_ENTRY(0, 0)};

  /* CsInitDuration - initialize the 'Duration' obj */
  void CsInitDuration(VM *c) {
    c->durationObject =
        CsEnterType(CsGlobalScope(c), "Duration", &CsDurationDispatch);
    CsEnterMethods(c, c->durationObject, methods);
    CsEnterVPMethods(c, c->durationObject, properties);
    CsEnterConstants(c, c->durationObject, constants);
  }

  /* CSF_parse - built-in method 'parse' */
  static value CSF_parse(VM *c) {
    wchars str;
    value  defval = UNDEFINED_VALUE;
    CsParseArguments(c, "**S#|V", &str.start, &str.length, &defval);

    double dur = tool::str_to_f(str, std::numeric_limits<double>::quiet_NaN());
    if (dur == std::numeric_limits<double>::quiet_NaN()) return defval;
    if (str.starts_with(WCHARS("s")))
      return CsMakeDuration(dur, DURATION_UNIT_TYPE::UT_S);
    else if (str.starts_with(WCHARS("ms")))
      return CsMakeDuration(dur, DURATION_UNIT_TYPE::UT_MS);
    return defval;
  }

  static value CSF_seconds(VM *c, value obj) {
    int t = unpack_int32(obj);
    return CsMakeFloat(t / 10000.0);
  }

  static value CSF_milliseconds(VM *c, value obj) {
    int t = unpack_int32(obj);
    return CsMakeFloat(t / 10.0);
  }

  static value CSF_morph(VM *c) {
    value  from;
    value  to;
    double ratio = 0;

    CsParseArguments(c, "**V=V=d", &from, &CsDurationDispatch, &to,
                     &CsDurationDispatch, &ratio);

    double df = CsDurationSeconds(from);
    double dt = CsDurationSeconds(to);

    return CsMakeDuration(df + (dt - df) * ratio, DURATION_UNIT_TYPE::UT_S);
  }
    
  value CsMakeDuration(float_t v, DURATION_UNIT_TYPE ut, bool raw) {
    if(raw)
      return pack_value(PT_DURATION, ut , int(v * 1000.0));
    else if (ut == DURATION_UNIT_TYPE::UT_MS) 
      return pack_value(PT_DURATION, DURATION_UNIT_TYPE::UT_MS, int(v));
    else 
      return pack_value(PT_DURATION, DURATION_UNIT_TYPE::UT_S, int(v * 1000.0));
  }

  float_t CsDurationSeconds(value obj) {
    int v = (int)unpack_uint32(obj);
    return v / 1000.0;
  }

  DURATION_UNIT_TYPE CsDurationUnits(value obj) {
    return unpack_unit<DURATION_UNIT_TYPE>(obj);
  }

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsDurationDispatch);
    int v = unpack_int32(obj);
    return CsMakeFloat(v / 1000.0);
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsDurationDispatch);
    int v = unpack_int32(obj);
    return CsMakeInteger(int((v)));
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value obj;
    CsParseArguments(c, "V=*|i", &obj, &CsDurationDispatch);
    return CsMakeCString(c, tool::value::duration_to_string(CsDurationSeconds(obj), CsDurationUnits(obj)));
    //int v = unpack_int32(obj);
    //return CsMakeCString(c, fixedtow(v, 4, W(""), W("s")));
  }

  // value CSF_angle(VM *c) { return angle_value( c ); }
  value CSF_seconds(VM *c) {
    double v;
    CsParseArguments(c, "**d", &v);
    return CsMakeDuration(v, DURATION_UNIT_TYPE::UT_S);
  }
  value CSF_milliseconds(VM *c) {
    double v;
    CsParseArguments(c, "**d", &v);
    return CsMakeDuration(v, DURATION_UNIT_TYPE::UT_MS);
  }

  static bool  GetDurationProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetDurationProperty(VM *c, value obj, value tag, value value);
  static bool  DurationPrint(VM *c, value obj, stream *s, bool toLocale);
  static long  DurationSize(value obj);
  static value DurationCopy(VM *c, value obj);
  static int_t DurationHash(value obj);

  dispatch CsDurationDispatch = {
      "Duration",          &CsDurationDispatch,  GetDurationProperty,
      SetDurationProperty, CsDefaultNewInstance, DurationPrint,
      DurationSize,        DurationCopy,         CsDefaultScan,
      DurationHash,        CsDefaultGetItem,     CsDefaultSetItem};

  /* GetDurationProperty - Duration get property handler */
  static bool GetDurationProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->durationObject, tag, pValue);
  }

  /* SetDurationProperty - Duration set property handler */
  static bool SetDurationProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->durationObject, tag, value);
  }

  /* DurationPrint - Duration print handler */
  static bool DurationPrint(VM *c, value obj, stream *s, bool toLocale) {
    // UNIT_TYPE u = unpack_unit(obj);
    //int v = unpack_int32(obj);
    //return s->put_str(fixedtow(v, 4, W(""), W("s")));
    return s->put_str(tool::value::duration_to_string(CsDurationSeconds(obj), CsDurationUnits(obj)));

  }

  /* DurationSize - Duration size handler */
  static long DurationSize(value obj) {
    // return sizeof(CsDuration);
    return sizeof(value);
  }

  /* DurationCopy - Duration copy handler */
  static value DurationCopy(VM *c, value obj) {
    // return CsPointerP(obj) ? CsDefaultCopy(c,obj) : obj;
    return obj;
  }

  /* DurationHash - Duration hash handler */
  static int_t DurationHash(value obj) { return (int_t)obj; }

  value CsDurationBinaryOp(VM *c, int op, value p1, value p2) {
    DURATION_UNIT_TYPE u = unpack_unit<DURATION_UNIT_TYPE>(p1);
    // int v = unpack_int32(p1);
    if (!CsDurationP(p1)) CsUnsupportedBinaryOp(c, op, p1, p2);

    float_t fv = CsDurationSeconds(p1);

    if (CsIntegerP(p2)) {
      int i = CsIntegerValue(p2);
      switch (op) {
      case BC_ADD: fv += CsDurationSeconds(CsMakeDuration(i, u)); break;
      case BC_SUB: assert(false); break;
      case BC_MUL: fv *= i; break;
      case BC_DIV:
        if (i != 0)
          fv /= i;
        else
          fv = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else if (CsFloatP(p2)) {
      double f = CsFloatValue(p2);
      switch (op) {
      case BC_ADD: fv += f; break;
      case BC_SUB: assert(false); break;
      case BC_MUL: fv *= f; break;
      case BC_DIV:
        if (f != 0)
          fv /= f;
        else
          fv = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else if (CsDurationP(p2)) {
      // int v2; UNIT_TYPE u2;
      // v2 = to_unit(p2,u2);
      switch (op) {
      case BC_ADD: fv += CsDurationSeconds(p2); break;
      case BC_SUB: fv -= CsDurationSeconds(p2); break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else
      CsUnsupportedBinaryOp(c, op, p1, p2);

    return CsMakeDuration(fv, DURATION_UNIT_TYPE::UT_S);
  }

  int CsDurationsCompare(VM *c, int op, value obj1, value obj2) {
    int v1 = unpack_int32(obj1);
    int v2 = unpack_int32(obj2);
    return v1 - v2;
  }

} // namespace tis
