/* cs_color.cpp - 'Color' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include <limits.h>

#ifdef SCITER
#include "gool/gool-types.h"
#endif

namespace tis {
  inline unsigned r(unsigned c) { return c & 0xff; }
  inline unsigned g(unsigned c) { return (c >> 8) & 0xff; }
  inline unsigned b(unsigned c) { return (c >> 16) & 0xff; }
  inline unsigned a(unsigned c) { return (c >> 24) & 0xff; }

  inline unsigned rgba(unsigned r, unsigned g, unsigned b, unsigned a) {
    return ((a & 0xff) << 24) | ((b & 0xff) << 16) | ((g & 0xff) << 8) |
           (r & 0xff);
  }
  inline unsigned rgba(unsigned r, unsigned g, unsigned b) {
    return ((b & 0xff) << 16) | ((g & 0xff) << 8) | (r & 0xff);
  }

  /* method handlers */
  static value CSF_rgba(VM *c);
  static value CSF_toFloat(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);

#ifdef SCITER
  static value CSF_parse(VM *c);
  static value CSF_hsl(VM *c);
  static value CSF_hsv(VM *c);
  static value CSF_to_hsl(VM *c);
  static value CSF_to_hsv(VM *c);
  static value CSF_tint(VM *c);
  static value CSF_morph(VM *c);
  static value CSF_hue(VM *c);
  static value CSF_rotate(VM *c);
  static value CSF_lightness(VM *c);
  static value CSF_lighten(VM *c);
  static value CSF_darken(VM *c);
  static value CSF_saturation(VM *c);
  static value CSF_saturate(VM *c);
  static value CSF_desaturate(VM *c);
  static value CSF_opacity(VM *c);
  static value CSF_opacify(VM *c);
  static value CSF_grayscale(VM *c);
#endif

  /* Integer methods */
  static c_method methods[] = {C_METHOD_ENTRY("rgba", CSF_rgba),
#ifdef SCITER
                               C_METHOD_ENTRY("parse", CSF_parse),
                               C_METHOD_ENTRY("hsl", CSF_hsl),
                               C_METHOD_ENTRY("hsv", CSF_hsv),
                               C_METHOD_ENTRY("toHSV", CSF_to_hsv),
                               C_METHOD_ENTRY("toHSL", CSF_to_hsl),
                               C_METHOD_ENTRY("tint", CSF_tint),
                               C_METHOD_ENTRY("morph", CSF_morph),

                               C_METHOD_ENTRY("hue", CSF_hue),
                               C_METHOD_ENTRY("rotate", CSF_rotate),
                               C_METHOD_ENTRY("lightness", CSF_lightness),
                               C_METHOD_ENTRY("lighten", CSF_lighten),
                               C_METHOD_ENTRY("darken", CSF_darken),
                               C_METHOD_ENTRY("saturation", CSF_saturation),
                               C_METHOD_ENTRY("saturate", CSF_saturate),
                               C_METHOD_ENTRY("desaturate", CSF_desaturate),
                               C_METHOD_ENTRY("opacity", CSF_opacity),
                               C_METHOD_ENTRY("opacify", CSF_opacify),
                               //C_METHOD_ENTRY("mix", CSF_mix),
                               C_METHOD_ENTRY("grayscale", CSF_grayscale),
                               //C_METHOD_ENTRY("sepia", CSF_sepia),
#endif
                               C_METHOD_ENTRY("toFloat", CSF_toFloat),
                               C_METHOD_ENTRY("toInteger", CSF_toInteger),
                               C_METHOD_ENTRY("toString", CSF_toString),
                               C_METHOD_ENTRY("toHtmlString", CSF_toString),
                               C_METHOD_ENTRY("toCssString", CSF_toString),
                               C_METHOD_ENTRY("toUrlString", CSF_toString),
                               C_METHOD_ENTRY(0, 0)};

  static value CSF_red(VM *c, value obj);
  static value CSF_green(VM *c, value obj);
  static value CSF_blue(VM *c, value obj);
  static value CSF_alpha(VM *c, value obj);
  //static value CSF_opacity(VM *c, value obj);

  /* Integer properties */
  static vp_method properties[] = {VP_METHOD_ENTRY("r", CSF_red, 0),
                                   VP_METHOD_ENTRY("g", CSF_green, 0),
                                   VP_METHOD_ENTRY("b", CSF_blue, 0),
                                   VP_METHOD_ENTRY("a", CSF_alpha, 0),
                                   //VP_METHOD_ENTRY("opacity", CSF_opacity, 0),
                                   VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {CONSTANT_ENTRY(0, 0)};

  /* CsInitColor - initialize the 'Color' obj */
  void CsInitColor(VM *c) {
    c->colorObject = CsEnterType(CsGlobalScope(c), "Color", &CsColorDispatch);
    CsEnterMethods(c, c->colorObject, methods);
    CsEnterVPMethods(c, c->colorObject, properties);
    CsEnterConstants(c, c->colorObject, constants);
  }

  unsigned color_value(value obj) { return unpack_uint32(obj); }

  static value CSF_red(VM *c, value obj) {
    return int_value(r(color_value(obj)));
  }
  static value CSF_green(VM *c, value obj) {
    return int_value(g(color_value(obj)));
  }
  static value CSF_blue(VM *c, value obj) {
    return int_value(b(color_value(obj)));
  }
  static value CSF_alpha(VM *c, value obj) {
    return int_value(255 - a(color_value(obj)));
  }

  //static value CSF_opacity(VM *c, value obj) {
  //  unsigned alpha = a(color_value(obj));
  //  return CsMakeFloat((255.0 - double(alpha)) / 255.0);
  //}

  value CSF_color(VM *c) { return CSF_rgba(c); }

  unsigned v2cc(value v) {
    if (CsIntegerP(v)) return unsigned(CsIntegerValue(v));
    if (CsFloatP(v)) return unsigned(CsFloatValue(v) * 255);
    return 0;
  }

  static value CSF_rgba(VM *c) {
    value first = CsGetArg(c, 3);
#ifdef SCITER
    if (CsStringP(first)) {
      const wchar *s       = W("");
      value        opacity = 0;
      CsParseArguments(c, "**S|V", &s, &opacity);

      gool::color_v cv = gool::parse_color(string(s));
      gool::argb    cc = cv.to_argb();
      if (opacity) {
        if (CsFloatP(opacity))
          cc.alfa = gool::argb::channel_t(
              255.0 * tool::limit<double>(CsFloatValue(opacity), 0, 1.0));
        else if (CsIntegerP(opacity))
          cc.alfa = gool::argb::channel_t(
              tool::limit(CsIntegerValue(opacity), 0, 255));
      }
      return pack_value(PT_COLOR, 0, gool::color(cc));
    }
#endif

    if (CsColorP(first)) return first;

    if (CsIntegerP(first) && (CsArgCnt(c) == 3)) {
      int cc = CsIntegerValue(first);
      return pack_value(PT_COLOR, 0, rgba(r(cc), g(cc), b(cc)));
    }

    value    vr, vg, vb;
    unsigned va  = 0;
    value    vva = 0;
    CsParseArguments(c, "**VVV|V", &vr, &vg, &vb, &vva);

    // return CsMakeFloat((float_t)CsIntegerValue(obj));
    if (vva) {
      if (CsFloatP(vva))
        va = 255 -
             unsigned(255.0 * tool::limit<double>(CsFloatValue(vva), 0, 1.0));
      else if (CsIntegerP(vva))
        va = 255 - tool::limit(CsIntegerValue(vva), 0, 255);
    }
    return pack_value(PT_COLOR, 0,
                      rgba(v2cc(vr), v2cc(vg), v2cc(vb), va));
  }

  value CsMakeColor(byte vr, byte vg, byte vb, byte va) {
    return pack_value(PT_COLOR, 0, rgba(vr, vg, vb, 255 - va));
  }


#ifdef SCITER

  static value CSF_parse(VM *c) {
    wchar *str;
    int    len;
    CsParseArguments(c, "**S#", &str, &len);
    gool::color_v cv = gool::parse_color(string(str, len));
    return pack_value(PT_COLOR, 0, gool::color(cv.to_argb()));
  }

  static value CSF_hsv(VM *c) {
    float    vh, vs, vv;
    unsigned va  = 0;
    value    vva = 0;
    CsParseArguments(c, "**gff|V", &vh, &vs, &vv, &vva);
    if (vva) {
      if (CsFloatP(vva))
        va = 255 -
             unsigned(255.0 * tool::limit<double>(CsFloatValue(vva), 0, 1.0));
      else if (CsIntegerP(vva))
        va = 255 - tool::limit(CsIntegerValue(vva), 0, 255);
    }
    gool::hsv hsv(vh, vs, vv);
    gool::rgb rgb(hsv);

    return pack_value(PT_COLOR, 0,
                      rgba(rgb.red, rgb.green, rgb.blue, va));
  }

  static value CSF_hsl(VM *c) {
    float    vh, vs, vl;
    unsigned va  = 0;
    value    vva = 0;
    CsParseArguments(c, "**gff|V", &vh, &vs, &vl, &vva);
    if (vva) {
      if (CsFloatP(vva))
        va = 255 -
             unsigned(255.0 * tool::limit<double>(CsFloatValue(vva), 0, 1.0));
      else if (CsIntegerP(vva))
        va = 255 - tool::limit(CsIntegerValue(vva), 0, 255);
    }
    gool::hsv hsl(vh, vs, vl);
    gool::rgb rgb(hsl);

    return pack_value(PT_COLOR, 0,
                      rgba(rgb.red, rgb.green, rgb.blue, va));
  }
  static value CSF_to_hsl(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsColorDispatch);
    gool::rgb rgb(color_value(obj));
    gool::hsl hsl(rgb);
    CS_RETURN3(c, CsMakeFloat(hsl.h), CsMakeFloat(hsl.s), CsMakeFloat(hsl.l));
  }
  static value CSF_to_hsv(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsColorDispatch);
    gool::rgb rgb(color_value(obj));
    gool::hsv hsv(rgb);
    CS_RETURN3(c, CsMakeFloat(hsv.h), CsMakeFloat(hsv.s), CsMakeFloat(hsv.v));
  }

  static value CSF_tint(VM *c) {
    gool::color base_color;
    float       saturation = 0, luminance = 0, hue = 0;

    CsParseArguments(c, "C*f|f|g", &base_color, &luminance, &saturation, &hue);

    gool::rgb in(base_color);

    gool::hsl z = in;

    saturation = limit(saturation, -1.0f, 1.0f);
    luminance  = limit(luminance, -1.0f, 1.0f);
    // hue = limit(luminance, -1.0f, 1.0f );

    if (saturation < 0.0f)
      z.s -= z.s * (-saturation);
    else if (saturation > 0.0f)
      z.s += (1.0f - z.s) * (saturation);

    if (luminance < 0.0f)
      z.l -= z.l * (-luminance);
    else if (luminance > 0.0f)
      z.l += (1.0f - z.l) * (luminance);

    z.h += hue;

    gool::rgb out = z;

    gool::color res = out;
    return CsMakeColor(res);
  }

  static value CSF_morph(VM *c) {
    uint   color_from;
    uint   color_to;
    double ratio = 0;

    CsParseArguments(c, "**CCd", &color_from, &color_to, &ratio);

    gool::argb c1(color_from);
    gool::argb c2(color_to);

    ratio = limit(ratio, 0.0, 1.0);

    gool::argb c3 = gool::argb::morph(c1, c2, ratio);

    return CsMakeColor(c3.red, c3.green, c3.blue, c3.alfa);
  }

  static value CSF_hue(VM *c) {
    gool::color base_color;
    float       hue = 0;

    CsParseArguments(c, "C*g", &base_color, &hue);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.h = hue;

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_rotate(VM *c) {
    gool::color base_color;
    float       hue = 0;

    CsParseArguments(c, "C*g", &base_color, &hue);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.h = fmod(z.h + hue, 360.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_lightness(VM *c) {
    gool::color base_color;

    float lightness;

    CsParseArguments(c, "C*F", &base_color, &lightness);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.l = limit(lightness, 0.0f, 1.0f);
    
    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_lighten(VM *c) {
    gool::color base_color;

    float lightness;

    CsParseArguments(c, "C*F", &base_color, &lightness);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.l = limit(z.l + lightness, 0.0f, 1.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_darken(VM *c) {
    gool::color base_color;

    float lightness;

    CsParseArguments(c, "C*F", &base_color, &lightness);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.l = limit(z.l - lightness, 0.0f, 1.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_saturation(VM *c) {
    gool::color base_color;

    float saturation;

    CsParseArguments(c, "C*F", &base_color, &saturation);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.s = limit(saturation, 0.0f, 1.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_saturate(VM *c) {
    gool::color base_color;

    float saturation;

    CsParseArguments(c, "C*F", &base_color, &saturation);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.s = limit(z.s + saturation, 0.0f, 1.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_desaturate(VM *c) {
    gool::color base_color;

    float saturation;

    CsParseArguments(c, "C*F", &base_color, &saturation);

    gool::argb base(base_color);

    gool::hsl z = gool::rgb(base);

    z.s = limit(z.s - saturation, 0.0f, 1.0f);

    gool::rgb out = z;

    base.red = out.red;
    base.green = out.green;
    base.blue = out.blue;

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_opacity(VM *c) {
    gool::color base_color;

    float opacity;

    CsParseArguments(c, "C*F", &base_color, &opacity);

    gool::argb base(base_color);

    base.alfa = (gool::argb::channel_t)limit(int(opacity * 255), 0, 255);

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_opacify(VM *c) {
    gool::color base_color;

    float opacity;

    CsParseArguments(c, "C*F", &base_color, &opacity);

    gool::argb base(base_color);

    base.alfa = (gool::argb::channel_t)limit(base.alfa - int(opacity * 255), 0, 255);

    gool::color res = base;
    return CsMakeColor(res);
  }

  static value CSF_grayscale(VM *c) {
    gool::color base_color;

    float ratio = 1.0f;

    CsParseArguments(c, "C*|F", &base_color, &ratio);

    gool::argb base(base_color);

    uint luma = base.luminance();
    ratio = limit(ratio, 0.0f, 1.0f);
    gool::argb out(luma, luma, luma, base.alfa);
    out = gool::argb::morph(base, out, ratio);

    gool::color res = base;
    return CsMakeColor(res);
  }


#endif

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsColorDispatch);
    return CsMakeFloat((float_t)CsIntegerValue(obj));
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsColorDispatch);
    return CsMakeInteger(CsIntegerValue(obj));
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value obj;
    value sym = 0;
    CsParseArguments(c, "V=*|V=", &obj, &CsColorDispatch, &sym,
                     &CsSymbolDispatch);

    unsigned clr = color_value(obj);

    ustring sv;
    if (!sym || sym == CsSymbolOf("RGB"))
      sv = ustring::format(W("#%02x%02x%02x"), r(clr), g(clr), b(clr));
    else if (sym == CsSymbolOf("rgb"))
      // swprintf(buf,100,L"rgb(%d,%d,%d)", r(clr), g(clr), b(clr) );
      sv = ustring::format(W("rgb(%d,%d,%d)"), r(clr), g(clr), b(clr));
    else if (sym == CsSymbolOf("rgba"))
      // swprintf(buf,100,L"rgb(%d,%d,%d,%f)", r(clr), g(clr), b(clr),
      // double(255-a(clr))/255.0);
      sv = ustring::format(W("rgba(%d,%d,%d,%f)"), r(clr), g(clr), b(clr),
                           double(255 - a(clr)) / 255.0);
    return CsMakeCString(c, sv);
  }

  static bool  GetColorProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetColorProperty(VM *c, value obj, value tag, value value);
  static bool  ColorPrint(VM *c, value obj, stream *s, bool toLocale);
  //static value ColorGetItem(VM *c, value obj, value tag);
  static long  ColorSize(value obj);
  static value ColorCopy(VM *c, value obj);
  static int_t ColorHash(value obj);

  dispatch CsColorDispatch = {"Color",
                              &CsColorDispatch,
                              GetColorProperty,
                              SetColorProperty,
                              CsDefaultNewInstance,
                              ColorPrint,
                              ColorSize,
                              ColorCopy,
                              CsDefaultScan,
                              ColorHash,
                              CsDefaultGetItem,
                              CsDefaultSetItem};

  /* GetColorProperty - Color get property handler */
  static bool GetColorProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->colorObject, tag, pValue);
  }

  /* SetColorProperty - Color set property handler */
  static bool SetColorProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->colorObject, tag, value);
  }

  value CsColorGetItem(VM *c, value obj, value tag) 
  {
#if defined(SCITER)
    if (CsSymbolP(tag) ) // color#fff form
    {
      gool::color_v cv = gool::parse_color( string::format("#%S",CsSymbolName(tag).c_str()));
      gool::argb    cc = cv.to_argb();
      return pack_value(PT_COLOR, 0, gool::color(cc));
    }
#endif
    return CsDefaultGetItem(c, obj, tag);
  }

  bool CsColorGetProperty(VM *c, value &obj, value tag, value *pValue) {
#if defined(SCITER)
    if (CsSymbolP(tag)) // color.violet form
    {
      gool::color_v cv = gool::parse_color(string(CsSymbolName(tag))());
      if (cv.is_defined()) {
        gool::argb    cc = cv.to_argb();
        *pValue = pack_value(PT_COLOR, 0, gool::color(cc));
        return true;
      }
    }
#endif
    return CsGetVirtualProperty(c, obj, c->methodObject, tag, pValue);
  }

  void CsColorToString(char *buf, value obj) {
    unsigned cv = unpack_uint32(obj);
    // if( (cv & 0xff000000) != 0 )
    sprintf(buf, "color(%d,%d,%d,%f)", r(cv), g(cv), b(cv),
            (255 - a(cv)) / 255.0);
    // else
    //  sprintf(buf,"color(%d,%d,%d)",r(cv),g(cv),b(cv));
  }

  /* ColorPrint - Color print handler */
  static bool ColorPrint(VM *c, value obj, stream *s, bool toLocale) {
    char buf[32];
    CsColorToString(buf, obj);
    return s->put_str(buf);
  }

  /* ColorSize - Color size handler */
  static long ColorSize(value obj) {
    // return sizeof(CsColor);
    return sizeof(value);
  }

  /* ColorCopy - Color copy handler */
  static value ColorCopy(VM *c, value obj) {
    // return CsPointerP(obj) ? CsDefaultCopy(c,obj) : obj;
    return obj;
  }

  /* ColorHash - Color hash handler */
  static int_t ColorHash(value obj) {
    uint32 cv = unpack_uint32(obj);
    return tool::hash(cv);
  }

} // namespace tis
