/* cs_bytevector.c - 'ByteVector' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <stdlib.h>
#include <string.h>
#include "cs.h"
#include "tool/tl_lzf.h"

namespace tis {

  value CsMakeFilledByteVector(VM *c, byte fill, int_t size);

  /* method handlers */
  static value CSF_ctor(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);
  static value CSF_save(VM *c);
  static value CSF_load(VM *c);
  static value CSF_fromBase64(VM *c);
  static value CSF_md5(VM *c);
  static value CSF_crc32(VM *c);
  static value CSF_fromString(VM *c);
  static value CSF_fromDataUrl(VM *c);
  static value CSF_toDataUrl(VM *c);
  static value CSF_compare(VM *c);
  static value CSF_compress(VM *c);
  static value CSF_decompress(VM *c);
  /* virtual property methods */
  static value CSF_length(VM *c, value obj);
  static value CSF_get_type(VM *c, value obj);
  static void  CSF_set_type(VM *c, value obj, value typ);
  static value CSF_get_name(VM *c, value obj);
  static void  CSF_set_name(VM *c, value obj, value typ);

  /* String methods */
  static c_method methods[] = {
      C_METHOD_ENTRY("this", CSF_ctor),
      C_METHOD_ENTRY("toLocaleString", CSF_std_toLocaleString),
      C_METHOD_ENTRY("toString", CSF_toString),
      C_METHOD_ENTRY("toInteger", CSF_toInteger),
      C_METHOD_ENTRY("save", CSF_save),
      C_METHOD_ENTRY("load", CSF_load),
      C_METHOD_ENTRY("fromBase64", CSF_fromBase64),
      C_METHOD_ENTRY("fromString", CSF_fromString),
      C_METHOD_ENTRY("md5", CSF_md5),
      C_METHOD_ENTRY("crc32", CSF_crc32),
      C_METHOD_ENTRY("compare", CSF_compare),
      C_METHOD_ENTRY("compress", CSF_compress),
      C_METHOD_ENTRY("decompress", CSF_decompress),
      C_METHOD_ENTRY("toDataUrl", CSF_toDataUrl),
      C_METHOD_ENTRY("fromDataUrl", CSF_fromDataUrl),
      
      C_METHOD_ENTRY(0, 0)};

  /* String properties */
  static vp_method properties[] = {
      VP_METHOD_ENTRY("length", CSF_length, 0),
      VP_METHOD_ENTRY("type", CSF_get_type, CSF_set_type),
      VP_METHOD_ENTRY("name", CSF_get_name, CSF_set_name),
      VP_METHOD_ENTRY(0, 0, 0)};

  /* CsInitByteVector - initialize the 'ByteVector' obj */
  void CsInitByteVector(VM *c) {
    c->byteVectorObject = CsEnterType(
        CsGlobalScope(c), CsByteVectorDispatch.typeName, &CsByteVectorDispatch);
    CsEnterMethods(c, c->byteVectorObject, methods);
    CsEnterVPMethods(c, c->byteVectorObject, properties);
  }

  /* CSF_ctor - built-in method 'initialize' */
  static value CSF_ctor(VM *c) {
    long  size = 0;
    value obj;
    CsParseArguments(c, "V=*|i", &obj, &CsByteVectorDispatch, &size);
    obj          = CsMakeFilledByteVector(c, 0, size);
    CsCtorRes(c) = obj;
    return obj;
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    // value obj;
    // CsParseArguments(c,"V=*",&obj,&CsByteVectorDispatch);
    // return CsMakeInteger(c,CsByteVectorSize(obj));
    return UNDEFINED_VALUE;
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value  obj;
    wchars encoding = WCHARS("base64");
    CsParseArguments(c, "V=*|S#", &obj, &CsByteVectorDispatch, &encoding.start,
                     &encoding.length);

    tool::bytes bytes(CsByteVectorAddress(obj), CsByteVectorSize(obj));

    if (encoding == WCHARS("base64")) {
      // to base 64
      tool::array<char> b64;
      tool::base64_encode(bytes, b64);
      value s = CsMakeFilledString(c, ' ', b64.size());
      if (s && CsStringP(s)) {
        wchar *     pd  = CsStringAddress(s);
        const char *ps  = b64.head();
        const char *pse = ps + b64.size();
        while (ps < pse)
          *pd++ = *ps++;
        return s;
      }
    } else {
      tool::ustring str;
      if (!tool::decode_bytes(bytes, str, tool::string(encoding)))
        CsThrowKnownError(c, CsErrGenericError, "Unsupported encoding");
      return string_to_value(c, str);
    }
    return NULL_VALUE;
  }

  static value CSF_fromBase64(VM *c) {
    tool::wchars data;
    CsParseArguments(c, "**S#", &data.start, &data.length);
    tool::array<byte> out;
    tool::base64_decode(data, out);
    return CsMakeByteVector(c, out.head(), out.size());
  }

  //var dataUrl = String.printf("data:%s;src=%s;base64,%s", mime, src.urlEscape(), data.toString());
  static value CSF_fromDataUrl(VM *c) {
    tool::wchars in;
    CsParseArguments(c, "**S#", &in.start, &in.length);
    tool::array<byte> data;
    tool::string u = u8::cvt(in);
    tool::string mime_type; 
    if (!crack_data_url(u(), mime_type, data))
      return NULL_VALUE;
      //CsThrowKnownError(c, CsErrGenericError, "Unsupported encoding");

    value v = CsMakeByteVector(c, data);
    PROTECT(v);
    value mt = CsMakeString(c,mime_type);
    CsSetByteVectorType(v, mt);
    return v;
  }

  static value CSF_toDataUrl(VM *c) {
    value  obj;
    CsParseArguments(c, "V=*", &obj, &CsByteVectorDispatch);

    tool::bytes bytes = CsByteVectorBytes(obj);

    // var dataUrl = String.printf("data:%s;src=%s;base64,%s",mime, src.urlEscape(),data.toString());
    string mime = value_to_string(CsByteVectorType(obj));
    string b64 = tool::base64_encode(bytes);
    //string out = string::format("data:%s;base64,%s",mime.c_str(),b64.c_str());
    array<char> out;
    out.push(CHARS("data:"));
    out.push(mime());
    out.push(CHARS(";base64,\n"));
    out.push(b64);

    return CsMakeString(c, out());
  }



  /* CSF_md5 - built-in method 'md5' - computes md5 digest */
  static value CSF_md5(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsByteVectorDispatch);

    tool::bytes bytes(CsByteVectorAddress(obj), CsByteVectorSize(obj));

    tool::string str = tool::md5(bytes).to_string();
    return string_to_value(c, str);
  }

  /* CSF_crc32 - built-in method 'crc32' - control sum */
  static value CSF_crc32(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsByteVectorDispatch);

    tool::bytes bytes(CsByteVectorAddress(obj), CsByteVectorSize(obj));

    uint v = tool::crc32(bytes);

    return CsMakeInteger(int(v));
  }

  static value symGZ = CsSymbolOf("gz");
  static value symGZIP = CsSymbolOf("gzip");

  /* CSF_compress - LZF or GZIP compression */
  static value CSF_compress(VM *c) {
    value obj;
    value type = 0;
    CsParseArguments(c, "V=*|V=", &obj, &CsByteVectorDispatch,&type,&CsSymbolDispatch);
    tool::bytes       bytes = CsByteVectorBytes(obj);
    tool::array<byte> compressed;
    if (type == symGZ)
      tool::gzip(true, bytes, compressed,false);
    else if (type == symGZIP)
      tool::gzip(true, bytes, compressed, true);
    else
      tool::lzf::compress(bytes, compressed);
    return CsMakeByteVector(c, compressed);
  }

  static value CSF_decompress(VM *c) {
    value obj;
    value type = 0;
    CsParseArguments(c, "V=*|V=", &obj, &CsByteVectorDispatch, &type, &CsSymbolDispatch);
    tool::bytes       bytes = CsByteVectorBytes(obj);
    tool::array<byte> decompressed;
    if (type == symGZ) {
      if(tool::gzip(false, bytes, decompressed,false))
        return CsMakeByteVector(c, decompressed);
    }
    else if (type == symGZIP) {
      if (tool::gzip(false, bytes, decompressed, true))
        return CsMakeByteVector(c, decompressed);
    }
    else {
      if (tool::lzf::decompress(bytes, decompressed))
        return CsMakeByteVector(c, decompressed);
    }
    return NULL_VALUE;
  }

  static value CSF_fromString(VM *c) {
    wchars str;
    wchars encoding = WCHARS("base64");
    CsParseArguments(c, "**S#|S#", &str.start, &str.length, &encoding.start,
                     &encoding.length);

    array<byte> data;

    if (encoding == WCHARS("base64"))
      tool::base64_decode(str, data);
    else
      tool::encode_bytes(str, data, encoding);
    return CsMakeByteVector(c, data.cbegin(), data.size());
  }

  static value CSF_compare(VM *c) {
    value obj, obj2;
    CsParseArguments(c, "V=*V=", &obj, &CsByteVectorDispatch, &obj2,
                     &CsByteVectorDispatch);

    tool::bytes bytes1(CsByteVectorAddress(obj), CsByteVectorSize(obj));
    tool::bytes bytes2(CsByteVectorAddress(obj2), CsByteVectorSize(obj2));

    if (bytes1.length > bytes2.length) return CsMakeInteger(1);
    if (bytes1.length < bytes2.length) return CsMakeInteger(-1);

    return CsMakeInteger(memcmp(bytes1.start, bytes2.start, bytes1.length));
  }

  /* CSF_save - saves bytes to file */

  static value CSF_save(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
    {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
      // return FALSE_VALUE;
    }
    tool::wchars filename;
    value        obj;
    CsParseArguments(c, "V=*S#", &obj, &CsByteVectorDispatch, &filename.start,
                     &filename.length);
    tool::bytes data(CsByteVectorAddress(obj), CsByteVectorSize(obj));
    if (filename.length == 0) return FALSE_VALUE;
    if (filename.like(W("file://*"))) filename.prune(7);

    FILE *f = fopen(tool::string(filename), "w+b");
    if (!f) return FALSE_VALUE;
    size_t r = fwrite(data.start, 1, data.length, f);
    fclose(f);
    return r == data.length ? TRUE_VALUE : FALSE_VALUE;
  }

  /* CSF_load - loads bytes from file */

  static value CSF_load(VM *c) {
    if (!c->can_use_feature(FEATURE_FILE_IO))
    {
      CsThrowKnownError(c, CsErrNotAllowed, "FILE IO");
      // return FALSE_VALUE;
    }
    tool::wchars fn;
    CsParseArguments(c, "**S#", &fn.start, &fn.length);
    if (fn.length == 0) return NULL_VALUE;

    ustring filename = fn;

    tool::array<byte> buf;
    stream *          s = c->open_stream(filename, false);
    if (s) {
      s->get_content(buf);
      s->close();
    } else
      CsThrowKnownError(c, CsErrFileNotFound, filename.c_str());

    value obj = 0, name = CsMakeString(c, filename);
    PROTECT(obj, name);

    obj = CsMakeByteVector(c, buf);
    if (CsByteVectorP(obj)) {
      CsSetByteVectorName(obj, name);
      return obj;
    }
    return NULL_VALUE;
  }

  /* CSF_size - built-in property 'length' */
  static value CSF_length(VM *c, value obj) {
    return CsMakeInteger((int_t)CsByteVectorSize(obj));
  }

  static value CSF_get_type(VM *c, value obj) {
    return CsByteVectorType(obj);
    // return UNDEFINED_VALUE;
  }
  static void CSF_set_type(VM *c, value obj, value typ) {
    CsSetByteVectorType(obj, typ);
  }
  static value CSF_get_name(VM *c, value obj) {
    return CsByteVectorName(obj);
    // return UNDEFINED_VALUE;
  }
  static void CSF_set_name(VM *c, value obj, value typ) {
    CsSetByteVectorName(obj, typ);
  }

  /* ByteVector handlers */
  static bool  GetByteVectorProperty(VM *c, value &obj, value tag,
                                     value *pValue);
  static bool  SetByteVectorProperty(VM *c, value obj, value tag, value value);
  static value ByteVectorNewInstance(VM *c, value proto);
  static bool  ByteVectorPrint(VM *c, value val, stream *s, bool toLocale);
  static long  ByteVectorSize(value obj);
  static int_t ByteVectorHash(value obj);

  static value CsByteVectorGetItem(VM *c, value obj, value tag);
  static void  CsByteVectorSetItem(VM *c, value obj, value tag, value value);

  static void ByteVectorScan(VM *c, value obj);

  /* ByteVector pdispatch */
  dispatch CsByteVectorDispatch = {"Bytes",
                                   &CsByteVectorDispatch,
                                   GetByteVectorProperty,
                                   SetByteVectorProperty,
                                   ByteVectorNewInstance,
                                   ByteVectorPrint,
                                   ByteVectorSize,
                                   CsDefaultCopy,
                                   ByteVectorScan,
                                   ByteVectorHash,
                                   CsByteVectorGetItem,
                                   CsByteVectorSetItem,
                                   0,
                                   0,
                                   0,
                                   0,
                                   0,
                                   0,
                                   0,
                                   0,
                                   0};

  static value CsByteVectorGetItem(VM *c, value obj, value tag) {
    if (CsIntegerP(tag)) {
      int i;
      if ((i = CsIntegerValue(tag)) < 0 || i >= int(CsByteVectorSize(obj)))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      return CsMakeInteger(CsByteVectorByte(obj, i));
    }
    return UNDEFINED_VALUE;
  }

  void CsSetByteVectorByte(value o, int i, byte b) {
    CsByteVectorAddress(o)[i] = b;
  }

  static void CsByteVectorSetItem(VM *c, value obj, value tag, value value) {
    if (CsIntegerP(tag)) {
      int i;
      if (!CsIntegerP(value)) CsTypeError(c, value);
      if ((i = CsIntegerValue(tag)) < 0 || i >= int(CsByteVectorSize(obj)))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      if (ptr<byte_vector>(obj)->heap_data) // vectors allocated on heap are
                                            // read only, are they actually?
        CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
      CsSetByteVectorByte(obj, i, (byte)CsIntegerValue(value));
    }
  }

  /* GetByteVectorProperty - ByteVector get property handler */
  static bool GetByteVectorProperty(VM *c, value &obj, value tag,
                                    value *pValue) {
    return CsGetVirtualProperty(c, obj, c->byteVectorObject, tag, pValue);
  }

  /* SetByteVectorProperty - ByteVector set property handler */
  static bool SetByteVectorProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->byteVectorObject, tag, value);
  }

  /* ByteVectorNewInstance - create a new ByteVector */
  static value ByteVectorNewInstance(VM *c, value proto) {
    return CsMakeFilledByteVector(c, 0, 0);
  }

  /* ByteVectorPrint - ByteVector print handler */
  static bool ByteVectorPrint(VM *c, value val, stream *s, bool toLocale) {
  /*byte *p = CsByteVectorAddress(val);
  long size = CsByteVectorSize(val);
  if (!s->put('#'))
    return false;
  while (--size >= 0)
      if (!s->put(*p++))
          return false;
  return s->put('#');
  */
#pragma TODO("Print base64 vector!")
    return s->put_str("Bytes");
  }

  /* ByteVectorSize - ByteVector size handler */
  static long ByteVectorSize(value obj) {
    if (ptr<byte_vector>(obj)->heap_data)
      return sizeof(byte_vector);
    else
      return sizeof(byte_vector) + CsRoundSize(CsByteVectorSize(obj));
  }

  static void ByteVectorScan(VM *c, value obj) {
    if (ptr<byte_vector>(obj)->heap_data) {
      ptr<byte_vector>(obj)->next = c->newSpace->bObjects;
      c->newSpace->bObjects       = obj;
    }
    CsSetByteVectorType(obj, CsCopyValue(c, CsByteVectorType(obj)));
    CsSetByteVectorName(obj, CsCopyValue(c, CsByteVectorName(obj)));
  }

  /* ByteVectorHash - ByteVector hash handler */
  static int_t ByteVectorHash(value obj) {
    return CsHashBytes(CsByteVectorBytes(obj));
  }

  static void CsSetByteVectorSize(value o, size_t sz) {
    ptr<byte_vector>(o)->size = sz;
  }

  /* CsMakeByteVector - make and initialize a new ByteVector value */
  value CsMakeByteVector(VM *c, const byte *data, int_t size) {
    long  allocSize = sizeof(byte_vector) + CsRoundSize(size);
    value newo      = CsAllocate(c, allocSize);
    byte *p         = CsByteVectorAddress(newo);
    CsSetDispatch(newo, &CsByteVectorDispatch);
    CsSetByteVectorSize(newo, size);
    if (data) target(p, size).copy(data);
    CsSetByteVectorType(newo, UNDEFINED_VALUE);
    CsSetByteVectorName(newo, UNDEFINED_VALUE);
    assert(allocSize == ValueSize(newo));
    return newo;
  }

  /* CsMakeString - make and initialize a new string value */
  value CsMakeFilledByteVector(VM *c, byte fill, int_t size) {
    long allocSize = sizeof(byte_vector) + CsRoundSize(size); /* space for zero terminator */
    value newo = CsAllocate(c, allocSize);
    byte *p    = CsByteVectorAddress(newo);
    CsSetDispatch(newo, &CsByteVectorDispatch);
    CsSetByteVectorSize(newo, size);
    memset(p, fill, size);
    CsSetByteVectorType(newo, UNDEFINED_VALUE);
    CsSetByteVectorName(newo, UNDEFINED_VALUE);
    assert(allocSize == ValueSize(newo));
    return newo;
  }

  value CsMakeByteVector(VM *c, const tool::array<byte> &data) {
    long  allocSize = sizeof(byte_vector); /* space for zero terminator */
    value newo      = CsAllocate(c, allocSize);
    CsSetDispatch(newo, &CsByteVectorDispatch);
    CsSetByteVectorSize(newo, 0);
    CsSetByteVectorType(newo, UNDEFINED_VALUE);
    CsSetByteVectorName(newo, UNDEFINED_VALUE);
    data.set_ref_to(ptr<byte_vector>(newo)->heap_data);
    // link it to CObjects list
    ptr<byte_vector>(newo)->next = c->newSpace->bObjects;
    c->newSpace->bObjects        = newo;
    return newo;
  }

  size_t CsByteVectorSize(value o) {
    byte_vector *me = ptr<byte_vector>(o);
    if (me->heap_data) {
      assert(me->size == 0);
      return me->heap_data->size;
    }
    return me->size;
  }
  byte *CsByteVectorAddress(value o) {
    byte_vector *me = ptr<byte_vector>(o);
    if (me->heap_data) {
      assert(me->size == 0);
      return me->heap_data->elements;
    }
    return ptr<byte>(o) + sizeof(byte_vector);
  }

  array<byte> CsByteVectorArray(value o) {
    byte_vector *me = ptr<byte_vector>(o);
    if (me->heap_data) {
      assert(me->size == 0);
      return tool::array<byte>(me->heap_data);
    }
    return array<byte>(CsByteVectorBytes(o));
  }

  void CsDestroyUnreachableByteVectors(VM *c) {
    value obj = c->oldSpace->bObjects;
    while (obj != 0) {
      if (!CsBrokenHeartP(obj)) {
        array<byte>::array_data::release(ptr<byte_vector>(obj)->heap_data);
        ptr<byte_vector>(obj)->heap_data = nullptr;
      }
      obj = ptr<byte_vector>(obj)->next;
    }
    c->oldSpace->bObjects = 0;
  }

  /* CsDestroyAllCObjects - destroy all cobjects */
  void CsDestroyAllByteVectors(VM *c) {
    value obj = c->newSpace->bObjects;
    while (obj != 0) {
      if (!CsBrokenHeartP(obj)) {
        array<byte>::array_data::release(ptr<byte_vector>(obj)->heap_data);
        ptr<byte_vector>(obj)->heap_data = nullptr;
      }
      obj = ptr<byte_vector>(obj)->next;
    }
    c->newSpace->bObjects = 0;
  }

} // namespace tis
