/* cobject.c - 'CObject' handler */
/*
Copyright (c) 2001-2004 Terra Informatica Software, Inc.
and Andrew Fedoniouk andrew@terrainformatica.com
All rights reserved
*/

#include <string.h>
#include "cs.h"

#include "sdk-headers.h"

namespace tis {

  value CsMakeAssetObject(VM *c, som_asset_t *ptr)
  {
    sciter::om::asset_add_ref(ptr);
    return CsMakeCPtrObject(c, c->assetDispatch, ptr);
  }

  value CsCObjectGetItem(VM *c, value obj, value tag);
  void CsCObjectSetItem(VM *c, value obj, value tag, value val);

  value AssetGetItem(VM *c, value obj, value tag) {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* psp = sciter::om::asset_get_passport(s)) {
        if (psp->item_getter) {
          tool::value key = value_to_value(c, tag);
          tool::value val;
          if (psp->item_getter(s, &key, &val))
            return value_to_value(c, val);
          else if (psp->flags & SOM_EXTENDABLE_OBJECT)
             return CsCObjectGetItem(c, obj, tag);
          return UNDEFINED_VALUE;
        }
      }
      CsThrowKnownError(c, CsErrGenericError, "[] unsupported");
    }
    CsThrowKnownError(c, CsErrGenericError, "already disposed");
    return UNDEFINED_VALUE;
  }

  void AssetSetItem(VM *c, value obj, value tag, value value) {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* psp = sciter::om::asset_get_passport(s)) {
        if (psp->item_getter) {
          tool::value key = value_to_value(c, tag);
          tool::value val = value_to_value(c, value);
          if (psp->item_setter(s, &key, &val))
            return;
        }
        if (psp->flags & SOM_EXTENDABLE_OBJECT) {
          CsCObjectSetItem(c,obj,tag,value);
          return;
        }
      }
      CsThrowKnownError(c, CsErrGenericError, "[] unsupported");
    }
    CsThrowKnownError(c, CsErrGenericError, "already disposed");
  }

  bool AssetGetProperty(VM *c, value &obj, value tag, value *pval) {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* pp = sciter::om::asset_get_passport(s))
      {
        if (pp->prop_getter && CsSymbolP(tag)) {
          tool::value tval;
          if (pp->prop_getter(s,tag,&tval)) {
            *pval = value_to_value(c, tval);
            return true;
          }
        }
        for (uint n = 0; n < pp->n_properties; ++n) {
          if (pp->properties[n].name == tag) {
            tool::value tval;
            if (!pp->properties[n].getter) return false;
            if (pp->properties[n].getter(s, &tval)) {
              *pval = value_to_value(c, tval);
              return true;
            }
          }
        }
        if (pp->flags & SOM_EXTENDABLE_OBJECT) {
          *pval = CsCObjectGetItem(c, obj, tag);
          return *pval != UNDEFINED_VALUE;
        }
        //CsThrowKnownError(c, CsErrNoProperty, obj, tag);
      } else
        CsThrowKnownError(c, CsErrGenericError, "asset:no passport");
      return false;
    }
    CsThrowKnownError(c, CsErrGenericError, "asset:already disposed");
    return false;
  }

  bool AssetSetProperty(VM *c, value obj, value tag, value val) {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* pp = sciter::om::asset_get_passport(s))
      {
        tool::value tval = value_to_value(c, val);
        if (pp->prop_setter && CsSymbolP(tag)) {
          if (pp->prop_setter(s, tag, &tval)) {
            return true;
          }
        }
        for (uint n = 0; n < pp->n_properties; ++n) {
          if (pp->properties[n].name == tag)
          {
            if (!pp->properties[n].setter) return false;
            return !!pp->properties[n].setter(s, &tval);
          }
        }
        if (pp->flags & SOM_EXTENDABLE_OBJECT) {
          CsCObjectSetItem(c, obj, tag, val);
          return true;
        }
        CsThrowKnownError(c, CsErrNoProperty, obj, tag);
      } else
        CsThrowKnownError(c, CsErrGenericError, "asset:no passport");
    }
    CsThrowKnownError(c, CsErrGenericError, "asset:already disposed");
    return false;
  }

#define ASSET_MAX_ARGS 32

  bool AssetHandleCall(VM *c, value obj, value tag, int argc, value *pretval)
  {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* pp = sciter::om::asset_get_passport(s))
      {
#if 0
        for (uint n = 0; n < pp->n_methods; ++n) {
          if (pp->methods[n].name == tag) {
            tool::value argv[8];
            uint argc = (uint)min(8, CsArgCnt(c) - 2);
            for (uint i = 0; i < argc; ++i)
              argv[i] = value_to_value(c, CsGetArg(c, i + 3));
            tool::value retval;
            BOOL r = pp->methods[n].func(s, argc, argv, &retval);
            if (r)
              *pretval = value_to_value(c, retval);
            return !!r;
          }
        }
#else
        som_method_t pm = nullptr;
        som_method_t pm_acceptable = nullptr;
        som_method_t pm_last_resort = nullptr;
        //uint argc = (uint)min(8, CsArgCnt(c) - 2);
        argc -= 2;



        if (argc > ASSET_MAX_ARGS)
          argc = ASSET_MAX_ARGS;

        for (uint n = 0; n < pp->n_methods; ++n) {
          if (pp->methods[n].name == tag) {
            if (pp->methods[n].params == uint(argc)) {
              pm = pp->methods[n].func;
              break;
            }
            if (pp->methods[n].params > uint(argc) && !pm_acceptable) {
              pm_acceptable = pp->methods[n].func;
            } else if(!pm_last_resort)
              pm_last_resort = pp->methods[n].func;
          }
        }

        if (!pm) {
          if (pm_acceptable)
            pm = pm_acceptable;
          else
            pm = pm_last_resort;
        }

        if (pm) {
          tool::value argv[ASSET_MAX_ARGS];
          for (int i = 0; i < argc; ++i)
            argv[i] = value_to_value(c, CsGetArg(c, i + 3));
          tool::value retval;
          SBOOL r = pm(s, argc, argv, &retval);
          if (r) {
            if (retval.is_error_string()) {
              string er = retval.get<ustring>();
              CsThrowKnownError(c, CsErrGenericError, er.c_str());
            }
            else
              *pretval = value_to_value(c, retval);
          }
          return !!r;
        }
#endif
        tool::string name = CsSymbolName(pp->name);
        CsThrowKnownError(c, CsErrNoMethod, name.c_str(), obj, tag);
      } else
        CsThrowKnownError(c, CsErrGenericError, "asset:no passport");
    }
    CsThrowKnownError(c, CsErrGenericError, "asset:already disposed");
    return false;
  }

  value AssetNextElement(VM *c, value *index, value obj, int nr) {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      if (som_passport_t* psp = sciter::om::asset_get_passport(s)) {
        if (psp->item_next) {
          tool::value idx = value_to_value(c, *index);
          tool::value val;
          if (psp->item_next(s, &idx, &val)) {
            *index = value_to_value(c, idx);
            return value_to_value(c, val);
          }
          return NOTHING_VALUE;
        }
      }
      CsThrowKnownError(c, CsErrGenericError, "Unsupported operation");
    }
    CsThrowKnownError(c, CsErrGenericError, "already disposed");
    return NOTHING_VALUE;
  }

  void AssetDispose(VM *c, value obj)
  {
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (s) {
      sciter::om::asset_release(s);
      CsSetCObjectValue(obj, nullptr);
    }
  }

  bool CsAssetP(value val) {
    dispatch *d = CsGetDispatch(val);
    return d->getProperty == AssetGetProperty;
  }


  static value CSF_typeOf(VM *c) {
    value obj;
    CsParseArguments(c, "**V=", &obj, c->assetDispatch);
    if (!CsAssetP(obj)) return UNDEFINED_VALUE;
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (!s) return NULL_VALUE;
    if (auto psp = sciter::om::asset_get_passport(s))
      return psp->name;
    return NULL_VALUE;
  }

  static value CSF_dispose(VM *c) {
    value obj;
    CsParseArguments(c, "**V=", &obj, c->assetDispatch);
    if (!CsAssetP(obj)) return UNDEFINED_VALUE;
     som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
     if (s) {
       sciter::om::asset_release(s);
       CsSetCObjectValue(obj, nullptr);
       return TRUE_VALUE;
     }
     return FALSE_VALUE;
  }

  static value CSF_hasMethod(VM *c) {
    value obj, name = 0;
    CsParseArguments(c, "**V=V=", &obj, c->assetDispatch, &name, &CsSymbolDispatch);
    if (!CsAssetP(obj)) return UNDEFINED_VALUE;
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (!s) return UNDEFINED_VALUE;
    if (auto psp = sciter::om::asset_get_passport(s)) {
      for (uint n = 0; n < psp->n_methods; ++n) {
        if (psp->methods[n].name == name)
          return TRUE_VALUE;
      }
    }
    return FALSE_VALUE;
  }

  static value CSF_hasProperty(VM *c) {
    value obj, name = 0;
    CsParseArguments(c, "**V=V=", &obj, c->assetDispatch, &name, &CsSymbolDispatch);
    if (!CsAssetP(obj)) return UNDEFINED_VALUE;
    som_asset_t *s = (som_asset_t *)CsCObjectValue(obj);
    if (!s) return UNDEFINED_VALUE;
    if (auto psp = sciter::om::asset_get_passport(s)) {
      for (uint n = 0; n < psp->n_properties; ++n) {
        if (psp->properties[n].name == name)
          return TRUE_VALUE;
      }
    }
    return FALSE_VALUE;
  }

  ustring CsAssetTypeName(value obj) {
    som_asset_t *sa = (som_asset_t *)CsCObjectValue(obj);
    if (!sa)
      return W("{disposed}");
    else if (auto psp = sciter::om::asset_get_passport(sa))
      return CsSymbolName(psp->name);
    return ustring();
  }

  tool::value _asset_to_map(som_asset_t *ptr) {
    if (auto psp = sciter::om::asset_get_passport(ptr)) {
      tool::value map;
      for (size_t n = 0; n < psp->n_properties; ++n) {
        tool::value val;
        if (psp->properties[n].getter(ptr, &val))
          map.set_item(tool::value(CsSymbolName(psp->properties[n].name)), val);
      }
      return map;
    }
    return tool::value();
  }

  tool::value CsAssetProps(value obj) {
    som_asset_t *sa = (som_asset_t *)CsCObjectValue(obj);
    if (!sa) return tool::value();
    return _asset_to_map(sa);
  }

  som_asset_t* CsAssetPtr(value obj) {
    assert(CsAssetP(obj));
    return (som_asset_t *)CsCObjectValue(obj);
  }


  bool AssetPrint(VM *c, value obj, stream *s, bool toLocale)
  {
    s->put_str("Asset(");
    s->put_str(CsAssetTypeName(obj));
    s->put_str(")");
    return true;
  }

  /* file methods */
  static c_method methods[] = {
    C_METHOD_ENTRY("typeOf", CSF_typeOf),
    C_METHOD_ENTRY("dispose", CSF_dispose),
    C_METHOD_ENTRY("hasMethod", CSF_hasMethod),
    C_METHOD_ENTRY("hasProperty", CSF_hasProperty),
    C_METHOD_ENTRY(0, 0),
  };

  void CsInitAsset(VM* c) {
    dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(c), "Asset", methods, nullptr);

    pd->baseType = &CsCObjectDispatch;

    pd->getProperty = AssetGetProperty;
    pd->setProperty = AssetSetProperty;
    //pd->hash = AssetHash;
    pd->getItem = AssetGetItem;
    pd->setItem = AssetSetItem;
    pd->print = AssetPrint;

    pd->getNextElement = (get_next_element_t)AssetNextElement;

    pd->destroy = (destructor_t)AssetDispose;

    pd->handleCall = (call_method_t)AssetHandleCall;

    c->assetDispatch = pd;

  }

}
